/**********************************************************************
Copyright (c) 2004 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License. 


Contributors:
    ...
**********************************************************************/
package org.datanucleus.metadata;

/**
 * MetaData representing a unique constraint.
 *
 * @since 1.1
 * @version $Revision: 1.17 $
 */
public class UniqueMetaData extends AbstractConstraintMetaData implements ColumnMetaDataContainer
{
    /**
     * Whether the unique is initially deferred.
     */
    final boolean deferred;

    /**
     * Constructor to create a copy of the passed metadata using the provided parent.
     * @param umd The metadata to copy
     */
    public UniqueMetaData(UniqueMetaData umd)
    {
        super(null, umd.name, umd.table);

        this.deferred = umd.deferred;
        // TODO Change these to copy rather than reference
        for (int i=0;i<umd.members.size();i++)
        {
            addMember((AbstractMemberMetaData)umd.members.get(i));
        }
        for (int i=0;i<umd.columns.size();i++)
        {
            addColumn((ColumnMetaData)umd.columns.get(i));
        }
    }

    /**
     * Constructor.
     * @param name Name of unique constraint
     * @param table Name of the table
     * @param deferredValue Whether the unique is deferred initially
     */
    public UniqueMetaData(final String name, final String table, final String deferredValue)
    {
        super(null, name, table); // Ignore parent

        if (deferredValue != null && deferredValue.equalsIgnoreCase("true"))
        {
            this.deferred = true;
        }
        else
        {
            this.deferred = false;
        }
    }

    /**
     * Method to initialise the object, creating internal convenience arrays.
     * Initialise all sub-objects.
     */
    public void initialise()
    {
        if (isInitialised())
        {
            return;
        }

        // Set up the fieldMetaData
        if (members.size() == 0)
        {
            memberMetaData = null;
        }
        else
        {
            memberMetaData = new AbstractMemberMetaData[members.size()];
            for (int i=0; i<memberMetaData.length; i++)
            {
                memberMetaData[i] = (AbstractMemberMetaData) members.get(i);
                memberMetaData[i].initialise();
            }
        }

        // Set up the columnMetaData
        if (columns.size() == 0)
        {
            columnMetaData = null;
        }
        else
        {
            columnMetaData = new ColumnMetaData[columns.size()];
            for (int i=0; i<columnMetaData.length; i++)
            {
                columnMetaData[i] = (ColumnMetaData) columns.get(i);
                columnMetaData[i].initialise();
            }
        }

        // Clear out parsing data
        members.clear();
        members = null;
        columns.clear();
        columns = null;

        setInitialised();
    }

    // -------------------------------- Mutators -------------------------------

    /**
     * Accessor for whether the unique constraint is deferred.
     * @return Returns whether the unique constraint is deferred
     */
    public final boolean isDeferred()
    {
        return deferred;
    }

    // -------------------------------- Utilities ------------------------------

    /**
     * Returns a string representation of the object.
     * This can be used as part of a facility to output a MetaData file. 
     * @param prefix prefix string
     * @param indent indent string
     * @return a string representation of the object.
     */
    public String toString(String prefix,String indent)
    {
        StringBuffer sb = new StringBuffer();
        sb.append(prefix).append("<unique");
        if (table != null)
        {
            sb.append(" table=\"" + table + "\"");
        }
        if (deferred)
        {
            sb.append(" deferred=\"true\"");
        }
        sb.append(name != null ? (" name=\"" + name + "\">\n") : ">\n");

        // Add fields
        if (memberMetaData != null)
        {
            for (int i=0;i<memberMetaData.length;i++)
            {
                sb.append(memberMetaData[i].toString(prefix + indent,indent));
            }
        }

        // Add columns
        if (columnMetaData != null)
        {
            for (int i=0;i<columnMetaData.length;i++)
            {
                sb.append(columnMetaData[i].toString(prefix + indent,indent));
            }
        }

        // Add extensions
        sb.append(super.toString(prefix + indent,indent));

        sb.append(prefix).append("</unique>\n");
        return sb.toString();
    }
}