/**********************************************************************
Copyright (c) 2006 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.metadata.xml;

import org.xml.sax.Attributes;
import org.xml.sax.EntityResolver;
import org.xml.sax.SAXException;

import org.datanucleus.metadata.MetaData;
import org.datanucleus.metadata.MetaDataManager;
import org.datanucleus.metadata.PersistenceFileMetaData;
import org.datanucleus.metadata.PersistenceUnitMetaData;
import org.datanucleus.util.NucleusLogger;

/**
 * Parser handler for "persistence.xml" files to convert them into a PersistenceFileMetaData.
 * Implements DefaultHandler and handles the extracting of MetaData from the
 * XML elements/attributes. This class simply constructs the MetaData representation
 * mirroring what is in the MetaData file.
 * <P>Operates the parse process using a Stack. MetaData components are added
 * to the stack as they are encountered and created. They are then popped off
 * the stack when the end element is encountered.</P>
 *
 * @since 1.1
 * @version $Revision: 1.2 $
 */
public class PersistenceFileMetaDataHandler extends AbstractMetaDataHandler
{
    /**
     * Constructor. Protected to prevent instantiation.
     * @param mgr the metadata manager
     * @param filename The name of the file to parse
     * @param resolver Entity Resolver to use (null if not available)
     */
    public PersistenceFileMetaDataHandler(MetaDataManager mgr, String filename, EntityResolver resolver)
    {
        super(mgr, filename, resolver);
        metadata = new PersistenceFileMetaData(filename);
        pushStack(metadata);
    }

    /**
     * Handler method called at the start of an element.
     * @param uri URI of the tag
     * @param localName Local name
     * @param qName Element name
     * @param attrs Attributes for this element 
     * @throws SAXException in parsing errors
     */
    public void startElement(String uri, 
                             String localName,
                             String qName,
                             Attributes attrs)
    throws SAXException 
    {
        if (NucleusLogger.METADATA.isDebugEnabled())
        {
            StringBuffer sb = new StringBuffer();
            sb.append("<" + qName);
            for (int i=0; i<attrs.getLength(); i++)
            {
                sb.append(" ");
                sb.append(attrs.getQName(i)).append("=\"").append(attrs.getValue(i)).append("\"");
            }
            sb.append(">");
            NucleusLogger.METADATA.debug(LOCALISER.msg("044034",sb.toString(), "" + stack.size()));
        }
        if (localName.length()<1)
        {
            localName = qName;
        }
        try
        {
            if (localName.equals("persistence"))
            {
                // New "persistence" file
                // Do nothing - created in our constructor
            }
            else if (localName.equals("persistence-unit"))
            {
                // New "persistence-unit"
                PersistenceFileMetaData filemd = (PersistenceFileMetaData)getStack();
                PersistenceUnitMetaData pumd = new PersistenceUnitMetaData(filemd, 
                    getAttr(attrs, "name"), getAttr(attrs, "transaction-type"));
                filemd.addPersistenceUnit(pumd);
                pushStack(pumd);
            }
            else if (localName.equals("properties"))
            {
                // Do nothing
            }
            else if (localName.equals("property"))
            {
                // New "property" for the current persistence unit
                PersistenceUnitMetaData pumd = (PersistenceUnitMetaData)getStack();
                pumd.addProperty(getAttr(attrs, "name"), getAttr(attrs, "value"));
            }
            else if (localName.equals("mapping-file"))
            {
                // Processed elsewhere
            }
            else if (localName.equals("class"))
            {
                // Processed elsewhere
            }
            else if (localName.equals("jar-file"))
            {
                // Processed elsewhere
            }
            else if (localName.equals("jta-data-source"))
            {
                // Processed elsewhere
            }
            else if (localName.equals("non-jta-data-source"))
            {
                // Processed elsewhere
            }
            else if (localName.equals("description"))
            {
                // Processed elsewhere
            }
            else if (localName.equals("provider"))
            {
                // Processed elsewhere
            }
            else if (localName.equals("exclude-unlisted-classes"))
            {
                PersistenceUnitMetaData pumd = (PersistenceUnitMetaData)getStack();
                pumd.setExcludeUnlistedClasses();
            }
            else
            {
                String message = LOCALISER.msg("044037",qName);
                NucleusLogger.METADATA.error(message);
                throw new RuntimeException(message);
            }
        }
        catch (RuntimeException ex)
        {
            NucleusLogger.METADATA.error(LOCALISER.msg("044042", qName, getStack(), uri), ex);
            throw ex;
        }
    }

    /**
     * Handler method called at the end of an element.
     * @param uri URI of the tag
     * @param localName local name
     * @param qName Name of element just ending
     * @throws SAXException in parsing errors
     */
    public void endElement(String uri, String localName, String qName)
    throws SAXException
    {
        if (NucleusLogger.METADATA.isDebugEnabled())
        {
            NucleusLogger.METADATA.debug(LOCALISER.msg("044035","<" + qName + ">", "" + stack.size()));
        }
        if (localName.length()<1)
        {
            localName = qName;
        }

        // Save the current string for elements that have a body value
        String currentString = getString().trim();
        if (currentString.length() > 0)
        {
            MetaData md = getStack();
            if (localName.equals("description"))
            {
                // Unit description
                ((PersistenceUnitMetaData)md).setDescription(currentString);
            }
            else if (localName.equals("provider"))
            {
                // Unit provider
                ((PersistenceUnitMetaData)md).setProvider(currentString);
            }
            else if (localName.equals("jta-data-source"))
            {
                // JTA data source
                ((PersistenceUnitMetaData)md).setJtaDataSource(currentString);
            }
            else if (localName.equals("non-jta-data-source"))
            {
                // Non-JTA data source
                ((PersistenceUnitMetaData)md).setNonJtaDataSource(currentString);
            }
            else if (localName.equals("class"))
            {
                // New persistent class
                ((PersistenceUnitMetaData)md).addClassName(currentString);
            }
            else if (localName.equals("mapping-file"))
            {
                // New mapping file
                ((PersistenceUnitMetaData)md).addMappingFile(currentString);
            }
            else if (localName.equals("jar-file"))
            {
                // New jar file
                ((PersistenceUnitMetaData)md).addJarFile(currentString);
            }
        }

        // Pop the tag
        // If startElement pushes an element onto the stack need a remove here for that type
        if (qName.equals("persistence-unit"))
        {
            popStack();
        }
    }
}