/**********************************************************************
Copyright (c) 2008 Erik Bengtson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
2008 Andy Jefferson - much restructuring, static methods, negate/complement operators
    ...
**********************************************************************/
package org.datanucleus.query.compiler;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Stack;

import org.datanucleus.ObjectManagerFactoryImpl;
import org.datanucleus.exceptions.NucleusUserException;
import org.datanucleus.query.node.Node;
import org.datanucleus.query.node.ParameterNode;
import org.datanucleus.store.query.QueryCompilerSyntaxException;
import org.datanucleus.util.Localiser;
import org.datanucleus.util.StringUtils;

/**
 * Implementation of a parser for JDOQL query language.
 * Generates Node tree(s) by use of the various compileXXX() methods.
 */
public class JDOQLParser implements Parser
{
    /** Localiser for messages. */
    protected static final Localiser LOCALISER = Localiser.getInstance(
        "org.datanucleus.Localisation", ObjectManagerFactoryImpl.class.getClassLoader());

    private static String[] jdoqlMethodNames = {"contains", "get", "containsKey", "containsValue", "isEmpty",
        "size", "toLowerCase", "toUpperCase", "indexOf", "matches", "substring", "startsWith", "endsWith",
        "Math.abs", "Math.sqrt", "JDOHelper.getObjectId", "JDOHelper.getVersion"
    };

    private String jdoqlMode = "DataNucleus";

    private Lexer p;
    private Stack stack = new Stack();

    /** Characters that parameters can be prefixed by. */
    private static String paramPrefixes = ":";

    /**
     * Constructor for a JDOQL Parser.
     * Supports "jdoql.level" option so can have strict JDO2 syntax, or flexible.
     * @param options parser options
     */
    public JDOQLParser(Map options)
    {
        if (options != null && options.containsKey("jdoql.level"))
        {
            jdoqlMode = (String)options.get("jdoql.level");
        }
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.compiler.Parser#compile(java.lang.String)
     */
    public Node compile(String expression)
    {
        p = new Lexer(expression, paramPrefixes);
        stack = new Stack();
        return compileExpression();
    }
    
    /* (non-Javadoc)
     * @see org.datanucleus.query.compiler.Parser#compileVariable(java.lang.String)
     */
    public Node compileVariable(String expression)
    {
        p = new Lexer(expression, paramPrefixes);
        stack = new Stack();
        if (!compileIdentifier())
        {
            throw new QueryCompilerSyntaxException("expected identifier", p.getIndex(), p.getInput());
        }
        if (!compileIdentifier())
        {
            throw new QueryCompilerSyntaxException("expected identifier", p.getIndex(), p.getInput());
        }
        Node nodeVariable = (Node) stack.pop();
        Node nodeType = (Node) stack.pop();
        nodeType.appendChildNode(nodeVariable);
        return nodeType;
    }

    /**
     * Method to compile the "from" clause, but JDOQL has no "from" so do nothing.
     * @param expression From string
     * @return Node trees for this from clause
     */
    public Node[] compileFrom(String expression)
    {
        return null;
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.compiler.Parser#compileOrder(java.lang.String)
     */
    public Node[] compileOrder(String expression)
    {
        p = new Lexer(expression, paramPrefixes);
        stack = new Stack();
        return compileOrderExpression();
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.compiler.Parser#compileTupple(java.lang.String)
     */
    public Node[] compileTupple(String expression)
    {
        p = new Lexer(expression, paramPrefixes);
        stack = new Stack();
        List nodes = new ArrayList();
        do
        {
            compileExpression();
            Node expr = (Node) stack.pop();
            nodes.add(expr);
        }
        while (p.parseString(","));
        return (Node[])nodes.toArray(new Node[nodes.size()]);
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.compiler.Parser#compileVariables(java.lang.String)
     */
    public Node[][] compileVariables(String expression)
    {
        p = new Lexer(expression, paramPrefixes);
        List nodes = new ArrayList();
        do
        {
            compilePrimary();
            if (stack.isEmpty())
            {
                throw new QueryCompilerSyntaxException("Parsing variable list and expected variable type", 
                    p.getIndex(), p.getInput());
            }
            if (!compileIdentifier())
            {
                throw new QueryCompilerSyntaxException("Parsing variable list and expected variable name",
                    p.getIndex(), p.getInput());
            }

            Node nodeVariable = (Node) stack.pop();
            String varName = (String)nodeVariable.getNodeValue();
            if (!StringUtils.isValidJavaIdentifierForJDOQL(varName))
            {
                throw new NucleusUserException(LOCALISER.msg("021105",varName));
            }

            Node nodeType = (Node) stack.pop();
            nodes.add(new Node[]{nodeType, nodeVariable});
        }
        while (p.parseString(";"));
        return (Node[][]) nodes.toArray(new Node[nodes.size()][2]);
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.compiler.Parser#compileParameters(java.lang.String)
     */
    public Node[][] compileParameters(String expression)
    {
        p = new Lexer(expression, paramPrefixes);
        List nodes = new ArrayList();
        do
        {
            compilePrimary();
            if (stack.isEmpty())
            {
                throw new QueryCompilerSyntaxException("expected identifier", p.getIndex(), p.getInput());
            }
            if (!compileIdentifier())
            {
                throw new QueryCompilerSyntaxException("expected identifier", p.getIndex(), p.getInput());
            }
            Node nodeVariable = (Node) stack.pop();
            Node nodeType = (Node) stack.pop();
            nodes.add(new Node[]{nodeType, nodeVariable});
        }
        while (p.parseString(","));
        return (Node[][]) nodes.toArray(new Node[nodes.size()][2]);
    }

    private Node[] compileOrderExpression()
    {
        List nodes = new ArrayList();
        do
        {
            compileExpression();
            if (p.parseString("asc") || p.parseString("ascending") ||
                p.parseString("ASC") || p.parseString("ASCENDING"))
            {
                Node expr = new Node(Node.OPERATOR, "ascending");
                stack.push(expr);
            }
            else if (p.parseString("desc") || p.parseString("descending") ||
                     p.parseString("DESC") || p.parseString("DESCENDING"))
            {
                Node expr = new Node(Node.OPERATOR, "descending");
                stack.push(expr);
            }
            Node expr = new Node(Node.OPERATOR, "order");
            expr.insertChildNode((Node) stack.pop());
            if (!stack.empty())
            {
                expr.insertChildNode((Node) stack.pop());
            }
            nodes.add(expr);
        }
        while (p.parseString(","));
        return (Node[]) nodes.toArray(new Node[nodes.size()]);
    }

    private Node compileExpression()
    {
        compileConditionalOrExpression();
        return (Node) stack.peek();
    }

    /**
     * This method deals with the OR condition
     * A condition specifies a combination of one or more expressions and logical (Boolean) operators and 
     * returns a value of TRUE, FALSE, or unknown
     */
    private void compileConditionalOrExpression()
    {
        compileConditionalAndExpression();

        while (p.parseString("||"))
        {
            compileConditionalAndExpression();
            Node expr = new Node(Node.OPERATOR, "||");
            expr.insertChildNode((Node) stack.pop());
            expr.insertChildNode((Node) stack.pop());
            stack.push(expr);
        }
    }

    /**
     * This method deals with the AND condition
     * A condition specifies a combination of one or more expressions and
     * logical (Boolean) operators and returns a value of TRUE, FALSE, or 
     * unknown
     */
    private void compileConditionalAndExpression()
    {
        compileInclusiveOrExpression();

        while (p.parseString("&&"))
        {
            compileInclusiveOrExpression();
            Node expr = new Node(Node.OPERATOR, "&&");
            expr.insertChildNode((Node) stack.pop());
            expr.insertChildNode((Node) stack.pop());
            stack.push(expr);
        }
    }

    private void compileInclusiveOrExpression()
    {
        compileExclusiveOrExpression();

        while (p.parseChar('|', '|'))
        {
            compileExclusiveOrExpression();
            Node expr = new Node(Node.OPERATOR, "|");
            expr.insertChildNode((Node) stack.pop());
            expr.insertChildNode((Node) stack.pop());
            stack.push(expr);
        }
    }

    private void compileExclusiveOrExpression()
    {
        compileAndExpression();

        while (p.parseChar('^'))
        {
            compileAndExpression();
            Node expr = new Node(Node.OPERATOR, "^");
            expr.insertChildNode((Node) stack.pop());
            expr.insertChildNode((Node) stack.pop());
            stack.push(expr);
        }
    }

    private void compileAndExpression()
    {
        compileEqualityExpression();

        while (p.parseChar('&', '&'))
        {
            compileEqualityExpression();
            Node expr = new Node(Node.OPERATOR, "&");
            expr.insertChildNode((Node) stack.pop());
            expr.insertChildNode((Node) stack.pop());
            stack.push(expr);
        }
    }

    private void compileEqualityExpression()
    {
        compileRelationalExpression();

        for (;;)
        {
            if (p.parseString("=="))
            {
                compileRelationalExpression();
                Node expr = new Node(Node.OPERATOR, "==");
                expr.insertChildNode((Node) stack.pop());
                expr.insertChildNode((Node) stack.pop());
                stack.push(expr);
            }
            else if (p.parseString("!="))
            {
                compileRelationalExpression();
                Node expr = new Node(Node.OPERATOR, "!=");
                expr.insertChildNode((Node) stack.pop());
                expr.insertChildNode((Node) stack.pop());
                stack.push(expr);
            }
            else if (p.parseString("="))
            {
                // Assignment operator is invalid (user probably meant to specify "==")
                throw new QueryCompilerSyntaxException("Invalid operator \"=\". Did you mean to use \"==\"?");
            }
            else
            {
                break;
            }
        }
    }

    private void compileRelationalExpression()
    {
        compileAdditiveExpression();

        for (;;)
        {
            if (p.parseString("<="))
            {
                compileAdditiveExpression();
                Node expr = new Node(Node.OPERATOR, "<=");
                expr.insertChildNode((Node) stack.pop());
                expr.insertChildNode((Node) stack.pop());
                stack.push(expr);
            }
            else if (p.parseString(">="))
            {
                compileAdditiveExpression();
                Node expr = new Node(Node.OPERATOR, ">=");
                expr.insertChildNode((Node) stack.pop());
                expr.insertChildNode((Node) stack.pop());
                stack.push(expr);
            }
            else if (p.parseChar('<'))
            {
                compileAdditiveExpression();
                Node expr = new Node(Node.OPERATOR, "<");
                expr.insertChildNode((Node) stack.pop());
                expr.insertChildNode((Node) stack.pop());
                stack.push(expr);
            }
            else if (p.parseChar('>'))
            {
                compileAdditiveExpression();
                Node expr = new Node(Node.OPERATOR, ">");
                expr.insertChildNode((Node) stack.pop());
                expr.insertChildNode((Node) stack.pop());
                stack.push(expr);
            }
            else if (p.parseString("instanceof"))
            {
                compileAdditiveExpression();
                Node expr = new Node(Node.OPERATOR, "instanceof");
                expr.insertChildNode((Node) stack.pop());
                expr.insertChildNode((Node) stack.pop());
                stack.push(expr);
            }
            else
            {
                break;
            }
        }
    }

    protected void compileAdditiveExpression()
    {
        compileMultiplicativeExpression();

        for (;;)
        {
            if (p.parseChar('+'))
            {
                compileMultiplicativeExpression();
                Node expr = new Node(Node.OPERATOR, "+");
                expr.insertChildNode((Node) stack.pop());
                expr.insertChildNode((Node) stack.pop());
                stack.push(expr);
            }
            else if (p.parseChar('-'))
            {
                compileMultiplicativeExpression();
                Node expr = new Node(Node.OPERATOR, "-");
                expr.insertChildNode((Node) stack.pop());
                expr.insertChildNode((Node) stack.pop());
                stack.push(expr);
            }
            else
            {
                break;
            }
        }
    }

    protected void compileMultiplicativeExpression()
    {
        compileUnaryExpression();

        for (;;)
        {
            if (p.parseChar('*'))
            {
                compileMultiplicativeExpression();
                Node expr = new Node(Node.OPERATOR, "*");
                expr.insertChildNode((Node) stack.pop());
                expr.insertChildNode((Node) stack.pop());
                stack.push(expr);
            }
            else if (p.parseChar('/'))
            {
                compileMultiplicativeExpression();
                Node expr = new Node(Node.OPERATOR, "/");
                expr.insertChildNode((Node) stack.pop());
                expr.insertChildNode((Node) stack.pop());
                stack.push(expr);
            }
            else if (p.parseChar('%'))
            {
                compileMultiplicativeExpression();
                Node expr = new Node(Node.OPERATOR, "%");
                expr.insertChildNode((Node) stack.pop());
                expr.insertChildNode((Node) stack.pop());
                stack.push(expr);
            }
            else
            {
                break;
            }
        }
    }

    protected void compileUnaryExpression()
    {
        if (p.parseString("++"))
        {
            throw new QueryCompilerSyntaxException("Unsupported operator '++'");
        }
        else if (p.parseString("--"))
        {
            throw new QueryCompilerSyntaxException("Unsupported operator '--'");
        }

        if (p.parseChar('+'))
        {
            // Just swallow + and leave remains on the stack
            compileUnaryExpression();
        }
        else if (p.parseChar('-'))
        {
            compileUnaryExpression();
            Node expr = new Node(Node.OPERATOR, "NEG");
            expr.insertChildNode((Node) stack.pop());
            stack.push(expr);
        }
        else
        {
            compileUnaryExpressionNotPlusMinus();
        }
    }

    protected void compileUnaryExpressionNotPlusMinus()
    {
        if (p.parseChar('~'))
        {
            compileUnaryExpression();
            Node expr = new Node(Node.OPERATOR, "~");
            expr.insertChildNode((Node) stack.pop());
            stack.push(expr);
        }
        else if (p.parseChar('!'))
        {
            compileUnaryExpression();
            Node expr = new Node(Node.OPERATOR, "!");
            expr.insertChildNode((Node) stack.pop());
            stack.push(expr);
        }
        else
        {
            compilePrimary();
        }
    }

    /**
     * Compiles the primary. First look for a literal (e.g. "text"), then
     * an identifier(e.g. variable) In the next step, call a function, if
     * executing a function, on the literal or the identifier found.
     */
    protected void compilePrimary()
    {
        if (p.parseStringIgnoreCase("DISTINCT"))
        {
            // Aggregates can have "count(DISTINCT field1)"
            Node distinctNode = new Node(Node.OPERATOR, "DISTINCT");
            compileIdentifier();
            Node identifierNode = (Node)stack.pop();
            distinctNode.appendChildNode(identifierNode);
            stack.push(distinctNode);
            return;
        }
        if (compileCreator())
        {
            return;
        }
        if (compileLiteral())
        {
            return;
        }
        if (compileMethod())
        {
            return;
        }

        if (p.parseChar('('))
        {
            compileExpression();
            if (!p.parseChar(')'))
            {
                throw new QueryCompilerSyntaxException("expected ')'", p.getIndex(), p.getInput());
            }
            return;
        }

        // if primary == null, literal not found...
        // We will have an identifier (variable, parameter, or field of candidate class)
        if (!compileIdentifier())
        {
            throw new QueryCompilerSyntaxException("Identifier expected", p.getIndex(), p.getInput());
        }
        int size = stack.size();

        // Generate Node tree, including chained operations
        // e.g identifier.methodX().methodY().methodZ() 
        //     -> node (IDENTIFIER) with child (INVOKE), with child (INVOKE), with child (INVOKE)
        // e.g identifier.fieldX.fieldY.fieldZ
        //     -> node (IDENTIFIER) with child (IDENTIFIER), with child (IDENTIFIER), with child (IDENTIFIER)
        while (p.parseChar('.'))
        {
            if (compileMethod())
            {
                // "a.method(...)"
            }
            else if (compileIdentifier())
            {
                // "a.field"
            }
            else
            {
                throw new QueryCompilerSyntaxException("Identifier expected", p.getIndex(), p.getInput());
            }
        }

        while (stack.size() > size)
        {
            Node top = (Node) stack.pop();
            Node peek = ((Node) stack.peek());
            peek.insertChildNode(top);
        }
    }

    /**
     * Method to parse "new a.b.c(param1[,param2], ...)" and create a Node of type CREATOR.
     * The Node at the top of the stack after this call will have any arguments defined in its "properties".
     * @return whether method syntax was found.
     */
    private boolean compileCreator()
    {
        if (p.parseString("new"))
        {
            int size = stack.size();
            if (!compileMethod())
            {
                if (!compileIdentifier())
                {
                    throw new QueryCompilerSyntaxException("Identifier expected", p.getIndex(), p.getInput());
                }

                while (p.parseChar('.'))
                {
                    if (compileMethod())
                    {
                        // "a.method(...)"
                    }
                    else if (compileIdentifier())
                    {
                        // "a.field"
                    }
                    else
                    {
                        throw new QueryCompilerSyntaxException("Identifier expected", p.getIndex(), p.getInput());
                    }
                }
            }
            while (stack.size() - 1 > size)
            {
                Node top = (Node) stack.pop();
                Node peek = ((Node) stack.peek());
                peek.insertChildNode(top);
            }
            Node expr = (Node) stack.pop();
            Node newExpr = new Node(Node.CREATOR);
            newExpr.insertChildNode(expr);
            stack.push(newExpr);
            return true;
        }
        return false;
    }

    /**
     * Method to parse "methodName(param1[,param2], ...)" and create a Node of type INVOKE.
     * The Node at the top of the stack after this call will have any arguments defined in its "properties".
     * @return whether method syntax was found.
     */
    private boolean compileMethod()
    {
        String method = p.parseMethod();
        if (method != null)
        {
            p.skipWS();
            p.parseChar('(');

            if (stack.size() > 0)
            {
                // Catch any statically supported methods and merge "primary expression" with method.
                // TODO Would be nice to do this using plugin extension-point info
                Node primExprNode = (Node)stack.peek();
                if (primExprNode != null && primExprNode.getNodeValue().equals("JDOHelper"))
                {
                    if (method.equals("getObjectId"))
                    {
                        method = "JDOHelper.getObjectId";
                        stack.pop();
                    }
                    else if (method.equals("getVersion"))
                    {
                        method = "JDOHelper.getVersion";
                        stack.pop();
                    }
                }
                if (primExprNode != null && primExprNode.getNodeValue().equals("Math"))
                {
                    if (method.equals("abs"))
                    {
                        method = "Math.abs";
                        stack.pop();
                    }
                    else if (method.equals("sqrt"))
                    {
                        method = "Math.sqrt";
                        stack.pop();
                    }
                }
            }

            if (jdoqlMode.equals("JDO2"))
            {
                // Enable strictness checking for levels of JDOQL
                // Note that this only checks the method name and not the arguments/types
                if (Arrays.binarySearch(jdoqlMethodNames, method) < 0)
                {
                    throw new QueryCompilerSyntaxException("Query uses method \"" + method + 
                        "\" but this is not a standard JDOQL method name");
                }
            }

            // Found syntax for a method, so invoke the method
            Node expr = new Node(Node.INVOKE, method);
            if (!p.parseChar(')'))
            {
                int numArgs = 0;
                do
                {
                    // Argument for the method call, add as a node property
                    compileExpression();
                    expr.addProperty((Node)stack.pop());
                    numArgs++;
                }
                while (p.parseChar(','));

                if (!p.parseChar(')'))
                {
                    throw new QueryCompilerSyntaxException("')' expected", p.getIndex(), p.getInput());
                }
            }

            stack.push(expr);
            return true;
        }
        return false;
    }

    /**
     * A literal is one value of any type.
     * Supported literals are of types String, Floating Point, Integer,
     * Character, Boolean and null e.g. 'J', "String", 1, 1.8, true, false, null.
     * @return The compiled literal
     */
    protected boolean compileLiteral()
    {
        Object litValue = null;

        String sLiteral;
        BigDecimal fLiteral;
        BigInteger iLiteral;
        Boolean bLiteral;

        boolean single_quote_next = p.nextIsSingleQuote();
        if ((sLiteral = p.parseStringLiteral()) != null)
        {
            // Both String and Character are allowed to use single-quotes
            // so we need to check if it was single-quoted and
            // use CharacterLiteral if length is 1.
            if (sLiteral.length() == 1 && single_quote_next)
            {
                litValue = new Character(sLiteral.charAt(0));
            }
            else
            {
                litValue = sLiteral;
            }
        }
        else if ((fLiteral = p.parseFloatingPointLiteral()) != null)
        {
            litValue = fLiteral;
        }
        else if ((iLiteral = p.parseIntegerLiteral()) != null)
        {
            litValue = new Long(iLiteral.longValue());
        }
        else if ((bLiteral = p.parseBooleanLiteral()) != null)
        {
            litValue = bLiteral;
        }
        else if (p.parseNullLiteral())
        {
            litValue = null;
        }
        else
        {
            return false;
        }

        stack.push(new Node(Node.LITERAL, litValue));
        return true;
    }

    int parameterPosition = 0;

    /**
     * An identifier always designates a reference to a single value.
     * A single value can be one collection, one field.
     * @return The compiled identifier
     */
    private boolean compileIdentifier()
    {
        String id = p.parseIdentifier();
        if (id == null)
        {
            return false;
        }
        char first = id.charAt(0);
        if (paramPrefixes.indexOf(first) >= 0)
        {
            // Parameter identifier, so strip off first char
            Node expr = new ParameterNode(Node.PARAMETER, id.substring(1), parameterPosition);
            parameterPosition++;
            stack.push(expr);
            return true;
        }
        else
        {
            Node expr = new Node(Node.IDENTIFIER, id);
            stack.push(expr);
            return true;
        }
    }
}