/**********************************************************************
Copyright (c) 2008 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.query.compiler;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Stack;

import org.datanucleus.exceptions.NucleusException;
import org.datanucleus.exceptions.NucleusUserException;
import org.datanucleus.query.node.Node;
import org.datanucleus.query.node.ParameterNode;
import org.datanucleus.store.query.QueryCompilerSyntaxException;

/**
 * Implementation of a parser for JPQL query language.
 */
public class JPQLParser implements Parser
{
    private Lexer p;
    private Stack stack = new Stack();

    /** Characters that parameters can be prefixed by. */
    private static String paramPrefixes = ":?";

    /**
     * Constructor for a JPQL Parser.
     * @param options parser options
     */
    public JPQLParser(Map options)
    {
        
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.compiler.Parser#compile(java.lang.String)
     */
    public Node compile(String expression)
    {
        p = new Lexer(expression, paramPrefixes);
        stack = new Stack();
        return compileExpression();
    }
    
    /* (non-Javadoc)
     * @see org.datanucleus.query.compiler.Parser#compileVariable(java.lang.String)
     */
    public Node compileVariable(String expression)
    {
        p = new Lexer(expression, paramPrefixes);
        stack = new Stack();
        if (!compileIdentifier())
        {
            throw new QueryCompilerSyntaxException("expected identifier", p.getIndex(), p.getInput());
        }
        if (!compileIdentifier())
        {
            throw new QueryCompilerSyntaxException("expected identifier", p.getIndex(), p.getInput());
        }
        Node nodeVariable = (Node) stack.pop();
        Node nodeType = (Node) stack.pop();
        nodeType.appendChildNode(nodeVariable);
        return nodeType;
    }

    /**
     * Method to compile the "from" clause, but JDOQL has no "from" so do nothing.
     * @param expression From string
     * @return Node trees for this from clause
     */
    public Node[] compileFrom(String expression)
    {
        p = new Lexer(expression, paramPrefixes);
        stack = new Stack();
        return compileFromExpression();
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.compiler.Parser#compileOrder(java.lang.String)
     */
    public Node[] compileOrder(String expression)
    {
        p = new Lexer(expression, paramPrefixes);
        stack = new Stack();
        return compileOrderExpression();
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.compiler.Parser#compileTupple(java.lang.String)
     */
    public Node[] compileTupple(String expression)
    {
        p = new Lexer(expression, paramPrefixes);
        stack = new Stack();
        List nodes = new ArrayList();
        do
        {
            compileExpression();
            Node expr = (Node) stack.pop();
            nodes.add(expr);
        }
        while (p.parseString(","));
        return (Node[])nodes.toArray(new Node[nodes.size()]);
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.compiler.Parser#compileVariables(java.lang.String)
     */
    public Node[][] compileVariables(String expression)
    {
        p = new Lexer(expression, paramPrefixes);
        List nodes = new ArrayList();
        do
        {
            compilePrimary();
            if (stack.isEmpty())
            {
                throw new QueryCompilerSyntaxException("expected identifier", p.getIndex(), p.getInput());
            }
            if (!compileIdentifier())
            {
                throw new QueryCompilerSyntaxException("expected identifier", p.getIndex(), p.getInput());
            }
            Node nodeVariable = (Node) stack.pop();
            Node nodeType = (Node) stack.pop();
            nodes.add(new Node[]{nodeType, nodeVariable});
        }
        while (p.parseString(";"));
        return (Node[][]) nodes.toArray(new Node[nodes.size()][2]);
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.compiler.Parser#compileParameters(java.lang.String)
     */
    public Node[][] compileParameters(String expression)
    {
        p = new Lexer(expression, paramPrefixes);
        List nodes = new ArrayList();
        do
        {
            compilePrimary();
            if (stack.isEmpty())
            {
                throw new QueryCompilerSyntaxException("expected identifier", p.getIndex(), p.getInput());
            }
            if (!compileIdentifier())
            {
                throw new QueryCompilerSyntaxException("expected identifier", p.getIndex(), p.getInput());
            }
            Node nodeVariable = (Node) stack.pop();
            Node nodeType = (Node) stack.pop();
            nodes.add(new Node[]{nodeType, nodeVariable});
        }
        while (p.parseString(","));
        return (Node[][]) nodes.toArray(new Node[nodes.size()][2]);
    }

    /**
     * The FROM expression in JPQL is a comma-separated list of expressions. Each expression can be
     * <ul>
     * <li>"IN {expression} [AS] alias [JOIN ... AS ...]"</li>
     * <li>mydomain.MyClass [AS] alias [JOIN ... AS ...]"</li>
     * </ul>
     * @return Node tree(s) for the FROM expression
     */
    private Node[] compileFromExpression()
    {
        String candidateClassName = null;
        String candidateAlias = null;
        List nodes = new ArrayList();
        do
        {
            if (p.parseStringIgnoreCase("IN"))
            {
                // IN expression
                // This will create a node of type "Node.CLASS" (candidate) and child of type "Node.NAME" (alias)
                // Any IN/JOINs will be child nodes of type "Node.OPERATOR"
                // Node(CLASS, "org.datanucleus.CandidateClass)
                // +--- Node(NAME, "c");
                // +--- Node(OPERATOR, "JOIN_INNER")
                //    +--- Node(IDENTIFIER, "p.myField")
                //    +--- Node(NAME, "f")
                // +--- Node(OPERATOR, "JOIN_INNER")
                //    +--- Node(IDENTIFIER, "f.myField2")
                //    +--- Node(NAME, "g")
                if (!p.parseChar('('))
                {
                    throw new QueryCompilerSyntaxException("Expected: '(' but got " + p.remaining(), 
                        p.getIndex(), p.getInput());
                }

                // Find what we are joining to
                String name = p.parseIdentifier();
                if (p.nextIsDot())
                {
                    p.parseChar('.');
                    name += ".";
                    name += p.parseName();
                }

                if (!p.parseChar(')'))
                {
                    throw new QueryCompilerSyntaxException("Expected: ')' but got " + p.remaining(), 
                        p.getIndex(), p.getInput());
                }
                p.parseStringIgnoreCase("AS"); // Optional
                String alias = p.parseIdentifier();

                // Create candidate class node with alias, and put at top of stack
                Node classNode = new Node(Node.CLASS, candidateClassName);
                Node classAliasNode = new Node(Node.NAME, candidateAlias);
                classNode.insertChildNode(classAliasNode);
                stack.push(classNode);

                // Translate the "IN(...) alias" into the equivalent JOIN syntax nodes
                Node joinNode = new Node(Node.OPERATOR, "JOIN_INNER");
                Node joinedToNode = new Node(Node.IDENTIFIER, name);
                joinNode.appendChildNode(joinedToNode);
                Node joinAliasNode = new Node(Node.NAME, alias);
                joinNode.appendChildNode(joinAliasNode);
                classNode.appendChildNode(joinNode);

                // Include any joins for this expression
                compileFromJoinExpression();

                nodes.add(classNode);
            }
            else
            {
                // "candidate [AS] alias"
                // This will create a node of type "Node.CLASS" and child of type "Node.NAME" (alias)
                // Any JOINs will be child nodes of type "Node.OPERATOR"
                compileExpression();
                Node id = (Node)stack.pop();
                String className = id.getNodeValue().toString();
                while (id.getChildNodes().size() > 0)
                {
                    id = id.getFirstChild();
                    className = className + "." + id.getNodeValue().toString();
                }

                String alias = p.parseIdentifier();
                if (alias != null && alias.equalsIgnoreCase("AS"))
                {
                    alias = p.parseIdentifier();
                }
                if (candidateClassName == null)
                {
                    candidateClassName = className;
                    candidateAlias = alias;
                }

                // Create candidate class node with alias, and put at top of stack
                Node classNode = new Node(Node.CLASS, className);
                Node aliasNode = new Node(Node.NAME, alias);
                classNode.insertChildNode(aliasNode);
                stack.push(classNode);

                // Include any joins for this expression
                compileFromJoinExpression();

                nodes.add(classNode);
            }
        }
        while (p.parseString(","));

        return (Node[]) nodes.toArray(new Node[nodes.size()]);
    }

    /**
     * Convenience method to process what remains of a component of the FROM clause, processing
     * the JOIN conditions. The leading part (candidate, or "IN") was processed just before.
     * This will append a child to the candidate node for each join condition encountered.
     * For example, "org.datanucleus.MyClass p INNER JOIN p.myField AS f" will translate to
     * <pre>
     * Node(CLASS, "org.datanucleus.MyClass)
     * +--- Node(NAME, "p")
     * +--- Node(OPERATOR, "JOIN_INNER")
     *    +--- Node(IDENTIFIER, "p.myField")
     *    +--- Node(NAME, "f")
     * </pre>
     * When we enter this method we expect the candidate node to be at the top of the stack, and when
     * we leave this method we leave the candidate node at the top of the stack.
     */
    private void compileFromJoinExpression()
    {
        Node candidateNode = (Node)stack.pop();
        boolean moreJoins = true;
        while (moreJoins)
        {
            // Check for JOIN syntax "[LEFT [OUTER] | INNER] JOIN ..."  (EJB3 syntax)
            boolean leftJoin = false;
            boolean innerJoin = false;
            if (p.parseStringIgnoreCase("INNER"))
            {
                innerJoin = true;
            }
            else if (p.parseStringIgnoreCase("LEFT"))
            {
                //optional and useless (for parser) outer keyword
                p.parseStringIgnoreCase("OUTER");
                leftJoin = true;
            }

            if (p.parseStringIgnoreCase("JOIN"))
            {
                // Process the join
                boolean fetch = false;
                if (p.parseStringIgnoreCase("FETCH"))
                {
                    fetch = true;
                }

                // Find what we are joining to
                String id = p.parseIdentifier();
                String name = id;
                if (p.nextIsDot())
                {
                    p.parseChar('.');
                    name += ".";
                    name += p.parseName();
                }

                // And the alias we know this joined field by
                p.parseStringIgnoreCase("AS"); // Optional
                String alias = p.parseName();

                // TODO Is the joinedNode correct in that it contains a name like "a.field"
                // whereas IDENTIFIER nodes are usually "a" with a subnode of "field"
                Node joinedNode = new Node(Node.IDENTIFIER, name);

                String joinType = "JOIN_INNER";
                if (innerJoin && fetch)
                {
                    joinType = "JOIN_INNER_FETCH";
                }
                else if (leftJoin)
                {
                    joinType = (fetch ? "JOIN_OUTER_FETCH" : "JOIN_OUTER");
                }
                Node joinNode = new Node(Node.OPERATOR, joinType);
                joinNode.appendChildNode(joinedNode);
                Node joinedAliasNode = new Node(Node.NAME, alias);
                joinNode.appendChildNode(joinedAliasNode);
                candidateNode.appendChildNode(joinNode);
            }
            else
            {
                if (innerJoin || leftJoin)
                {
                    throw new NucleusUserException("Expected JOIN after INNER/LEFT keyword at"+p.remaining());
                }
                moreJoins = false;
            }
        }
        stack.push(candidateNode);
    }

    private Node[] compileOrderExpression()
    {
        List nodes = new ArrayList();
        do
        {
            compileExpression();
            if (p.parseStringIgnoreCase("asc"))
            {
                Node expr = new Node(Node.OPERATOR, "ascending");
                stack.push(expr);
            }
            else if (p.parseStringIgnoreCase("desc"))
            {
                Node expr = new Node(Node.OPERATOR, "descending");
                stack.push(expr);
            }

            Node expr = new Node(Node.OPERATOR, "order");
            expr.insertChildNode((Node) stack.pop());
            if (!stack.empty())
            {
                expr.insertChildNode((Node) stack.pop());
            }
            nodes.add(expr);
        }
        while (p.parseString(","));
        return (Node[]) nodes.toArray(new Node[nodes.size()]);
    }

    private Node compileExpression()
    {
        compileOrExpression();
        return (Node) stack.peek();
    }

    /**
     * This method deals with the OR condition.
     * A condition specifies a combination of one or more expressions and logical (Boolean) operators and 
     * returns a value of TRUE, FALSE, or unknown
     */
    private void compileOrExpression()
    {
        compileAndExpression();

        while (p.parseStringIgnoreCase("OR"))
        {
            compileAndExpression();
            Node expr = new Node(Node.OPERATOR, "||");
            expr.insertChildNode((Node) stack.pop());
            expr.insertChildNode((Node) stack.pop());
            stack.push(expr);
        }
    }

    /**
     * This method deals with the AND condition.
     * A condition specifies a combination of one or more expressions and
     * logical (Boolean) operators and returns a value of TRUE, FALSE, or 
     * unknown
     */
    private void compileAndExpression()
    {
        compileEqualityExpression();

        while (p.parseStringIgnoreCase("AND"))
        {
            compileEqualityExpression();
            Node expr = new Node(Node.OPERATOR, "&&");
            expr.insertChildNode((Node) stack.pop());
            expr.insertChildNode((Node) stack.pop());
            stack.push(expr);
        }
    }

    // TODO Should we parse any NOT here rather than later?

    private void compileEqualityExpression()
    {
        compileRelationalExpression();

        for (;;)
        {
            if (p.parseString("="))
            {
                compileRelationalExpression();
                Node expr = new Node(Node.OPERATOR, "==");
                expr.insertChildNode((Node) stack.pop());
                expr.insertChildNode((Node) stack.pop());
                stack.push(expr);
            }
            else if (p.parseString("!=")) // Valid in JPQL ?
            {
                compileRelationalExpression();
                Node expr = new Node(Node.OPERATOR, "!=");
                expr.insertChildNode((Node) stack.pop());
                expr.insertChildNode((Node) stack.pop());
                stack.push(expr);
            }
            else if (p.parseStringIgnoreCase("NOT"))
            {
                if (p.parseStringIgnoreCase("BETWEEN"))
                {
                    // {expression} NOT BETWEEN {lower} AND {upper}
                    Node inputNode = (Node)stack.pop();
                    compileAdditiveExpression();
                    Node lowerNode = (Node)stack.pop();
                    if (p.parseStringIgnoreCase("AND"))
                    {
                        compileAdditiveExpression();
                        Node upperNode = (Node)stack.pop();

                        Node leftNode = new Node(Node.OPERATOR, "<");
                        leftNode.insertChildNode(lowerNode);
                        leftNode.insertChildNode(inputNode);

                        Node rightNode = new Node(Node.OPERATOR, ">");
                        rightNode.insertChildNode(upperNode);
                        rightNode.insertChildNode(inputNode);

                        Node betweenNode = new Node(Node.OPERATOR, "&&");
                        betweenNode.insertChildNode(leftNode);
                        betweenNode.insertChildNode(rightNode);
                        stack.push(betweenNode);
                    }
                    else
                    {
                        throw new NucleusUserException("Query has BETWEEN keyword with no AND clause");
                    }
                }
                else if (p.parseStringIgnoreCase("LIKE"))
                {
                    compileLikeExpression();
                    Node notNode = new Node(Node.OPERATOR, "!");
                    notNode.insertChildNode((Node) stack.pop());
                    stack.push(notNode);
                }
                else if (p.parseStringIgnoreCase("IN"))
                {
                    // {expression} NOT IN (expr1 [,expr2[,expr3]])
                    compileInExpression(true);
                }
                else if (p.parseStringIgnoreCase("MEMBER"))
                {
                    compileMemberExpression(true);
                }
                else
                {
                    throw new NucleusException("Unsupported query syntax NOT followed by unsupported keyword");
                }
            }
            else if (p.parseStringIgnoreCase("BETWEEN"))
            {
                // {expression} BETWEEN {lower} AND {upper}
                Node inputNode = (Node)stack.pop();
                compileAdditiveExpression();
                Node lowerNode = (Node)stack.pop();
                if (p.parseStringIgnoreCase("AND"))
                {
                    compileAdditiveExpression();
                    Node upperNode = (Node)stack.pop();
                    Node leftNode = new Node(Node.OPERATOR, ">=");
                    leftNode.insertChildNode(lowerNode);
                    leftNode.insertChildNode(inputNode);

                    Node rightNode = new Node(Node.OPERATOR, "<=");
                    rightNode.insertChildNode(upperNode);
                    rightNode.insertChildNode(inputNode);

                    Node betweenNode = new Node(Node.OPERATOR, "&&");
                    betweenNode.insertChildNode(leftNode);
                    betweenNode.insertChildNode(rightNode);
                    stack.push(betweenNode);
                }
                else
                {
                    throw new NucleusUserException("Query has BETWEEN keyword with no AND clause");
                }
            }
            else if (p.parseStringIgnoreCase("LIKE"))
            {
                // {expression} LIKE {pattern_value} [ESCAPE {escape_char}]
                compileLikeExpression();
            }
            else if (p.parseStringIgnoreCase("IN"))
            {
                // {expression} IN (expr1 [,expr2[,expr3]])
                compileInExpression(false);
            }
            else if (p.parseStringIgnoreCase("MEMBER"))
            {
                compileMemberExpression(false);
            }
            else if (p.parseStringIgnoreCase("IS"))
            {
                // {expression} IS [NOT] [NULL | EMPTY]
                Node inputNode = (Node)stack.pop();
                boolean not = false;
                if (p.parseStringIgnoreCase("NOT"))
                {
                    not = true;
                }
                if (p.parseStringIgnoreCase("NULL"))
                {
                    Node isNode = new Node(Node.OPERATOR, (not ? "!=" : "=="));
                    Node compareNode = new Node(Node.LITERAL, null);
                    isNode.insertChildNode(compareNode);
                    isNode.insertChildNode(inputNode);
                    stack.push(isNode);
                }
                else if (p.parseStringIgnoreCase("EMPTY"))
                {
                    // Convert IS EMPTY to a method call of isEmpty() on collection/map
                    Node isNode = new Node(Node.INVOKE, "isEmpty");
                    inputNode.insertChildNode(isNode);
                    if (not)
                    {
                        Node notExpr = new Node(Node.OPERATOR, "!");
                        notExpr.insertChildNode(inputNode);
                        stack.push(notExpr);
                    }
                    else
                    {
                        stack.push(inputNode);
                    }
                }
                else
                {
                    throw new NucleusException("Encountered IS " + (not ? "NOT " : " ") + 
                        " that should be followed by NULL | EMPTY but isnt");
                }
            }
            else
            {
                break;
            }
        }
    }

    /**
     * Convenience handler to compile a LIKE expression.
     * Expression is of the form "{expression} LIKE {pattern_value} [ESCAPE {escape_char}]".
     * Requires that the node at the top of the stack is the field expression node.
     * At return the like expression node will be at the top of the stack.
     */
    private void compileLikeExpression()
    {
        Node inputNode = (Node)stack.pop();
        compileAdditiveExpression();
        Node likeExprNode = (Node)stack.pop();
        if (p.parseStringIgnoreCase("ESCAPE"))
        {
            compileAdditiveExpression();
            /*Node escapeNode = (Node)*/stack.pop();
            // TODO Use the escapeNode
            Node likeNode = new Node(Node.OPERATOR, "LIKE");
            likeNode.insertChildNode(likeExprNode);
            likeNode.insertChildNode(inputNode);
            stack.push(likeNode);
        }
        else
        {
            Node likeNode = new Node(Node.OPERATOR, "LIKE");
            likeNode.insertChildNode(likeExprNode);
            likeNode.insertChildNode(inputNode);
            stack.push(likeNode);
        }
    }

    /**
     * Convenience handler to compile an IN expression.
     * Expression is of the form "{expression} IN (expr1 [,expr2 [,expr3]] | subquery)".
     * Generates a node tree like
     * <pre>({expression} == expr1) || ({expression} == expr2) || ({expression} == expr3)</pre> 
     * @param not Whether this is an expression "NOT IN"
     */
    private void compileInExpression(boolean not)
    {
        Node inputNode = (Node)stack.pop(); // The left hand side expression

        if (!p.parseChar('('))
        {
            // Subquery
            // TODO Support subqueries
            throw new NucleusException("IN subquery not yet implemented for generic JPQL compiler");
        }

        Node inNode = null;
        do
        {
            // IN ((literal|parameter) [, (literal|parameter)])
            compilePrimary();
            if (stack.peek() == null)
            {
                throw new QueryCompilerSyntaxException("Expected literal|parameter but got " + 
                    p.remaining(), p.getIndex(), p.getInput());
            }

            // Generate node for comparison with this value
            Node valueNode = (Node)stack.pop();
            Node compareNode = new Node(Node.OPERATOR, (not ? "!=" : "=="));
            compareNode.insertChildNode(inputNode);
            compareNode.insertChildNode(valueNode);

            if (inNode == null)
            {
                inNode = compareNode;
            }
            else
            {
                Node newInNode = new Node(Node.OPERATOR, (not ? "&&" : "||"));
                newInNode.insertChildNode(inNode);
                newInNode.insertChildNode(compareNode);
                inNode = newInNode;
            }
        } while (p.parseChar(','));

        if (!p.parseChar(')'))
        {
            throw new QueryCompilerSyntaxException("Expected: ')' but got " + p.remaining(), 
                p.getIndex(), p.getInput());
        }
        stack.push(inNode);
    }

    /**
     * Convenience handler to compile an MEMBER expression.
     * Expression is of the form "{expr1} MEMBER [OF] expr2".
     * @param not Whether this is an expression "NOT MEMBER"
     */
    private void compileMemberExpression(boolean not)
    {
        Node inputNode = (Node)stack.pop(); // The left hand side expression
        p.parseStringIgnoreCase("OF"); // Ignore any "OF" keyword here (optional)
        compilePrimary(); // Container node at top of stack
        Node containerNode = (Node)stack.peek();

        if (not)
        {
            Node notNode = new Node(Node.OPERATOR, "!");
            stack.pop();
            notNode.insertChildNode(containerNode);
            stack.push(notNode);
        }

        // Node (IDENTIFIER, container)
        // ---> Node (INVOKE, "contains")
        //      ---> Node(IDENTIFIER, containsNode)
        Node containsNode = new Node(Node.INVOKE, "contains");
        containsNode.insertChildNode(inputNode);
        containerNode.appendChildNode(containsNode);
    }

    private void compileRelationalExpression()
    {
        compileAdditiveExpression();

        for (;;)
        {
            if (p.parseString("<="))
            {
                compileAdditiveExpression();
                Node expr = new Node(Node.OPERATOR, "<=");
                expr.insertChildNode((Node) stack.pop());
                expr.insertChildNode((Node) stack.pop());
                stack.push(expr);
            }
            else if (p.parseString(">="))
            {
                compileAdditiveExpression();
                Node expr = new Node(Node.OPERATOR, ">=");
                expr.insertChildNode((Node) stack.pop());
                expr.insertChildNode((Node) stack.pop());
                stack.push(expr);
            }
            else if (p.parseChar('<'))
            {
                compileAdditiveExpression();
                Node expr = new Node(Node.OPERATOR, "<");
                expr.insertChildNode((Node) stack.pop());
                expr.insertChildNode((Node) stack.pop());
                stack.push(expr);
            }
            else if (p.parseChar('>'))
            {
                compileAdditiveExpression();
                Node expr = new Node(Node.OPERATOR, ">");
                expr.insertChildNode((Node) stack.pop());
                expr.insertChildNode((Node) stack.pop());
                stack.push(expr);
            }
            else if (p.parseStringIgnoreCase("instanceof"))
            {
                compileAdditiveExpression();
                Node expr = new Node(Node.OPERATOR, "instanceof");
                expr.insertChildNode((Node) stack.pop());
                expr.insertChildNode((Node) stack.pop());
                stack.push(expr);
            }
            else
            {
                break;
            }
        }
    }

    protected void compileAdditiveExpression()
    {
        compileMultiplicativeExpression();

        for (;;)
        {
            if (p.parseChar('+'))
            {
                compileMultiplicativeExpression();
                Node expr = new Node(Node.OPERATOR, "+");
                expr.insertChildNode((Node) stack.pop());
                expr.insertChildNode((Node) stack.pop());
                stack.push(expr);
            }
            else if (p.parseChar('-'))
            {
                compileMultiplicativeExpression();
                Node expr = new Node(Node.OPERATOR, "-");
                expr.insertChildNode((Node) stack.pop());
                expr.insertChildNode((Node) stack.pop());
                stack.push(expr);
            }
            else
            {
                break;
            }
        }
    }

    protected void compileMultiplicativeExpression()
    {
        compileUnaryExpression();

        for (;;)
        {
            if (p.parseChar('*'))
            {
                compileMultiplicativeExpression();
                Node expr = new Node(Node.OPERATOR, "*");
                expr.insertChildNode((Node) stack.pop());
                expr.insertChildNode((Node) stack.pop());
                stack.push(expr);
            }
            else if (p.parseChar('/'))
            {
                compileMultiplicativeExpression();
                Node expr = new Node(Node.OPERATOR, "/");
                expr.insertChildNode((Node) stack.pop());
                expr.insertChildNode((Node) stack.pop());
                stack.push(expr);
            }
            else if (p.parseChar('%'))
            {
                compileMultiplicativeExpression();
                Node expr = new Node(Node.OPERATOR, "%");
                expr.insertChildNode((Node) stack.pop());
                expr.insertChildNode((Node) stack.pop());
                stack.push(expr);
            }
            else
            {
                break;
            }
        }
    }

    protected void compileUnaryExpression()
    {
        if (p.parseString("++"))
        {
            throw new NucleusUserException("Unsupported operator '++'");
        }
        else if (p.parseString("--"))
        {
            throw new NucleusUserException("Unsupported operator '--'");
        }

        if (p.parseChar('+'))
        {
            // Just swallow + and leave remains on the stack
            compileUnaryExpression();
        }
        else if (p.parseChar('-'))
        {
            compileUnaryExpression();
            Node expr = new Node(Node.OPERATOR, "NEG");
            expr.insertChildNode((Node) stack.pop());
            stack.push(expr);
        }
        else
        {
            compileUnaryExpressionNotPlusMinus();
        }
    }

    protected void compileUnaryExpressionNotPlusMinus()
    {
        if (p.parseStringIgnoreCase("NOT"))
        {
            compileUnaryExpression();
            Node expr = new Node(Node.OPERATOR, "!");
            expr.insertChildNode((Node) stack.pop());
            stack.push(expr);
        }
        else
        {
            compilePrimary();
        }
    }

    /**
     * Compiles the primary. First look for a literal (e.g. "text"), then
     * an identifier(e.g. variable) In the next step, call a function, if
     * executing a function, on the literal or the identifier found.
     */
    protected void compilePrimary()
    {
        if (p.parseStringIgnoreCase("CURRENT_DATE"))
        {
            // Convert to a method call
            Node expr = new Node(Node.INVOKE, "CURRENT_DATE");
            stack.push(expr);
            return;
        }
        else if (p.parseStringIgnoreCase("CURRENT_TIMESTAMP"))
        {
            // Convert to a method call
            Node expr = new Node(Node.INVOKE, "CURRENT_TIMESTAMP");
            stack.push(expr);
            return;
        }
        else if (p.parseStringIgnoreCase("CURRENT_TIME"))
        {
            // Convert to a method call
            Node expr = new Node(Node.INVOKE, "CURRENT_TIME");
            stack.push(expr);
            return;
        }
        else if (p.parseStringIgnoreCase("DISTINCT"))
        {
            // Aggregates can have "count(DISTINCT field1)"
            Node distinctNode = new Node(Node.OPERATOR, "DISTINCT");
            compileIdentifier();
            Node identifierNode = (Node)stack.pop();
            distinctNode.appendChildNode(identifierNode);
            stack.push(distinctNode);
            return;
        }
        if (compileCreator())
        {
            return;
        }
        if (compileLiteral())
        {
            return;
        }
        if (compileMethod())
        {
            return;
        }

        if (p.parseChar('('))
        {
            compileExpression();
            if (!p.parseChar(')'))
            {
                throw new QueryCompilerSyntaxException("expected ')'", p.getIndex(), p.getInput());
            }
            return;
        }

        // if primary == null, literal not found...
        // We will have an identifier (variable, parameter, or field of candidate class)
        if (!compileIdentifier())
        {
            throw new QueryCompilerSyntaxException("Identifier expected", p.getIndex(), p.getInput());
        }
        int size = stack.size();

        // Generate Node tree, including chained operations
        // e.g identifier.methodX().methodY().methodZ() 
        //     -> node (IDENTIFIER) with child (INVOKE), with child (INVOKE), with child (INVOKE)
        // e.g identifier.fieldX.fieldY.fieldZ
        //     -> node (IDENTIFIER) with child (IDENTIFIER), with child (IDENTIFIER), with child (IDENTIFIER)
        while (p.parseChar('.'))
        {
            if (compileMethod())
            {
                // "a.method(...)"
            }
            else if (compileIdentifier())
            {
                // "a.field"
            }
            else
            {
                throw new QueryCompilerSyntaxException("Identifier expected", p.getIndex(), p.getInput());
            }
        }

        while (stack.size() > size)
        {
            Node top = (Node) stack.pop();
            Node peek = ((Node) stack.peek());
            peek.insertChildNode(top);
        }
    }

    /**
     * Method to parse "new a.b.c(param1[,param2], ...)" and create a Node of type CREATOR.
     * The Node at the top of the stack after this call will have any arguments defined in its "properties".
     * @return whether method syntax was found.
     */
    private boolean compileCreator()
    {
        if (p.parseStringIgnoreCase("new"))
        {
            // "new MyClass(arg1, arg2)"
            int size = stack.size();
            if (!compileMethod())
            {
                if (!compileIdentifier())
                {
                    throw new QueryCompilerSyntaxException("Identifier expected", p.getIndex(), p.getInput());
                }

                // run function on literals or identifiers e.g. "primary.runMethod(arg)"
                while (p.parseChar('.'))
                {
                    if (compileMethod())
                    {
                        // "a.method(...)"
                    }
                    else if (compileIdentifier())
                    {
                        // "a.field"
                    }
                    else
                    {
                        throw new QueryCompilerSyntaxException("Identifier expected", p.getIndex(), p.getInput());
                    }
                }
            }
            while (stack.size() - 1 > size)
            {
                Node top = (Node) stack.pop();
                Node peek = ((Node) stack.peek());
                peek.insertChildNode(top);
            }
            Node expr = (Node) stack.pop();
            Node newExpr = new Node(Node.CREATOR);
            newExpr.insertChildNode(expr);
            stack.push(newExpr);
            return true;
        }
        return false;
    }

    /**
     * Method to parse "methodName(param1[,param2], ...)" and create a Node of type INVOKE.
     * The Node at the top of the stack after this call will have any arguments defined in its "properties".
     * @return whether method syntax was found.
     */
    private boolean compileMethod()
    {
        String method = p.parseMethod();
        if (method != null)
        {
            p.skipWS();
            p.parseChar('(');

            if (method.equals("Object"))
            {
                // "Object(p)", so interpret as "p"
                compileExpression(); // identifier at top of stack
                if (!p.parseChar(')'))
                {
                    throw new QueryCompilerSyntaxException("')' expected", p.getIndex(), p.getInput());
                }
                return true;
            }
            if (method.equalsIgnoreCase("EXISTS"))
            {
                // TODO Implement EXISTS (subquery)
                throw new NucleusException("EXISTS is not yet supported for generic JPQL compilation");
            }
            else if (method.equalsIgnoreCase("ANY"))
            {
                // TODO Implement ANY (subquery)
                throw new NucleusException("ANY is not yet supported for generic JPQL compilation");
            }
            else if (method.equalsIgnoreCase("ALL"))
            {
                // TODO Implement ALL (subquery)
                throw new NucleusException("ALL is not yet supported for generic JPQL compilation");
            }
            else if (method.equalsIgnoreCase("SOME"))
            {
                // TODO Implement SOME (subquery)
                throw new NucleusException("SOME is not yet supported for generic JPQL compilation");
            }
            else if (method.equalsIgnoreCase("TRIM"))
            {
                // TRIM([[LEADING | TRAILING | BOTH] [trim_character] FROM] string_primary)
                String methodName = "TRIM";
                if (p.parseStringIgnoreCase("LEADING"))
                {
                    methodName = "TRIM_LEADING";
                }
                else if (p.parseStringIgnoreCase("TRAILING"))
                {
                    methodName = "TRIM_TRAILING";
                }
                else if (p.parseStringIgnoreCase("BOTH"))
                {
                    // Default
                }

                Node trimCharNode = null;
                Node expr = new Node(Node.INVOKE, methodName);
                compileExpression();
                Node next = (Node)stack.pop();
                if (next.getNodeType() == Node.LITERAL)
                {
                    Object litValue = next.getNodeValue();
                    if (litValue instanceof String && ((String)litValue).equals("FROM"))
                    {
                        // FROM so ignore
                        compileExpression(); // field expression that we are trimming
                        next = (Node)stack.pop();
                    }
                    else
                    {
                        // trim character first
                        trimCharNode = next;
                        if (p.parseStringIgnoreCase("FROM"))
                        {
                            // Ignore the FROM
                        }
                        compileExpression();
                        next = (Node)stack.pop();
                    }
                }
                else
                {
                    // No "trimChar" or FROM
                }

                if (!p.parseChar(')'))
                {
                    throw new QueryCompilerSyntaxException("')' expected", p.getIndex(), p.getInput());
                }
                expr.appendChildNode(next);
                if (trimCharNode != null)
                {
                    expr.appendChildNode(trimCharNode);
                }
                stack.push(expr);
                return true;
            }
            else
            {
                // Found syntax for a method, so invoke the method
                // TODO What if the method is not supported for JPQL?
                Node expr = new Node(Node.INVOKE, method);
                if (!p.parseChar(')'))
                {
                    do
                    {
                        // Argument for the method call, add as a node property
                        compileExpression();
                        expr.addProperty((Node)stack.pop());
                    }
                    while (p.parseChar(','));

                    if (!p.parseChar(')'))
                    {
                        throw new QueryCompilerSyntaxException("')' expected", p.getIndex(), p.getInput());
                    }
                }
                stack.push(expr);
                return true;
            }
        }
        return false;
    }

    /**
     * A literal is one value of any type.
     * Supported literals are of types String, Floating Point, Integer,
     * Character, Boolean and null e.g. 'J', "String", 1, 1.8, true, false, null.
     * @return The compiled literal
     */
    protected boolean compileLiteral()
    {
        Object litValue = null;

        String sLiteral;
        BigDecimal fLiteral;
        BigInteger iLiteral;
        Boolean bLiteral;

        boolean single_quote_next = p.nextIsSingleQuote();
        if ((sLiteral = p.parseStringLiteral()) != null)
        {
            // Both String and Character are allowed to use single-quotes
            // so we need to check if it was single-quoted and
            // use CharacterLiteral if length is 1.
            if (sLiteral.length() == 1 && single_quote_next)
            {
                litValue = new Character(sLiteral.charAt(0));
            }
            else
            {
                litValue = sLiteral;
            }
        }
        else if ((fLiteral = p.parseFloatingPointLiteral()) != null)
        {
            litValue = fLiteral;
        }
        else if ((iLiteral = p.parseIntegerLiteral()) != null)
        {
            litValue = new Long(iLiteral.longValue());
        }
        else if ((bLiteral = p.parseBooleanLiteralIgnoreCase()) != null)
        {
            litValue = bLiteral;
        }
        else if (p.parseNullLiteralIgnoreCase())
        {
            litValue = null;
        }
        else
        {
            return false;
        }

        stack.push(new Node(Node.LITERAL, litValue));
        return true;
    }

    int parameterPosition = 0;

    /**
     * An identifier always designates a reference to a single value.
     * A single value can be one collection, one field.
     * @return The compiled identifier
     */
    private boolean compileIdentifier()
    {
        String id = p.parseIdentifier();
        if (id == null || id.length() == 0)
        {
            return false;
        }
        char first = id.charAt(0);
        if (paramPrefixes.indexOf(first) >= 0)
        {
            // Parameter identifier, so strip off first char
            Node expr = new ParameterNode(Node.PARAMETER, id.substring(1), parameterPosition);
            parameterPosition++;
            stack.push(expr);
            return true;
        }
        else
        {
            Node expr = new Node(Node.IDENTIFIER, id);
            stack.push(expr);
            return true;
        }
    }
}