/**********************************************************************
Copyright (c) 2008 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
 **********************************************************************/
package org.datanucleus.query.compiler;

import java.lang.reflect.Field;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.exceptions.NucleusUserException;
import org.datanucleus.metadata.AbstractClassMetaData;
import org.datanucleus.metadata.AbstractMemberMetaData;
import org.datanucleus.metadata.MetaDataManager;
import org.datanucleus.query.expression.Expression;
import org.datanucleus.query.expression.ExpressionCompiler;
import org.datanucleus.query.node.Node;
import org.datanucleus.query.symbol.PropertySymbol;
import org.datanucleus.query.symbol.Symbol;
import org.datanucleus.query.symbol.SymbolResolver;
import org.datanucleus.query.symbol.SymbolTable;
import org.datanucleus.util.ClassUtils;
import org.datanucleus.util.Imports;
import org.datanucleus.util.NucleusLogger;

/**
 * Typical implementation of a compiler for a java-based query language.
 * The constructor takes in the components of the query, and the method compile() compiles it
 * returning the compiled query, for use elsewhere.
 * <p>
 * Each "Expression" is effectively a tree of Expressions. You can navigate through each expression based on
 * their type. For example, a DyadicExpression has a "left" and "right" and an operator between them.
 * The left could also be a DyadicExpression, so you would navigate to its left/right components etc etc.
 * </p>
 */
public abstract class JavaQueryCompiler implements SymbolResolver
{
    protected final MetaDataManager metaDataManager;
    protected final ClassLoaderResolver clr;

    /** Primary candidate class (if defined). */
    protected Class candidateClass;

    /** Alias for the primary candidate. Default to "this" (JDOQL) but can be set. */
    protected String candidateAlias = "this";

    protected String from;
    protected Collection candidates;

    protected String filter;
    protected String ordering;
    protected String parameters;
    protected String variables;
    protected String grouping;
    protected String having;
    protected String result;
    protected Imports imports;

    /** Compiled Symbol Table. */
    protected SymbolTable symtbl;

    /** Parser specific to the type of query being compiled. */
    protected Parser parser;

    public JavaQueryCompiler(MetaDataManager metaDataManager, ClassLoaderResolver clr, 
            String from, Class candidateClass, Collection candidates, 
            String filter, Imports imports, String ordering, String result, String grouping, String having, 
            String params, String variables)
    {
        this.metaDataManager = metaDataManager;
        this.clr = clr;

        this.from = from;
        this.candidateClass = candidateClass;
        this.candidates = candidates;

        this.filter = filter;
        this.result = result;
        this.grouping = grouping;
        this.having = having;
        this.ordering = ordering;
        this.parameters = params;
        this.variables = variables;

        this.imports = imports;
        if (imports == null)
        {
            this.imports = new Imports();
        }
    }

    /**
     * Method to compile the query.
     * @param parameters The parameter values keyed by name.
     * @return The query compilation
     */
    public abstract QueryCompilation compile(Map parameters);

    /**
     * Compile the candidates, variables and parameters.
     * @param parameters Map of parameter values keyed by their name
     */
    public void compileCandidatesParametersVariables(Map parameters)
    {
        compileCandidates();
        compileVariables();
        compileParameters();

        if (parameters != null)
        {
            // Bind the parameters values to their names
            // This is really to be performed at execute() so could be removed from here since compiling
            Iterator it = parameters.entrySet().iterator();
            while (it.hasNext())
            {
                Map.Entry entry = (Map.Entry)it.next();
                String paramName = (String)entry.getKey();
                Symbol symbol = symtbl.getSymbol(paramName);
                if (symbol == null)
                {
                    // Implicit parameter so add symbol
                    // TODO Check for invalid implicit parameter (not defined in the query) where the param
                    // input is not positional (DATANUCLEUS_PARAM_{n})
                    if (entry.getValue() == null)
                    {
                        symtbl.addSymbol(new PropertySymbol(paramName));
                    }
                    else
                    {
                        symtbl.addSymbol(new PropertySymbol(paramName, entry.getValue().getClass()));
                    }
                    symbol = symtbl.getSymbol(paramName);
                }
                else if (symbol.getValueType() == null && entry.getValue() != null)
                {
                    symbol.setValueType(entry.getValue().getClass());
                }

                symbol.setValue(entry.getValue());
            }
        }
    }

    /**
     * Method to compile the "from" clause (if present for the query language).
     */
    protected Expression[] compileFrom()
    {
        if (from == null)
        {
            return null;
        }

        Node[] node = parser.compileFrom(from);
        Expression[] expr = new Expression[node.length];
        for (int i = 0; i < node.length; i++)
        {
            String className = (String)node[i].getNodeValue();
            String classAlias = null;
            Class cls = imports.resolveClassDeclaration(className, clr, null);
            List children = node[i].getChildNodes();
            for (int j=0;j<children.size();j++)
            {
                Node child = (Node)children.get(j);
                if (child.getNodeType() == Node.NAME) // Alias - maybe should assume it is the first child
                {
                    classAlias = (String)child.getNodeValue();
                }
            }

            if (classAlias != null && symtbl.getSymbol(classAlias) == null)
            {
                // Add symbol for this candidate under its alias
                symtbl.addSymbol(new PropertySymbol(classAlias, cls));
            }

            if (i == 0)
            {
                // First expression so set up candidateClass/alias
                candidateClass = cls;
                candidateAlias = classAlias;
            }

            Iterator childIter = node[i].getChildNodes().iterator();
            while (childIter.hasNext())
            {
                // Add entries in symbol table for any joined aliases
                Node childNode = (Node)childIter.next();
                if (childNode.getNodeType() == Node.OPERATOR)
                {
                    childNode.getFirstChild(); // Joined node
                    Node aliasNode = childNode.getNextChild();
                    if (aliasNode.getNodeType() == Node.NAME)
                    {
                        symtbl.addSymbol(new PropertySymbol((String)aliasNode.getNodeValue()));
                    }
                }
            }

            ExpressionCompiler comp = new ExpressionCompiler();
            comp.setSymbolTable(symtbl);
            expr[i] = comp.compileFromExpression(node[i]);
            if (expr[i] != null)
            {
                expr[i].bind();
            }
        }
        return expr;
    }

    private void compileCandidates()
    {
        if (symtbl.getSymbol(candidateAlias) == null)
        {
            // Add candidate symbol if not already present (from "compileFrom")
            PropertySymbol symbol = new PropertySymbol(candidateAlias, candidateClass);
            symbol.setValue(candidates);
            symtbl.addSymbol(symbol);
        }
    }

    /**
     * Compile the filter and return the compiled expression.
     * @return The compiled expression
     */
    public Expression compileFilter()
    {
        if (filter != null)
        {
            // Generate the node tree for the filter
            Node node = parser.compile(filter);

            ExpressionCompiler comp = new ExpressionCompiler();
            comp.setSymbolTable(symtbl);
            Expression compiledExpression = comp.compileExpression(node);
            compiledExpression.bind();
            return compiledExpression;
        }
        return null;
    }

    public Expression[] compileResult()
    {
        if (result == null)
        {
            return null;
        }
        Node[] node = parser.compileTupple(result);
        Expression[] expr = new Expression[node.length];
        for (int i = 0; i < node.length; i++)
        {
            ExpressionCompiler comp = new ExpressionCompiler();
            comp.setSymbolTable(symtbl);
            expr[i] = comp.compileExpression(node[i]);
            // expr[i].bind();
        }
        return expr;
    }

    public Expression[] compileGrouping()
    {
        if (grouping == null)
        {
            return null;
        }
        Node[] node = parser.compileTupple(grouping);
        Expression[] expr = new Expression[node.length];
        for (int i = 0; i < node.length; i++)
        {
            ExpressionCompiler comp = new ExpressionCompiler();
            comp.setSymbolTable(symtbl);
            expr[i] = comp.compileExpression(node[i]);
            // expr[i].bind();
        }
        return expr;
    }

    public Expression compileHaving()
    {
        if (having == null)
        {
            return null;
        }
        Node node = parser.compile(having);
        ExpressionCompiler comp = new ExpressionCompiler();
        comp.setSymbolTable(symtbl);
        Expression expr = comp.compileExpression(node);
        // expr.bind();
        return expr;
    }

    private void compileVariables()
    {
        if (variables == null)
        {
            return;
        }

        Node[][] node = parser.compileVariables(variables);
        for (int i = 0; i < node.length; i++)
        {
            String varName = (String) node[i][1].getNodeValue();
            if (symtbl.getSymbol(varName) != null)
            {
                NucleusLogger.QUERY.warn(">> compileVariables param=" + varName + " but symbol already exists in table");
            }
            NucleusLogger.QUERY.debug(">> Adding variable symbol " + varName + " type=" + node[i][1].getNodeType());
            PropertySymbol sym = new PropertySymbol(varName,
                imports.resolveClassDeclaration(node[i][0].getNodeChildId(), clr, null));
            sym.setType(Symbol.VARIABLE);
            symtbl.addSymbol(sym);
        }
    }

    private void compileParameters()
    {
        if (parameters == null)
        {
            return;
        }

        Node[][] node = parser.compileParameters(parameters);
        for (int i = 0; i < node.length; i++)
        {
            String paramName = (String) node[i][1].getNodeValue();
            if (symtbl.getSymbol(paramName) != null)
            {
                NucleusLogger.QUERY.warn(">> compileParameters param=" + paramName + " but symbol already exists in table");
            }
            PropertySymbol sym = new PropertySymbol(paramName,
                imports.resolveClassDeclaration(node[i][0].getNodeChildId(), clr, null));
            sym.setType(Symbol.PARAMETER);
            symtbl.addSymbol(sym);
        }
    }

    public Expression[] compileOrdering()
    {
        if (ordering == null)
        {
            return null;
        }
        Node[] node = parser.compileOrder(ordering);
        Expression[] expr = new Expression[node.length];
        for (int i = 0; i < node.length; i++)
        {
            ExpressionCompiler comp = new ExpressionCompiler();
            comp.setSymbolTable(symtbl);
            expr[i] = comp.compileOrderExpression(node[i]);
            expr[i].bind();
        }
        return expr;
    }

    public Class getType(List tuples)
    {
        Class type = null;
        Symbol symbol = symtbl.getSymbol((String)tuples.get(0));
        if (symbol != null)
        {
            type = symbol.getValueType();
            for (int i=1; i<tuples.size(); i++)
            {
                type = getType(type, (String)tuples.get(i));
            }
        }
        else
        {
            symbol = symtbl.getSymbol(candidateAlias);
            type = symbol.getValueType();
            for (int i=0; i<tuples.size(); i++)
            {
                type = getType(type, (String)tuples.get(i));
            }
        }
        return type;
    }

    Class getType(Class cls, String fieldName)
    {
        AbstractClassMetaData acmd = metaDataManager.getMetaDataForClass(cls, clr);
        if (acmd != null)
        {
            AbstractMemberMetaData fmd = acmd.getMetaDataForMember(fieldName);
            if (fmd == null)
            {
                throw new NucleusUserException("Cannot access field "+fieldName+" on type "+cls.getName());
            }
            return fmd.getType();
        }
        else
        {
            Field field = ClassUtils.getFieldForClass(cls, fieldName);
            if (field == null)
            {
                throw new NucleusUserException("Cannot access field "+fieldName+" on type "+cls.getName());
            }
            return field.getType();
        }
    }
}