/**********************************************************************
Copyright (c) 2008 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.query.compiler;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.datanucleus.ObjectManagerFactoryImpl;
import org.datanucleus.query.expression.Expression;
import org.datanucleus.query.symbol.Symbol;
import org.datanucleus.query.symbol.SymbolTable;
import org.datanucleus.store.query.QueryInvalidParametersException;
import org.datanucleus.util.Localiser;

/**
 * Representation of the components of a compiled java "string-based" query.
 * Assumes that the query has the form
 * <pre>
 * SELECT {result} FROM {from} WHERE {filter} GROUP BY {grouping} HAVING {having} ORDER BY {order}
 * </pre>
 */
public class QueryCompilation
{
    /** Localiser for messages. */
    protected static final Localiser LOCALISER=Localiser.getInstance(
        "org.datanucleus.Localisation", ObjectManagerFactoryImpl.class.getClassLoader());

    /** Primary candidate class. */
    protected Class candidateClass;

    /** Alias for the (primary) candidate. Defaults to "this". */
    protected String candidateAlias = "this";

    /** Compiled Symbol Table. */
    protected SymbolTable symtbl;

    /** Compiled result expression. */
    protected Expression[] exprResult;

    /** Compled from expression. */
    protected Expression[] exprFrom;

    /** Compiled filter expression */
    protected Expression exprFilter = null;

    /** Compiled grouping expression. */
    protected Expression[] exprGrouping;

    /** Compiled having expression. */
    protected Expression exprHaving;

    /** Compiled ordering expression. */
    protected Expression[] exprOrdering;

    /** Compilations of any subqueries, keyed by the subquery variable name. */
    protected Map subqueryCompilations = null;

    public QueryCompilation(Class candidateCls, String candidateAlias, SymbolTable symtbl, 
            Expression[] results, Expression[] froms, Expression filter, Expression[] groupings, 
            Expression having, Expression[] orderings)
    {
        this.candidateClass = candidateCls;
        this.candidateAlias = candidateAlias;
        this.symtbl = symtbl;
        this.exprResult = results;
        this.exprFrom = froms;
        this.exprFilter = filter;
        this.exprGrouping = groupings;
        this.exprHaving = having;
        this.exprOrdering = orderings;
    }

    /**
     * Method to add the compilation for a subquery of this query.
     * @param alias Alias for the subquery (variable name)
     * @param compilation The compilation
     */
    public void addSubqueryCompilation(String alias, QueryCompilation compilation)
    {
        if (subqueryCompilations == null)
        {
            subqueryCompilations = new HashMap();
        }
        subqueryCompilations.put(alias, compilation);
    }

    /**
     * Accessor for the compilation for a subquery with the specified alias.
     * @param alias Alias of subquery
     * @return The compilation
     */
    public QueryCompilation getCompilationForSubquery(String alias)
    {
        return (subqueryCompilations != null ? (QueryCompilation)subqueryCompilations.get(alias) : null);
    }

    /**
     * Accessor for the candidate class.
     * @return Candidate class
     */
    public Class getCandidateClass()
    {
        return candidateClass;
    }

    /**
     * Accessor for the candidate alias.
     * @return Candidate alias
     */
    public String getCandidateAlias()
    {
        return candidateAlias;
    }

    /**
     * Accessor for the symbol table for the query.
     * @return Symbol table, for parameter, variable lookup.
     */
    public SymbolTable getSymbolTable()
    {
        return symtbl;
    }

    /**
     * Accessor for any result expression(s).
     * @return The results
     */
    public Expression[] getExprResult()
    {
        return exprResult;
    }

    /**
     * Accessor for any from expression(s).
     * @return The from clauses
     */
    public Expression[] getExprFrom()
    {
        return exprFrom;
    }

    /**
     * Accessor for the filter expression.
     * @return The filter
     */
    public Expression getExprFilter()
    {
        return exprFilter;
    }

    /**
     * Accessor for any grouping expression(s).
     * @return The grouping
     */
    public Expression[] getExprGrouping()
    {
        return exprGrouping;
    }

    /**
     * Accessor for any having expression.
     * @return The having clause
     */
    public Expression getExprHaving()
    {
        return exprHaving;
    }

    /**
     * Accessor for any ordering expression(s).
     * @return The ordering
     */
    public Expression[] getExprOrdering()
    {
        return exprOrdering;
    }

    /**
     * Convenience method to bind the parameter values (at execution) to the compiled query.
     * This updates the symbols in the SymbolTable to the specified values.
     * @param parameters Map of parameter values keyed by parameter name
     */
    public void bindParameterValuesToNames(Map parameters)
    {
        // Bind the parameters values to their names
        if (parameters != null)
        {
            Iterator it = parameters.entrySet().iterator();
            while (it.hasNext())
            {
                Map.Entry entry = (Map.Entry)it.next();
                String paramName = (String)entry.getKey();
                Symbol symbol = symtbl.getSymbol(paramName);
                if (symbol != null)
                {
                    // TODO Type checking ?
                    symbol.setValue(entry.getValue());
                }
                else
                {
                    throw new QueryInvalidParametersException(LOCALISER.msg("021116", paramName));
                }
            }
        }
    }
}