/**********************************************************************
Copyright (c) 2008 Erik Bengtson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
2008 Andy Jefferson - support for all methods
2008 Andy Jefferson - support for "+", "-", "/", "%", ">", "<", ">=", "<=", "instanceof", !, ~
2008 Andy Jefferson - support for implicit parameters
2008 Andy Jefferson - support for chained PrimaryExpression/InvokeExpressions
    ...
**********************************************************************/
package org.datanucleus.query.evaluator.memory;

import java.lang.reflect.Field;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Stack;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.ObjectManagerFactoryImpl;
import org.datanucleus.exceptions.ClassNotResolvedException;
import org.datanucleus.exceptions.NucleusException;
import org.datanucleus.exceptions.NucleusUserException;
import org.datanucleus.query.QueryUtils;
import org.datanucleus.query.evaluator.AbstractExpressionEvaluator;
import org.datanucleus.query.expression.CreatorExpression;
import org.datanucleus.query.expression.Expression;
import org.datanucleus.query.expression.InvokeExpression;
import org.datanucleus.query.expression.Literal;
import org.datanucleus.query.expression.ParameterExpression;
import org.datanucleus.query.expression.PrimaryExpression;
import org.datanucleus.query.expression.Expression.Operator;
import org.datanucleus.query.symbol.Symbol;
import org.datanucleus.query.symbol.SymbolTable;
import org.datanucleus.store.query.QueryManager;
import org.datanucleus.util.Imports;
import org.datanucleus.util.Localiser;

/**
 * Class providing evaluation of java "string-based" queries in-memory.
 */
public class InMemoryExpressionEvaluator extends AbstractExpressionEvaluator
{
    /** Localisation utility for output messages */
    protected static final Localiser LOCALISER = Localiser.getInstance(
        "org.datanucleus.Localisation", ObjectManagerFactoryImpl.class.getClassLoader());

    Stack stack = new Stack();

    SymbolTable symtbl;

    Imports imports;

    ClassLoaderResolver clr;

    QueryManager queryMgr;

    /** Alias name for the candidate. */
    final String candidateAlias;

    /**
     * Constructor for an in-memory evaluator.
     * @param queryMgr Manager for queries
     * @param symtbl Symbol table
     * @param imports Any imports
     * @param clr ClassLoader resolver 
     * @param candidateAlias Alias for the candidate class. With JDOQL this is usually "this".
     */
    public InMemoryExpressionEvaluator(QueryManager queryMgr, SymbolTable symtbl, Imports imports, 
            ClassLoaderResolver clr, String candidateAlias)
    {
        this.queryMgr = queryMgr;
        this.symtbl = symtbl;
        this.imports = imports;
        this.clr = clr;
        this.candidateAlias = candidateAlias;
    }

    public SymbolTable getSymbolTable()
    {
        return symtbl;
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.evaluator.AbstractExpressionEvaluator#processAndExpression(org.datanucleus.query.expression.Expression)
     */
    protected Object processAndExpression(Expression expr)
    {
        Object right = stack.pop();
        Object left = stack.pop();
        stack.push((left == Boolean.TRUE && right == Boolean.TRUE) ? Boolean.TRUE : Boolean.FALSE);
        return stack.peek();
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.evaluator.AbstractExpressionEvaluator#processEqExpression(org.datanucleus.query.expression.Expression)
     */
    protected Object processEqExpression(Expression expr)
    {
        Object right = stack.pop();
        Object left = stack.pop();
        Boolean result = compareValues(left, right, expr.getOperator()) ? Boolean.TRUE : Boolean.FALSE;
        stack.push(result);
        return stack.peek();
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.evaluator.AbstractExpressionEvaluator#processLikeExpression(org.datanucleus.query.expression.Expression)
     */
    protected Object processLikeExpression(Expression expr)
    {
        Object right = stack.pop();
        Object left = stack.pop();
        if (!(left instanceof String))
        {
            throw new NucleusUserException(
                "LIKE expression can only be used on a String expression, but found on " + 
                left.getClass().getName());
        }
        if (right instanceof String)
        {
            // Just use String.matches(String)
            Boolean result = ((String)left).matches((String)right) ? Boolean.TRUE : Boolean.FALSE;
            stack.push(result);
            return result;
        }
        else
        {
            throw new NucleusUserException(
                "Dont currently support expression on right of LIKE to be other than String but was " + 
                right.getClass().getName());
        }
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.evaluator.AbstractExpressionEvaluator#processNoteqExpression(org.datanucleus.query.expression.Expression)
     */
    protected Object processNoteqExpression(Expression expr)
    {
        Object right = stack.pop();
        Object left = stack.pop();
        Boolean result = compareValues(left, right, expr.getOperator()) ? Boolean.TRUE : Boolean.FALSE;
        stack.push(result);
        return stack.peek();
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.evaluator.AbstractExpressionEvaluator#processOrExpression(org.datanucleus.query.expression.Expression)
     */
    protected Object processOrExpression(Expression expr)
    {
        Object right = stack.pop();
        Object left = stack.pop();
        stack.push((left == Boolean.TRUE || right == Boolean.TRUE) ? Boolean.TRUE : Boolean.FALSE);
        return stack.peek();
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.evaluator.AbstractExpressionEvaluator#processGteqExpression(org.datanucleus.query.expression.Expression)
     */
    protected Object processGteqExpression(Expression expr)
    {
        Object right = stack.pop();
        Object left = stack.pop();
        Boolean result = compareValues(left, right, expr.getOperator()) ? Boolean.TRUE : Boolean.FALSE;
        stack.push(result);
        return stack.peek();
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.evaluator.AbstractExpressionEvaluator#processGtExpression(org.datanucleus.query.expression.Expression)
     */
    protected Object processGtExpression(Expression expr)
    {
        Object right = stack.pop();
        Object left = stack.pop();
        Boolean result = compareValues(left, right, expr.getOperator()) ? Boolean.TRUE : Boolean.FALSE;
        stack.push(result);
        return stack.peek();
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.evaluator.AbstractExpressionEvaluator#processIsExpression(org.datanucleus.query.expression.Expression)
     */
    protected Object processIsExpression(Expression expr)
    {
        // field instanceof className
        Object right = stack.pop();
        Object left = stack.pop();
        if (!(right instanceof Class))
        {
            throw new NucleusException("Attempt to invoke instanceof with argument of type " + 
                right.getClass().getName() + " has to be Class");
        }
        try
        {
            Boolean result = ((Class)right).isAssignableFrom(left.getClass()) ? Boolean.TRUE : Boolean.FALSE;
            stack.push(result);
            return result;
        }
        catch (ClassNotResolvedException cnre)
        {
            throw new NucleusException("Attempt to invoke instanceof with " + 
                right + " yet class was not found!");
        }
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.evaluator.AbstractExpressionEvaluator#processLteqExpression(org.datanucleus.query.expression.Expression)
     */
    protected Object processLteqExpression(Expression expr)
    {
        Object right = stack.pop();
        Object left = stack.pop();
        Boolean result = compareValues(left, right, expr.getOperator()) ? Boolean.TRUE : Boolean.FALSE;
        stack.push(result);
        return stack.peek();
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.evaluator.AbstractExpressionEvaluator#processLtExpression(org.datanucleus.query.expression.Expression)
     */
    protected Object processLtExpression(Expression expr)
    {
        Object right = stack.pop();
        Object left = stack.pop();
        Boolean result = compareValues(left, right, expr.getOperator()) ? Boolean.TRUE : Boolean.FALSE;
        stack.push(result);
        return stack.peek();
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.evaluator.AbstractExpressionEvaluator#processAddExpression(org.datanucleus.query.expression.Expression)
     */
    protected Object processAddExpression(Expression expr)
    {
        Object right = stack.pop();
        Object left = stack.pop();
        Object value = new BigDecimal(left.toString()).add(new BigDecimal(right.toString()));
        stack.push(value);
        return stack.peek();
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.evaluator.AbstractExpressionEvaluator#processSubExpression(org.datanucleus.query.expression.Expression)
     */
    protected Object processSubExpression(Expression expr)
    {
        Object right = stack.pop();
        Object left = stack.pop();
        Object value = new BigDecimal(left.toString()).subtract(new BigDecimal(right.toString()));
        stack.push(value);
        return stack.peek();
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.evaluator.AbstractExpressionEvaluator#processDivExpression(org.datanucleus.query.expression.Expression)
     */
    protected Object processDivExpression(Expression expr)
    {
        Object right = stack.pop();
        Object left = stack.pop();
        double firstValue = new BigDecimal(left.toString()).doubleValue();
        double secondValue = new BigDecimal(right.toString()).doubleValue();
        BigDecimal value = new BigDecimal(firstValue/secondValue);
        stack.push(value);
        return stack.peek();
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.evaluator.AbstractExpressionEvaluator#processModExpression(org.datanucleus.query.expression.Expression)
     */
    protected Object processModExpression(Expression expr)
    {
        Object right = stack.pop();
        Object left = stack.pop();
        BigDecimal firstValue = new BigDecimal(left.toString());
        BigDecimal divisor = new BigDecimal(right.toString());
        Object value = firstValue.subtract(firstValue.divideToIntegralValue(divisor).multiply(divisor));
        stack.push(value);
        return stack.peek();
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.evaluator.AbstractExpressionEvaluator#processMulExpression(org.datanucleus.query.expression.Expression)
     */
    protected Object processMulExpression(Expression expr)
    {
        Object right = stack.pop();
        Object left = stack.pop();
        Object value = new BigDecimal(left.toString()).multiply(new BigDecimal(right.toString()));
        stack.push(value);
        return stack.peek();
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.evaluator.AbstractExpressionEvaluator#processNegExpression(org.datanucleus.query.expression.Expression)
     */
    protected Object processNegExpression(Expression expr)
    {
        Number val = null;
        if (expr.getLeft() instanceof PrimaryExpression)
        {
            val = (Number)getValueForPrimaryExpression((PrimaryExpression)expr.getLeft());
        }
        else if (expr.getLeft() instanceof ParameterExpression)
        {
            val = (Number)QueryUtils.getValueForParameterExpression(symtbl, (ParameterExpression)expr.getLeft());
        }
        // Other types?
        if (val instanceof Integer)
        {
            stack.push(new Integer(-val.intValue()));
            return stack.peek();
        }
        else if (val instanceof Long)
        {
            stack.push(new Long(-val.longValue()));
            return stack.peek();
        }
        else if (val instanceof Short)
        {
            stack.push(new Short((short)-val.shortValue()));
            return stack.peek();
        }
        else if (val instanceof BigInteger)
        {
            stack.push(new BigInteger("" + -val.longValue()));
            return stack.peek();
        }
        else if (val instanceof Double)
        {
            stack.push(new Double(-val.doubleValue()));
            return stack.peek();
        }
        else if (val instanceof Float)
        {
            stack.push(new Float(-val.floatValue()));
            return stack.peek();
        }
        else if (val instanceof BigDecimal)
        {
            stack.push(new BigDecimal(-val.doubleValue()));
            return stack.peek();
        }
        else
        {
            throw new NucleusException("Attempt to negate value of type " + val + " not supported");
        }
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.evaluator.AbstractExpressionEvaluator#processComExpression(org.datanucleus.query.expression.Expression)
     */
    protected Object processComExpression(Expression expr)
    {
        // Bitwise complement - only for integer values
        PrimaryExpression primExpr = (PrimaryExpression)expr.getLeft();
        Object primVal = getValueForPrimaryExpression(primExpr);
        int val = -1;
        if (primVal instanceof Number)
        {
            val = ((Number)primVal).intValue();
        }
        Integer result = new Integer(~val);
        stack.push(result);
        return stack.peek();
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.evaluator.AbstractExpressionEvaluator#processNotExpression(org.datanucleus.query.expression.Expression)
     */
    protected Object processNotExpression(Expression expr)
    {
        // Logical complement - only for boolean values
        Boolean left = (Boolean)stack.pop();
        Boolean result = (left.booleanValue() ? Boolean.FALSE : Boolean.TRUE);
        stack.push(result);
        return stack.peek();
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.evaluator.AbstractExpressionEvaluator#processCreatorExpression(org.datanucleus.query.expression.CreatorExpression)
     */
    protected Object processCreatorExpression(CreatorExpression expr)
    {
        List params = new ArrayList();
        for (int i = 0; i < expr.getParameters().size(); i++)
        {
            params.add(((Expression) expr.getParameters().get(i)).evaluate(this));
        }
        Class cls = imports.resolveClassDeclaration(expr.getId(), clr, null);
        Object value = QueryUtils.createResultObjectUsingArgumentedConstructor(cls, params.toArray());
        stack.push(value);
        // TODO What about CreateExpression.InvokeExpression or CreateExpression.PrimaryExpression ?
        return value;
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.evaluator.AbstractExpressionEvaluator#processInvokeExpression(org.datanucleus.query.expression.InvokeExpression)
     */
    protected Object processInvokeExpression(InvokeExpression expr)
    {
        // Process expressions like :-
        // a). aggregates : count(...), avg(...), sum(...), min(...), max(...)
        // b). methods/functions : FUNCTION(...), field.method(...)
        Object result = getValueForInvokeExpression(expr);
        stack.push(result);
        return result;
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.evaluator.AbstractExpressionEvaluator#processLiteral(org.datanucleus.query.expression.Literal)
     */
    protected Object processLiteral(Literal expr)
    {
        Object value = expr.getLiteral();
        stack.push(value);
        return value;
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.evaluator.AbstractExpressionEvaluator#processParameterExpression(org.datanucleus.query.expression.ParameterExpression)
     */
    protected Object processParameterExpression(ParameterExpression expr)
    {
        Object value = QueryUtils.getValueForParameterExpression(symtbl, expr);
        stack.push(value);
        return value;
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.evaluator.AbstractExpressionEvaluator#processPrimaryExpression(org.datanucleus.query.expression.PrimaryExpression)
     */
    protected Object processPrimaryExpression(PrimaryExpression expr)
    {
        Symbol primSym = symtbl.getSymbol(expr.getId());
        if (primSym != null)
        {
            // Explicit Parameter or Variable
            Object value = primSym.getValue();
            stack.push(value);
            return value;
        }
        else
        {
            // Field
            Object value = getValueForPrimaryExpression(expr);
            stack.push(value);
            return value;
        }
    }

    /**
     * Method to evaluate an InvokeExpression.
     * Will navigate along chained invocations, evaluating the first one, then the second one etc
     * until it gets the value for the passed in expression.
     * @param invokeExpr The InvokeExpression
     * @return The value
     */
    public Object getValueForInvokeExpression(InvokeExpression invokeExpr)
    {
        String method = invokeExpr.getOperation();
        if (invokeExpr.getLeft() == null)
        {
            // Static function
            if (method.equals("count"))
            {
                Symbol symbol = symtbl.getSymbol(Symbol.RESULTS_SET);
                Collection coll = (Collection)symbol.getValue();
                SetExpression setexpr = new SetExpression(coll, candidateAlias);
                Expression paramExpr = (Expression)invokeExpr.getArguments().get(0);
                if (paramExpr.getOperator() == Expression.OP_DISTINCT)
                {
                    Collection processable = new HashSet(coll); // No dups in HashSet
                    coll = processable;
                }
                return setexpr.count(paramExpr, this);
            }
            else if (method.equals("sum"))
            {
                Symbol symbol = symtbl.getSymbol(Symbol.RESULTS_SET);
                Collection coll = (Collection)symbol.getValue();
                SetExpression setexpr = new SetExpression(coll, candidateAlias);
                Expression paramExpr = (Expression)invokeExpr.getArguments().get(0);
                if (paramExpr.getOperator() == Expression.OP_DISTINCT)
                {
                    Collection processable = new HashSet(coll); // No dups in HashSet
                    coll = processable;
                }
                return setexpr.sum(paramExpr, this, symtbl);
            }
            else if (method.equals("avg"))
            {
                Symbol symbol = symtbl.getSymbol(Symbol.RESULTS_SET);
                Collection coll = (Collection)symbol.getValue();
                SetExpression setexpr = new SetExpression(coll, candidateAlias);
                Expression paramExpr = (Expression)invokeExpr.getArguments().get(0);
                if (paramExpr.getOperator() == Expression.OP_DISTINCT)
                {
                    Collection processable = new HashSet(coll); // No dups in HashSet
                    coll = processable;
                }
                return setexpr.avg(paramExpr, this, symtbl);
            }
            else if (method.equals("min"))
            {
                Symbol symbol = symtbl.getSymbol(Symbol.RESULTS_SET);
                Collection coll = (Collection)symbol.getValue();
                SetExpression setexpr = new SetExpression(coll, candidateAlias);
                Expression paramExpr = (Expression)invokeExpr.getArguments().get(0);
                if (paramExpr.getOperator() == Expression.OP_DISTINCT)
                {
                    Collection processable = new HashSet(coll); // No dups in HashSet
                    coll = processable;
                }
                return setexpr.min(paramExpr, this, symtbl);
            }
            else if (method.equals("max"))
            {
                Symbol symbol = symtbl.getSymbol(Symbol.RESULTS_SET);
                Collection coll = (Collection)symbol.getValue();
                SetExpression setexpr = new SetExpression(coll, candidateAlias);
                Expression paramExpr = (Expression)invokeExpr.getArguments().get(0);
                if (paramExpr.getOperator() == Expression.OP_DISTINCT)
                {
                    Collection processable = new HashSet(coll); // No dups in HashSet
                    coll = processable;
                }
                return setexpr.max(paramExpr, this, symtbl);
            }
            else
            {
                // Try to find a supported static method with this name
                InvocationEvaluator methodEval = queryMgr.getInMemoryEvaluatorForMethod(null, method);
                if (methodEval != null)
                {
                    return methodEval.evaluate(invokeExpr, null, this);
                }
                else
                {
                    throw new NucleusException("Query contains call to static method " + method + 
                    " yet no support is available for in-memory evaluation of this");
                }
            }
        }
        else if (invokeExpr.getLeft() instanceof PrimaryExpression)
        {
            // {primaryExpr}.method(...)
            Object invokedValue = getValueForPrimaryExpression((PrimaryExpression)invokeExpr.getLeft());

            // Invoke method on this object
            InvocationEvaluator methodEval = queryMgr.getInMemoryEvaluatorForMethod(invokedValue.getClass(), method);
            if (methodEval != null)
            {
                return methodEval.evaluate(invokeExpr, invokedValue, this);
            }
            else
            {
                throw new NucleusException("Query contains call to method " + 
                    invokedValue.getClass().getName() + "." + method + " yet no support is available for this");
            }
        }
        else if (invokeExpr.getLeft() instanceof InvokeExpression)
        {
            // {invokeExpr}.method(...)
            Object invokedValue = getValueForInvokeExpression((InvokeExpression)invokeExpr.getLeft());

            // Invoke method on this object
            InvocationEvaluator methodEval = queryMgr.getInMemoryEvaluatorForMethod(invokedValue.getClass(), method);
            if (methodEval != null)
            {
                return methodEval.evaluate(invokeExpr, invokedValue, this);
            }
            else
            {
                throw new NucleusException("Query contains call to method " + 
                    invokedValue.getClass().getName() + "." + method + " yet no support is available for this");
            }
        }
        else
        {
            throw new NucleusException("No support is available for in-memory evaluation of methods invoked" +
                " on expressions of type " + invokeExpr.getLeft().getClass().getName());
        }
    }

    /**
     * Convenience method to get an int value from the supplied literal.
     * Returns a value if it is convertible into an int.
     * @param lit The literal
     * @return The int value
     * @throws NucleusException if impossible to convert into an int
     */
    public int getIntegerForLiteral(Literal lit)
    {
        Object val = lit.getLiteral();
        if (val instanceof BigDecimal)
        {
            return ((BigDecimal)val).intValue();
        }
        else if (val instanceof BigInteger)
        {
            return ((BigInteger)val).intValue();
        }
        else if (val instanceof Long)
        {
            return ((Long)val).intValue();
        }
        else if (val instanceof Integer)
        {
            return ((Integer)val).intValue();
        }
        else if (val instanceof Short)
        {
            return ((Short)val).intValue();
        }
        throw new NucleusException("Attempt to convert literal with value " + val + " (" + 
            val.getClass().getName() + ") into an int failed");
    }

    /**
     * Convenience method to get the value for a PrimaryExpression.
     * @param primExpr Expression
     * @return The value in the object for this expression
     */
    public Object getValueForPrimaryExpression(PrimaryExpression primExpr)
    {
        Symbol symbol = symtbl.getSymbol((String)primExpr.getTuples().get(0));
        if (symbol == null)
        {
            // no symbol found, so defaults to candidate
            symbol = symtbl.getSymbol(candidateAlias);
        }
        Object value = symbol.getValue();
        for (int i = 0; i < primExpr.getTuples().size(); i++)
        {
            String fieldName = (String)primExpr.getTuples().get(i);
            if (!fieldName.equals(candidateAlias))
            {
                value = getFieldValue(value, fieldName);
            }
        }
        return value;
    }

    /**
     * Convenience method to compare two values against the specified operator.
     * Returns true if "left {operator} right" is true. The operator can be <, >, <=, <=, ==, !=.
     * @param left Left object
     * @param right Right object
     * @param op Operator
     * @return Whether the expression is true
     */
    private boolean compareValues(Object left, Object right, Operator op)
    {
        if (left == null || right == null)
        {
            // Null comparisons - not all operations are valid (e.g "5.0 > null")
            if (op == Expression.OP_GT)
            {
                throw new NucleusException("Impossible to evaluate greater_than expression between " + left + 
                    " and " + right + " due to presence of null!");
            }
            else if (op == Expression.OP_LT)
            {
                throw new NucleusException("Impossible to evaluate less_than expression between " + left + 
                    " and " + right + " due to presence of null!");
            }
            else if (op == Expression.OP_GTEQ)
            {
                if (left == right)
                {
                    return true;
                }
                throw new NucleusException("Impossible to evaluate greater_equals expression between " + left + 
                    " and " + right + " due to presence of null!");
            }
            else if (op == Expression.OP_LTEQ)
            {
                if (left == right)
                {
                    return true;
                }
                throw new NucleusException("Impossible to evaluate less_equals expression between " + left + 
                    " and " + right + " due to presence of null!");
            }
            else if (op == Expression.OP_EQ)
            {
                return left == right;
            }
            else if (op == Expression.OP_NOTEQ)
            {
                return left != right;
            }
        }
        else if (left instanceof Float || left instanceof Double || left instanceof BigDecimal ||
            right instanceof Float || right instanceof Double || right instanceof BigDecimal)
        {
            // One of the two numbers is floating point based so compare using Double
            // NOTE : Assumes double is the largest precision required
            Double leftVal = null;
            Double rightVal = null;
            if (left instanceof BigDecimal)
            {
                leftVal = new Double(((BigDecimal)left).doubleValue());
            }
            else if (left instanceof Double)
            {
                leftVal = (Double)left;
            }
            else if (left instanceof Float)
            {
                leftVal = new Double(((Float)left).doubleValue());
            }
            else if (left instanceof BigInteger)
            {
                leftVal = new Double(((BigInteger)left).doubleValue());
            }
            else if (left instanceof Long)
            {
                leftVal = new Double(((Long)left).doubleValue());
            }
            else if (left instanceof Integer)
            {
                leftVal = new Double(((Integer)left).doubleValue());
            }
            else if (left instanceof Short)
            {
                leftVal = new Double(((Short)left).doubleValue());
            }
            if (right instanceof BigDecimal)
            {
                rightVal = new Double(((BigDecimal)right).doubleValue());
            }
            else if (right instanceof Double)
            {
                rightVal = (Double)right;
            }
            else if (right instanceof Float)
            {
                rightVal = new Double(((Float)right).doubleValue());
            }
            else if (right instanceof BigInteger)
            {
                rightVal = new Double(((BigInteger)right).doubleValue());
            }
            else if (right instanceof Long)
            {
                rightVal = new Double(((Long)right).doubleValue());
            }
            else if (right instanceof Integer)
            {
                rightVal = new Double(((Integer)right).doubleValue());
            }
            else if (right instanceof Short)
            {
                rightVal = new Double(((Short)right).doubleValue());
            }

            if (leftVal == null || rightVal == null)
            {
                throw new NucleusException("Attempt to evaluate relational expression between" + 
                    "\"" + left + "\" (type=" + left.getClass().getName() + ") and" +
                    "\"" + right + "\" (type=" + right.getClass().getName() + ") not possible due to types");
            }

            int comparison = leftVal.compareTo(rightVal);
            if (op == Expression.OP_GT)
            {
                return comparison > 0 ? true : false;
            }
            else if (op == Expression.OP_LT)
            {
                return comparison < 0 ? true : false;
            }
            else if (op == Expression.OP_GTEQ)
            {
                return comparison >= 0 ? true : false;
            }
            else if (op == Expression.OP_LTEQ)
            {
                return comparison <= 0 ? true : false;
            }
            else if (op == Expression.OP_EQ)
            {
                return comparison == 0 ? true : false;
            }
            else if (op == Expression.OP_NOTEQ)
            {
                return comparison != 0 ? true : false;
            }
        }
        else if (left instanceof Short || left instanceof Integer || left instanceof Long || left instanceof BigInteger ||
            right instanceof Short || right instanceof Integer || right instanceof Long || right instanceof BigInteger)
        {
            // Not floating point based and (at least) one of numbers is integral based so compare using long
            // NOTE : Assumes long is the largest precision required
            long leftVal = Long.MAX_VALUE;
            long rightVal = Long.MAX_VALUE;
            if (left instanceof BigInteger)
            {
                leftVal = ((BigInteger)left).longValue();
            }
            else if (left instanceof Long)
            {
                leftVal = ((Long)left).longValue();
            }
            else if (left instanceof Integer)
            {
                leftVal = ((Integer)left).longValue();
            }
            else if (left instanceof Short)
            {
                leftVal = ((Short)left).longValue();
            }
            else if (left instanceof BigDecimal)
            {
                leftVal = ((BigDecimal)left).longValue();
            }
            else if (left instanceof Double)
            {
                leftVal = ((Double)left).longValue();
            }
            else if (left instanceof Float)
            {
                leftVal = ((Float)left).longValue();
            }
            if (right instanceof BigInteger)
            {
                rightVal = ((BigInteger)right).longValue();
            }
            else if (right instanceof Long)
            {
                rightVal = ((Long)right).longValue();
            }
            else if (right instanceof Integer)
            {
                rightVal = ((Integer)right).longValue();
            }
            else if (right instanceof Short)
            {
                rightVal = ((Short)right).longValue();
            }
            else if (right instanceof BigDecimal)
            {
                rightVal = ((BigDecimal)right).longValue();
            }
            else if (right instanceof Double)
            {
                rightVal = ((Double)right).longValue();
            }
            else if (right instanceof Float)
            {
                rightVal = ((Float)right).longValue();
            }

            if (leftVal == Long.MAX_VALUE || rightVal == Long.MAX_VALUE)
            {
                throw new NucleusException("Attempt to evaluate relational expression between" + 
                    "\"" + left + "\" (type=" + left.getClass().getName() + ") and" +
                    "\"" + right + "\" (type=" + right.getClass().getName() + ") not possible due to types");
            }

            if (op == Expression.OP_GT)
            {
                return leftVal > rightVal ? true : false;
            }
            else if (op == Expression.OP_LT)
            {
                return leftVal < rightVal ? true : false;
            }
            else if (op == Expression.OP_GTEQ)
            {
                return leftVal >= rightVal ? true : false;
            }
            else if (op == Expression.OP_LTEQ)
            {
                return leftVal <= rightVal ? true : false;
            }
            else if (op == Expression.OP_EQ)
            {
                return leftVal == rightVal ? true : false;
            }
            else if (op == Expression.OP_NOTEQ)
            {
                return leftVal != rightVal ? true : false;
            }
        }
        else
        {
            if (op == Expression.OP_EQ)
            {
                // Use object comparison
                if (left == null)
                {
                    return right == null;
                }
                else
                {
                    return left.equals(right);
                }
            }
            else if (op == Expression.OP_NOTEQ)
            {
                // Use object comparison
                if (left == null)
                {
                    return right != null;
                }
                else
                {
                    return !left.equals(right);
                }
            }
            else
            {
                // Can't do >, <, >=, <= with non-numeric. Maybe allow Character in future?
                throw new NucleusException("Attempt to evaluate relational expression between" + 
                    "\"" + left + "\" (type=" + left.getClass().getName() + ") and" +
                    "\"" + right + "\" (type=" + right.getClass().getName() + ") not possible due to types");
            }
        }

        throw new NucleusException("Attempt to evaluate relational expression between " + left + 
            " and " + right + " with operation = " + op + " impossible to perform");
    }

    /**
     * Helper method to retrieve the java.lang.reflect.Field
     * @param clazz the Class instance of the declaring class or interface
     * @param fieldName the field name
     * @return The field
     */
    private Field getDeclaredFieldPrivileged(final Class clazz, final String fieldName)
    {
        if ((clazz == null) || (fieldName == null))
        {
            return null;
        }

        return (Field) AccessController.doPrivileged(
            new PrivilegedAction()
            {
                public Object run ()
                {
                    Class seekingClass = clazz;
                    do
                    {
                        try
                        {
                            return seekingClass.getDeclaredField(fieldName);
                        }
                        catch (SecurityException ex)
                        {
                            throw new NucleusException("CannotGetDeclaredField",ex).setFatal();
                        }
                        catch (NoSuchFieldException ex)
                        {
                            // do nothing, we will return null later if no 
                            // field is found in this class or superclasses
                        }
                        catch (LinkageError ex)
                        {
                            throw new NucleusException("ClassLoadingError",ex).setFatal();
                        }
                        seekingClass = seekingClass.getSuperclass();
                    } while(seekingClass != null);

                    //no field found
                    return null;
                }
            });
    }

    private Object getFieldValue(Object object, String fieldName)
    {
        Object fieldValue;
        if (object == null)
        {
            return null;
        }
        final Field field = getDeclaredFieldPrivileged(object.getClass(), fieldName);
        if (field == null)
        {
            throw new NucleusUserException("Cannot access field: " + fieldName + " in type " + object.getClass());
        }

        try
        {
            // if the field is not accessible, try to set the accessible flag.
            if (!field.isAccessible())
            {
                try
                {
                    AccessController.doPrivileged(
                        new PrivilegedAction()
                        {
                            public Object run()
                            {
                                field.setAccessible(true);
                                return null;
                            }
                        });
                }
                catch (SecurityException ex)
                {
                    throw new NucleusException("Cannot access field: " + fieldName,ex).setFatal();
                }
            }
            fieldValue = field.get(object);
        }
        catch (IllegalArgumentException e2)
        {
            throw new NucleusUserException("Cannot access field: " + fieldName,e2);
        }
        catch (IllegalAccessException e2)
        {
            throw new NucleusUserException("Cannot access field: " + fieldName,e2);
        }
        return fieldValue;
    }
}