/**********************************************************************
Copyright (c) 2008 Erik Bengtson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
2008 Andy Jefferson - javadocs, toString()
    ...
**********************************************************************/
package org.datanucleus.query.expression;

import java.util.List;

import org.datanucleus.query.symbol.PropertySymbol;
import org.datanucleus.query.symbol.Symbol;
import org.datanucleus.query.symbol.SymbolTable;
import org.datanucleus.util.StringUtils;

/**
 * Expression representing something like "new X.Y.Z([param1[,param2[,param3]]])".
 */
public class CreatorExpression extends Expression
{
    SymbolTable symtbl;

    /** Components of the class name being created e.g ["org", "datanucleus", "MyClass"]. */
    List tuples;

    /** Parameters for the creation call. */
    List parameters;

    public CreatorExpression(SymbolTable symtbl, List tuples, List parameters)
    {
        this.symtbl = symtbl;
        this.tuples = tuples;
        this.parameters = parameters;
    }
    
    public Object evaluate(ExpressionEvaluator eval)
    {
        return eval.evaluate(this);
    }

    /**
     * Accessor for the class name of the object being created.
     * @return Name of the class
     */
    public String getId()
    {
        StringBuffer id = new StringBuffer();
        for (int i = 0; i < tuples.size(); i++)
        {
            if (id.length() > 0)
            {
                id.append('.');
            }
            id.append((String) tuples.get(i));
        }
        return id.toString();
    }

    /**
     * Accessor for the parameters to use in the creation call.
     * @return Parameter list
     */
    public List getParameters()
    {
        return parameters;
    }

    public List getTuples()
    {
        return tuples;
    }

    public Symbol bind()
    {
        if (symtbl.hasSymbol(getId()))
        {
            symbol = symtbl.getSymbol(getId());
        }
        else
        {
            symbol = new PropertySymbol(getId(), symtbl.getType(tuples));
        }
        return symbol;
    }

    public String toString()
    {
        return "CreatorExpression: " + getId() + "(" + StringUtils.collectionToString(parameters) + ")";
    }
}