/**********************************************************************
Copyright (c) 2008 Erik Bengtson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
2008 Andy Jefferson - support for swapping PrimaryExpression to class Literal
    ...
**********************************************************************/
package org.datanucleus.query.expression;

import org.datanucleus.query.symbol.Symbol;

/**
 * Expression between two other expressions and an operation.
 * For example, "this.myField < myValue" will become
 * left = PrimaryExpression, right = Literal, op = Expression.OP_LT.
 * A special case is where we have an expression like "!(condition)". In this case we become
 * left = expression, right = null, op = Expression.OP_NOT.
 */
public class DyadicExpression extends Expression
{
    /**
     * Perform a function <code>op</code> on <code>operand</code> 
     * @param op operator
     * @param operand operand
     */
    public DyadicExpression(MonadicOperator op, Expression operand)
    {
        super(op,operand);
    }

    /**
     * Performs a function on two arguments.
     * op(operand1,operand2)
     * operand1 op operand2 
     * @param operand1 the first expression
     * @param op the operator between operands
     * @param operand2 the second expression
     */
    public DyadicExpression(Expression operand1, DyadicOperator op, Expression operand2)
    {
        super(operand1,op,operand2);
    }

    public Object evaluate(ExpressionEvaluator eval)
    {
        left.evaluate(eval);
        if (right != null)
        {
            right.evaluate(eval);
        }
        return eval.evaluate(this);
    }
    
    public Symbol bind()
    {
        try
        {
            left.bind();
        }
        catch (PrimaryExpressionIsClassLiteralException peil)
        {
            // PrimaryExpression should be swapped for a class Literal
            left = new Literal(peil.getLiteralClass());
            left.bind();
        }

        if (right != null)
        {
            try
            {
                right.bind();
            }
            catch (PrimaryExpressionIsClassLiteralException peil)
            {
                // PrimaryExpression should be swapped for a class Literal
                right = new Literal(peil.getLiteralClass());
                right.bind();
            }
        }
        return null;
    }
    
    public String toString()
    {
        return "DyadicExpression "+ getLeft() + " " + getOperator() + " " + getRight();
    }
}