/**********************************************************************
Copyright (c) 2008 Erik Bengtson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
2008 Andy Jefferson - javadocs. Support for DISTINCT. Support for implicit parameters
2008 Andy Jefferson - compileFrom()
    ...
**********************************************************************/
package org.datanucleus.query.expression;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.datanucleus.query.node.Node;
import org.datanucleus.query.node.ParameterNode;
import org.datanucleus.query.symbol.SymbolTable;

/**
 * Compiler for expressions. Responsible for taking a Node tree and creating an Expression tree.
 */
public class ExpressionCompiler
{
    SymbolTable symtbl;

    public void setSymbolTable(SymbolTable symtbl)
    {
        this.symtbl = symtbl;
    }

    /**
     * Primary entry point for compiling a node for the order clause.
     * @param node The node
     * @return Its compiled expression
     */
    public Expression compileOrderExpression(Node node)
    {
        if (isOperator(node, "order"))
        {
            if (node.getChildNodes().size() > 1)
            {
                return new OrderExpression(symtbl, compileExpression(node.getFirstChild()), (String) node.getNextChild().getNodeValue());
            }
            if (node.getChildNodes().size() == 1)
            {
                return new OrderExpression(symtbl, compileExpression(node.getFirstChild()));
            }
        }
        return compileExpression(node.getFirstChild());
    }

    /**
     * Primary entry point for compiling a node for the from clause.
     * @param node The node
     * @return Its compiled expression
     */
    public Expression compileFromExpression(Node node)
    {
        if (node.getNodeType() == Node.CLASS)
        {
            Node aliasNode = node.getFirstChild();
            ClassExpression clsExpr = new ClassExpression(symtbl, (String)aliasNode.getNodeValue());

            // Process any joins, chained down off the ClassExpression
            // So you can do clsExpr.getRight() to get the JoinExpression
            // then joinExpr.getRight() to get the next JoinExpression (if any)
            JoinExpression currentJoinExpr = null;
            Iterator childIter = node.getChildNodes().iterator();
            while (childIter.hasNext())
            {
                Node childNode = (Node)childIter.next();
                if (childNode.getNodeType() == Node.OPERATOR)
                {
                    String joinType = (String)childNode.getNodeValue();
                    int joinTypeId = JoinExpression.JOIN_INNER;
                    if (joinType.equals("JOIN_INNER_FETCH"))
                    {
                        joinTypeId = JoinExpression.JOIN_INNER_FETCH;
                    }
                    else if (joinType.equals("JOIN_OUTER_FETCH"))
                    {
                        joinTypeId = JoinExpression.JOIN_OUTER_FETCH;
                    }
                    else if (joinType.equals("JOIN_OUTER"))
                    {
                        joinTypeId = JoinExpression.JOIN_OUTER;
                    }
                    Node joinedNode = childNode.getFirstChild();
                    Node joinedAliasNode = childNode.getNextChild();
                    PrimaryExpression primExpr = (PrimaryExpression)compilePrimaryExpression(joinedNode);

                    JoinExpression joinExpr = new JoinExpression(symtbl, primExpr, 
                        (String)joinedAliasNode.getNodeValue(), joinTypeId);
                    if (currentJoinExpr != null)
                    {
                        currentJoinExpr.setJoinExpression(joinExpr);
                    }
                    else
                    {
                        clsExpr.setJoinExpression(joinExpr);
                    }
                    currentJoinExpr = joinExpr;
                }
            }
            return clsExpr;
        }
        return null;
    }

    /**
     * Primary entry point for compiling a node for the filter, grouping, having, result clauses.
     * @param node The node
     * @return Its compiled expression
     */
    public Expression compileExpression(Node node)
    {
        return compileConditionalOrExpression(node);
    }

    /**
     * This method deals with the OR condition.
     * A condition specifies a combination of one or more expressions and
     * logical (Boolean) operators and returns a value of TRUE, FALSE, or unknown
     */
    private Expression compileConditionalOrExpression(Node node)
    {
        if (isOperator(node, "||"))
        {
            Expression left = compileExpression(node.getFirstChild());
            Expression right = compileExpression(node.getNextChild());
            return new DyadicExpression(left,Expression.OP_OR,right);
        }
        return compileConditionalAndExpression(node);
    }    

    /**
     * This method deals with the AND condition
     * A condition specifies a combination of one or more expressions and
     * logical (Boolean) operators and returns a value of TRUE, FALSE, or unknown
     */
    private Expression compileConditionalAndExpression(Node node)
    {
        if (isOperator(node, "&&"))
        {
            Expression left = compileExpression(node.getFirstChild());
            Expression right = compileExpression(node.getNextChild());
            return new DyadicExpression(left,Expression.OP_AND,right);
        }
        return compileInclusiveOrExpression(node);
    }    

    private Expression compileInclusiveOrExpression(Node node)
    {
        if (isOperator(node, "|"))
        {
            Expression left = compileExpression(node.getFirstChild());
            Expression right = compileExpression(node.getNextChild());
            return new DyadicExpression(left,Expression.OP_OR,right);
        }
        return compileExclusiveOrExpression(node);
    }
    
    private Expression compileExclusiveOrExpression(Node node)
    {
        if (isOperator(node, "^"))
        {
            Expression left = compileExpression(node.getFirstChild());
            Expression right = compileExpression(node.getNextChild());
            return new DyadicExpression(left,Expression.OP_OR,right);
        }
        return compileAndExpression(node);
    }    
    
    private Expression compileAndExpression(Node node)
    {
        if (isOperator(node, "&"))
        {
            Expression left = compileExpression(node.getFirstChild());
            Expression right = compileExpression(node.getNextChild());
            return new DyadicExpression(left,Expression.OP_AND,right);
        }
        return compileEqualityExpression(node);
    }
    
    private Expression compileEqualityExpression(Node node)
    {
        if (isOperator(node, "=="))
        {
            Expression left = compileExpression(node.getFirstChild());
            Expression right = compileExpression(node.getNextChild());
            return new DyadicExpression(left, Expression.OP_EQ, right);
        }
        else if (isOperator(node, "!="))
        {
            Expression left = compileExpression(node.getFirstChild());
            Expression right = compileExpression(node.getNextChild());
            return new DyadicExpression(left, Expression.OP_NOTEQ, right);
        }
        else if (isOperator(node, "LIKE"))
        {
            Expression left = compileExpression(node.getFirstChild());
            Expression right = compileExpression(node.getNextChild());
            return new DyadicExpression(left, Expression.OP_LIKE, right);
        }
        return compileRelationalExpression(node);
    }

    private Expression compileRelationalExpression(Node node)
    {
        if (isOperator(node, "<="))
        {
            Expression left = compileExpression(node.getFirstChild());
            Expression right = compileExpression(node.getNextChild());
            return new DyadicExpression(left,Expression.OP_LTEQ,right);
        }
        else if (isOperator(node, ">="))
        {
            Expression left = compileExpression(node.getFirstChild());
            Expression right = compileExpression(node.getNextChild());
            return new DyadicExpression(left,Expression.OP_GTEQ,right);
        }
        else if (isOperator(node, "<"))
        {
            Expression left = compileExpression(node.getFirstChild());
            Expression right = compileExpression(node.getNextChild());
            return new DyadicExpression(left,Expression.OP_LT,right);
        }
        else if (isOperator(node, ">"))
        {
            Expression left = compileExpression(node.getFirstChild());
            Expression right = compileExpression(node.getNextChild());
            return new DyadicExpression(left,Expression.OP_GT,right);
        }
        else if (isOperator(node, "instanceof"))
        {
            Expression left = compileExpression(node.getFirstChild());
            Expression right = compileExpression(node.getNextChild());
            return new DyadicExpression(left,Expression.OP_IS,right);
        }
        return compileAdditiveExpression(node);
    }

    private Expression compileAdditiveExpression(Node node)
    {
        if (isOperator(node, "+"))
        {
            Expression left = compileExpression(node.getFirstChild());
            Expression right = compileExpression(node.getNextChild());
            return new DyadicExpression(left,Expression.OP_ADD,right);
        }
        else if (isOperator(node, "-"))
        {
            Expression left = compileExpression(node.getFirstChild());
            Expression right = compileExpression(node.getNextChild());
            return new DyadicExpression(left,Expression.OP_SUB,right);
        }
        return compileMultiplicativeExpression(node);
    }
   
    private Expression compileMultiplicativeExpression(Node node)
    {
        if (isOperator(node, "*"))
        {
            Expression left = compileExpression(node.getFirstChild());
            Expression right = compileExpression(node.getNextChild());
            return new DyadicExpression(left,Expression.OP_MUL,right);
        }
        else if (isOperator(node, "/"))
        {
            Expression left = compileExpression(node.getFirstChild());
            Expression right = compileExpression(node.getNextChild());
            return new DyadicExpression(left,Expression.OP_DIV,right);
        }
        else if (isOperator(node, "%"))
        {
            Expression left = compileExpression(node.getFirstChild());
            Expression right = compileExpression(node.getNextChild());
            return new DyadicExpression(left,Expression.OP_MOD,right);
        }
        return compileUnaryExpression(node);
    }

    private Expression compileUnaryExpression(Node node)
    {
        if (isOperator(node, "NEG"))
        {
            Expression left = compileExpression(node.getFirstChild());
            return new DyadicExpression(Expression.OP_NEG, left);
        }
        return compileUnaryExpressionNotPlusMinus(node);
    }
    
    private Expression compileUnaryExpressionNotPlusMinus(Node node)
    {
        if (isOperator(node, "~"))
        {
            Expression left = compileExpression(node.getFirstChild());
            return new DyadicExpression(Expression.OP_COM, left);
        }
        else if (isOperator(node, "!"))
        {
            Expression left = compileExpression(node.getFirstChild());
            return new DyadicExpression(Expression.OP_NOT, left);
        }
        else if (isOperator(node, "DISTINCT"))
        {
            Expression left = compileExpression(node.getFirstChild());
            return new DyadicExpression(Expression.OP_DISTINCT, left);
        }
        return compilePrimaryExpression(node);
    }    

    private Expression compilePrimaryExpression(Node node)
    {
        if (node.getNodeType() == Node.IDENTIFIER)
        {
            // TODO Cater for the node here being a variable, and set the expression to a VariableExpression
            Node currentNode = node;
            List tupple = new ArrayList();
            Expression currentExpr = null;
            while (currentNode != null)
            {
                tupple.add(currentNode.getNodeValue());

                if (currentNode.getNodeType() == Node.INVOKE)
                {
                    if (currentExpr == null && tupple.size() > 1)
                    {
                        // Start from PrimaryExpression and invoke on that
                        currentExpr = new PrimaryExpression(symtbl, tupple.subList(0, tupple.size()-1));
                    }

                    String methodName = (String)tupple.get(tupple.size()-1);
                    List parameterExprs = getExpressionsForPropertiesOfNode(currentNode);
                    currentExpr = new InvokeExpression(symtbl, currentExpr, methodName, parameterExprs);

                    currentNode = currentNode.getFirstChild();
                    if (currentNode != null)
                    {
                        // Continue on along the chain
                        tupple = new ArrayList();
                        tupple.add(currentExpr); // Start from this expression
                    }
                }
                else
                {
                    currentNode = currentNode.getFirstChild();
                }
            }

            if (currentExpr == null)
            {
                return new PrimaryExpression(symtbl, tupple);
            }
            return currentExpr;
        }
        else if (node.getNodeType() == Node.PARAMETER)
        {
            return new ParameterExpression(symtbl, (String)node.getNodeValue(),
                ((ParameterNode)node).getPosition());
        }
        else if (node.getNodeType() == Node.INVOKE)
        {
            Node currentNode = node;
            List tupple = new ArrayList();
            Expression currentExpr = null;
            while (currentNode != null)
            {
                tupple.add(currentNode.getNodeValue());

                if (currentNode.getNodeType() == Node.INVOKE)
                {
                    String methodName = (String)tupple.get(tupple.size()-1);
                    List parameterExprs = getExpressionsForPropertiesOfNode(currentNode);
                    currentExpr = new InvokeExpression(symtbl, currentExpr, methodName, parameterExprs);

                    currentNode = currentNode.getFirstChild();
                    if (currentNode != null)
                    {
                        // Continue on along the chain
                        tupple = new ArrayList();
                        tupple.add(currentExpr); // Start from this expression
                    }
                }
                else
                {
                    // TODO What node type is this that comes after an INVOKE?
                    currentNode = currentNode.getFirstChild();
                }
            }
            return currentExpr;
        }
        else if (node.getNodeType() == Node.CREATOR)
        {
            Node currentNode = node.getFirstChild();
            List tupple = new ArrayList();
            boolean method = false;
            while (currentNode != null)
            {
                tupple.add(currentNode.getNodeValue());
                if (currentNode.getNodeType() == Node.INVOKE)
                {
                    method = true;
                    break;
                }
                currentNode = currentNode.getFirstChild();
            }

            List parameterExprs = null;
            if (method)
            {
                parameterExprs = getExpressionsForPropertiesOfNode(currentNode);
            }
            else
            {
                parameterExprs = new ArrayList();
            }
            return new CreatorExpression(symtbl, tupple, parameterExprs);
        }
        else if (node.getNodeType() == Node.LITERAL)
        {
            // TODO Can we have a Literal.INVOKE ? Probably
            return new Literal(node.getNodeValue());
        }
        return null;
    }

    /**
     * Convenience method to extract properties for this node and return the associated list of expressions.
     * @param node The node
     * @return The list of expressions for the properties
     */
    private List getExpressionsForPropertiesOfNode(Node node)
    {
        // TODO Don't create ArrayList if no properties for efficiency (needs checks elsewhere for null)
        List parameterExprs = new ArrayList();
        if (node.hasProperties())
        {
            List propNodes = node.getProperties();
            for (int i=0;i<propNodes.size();i++)
            {
                parameterExprs.add(compileExpression((Node)propNodes.get(i)));
            }
        }
        return parameterExprs;
    }

    private boolean isOperator(Node node, String operator)
    {
        return node.getNodeType() == Node.OPERATOR && node.getNodeValue().equals(operator);
    }
}