/**********************************************************************
Copyright (c) 2008 Erik Bengtson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
2008 Andy Jefferson - javadocs
2008 Andy Jefferson - rewritten to allow chaining. Use "left" for what we invoke on, and store "method" explicitly.
    ...
**********************************************************************/
package org.datanucleus.query.expression;

import java.util.List;

import org.datanucleus.query.symbol.Symbol;
import org.datanucleus.query.symbol.SymbolTable;
import org.datanucleus.util.StringUtils;

/**
 * Expression representing invocation of a method.
 * This may be an aggregate in a result clause (like "count(this)"), or a method on a class, or a function.
 * The "left" expression is what we are invoking on. This is typically a PrimaryExpression, or an InvokeExpression.
 * This then allows chaining of invocations.
 */
public class InvokeExpression extends Expression
{
    SymbolTable symtbl;

    /** Name of the method to invoke. */
    String methodName;

    /** Parameters for the method invocation. */
    List arguments;

    /**
     * Constructor for an expression for the invocation of a method/function.
     * @param symtbl Symbol table
     * @param invoked Expression on which we are invoking
     * @param methodName Name of the method
     * @param args Arguments passed in to the method/function call
     */
    public InvokeExpression(SymbolTable symtbl, Expression invoked, String methodName, List args)
    {
        this.symtbl = symtbl;
        this.left = invoked;
        this.methodName = methodName;
        this.arguments = args;
    }

    public Object evaluate(ExpressionEvaluator eval)
    {
        return eval.evaluate(this);
    }

    /**
     * The method/function invoked.
     * @return The method/function invoked.
     */
    public String getOperation()
    {
        return methodName;
    }

    /**
     * Accessor for any arguments to be passed in the invocation.
     * @return The arguments.
     */
    public List getArguments()
    {
        return arguments;
    }

    public Symbol bind()
    {
        if (left != null)
        {
            symbol = left.bind();
        }
        /*if (symtbl.hasSymbol(getId()))
        {
            symbol = symtbl.getSymbol(getId());
        }
        else
        {
            if (tuples.size() > 1)
            {
                symbol = new PropertySymbol(getId(), symtbl.getType(tuples.subList(0, tuples.size()-1)));
            }
        }*/
        return symbol;
    }

    public String toString()
    {
        return "InvokeExpression: [" + left + "]." + methodName + "(" + StringUtils.collectionToString(arguments) + ")";
    }
}