/**********************************************************************
Copyright (c) 2008 Erik Bengtson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
2008 Andy Jefferson - added bind check on whether class literal. Javadocs
    ...
**********************************************************************/
package org.datanucleus.query.expression;

import java.util.Iterator;
import java.util.List;

import org.datanucleus.exceptions.ClassNotResolvedException;
import org.datanucleus.exceptions.NucleusUserException;
import org.datanucleus.query.symbol.PropertySymbol;
import org.datanucleus.query.symbol.Symbol;
import org.datanucleus.query.symbol.SymbolTable;
import org.datanucleus.util.StringUtils;

/**
 * Expression for a primary object. This may be a field, or an explicit variable/parameter.
 * Implicit parameters are expressed by ParameterExpression since we know at compile time that they
 * are parameters.
 */
public class PrimaryExpression extends Expression
{
    SymbolTable symtbl;

    List tuples;

    public PrimaryExpression(SymbolTable symtbl, List tuples)
    {
        this.symtbl = symtbl;
        this.tuples = tuples;
    }
    
    public Object evaluate(ExpressionEvaluator eval)
    {
        return eval.evaluate(this);
    }

    public String getId()
    {
        StringBuffer id = new StringBuffer();
        for (int i=0; i<tuples.size(); i++)
        {
            if (id.length() > 0)
            {
                id.append('.');
            }
            id.append((String)tuples.get(i));
        }
        return id.toString();
    }

    public List getTuples()
    {
        return tuples;
    }

    public Symbol bind()
    {
        if (symtbl.hasSymbol(getId()))
        {
            symbol = symtbl.getSymbol(getId());
        }
        else
        {
            try
            {
                Class symbolType = symtbl.getType(tuples);
                symbol = new PropertySymbol(getId(), symbolType);
            }
            catch (NucleusUserException nue)
            {
                // Thrown if a field in the primary expression doesn't exist.
                // This may be due to an entry like "org.jpox.samples.MyClass" used for "instanceof"
                StringBuffer str = new StringBuffer();
                Iterator iter = tuples.iterator();
                while (iter.hasNext())
                {
                    String tuple = (String)iter.next();
                    if (str.length() > 0)
                    {
                        str.append('.');
                    }
                    str.append(tuple);
                }
                try
                {
                    // Try to find the class that this represents
                    Class cls = symtbl.getClassLoaderResolver().classForName(str.toString());

                    // Represents a valid class so throw exception to get the PrimaryExpression swapped
                    throw new PrimaryExpressionIsClassLiteralException(cls);
                }
                catch (ClassNotResolvedException cnre)
                {
                    // Just throw the original exception
                    throw nue;
                }
            }
        }
        return symbol;
    }

    public String toString()
    {
        return "PrimaryExpression: " + StringUtils.collectionToString(tuples);
    }
}