/**********************************************************************
Copyright (c) 2008 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.state;

import java.util.HashMap;
import java.util.Map;

import org.datanucleus.cache.CachedPC;

/**
 * Holder for the state information for the (Level2) cache process.
 * This is used when we are preparing a graph of objects for caching in the L2 cache.
 * The persistence process requests that object X is cached in the L2 cache, and we process
 * that object down the fetch plan finding a collection of objects that can be cached with
 * it (following what the user has defined in MetaData for the fields that can be cached).
 */
public class CacheState extends FetchPlanState
{
    /** Map of cacheable objects to be stored in the L2 cache, keyed by the object id. **/
    private Map cacheObjectsById = new HashMap();

    /**
     * Constructor.
     */
    public CacheState()
    {
    }

    /**
     * Method to add an object to be cached.
     * @param id Id of the object
     * @param cacheablePC the cacheable persistable object
     */
    public void addCacheableObject(Object id, CachedPC cacheablePC)
    {
        cacheObjectsById.put(id, cacheablePC);
    }

    /**
     * Accessor for the cacheable object for this id (if any).
     * Returns the object reference that will be put in the (level2) cache when
     * this process completes.
     * @param id The identity
     * @return The cacheable object (CachedPC)
     */
    public CachedPC getCacheableObject(Object id)
    {
        return (CachedPC)cacheObjectsById.get(id);
    }

    /**
     * Accessor for whether the object with the supplied identity is already to be cached.
     * @param id Identity of the object
     * @return Whether it is already in the map
     */
    public boolean contains(Object id)
    {
        return cacheObjectsById.containsKey(id);
    }

    /**
     * Accessor for the cacheable objects as a Map<Object, CachedPC> with the key being the identity.
     * @return the cacheable objects map
     */
    public Map getCacheableObjectsIdMap()
    {
        return cacheObjectsById;
    }
}