/**********************************************************************
Copyright (c) 2005 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License. 


Contributors:
    ...
**********************************************************************/
package org.datanucleus.state;

import java.util.ArrayList;
import java.util.List;
import java.util.ListIterator;

/**
 * Holder for the state control for FetchPlan processing.
 * Maintains a list of the field names being fetched. The first item in the List will be the root.
 * When a new branch of the graph of processed the field name is added, and is removed when it is
 * processed. This provides a means of always knowing the depth in the current graph, and also of
 * allowing detection of recursion of field names.
 * 
 * @version $Revision: 1.6 $
 */
public class FetchPlanState
{
    /**
     * List of field names in the graph. The first is the root of the tree, and fields are added as they are encountered
     * and removed when they are finished with.
     */
    protected List fetchFieldNames = new ArrayList();

    /**
     * Method to add an field name to the list since it is being processed
     * @param field The field to add
     */
    public void addFieldName(String field)
    {
        fetchFieldNames.add(field);
    }

    /**
     * Method to remove the latest field name from the list since it is now processed
     */
    public void removeLatestFieldName()
    {
        fetchFieldNames.remove(fetchFieldNames.size()-1);
    }

    /**
     * Accessor for the object graph depth currently
     * @return The graph depth
     */
    public int getCurrentFetchDepth()
    {
        return fetchFieldNames.size();
    }

    /**
     * Accessor for the current depth for the specified field name
     * @param fieldName The name of the field
     * @return The depth for this field name
     */
    public int getObjectDepthForType(String fieldName)
    {
        ListIterator iter = fetchFieldNames.listIterator(fetchFieldNames.size()); // Start at the end
        int number = 0;
        while (iter.hasPrevious())
        {
            String field = (String)iter.previous();
            if (field.equals(fieldName))
            {
                number++;
            }
            else
            {
                break;
            }
        }
        return number;
    }
}