/**********************************************************************
Copyright (c) 2007 Erik Bengtson and others. All rights reserved. 
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License. 
 

Contributors:
    ...
**********************************************************************/
package org.datanucleus.store;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;

import org.datanucleus.ObjectManagerFactoryImpl;
import org.datanucleus.metadata.AbstractMemberMetaData;
import org.datanucleus.metadata.ClassMetaData;
import org.datanucleus.metadata.IdentityType;
import org.datanucleus.util.NucleusLogger;
import org.datanucleus.util.Localiser;
import org.datanucleus.util.MultiMap;

/**
 * StoreData Cache Manager
 */
public class StoreDataManager
{
    protected static final Localiser LOCALISER = Localiser.getInstance(
        "org.datanucleus.Localisation", ObjectManagerFactoryImpl.class.getClassLoader());    
    
    /** Map of all managed store data, keyed by the class/field name. */
    protected Map storeDataByClass = Collections.synchronizedMap(new HashMap());

    /** Map of all managed store data using Application Identity, keyed by the app id PK class */
    protected MultiMap storeDataByAppIdClass = new MultiMap();

    /** the memory image of schema data beforing running it **/
    protected Map savedStoreDataByClass;

    /** the memory image of schema data beforing running it **/
    protected MultiMap savedStoreDataByAppIdClass;
    
    /**
     * Clear the cache
     */
    public void clear()
    {
        if (NucleusLogger.PERSISTENCE.isInfoEnabled())
        {
            NucleusLogger.PERSISTENCE.info(LOCALISER.msg("032002"));
        }

        storeDataByClass.clear();
        storeDataByAppIdClass.clear();
    }
    
    /**
     * Method to register some data with the store.
     * This will also register the data with the starter process.
     * @param data The StoreData to add
     */
    protected void registerStoreData(StoreData data)
    {
        if (data.isFCO())
        {
            // Index any classes by the class name
            storeDataByClass.put(data.getName(), data);

            // If it's a class, using APPLICATION identity and is users own AID then store the PK class.
            // We don't need SingleFieldIdentity in here since they define the class being used
            ClassMetaData cmd = (ClassMetaData)data.getMetaData();
            if (cmd.getIdentityType() == IdentityType.APPLICATION && !cmd.usesSingleFieldIdentityClass())
            {
                storeDataByAppIdClass.put(cmd.getObjectidClass(), data);
            }
        }
        else
        {
            // Index any fields by the class name of the field
            storeDataByClass.put(data.getMetaData(), data);
        }

        if (NucleusLogger.PERSISTENCE.isInfoEnabled())
        {
            NucleusLogger.PERSISTENCE.info(LOCALISER.msg("032001", data));
        }
    }

    /**
     * Convenience accessor for all store data where property 1 has value1 and property 2 has value2.
     * Uses equals() on the values. Doesnt cater for null values.
     * @param key1 Property 1 name
     * @param value1 Property 1 value
     * @param key2 Property 2 name
     * @param value2 Property 2 value
     * @return Store data with the specified property values
     */
    public synchronized StoreData[] getStoreDataForProperties(String key1, Object value1,
            String key2, Object value2)
    {
        Collection results = null;
        Collection storeDatas = storeDataByClass.values();
        synchronized (storeDataByClass) {
			// Collections.synchronizedMap() says: 
			// "It is imperative that the user manually synchronize on the returned
			// map when iterating over any of its collection views"
			Iterator iterator = storeDatas.iterator();
			while (iterator.hasNext()) {
				StoreData data = (StoreData) iterator.next();
				if (data.getProperties() != null) {
					Object prop1Value = data.getProperties().get(key1);
					Object prop2Value = data.getProperties().get(key2);
					if (prop1Value != null && prop1Value.equals(value1)
							&& prop2Value != null && prop2Value.equals(value2)) {
						if (results == null) {
							results = new HashSet();
						}
						results.add(data);
					}
				}
			}
		}
		if (results != null)
        {
            return (StoreData[])results.toArray(new StoreData[results.size()]);
        }
        return null;
    }

    /**
     * Accessor for whether the specified class is managed currently
     * @param className The name of the class
     * @return Whether it is managed
     */
    public boolean managesClass(String className)
    {
        return storeDataByClass.containsKey(className);
    }
    
    /**
     * Accessor for the StoreData currently managed by this store.
     * @return Collection of the StoreData being managed
     */
    public Collection getManagedStoreData()
    {
        return Collections.unmodifiableCollection(storeDataByClass.values());
    }    

    /**
     * Get the StoreData by the given className
     * @param className the fully qualified class name
     * @return the StoreData
     */
    public StoreData get(String className)
    {
        return (StoreData) storeDataByClass.get(className);
    }

    /**
     * Get the StoreData by the given field/property
     * @param apmd the field/property
     * @return the StoreData
     */
    public StoreData get(AbstractMemberMetaData apmd)
    {
        return (StoreData) storeDataByClass.get(apmd);
    }

    /**
     * Get the Collection of StoreData that currently uses this primary key class
     * @param className the primary key class name
     * @return the Collection of StoreData
     */
    public Collection getByPrimaryKeyClass(String className)
    {
        return (Collection) storeDataByAppIdClass.get(className);
    }

    /**
     * Acessor to the number of StoreData in cache
     * @return the number of StoreData in cache
     */
    public int size()
    {
        return storeDataByClass.size();
    }

    /**
     * Begin a transaction that changes the StoreData cache
     */
    public void begin()
    {
        savedStoreDataByClass = new HashMap(storeDataByClass);
        savedStoreDataByAppIdClass = new MultiMap(storeDataByAppIdClass);
    }
    
    /**
     * Rollbacks the transaction changes to the StoreData cache
     */
    public void rollback()
    {
        storeDataByClass = savedStoreDataByClass;
        storeDataByAppIdClass = savedStoreDataByAppIdClass;
    }
    
    /**
     * Commit the transaction changes to the StoreData cache
     */
    public void commit()
    {
        savedStoreDataByClass = null;
        savedStoreDataByAppIdClass = null;
    }
}