/**********************************************************************
Copyright (c) 2008 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.store;

import org.datanucleus.ObjectManager;
import org.datanucleus.StateManager;
import org.datanucleus.exceptions.NucleusDataStoreException;
import org.datanucleus.exceptions.NucleusObjectNotFoundException;

/**
 * Interface defining persistence operations of a StoreManager.
 * @version $Revision$
 */
public interface StorePersistenceHandler
{
    /**
     * Method to close the persistence handler, and release any resources.
     */
    public void close();

    /**
     * Inserts a persistent object into the database.
     * @param sm The state manager of the object to be inserted.
     * @throws NucleusDataStoreException when an error occurs in the datastore communication
     */
    public void insertObject(StateManager sm);

    /**
     * Fetches a persistent object from the database.
     * @param sm The state manager of the object to be fetched.
     * @param fieldNumbers The numbers of the fields to be fetched.
     * @throws NucleusObjectNotFoundException if the object doesnt exist
     * @throws NucleusDataStoreException when an error occurs in the datastore communication
     */
    public void fetchObject(StateManager sm, int fieldNumbers[]);

    /**
     * Updates a persistent object in the datastore.
     * @param sm The state manager of the object to be updated.
     * @param fieldNumbers The numbers of the fields to be updated.
     * @throws NucleusDataStoreException when an error occurs in the datastore communication
     */
    public void updateObject(StateManager sm, int fieldNumbers[]);

    /**
     * Deletes a persistent object from the datastore.
     * @param sm The state manager of the object to be deleted.
     * @throws NucleusDataStoreException when an error occurs in the datastore communication
     */
    public void deleteObject(StateManager sm);

    /**
     * Locates this object in the datastore.
     * @param sm The StateManager for the object to be found
     * @throws NucleusObjectNotFoundException if the object doesnt exist
     * @throws NucleusDataStoreException when an error occurs in the datastore communication
     */
    public void locateObject(StateManager sm);

    /**
     * Method to find a persistable object with the specified id from the datastore, if the StoreManager supports
     * this operation (optional). This allows for datastores that perform the instantiation of objects directly
     * (such as ODBMS). With other types of datastores (e.g RDBMS) this method returns null. If the StoreManager
     * supports this operation yet the object is not found an exception should be thrown.
     * @param om the ObjectManager which will manage the object
     * @param id the id of the object in question.
     * @return a persistable object with a valid object state (for example: hollow) or null, 
     *     indicating that the implementation leaves the instantiation work to JPOX.
     */
    public Object findObject(ObjectManager om, Object id);
}