/**********************************************************************
Copyright (c) 2008 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.store.fieldmanager;

import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.datanucleus.StateManager;
import org.datanucleus.FetchPlan.FetchPlanForClass;
import org.datanucleus.api.ApiAdapter;
import org.datanucleus.cache.CachedPC;
import org.datanucleus.exceptions.NucleusException;
import org.datanucleus.jdo.NucleusJDOHelper;
import org.datanucleus.metadata.AbstractMemberMetaData;
import org.datanucleus.metadata.MetaDataUtils;
import org.datanucleus.sco.SCO;
import org.datanucleus.sco.SCOContainer;
import org.datanucleus.state.CacheState;
import org.datanucleus.state.JDOStateManagerImpl;
import org.datanucleus.util.NucleusLogger;
import org.datanucleus.util.StringUtils;

/**
 * FieldManager to handle the population of the Level2 cache with objects.
 * Recurses down the object graph to the fetchDepth limit adding objects to the L2 cache as it goes.
 * Any field that has an SCO wrapper will be replaced by a non-SCO object.
 */
public class CachePopulateFieldManager extends AbstractFetchFieldManager
{
    CachedPC cachedPC;

    /**
     * Constructor.
     * @param sm the StateManager of the instance being cached.
     * @param secondClassMutableFields The second class mutable fields for the class of this object
     * @param fpClass Fetch Plan for the class of this instance
     * @param state State object to hold any pertinent controls for the caching process
     */
    public CachePopulateFieldManager(StateManager sm, boolean[] secondClassMutableFields, 
            FetchPlanForClass fpClass, CacheState state, CachedPC cacheable)
    {
        super(sm, secondClassMutableFields, fpClass, state);
        this.cachedPC = cacheable;
    }

    /**
     * Utility method to process the passed persistable object.
     * @param pc The PC object
     */
    protected Object processPersistable(AbstractMemberMetaData mmd, Object pc)
    {
        if (pc == null)
        {
            return null;
        }

        ApiAdapter api = sm.getObjectManager().getApiAdapter();
        if (!api.isPersistable(pc))
        {
            return pc;
        }
        if (api.isDetached(pc))
        {
            NucleusLogger.CACHE.warn(">> CachePopulateFM.processPersistable" +
                " this=" + StringUtils.toJVMIDString(sm.getObject()) +
                " field=" + mmd.getFullFieldName() + " pc=" + StringUtils.toJVMIDString(pc) +
                " state=" + NucleusJDOHelper.getObjectStateAsString(pc) + " IS DETACHED!");
            return null;
        }

        CacheState cacheState = (CacheState)state;
        if (cacheState.contains(api.getIdForObject(pc)))
        {
            // Already have an object ready for caching with this id so return it
            CachedPC cachePC = cacheState.getCacheableObject(api.getIdForObject(pc));
            return cachePC.getPersistableObject();
        }
        else
        {
            // Not yet in the collection of objects to be cached so prepare it for caching
            StateManager pcSM = sm.getObjectManager().findStateManager(pc);
            if (pcSM != null)
            {
                return ((JDOStateManagerImpl)pcSM).cache(cacheState);
            }

            // For some reason this PC has no StateManager (transient/detached?)
            // TODO Investigate if this is possible
            return null;
        }
    }

    /**
     * Method to fetch an object field whether it is collection/map, PC, or whatever for the cache process.
     * @param fieldNumber Number of the field
     * @return The object
     */
    protected Object internalFetchObjectField(int fieldNumber)
    {
        SingleValueFieldManager sfv = new SingleValueFieldManager();
        sm.provideFields(new int[]{fieldNumber}, sfv);
        Object value = sfv.fetchObjectField(fieldNumber);
        ApiAdapter api = sm.getObjectManager().getApiAdapter();
        if (value == null)
        {
            return null;
        }
        else
        {
            AbstractMemberMetaData fmd = 
                sm.getClassMetaData().getMetaDataForManagedMemberAtAbsolutePosition(fieldNumber);
            if (!fmd.isCacheable())
            {
                // Field is marked as not cacheable so unset its loaded flag and return null
                cachedPC.getLoadedFields()[fieldNumber] = false;
                return null;
            }

            if (api.isPersistable(value))
            {
                // 1-1, N-1 PC field
                if (fmd.isSerialized() || fmd.isEmbedded())
                {
                    // TODO Support serialised/embedded PC fields
                    cachedPC.getLoadedFields()[fieldNumber] = false;
                    return null;
                }

                Object cachePC = processPersistable(fmd, value);
                if (cachePC == null)
                {
                    // Object was detached?
                    cachedPC.getLoadedFields()[fieldNumber] = false;
                    return null;
                }
                else
                {
                    // Put field OID in CachedPC "relationFields" and store null in this field
                    cachedPC.setRelationField(fmd.getName(), api.getIdForObject(value));
                    return null;
                }
            }
            else if (value instanceof Collection)
            {
                // 1-N, M-N Collection
                if (MetaDataUtils.getInstance().storesPersistable(fmd, sm.getObjectManager().getClassLoaderResolver()))
                {
                    if (fmd.isSerialized() || fmd.isEmbedded() ||
                        fmd.getCollection().isSerializedElement() || fmd.getCollection().isEmbeddedElement())
                    {
                        // TODO Support serialised/embedded elements
                        cachedPC.getLoadedFields()[fieldNumber] = false;
                        return null;
                    }
                    Collection collValue = (Collection)value;
                    if (collValue instanceof SCO && !((SCOContainer)value).isLoaded())
                    {
                        // Contents not loaded so just mark as unloaded
                        cachedPC.getLoadedFields()[fieldNumber] = false;
                        return null;
                    }

                    Iterator collIter = collValue.iterator();
                    Collection returnColl = null;
                    try
                    {
                        if (value.getClass().isInterface())
                        {
                            if (List.class.isAssignableFrom(value.getClass()) || fmd.getOrderMetaData() != null)
                            {
                                // List based
                                returnColl = new ArrayList();
                            }
                            else
                            {
                                // Set based
                                returnColl = new HashSet();
                            }
                        }
                        else
                        {
                            if (value instanceof SCO)
                            {
                                returnColl = (Collection)((SCO)value).getValue().getClass().newInstance();
                            }
                            else
                            {
                                returnColl = (Collection)value.getClass().newInstance();
                            }
                        }

                        // Recurse through elements, and put ids of elements in return value
                        while (collIter.hasNext())
                        {
                            Object elem = collIter.next();
                            processPersistable(fmd, elem);
                            returnColl.add(api.getIdForObject(elem));
                        }

                        // Put Collection<OID> in CachedPC "relationFields" and store null in this field
                        cachedPC.setRelationField(fmd.getName(), returnColl);
                        return null;
                    }
                    catch (Exception e)
                    {
                        throw new NucleusException("Unable to create object of type " + value.getClass().getName(), e);
                    }
                }
                else
                {
                    // Collection<Non-PC> so just return it
                    if (value instanceof SCOContainer)
                    {
                        if (((SCOContainer)value).isLoaded())
                        {
                            // Return unwrapped collection
                            return ((SCO)value).getValue();
                        }
                        else
                        {
                            // Contents not loaded so just mark as unloaded
                            cachedPC.getLoadedFields()[fieldNumber] = false;
                            return null;
                        }
                    }
                    else
                    {
                        return value;
                    }
                }
            }
            else if (value instanceof Map)
            {
                // 1-N, M-N Map
                if (MetaDataUtils.getInstance().storesPersistable(fmd, sm.getObjectManager().getClassLoaderResolver()))
                {
                    if (fmd.isSerialized() || fmd.isEmbedded() ||
                        fmd.getMap().isSerializedKey() || fmd.getMap().isEmbeddedKey() ||
                        fmd.getMap().isSerializedValue() || fmd.getMap().isEmbeddedValue())
                    {
                        // TODO Support serialised/embedded keys/values
                        cachedPC.getLoadedFields()[fieldNumber] = false;
                        return null;
                    }

                    if (value instanceof SCO && !((SCOContainer)value).isLoaded())
                    {
                        // Contents not loaded so just mark as unloaded
                        cachedPC.getLoadedFields()[fieldNumber] = false;
                        return null;
                    }

                    try
                    {
                        Map returnMap = null;
                        if (value.getClass().isInterface())
                        {
                            returnMap = new HashMap();
                        }
                        else
                        {
                            if (value instanceof SCO)
                            {
                                returnMap = (Map)((SCO)value).getValue().getClass().newInstance();
                            }
                            else
                            {
                                returnMap = (Map)value.getClass().newInstance();
                            }
                        }
                        Iterator mapIter = ((Map)value).entrySet().iterator();
                        while (mapIter.hasNext())
                        {
                            Map.Entry entry = (Map.Entry)mapIter.next();
                            Object mapKey = null;
                            Object mapValue = null;
                            if (fmd.getMap().getKeyClassMetaData() != null)
                            {
                                processPersistable(fmd, entry.getKey());
                                mapKey = api.getIdForObject(entry.getKey());
                            }
                            else
                            {
                                mapKey = entry.getKey();
                            }
                            if (fmd.getMap().getValueClassMetaData() != null)
                            {
                                processPersistable(fmd, entry.getValue());
                                mapValue = api.getIdForObject(entry.getValue());
                            }
                            else
                            {
                                mapValue = entry.getValue();
                            }
                            returnMap.put(mapKey, mapValue);
                        }

                        // Put Map<X, Y> in CachedPC "relationFields" and store null in this field
                        // where X, Y can be OID if they are persistable objects
                        cachedPC.setRelationField(fmd.getName(), returnMap);
                        return null;
                    }
                    catch (Exception e)
                    {
                        throw new NucleusException("Unable to create object of type " + value.getClass().getName(), e);
                    }
                }
                else
                {
                    // Map<Non-PC, Non-PC> so just return it
                    if (value instanceof SCOContainer)
                    {
                        if (((SCOContainer)value).isLoaded())
                        {
                            // Return unwrapped map
                            return ((SCO)value).getValue();
                        }
                        else
                        {
                            // Contents not loaded so just mark as unloaded
                            cachedPC.getLoadedFields()[fieldNumber] = false;
                            return null;
                        }
                    }
                    else
                    {
                        return value;
                    }
                }
            }
            else if (value instanceof Object[])
            {
                // Array, maybe of Persistable objects
                if (!api.isPersistable(fmd.getType().getComponentType()))
                {
                    // Array element type is not persistable so just return the value
                    return value;
                }

                Object[] returnArr = new Object[Array.getLength(value)];
                for (int i=0;i<Array.getLength(value);i++)
                {
                    Object element = Array.get(value, i);
                    processPersistable(fmd, element);
                    returnArr[i] = api.getIdForObject(element);
                }

                // Put OID[] in CachedPC "relationFields" and store null in this field
                cachedPC.setRelationField(fmd.getName(), returnArr);
                return null;
            }
            else if (value instanceof SCO)
            {
                // SCO wrapper - so replace with unwrapped
                if (NucleusLogger.CACHE.isDebugEnabled())
                {
                    // TODO Localise this
                    NucleusLogger.CACHE.debug("CachePopulateFM.fetchObjectField this=" + 
                        StringUtils.toJVMIDString(sm.getObject()) +
                        " field=" + fieldNumber + " is having its SCO wrapper replaced prior to L2 caching.");
                }
                return ((SCO)value).getValue();
            }
            else
            {
                return value;
            }
        }
    }

    /**
     * Method to throw and EndOfFetchPlanGraphException since we're at the end of a branch in the tree.
     * @param fieldNumber Number of the field
     * @return Object to return
     */
    protected Object endOfGraphOperation(int fieldNumber)
    {
        // Reached end of our graph so return
        SingleValueFieldManager sfv = new SingleValueFieldManager();
        sm.provideFields(new int[]{fieldNumber}, sfv);
        Object value = sfv.fetchObjectField(fieldNumber);
        AbstractMemberMetaData fmd = 
            sm.getClassMetaData().getMetaDataForManagedMemberAtAbsolutePosition(fieldNumber);
        ApiAdapter api = sm.getObjectManager().getApiAdapter();

        CacheState cacheState = (CacheState)state;
        if (api.isPersistable(value) && cacheState.contains(api.getIdForObject(value)))
        {
            // Object is persistable and already in the objects to cache so set the field and return
            cachedPC.setRelationField(fmd.getName(), api.getIdForObject(value));
            return null;
        }

        // Update the cacheable object loaded flags so this field is not marked as loaded when cached
        cachedPC.getLoadedFields()[fieldNumber] = false;

        // end of object graph
        throw new EndOfFetchPlanGraphException();
    }
}