/**********************************************************************
Copyright (c) 2007 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.store.query;

import java.util.Collection;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Map;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.ObjectManager;
import org.datanucleus.exceptions.NucleusUserException;
import org.datanucleus.plugin.ConfigurationElement;
import org.datanucleus.plugin.Extension;
import org.datanucleus.plugin.PluginManager;
import org.datanucleus.store.Extent;
import org.datanucleus.util.Imports;
import org.datanucleus.util.NucleusLogger;

/**
 * Abstract representation of a Java-based query.
 * To be extended by Java-based query languages.
 */
public abstract class AbstractJavaQuery extends Query
{
    /** Extent of candidates for this query. */
    protected transient Extent candidateExtent = null;

    /** Collection of candidates for this query. */
    protected transient Collection candidateCollection = null;

    /** Cached form of the single string form of the query. */
    protected String singleString = null;

    /** Result metadata (extension, allowing access to more info about results). **/
    protected transient QueryResultsMetaData resultMetaData;

    // TODO This is currently using the old-style RDBMS extension. Change it to new compiler
    /** Register of user-defined ScalarExpression, provided via plugins. */
    protected static transient Map userDefinedScalarExpressions = new Hashtable();

    /**
     * Constructor for a Java-based query.
     * @param om The ObjectManager
     */
    public AbstractJavaQuery(ObjectManager om)
    {
        super(om);
        registerScalarExpressions(om.getOMFContext().getPluginManager(), om.getClassLoaderResolver());
    }

    /**
     * Set the candidate Extent to query.
     * @param pcs the Candidate Extent.
     * @see javax.jdo.Query#setCandidates(javax.jdo.Extent)
     */
    public void setCandidates(Extent pcs)
    {
        discardCompiled();
        assertIsModifiable();
    
        if (pcs == null)
        {
            NucleusLogger.QUERY.warn(LOCALISER.msg("021073"));
            return;
        }
    
        setSubclasses(pcs.hasSubclasses());
        setClass(pcs.getCandidateClass());
    
        candidateExtent = pcs;
        candidateCollection = null; // We have an Extent, so remove any collection
    }

    /**
     * Set the candidate Collection to query.
     * @param pcs the Candidate collection.
     * @see javax.jdo.Query#setCandidates(java.util.Collection)
     */
    public void setCandidates(Collection pcs)
    {
        discardCompiled();
        assertIsModifiable();
        if (pcs == null)
        {
            NucleusLogger.QUERY.warn(LOCALISER.msg("021072"));
            return;
        }
    
        candidateExtent = null;
        candidateCollection = pcs; // We have a Collection, so remove any Extent
    }

    /**
     * Accessor for the candidate Extent (if specified using an Extent).
     * @return Candidate Extent
     */
    public Extent getCandidateExtent()
    {
        return candidateExtent;
    }

    /**
     * Accessor for the candidate collection (if specified using a collection).
     * @return Candidate collection
     */
    public Collection getCandidateCollection()
    {
        return candidateCollection;
    }

    /**
     * Method to discard our current compiled query due to changes.
     * @see org.datanucleus.store.query.Query#discardCompiled()
     */
    protected void discardCompiled()
    {
        super.discardCompiled();
        singleString = null;
        parsedImports = null;
    }

    /**
     * Accessor for the parsed imports.
     * If no imports are set then adds candidate class, user imports, and any user-defined expression packages.
     * @return Parsed imports
     */
    public Imports getParsedImports()
    {
        if (parsedImports == null)
        {
            super.getParsedImports();

            // Add any imports for expressions etc
            Iterator it = userDefinedScalarExpressions.keySet().iterator();
            while (it.hasNext())
            {
                parsedImports.importClass((String)it.next());
            }
        }
        return parsedImports;
    }

    /**
     * Retrieve the metadata for the results 
     * @return the ResultSetMetaData
     */
    public QueryResultsMetaData getResultSetMetaData()
    {
        // TODO Set resultMetaData for generic compiler process
        if (resultMetaData == null)
        {
            throw new NucleusUserException("You must compile the query before calling this method.");
        }
        return resultMetaData;
    }

    /**
     * Execute the query to delete persistent objects.
     * @param parameters the Map containing all of the parameters.
     * @return the number of deleted objects.
     */
    protected long performDeletePersistentAll(Map parameters)
    {
        boolean requiresUnique = unique;
        try
        {
            // Make sure we have a compiled query without any "unique" setting
            if (unique)
            {
                unique = false;
                discardCompiled();
            }
            compileInternal(true, parameters);

            if (candidateCollection != null && candidateCollection.isEmpty())
            {
                // Candidates specified, but was empty so nothing to delete
                return 0;
            }

            // Execute the query to get the instances affected
            Collection results = (Collection)performExecute(parameters);
            if (results == null)
            {
                return 0;
            }
            else
            {
                // TODO : To make this most efficient we shouldn't instantiate things into memory
                // but instead check if the object to be deleted implements DeleteCallback, or there are
                // any lifecycle listeners for this object type, or if there are any dependent fields
                // and only then do we instantiate it.
                int number = results.size();
                if (requiresUnique && number > 1)
                {
                    throw new NucleusUserException(LOCALISER.msg("021032"));
                }

                // Instances to be deleted are flushed first (JDO2 [14.8-4])
                Iterator resultsIter = results.iterator();
                while (resultsIter.hasNext())
                {
                    om.findStateManager(resultsIter.next()).flush();
                }

                // Perform the deletion - deletes any dependent objects
                om.deleteObjects(results);

                if (results instanceof QueryResult)
                {
                    // Close any results that are of type QueryResult (which manage the access to the results)
                    // TODO Do we need to remove these from the PM.queryRun store and this.queryResults ?
                    ((QueryResult)results).close();
                }

                return number;
            }
        }
        finally
        {
            if (requiresUnique != unique)
            {
                // Reinstate unique setting
                unique = requiresUnique;
                discardCompiled();
            }
        }
    }

    /**
     * Accessor for a single string form of the query.
     * @return Single string form of the query.
     */
    public abstract String getSingleStringQuery();

    /**
     * Stringifier method
     * @return Single-string form of this JDOQL query.
     */
    public String toString()
    {
        return getSingleStringQuery();
    }

    /**
     * Register ScalarExpressions for the given <code>cls</code>. It allows
     * to perform operations in the query on <i>cls.method([arglist])</i>.
     * @param literal the class providing the operations; e.g. java.lang.Math.class
     * @param scalarExpressionClass the class with the corresponding ScalarExpression. eg. org.datanucleus.store.expression.MathExpression.class
     * @param name alternative name of the given literal class
     */
    public static void registerScalarExpression(Class literal, Class scalarExpressionClass, String name)
    {
        userDefinedScalarExpressions.put(name == null ? literal.getName() : name, scalarExpressionClass);
    }

    /**
     * Register ScalarExpression classes delcared as plug-ins extensions
     * TODO currently register only the first time this class is instantiated. Should be registered per PMF?
     * @param pluginMgr The PluginManager
     * @param clr The ClassLoaderResolver to load the literal and ScalarExpression classes
     */
    protected void registerScalarExpressions(PluginManager pluginMgr, ClassLoaderResolver clr)
    {
        if (userDefinedScalarExpressions.isEmpty())
        {
            Extension[] ex = pluginMgr.getExtensionPoint("org.datanucleus.store_expression_scalarexpression").getExtensions();
            for (int i=0; i<ex.length; i++)
            {
                ConfigurationElement[] confElm = ex[i].getConfigurationElements();
                for (int c=0; c<confElm.length; c++)
                {       
                    Class literalClass = null;
                    if (confElm[c].getAttribute("literal-class") != null)
                    {
                        literalClass = pluginMgr.loadClass(confElm[c].getExtension().getPlugin().getSymbolicName(),
                            confElm[c].getAttribute("literal-class"));
                    }
                    Class scalarExpression = null;
                    if (confElm[c].getAttribute("scalar-expression-class") != null)
                    {
                        scalarExpression = pluginMgr.loadClass(confElm[c].getExtension().getPlugin().getSymbolicName(),
                            confElm[c].getAttribute("scalar-expression-class"));
                    }
                    registerScalarExpression(literalClass, scalarExpression, confElm[c].getAttribute("name"));
                }
            }
        }
    }

    /**
     * Accessor for the user-defined scalar expressions.
     * @return Map of user-defined scalar expressions
     */
    public static Map getUserDefinedScalarExpressions()
    {
        return userDefinedScalarExpressions;
    }

    /**
     * Convenience method to return whether the query should be evaluated in-memory.
     * @return Use in-memory evaluation?
     */
    protected boolean evaluateInMemory()
    {
        boolean inMemory = false;
        String queryInMemoryStr = (String)getExtension("datanucleus.query.evaluateInMemory");
        if (queryInMemoryStr != null && queryInMemoryStr.equalsIgnoreCase("true"))
        {
            inMemory = true;
        }
        return inMemory;
    }
}