/**********************************************************************
Copyright (c) 2008 Andy Jefferson and others. All rights reserved. 
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.store.query;

import java.util.AbstractList;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;

import javax.jdo.JDOUserException;

import org.datanucleus.ManagedConnectionResourceListener;
import org.datanucleus.ObjectManagerFactoryImpl;
import org.datanucleus.store.query.Query;
import org.datanucleus.store.query.QueryResult;
import org.datanucleus.util.Localiser;

/**
 * Abstract representation of a QueryResult.
 * Provides default implementations of the majority of list methods that we aren't likely to
 * be providing in a concrete query result.
 * This class is used where your query implementation needs to return a wrapper to a List so that
 * you can intercept calls and convert a row of the results into object(s), to avoid full instantiation
 * at creation.
 */
public abstract class AbstractQueryResult extends AbstractList implements QueryResult, java.io.Serializable
{
    /** Localiser for messages. */
    protected static final Localiser LOCALISER = Localiser.getInstance(
        "org.datanucleus.Localisation", ObjectManagerFactoryImpl.class.getClassLoader());

    /** The Query object. */
    protected Query query;

    /** Whether the results are close. */
    protected boolean closed = false;

    /** List of listeners to notify when the query results are closed. */
    protected List connectionListeners = null;

    /**
     * Constructor of the result from a Query.
     * @param query The Query
     */
    public AbstractQueryResult(Query query)
    {
        this.query = query;
    }

    /**
     * Method to disconnect the results from the ObjectManager, meaning that thereafter it just behaves
     * like a List. All remaining results are read in at this point (unless selected not to be).
     */
    public void disconnect()
    {
        if (query == null)
        {
            // Already disconnected
            return;
        }

        // Inform that we are closing the connection, so all results are read as necessary
        closingConnection();

        // Close the result set
        closeResults();

        // Release all resources
        query = null;
    }

    /**
     * Inform the query result that the connection is being closed so perform
     * any operations now, or rest in peace.
     */
    protected abstract void closingConnection();

    /**
     * Inform the query result that we are closing the results now.
     */
    protected abstract void closeResults();

    /**
     * Method to close the results, meaning that they are inaccessible after this point.
     */
    public synchronized void close()
    {
        // Close the result set
        closeResults();

        // Release all resources
        query = null;
        closed = true;

        if (connectionListeners != null)
        {
            // Call all listeners that we are closing
            Iterator iter = connectionListeners.iterator();
            while (iter.hasNext())
            {
                ManagedConnectionResourceListener listener = (ManagedConnectionResourceListener)iter.next();
                listener.resourcePostClose();
            }
            connectionListeners.clear();
            connectionListeners = null;
        }
    }

    /**
     * Method to register a listener to be notified when the query result is closing.
     * @param listener The listener
     */
    public void addConnectionListener(ManagedConnectionResourceListener listener)
    {
        if (connectionListeners == null)
        {
            connectionListeners = new ArrayList();
        }
        connectionListeners.add(listener);
    }

    /**
     * Accessor whether the results are open.
     * @return Whether it is open.
     */
    protected boolean isOpen()
    {
        return closed == false;
    }

    /**
     * Internal method to throw an Exception if the ResultSet is open.
     */
    protected void assertIsOpen()
    {
        if (!isOpen())
        {
            // TODO Should not throw JDOException when not using JDO
            throw new JDOUserException(LOCALISER.msg("052600"));
        }
    }

    // ------------------------- Implementation of the List -------------------------

    /**
     * Method to add a result. Unsupported.
     * @param index The position to add
     * @param element The results to add
     */
    public void add(int index, Object element)
    {
        throw new UnsupportedOperationException(LOCALISER.msg("052603"));
    }

    /**
     * Method to add results. Unsupported.
     * @param o The result to add
     * @return true if added successfully
     */
    public boolean add(Object o)
    {
        throw new UnsupportedOperationException(LOCALISER.msg("052603"));
    }

    /**
     * Method to add results. Unsupported.
     * @param index The position to add
     * @param c The results to add
     * @return true if added successfully
     */
    public boolean addAll(int index, Collection c)
    {
        throw new UnsupportedOperationException(LOCALISER.msg("052603"));
    }

    /**
     * Method to clear the results.
     */
    public void clear()
    {
        throw new UnsupportedOperationException(LOCALISER.msg("052603"));
    }

    /**
     * Method to check if the specified object is contained in this result.
     * @param o The object
     * @return Whether it is contained here.
     */
    public boolean contains(Object o)
    {
        throw new UnsupportedOperationException(LOCALISER.msg("052604"));
    }

    /**
     * Method to check if all of the specified objects are contained here.
     * @param c The collection of objects
     * @return Whether they are all contained here.
     */
    public boolean containsAll(Collection c)
    {
        throw new UnsupportedOperationException(LOCALISER.msg("052604"));
    }

    /**
     * Equality operator for QueryResults.
     * Overrides the AbstractList implementation since that uses 
     * size() and iterator() and that would cause problems when closed.
     * @param o The object to compare against
     * @return Whether they are equal
     */
    public abstract boolean equals(Object o);

    /**
     * Method to retrieve a particular element from the list.
     * @param index The index of the element
     * @return The element at index
     */
    public abstract Object get(int index);

    /**
     * Accessor for the hashcode of this object
     * @return The hash code
     */
    public int hashCode()
    {
        if (query != null)
        {
            return query.hashCode();
        }
        else
        {
            // Disconnected
            return super.hashCode();
        }
    }

    /**
     * Method to check the index of a result. Not supported.
     * @param o The result
     * @return The position
     */
    public int indexOf(Object o)
    {
        throw new UnsupportedOperationException(LOCALISER.msg("052604"));
    }

    /**
     * Returns <tt>true</tt> if this collection contains no elements.<p>
     * @return <tt>true</tt> if this collection contains no elements.
     */
    public boolean isEmpty()
    {
        return size() < 1;
    }

    /**
     * Accessor for an iterator for the results.
     * @return The iterator
     */
    public abstract Iterator iterator();

    /**
     * Method to check the last index of a result. Not supported.
     * @param o The result
     * @return The last index
     */
    public int lastIndexOf(Object o)
    {
        throw new UnsupportedOperationException(LOCALISER.msg("052604"));
    }

    /**
     * Accessor for a list iterator for the results.
     * @return a ListIterator with the query results
     */
    public abstract ListIterator listIterator();

    /**
     * Method to remove a result. Not supported.
     * @param index The position of the result.
     * @return The removed object.
     */
    public Object remove(int index)
    {
        throw new UnsupportedOperationException(LOCALISER.msg("052603"));
    }

    /**
     * Method to set the position of a result. Not supported.
     * @param index Position of the result
     * @param element The result
     * @return The element
     */
    public Object set(int index, Object element)
    {
        throw new UnsupportedOperationException(LOCALISER.msg("052603"));
    }

    /**
     * Method to return the size of the result.
     * @return The size of the result.
     */
    public abstract int size();

    /**
     * Method return a sub list of results.
     * Method create new ArrayList, iterate and call get() in subclass for optimum perfomance.
     * @param fromIndex start position
     * @param toIndex end position (exclusive)
     * @return The list of results
     */
    public List subList(int fromIndex, int toIndex)
    {
        int subListLength = toIndex - fromIndex;
        ArrayList subList = new ArrayList(subListLength);
        for (int i = fromIndex; i < toIndex; i++)
        {
            subList.add(get(i));
        }
        return subList;
    }

    /**
     * Method to return the results as an array.
     * @return The array.
     */
    public Object[] toArray()
    {
        Object[] array = new Object[size()];
        for (int i = 0; i < array.length; i++)
        {
            array[i] = get(i);
        }
        return array;
    }

    /**
     * Method to return the results as an array.
     * @param a The array to copy into. 
     * @return The array.
     */
    public Object[] toArray(Object[] a)
    {
        if (a.length >= size())// collection fits in the specified array
        {
            for (int i = 0; i < a.length; i++)
            {
                if (i < size())
                {
                    a[i] = get(i);
                }
                else
                {
                    a[i] = null;
                }
            }
            return a;
        }
        return toArray(); // collection not fits in the specified array
    }
}