/**********************************************************************
Copyright (c) 2008 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.store.query;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;

import org.datanucleus.ObjectManagerFactoryImpl;
import org.datanucleus.exceptions.NucleusException;
import org.datanucleus.exceptions.NucleusUserException;
import org.datanucleus.util.Imports;
import org.datanucleus.util.Localiser;
import org.datanucleus.util.StringUtils;

/**
 * Base definition of a query compiler for a query language.
 * Should be extended to compile details of individual query languages.
 * This was designed for the old-style compilation process, and is being phased out.
 * New implementations of querying in DataNucleus should make use of compilers under
 * "org.datanucleus.query.compiler".
 */
public class QueryCompiler
{
    public static final int COMPILE_EXPLICIT_PARAMETERS = 1;
    public static final int COMPILE_EXPLICIT_VARIABLES = 2;
    public static final int COMPILE_SYNTAX = 3;
    public static final int COMPILE_EXECUTION = 4;

    /** Localiser for messages. */
    protected static final Localiser LOCALISER = Localiser.getInstance(
        "org.datanucleus.Localisation", ObjectManagerFactoryImpl.class.getClassLoader());

    /** Language of this query (e.g JDOQL, JPQL). */
    protected String language;

    /** The query being compiled. */
    protected Query query;

    /** Imports to use for the resolution of classes. */
    protected Imports imports;

    /** Map of parameter values, keyed by their name. */
    protected Map parameters;

    /** Flag for whether the current compile is for execution (using param values). */
    protected boolean executionCompile = true;

    /** Candidate class for the query. May be updated during the compilation. */
    protected Class candidateClass = null;

    /** Alias for the candidate class. */
    protected String candidateAlias = "this";

    /** The parameter names. */
    protected List parameterNames = null;

    /** Look-up for the parameter types, keyed by the name. */
    protected Map parameterTypesByName = null;

    /** List of variable names. */
    protected List variableNames = null;

    /** Look-up for the variables types, keyed by the name. */
    protected Map variableTypesByName = null;

    /**
     * Constructor for a compiler of java queries.
     * @param query The query to compile
     * @param imports The imports to use
     * @param parameters Any parameters
     */
    public QueryCompiler(Query query, Imports imports, Map parameters)
    {
        this.query = query;
        this.imports = imports;
        this.parameters = parameters;
        this.candidateClass = query.getCandidateClass();
    }

    /**
     * Method to close the Compiler.
     */
    public void close()
    {
        this.query = null;
        this.imports = null;
        this.variableNames = null;
        this.parameterNames = null;
        this.variableTypesByName = null;
        this.parameterTypesByName = null;
        this.parameters = null;
    }

    /**
     * Method to compile the query.
     * @param type Type of compilation. This compiler only supports explicit parameters, explicit variables
     * @return the compilation artifact (if any)
     */
    public Object compile(int type)
    {
        switch (type)
        {
            case COMPILE_EXPLICIT_PARAMETERS :
                compileExplicitParameters();
                return null;

            case COMPILE_EXPLICIT_VARIABLES :
                compileExplicitVariables();
                return null;

            default :
                throw new NucleusException("Query Compiler doesnt support compilation of type " + type);
        }
    }

    /**
     * Accessor for the candidate class. May have been updated during the compile process.
     * @return Candidate class
     */
    public Class getCandidateClass()
    {
        return candidateClass;
    }

    /**
     * Accessor for the candidate alias.
     * @return Candidate alias
     */
    public String getCandidateAlias()
    {
        return candidateAlias;
    }

    /**
     * Accessor for the (explicit) parameter names.
     * @return Parameter names
     */
    public String[] getParameterNames()
    {
        if (parameterNames == null)
        {
            return null;
        }
        return (String[])parameterNames.toArray(new String[parameterNames.size()]);
    }

    /**
     * Accessor for the parameter types keyed by the parameter name.
     * Generated during compile of explicit parameters.
     * @return Map of parameter type keyed by name
     */
    public Map getParameterTypesByName()
    {
        return parameterTypesByName;
    }

    /**
     * Method to compile all parameters declared for this query.
     * Takes the input "parameters" and populates "parameterNames", "parameterTypesByName" for convenience.
     */
    protected void compileExplicitParameters()
    {
        parameterNames = new ArrayList();
        parameterTypesByName = new HashMap();
        String explicitParameters = query.getExplicitParameters();
        if (explicitParameters != null && explicitParameters.length() > 0)
        {
            // Explicit parameters defined, so validate them
            StringTokenizer t1 = new StringTokenizer(explicitParameters, ",");
            while (t1.hasMoreTokens())
            {
                StringTokenizer t2 = new StringTokenizer(t1.nextToken(), " ");
                if (t2.countTokens() != 2)
                {
                    throw new NucleusUserException(LOCALISER.msg("021101", explicitParameters));
                }

                String classDecl = t2.nextToken();
                String parameterName = t2.nextToken();
                if (!StringUtils.isValidJavaIdentifierForJDOQL(parameterName))
                {
                    throw new NucleusUserException(LOCALISER.msg("021102",parameterName));
                }

                if (parameterNames.contains(parameterName))
                {
                    throw new NucleusUserException(LOCALISER.msg("021103",parameterName));
                }

                parameterNames.add(parameterName);
                parameterTypesByName.put(parameterName, query.resolveClassDeclaration(classDecl));
            }
        }
    }

    /**
     * Method to compile all variables declared for this query.
     * Takes the input "variables" and populates "variableNames", "variableTypesByName" for convenience.
     */
    protected void compileExplicitVariables()
    {
        variableNames = new ArrayList();
        variableTypesByName = new HashMap();

        String explicitVariables = query.getExplicitVariables();
        if (explicitVariables != null && explicitVariables.length() > 0)
        {
            // Explicit variables defined, so validate them
            StringTokenizer t1 = new StringTokenizer(explicitVariables, ";");
            while (t1.hasMoreTokens())
            {
                StringTokenizer t2 = new StringTokenizer(t1.nextToken(), " ");
                if (t2.countTokens() != 2)
                {
                    throw new NucleusUserException(LOCALISER.msg("021104", explicitVariables));
                }

                String classDecl = t2.nextToken();
                String variableName = t2.nextToken();
                if (!StringUtils.isValidJavaIdentifierForJDOQL(variableName))
                {
                    throw new NucleusUserException(LOCALISER.msg("021105",variableName));
                }

                if (parameterNames.contains(variableName))
                {
                    throw new NucleusUserException(LOCALISER.msg("021106",variableName));
                }

                if (variableNames.contains(variableName))
                {
                    throw new NucleusUserException(LOCALISER.msg("021107",variableName));
                }

                variableNames.add(variableName);
                variableTypesByName.put(variableName, query.resolveClassDeclaration(classDecl));
            }
        }
    }
}