/**********************************************************************
Copyright (c) 2007 Erik Bengtson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
2008 Andy Jefferson - check on datastore when getting query support
2008 Andy Jefferson - query cache
     ...
***********************************************************************/
package org.datanucleus.store.query;

import java.lang.reflect.InvocationTargetException;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.WeakHashMap;

import org.datanucleus.OMFContext;
import org.datanucleus.ObjectManager;
import org.datanucleus.exceptions.NucleusException;
import org.datanucleus.management.ManagementServer;
import org.datanucleus.management.runtime.QueryRuntime;
import org.datanucleus.query.compiler.QueryCompilation;
import org.datanucleus.query.evaluator.memory.InvocationEvaluator;
import org.datanucleus.util.ClassUtils;
import org.datanucleus.util.NucleusLogger;

/**
 * Manages the runtime, metadata and lifecycle of queries.
 * Provides caching of query compilations.
 */
public class QueryManager
{
    OMFContext omfCtx;

    // TODO Make this pluggable and consider merging with L2 cache
    /** Cache of compiled queries, keyed by the string form of the query. */
    Map queryCompilationCache = new WeakHashMap();

    /** Cache of InvocationEvaluator objects keyed by the method name. */
    Map queryMethodEvaluatorMap = new HashMap();

    /** Query Runtime. Used when providing management of services. */
    QueryRuntime queryRuntime = null;

    public QueryManager(OMFContext omfContext)
    {
        this.omfCtx = omfContext;
        if (omfContext.getManagement() != null)
        {
            // register MBean in MbeanServer
            ManagementServer mgntServer = omfContext.getManagement().getManagementServer();
            queryRuntime = new QueryRuntime();
            String mbeanName = omfContext.getDomainName() + ":InstanceName=" + omfContext.getInstanceName() +
                ",Type=" + ClassUtils.getClassNameForClass(queryRuntime.getClass())+
                ",Name=QueryRuntime";
            mgntServer.registerMBean(this.queryRuntime, mbeanName);
        }
    }

    public void close()
    {
        queryCompilationCache.clear();
        queryCompilationCache = null;
        queryRuntime = null;
    }

    public QueryRuntime getQueryRuntime()
    {
        return queryRuntime;
    }
    
    /**
     * Method to generate a new query using the passed query as basis.
     * @param language The query language
     * @param om The Object Manager
     * @param query The query filter (String) or a previous Query
     * @return The Query
     */
    public Query newQuery(String language, ObjectManager om, Object query)
    {
        if (language == null)
        {
            return null;
        }

        // Find the query support for this language and this datastore
        try
        {
            if (query == null)
            {
                Class[] argsClass = new Class[] {org.datanucleus.ObjectManager.class};
                Object[] args = new Object[] {om};
                Query q = (Query) om.getOMFContext().getPluginManager().createExecutableExtension(
                    "org.datanucleus.store_query_query", new String[] {"name", "datastore"},
                    new String[] {language, om.getStoreManager().getStoreManagerKey()}, 
                    "class-name", argsClass, args);
                if (q == null)
                {
                    // No query support for this language
                    throw new NucleusException("DataNucleus doesnt currently support queries of language " + language + " for this datastore");
                }
                return q;
            }
            else
            {
                Query q = null;
                if (query instanceof String)
                {
                    // Try XXXQuery(ObjectManager, String);
                    Class[] argsClass = new Class[]{org.datanucleus.ObjectManager.class, String.class};
                    Object[] args = new Object[]{om, query};
                    q = (Query) om.getOMFContext().getPluginManager().createExecutableExtension(
                        "org.datanucleus.store_query_query", new String[] {"name", "datastore"},
                        new String[] {language, om.getStoreManager().getStoreManagerKey()}, 
                        "class-name", argsClass, args);
                    if (q == null)
                    {
                        // No query support for this language
                        throw new NucleusException("DataNucleus doesnt currently support queries of language " + language + " for this datastore");
                    }
                }
                else if (query instanceof Query)
                {
                    // Try XXXQuery(ObjectManager, query.class);
                    Class[] argsClass = new Class[]{org.datanucleus.ObjectManager.class, query.getClass()};
                    Object[] args = new Object[]{om, query};
                    q = (Query) om.getOMFContext().getPluginManager().createExecutableExtension(
                        "org.datanucleus.store_query_query", new String[] {"name", "datastore"},
                        new String[] {language, om.getStoreManager().getStoreManagerKey()}, 
                        "class-name", argsClass, args);
                    if (q == null)
                    {
                        // No query support for this language
                        throw new NucleusException("DataNucleus doesnt currently support queries of language " + language + " for this datastore");
                    }
                }
                else
                {
                    // Try XXXQuery(ObjectManager, Object);
                    Class[] argsClass = new Class[]{org.datanucleus.ObjectManager.class, Object.class};
                    Object[] args = new Object[]{om, query};
                    q = (Query) om.getOMFContext().getPluginManager().createExecutableExtension(
                        "org.datanucleus.store_query_query", new String[] {"name", "datastore"},
                        new String[] {language, om.getStoreManager().getStoreManagerKey()}, 
                        "class-name", argsClass, args);
                    if (q == null)
                    {
                        // No query support for this language
                        throw new NucleusException("DataNucleus doesnt currently support queries of language " + language + " for this datastore");
                    }
                }
                return q;
            }
        }
        catch (InvocationTargetException e)
        {
            Throwable t = e.getTargetException();
            if (t instanceof RuntimeException)
            {
                throw (RuntimeException) t;
            }
            else if (t instanceof Error)
            {
                throw (Error) t;
            }
            else
            {
                throw new NucleusException(t.getMessage(), t).setFatal();
            }
        }
        catch (Exception e)
        {
            throw new NucleusException(e.getMessage(), e).setFatal();
        }
    }

    /**
     * Accessor for a Query compilation for the specified query and language.
     * @param language Language of the query
     * @param query Query string
     * @return The compilation (if present)
     */
    public QueryCompilation getQueryCompilationForQuery(String language, String query)
    {
        return (QueryCompilation)queryCompilationCache.get(language + ":" + query);
    }

    /**
     * Method to store the compilation for a query.
     * @param language Language of the query
     * @param query The query string
     * @param compilation The compilation of this query
     */
    public void addQueryCompilation(String language, String query, QueryCompilation compilation)
    {
        queryCompilationCache.put(language + ":" + query, compilation);
    }

    /**
     * Accessor for an evaluator for invocation of the specified method.
     * If it is not a supported method then returns null.
     * @param methodName Name of the method
     * @return Evaluator suitable for this type with this method name
     */
    public InvocationEvaluator getInMemoryEvaluatorForMethod(Class type, String methodName)
    {
        Collection methods = (Collection)queryMethodEvaluatorMap.get(methodName);
        if (methods != null)
        {
            Iterator evaluatorIter = methods.iterator();
            while (evaluatorIter.hasNext())
            {
                InvocationEvaluator eval = (InvocationEvaluator)evaluatorIter.next();
                if (eval.supportsType(type))
                {
                    return eval;
                }
            }
        }
        else
        {
            // Not yet loaded for this method
            String[] evaluatorNames = omfCtx.getPluginManager().getAttributeValuesForExtension(
                "org.datanucleus.store_query_methods", "name", methodName, "memory-evaluator");
            if (evaluatorNames != null)
            {
                Collection evaluators = new HashSet();
                InvocationEvaluator evalToUse = null;
                for (int i=0;i<evaluatorNames.length;i++)
                {
                    try
                    {
                        Class evalCls = omfCtx.getClassLoaderResolver(null).classForName(evaluatorNames[i]);
                        InvocationEvaluator eval = (InvocationEvaluator)evalCls.newInstance();
                        evaluators.add(eval);
                        if (eval.supportsType(type))
                        {
                            evalToUse = eval;
                        }
                    }
                    catch (Exception e)
                    {
                        NucleusLogger.QUERY.warn("Extension org.datanucleus.store_query_methods has method=" + 
                            methodName + " referencing evaluator " + evaluatorNames[i] + 
                            " but an error occurred in construction : " + e.getMessage());
                    }
                }
                queryMethodEvaluatorMap.put(methodName, evaluators);
                return evalToUse;
            }
        }
        return null;
    }
}