/**********************************************************************
Copyright (c) 2002 Mike Martin (TJDO) and others. All rights reserved. 
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License. 
 

Contributors:
2003 Andy Jefferson - coding standards
2004 Andy Jefferson - added query methods and comments
2004 Erik Bengtson - added joinKeysValuesTo
2004 Erik Bengtson - added joinKeysToGet
    ...
**********************************************************************/
package org.datanucleus.store.scostore;

import java.util.Map;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.StateManager;

/**
 * Interface representation of the backing store for a Map.
 **/
public interface MapStore extends Store
{
    // --------------------------- Accessor Methods ----------------------------

    /**
     * Accessor for the key type in the collection.
     * @return The key type.
     **/
    String getKeyType();

    /**
     * Accessor for the value type in the collection.
     * @return The value type.
     **/
    String getValueType();

    /**
     * Accessor for whether the keys are embedded
     * @return Whether we have embedded keys
     */
    boolean keysAreEmbedded();

    /**
     * Accessor for whether the keys are serialised
     * @return Whether we have serialised keys
     */
    boolean keysAreSerialised();

    /**
     * Accessor for whether the values are embedded
     * @return Whether we have embedded values
     */
    boolean valuesAreEmbedded();

    /**
     * Accessor for whether the values are serialised
     * @return Whether we have serialised values
     */
    boolean valuesAreSerialised();

    // -------------------------------- Map Methods ----------------------------
 
    /**
     * Accessor for whether the Map contains this value.
     * @param ownerSM State Manager for the Map.
     * @param value The value to check
     * @return Whether it is contained.
     **/
    boolean containsValue(StateManager ownerSM, Object value);

    /**
     * Accessor for whether the Map contains this key.
     * @param ownerSM State Manager for the Map.
     * @param key The key to check
     * @return Whether it is contained.
     **/
    boolean containsKey(StateManager ownerSM, Object key);

    /**
     * Accessor for a value from the Map.
     * @param ownerSM State Manager for the Map. 
     * @param key Key for the value.
     * @return Value for this key.
     **/
    Object get(StateManager ownerSM, Object key);

    /**
     * Method to add a value to the Map against this key.
     * @param ownerSM State Manager for the Map. 
     * @param key The key.
     * @param value The value.
     * @return Value that was previously against this key.
     **/
    Object put(StateManager ownerSM, Object key, Object value);

    /**
     * Method to add a map of values to the Map.
     * @param ownerSM State Manager for the Map. 
     * @param m The map to add.
     **/ 
    void putAll(StateManager ownerSM, Map m);

    /**
     * Method to remove a value from the Map.
     * @param ownerSM State Manager for the Map. 
     * @param key Key whose value is to be removed.
     * @return Value that was removed.
     **/
    Object remove(StateManager ownerSM, Object key);

    /**
     * Method to clear the collection.
     * @param ownerSM State Manager for the collection. 
     **/
    void clear(StateManager ownerSM);

    /**
     * Accessor for the keys in the Map.
     * @param clr The ClassLoaderResolver
     * @return Keys for the Map.
     **/
    SetStore keySetStore(ClassLoaderResolver clr);

    /**
     * Accessor for the values in the Map.
     * @param clr The ClassLoaderResolver
     * @return Values for the Map.
     **/
    SetStore valueSetStore(ClassLoaderResolver clr);

    /**
     * Accessor for the entry set for the Map.
     * @return Entry set for the Map.
     **/
    SetStore entrySetStore();

    /**
     * Method to update en embedded key in the collection.
     * @param sm State Manager of the owner
     * @param key The element
     * @param fieldNumber Field to update in the key
     * @param newValue The new value for the field
     * @return Whether the element was modified
     */
    boolean updateEmbeddedKey(StateManager sm, Object key, int fieldNumber, Object newValue);

    /**
     * Method to update en embedded value in the collection.
     * @param sm State Manager of the owner
     * @param value The element
     * @param fieldNumber Field to update in the value
     * @param newValue The new value for the field
     * @return Whether the element was modified
     */
    boolean updateEmbeddedValue(StateManager sm, Object value, int fieldNumber, Object newValue);
}