/**********************************************************************
Copyright (c) 2004 Andy Jefferson and others. All rights reserved. 
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.util;

/**
 * Logging framework for DataNucleus. Allows use of Log4J, JDK1.4, or no logging.
 * Performs a similar role to Apache CommonsLogging yet doesn't need an extra jar to be present in 
 * the CLASSPATH and also allows for no available logger.
 */
public abstract class NucleusLogger
{
    /** Implementation of NucleusLogger providing the logger. */
    private static Class LOGGER_CLASS = null;

    /** Log for JDO issues */
    public static final NucleusLogger JDO;

    /** Log for JPA issues */
    public static final NucleusLogger JPA;

    /** Log for Persistence issues */
    public static final NucleusLogger PERSISTENCE;

    /** Log for Lifecycle issues */
    public static final NucleusLogger LIFECYCLE;

    /** Log for Query issues */
    public static final NucleusLogger QUERY;

    /** Log for REACHABILITY issues */
    public static final NucleusLogger REACHABILITY;

    /** Log for METADATA issues */
    public static final NucleusLogger METADATA;

    /** Log for MANAGEMENT issues */
    public static final NucleusLogger MANAGEMENT;
    
    /** Log for Cache issues */
    public static final NucleusLogger CACHE;

    /** Log for General issues */
    public static final NucleusLogger GENERAL;

    /** Log for Transaction issues */
    public static final NucleusLogger TRANSACTION;
    
    /** Log for Connection issues */
    public static final NucleusLogger CONNECTION;    

    /** Log for JCA issues */
    public static final NucleusLogger JCA;   
    
    /** Log for ClassLoading issues */
    public static final NucleusLogger CLASSLOADING;

    /** Log for PLUGIN issues */
    public static final NucleusLogger PLUGIN;

    /** Log for value generation issues */
    public static final NucleusLogger VALUEGENERATION;

    /** Log for javax.naming issues */
    public static final NucleusLogger NAMING;

    /** Log for Datastore issues */
    public static final NucleusLogger DATASTORE;

    /** Log for Datastore persistence issues */
    public static final NucleusLogger DATASTORE_PERSIST;

    /** Log for Datastore retrieval issues */
    public static final NucleusLogger DATASTORE_RETRIEVE;

    /** Log for Datastore Schema issues */
    public static final NucleusLogger DATASTORE_SCHEMA;

    /** Log for SchemaTool */
    public static final NucleusLogger SCHEMATOOL;

    static
    {
        // Set the log type to be used based on what is available from this ClassLoader
        // Note that we could have registered in the PluginManager but that needs to log too
        Class loggerClass = org.datanucleus.util.NullLogger.class;
        try
        {
            NucleusLogger.class.getClassLoader().loadClass("org.apache.log4j.Logger");
            loggerClass = org.datanucleus.util.Log4JLogger.class;
        }
        catch (Exception e)
        {
            if (JavaUtils.isJRE1_4OrAbove())
            {
                loggerClass = org.datanucleus.util.JDK14Logger.class;
            }
        }
        LOGGER_CLASS = loggerClass;

        // Create the Loggers for our predefined categories
        JDO = getLoggerInstance("DataNucleus.JDO");
        JPA = getLoggerInstance("DataNucleus.JPA");
        PERSISTENCE = getLoggerInstance("DataNucleus.Persistence");
        LIFECYCLE = getLoggerInstance("DataNucleus.Lifecycle");
        QUERY = getLoggerInstance("DataNucleus.Query");
        REACHABILITY = getLoggerInstance("DataNucleus.Reachability");
        METADATA = getLoggerInstance("DataNucleus.MetaData");
        CACHE = getLoggerInstance("DataNucleus.Cache");
        GENERAL = getLoggerInstance("DataNucleus.General");
        TRANSACTION = getLoggerInstance("DataNucleus.Transaction");
        PLUGIN = getLoggerInstance("DataNucleus.Plugin");
        VALUEGENERATION = getLoggerInstance("DataNucleus.Store.Poid");
        CLASSLOADING = getLoggerInstance("DataNucleus.ClassLoading");
        NAMING = getLoggerInstance("DataNucleus.Naming");
        MANAGEMENT = getLoggerInstance("DataNucleus.Management");
        CONNECTION = getLoggerInstance("DataNucleus.Connection");
        JCA = getLoggerInstance("DataNucleus.JCA");

        DATASTORE = getLoggerInstance("DataNucleus.Datastore");
        DATASTORE_PERSIST = getLoggerInstance("DataNucleus.Datastore.Persist");
        DATASTORE_RETRIEVE = getLoggerInstance("DataNucleus.Datastore.Retrieve");
        DATASTORE_SCHEMA = getLoggerInstance("DataNucleus.Datastore.Schema");

        SCHEMATOOL = getLoggerInstance("DataNucleus.SchemaTool");
    }

    /**
     * Method to create a logger instance.
     * @param logCategory The category (or null)
     * @return The logger
     */
    public static NucleusLogger getLoggerInstance(String logCategory)
    {
        return (NucleusLogger)ClassUtils.newInstance(LOGGER_CLASS, 
            new Class[] {String.class}, new Object[] {logCategory});
    }

    /**
     * Log a debug message.
     * @param msg The message
     */
    public abstract void debug(Object msg);

    /**
     * Log a debug message with throwable.
     * @param msg The message
     * @param thr A throwable
     */
    public abstract void debug(Object msg, Throwable thr);

    /**
     * Log an info message.
     * @param msg The message
     */
    public abstract void info(Object msg);

    /**
     * Log an info message with throwable.
     * @param msg The message
     * @param thr A throwable
     */
    public abstract void info(Object msg, Throwable thr);

    /**
     * Log a warning message.
     * @param msg The message
     */
    public abstract void warn(Object msg);

    /**
     * Log a warning message with throwable.
     * @param msg The message
     * @param thr A throwable
     */
    public abstract void warn(Object msg, Throwable thr);

    /**
     * Log an error message.
     * @param msg The message
     */
    public abstract void error(Object msg);

    /**
     * Log an error message with throwable.
     * @param msg The message
     * @param thr A throwable
     */
    public abstract void error(Object msg, Throwable thr);

    /**
     * Log a fatal message.
     * @param msg The message
     */
    public abstract void fatal(Object msg);

    /**
     * Log a fatal message with throwable.
     * @param msg The message
     * @param thr A throwable
     */
    public abstract void fatal(Object msg, Throwable thr);

    /**
     * Accessor for whether debug logging is enabled
     * @return Whether it is enabled
     */
    public abstract boolean isDebugEnabled();

    /**
     * Accessor for whether info logging is enabled
     * @return Whether it is enabled
     */
    public abstract boolean isInfoEnabled();
}