/**********************************************************************
Copyright (c) 2006 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.jdo.metadata;

import org.xml.sax.Attributes;
import org.xml.sax.EntityResolver;
import org.xml.sax.SAXException;

import org.datanucleus.metadata.AbstractClassMetaData;
import org.datanucleus.metadata.AbstractElementMetaData;
import org.datanucleus.metadata.AbstractMemberMetaData;
import org.datanucleus.metadata.ArrayMetaData;
import org.datanucleus.metadata.ClassMetaData;
import org.datanucleus.metadata.ClassPersistenceModifier;
import org.datanucleus.metadata.CollectionMetaData;
import org.datanucleus.metadata.ColumnMetaData;
import org.datanucleus.metadata.DiscriminatorMetaData;
import org.datanucleus.metadata.ElementMetaData;
import org.datanucleus.metadata.EmbeddedMetaData;
import org.datanucleus.metadata.FetchGroupMetaData;
import org.datanucleus.metadata.FetchPlanMetaData;
import org.datanucleus.metadata.FieldMetaData;
import org.datanucleus.metadata.FileMetaData;
import org.datanucleus.metadata.ForeignKeyAction;
import org.datanucleus.metadata.ForeignKeyMetaData;
import org.datanucleus.metadata.IdentityMetaData;
import org.datanucleus.metadata.IdentityStrategy;
import org.datanucleus.metadata.IdentityType;
import org.datanucleus.metadata.ImplementsMetaData;
import org.datanucleus.metadata.IndexMetaData;
import org.datanucleus.metadata.IndexedValue;
import org.datanucleus.metadata.InheritanceMetaData;
import org.datanucleus.metadata.InterfaceMetaData;
import org.datanucleus.metadata.JoinMetaData;
import org.datanucleus.metadata.KeyMetaData;
import org.datanucleus.metadata.MapMetaData;
import org.datanucleus.metadata.MetaData;
import org.datanucleus.metadata.MetaDataManager;
import org.datanucleus.metadata.MetadataFileType;
import org.datanucleus.metadata.NullValue;
import org.datanucleus.metadata.OrderMetaData;
import org.datanucleus.metadata.PackageMetaData;
import org.datanucleus.metadata.PrimaryKeyMetaData;
import org.datanucleus.metadata.PropertyMetaData;
import org.datanucleus.metadata.QueryMetaData;
import org.datanucleus.metadata.SequenceMetaData;
import org.datanucleus.metadata.UniqueMetaData;
import org.datanucleus.metadata.ValueMetaData;
import org.datanucleus.metadata.VersionMetaData;
import org.datanucleus.metadata.xml.AbstractMetaDataHandler;
import org.datanucleus.util.NucleusLogger;

/**
 * Parser handler for JDO MetaData.
 * Implements DefaultHandler and handles the extracting of MetaData for JDO
 * from the XML elements/attributes. This class simply constructs the MetaData
 * representation mirroring what is in the MetaData file. It has no knowledge
 * of the class(es) that it represents, simply the information in the MetaData
 * file. The knowledge of the classes is imposed on the representation at a
 * later stage where necessary.
 * <P>Operates the parse process using a Stack. MetaData components are added
 * to the stack as they are encountered and created. They are then popped off
 * the stack when the end element is encountered.</P>
 */
public class JDOMetaDataHandler extends AbstractMetaDataHandler
{
    /**
     * Constructor. Protected to prevent instantiation.
     * @param mgr the metadata manager
     * @param filename The name of the file to parse
     * @param resolver Entity Resolver to use (null if not available)
     */
    public JDOMetaDataHandler(MetaDataManager mgr, String filename, EntityResolver resolver)
    {
        super(mgr, filename, resolver);
        metadata = new FileMetaData();
        ((FileMetaData)metadata).setFilename(filename);
        ((FileMetaData)metadata).setMetaDataManager(mgr); // Parsing via this manager, so set it
        pushStack(metadata);
    }
    
    /**
     * Utility to create a new class component.
     * @param pmd The parent PackageMetaData
     * @param attrs The attributes
     * @return The ClassMetaData
     */
    protected ClassMetaData newClassObject(PackageMetaData pmd, Attributes attrs)
    {
        ClassMetaData cmd = new ClassMetaData(pmd, getAttr(attrs,"name"));
        cmd.setTable(getAttr(attrs,"table"));
        cmd.setCatalog(getAttr(attrs,"catalog"));
        cmd.setSchema(getAttr(attrs,"schema"));
        cmd.setRequiresExtent(getAttr(attrs,"requires-extent"));
        cmd.setDetachable(getAttr(attrs,"detachable"));
        cmd.setObjectIdClass(getAttr(attrs,"objectid-class"));
        cmd.setEmbeddedOnly(getAttr(attrs,"embedded-only"));
        cmd.setPersistenceModifier(
            ClassPersistenceModifier.getClassPersistenceModifier(getAttr(attrs,"persistence-modifier")));
        cmd.setIdentityType(IdentityType.getIdentityType(getAttr(attrs,"identity-type")));
        cmd.setPersistenceCapableSuperclass(getAttr(attrs,"persistence-capable-superclass"));
        String cacheableAttr = getAttr(attrs, "cacheable");
        if (cacheableAttr != null)
        {
            cmd.setCacheable(cacheableAttr.equalsIgnoreCase("false") ? false : true);
        }
        String serializeReadAttr = getAttr(attrs, "serialize-read");
        if (serializeReadAttr != null)
        {
            cmd.setSerializeRead(serializeReadAttr.equalsIgnoreCase("true") ? true : false);
        }
        return cmd;
    }

    /**
     * Utility to create a new interface component.
     * @param pmd The parent PackageMetaData
     * @param attrs The attributes
     * @return The InterfaceMetaData
     */
    protected InterfaceMetaData newInterfaceObject(PackageMetaData pmd, Attributes attrs)
    {
        InterfaceMetaData imd = new InterfaceMetaData(pmd, getAttr(attrs, "name"));
        imd.setTable(getAttr(attrs,"table"));
        imd.setCatalog(getAttr(attrs,"catalog"));
        imd.setSchema(getAttr(attrs,"schema"));
        imd.setDetachable(getAttr(attrs, "detachable"));
        imd.setRequiresExtent(getAttr(attrs, "requires-extent"));
        imd.setObjectIdClass(getAttr(attrs, "objectid-class"));
        imd.setEmbeddedOnly(getAttr(attrs, "embedded-only"));
        imd.setIdentityType(IdentityType.getIdentityType(getAttr(attrs, "identity-type")));
        imd.setPersistenceModifier(ClassPersistenceModifier.PERSISTENCE_CAPABLE);
        String cacheableAttr = getAttr(attrs, "cacheable");
        if (cacheableAttr != null)
        {
            imd.setCacheable(cacheableAttr.equalsIgnoreCase("false") ? false : true);
        }
        return imd;
    }

    /**
     * Utility to create a new field component.
     * @param md The parent MetaData
     * @param attrs The attributes
     * @return The FieldMetaData
     */
    protected FieldMetaData newFieldObject(MetaData md, Attributes attrs)
    {
        FieldMetaData fmd = new FieldMetaData(md, getAttr(attrs,"name"));
        fmd.setPersistenceModifier(getAttr(attrs,"persistence-modifier"));
        fmd.setDeleteAction(getAttr(attrs,"delete-action"));
        fmd.setPrimaryKey(getAttr(attrs,"primary-key"));
        fmd.setDefaultFetchGroup(getAttr(attrs,"default-fetch-group"));
        fmd.setEmbedded(getAttr(attrs,"embedded"));
        fmd.setSerialised(getAttr(attrs,"serialized"));
        fmd.setDependent(getAttr(attrs,"dependent"));
        fmd.setNullValue(NullValue.getNullValue(getAttr(attrs,"null-value")));
        fmd.setMappedBy(getAttr(attrs,"mapped-by"));
        fmd.setColumn(getAttr(attrs,"column"));
        fmd.setIndexed(IndexedValue.getIndexedValue(getAttr(attrs,"indexed")));
        fmd.setUnique(getAttr(attrs,"unique"));
        fmd.setTable(getAttr(attrs,"table"));
        fmd.setLoadFetchGroup(getAttr(attrs,"load-fetch-group"));
        fmd.setRecursionDepth(getAttr(attrs,"recursion-depth"));
        fmd.setValueStrategy(getAttr(attrs,"value-strategy"));
        fmd.setSequence(getAttr(attrs,"sequence"));
        fmd.setFieldTypes(getAttr(attrs,"field-type"));
        String cacheableAttr = getAttr(attrs, "cacheable");
        if (cacheableAttr != null)
        {
            fmd.setCacheable(cacheableAttr.equalsIgnoreCase("false") ? false : true);
        }
        return fmd;
    }

    /**
     * Utility to create a new property component.
     * @param md The parent MetaData
     * @param attrs The attributes
     * @return The PropertyMetaData
     */
    protected PropertyMetaData newPropertyObject(MetaData md, Attributes attrs)
    {
        PropertyMetaData pmd = new PropertyMetaData(md, getAttr(attrs,"name"));
        pmd.setPersistenceModifier(getAttr(attrs,"persistence-modifier"));
        pmd.setDeleteAction(getAttr(attrs,"delete-action"));
        pmd.setPrimaryKey(getAttr(attrs,"primary-key"));
        pmd.setDefaultFetchGroup(getAttr(attrs,"default-fetch-group"));
        pmd.setEmbedded(getAttr(attrs,"embedded"));
        pmd.setSerialised(getAttr(attrs,"serialized"));
        pmd.setDependent(getAttr(attrs,"dependent"));
        pmd.setNullValue(NullValue.getNullValue(getAttr(attrs,"null-value")));
        pmd.setMappedBy(getAttr(attrs,"mapped-by"));
        pmd.setColumn(getAttr(attrs,"column"));
        pmd.setIndexed(IndexedValue.getIndexedValue(getAttr(attrs,"indexed")));
        pmd.setUnique(getAttr(attrs,"unique"));
        pmd.setTable(getAttr(attrs, "table"));
        pmd.setLoadFetchGroup(getAttr(attrs,"load-fetch-group"));
        pmd.setRecursionDepth(getAttr(attrs,"recursion-depth"));
        pmd.setValueStrategy(getAttr(attrs,"value-strategy"));
        pmd.setSequence(getAttr(attrs,"sequence"));
        pmd.setFieldTypes(getAttr(attrs,"field-type"));
        pmd.setFieldName(getAttr(attrs,"field-name"));
        String cacheableAttr = getAttr(attrs, "cacheable");
        if (cacheableAttr != null)
        {
            pmd.setCacheable(cacheableAttr.equalsIgnoreCase("false") ? false : true);
        }
        return pmd;
    }    
    
    /**
     * Handler method called at the start of an element.
     * @param uri URI of the tag
     * @param localName Local name
     * @param qName Element name
     * @param attrs Attributes for this element 
     * @throws SAXException in parsing errors
     */
    public void startElement(String uri, String localName, String qName, Attributes attrs)
    throws SAXException 
    {
        if (charactersBuffer.length() > 0)
        {
            // Cater for subelements that appear before the end of the body text (save the body text with the parent)
            String currentString = getString().trim();
            if (getStack() instanceof QueryMetaData)
            {
                ((QueryMetaData)getStack()).setQuery(currentString.trim());
            }
        }

        if (localName.length()<1)
        {
            localName = qName;
        }
        try
        {
            if (localName.equals("jdo"))
            {
                FileMetaData filemd = (FileMetaData)getStack();
                filemd.setType(MetadataFileType.JDO_FILE);
                filemd.setCatalog(getAttr(attrs, "catalog"));
                filemd.setSchema(getAttr(attrs, "schema"));
            }
            else if (localName.equals("orm"))
            {
                FileMetaData filemd = (FileMetaData)getStack();
                filemd.setType(MetadataFileType.JDO_ORM_FILE);
                filemd.setCatalog(getAttr(attrs, "catalog"));
                filemd.setSchema(getAttr(attrs, "schema"));
            }
            else if (localName.equals("jdoquery"))
            {
                FileMetaData filemd = (FileMetaData)getStack();
                filemd.setType(MetadataFileType.JDO_QUERY_FILE);
            }
            else if (localName.equals("fetch-plan"))
            {
                FileMetaData filemd = (FileMetaData)metadata;
                FetchPlanMetaData fpmd = filemd.newFetchPlanMetadata(getAttr(attrs, "name"));
                fpmd.setMaxFetchDepth(getAttr(attrs, "max-fetch-depth"));
                fpmd.setFetchSize(getAttr(attrs, "fetch-size"));
                pushStack(fpmd);
            }
            else if (localName.equals("package"))
            {
                FileMetaData filemd = (FileMetaData)getStack();
                PackageMetaData pmd = filemd.newPackageMetadata(getAttr(attrs, "name"));
                pmd.setCatalog(getAttr(attrs, "catalog"));
                pmd.setSchema(getAttr(attrs, "schema"));
                pushStack(pmd);
            }
            else if (localName.equals("class"))
            {
                PackageMetaData pmd = (PackageMetaData)getStack();
                ClassMetaData cmd = newClassObject(pmd,attrs);
                pmd.addClass(cmd);
    
                pushStack(cmd);
            }
            else if (localName.equals("interface"))
            {
                PackageMetaData pmd = (PackageMetaData)getStack();
                InterfaceMetaData imd = newInterfaceObject(pmd, attrs);
                pmd.addInterface(imd);
                pushStack(imd);
            }
            else if (localName.equals("primary-key"))
            {
                MetaData md = getStack();
                PrimaryKeyMetaData pkmd = new PrimaryKeyMetaData();
                pkmd.setName(getAttr(attrs,"name"));
                pkmd.setColumnName(getAttr(attrs, "column"));
                if (md instanceof AbstractClassMetaData)
                {
                    ((AbstractClassMetaData)md).setPrimaryKeyMetaData(pkmd);
                }
                else if (md instanceof JoinMetaData)
                {
                    ((JoinMetaData)md).setPrimaryKeyMetaData(pkmd);
                }
                pushStack(pkmd);
            }
            else if (localName.equals("implements"))
            {
                ClassMetaData cmd = (ClassMetaData)getStack();
                ImplementsMetaData imd = new ImplementsMetaData(getAttr(attrs,"name"));
                cmd.addImplements(imd);
                pushStack(imd);
            }
            else if (localName.equals("property"))
            {
                MetaData parent = getStack();
                if (parent instanceof AbstractClassMetaData)
                {
                    AbstractClassMetaData acmd = (AbstractClassMetaData)parent;
                    PropertyMetaData propmd = newPropertyObject(acmd, attrs);
                    acmd.addMember(propmd);
                    pushStack(propmd);
                }
                else if (parent instanceof EmbeddedMetaData)
                {
                    EmbeddedMetaData emd = (EmbeddedMetaData)parent;
                    PropertyMetaData propmd = newPropertyObject(emd, attrs);
                    emd.addMember(propmd);
                    pushStack(propmd);
                }
                else if (parent instanceof ImplementsMetaData)
                {
                    ImplementsMetaData implmd = (ImplementsMetaData)parent;
                    PropertyMetaData propmd = newPropertyObject(implmd, attrs);
                    implmd.addProperty(propmd);
                    pushStack(propmd);
                }
                else if (parent instanceof FetchGroupMetaData)
                {
                    FetchGroupMetaData fgmd = (FetchGroupMetaData)parent;
                    PropertyMetaData propmd = newPropertyObject(fgmd, attrs);
                    fgmd.addMember(propmd);
                    pushStack(propmd);
                }
            }
            else if (localName.equals("datastore-identity"))
            {
                AbstractClassMetaData acmd = (AbstractClassMetaData)getStack();
                IdentityMetaData idmd = new IdentityMetaData();
                idmd.setColumnName(getAttr(attrs,"column"));
                idmd.setValueStrategy(IdentityStrategy.getIdentityStrategy(getAttr(attrs,"strategy")));
                idmd.setSequence(getAttr(attrs,"sequence"));
                acmd.setIdentityMetaData(idmd);
                pushStack(idmd);
            }
            else if (localName.equals("inheritance"))
            {
                MetaData parent = getStack();
                AbstractClassMetaData acmd = (AbstractClassMetaData)parent;
                InheritanceMetaData inhmd = new InheritanceMetaData();
                inhmd.setStrategy(getAttr(attrs,"strategy"));
                acmd.setInheritanceMetaData(inhmd);
                pushStack(inhmd);
            }
            else if (localName.equals("discriminator"))
            {
                InheritanceMetaData inhmd = (InheritanceMetaData)getStack();
                DiscriminatorMetaData dismd = inhmd.newDiscriminatorMetadata();
                dismd.setColumnName(getAttr(attrs,"column"));
                dismd.setValue(getAttr(attrs,"value"));
                dismd.setStrategy(getAttr(attrs,"strategy"));
                dismd.setIndexed(getAttr(attrs,"indexed"));
                pushStack(dismd);
            }
            else if (localName.equals("query"))
            {
                MetaData emd = getStack();
                if (emd instanceof ClassMetaData)
                {
                    ClassMetaData cmd = (ClassMetaData)emd;
                    QueryMetaData qmd = new QueryMetaData(getAttr(attrs, "name"));
                    qmd.setScope(cmd.getFullClassName());
                    qmd.setLanguage(getAttr(attrs, "language"));
                    qmd.setUnmodifiable(getAttr(attrs, "unmodifiable"));
                    qmd.setResultClass(getAttr(attrs, "result-class"));
                    qmd.setUnique(getAttr(attrs, "unique"));
                    qmd.setFetchPlanName(getAttr(attrs, "fetch-plan"));
                    cmd.addQuery(qmd);
                    pushStack(qmd);
                }
                else if (emd instanceof InterfaceMetaData)
                {
                    InterfaceMetaData imd = (InterfaceMetaData)emd;
                    QueryMetaData qmd = new QueryMetaData(getAttr(attrs, "name"));
                    qmd.setScope(imd.getFullClassName());
                    qmd.setLanguage(getAttr(attrs, "language"));
                    qmd.setUnmodifiable(getAttr(attrs, "unmodifiable"));
                    qmd.setResultClass(getAttr(attrs, "result-class"));
                    qmd.setUnique(getAttr(attrs, "unique"));
                    qmd.setFetchPlanName(getAttr(attrs, "fetch-plan"));
                    imd.addQuery(qmd);
                    pushStack(qmd);
                }
                else if (emd instanceof FileMetaData)
                {
                    FileMetaData filemd = (FileMetaData)emd;
                    QueryMetaData qmd = filemd.newQueryMetadata(getAttr(attrs, "name"));
                    qmd.setLanguage(getAttr(attrs, "language"));
                    qmd.setUnmodifiable(getAttr(attrs, "unmodifiable"));
                    qmd.setResultClass(getAttr(attrs, "result-class"));
                    qmd.setUnique(getAttr(attrs, "unique"));
                    qmd.setFetchPlanName(getAttr(attrs, "fetch-plan"));
                    pushStack(qmd);
                }
            }
            else if (localName.equals("sequence"))
            {
                PackageMetaData pmd = (PackageMetaData)getStack();
                SequenceMetaData seqmd = 
                    pmd.newSequenceMetadata(getAttr(attrs,"name"), getAttr(attrs,"strategy"));
                seqmd.setFactoryClass(getAttr(attrs,"factory-class"));
                seqmd.setDatastoreSequence(getAttr(attrs,"datastore-sequence"));
                pushStack(seqmd);
            }
            else if (localName.equals("field"))
            {
                MetaData md = getStack();
                FieldMetaData fmd = newFieldObject(md,attrs);
                if (md instanceof ClassMetaData)
                {
                    ClassMetaData cmd = (ClassMetaData)md;
                    cmd.addMember(fmd);
                }
                else if (md instanceof FetchGroupMetaData)
                {
                    FetchGroupMetaData fgmd = (FetchGroupMetaData)md;
                    fgmd.addMember(fmd);
                }
                else if (md instanceof EmbeddedMetaData)
                {
                    EmbeddedMetaData emd = (EmbeddedMetaData)md;
                    emd.addMember(fmd);
                }
                else if (md instanceof ForeignKeyMetaData)
                {
                    ForeignKeyMetaData fkmd = (ForeignKeyMetaData)md;
                    fkmd.addMember(fmd);
                }
                else if (md instanceof IndexMetaData)
                {
                    IndexMetaData imd = (IndexMetaData)md;
                    imd.addMember(fmd);
                }
                else if (md instanceof UniqueMetaData)
                {
                    UniqueMetaData umd = (UniqueMetaData)md;
                    umd.addMember(fmd);
                }
                pushStack(fmd);
            }
            else if (localName.equals("join"))
            {
                MetaData parent = getStack();
                String tableName = getAttr(attrs, "table");
                String columnName = getAttr(attrs, "column");
                String outer = getAttr(attrs, "outer");
                IndexedValue indexed = IndexedValue.getIndexedValue(getAttr(attrs, "indexed"));
                String unique = getAttr(attrs, "unique");
                String deleteAction = getAttr(attrs, "delete-action");

                JoinMetaData joinmd = null;
                if (parent instanceof AbstractMemberMetaData)
                {
                    AbstractMemberMetaData fmd = (AbstractMemberMetaData)parent;
                    joinmd = fmd.newJoinMetadata();
                }
                else if (parent instanceof AbstractClassMetaData)
                {
                    AbstractClassMetaData cmd = (AbstractClassMetaData)parent;
                    joinmd = new JoinMetaData();
                    cmd.addJoin(joinmd);
                }
                else if (parent instanceof InheritanceMetaData)
                {
                    InheritanceMetaData inhmd = (InheritanceMetaData)parent;
                    joinmd = inhmd.newJoinMetadata();
                }
                joinmd.setTable(tableName);
                joinmd.setColumnName(columnName);
                joinmd.setOuter(outer);
                joinmd.setIndexed(indexed);
                joinmd.setUnique(unique);
                joinmd.setDeleteAction(deleteAction);
                pushStack(joinmd);
            }
            else if (localName.equals("map"))
            {
                AbstractMemberMetaData fmd = (AbstractMemberMetaData)getStack();
                MapMetaData mapmd = fmd.newMapMetaData();
                mapmd.setKeyType(getAttr(attrs,"key-type"));
                mapmd.setEmbeddedKey(getAttr(attrs,"embedded-key"));
                mapmd.setSerializedKey(getAttr(attrs,"serialized-key"));
                mapmd.setDependentKey(getAttr(attrs,"dependent-key"));
                mapmd.setValueType(getAttr(attrs,"value-type"));
                mapmd.setEmbeddedValue(getAttr(attrs,"embedded-value"));
                mapmd.setSerializedValue(getAttr(attrs,"serialized-value"));
                mapmd.setDependentValue(getAttr(attrs,"dependent-value"));
                pushStack(mapmd);
            }
            else if (localName.equals("array"))
            {
                AbstractMemberMetaData fmd = (AbstractMemberMetaData)getStack();
                ArrayMetaData arrmd = fmd.newArrayMetaData();
                arrmd.setElementType(getAttr(attrs, "element-type"));
                arrmd.setEmbeddedElement(getAttr(attrs, "embedded-element"));
                arrmd.setSerializedElement(getAttr(attrs, "serialized-element"));
                arrmd.setDependentElement(getAttr(attrs, "dependent-element"));
                pushStack(arrmd);
            }
            else if (localName.equals("collection"))
            {
                AbstractMemberMetaData fmd = (AbstractMemberMetaData)getStack();
                CollectionMetaData collmd = fmd.newCollectionMetaData();
                collmd.setElementType(getAttr(attrs,"element-type"));
                collmd.setEmbeddedElement(getAttr(attrs,"embedded-element"));
                collmd.setSerializedElement(getAttr(attrs,"serialized-element"));
                collmd.setDependentElement(getAttr(attrs,"dependent-element"));
                pushStack(collmd);
            }
            else if (localName.equals("column"))
            {
                MetaData md = getStack();
                ColumnMetaData colmd = new ColumnMetaData();
                colmd.setName(getAttr(attrs,"name"));
                colmd.setTarget(getAttr(attrs,"target"));
                colmd.setTargetMember(getAttr(attrs,"target-field"));
                colmd.setJdbcType(getAttr(attrs,"jdbc-type"));
                colmd.setSqlType(getAttr(attrs,"sql-type"));
                colmd.setLength(getAttr(attrs,"length"));
                colmd.setScale(getAttr(attrs,"scale"));
                colmd.setAllowsNull(getAttr(attrs,"allows-null"));
                colmd.setDefaultValue(getAttr(attrs,"default-value"));
                colmd.setInsertValue(getAttr(attrs,"insert-value"));
                if (md instanceof AbstractMemberMetaData)
                {
                    AbstractMemberMetaData fmd = (AbstractMemberMetaData)md;
                    fmd.addColumn(colmd);
                }
                else if (md instanceof AbstractElementMetaData)
                {
                    AbstractElementMetaData elemd = (AbstractElementMetaData)md;
                    elemd.addColumn(colmd);
                }
                else if (md instanceof JoinMetaData)
                {
                    JoinMetaData jnmd = (JoinMetaData)md;
                    jnmd.addColumn(colmd);
                }
                else if (md instanceof IdentityMetaData)
                {
                    IdentityMetaData idmd = (IdentityMetaData)md;
                    idmd.addColumn(colmd);
                }
                else if (md instanceof ForeignKeyMetaData)
                {
                    ForeignKeyMetaData fkmd = (ForeignKeyMetaData)md;
                    fkmd.addColumn(colmd);
                }
                else if (md instanceof IndexMetaData)
                {
                    IndexMetaData idxmd = (IndexMetaData)md;
                    idxmd.addColumn(colmd);
                }
                else if (md instanceof UniqueMetaData)
                {
                    UniqueMetaData unimd = (UniqueMetaData)md;
                    unimd.addColumn(colmd);
                }
                else if (md instanceof OrderMetaData)
                {
                    OrderMetaData ormd = (OrderMetaData)md;
                    ormd.addColumn(colmd);
                }
                else if (md instanceof DiscriminatorMetaData)
                {
                    DiscriminatorMetaData dismd = (DiscriminatorMetaData)md;
                    dismd.setColumnMetaData(colmd);
                }
                else if (md instanceof VersionMetaData)
                {
                    VersionMetaData vermd = (VersionMetaData)md;
                    vermd.addColumn(colmd);
                }
                else if (md instanceof AbstractClassMetaData)
                {
                    AbstractClassMetaData cmd = (AbstractClassMetaData)md;
                    cmd.addUnmappedColumn(colmd);
                }
                else if (md instanceof PrimaryKeyMetaData)
                {
                    PrimaryKeyMetaData pkmd = (PrimaryKeyMetaData)md;
                    pkmd.addColumn(colmd);
                }
                pushStack(colmd);
            }
            else if (localName.equals("element"))
            {
                AbstractMemberMetaData fmd = (AbstractMemberMetaData)getStack();
                ElementMetaData elemmd = new ElementMetaData();
                elemmd.setColumnName(getAttr(attrs, "column"));
                elemmd.setDeleteAction(getAttr(attrs, "delete-action"));
                elemmd.setUpdateAction(getAttr(attrs, "update-action"));
                elemmd.setIndexed(IndexedValue.getIndexedValue(getAttr(attrs, "indexed")));
                elemmd.setUnique(getAttr(attrs, "unique"));
                elemmd.setMappedBy(getAttr(attrs, "mapped-by"));
                fmd.setElementMetaData(elemmd);
                pushStack(elemmd);
            }
            else if (localName.equals("key"))
            {
                AbstractMemberMetaData fmd = (AbstractMemberMetaData)getStack();
                KeyMetaData keymd = new KeyMetaData();
                keymd.setColumnName(getAttr(attrs, "column"));
                keymd.setDeleteAction(getAttr(attrs, "delete-action"));
                keymd.setUpdateAction(getAttr(attrs, "update-action"));
                keymd.setIndexed(IndexedValue.getIndexedValue(getAttr(attrs, "indexed")));
                keymd.setUnique(getAttr(attrs, "unique"));
                keymd.setMappedBy(getAttr(attrs, "mapped-by"));
                fmd.setKeyMetaData(keymd);
                pushStack(keymd);
            }
            // New value
            else if (localName.equals("value"))
            {
                AbstractMemberMetaData fmd = (AbstractMemberMetaData)getStack();
                ValueMetaData valuemd = new ValueMetaData();
                valuemd.setColumnName(getAttr(attrs, "column"));
                valuemd.setDeleteAction(getAttr(attrs, "delete-action"));
                valuemd.setUpdateAction(getAttr(attrs, "update-action"));
                valuemd.setIndexed(IndexedValue.getIndexedValue(getAttr(attrs, "indexed")));
                valuemd.setUnique(getAttr(attrs, "unique"));
                valuemd.setMappedBy(getAttr(attrs, "mapped-by"));
                fmd.setValueMetaData(valuemd);
                pushStack(valuemd);
            }
            // New fetch-group
            else if (localName.equals("fetch-group"))
            {
                MetaData md = getStack();
                FetchGroupMetaData fgmd = new FetchGroupMetaData(getAttr(attrs,"name"));
                fgmd.setPostLoad(getAttr(attrs,"post-load"));
                if (md instanceof FetchGroupMetaData)
                {
                    FetchGroupMetaData fgmdParent = (FetchGroupMetaData)md;
                    fgmdParent.addFetchGroup(fgmd);
                }
                else if (md instanceof AbstractClassMetaData)
                {
                    AbstractClassMetaData cmd = (AbstractClassMetaData)md;
                    cmd.addFetchGroup(fgmd);
                }
                else if (md instanceof FetchPlanMetaData)
                {
                    FetchPlanMetaData fpmd = (FetchPlanMetaData)md;
                    fpmd.addFetchGroup(fgmd);
                }
                pushStack(fgmd);
            }
            else if (localName.equals("extension"))
            {
                MetaData md = getStack();
                md.addExtension(getAttr(attrs,"vendor-name"), getAttr(attrs,"key"), getAttr(attrs,"value"));
            }
            else if (localName.equals("version"))
            {
                AbstractClassMetaData cmd = (AbstractClassMetaData)getStack();
                VersionMetaData vermd = cmd.newVersionMetadata();
                vermd.setStrategy(getAttr(attrs,"strategy")).setColumnName(getAttr(attrs,"column"));
                vermd.setIndexed(IndexedValue.getIndexedValue(getAttr(attrs,"indexed")));
                pushStack(vermd);
            }
            else if (localName.equals("index"))
            {
                MetaData md = getStack();
                IndexMetaData idxmd = new IndexMetaData();
                idxmd.setName(getAttr(attrs,"name"));
                idxmd.setTable(getAttr(attrs,"table"));
                idxmd.setUnique(getAttr(attrs,"unique"));
                if (md instanceof AbstractClassMetaData)
                {
                    AbstractClassMetaData cmd = (AbstractClassMetaData)md;
                    cmd.addIndex(idxmd);
                }
                else if (md instanceof AbstractMemberMetaData)
                {
                    AbstractMemberMetaData fmd = (AbstractMemberMetaData)md;
                    fmd.setIndexMetaData(idxmd);
                }
                else if (md instanceof JoinMetaData)
                {
                    JoinMetaData jmd = (JoinMetaData)md;
                    jmd.setIndexMetaData(idxmd);
                }
                else if (md instanceof AbstractElementMetaData)
                {
                    AbstractElementMetaData elmd = (AbstractElementMetaData)md;
                    elmd.setIndexMetaData(idxmd);
                }
                else if (md instanceof OrderMetaData)
                {
                    OrderMetaData omd = (OrderMetaData)md;
                    omd.setIndexMetaData(idxmd);
                }
                else if (md instanceof VersionMetaData)
                {
                    VersionMetaData vermd = (VersionMetaData)md;
                    vermd.setIndexMetaData(idxmd);
                }
                else if (md instanceof DiscriminatorMetaData)
                {
                    DiscriminatorMetaData dismd = (DiscriminatorMetaData)md;
                    dismd.setIndexMetaData(idxmd);
                }
                pushStack(idxmd);
            }
            else if (localName.equals("unique"))
            {
                MetaData md = getStack();
                UniqueMetaData unimd = new UniqueMetaData();
                unimd.setName(getAttr(attrs,"name"));
                unimd.setTable(getAttr(attrs,"table"));
                unimd.setDeferred(getAttr(attrs,"deferred"));
                if (md instanceof AbstractClassMetaData)
                {
                    AbstractClassMetaData cmd = (AbstractClassMetaData)md;
                    cmd.addUniqueConstraint(unimd);
                }
                else if (md instanceof AbstractMemberMetaData)
                {
                    AbstractMemberMetaData fmd = (AbstractMemberMetaData)md;
                    fmd.setUniqueMetaData(unimd);
                }
                else if (md instanceof JoinMetaData)
                {
                    JoinMetaData jmd = (JoinMetaData)md;
                    jmd.setUniqueMetaData(unimd);
                }
                else if (md instanceof AbstractElementMetaData)
                {
                    AbstractElementMetaData elmd = (AbstractElementMetaData)md;
                    elmd.setUniqueMetaData(unimd);
                }
                pushStack(unimd);
            }
            else if (localName.equals("foreign-key"))
            {
                MetaData md = getStack();
                ForeignKeyMetaData fkmd = new ForeignKeyMetaData();
                fkmd.setName(getAttr(attrs,"name"));
                fkmd.setTable(getAttr(attrs,"table"));
                fkmd.setUnique(getAttr(attrs,"unique"));
                fkmd.setDeferred(getAttr(attrs,"deferred"));
                fkmd.setDeleteAction(ForeignKeyAction.getForeignKeyAction(getAttr(attrs,"delete-action")));
                fkmd.setUpdateAction(ForeignKeyAction.getForeignKeyAction(getAttr(attrs,"update-action")));
                if (md instanceof AbstractClassMetaData)
                {
                    AbstractClassMetaData cmd = (AbstractClassMetaData)md;
                    cmd.addForeignKey(fkmd);
                }
                else if (md instanceof AbstractMemberMetaData)
                {
                    AbstractMemberMetaData fmd = (AbstractMemberMetaData)md;
                    fmd.setForeignKeyMetaData(fkmd);
                }
                else if (md instanceof JoinMetaData)
                {
                    JoinMetaData jmd = (JoinMetaData)md;
                    jmd.setForeignKeyMetaData(fkmd);
                }
                else if (md instanceof AbstractElementMetaData)
                {
                    AbstractElementMetaData elmd = (AbstractElementMetaData)md;
                    elmd.setForeignKeyMetaData(fkmd);
                }
                pushStack(fkmd);
            }
            else if (localName.equals("order"))
            {
                OrderMetaData ordmd = new OrderMetaData();
                ordmd.setIndexed(IndexedValue.getIndexedValue(getAttr(attrs, "indexed")));
                ordmd.setColumnName(getAttr(attrs, "column"));
                ordmd.setMappedBy(getAttr(attrs, "mapped-by"));
                AbstractMemberMetaData fmd = (AbstractMemberMetaData)getStack();
                fmd.setOrderMetaData(ordmd);
                pushStack(ordmd);
            }
            else if (localName.equals("embedded"))
            {
                MetaData md = getStack();
                EmbeddedMetaData embmd = new EmbeddedMetaData();
                embmd.setOwnerMember(getAttr(attrs,"owner-field"));
                embmd.setNullIndicatorColumn(getAttr(attrs,"null-indicator-column"));
                embmd.setNullIndicatorValue(getAttr(attrs,"null-indicator-value"));
                if (md instanceof AbstractMemberMetaData)
                {
                    AbstractMemberMetaData fmd = (AbstractMemberMetaData)md;
                    fmd.setEmbeddedMetaData(embmd);
                }
                else if (md instanceof KeyMetaData)
                {
                    KeyMetaData kmd = (KeyMetaData)md;
                    kmd.setEmbeddedMetaData(embmd);
                }
                else if (md instanceof ValueMetaData)
                {
                    ValueMetaData vmd = (ValueMetaData)md;
                    vmd.setEmbeddedMetaData(embmd);
                }
                else if (md instanceof ElementMetaData)
                {
                    ElementMetaData elmd = (ElementMetaData)md;
                    elmd.setEmbeddedMetaData(embmd);
                }
                pushStack(embmd);
            }
            else
            {
                String message = LOCALISER.msg("044037",qName);
                NucleusLogger.METADATA.error(message);
                throw new RuntimeException(message);
            }
        }
        catch(RuntimeException ex)
        {
            NucleusLogger.METADATA.error(LOCALISER.msg("044042", qName, getStack(), uri), ex);
            throw ex;
        }
    }

    /**
     * Handler method called at the end of an element.
     * @param uri URI of the tag
     * @param localName local name
     * @param qName Name of element just ending
     * @throws SAXException in parsing errors
     */
    public void endElement(String uri, String localName, String qName)
    throws SAXException
    {
        if (localName.length() < 1)
        {
            localName = qName;
        }

        // Save the current string for elements that have a body value
        String currentString = getString().trim();
        if (currentString.length() > 0)
        {
            MetaData md = getStack();
            if (localName.equals("query"))
            {
                ((QueryMetaData)md).setQuery(currentString);
            }
        }

        // Pop the tag
        // If startElement pushes an element onto the stack need a remove here for that type
        if (localName.equals("package") ||
            localName.equals("fetch-plan") ||
            localName.equals("class") ||
            localName.equals("interface") ||
            localName.equals("implements") ||
            localName.equals("property") ||
            localName.equals("datastore-identity") ||
            localName.equals("inheritance") ||
            localName.equals("primary-key") ||
            localName.equals("version") ||
            localName.equals("unmapped") ||
            localName.equals("query") ||
            localName.equals("sequence") ||
            localName.equals("field") ||
            localName.equals("map") ||
            localName.equals("element") ||
            localName.equals("embedded") ||
            localName.equals("key") ||
            localName.equals("value") ||
            localName.equals("array") ||
            localName.equals("collection") ||
            localName.equals("join") ||
            localName.equals("index") ||
            localName.equals("unique") ||
            localName.equals("foreign-key") ||
            localName.equals("order") ||
            localName.equals("fetch-group") ||
            localName.equals("column") ||
            localName.equals("discriminator"))
        {
            popStack();
        }
    }
}