/**********************************************************************
Copyright (c) 2009 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
   ...
**********************************************************************/
package org.datanucleus.store;

import org.datanucleus.ObjectManager;
import org.datanucleus.StateManager;

import org.datanucleus.store.StorePersistenceHandler;

/**
 * Abstract representation of a persistence handler, to be extended by datastores own variant.
 */
public abstract class AbstractPersistenceHandler implements StorePersistenceHandler
{
    /* (non-Javadoc)
     * @see org.datanucleus.store.StorePersistenceHandler#batchEnd(org.datanucleus.ObjectManager)
     */
    public void batchEnd(ObjectManager om)
    {
        // Override in subclasses if supporting batching using this mechanism
    }

    /* (non-Javadoc)
     * @see org.datanucleus.store.StorePersistenceHandler#batchStart(org.datanucleus.ObjectManager)
     */
    public void batchStart(ObjectManager om)
    {
        // Override in subclasses if supporting batching using this mechanism
    }

    /**
     * Simple implementation of location of multiple objects, relaying the locate check for each object
     * to <pre>locateObject</pre>.
     * Should be overridden by the datastore implementation if it is possible to do bulk locates.
     * @param sms StateManagers for the objects to locate
     * @throws NucleusObjectNotFoundException if an object doesn't exist
     * @throws NucleusDataStoreException when an error occurs in the datastore communication
     */
    public void locateObjects(StateManager[] sms)
    {
        for (int i=0;i<sms.length;i++)
        {
            locateObject(sms[i]);
        }
    }

    /**
     * Simple implementation of find of multiple objects, relaying the find for each object
     * to <pre>findObject</pre>.
     * Should be overridden by the datastore implementation if it is possible to do bulk retrieval.
     * @param om the ObjectManager which will manage the object
     * @param ids identities of the object(s) to retrieve
     * @return The persistable objects with these identities (in the same order as <pre>ids</pre>)
     * @throws NucleusObjectNotFoundException if an object doesn't exist
     * @throws NucleusDataStoreException when an error occurs in the datastore communication
     */
    public Object[] findObjects(ObjectManager om, Object[] ids)
    {
        Object[] objects = new Object[ids.length];
        for (int i=0;i<ids.length;i++)
        {
            objects[i] = findObject(om, ids[i]);
        }
        return objects;
    }
}