/**********************************************************************
Copyright (c) 2009 Erik Bengtson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.store;

import org.datanucleus.exceptions.NucleusDataStoreException;
import org.datanucleus.exceptions.NucleusObjectNotFoundException;

/**
 * Interface defining persistence operations of a StoreManager.
 * This performs the low level communication with the actual datastore.
 */
public interface StorePersistenceHandler2
{
    /**
     * Method to close the persistence handler, and release any resources.
     */
    public void close();

    /**
     * Signal that a batch of operations are starting for the specified ExecutionContext.
     * This is usually a batch persist, or batch delete.
     * @param ectx The ExecutionContext
     */
    void batchStart(ExecutionContext ectx);

    /**
     * Signal that a batch of operations are ending for the specified ExecutionContext.
     * This is usually a batch persist, or batch delete.
     * @param ectx The ExecutionContext
     */
    void batchEnd(ExecutionContext ectx);

    /**
     * Inserts a persistent object into the database.
     * @param op The ObjectProvider of the object to be inserted.
     * @throws NucleusDataStoreException when an error occurs in the datastore communication
     */
    public void insertObject(ObjectProvider op);

    /**
     * Updates a persistent object in the datastore.
     * @param op The ObjectProvider of the object to be updated.
     * @param fieldNumbers The numbers of the fields to be updated.
     * @throws NucleusDataStoreException when an error occurs in the datastore communication
     */
    public void updateObject(ObjectProvider op, int fieldNumbers[]);

    /**
     * Deletes a persistent object from the datastore.
     * @param op The ObjectProvider of the object to be deleted.
     * @throws NucleusDataStoreException when an error occurs in the datastore communication
     */
    public void deleteObject(ObjectProvider op);

    /**
     * Fetches a persistent object from the database.
     * @param op The ObjectProvider of the object to be fetched.
     * @param fieldNumbers The numbers of the fields to be fetched.
     * @throws NucleusObjectNotFoundException if the object doesn't exist
     * @throws NucleusDataStoreException when an error occurs in the datastore communication
     */
    public void fetchObject(ObjectProvider op, int fieldNumbers[]);

    /**
     * Locates this object in the datastore.
     * @param op The ObjectProvider for the object to be found
     * @throws NucleusObjectNotFoundException if the object doesn't exist
     * @throws NucleusDataStoreException when an error occurs in the datastore communication
     */
    public void locateObject(ObjectProvider op);

    /**
     * Locates object(s) in the datastore.
     * @param ops ObjectProvider(s) for the object(s) to be found
     * @throws NucleusObjectNotFoundException if an object doesn't exist
     * @throws NucleusDataStoreException when an error occurs in the datastore communication
     */
    public void locateObjects(ObjectProvider[] ops);

    /**
     * Method to find a persistable object with the specified id from the datastore, if the StoreManager 
     * supports this operation (optional). This allows for datastores that perform the instantiation of 
     * objects directly (such as ODBMS). With other types of datastores (e.g RDBMS) this method returns null.
     * @param ectx The ExecutionContext
     * @param id the id of the object in question.
     * @return a persistable object with a valid object state (for example: hollow) or null, 
     *     indicating that the implementation leaves the instantiation work to DataNucleus.
     * @throws NucleusObjectNotFoundException if this route is supported yet the object doesn't exist
     * @throws NucleusDataStoreException when an error occurs in the datastore communication
     */
    public Object findObject(ExecutionContext ectx, Object id);

    /**
     * Method to find an array of objects with the specified identities from the datastore.
     * This allows for datastores that perform the instantiation of objects directly (such as ODBMS). 
     * With other types of datastores (e.g RDBMS) this method returns null.
     * @param ectx The ExecutionContext
     * @param ids identities of the object(s) to retrieve
     * @return The persistable objects with these identities (in the same order as <pre>ids</pre>)
     * @throws NucleusObjectNotFoundException if an object doesn't exist
     * @throws NucleusDataStoreException when an error occurs in the datastore communication
     */
    public Object[] findObjects(ExecutionContext ectx, Object[] ids);
}