/**********************************************************************
Copyright (c) 2004 Barry Haddow and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
 

Contributors:
2004 Erik Bengtson - added cast and other operators
2005 Andy Jefferson - added javadoc
2006 Andy Jefferson - generalised as reference expression
2007 Andy Jefferson - cater for RefExpr == null
    ...
**********************************************************************/
package org.datanucleus.store.mapped.expression;

import org.datanucleus.exceptions.NucleusUserException;
import org.datanucleus.store.mapped.mapping.JavaTypeMapping;
import org.datanucleus.store.mapped.mapping.NullMapping;
import org.datanucleus.store.mapped.mapping.ReferenceMapping;

/**
 * Represents a reference field/object (interface/object) in a JDOQL query.
 *
 * @version $Revision: 1.9 $ 
 */
public class ReferenceExpression extends ScalarExpression
{
    /**
     * Constructor for a reference expression, using the mapping of the field, and the expression for the table.
     * @param qs The Query Statement
     * @param mapping The mapping for the field whose interface we are expressing
     * @param te The expression for the table of the interface.
     */
    public ReferenceExpression(QueryExpression qs, JavaTypeMapping mapping, LogicSetExpression te)
    {
        super(qs, mapping, te);
    }

    // TODO Likely to need another constructor for 1-1 bidir/1-N bidir in the same way as ObjectExpression.

    /**
     * Method for casting a reference expression to some other type in the query.
     * @param castType The type to cast it to.
     * @return The expression for the casted reference
     */
    public ScalarExpression cast(Class castType)
    {
        JavaTypeMapping[] javaTypeMappings = ((ReferenceMapping)mapping).getJavaTypeMapping();
        for (int i = 0; i < javaTypeMappings.length; ++i)
        {
            JavaTypeMapping m = javaTypeMappings[i];
            if (castType.getName().equals(m.getType()))
            {          
                return m.newScalarExpression(qs,te);
            }
        }
        return super.cast(castType);
    }

    /**
     * Method for use when handling the equality of reference expressions.
     * @param expr The expression to compare against
     * @return Expression whether the expressions are equal
     */
    public BooleanExpression eq(ScalarExpression expr)
    {
		BooleanExpression bexpr = null;
        JavaTypeMapping[] javaTypeMappings = ((ReferenceMapping) mapping).getJavaTypeMapping();
        if (expr.mapping instanceof ReferenceMapping)
        {
            // Expression of form "ReferenceExpression == ReferenceExpression"
            // TODO Should really split into an expression for each part and line up the JavaTypeMappings
            for (int i = 0; i < javaTypeMappings.length; ++i)
            {
                JavaTypeMapping m = javaTypeMappings[i];

                if (bexpr == null)
                {
                    bexpr = m.newScalarExpression(qs, te).eq(expr);
                    bexpr.encloseWithInParentheses();
                }
                else
                {
                    bexpr = bexpr.ior(m.newScalarExpression(qs, te).eq(expr).encloseWithInParentheses());
                    bexpr.encloseWithInParentheses();
                }
            }
        }
        else if (expr.mapping instanceof NullMapping)
        {
            // Expression of form "ReferenceExpression == null"
            for (int i = 0; i < javaTypeMappings.length; ++i)
            {
                JavaTypeMapping m = javaTypeMappings[i];
                if (bexpr == null)
                {
                    bexpr = m.newScalarExpression(qs, te).eq(expr);
                }
                else
                {
                    bexpr = bexpr.and(m.newScalarExpression(qs, te).eq(expr));
                }
            }
        }
        else
        {
            // Expression of form "ReferenceExpression == PC"
            for (int i = 0; i < javaTypeMappings.length; ++i)
            {
                JavaTypeMapping m = javaTypeMappings[i];
                JavaTypeMapping exprMapping = expr.mapping;
                if (m.getNumberOfDatastoreMappings() == exprMapping.getNumberOfDatastoreMappings())
                {
                    // TODO Should apply the above check on noteq() too but in reverse
                    // If expression implementation and this implementation have same number of PK fields allow the compare
                    // TODO Should really also compare the types of the mappings
                    if (bexpr == null)
                    {
                        bexpr = m.newScalarExpression(qs, te).eq(expr);
                        bexpr.encloseWithInParentheses();
                    }
                    else
                    {
                        bexpr = bexpr.ior(m.newScalarExpression(qs, te).eq(expr).encloseWithInParentheses());
                        bexpr.encloseWithInParentheses();
                    }
                }
            }
        }

        bexpr.encloseWithInParentheses();
        return bexpr;
    }

    /**
     * Method for use when handling the inequality of reference expressions.
     * @param expr The expression to compare against
     * @return Expression whether the expressions are not equal
     */
    public BooleanExpression noteq(ScalarExpression expr)
    {
		BooleanExpression bexpr = null;
        JavaTypeMapping[] javaTypeMappings = ((ReferenceMapping)mapping).getJavaTypeMapping();
        if (expr.mapping instanceof NullMapping)
        {
            // Expression of form "ReferenceExpression != null"
            // So find a mapping that is not null amongst the possible impls
            for (int i = 0; i < javaTypeMappings.length; ++i)
            {
                JavaTypeMapping m = javaTypeMappings[i];
                if (bexpr == null)
                {
                    bexpr = m.newScalarExpression(qs, te).noteq(expr);
                }
                else
                {
                    bexpr = bexpr.ior(m.newScalarExpression(qs, te).noteq(expr));
                }
            }
        }
        else
        {
            for (int i = 0; i < javaTypeMappings.length; ++i)
            {
                JavaTypeMapping m = javaTypeMappings[i];
                if (bexpr == null)
                {
                    bexpr = m.newScalarExpression(qs,te).noteq(expr); 
                }
                else
                {
                    bexpr = bexpr.and(m.newScalarExpression(qs,te).noteq(expr)); 
                }
            }
        }
		bexpr.encloseWithInParentheses();
        return bexpr;
    }

    /**
     * Method invoked when accessing a field in the reference.
     * This is currently not available - you must cast it to a concrete type (since the
     * reference doesnt have any fields!).
     * @param fieldName Name of the field to access
     * @param innerJoin whether to join using an inner join
     * @return Expression representing the field of this reference
     */
    public ScalarExpression accessField(String fieldName, boolean innerJoin)
    {
        throw new NucleusUserException(LOCALISER.msg("037000", fieldName, mapping.getType()));
    }
}