/**********************************************************************
Copyright (c) 2005 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License. 


Contributors:
    ...
**********************************************************************/
package org.datanucleus.store.mapped.scostore;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.exceptions.NucleusUserException;
import org.datanucleus.metadata.AbstractClassMetaData;
import org.datanucleus.metadata.AbstractMemberMetaData;
import org.datanucleus.metadata.FieldRole;
import org.datanucleus.metadata.MetaDataUtils;
import org.datanucleus.store.mapped.DatastoreClass;
import org.datanucleus.store.mapped.DatastoreContainerObject;
import org.datanucleus.store.mapped.DatastoreIdentifier;
import org.datanucleus.store.mapped.exceptions.IncompatibleQueryElementTypeException;
import org.datanucleus.store.mapped.expression.LogicSetExpression;
import org.datanucleus.store.mapped.expression.QueryExpression;
import org.datanucleus.store.mapped.expression.ScalarExpression;
import org.datanucleus.store.mapped.mapping.JavaTypeMapping;
import org.datanucleus.util.ClassUtils;

/**
 * Backing store for an array stored using a join table.
 * Can be used for all types of arrays :- PC arrays, non-PC arrays, reference arrays.
 */
public abstract class JoinArrayStore extends AbstractArrayStore
{
    /**
     * Constructor.
     * @param joinTable Join table storing the relationship between owner and element
     * @param clr ClassLoader resolver
     */
    public JoinArrayStore(DatastoreContainerObject joinTable, AbstractMemberMetaData ownerFieldMetaData,
                          JavaTypeMapping ownerMapping, JavaTypeMapping elementMapping, JavaTypeMapping orderMapping,
                          JavaTypeMapping relationDiscriminatorMapping, String relationDiscriminatorValue,
                          String elementType, boolean elementsAreEmbedded, boolean elementsAreSerialised,
                          ClassLoaderResolver clr, JoinArrayStoreSpecialization specialization)
    {
        super(joinTable.getStoreManager(), clr, specialization);

        this.containerTable = joinTable;
        setOwner(ownerFieldMetaData, clr);

        this.ownerMapping = ownerMapping;
        this.elementMapping = elementMapping;
        this.orderMapping = orderMapping;
        this.relationDiscriminatorMapping = relationDiscriminatorMapping;
        this.relationDiscriminatorValue = relationDiscriminatorValue;

        this.elementType = elementType;
        this.elementsAreEmbedded = elementsAreEmbedded;
        this.elementsAreSerialised = elementsAreSerialised;

        if (this.elementsAreSerialised)
        {
            this.elementInfo = null;
        }
        else
        {
            Class element_class=clr.classForName(elementType);
            if (ClassUtils.isReferenceType(element_class))
            {
                // Array of reference types (interfaces/Objects)
                String[] implNames = MetaDataUtils.getInstance().getImplementationNamesForReferenceField(ownerMemberMetaData,
                    FieldRole.ROLE_ARRAY_ELEMENT, clr, storeMgr.getMetaDataManager());
                elementInfo = new ElementInfo[implNames.length];
                for (int i=0;i<implNames.length;i++)
                {
                    DatastoreClass table = storeMgr.getDatastoreClass(implNames[i], clr);
                    AbstractClassMetaData cmd = storeMgr.getOMFContext().getMetaDataManager().getMetaDataForClass(implNames[i], clr);
                    elementInfo[i] = new ElementInfo(cmd, table);
                }
            }
            else
            {
                // Collection of PC or non-PC
                emd = storeMgr.getOMFContext().getMetaDataManager().getMetaDataForClass(element_class, clr);
                if (emd != null)
                {
                    elementType  = emd.getFullClassName();
                    if (!elementsAreEmbedded && !elementsAreSerialised)
                    {
                        elementInfo = getElementInformationForClass();
                        if (elementInfo != null && elementInfo.length > 1)
                        {
                            throw new NucleusUserException(LOCALISER.msg("056045", 
                                ownerMemberMetaData.getFullFieldName()));
                        }
                    }
                    else
                    {
                        elementInfo = null;
                    }
                }
                else
                {
                    elementInfo = null;
                }
            }
        }
    }

    // ----------------------------- TODO Remove these when we replace JDOQL -----------------------------------

    /**
     * Method used in queries when contains() has been invoked.
     * @param stmt The Query Statement 
     * @param parentStmt the parent Query Statement. If there is no parent, <code>parentStmt</code> must be equals to <code>stmt</code> 
     * @param ownerMapping the mapping for the owner. 
     * @param ownerTe Table Expression for the owner
     * @param listTableAlias Alias for the "List" table. 
     * @param filteredElementType The Class Type for the filtered element
     * @param elmExpr The Expression for the element
     * @param elementTableAlias The SQL alias to assign to the expression or to the element table.
     * @return expression to the join
     **/
    public ScalarExpression joinElementsTo(QueryExpression stmt, 
                                           QueryExpression parentStmt,
                                           JavaTypeMapping ownerMapping,
                                           LogicSetExpression ownerTe,
                                           DatastoreIdentifier listTableAlias,
                                           Class filteredElementType,
                                           ScalarExpression elmExpr,
                                           DatastoreIdentifier elementTableAlias)
    {
        ClassLoaderResolver clr=stmt.getClassLoaderResolver();
        Class primitiveType = ClassUtils.getWrapperTypeForPrimitiveType(clr.classForName(elementType));
        if (!clr.isAssignableFrom(elementType, filteredElementType) &&
            !clr.isAssignableFrom(filteredElementType, elementType) &&
            (primitiveType !=null && clr.isAssignableFrom(elementType,primitiveType.getName())))
        {
            throw new IncompatibleQueryElementTypeException(elementType, filteredElementType.getName());
        }

        LogicSetExpression ownTblExpr = stmt.newTableExpression(containerTable, listTableAlias);
        ScalarExpression ownerExpr = ownerMapping.newScalarExpression(stmt, ownerTe);
        ScalarExpression ownerSetExpr = this.ownerMapping.newScalarExpression(stmt, stmt.getTableExpression(listTableAlias));
        if (!parentStmt.hasCrossJoin(ownTblExpr))
        {
            stmt.crossJoin(ownTblExpr, true);
        }
        stmt.andCondition(ownerExpr.eq(ownerSetExpr),true);

        if (storeMgr.getMappedTypeManager().isSupportedMappedType(filteredElementType.getName()))
        {
            // Element = Non-PC(embedded)
            return elementMapping.newScalarExpression(stmt, stmt.getTableExpression(listTableAlias));
        }
        else if (elementsAreEmbedded || elementsAreSerialised)
        {
            // Element = PC(embedded), PC(serialised)
            return elementMapping.newScalarExpression(stmt, stmt.getTableExpression(listTableAlias));
        }
        else
        {
            // Element = PC
            // Join to element table on id column(s) of element
            DatastoreClass elementTable = storeMgr.getDatastoreClass(filteredElementType.getName(), stmt.getClassLoaderResolver());
            DatastoreClass joiningClass = (elmExpr.getLogicSetExpression() == null ? elementTable : (DatastoreClass)elmExpr.getLogicSetExpression().getMainTable());
            JavaTypeMapping elementTableID = joiningClass.getIdMapping();

            /* TODO this comment is no longer valid, since we use sub queries
             * if elementType == filteredElementType
             *    INNER JOIN ELEMENT THIS_FILTER_ELEMENT_VARIABLE
             *    ON THIS_SETTABLE.ELEMENT_EID = THIS_FILTER_ELEMENT_VARIABLE.ELEMENT_ID
             * 
             * else if elementType != filteredElementType but filteredElementType is a superclass of elementType
             * 
             *    INNER JOIN FILTER_ELEMENT_TYPE THIS_FILTER_ELEMENT_VARIABLE
             *    ON THIS_SETTABLE.ELEMENT_EID = THIS_FILTER_ELEMENT_VARIABLE.FILTER_ELEMENT_TYPE_ID */
            LogicSetExpression elmTblExpr = stmt.getTableExpression(elementTableAlias);
            if (elmTblExpr==null)
            {
                elmTblExpr = stmt.newTableExpression(elementTable,elementTableAlias);
            }
            ScalarExpression elmSetExpr = elementMapping.newScalarExpression(stmt, stmt.getTableExpression(listTableAlias));
            if (!parentStmt.hasCrossJoin(elmTblExpr))
            {
                stmt.crossJoin(elmTblExpr, true);
            }
            if (elmExpr.getLogicSetExpression()!= null && !elementTable.equals(elmExpr.getLogicSetExpression().getMainTable()))
            {
                //elmExpr might express a FK in another to the ELEMENT table 
                stmt.andCondition(elmSetExpr.eq(elmExpr),true);
                return this.elementMapping.newScalarExpression(stmt,stmt.getTableExpression(listTableAlias));
            }
            else
            {
                //elmExpr might be a PK of the ELEMENT table 
                ScalarExpression elementExpr = elementTableID.newScalarExpression(stmt, stmt.getTableExpression(elementTableAlias));
                stmt.andCondition(elmSetExpr.eq(elementExpr),true);
                return elementExpr;
            }                    
        }
    }
}
