/**********************************************************************
Copyright (c) 2004 Erik Bengtson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
2004 Andy Jefferson - added description of addition process.
2004 Andy Jefferson - added constructor, and try-catch on initialisation
2006 Andy Jefferson - renamed to PersistenceConfiguration so that it is API agnostic
2008 Andy Jefferson - rewritten to have properties map and not need Java beans setters/getters
    ...
**********************************************************************/
package org.datanucleus;

import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import org.datanucleus.exceptions.NucleusUserException;
import org.datanucleus.plugin.ConfigurationElement;
import org.datanucleus.plugin.PluginManager;
import org.datanucleus.properties.PersistencePropertyValidator;
import org.datanucleus.properties.PropertyTypeInvalidException;
import org.datanucleus.store.ExecutionContext;
import org.datanucleus.util.Localiser;
import org.datanucleus.util.NucleusLogger;
import org.datanucleus.util.PersistenceUtils;

/**
 * Class providing configuration for persistence. 
 * Properties are defined in plugin.xml (aliases, default value, validators etc). 
 * Property values are stored in two maps. 
 * <ul>
 * <li>The first is the default value for the property (where a default is defined).</li>
 * <li>The second is the user-provided value (where the user has provided one).</li>
 * </ul>
 * Components can then access these properties using any of the convenience accessors for boolean, Boolean, long, 
 * int, Object, String types. When accessing properties the user-provided value is taken first (if available),
 * otherwise the default value is used (or null).
 */
public class PersistenceConfiguration
{
    /** Localisation of messages. */
    protected static final Localiser LOCALISER = Localiser.getInstance("org.datanucleus.Localisation",
        ExecutionContext.class.getClassLoader());

    /** Map of user-provided persistence properties. */
    private Map<String, Object> properties = new HashMap<String, Object>();

    /** Map of default persistence properties. */
    private Map<String, Object> defaultProperties = new HashMap<String, Object>();

    /** Mapping for the properties of the plugins, PropertyMapping, keyed by the property name. */
    private Map<String, PropertyMapping> propertyMappings = new HashMap<String, PropertyMapping>();

    /**
     * Convenience class wrapping the plugin property specification information.
     */
    class PropertyMapping
    {
        String name;
        String internalName;
        String value;
        String validatorName;
        public PropertyMapping(String name, String intName, String val, String validator)
        {
            this.name = name;
            this.internalName = intName;
            this.value = val;
            this.validatorName = validator;
        }
    }

    /**
     * Constructor.
     */
    public PersistenceConfiguration()
    {
    }

    /**
     * Accessor for the names of the supported persistence properties.
     * @return The persistence properties that we support
     */
    public Set<String> getSupportedProperties()
    {
        return propertyMappings.keySet();
    }

    /**
     * Method to set the persistence property defaults based on what is defined for plugins.
     * This should only be called after the other setDefaultProperties method is called, which sets up the mappings
     * @param props Properties to use in the default set
     */
    public void setDefaultProperties(Map props)
    {
        if (props != null && props.size() > 0)
        {
            Iterator<Map.Entry> entryIter = props.entrySet().iterator();

            while (entryIter.hasNext())
            {
                Map.Entry entry = entryIter.next();
                defaultProperties.put((String)entry.getKey(), entry.getValue());
            }
        }
    }

    /**
     * Method to set the persistence property defaults based on what is defined for plugins.
     * @param pluginMgr The plugin manager
     */
    public void setDefaultProperties(PluginManager pluginMgr)
    {
        ConfigurationElement[] propElements =
            pluginMgr.getConfigurationElementsForExtension("org.datanucleus.persistence_properties", null, null);
        if (propElements != null)
        {
            for (int i=0;i<propElements.length;i++)
            {
                String name = propElements[i].getAttribute("name");
                String intName = propElements[i].getAttribute("internal-name");
                String value = propElements[i].getAttribute("value");
                propertyMappings.put(name.toLowerCase(), new PropertyMapping(name, intName, value, 
                    propElements[i].getAttribute("validator")));

                Object systemValue = System.getProperty(name);
                if (systemValue != null)
                {
                    // System value provided so add property with that value
                    // TODO Should this be stored as "default" value when provided by user as system value? don't think so
                    this.defaultProperties.put(intName != null ? intName.toLowerCase() : name.toLowerCase(), systemValue);
                }
                else
                {
                    if (!defaultProperties.containsKey(name) && value != null)
                    {
                        // Property has default value and not yet set so add property with that value
                        this.defaultProperties.put(intName != null ? intName.toLowerCase() : name.toLowerCase(), value);
                    }
                }
            }
        }
    }

    /**
     * Accessor for the specified property as an Object.
     * Returns user-specified value if propvided, otherwise the default value, otherwise null.
     * @param name Name of the property
     * @return Value for the property
     */
    public Object getProperty(String name)
    {
        if (properties.containsKey(name.toLowerCase()))
        {
            return properties.get(name.toLowerCase());
        }
        return defaultProperties.get(name.toLowerCase());
    }

    /**
     * Accessor for whether a particular property is defined.
     * @param name Property name
     * @return Whether the property is defined
     */
    public boolean hasProperty(String name)
    {
        return getProperty(name) != null;
    }

    /**
     * Accessor for the specified property as a long.
     * If the specified property isn't found returns 0.
     * @param name Name of the property
     * @return Long value for the property
     * @throws PropertyTypeInvalidException thrown when the property is not available as this type
     */
    public long getLongProperty(String name)
    {
        Object obj = getProperty(name);
        if (obj != null)
        {
            if (obj instanceof Number)
            {
                return ((Number)obj).longValue();
            }
            else if (obj instanceof String)
            {
                Long longVal = Long.valueOf((String)obj);
                setPropertyInternal(name, longVal); // Replace String value with Long
                return longVal.longValue();
            }
        }
        else
        {
            return 0;
        }
        throw new PropertyTypeInvalidException(name, "long");
    }

    /**
     * Accessor for the specified property as an int.
     * If the specified property isn't found returns 0.
     * @param name Name of the property
     * @return Int value for the property
     * @throws PropertyTypeInvalidException thrown when the property is not available as this type
     */
    public int getIntProperty(String name)
    {
        Object obj = getProperty(name);
        if (obj != null)
        {
            if (obj instanceof Number)
            {
                return ((Number)obj).intValue();
            }
            else if (obj instanceof String)
            {
                Integer intVal = Integer.valueOf((String)obj);
                setPropertyInternal(name, intVal); // Replace String value with Integer
                return intVal.intValue();
            }
        }
        else
        {
            return 0;
        }
        throw new PropertyTypeInvalidException(name, "int");
    }

    /**
     * Accessor for the specified property as a boolean.
     * If the specified property isn't found returns false.
     * @param name Name of the property
     * @return Boolean value for the property
     * @throws PropertyTypeInvalidException thrown when the property is not available as this type
     */
    public boolean getBooleanProperty(String name)
    {
        return getBooleanProperty(name, false);
    }

    /**
     * Accessor for the specified property as a boolean.
     * @param name Name of the property
     * @param resultIfNotSet The value to return if no value for the specified property is found.
     * @return Boolean value for the property
     * @throws PropertyTypeInvalidException thrown when the property is not available as this type
     */
    public boolean getBooleanProperty(String name, boolean resultIfNotSet)
    {
        Object obj = getProperty(name);
        if (obj != null)
        {
            if (obj instanceof Boolean)
            {
                return ((Boolean)obj).booleanValue();
            }
            else if (obj instanceof String)
            {
                Boolean boolVal = Boolean.valueOf((String)obj);
                setPropertyInternal(name, boolVal); // Replace String value with Boolean
                return boolVal.booleanValue();
            }
        }
        else
        {
            return resultIfNotSet;
        }
        throw new PropertyTypeInvalidException(name, "boolean");
    }

    /**
     * Accessor for the specified property as a Boolean.
     * If the specified property isn't found returns false.
     * @param name Name of the property
     * @return Boolean value for the property
     * @throws PropertyTypeInvalidException thrown when the property is not available as this type
     */
    public Boolean getBooleanObjectProperty(String name)
    {
        Object obj = getProperty(name);
        if (obj != null)
        {
            if (obj instanceof Boolean)
            {
                return ((Boolean)obj);
            }
            else if (obj instanceof String)
            {
                Boolean boolVal = Boolean.valueOf((String)obj);
                setPropertyInternal(name, boolVal); // Replace String value with Boolean
                return boolVal;
            }
        }
        else
        {
            return null;
        }
        throw new PropertyTypeInvalidException(name, "Boolean");
    }

    /**
     * Accessor for the specified property as a String.
     * If the specified property isn't found returns null.
     * @param name Name of the property
     * @return String value for the property
     * @throws PropertyTypeInvalidException thrown when the property is not available as this type
     */
    public String getStringProperty(String name)
    {
        Object obj = getProperty(name);
        if (obj != null)
        {
            if (obj instanceof String)
            {
                return ((String)obj);
            }
        }
        else
        {
            return null;
        }
        throw new PropertyTypeInvalidException(name, "String");
    }

    /**
     * Set the value for this persistence property.
     * @param name Name of the property
     * @param value The value
     */
    private void setPropertyInternal(String name, Object value)
    {
        this.properties.put(name.toLowerCase(), value);
    }

    /**
     * Method to set the persistence properties using those defined in a file.
     * @param filename Name of the file containing the properties
     */
    public synchronized void setPropertiesUsingFile(String filename)
    {
        if (filename == null)
        {
            return;
        }

        Properties props = null;
        try
        {
            props = PersistenceUtils.setPropertiesUsingFile(filename);
            setPropertyInternal("datanucleus.propertiesFile", filename);
        }
        catch (NucleusUserException nue)
        {
            properties.remove("datanucleus.propertiesFile");
            throw nue;
        }
        if (props != null && !props.isEmpty())
        {
            setPersistenceProperties(props);
        }
    }

    /**
     * Accessor for the persistence properties default values.
     * This returns the defaulted properties
     * @return The persistence properties
     */
    public Map<String, Object> getPersistencePropertiesDefaults()
    {
        return Collections.unmodifiableMap(defaultProperties);
    }

    /**
     * Accessor for the persistence properties.
     * This returns just the user-supplied properties, not the defaulted properties
     * @see getPersistenceProperties()
     * @return The persistence properties
     */
    public Map<String, Object> getPersistenceProperties()
    {
        return Collections.unmodifiableMap(properties);
    }

    /**
     * Set the properties for this configuration.
     * Note : this has this name so it has a getter/setter pair for use by things like Spring.
     * @see getPersistencePropertiesDefaults()
     * @param props The persistence properties
     */
    public void setPersistenceProperties(Map props)
    {
        Set keys = props.keySet();
        Iterator keyIter = keys.iterator();
        while (keyIter.hasNext())
        {
            Object keyObj = keyIter.next();
            if (keyObj instanceof String)
            {
                String key = (String)keyObj;
                Object valueObj = props.get(keyObj);
                setProperty(key, valueObj);
            }
        }
    }

    /**
     * Convenience method to set a persistence property.
     * Uses any validator defined for the property to govern whether the value is suitable.
     * @param name Name of the property
     * @param value Value
     */
    public void setProperty(String name, Object value)
    {
        if (name != null)
        {
            String propertyName = name.trim();
            PropertyMapping mapping = propertyMappings.get(propertyName.toLowerCase());
            if (mapping != null)
            {
                if (mapping.validatorName != null)
                {
                    // TODO Use ClassLoaderResolver
                    PersistencePropertyValidator validator = null;
                    try
                    {
                        Class validatorCls = Class.forName(mapping.validatorName);
                        validator = (PersistencePropertyValidator)validatorCls.newInstance();
                    }
                    catch (Exception e)
                    {
                        NucleusLogger.PERSISTENCE.warn("Error creating validator of type " + mapping.validatorName, e);
                    }

                    if (validator != null)
                    {
                        boolean validated = (mapping.internalName != null ? 
                                validator.validate(mapping.internalName, value) : 
                                    validator.validate(propertyName, value));
                        if (!validated)
                        {
                            throw new IllegalArgumentException(LOCALISER.msg("008012", propertyName, value));
                        }
                    }
                }

                if (mapping.internalName != null)
                {
                    setPropertyInternal(mapping.internalName, value);
                }
                else
                {
                    setPropertyInternal(mapping.name, value);
                }

                // Special behaviour properties
                if (propertyName.equals("datanucleus.propertiesFile"))
                {
                    // Load all properties from the specified file
                    setPropertiesUsingFile((String)value);
                }
                else if (propertyName.equals("datanucleus.localisation.messageCodes"))
                {
                    // Set global log message code flag
                    boolean included = getBooleanProperty("datanucleus.localisation.messageCodes");
                    Localiser.setDisplayCodesInMessages(included);
                }
                else if (propertyName.equals("datanucleus.localisation.language"))
                {
                    String language = getStringProperty("datanucleus.localisation.language");
                    Localiser.setLanguage(language);
                }
            }
            else
            {
                // Unknown property so just add it.
                setPropertyInternal(propertyName, value);
                if (propertyMappings.size() > 0)
                {
                    NucleusLogger.PERSISTENCE.info(LOCALISER.msg("008015", propertyName));
                }
            }
        }
    }

    /**
     * Equality operator.
     * @param obj Object to compare against.
     * @return Whether the objects are equal.
     */
    public synchronized boolean equals(Object obj)
    {
        if (obj == this)
        {
            return true;
        }
        if (!(obj instanceof PersistenceConfiguration))
        {
            return false;
        }

        PersistenceConfiguration config = (PersistenceConfiguration)obj;
        if (properties == null)
        {
            if (config.properties != null)
            {
                return false;
            }
        }
        else if (!properties.equals(config.properties))
        {
            return false;
        }

        if (defaultProperties == null)
        {
            if (config.defaultProperties != null)
            {
                return false;
            }
        }
        else if (!defaultProperties.equals(config.defaultProperties))
        {
            return false;
        }

        return true;
    }
}