/**********************************************************************
Copyright (c) 2006 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.jdo;

import java.security.AccessController;
import java.security.PrivilegedAction;
import java.sql.SQLException;
import java.util.Collection;

import javax.jdo.JDODataStoreException;
import javax.jdo.JDOException;
import javax.jdo.JDOFatalDataStoreException;
import javax.jdo.JDOFatalInternalException;
import javax.jdo.JDOFatalUserException;
import javax.jdo.JDOHelper;
import javax.jdo.JDOObjectNotFoundException;
import javax.jdo.JDOOptimisticVerificationException;
import javax.jdo.JDOQueryInterruptedException;
import javax.jdo.JDOUnsupportedOptionException;
import javax.jdo.JDOUserException;
import javax.jdo.PersistenceManager;
import javax.jdo.PersistenceManagerFactory;
import javax.jdo.spi.JDOImplHelper;
import javax.jdo.spi.PersistenceCapable;
import javax.transaction.xa.XAException;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.ObjectManager;
import org.datanucleus.ObjectManagerFactoryImpl;
import org.datanucleus.StateManager;
import org.datanucleus.exceptions.ClassNotPersistableException;
import org.datanucleus.exceptions.NucleusDataStoreException;
import org.datanucleus.exceptions.NucleusException;
import org.datanucleus.exceptions.NucleusObjectNotFoundException;
import org.datanucleus.exceptions.NucleusOptimisticException;
import org.datanucleus.exceptions.NucleusUnsupportedOptionException;
import org.datanucleus.exceptions.NucleusUserException;
import org.datanucleus.exceptions.NoPersistenceInformationException;
import org.datanucleus.exceptions.TransactionNotActiveException;
import org.datanucleus.exceptions.TransactionNotReadableException;
import org.datanucleus.exceptions.TransactionNotWritableException;
import org.datanucleus.jdo.exceptions.ClassNotPersistenceCapableException;
import org.datanucleus.metadata.ClassMetaData;
import org.datanucleus.metadata.MetaDataManager;
import org.datanucleus.state.StateManagerFactory;
import org.datanucleus.store.exceptions.DatastoreReadOnlyException;
import org.datanucleus.store.query.QueryInterruptedException;
import org.datanucleus.util.ClassUtils;
import org.datanucleus.util.Localiser;

/**
 * Helper for persistence operations with DataNucleus.
 * Extends JDOHelper so that people can use this class if they wish.
 */
public class NucleusJDOHelper extends JDOHelper
{
    /** Localisation utility for output messages */
    protected static final Localiser LOCALISER = Localiser.getInstance("org.datanucleus.Localisation",
        ObjectManagerFactoryImpl.class.getClassLoader());

    /**
     * Convenience accessor for the query results cache.
     * @param pmf The PMF
     * @return The Query results cache
     */
    public static JDOQueryCache getQueryResultCache(PersistenceManagerFactory pmf)
    {
        return ((JDOPersistenceManagerFactory)pmf).getQueryCache();
    }

    // ---------------------------------- Replication -------------------------------

    /**
     * Convenience method to replicate a group of objects from one datastore (managed by PMF1)
     * to a second datastore (managed by PMF2).
     * @param pmf1 PersistenceManagerFactory for the source of the objects
     * @param pmf2 PersistenceManagerFactory for the target of the objects
     * @param oids Identities of the objects to replicate
     */
    public static void replicate(PersistenceManagerFactory pmf1, PersistenceManagerFactory pmf2,
            Object... oids)
    {
        JDOReplicationManager replicator = new JDOReplicationManager(pmf1, pmf2);
        replicator.replicate(oids);
    }

    /**
     * Convenience method to replicate objects of particular types from one datastore (managed by PMF1)
     * to a second datastore (managed by PMF2).
     * @param pmf1 PersistenceManagerFactory for the source of the objects
     * @param pmf2 PersistenceManagerFactory for the target of the objects
     * @param types Types of objects to replicate
     */
    public static void replicate(PersistenceManagerFactory pmf1, PersistenceManagerFactory pmf2,
            Class... types)
    {
        JDOReplicationManager replicator = new JDOReplicationManager(pmf1, pmf2);
        replicator.replicate(types);
    }

    /**
     * Convenience method to replicate objects of particular types from one datastore (managed by PMF1)
     * to a second datastore (managed by PMF2).
     * @param pmf1 PersistenceManagerFactory for the source of the objects
     * @param pmf2 PersistenceManagerFactory for the target of the objects
     * @param classNames Names of classes to replicate
     */
    public static void replicate(PersistenceManagerFactory pmf1, PersistenceManagerFactory pmf2,
            String... classNames)
    {
        JDOReplicationManager replicator = new JDOReplicationManager(pmf1, pmf2);
        replicator.replicate(classNames);
    }

    // ------------------------------------ MetaData --------------------------------

    /**
     * Accessor for the MetaData for the specified class
     * @param pmf PersistenceManager factory
     * @param cls The class
     * @return The MetaData for the class
     */
    public static ClassMetaData getMetaDataForClass(PersistenceManagerFactory pmf, Class cls)
    {
        if (pmf == null || cls == null)
        {
            return null;
        }
        if (!(pmf instanceof JDOPersistenceManagerFactory))
        {
            return null;
        }

        JDOPersistenceManagerFactory myPMF = (JDOPersistenceManagerFactory)pmf;
        MetaDataManager mdmgr = myPMF.getOMFContext().getMetaDataManager();
        return (ClassMetaData)mdmgr.getMetaDataForClass(cls, myPMF.getOMFContext().getClassLoaderResolver(null));
    }

    /**
     * Accessor for the names of the classes that have MetaData for this PMF.
     * @param pmf The PMF
     * @return The class names
     */
    public static String[] getClassesWithMetaData(PersistenceManagerFactory pmf)
    {
        if (pmf == null || !(pmf instanceof JDOPersistenceManagerFactory))
        {
            return null;
        }

        JDOPersistenceManagerFactory myPMF = (JDOPersistenceManagerFactory)pmf;
        Collection classes = myPMF.getOMFContext().getMetaDataManager().getClassesWithMetaData();
        return (String[])classes.toArray(new String[classes.size()]);
    }

    // ------------------------------ Object Lifecycle --------------------------------

    /**
     * Accessor for the names of the dirty fields of the persistable object.
     * @param obj The persistable object
     * @param pm The Persistence Manager (only required if the object is detached)
     * @return Names of the dirty fields
     */
    public static String[] getDirtyFields(Object obj, PersistenceManager pm)
    {
        if (obj == null || !(obj instanceof PersistenceCapable))
        {
            return null;
        }
        PersistenceCapable pc = (PersistenceCapable)obj;

        if (isDetached(pc))
        {
            org.datanucleus.ObjectManager om = ((JDOPersistenceManager)pm).getObjectManager();

            // Temporarily attach a StateManager to access the detached field information
            StateManager sm = StateManagerFactory.newStateManagerForDetached(om.getExecutionContext(), pc, 
                getObjectId(pc), null);
            pc.jdoReplaceStateManager((javax.jdo.spi.StateManager) sm);
            sm.retrieveDetachState(sm);
            String[] dirtyFieldNames = sm.getDirtyFieldNames();
            pc.jdoReplaceStateManager(null);

            return dirtyFieldNames;
        }
        else
        {
            org.datanucleus.ObjectManager om = ((JDOPersistenceManager)pm).getObjectManager();
            StateManager sm = om.findStateManager(pc);
            if (sm == null)
            {
                return null;
            }
            return sm.getDirtyFieldNames();
        }
    }

    /**
     * Accessor for the names of the loaded fields of the persistable object.
     * @param obj Persistable object
     * @param pm The Persistence Manager (only required if the object is detached)
     * @return Names of the loaded fields
     */
    public static String[] getLoadedFields(Object obj, PersistenceManager pm)
    {
        if (obj == null || !(obj instanceof PersistenceCapable))
        {
            return null;
        }
        PersistenceCapable pc = (PersistenceCapable)obj;

        if (isDetached(pc))
        {
            // Temporarily attach a StateManager to access the detached field information
            org.datanucleus.ObjectManager om = ((JDOPersistenceManager)pm).getObjectManager();
            StateManager sm = StateManagerFactory.newStateManagerForDetached(om.getExecutionContext(), pc, 
                getObjectId(pc), null);
            pc.jdoReplaceStateManager((javax.jdo.spi.StateManager) sm);
            sm.retrieveDetachState(sm);
            String[] loadedFieldNames = sm.getLoadedFieldNames();
            pc.jdoReplaceStateManager(null);

            return loadedFieldNames;
        }
        else
        {
            org.datanucleus.ObjectManager om = ((JDOPersistenceManager)pm).getObjectManager();
            StateManager sm = om.findStateManager(pc);
            if (sm == null)
            {
                return null;
            }
            return sm.getLoadedFieldNames();
        }
    }

    /**
     * Accessor for whether the specified member (field/property) of the passed persistable object is loaded.
     * @param obj The persistable object
     * @param memberName Name of the field/property
     * @param pm PersistenceManager (if the object is detached)
     * @return Whether the member is loaded
     */
    public static Boolean isLoaded(Object obj, String memberName, PersistenceManager pm)
    {
        if (obj == null || !(obj instanceof PersistenceCapable))
        {
            return null;
        }
        PersistenceCapable pc = (PersistenceCapable)obj;

        if (isDetached(pc))
        {
            // Temporarily attach a StateManager to access the detached field information
            org.datanucleus.ObjectManager om = ((JDOPersistenceManager)pm).getObjectManager();
            StateManager sm = StateManagerFactory.newStateManagerForDetached(om.getExecutionContext(), pc,
                getObjectId(pc), null);
            pc.jdoReplaceStateManager((javax.jdo.spi.StateManager) sm);
            sm.retrieveDetachState(sm);
            int position = sm.getClassMetaData().getAbsolutePositionOfMember(memberName);
            boolean loaded = sm.isFieldLoaded(position);
            pc.jdoReplaceStateManager(null);

            return loaded;
        }
        else
        {
            ObjectManager om = ((JDOPersistenceManager)pc.jdoGetPersistenceManager()).getObjectManager();
            StateManager sm = om.findStateManager(pc);
            if (sm == null)
            {
                return null;
            }
            int position = sm.getClassMetaData().getAbsolutePositionOfMember(memberName);
            return sm.isFieldLoaded(position);
        }
    }

    /**
     * Accessor for whether the specified member (field/property) of the passed persistable object is dirty.
     * @param obj The persistable object
     * @param memberName Name of the field/property
     * @param pm PersistenceManager (if the object is detached)
     * @return Whether the member is dirty
     */
    public static Boolean isDirty(Object obj, String memberName, PersistenceManager pm)
    {
        if (obj == null || !(obj instanceof PersistenceCapable))
        {
            return null;
        }
        PersistenceCapable pc = (PersistenceCapable)obj;

        if (isDetached(pc))
        {
            // Temporarily attach a StateManager to access the detached field information
            org.datanucleus.ObjectManager om = ((JDOPersistenceManager)pm).getObjectManager();
            StateManager sm = StateManagerFactory.newStateManagerForDetached(om.getExecutionContext(), pc,
                getObjectId(pc), null);
            pc.jdoReplaceStateManager((javax.jdo.spi.StateManager) sm);
            sm.retrieveDetachState(sm);
            int position = sm.getClassMetaData().getAbsolutePositionOfMember(memberName);
            boolean[] dirtyFieldNumbers = sm.getDirtyFields();
            pc.jdoReplaceStateManager(null);

            return dirtyFieldNumbers[position];
        }
        else
        {
            ObjectManager om = ((JDOPersistenceManager)pc.jdoGetPersistenceManager()).getObjectManager();
            StateManager sm = om.findStateManager(pc);
            if (sm == null)
            {
                return null;
            }
            int position = sm.getClassMetaData().getAbsolutePositionOfMember(memberName);
            boolean[] dirtyFieldNumbers = sm.getDirtyFields();
            return dirtyFieldNumbers[position];
        }
    }

    // ------------------------------ Convenience --------------------------------

    /**
     * Convenience method to convert an exception into a JDO exception.
     * If the incoming exception has a "failed object" then create the new exception with
     * a failed object. Otherwise if the incoming exception has nested exceptions then
     * create this exception with those nested exceptions. Else create this exception with
     * the incoming exception as its nested exception.
     * @param jpe NucleusException
     * @return The JDOException
     */
    public static JDOException getJDOExceptionForNucleusException(NucleusException jpe)
    {
        // Specific exceptions first
        if (jpe instanceof ClassNotPersistableException)
        {
            return new ClassNotPersistenceCapableException(jpe.getMessage(), jpe);
        }
        else if (jpe instanceof NoPersistenceInformationException)
        {
            return new org.datanucleus.jdo.exceptions.NoPersistenceInformationException(jpe.getMessage(), jpe);
        }
        else if (jpe instanceof TransactionNotReadableException)
        {
            return new org.datanucleus.jdo.exceptions.TransactionNotReadableException(jpe.getMessage(), jpe.getCause());
        }
        else if (jpe instanceof TransactionNotWritableException)
        {
            return new org.datanucleus.jdo.exceptions.TransactionNotWritableException(jpe.getMessage(), jpe.getCause());
        }
        else if (jpe instanceof TransactionNotActiveException)
        {
            return new org.datanucleus.jdo.exceptions.TransactionNotActiveException(jpe.getMessage(), jpe);
        }
        else if (jpe instanceof QueryInterruptedException)
        {
            return new JDOQueryInterruptedException(jpe.getMessage());
        }
        else if (jpe instanceof NucleusUnsupportedOptionException)
        {
            return new JDOUnsupportedOptionException(jpe.getMessage(), jpe);
        }
        else if (jpe instanceof DatastoreReadOnlyException)
        {
            ClassLoaderResolver clr = ((DatastoreReadOnlyException)jpe).getClassLoaderResolver();
            try
            {
                Class cls = clr.classForName("javax.jdo.JDOReadOnlyException");
                throw (JDOUserException)ClassUtils.newInstance(cls, 
                    new Class[] {String.class}, new Object[] {jpe.getMessage()});
            }
            catch (NucleusException ne)
            {
                // No JDOReadOnlyException so JDO1.0-JDO2.1
                throw new JDOUserException(jpe.getMessage());
            }
        }
        else if (jpe instanceof NucleusDataStoreException)
        {
            if (jpe.isFatal())
            {
                //sadly JDOFatalDataStoreException dont allow nested exceptions and failed objects together
                if (jpe.getFailedObject() != null)
                {
                    return new JDOFatalDataStoreException(jpe.getMessage(), jpe.getFailedObject());
                }
                else if (jpe.getNestedExceptions() != null)
                {
                    return new JDOFatalDataStoreException(jpe.getMessage(), jpe.getNestedExceptions());
                }
                else
                {
                    return new JDOFatalDataStoreException(jpe.getMessage(), jpe);
                }
            }
            else
            {
                if (jpe.getNestedExceptions() != null)
                {
                    if (jpe.getFailedObject() != null)
                    {
                        return new JDODataStoreException(jpe.getMessage(), jpe.getNestedExceptions(), jpe.getFailedObject());
                    }
                    return new JDODataStoreException(jpe.getMessage(), jpe.getNestedExceptions());
                }
                else if (jpe.getFailedObject() != null)
                {
                    JDOPersistenceManager.LOGGER.info("Exception thrown", jpe);
                    return new JDODataStoreException(jpe.getMessage(), jpe.getFailedObject());
                }
                else
                {
                    JDOPersistenceManager.LOGGER.info("Exception thrown", jpe);
                    return new JDODataStoreException(jpe.getMessage(), jpe);
                }
            }
        }
        else if (jpe instanceof NucleusObjectNotFoundException)
        {
            if (jpe.getFailedObject() != null)
            {
                if (jpe.getNestedExceptions() != null)
                {
                    return new JDOObjectNotFoundException(jpe.getMessage(), jpe.getNestedExceptions(), jpe.getFailedObject());
                }
                else
                {
                    return new JDOObjectNotFoundException(jpe.getMessage(), jpe, jpe.getFailedObject());
                }
            }
            else if (jpe.getNestedExceptions() != null)
            {
                return new JDOObjectNotFoundException(jpe.getMessage(), jpe.getNestedExceptions());
            }
            else
            {
                return new JDOObjectNotFoundException(jpe.getMessage(), new Throwable[]{jpe});
            }
        }
        else if (jpe instanceof NucleusUserException)
        {
            if (jpe.isFatal())
            {
                if (jpe.getNestedExceptions() != null)
                {
                    if (jpe.getFailedObject() != null)
                    {
                        return new JDOFatalUserException(jpe.getMessage(), jpe.getNestedExceptions(), jpe.getFailedObject());
                    }
                    return new JDOFatalUserException(jpe.getMessage(), jpe.getNestedExceptions());
                }
                else if (jpe.getFailedObject() != null)
                {
                    JDOPersistenceManager.LOGGER.info("Exception thrown", jpe);
                    return new JDOFatalUserException(jpe.getMessage(), jpe.getFailedObject());
                }
                else
                {
                    JDOPersistenceManager.LOGGER.info("Exception thrown", jpe);
                    return new JDOFatalUserException(jpe.getMessage(), jpe);
                }
            }
            else
            {
                if (jpe.getNestedExceptions() != null)
                {
                    if (jpe.getFailedObject() != null)
                    {
                        return new JDOUserException(jpe.getMessage(), jpe.getNestedExceptions(), jpe.getFailedObject());
                    }
                    return new JDOUserException(jpe.getMessage(), jpe.getNestedExceptions());
                }
                else if (jpe.getFailedObject() != null)
                {
                    JDOPersistenceManager.LOGGER.info("Exception thrown", jpe);
                    return new JDOUserException(jpe.getMessage(), jpe.getFailedObject());
                }
                else
                {
                    JDOPersistenceManager.LOGGER.info("Exception thrown", jpe);
                    return new JDOUserException(jpe.getMessage(), jpe);
                }
            }
        }
        else if (jpe instanceof NucleusOptimisticException)
        {
            //sadly JDOOptimisticVerificationException dont allow nested exceptions and failed objects together
            if (jpe.getFailedObject() != null)
            {
                return new JDOOptimisticVerificationException(jpe.getMessage(), jpe.getFailedObject());
            }
            else if (jpe.getNestedExceptions() != null)
            {
                return new JDOOptimisticVerificationException(jpe.getMessage(), jpe.getNestedExceptions());
            }
            else
            {
                return new JDOOptimisticVerificationException(jpe.getMessage(), jpe);
            }
        }
        else if (jpe instanceof org.datanucleus.transaction.HeuristicRollbackException 
                && jpe.getNestedExceptions().length==1 
                && jpe.getNestedExceptions()[0].getCause() instanceof SQLException) 
        {
            // if there was a single failure in the transaction, we want to properly propagate the 
            // single nested SQLException that was the cause of the failure, so application code
            // can decide on what to do with it
            return new JDODataStoreException(jpe.getMessage(), ((XAException)jpe.getNestedExceptions()[0]).getCause());
        }
        else
        {
            if (jpe.isFatal())
            {
                if (jpe.getNestedExceptions() != null)
                {
                    return new JDOFatalInternalException(jpe.getMessage(), jpe.getNestedExceptions());
                }
                else
                {
                    return new JDOFatalInternalException(jpe.getMessage(), jpe);
                }
            }
            else if (jpe.getNestedExceptions() != null)
            {
                return new JDOException(jpe.getMessage(), jpe.getNestedExceptions());
            }
            else
            {
                return new JDOException(jpe.getMessage(), jpe);
            }
        }
    }

    /**
     * Get the JDOImplHelper instance.
     * This must be done in a doPrivileged block.
     * @return The JDOImplHelper.
     */
    public static JDOImplHelper getJDOImplHelper() 
    {
        return (JDOImplHelper) AccessController.doPrivileged(new PrivilegedAction()
            {
                public Object run()
                {
                    try
                    {
                        return JDOImplHelper.getInstance();
                    }
                    catch (SecurityException e)
                    {
                        throw new JDOFatalUserException(LOCALISER.msg("026000"), e);
                    }
                }
            });
    }
}