/**********************************************************************
Copyright (c) 2002 Mike Martin (TJDO) and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License. 
 

Contributors:
2003 Andy Jefferson - coding standards
2005 Andy Jefferson - allow for embedded keys/values in join table
2005 Andy Jefferson - add ROF extraction of key/value in same iterator stmt
    ...
**********************************************************************/
package org.datanucleus.store.mapped.scostore;

import java.util.Collection;
import java.util.Iterator;
import java.util.Map.Entry;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.metadata.AbstractMemberMetaData;
import org.datanucleus.store.ObjectProvider;
import org.datanucleus.store.mapped.DatastoreContainerObject;
import org.datanucleus.store.mapped.mapping.JavaTypeMapping;
import org.datanucleus.store.scostore.MapStore;
import org.datanucleus.store.scostore.SetStore;

/**
 * Representation of backing store for a Map EntrySet.
 */
public abstract class MapEntrySetStore extends BaseContainerStore implements SetStore
{
    /** Table containing the key and value forming the entry. */
    protected DatastoreContainerObject mapTable;

    /** The backing store for the Map. */
    protected MapStore mapStore;

    /** Mapping for the key. */
    protected JavaTypeMapping keyMapping;

    /** Mapping for the value. */
    protected JavaTypeMapping valueMapping;

    protected ClassLoaderResolver clr;

    /**
     * Constructor for a store for the entries of a map.
     * @param mapTable The table handling the relation (can be join table, or key table, or value table)
     * @param ownerMemberMetaData Metadata for the owning field/property
     * @param ownerMapping Mapping back to the owner from this table
     * @param keyMapping Mapping for the key
     * @param valueMapping Mapping for the value
     * @param mapStore Backing store for the map
     * @param clr ClassLoader resolver
     */
    public MapEntrySetStore(DatastoreContainerObject mapTable, AbstractMemberMetaData ownerMemberMetaData,
        JavaTypeMapping ownerMapping, JavaTypeMapping keyMapping, JavaTypeMapping valueMapping,
        MapStore mapStore, ClassLoaderResolver clr)
    {
        super(mapTable.getStoreManager());

        this.mapTable = mapTable;
        this.mapStore = mapStore;
        this.ownerMapping = ownerMapping;
        this.keyMapping   = keyMapping;
        this.valueMapping = valueMapping;
        this.ownerMemberMetaData = ownerMemberMetaData;
        this.clr = clr;
    }

    /**
     * Accessor for whether this store has an order mapping to allow for duplicates, or ordering.
     * @return Whether it has an order mapping.
     */
    public boolean hasOrderMapping()
    {
        return false;
    }

    /**
     * Method to update an embedded element.
     * @param sm State Manager of the owner
     * @param element The element to update
     * @param fieldNumber The number of the field to update
     * @param value The value
     * @return Whether the element was modified
     */
    public boolean updateEmbeddedElement(ObjectProvider sm, Object element, int fieldNumber, Object value)
    {
        // Do nothing since of no use here
        return false;
    }

    /**
     * Accessor for the owner mapping.
     * @return The owner mapping
     */
    public JavaTypeMapping getOwnerMapping()
    {
        return ownerMapping;
    }

    protected boolean validateElementType(Object element)
    {
        return element instanceof Entry;
    }

    /**
     * Accessor for an iterator for the entries of the Map.
     * Implemented by the subclass using whatever querying mechanism is suitable for the underlying datastore.
     * @param sm State manager for the owner
     * @return The iterator
     */
    public abstract Iterator iterator(ObjectProvider sm);


    /**
     * Accessor for the size of the Map.
     * @param sm The state manager
     * @return The size
     */
    public abstract int size(ObjectProvider sm);

    /**
     * Method to update the collection to be the supplied collection of elements.
     * @param sm StateManager of the object
     * @param coll The collection to use
     */
    public void update(ObjectProvider sm, Collection coll)
    {
        // Crude update - remove existing and add new!
        // TODO Update this to just remove what is not needed, and add what is really new
        clear(sm);
        addAll(sm, coll, 0);
    }

    public boolean contains(ObjectProvider sm, Object element)
    {
        if (!validateElementType(element))
        {
            return false;
        }
        Entry entry = (Entry)element;

        return mapStore.containsKey(sm, entry.getKey());
    }

    /**
     * Method to add an entry to the Map.
     * @param sm State Manager for the owner
     * @param element Entry to add
     * @return Whether it was added
     */
    public boolean add(ObjectProvider sm, Object element, int size)
    {
        throw new UnsupportedOperationException("Cannot add to a map through its entry set");
    }

    /**
     * Method to add entries to the Map.
     * @param sm State Manager for the owner
     * @param elements Entries to add
     * @return Whether they were added
     */
    public boolean addAll(ObjectProvider sm, Collection elements, int size)
    {
        throw new UnsupportedOperationException("Cannot add to a map through its entry set");
    }

    /**
     * Method to remove an entry from the Map.
     * @param sm State Manager for the owner
     * @param element Entry to remove
     * @return Whether it was removed
     */
    public boolean remove(ObjectProvider sm, Object element, int size, boolean allowDependentField)
    {
        if (!validateElementType(element))
        {
            return false;
        }

        Entry entry = (Entry)element;
        Object removed = mapStore.remove(sm, entry.getKey());

        // NOTE: this may not return an accurate result if a null value is being removed
        return removed == null ? entry.getValue() == null : removed.equals(entry.getValue());
    }

    /**
     * Method to remove entries from the Map.
     * @param sm State Manager for the owner
     * @param elements Entries to remove
     * @return Whether they were removed
     */
    public boolean removeAll(ObjectProvider sm, Collection elements, int size)
    {
        if (elements == null || elements.size() == 0)
        {
            return false;
        }

        Iterator iter=elements.iterator();
        boolean modified=false;
        while (iter.hasNext())
        {
            Object element=iter.next();
            Entry entry = (Entry)element;

            Object removed = mapStore.remove(sm, entry.getKey());

            // NOTE: this may not return an accurate result if a null value is being removed.
            modified = removed == null ? entry.getValue() == null : removed.equals(entry.getValue());
        }

        return modified;
    }

    /**
     * Method to clear the Map.
     * @param sm State Manager for the owner.
     */
    public void clear(ObjectProvider sm)
    {
        mapStore.clear(sm);
    }

    public MapStore getMapStore()
    {
        return mapStore;
    }

    public JavaTypeMapping getKeyMapping()
    {
        return keyMapping;
    }

    public JavaTypeMapping getValueMapping()
    {
        return valueMapping;
    }
}