/**********************************************************************
Copyright (c) 2002 Kelly Grizzle (TJDO) and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
 

Contributors:
2002 Mike Martin (TJDO)
2003 Erik Bengtson - removed unused variable and import
2003 Andy Jefferson - coding standards
2004 Andy Jefferson - moved statements to AbstractSetStore
    ...
**********************************************************************/
package org.datanucleus.store.mapped.scostore;

import java.util.Collection;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.metadata.AbstractMemberMetaData;
import org.datanucleus.store.ObjectProvider;
import org.datanucleus.store.mapped.DatastoreContainerObject;
import org.datanucleus.store.mapped.mapping.JavaTypeMapping;
import org.datanucleus.store.scostore.MapStore;
import org.datanucleus.util.ClassUtils;

/**
 * Representation of the backing store for the keys of a Map.
 * This is used where the user calls Map.keySet() and then wants to perform some operation on the resulting Set.
 * The keys for a Map can be stored in several ways. There are the following possibilities
 * <UL>
 * <LI><B>Map using join table</B> - join table contains the key and value (or FK's to their tables)</LI>
 * <LI><B>Map using key table</B> - Key table stores the value (or an FK to its table)</LI>
 * <LI><B>Map using value table</B> - Value table stores the key (or an FK to its table)</LI>
 * </UL>
 */
public abstract class MapKeySetStore extends AbstractSetStore
{
    protected final MapStore mapStore;

    /**
     * Constructor for a backing store for the keys of a map relation.
     * @param mapTable The table for the map (table storing the map linkage)
     * @param ownerMmd Metadata for the owner member
     * @param ownerMapping mapping from the map table to the owner table
     * @param keyMapping mapping from the map table to the key
     * @param mapStore The backing store for the map
     * @param clr The ClassLoaderResolver
     * @param specialization The specialisation
     */
    public MapKeySetStore(DatastoreContainerObject mapTable, AbstractMemberMetaData ownerMmd, 
            JavaTypeMapping ownerMapping, JavaTypeMapping keyMapping, MapStore mapStore, 
            ClassLoaderResolver clr, AbstractSetStoreSpecialization specialization)
    {
        super(mapTable.getStoreManager(), clr, specialization);

        this.mapStore = mapStore;
        this.containerTable = mapTable;
        this.ownerMemberMetaData = ownerMmd;
        this.ownerMapping = ownerMapping;
        this.elementMapping = keyMapping;

        initialize(clr);
    }

    /**
     * Initialisation method.
     */
    private void initialize(ClassLoaderResolver clr)
    {
        elementType = elementMapping.getType();
        elementsAreEmbedded = isEmbeddedMapping(elementMapping);
        elementsAreSerialised = isEmbeddedMapping(elementMapping);

        // Load the element class
        Class element_class = clr.classForName(elementType);

        if (ClassUtils.isReferenceType(element_class))
        {
            emd = storeMgr.getOMFContext().getMetaDataManager().getMetaDataForImplementationOfReference(element_class, null, clr);
        }
        else
        {
            emd = storeMgr.getOMFContext().getMetaDataManager().getMetaDataForClass(element_class, clr);
        }
        if (emd != null)
        {
            elementType = emd.getFullClassName();
            elementInfo = getElementInformationForClass();
        } 
    }

    /**
     * Method to add an element. Overridden because we want to prevent it.
     * @param sm State Manager of collection.
     * @param element Element to add.
     * @return Whether it was successful
     **/
    public boolean add(ObjectProvider sm, Object element, int size)
    {
        throw new UnsupportedOperationException("Cannot add to a map through its key set");
    }

    /**
     * Method to add a collection of elements. Overridden because we want to prevent it.
     * @param sm State Manager of collection.
     * @param elements Elements to add.
     * @return Whether it was successful
     **/
    public boolean addAll(ObjectProvider sm, Collection elements, int size)
    {
        throw new UnsupportedOperationException("Cannot add to a map through its key set");
    }

    /**
     * Method to remove an element. Overridden because we want to prevent it.
     * @param sm State Manager of collection.
     * @param element Element to remove.
     * @return Whether it was successful
     **/
    public boolean remove(ObjectProvider sm, Object element, int size, boolean allowDependentField)
    {
        if (!canRemove())
        {
            throw new UnsupportedOperationException("Cannot remove from an inverse map through its key set");
        }

        return super.remove(sm, element, size, allowDependentField);
    }

    /**
     * Method to remove a collection of elements. Overridden because we want to prevent it.
     * @param sm State Manager of collection.
     * @param elements Elements to remove.
     * @return Whether it was successful
     **/
    public boolean removeAll(ObjectProvider sm, Collection elements, int size)
    {
        if (!canRemove())
        {
            throw new UnsupportedOperationException("Cannot remove from an inverse map through its key set");
        }

        return super.removeAll(sm, elements, size);
    }

    /**
     * Method to clear the collection. Overridden because we want to prevent it.
     * @param sm State Manager of collection.
     **/
    public void clear(ObjectProvider sm)
    {
        if (!canClear())
        {
            throw new UnsupportedOperationException("Cannot clear an inverse map through its key set");
        }

        super.clear(sm);
    }

    protected abstract boolean canRemove();
    protected abstract boolean canClear();
}