/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.quic.common;

import java.io.IOException;
import java.net.SocketAddress;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedChannelException;
import java.util.Arrays;
import java.util.Collection;
import java.util.EventListener;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Consumer;
import org.eclipse.jetty.io.ByteBufferPool;
import org.eclipse.jetty.io.Connection;
import org.eclipse.jetty.io.CyclicTimeout;
import org.eclipse.jetty.io.RetainableByteBuffer;
import org.eclipse.jetty.quic.common.CloseInfo;
import org.eclipse.jetty.quic.common.ProtocolSession;
import org.eclipse.jetty.quic.common.QuicConnection;
import org.eclipse.jetty.quic.common.QuicErrorCode;
import org.eclipse.jetty.quic.common.QuicStreamEndPoint;
import org.eclipse.jetty.quic.common.StreamType;
import org.eclipse.jetty.quic.quiche.QuicheConnection;
import org.eclipse.jetty.quic.quiche.QuicheConnectionId;
import org.eclipse.jetty.util.BufferUtil;
import org.eclipse.jetty.util.Callback;
import org.eclipse.jetty.util.IteratingCallback;
import org.eclipse.jetty.util.component.ContainerLifeCycle;
import org.eclipse.jetty.util.component.DumpableCollection;
import org.eclipse.jetty.util.component.LifeCycle;
import org.eclipse.jetty.util.thread.Invocable;
import org.eclipse.jetty.util.thread.Scheduler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class QuicSession
extends ContainerLifeCycle {
    private static final Logger LOG = LoggerFactory.getLogger(QuicSession.class);
    private final AtomicLong[] ids = new AtomicLong[StreamType.values().length];
    private final ConcurrentMap<Long, QuicStreamEndPoint> endPoints = new ConcurrentHashMap<Long, QuicStreamEndPoint>();
    private final Executor executor;
    private final Scheduler scheduler;
    private final ByteBufferPool byteBufferPool;
    private final QuicheConnection quicheConnection;
    private final QuicConnection connection;
    private final Flusher flusher;
    private SocketAddress remoteAddress;
    private volatile ProtocolSession protocolSession;
    private QuicheConnectionId quicheConnectionId;
    private long idleTimeout;

    protected QuicSession(Executor executor, Scheduler scheduler, ByteBufferPool byteBufferPool, QuicheConnection quicheConnection, QuicConnection connection, SocketAddress remoteAddress) {
        this.executor = executor;
        this.scheduler = scheduler;
        this.byteBufferPool = byteBufferPool;
        this.quicheConnection = quicheConnection;
        this.connection = connection;
        this.flusher = new Flusher(scheduler);
        this.addBean((Object)this.flusher);
        this.remoteAddress = remoteAddress;
        Arrays.setAll(this.ids, i -> new AtomicLong());
    }

    protected void doStart() throws Exception {
        super.doStart();
        this.getEventListeners().stream().filter(Listener.class::isInstance).map(Listener.class::cast).forEach(this::notifyOpened);
    }

    private void notifyOpened(Listener listener) {
        try {
            listener.onOpened(this);
        }
        catch (Throwable x) {
            LOG.info("failure notifying listener {}", (Object)listener, (Object)x);
        }
    }

    protected void doStop() throws Exception {
        super.doStop();
        this.getEventListeners().stream().filter(Listener.class::isInstance).map(Listener.class::cast).forEach(this::notifyClosed);
    }

    private void notifyClosed(Listener listener) {
        try {
            listener.onClosed(this);
        }
        catch (Throwable x) {
            LOG.info("failure notifying listener {}", (Object)listener, (Object)x);
        }
    }

    public CompletableFuture<Void> shutdown() {
        ProtocolSession session = this.protocolSession;
        if (session != null) {
            return session.shutdown();
        }
        return CompletableFuture.completedFuture(null);
    }

    public Executor getExecutor() {
        return this.executor;
    }

    public Scheduler getScheduler() {
        return this.scheduler;
    }

    public ByteBufferPool getByteBufferPool() {
        return this.byteBufferPool;
    }

    public ProtocolSession getProtocolSession() {
        return this.protocolSession;
    }

    public int getMaxLocalStreams() {
        return this.quicheConnection.maxLocalStreams();
    }

    public String getNegotiatedProtocol() {
        return this.quicheConnection.getNegotiatedProtocol();
    }

    public QuicConnection getQuicConnection() {
        return this.connection;
    }

    public Collection<QuicStreamEndPoint> getQuicStreamEndPoints() {
        return List.copyOf(this.endPoints.values());
    }

    public CloseInfo getRemoteCloseInfo() {
        QuicheConnection.CloseInfo info = this.quicheConnection.getRemoteCloseInfo();
        if (info != null) {
            return new CloseInfo(info.error(), info.reason());
        }
        return null;
    }

    public long getIdleTimeout() {
        return this.idleTimeout;
    }

    public void setIdleTimeout(long idleTimeout) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("setting idle timeout {} ms for {}", (Object)idleTimeout, (Object)this);
        }
        this.idleTimeout = idleTimeout;
    }

    public boolean onIdleTimeout() {
        return this.protocolSession.onIdleTimeout();
    }

    public void onFailure(Throwable failure) {
        this.protocolSession.onFailure(QuicErrorCode.NO_ERROR.code(), "failure", failure);
    }

    public long newStreamId(StreamType streamType) {
        int type = streamType.type();
        long id = this.ids[type].getAndIncrement();
        return (id << 2) + (long)type;
    }

    public int fill(long streamId, ByteBuffer buffer) throws IOException {
        int drained = this.quicheConnection.drainClearBytesForStream(streamId, buffer);
        this.flush();
        return drained;
    }

    public int flush(long streamId, ByteBuffer buffer, boolean last) throws IOException {
        int flushed = this.quicheConnection.feedClearBytesForStream(streamId, buffer, last);
        this.flush();
        return flushed;
    }

    public boolean isFinished(long streamId) {
        return this.quicheConnection.isStreamFinished(streamId);
    }

    public long getWindowCapacity() {
        return this.quicheConnection.windowCapacity();
    }

    public long getWindowCapacity(long streamId) throws IOException {
        return this.quicheConnection.windowCapacity(streamId);
    }

    public void shutdownInput(long streamId, long error) throws IOException {
        this.quicheConnection.shutdownStream(streamId, false, error);
        this.flush();
    }

    public void shutdownOutput(long streamId, long error) throws IOException {
        this.quicheConnection.shutdownStream(streamId, true, error);
        this.flush();
    }

    public void remove(QuicStreamEndPoint endPoint, Throwable failure) {
        boolean removed;
        boolean bl = removed = this.endPoints.remove(endPoint.getStreamId()) != null;
        if (removed) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("removed {} from {}", (Object)endPoint, (Object)this);
            }
            endPoint.closed(failure);
        }
    }

    public SocketAddress getLocalAddress() {
        return this.connection.getEndPoint().getLocalSocketAddress();
    }

    public SocketAddress getRemoteAddress() {
        return this.remoteAddress;
    }

    public boolean isConnectionEstablished() {
        return this.quicheConnection.isConnectionEstablished();
    }

    public QuicheConnectionId getConnectionId() {
        return this.quicheConnectionId;
    }

    public void setConnectionId(QuicheConnectionId quicheConnectionId) {
        this.quicheConnectionId = quicheConnectionId;
    }

    public Runnable process(SocketAddress remoteAddress, ByteBuffer cipherBufferIn) throws IOException {
        int accepted;
        this.remoteAddress = remoteAddress;
        int remaining = cipherBufferIn.remaining();
        if (LOG.isDebugEnabled()) {
            LOG.debug("feeding {} cipher bytes to {}", (Object)remaining, (Object)this);
        }
        if ((accepted = this.quicheConnection.feedCipherBytes(cipherBufferIn, (SocketAddress)this.connection.getLocalInetSocketAddress(), remoteAddress)) != remaining) {
            throw new IllegalStateException();
        }
        if (this.isConnectionEstablished()) {
            ProtocolSession protocol = this.protocolSession;
            if (protocol == null) {
                this.protocolSession = protocol = this.createProtocolSession();
                this.addManaged((LifeCycle)protocol);
            }
            if (LOG.isDebugEnabled()) {
                LOG.debug("processing {}", (Object)protocol);
            }
            return protocol.getProducerTask();
        }
        this.flush();
        return null;
    }

    protected Runnable pollTask() {
        return null;
    }

    protected abstract ProtocolSession createProtocolSession();

    List<Long> getWritableStreamIds() {
        return this.quicheConnection.writableStreamIds();
    }

    List<Long> getReadableStreamIds() {
        return this.quicheConnection.readableStreamIds();
    }

    QuicStreamEndPoint getStreamEndPoint(long streamId) {
        return (QuicStreamEndPoint)((Object)this.endPoints.get(streamId));
    }

    public abstract Connection newConnection(QuicStreamEndPoint var1);

    public void flush() {
        if (LOG.isDebugEnabled()) {
            LOG.debug("flushing {}", (Object)this);
        }
        this.flusher.iterate();
    }

    public QuicStreamEndPoint getOrCreateStreamEndPoint(long streamId, Consumer<QuicStreamEndPoint> consumer) {
        AtomicBoolean created = new AtomicBoolean();
        QuicStreamEndPoint endPoint = this.endPoints.computeIfAbsent(streamId, id -> {
            if (LOG.isDebugEnabled()) {
                LOG.debug("creating endpoint for stream #{} for {}", id, (Object)this);
            }
            QuicStreamEndPoint result = this.newQuicStreamEndPoint((long)id);
            created.set(true);
            return result;
        });
        if (created.get()) {
            consumer.accept(endPoint);
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("returning {} for {}", (Object)endPoint, (Object)this);
        }
        return endPoint;
    }

    private QuicStreamEndPoint newQuicStreamEndPoint(long streamId) {
        return new QuicStreamEndPoint(this.getScheduler(), this, streamId);
    }

    public void inwardClose(long error, String reason) {
        this.protocolSession.inwardClose(error, reason);
    }

    public void outwardClose(long error, String reason) {
        boolean closed = this.quicheConnection.close(error, reason);
        if (LOG.isDebugEnabled()) {
            LOG.debug("outward closing ({}) 0x{}/{} on {}", new Object[]{closed, Long.toHexString(error), reason, this});
        }
        if (closed) {
            this.flush();
        }
    }

    private void finishOutwardClose(Throwable failure) {
        try {
            this.endPoints.clear();
            this.getQuicConnection().outwardClose(this, failure);
        }
        finally {
            this.quicheConnection.dispose();
        }
    }

    public void dump(Appendable out, String indent) throws IOException {
        this.dumpObjects(out, indent, new Object[]{new DumpableCollection("endPoints", this.getQuicStreamEndPoints())});
    }

    public String toString() {
        return String.format("%s@%x[id=%s]", ((Object)((Object)this)).getClass().getSimpleName(), ((Object)((Object)this)).hashCode(), this.quicheConnectionId);
    }

    private class Flusher
    extends IteratingCallback {
        private final CyclicTimeout timeout;
        private RetainableByteBuffer cipherBuffer;

        public Flusher(Scheduler scheduler) {
            this.timeout = new CyclicTimeout(scheduler){

                public void onTimeoutExpired() {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("quiche timeout expired {}", (Object)QuicSession.this);
                    }
                    QuicSession.this.quicheConnection.onTimeout();
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("re-iterating after quiche timeout {}", (Object)QuicSession.this);
                    }
                    QuicSession.this.getExecutor().execute(() -> Flusher.this.iterate());
                }
            };
        }

        protected IteratingCallback.Action process() throws IOException {
            this.cipherBuffer = QuicSession.this.byteBufferPool.acquire(QuicSession.this.connection.getOutputBufferSize(), QuicSession.this.connection.isUseOutputDirectByteBuffers());
            ByteBuffer cipherByteBuffer = this.cipherBuffer.getByteBuffer();
            int pos = BufferUtil.flipToFill((ByteBuffer)cipherByteBuffer);
            int drained = QuicSession.this.quicheConnection.drainCipherBytes(cipherByteBuffer);
            if (LOG.isDebugEnabled()) {
                LOG.debug("drained {} byte(s) of cipher bytes from {}", (Object)drained, (Object)QuicSession.this);
            }
            long nextTimeoutInMs = QuicSession.this.quicheConnection.nextTimeout();
            if (LOG.isDebugEnabled()) {
                LOG.debug("next quiche timeout: {} ms on {}", (Object)nextTimeoutInMs, (Object)QuicSession.this);
            }
            if (nextTimeoutInMs < 0L) {
                this.timeout.cancel();
            } else {
                this.timeout.schedule(nextTimeoutInMs, TimeUnit.MILLISECONDS);
            }
            if (drained == 0) {
                IteratingCallback.Action action;
                boolean connectionClosed = QuicSession.this.quicheConnection.isConnectionClosed();
                IteratingCallback.Action action2 = action = connectionClosed ? IteratingCallback.Action.SUCCEEDED : IteratingCallback.Action.IDLE;
                if (LOG.isDebugEnabled()) {
                    LOG.debug("connection draining={} closed={}, action={} on {}", new Object[]{QuicSession.this.quicheConnection.isDraining(), connectionClosed, action, QuicSession.this});
                }
                if (action == IteratingCallback.Action.IDLE) {
                    this.cipherBuffer.release();
                }
                return action;
            }
            BufferUtil.flipToFlush((ByteBuffer)cipherByteBuffer, (int)pos);
            if (LOG.isDebugEnabled()) {
                LOG.debug("writing cipher bytes for {} on {}", (Object)QuicSession.this.remoteAddress, (Object)QuicSession.this);
            }
            QuicSession.this.connection.write((Callback)this, QuicSession.this.remoteAddress, cipherByteBuffer);
            return IteratingCallback.Action.SCHEDULED;
        }

        public void succeeded() {
            if (LOG.isDebugEnabled()) {
                LOG.debug("written cipher bytes on {}", (Object)QuicSession.this);
            }
            this.cipherBuffer.release();
            super.succeeded();
        }

        public Invocable.InvocationType getInvocationType() {
            return Invocable.InvocationType.NON_BLOCKING;
        }

        protected void onCompleteSuccess() {
            if (LOG.isDebugEnabled()) {
                LOG.debug("connection closed {}", (Object)QuicSession.this);
            }
            this.finish(new ClosedChannelException());
        }

        protected void onCompleteFailure(Throwable failure) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("failed to write cipher bytes, closing session on {}", (Object)QuicSession.this, (Object)failure);
            }
            this.finish(failure);
        }

        private void finish(Throwable failure) {
            this.cipherBuffer.release();
            QuicSession.this.finishOutwardClose(failure);
            this.timeout.destroy();
        }
    }

    public static interface Listener
    extends EventListener {
        default public void onOpened(QuicSession session) {
        }

        default public void onClosed(QuicSession session) {
        }
    }
}

