/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.milo.opcua.stack.core.application;

import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Sets;
import com.google.common.util.concurrent.Uninterruptibles;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufUtil;
import io.netty.buffer.Unpooled;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.URLEncoder;
import java.nio.file.ClosedWatchServiceException;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.security.cert.CertificateEncodingException;
import java.security.cert.X509Certificate;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.eclipse.milo.opcua.stack.core.UaException;
import org.eclipse.milo.opcua.stack.core.application.CertificateValidator;
import org.eclipse.milo.opcua.stack.core.util.CertificateUtil;
import org.eclipse.milo.opcua.stack.core.util.CertificateValidationUtil;
import org.eclipse.milo.opcua.stack.core.util.DigestUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DefaultCertificateValidator
implements CertificateValidator,
AutoCloseable {
    private final Logger logger = LoggerFactory.getLogger(this.getClass());
    private final Set<X509Certificate> trustedCertificates = Sets.newConcurrentHashSet();
    private final Set<X509Certificate> authorityCertificates = Sets.newConcurrentHashSet();
    private final File trustedDir;
    private final File rejectedDir;
    private final File revocationDir;
    private WatchService watchService;
    private Thread thread;

    public DefaultCertificateValidator(File certificatesBaseDir) {
        this.trustedDir = new File(certificatesBaseDir.getAbsolutePath() + File.separator + "trusted");
        if (!this.trustedDir.exists() && !this.trustedDir.mkdirs()) {
            this.logger.warn("Could not create trusted certificate dir: {}", (Object)this.trustedDir);
        }
        this.rejectedDir = new File(certificatesBaseDir.getAbsolutePath() + File.separator + "rejected");
        if (!this.rejectedDir.exists() && !this.rejectedDir.mkdirs()) {
            this.logger.warn("Could not create rejected certificate dir: {}", (Object)this.rejectedDir);
        }
        this.revocationDir = new File(certificatesBaseDir.getAbsolutePath() + File.separator + "revocation");
        if (!this.revocationDir.exists() && !this.revocationDir.mkdirs()) {
            this.logger.warn("Could not create revocation certificate dir: {}", (Object)this.revocationDir);
        }
        this.createWatchService();
        this.synchronizeTrustedCertificates();
    }

    public DefaultCertificateValidator(@Nonnull File trustedDir, @Nullable File rejectedDir, @Nullable File revocationDir) {
        Objects.requireNonNull(trustedDir);
        this.trustedDir = trustedDir.getAbsoluteFile();
        this.rejectedDir = rejectedDir != null ? rejectedDir.getAbsoluteFile() : null;
        File file = this.revocationDir = revocationDir != null ? revocationDir.getAbsoluteFile() : null;
        if (!trustedDir.isDirectory()) {
            throw new IllegalArgumentException(String.format("Directory of trusted certificates could not be found: %s", trustedDir.getAbsolutePath()));
        }
        if (rejectedDir != null && !rejectedDir.isDirectory()) {
            throw new IllegalArgumentException(String.format("Directory of rejected certificates must be an existing, writable directory: %s", rejectedDir.getAbsolutePath()));
        }
        if (revocationDir != null && !revocationDir.isDirectory()) {
            throw new IllegalArgumentException(String.format("Directory of revoked certificates must be an existing, writable directory: %s", revocationDir.getAbsolutePath()));
        }
        this.createWatchService();
        this.synchronizeTrustedCertificates();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() throws IOException {
        Thread thread;
        WatchService watchService;
        DefaultCertificateValidator defaultCertificateValidator = this;
        synchronized (defaultCertificateValidator) {
            this.logger.info("Closing default certificate validator");
            watchService = this.watchService;
            thread = this.thread;
            this.watchService = null;
            this.thread = null;
            this.trustedCertificates.clear();
            this.authorityCertificates.clear();
        }
        if (watchService != null) {
            watchService.close();
        }
        if (thread != null) {
            Uninterruptibles.joinUninterruptibly((Thread)thread);
        }
    }

    @Override
    public void validate(X509Certificate certificate) throws UaException {
        try {
            CertificateValidationUtil.validateCertificateValidity(certificate);
        }
        catch (UaException e) {
            this.certificateRejected(certificate);
            throw e;
        }
    }

    @Override
    public synchronized void verifyTrustChain(List<X509Certificate> certificateChain) throws UaException {
        try {
            CertificateValidationUtil.verifyTrustChain(certificateChain, this.trustedCertificates, this.authorityCertificates);
        }
        catch (UaException e) {
            X509Certificate certificate = certificateChain.get(0);
            this.certificateRejected(certificate);
            throw e;
        }
    }

    public synchronized ImmutableSet<X509Certificate> getTrustedCertificates() {
        return ImmutableSet.copyOf(this.trustedCertificates);
    }

    public synchronized ImmutableSet<X509Certificate> getAuthorityCertificates() {
        return ImmutableSet.copyOf(this.authorityCertificates);
    }

    private void createWatchService() {
        try {
            this.watchService = this.trustedDir.toPath().getFileSystem().newWatchService();
            WatchKey trustedKey = this.trustedDir.toPath().register(this.watchService, StandardWatchEventKinds.ENTRY_CREATE, StandardWatchEventKinds.ENTRY_DELETE, StandardWatchEventKinds.ENTRY_MODIFY);
            this.thread = new Thread(new Watcher(this.watchService, trustedKey));
            this.thread.setName("ua-certificate-directory-watcher");
            this.thread.setDaemon(true);
            this.thread.start();
        }
        catch (IOException e) {
            this.logger.error("Error creating WatchService.", (Throwable)e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void synchronizeTrustedCertificates() {
        this.logger.debug("Synchronizing trusted certificates...");
        Set<X509Certificate> certificates = this.certificatesFromDir(this.trustedDir);
        Set trusted = certificates.stream().filter(c -> c.getBasicConstraints() == -1).collect(Collectors.toSet());
        Set authority = certificates.stream().filter(c -> c.getBasicConstraints() != -1).collect(Collectors.toSet());
        DefaultCertificateValidator defaultCertificateValidator = this;
        synchronized (defaultCertificateValidator) {
            this.trustedCertificates.clear();
            this.trustedCertificates.addAll(trusted);
            this.authorityCertificates.clear();
            this.authorityCertificates.addAll(authority);
        }
        this.logger.debug("trustedCertificates.size()={}, authorityCertificates.size()={}", (Object)this.trustedCertificates.size(), (Object)this.authorityCertificates.size());
    }

    private Set<X509Certificate> certificatesFromDir(File dir) {
        File[] files = dir.listFiles();
        if (files == null) {
            files = new File[]{};
        }
        return Arrays.stream(files).map(this::file2certificate).filter(c -> c != null).collect(Collectors.toSet());
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private X509Certificate file2certificate(File f) {
        try (FileInputStream inputStream = new FileInputStream(f);){
            X509Certificate x509Certificate = CertificateUtil.decodeCertificate(inputStream);
            return x509Certificate;
        }
        catch (Throwable ignored) {
            return null;
        }
    }

    private void certificateRejected(X509Certificate certificate) {
        if (this.rejectedDir == null) {
            return;
        }
        try {
            String[] ss = certificate.getSubjectX500Principal().getName().split(",");
            String name = ss.length > 0 ? ss[0] : certificate.getSubjectX500Principal().getName();
            String thumbprint = ByteBufUtil.hexDump((ByteBuf)Unpooled.wrappedBuffer((byte[])DigestUtil.sha1(certificate.getEncoded())));
            String filename = String.format("%s [%s].der", URLEncoder.encode(name, "UTF-8"), thumbprint);
            File f = new File(this.rejectedDir.getAbsolutePath() + File.separator + filename);
            try (FileOutputStream fos = new FileOutputStream(f);){
                fos.write(certificate.getEncoded());
                fos.flush();
            }
            this.logger.debug("Added rejected certificate entry: {}", (Object)filename);
        }
        catch (IOException | CertificateEncodingException e) {
            this.logger.error("Error adding rejected certificate entry.", (Throwable)e);
        }
    }

    private class Watcher
    implements Runnable {
        private final WatchService watchService;
        private final WatchKey trustedKey;

        Watcher(WatchService watchService, WatchKey trustedKey) {
            this.watchService = watchService;
            this.trustedKey = trustedKey;
        }

        @Override
        public void run() {
            while (true) {
                try {
                    WatchKey key;
                    do {
                        if ((key = this.watchService.take()) != this.trustedKey) continue;
                        for (WatchEvent<?> watchEvent : key.pollEvents()) {
                            WatchEvent.Kind<?> kind = watchEvent.kind();
                            if (kind == StandardWatchEventKinds.OVERFLOW) continue;
                            DefaultCertificateValidator.this.synchronizeTrustedCertificates();
                        }
                    } while (key.reset());
                    DefaultCertificateValidator.this.logger.warn("Failed to reset watch key");
                }
                catch (ClosedWatchServiceException e) {
                    DefaultCertificateValidator.this.logger.info("Watcher got closed");
                    return;
                }
                catch (InterruptedException e) {
                    DefaultCertificateValidator.this.logger.error("Watcher interrupted.", (Throwable)e);
                    continue;
                }
                break;
            }
        }
    }
}

