/*
 * Decompiled with CFR 0.152.
 */
package org.everrest.websockets.client;

import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.net.Socket;
import java.net.URI;
import java.nio.charset.Charset;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Random;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.apache.commons.codec.binary.Base64;
import org.everrest.core.util.Logger;
import org.everrest.websockets.client.ClientMessageListener;

public class WSClient {
    public static final int DEFAULT_MAX_MESSAGE_PAYLOAD_SIZE = 0x200000;
    private static final Logger LOG;
    private static final String GLOBAL_WS_SERVER_UUID = "258EAFA5-E914-47DA-95CA-C5AB0DC85B11";
    private static final Random RANDOM;
    private static final Charset UTF8_CS;
    private static final char[] CHARS;
    private static final int MASK_SIZE = 4;
    private final ExecutorService executor;
    private final URI target;
    private final int maxMessagePayloadSize;
    private final String secWebSocketKey;
    private final List<ClientMessageListener> listeners;
    private Socket socket;
    private InputStream in;
    private OutputStream out;
    private volatile boolean connected;

    public WSClient(URI target, ClientMessageListener ... listeners) {
        this(target, 0x200000, listeners);
    }

    public WSClient(URI target, int maxMessagePayloadSize, ClientMessageListener ... listeners) {
        if (target == null) {
            throw new IllegalArgumentException("Connection URI may not be null. ");
        }
        if (!"ws".equals(target.getScheme())) {
            throw new IllegalArgumentException("Unsupported scheme: " + target.getScheme());
        }
        if (maxMessagePayloadSize < 1) {
            throw new IllegalArgumentException("Invalid max message payload size: " + maxMessagePayloadSize);
        }
        if (listeners == null) {
            throw new IllegalArgumentException("listeners may not be null. ");
        }
        this.target = target;
        this.maxMessagePayloadSize = maxMessagePayloadSize;
        this.executor = Executors.newSingleThreadExecutor();
        this.listeners = new ArrayList<ClientMessageListener>(listeners.length);
        Collections.addAll(this.listeners, listeners);
        this.secWebSocketKey = this.generateSecKey();
    }

    public synchronized void connect(long timeout) throws IOException {
        if (timeout < 1L) {
            throw new IllegalArgumentException("Invalid timeout: " + timeout);
        }
        if (this.connected) {
            throw new IOException("Already connected.");
        }
        try {
            this.executor.submit(new Runnable(){

                @Override
                public void run() {
                    try {
                        WSClient.this.socket = new Socket(WSClient.this.target.getHost(), WSClient.this.target.getPort());
                        WSClient.this.in = WSClient.this.socket.getInputStream();
                        WSClient.this.out = WSClient.this.socket.getOutputStream();
                        WSClient.this.out.write(WSClient.this.getHandshake());
                        WSClient.this.validateResponseHeaders();
                        WSClient.this.connected = true;
                    }
                    catch (IOException e) {
                        throw new RuntimeException(e);
                    }
                }
            }).get(timeout, TimeUnit.SECONDS);
        }
        catch (InterruptedException e) {
            throw new IOException(e.getMessage(), e);
        }
        catch (ExecutionException e) {
            RuntimeException re = (RuntimeException)e.getCause();
            Throwable cause = re.getCause();
            if (cause instanceof IOException) {
                throw (IOException)cause;
            }
            throw re;
        }
        catch (TimeoutException e) {
            throw new IOException("Connection timeout. ");
        }
        finally {
            if (!this.connected) {
                this.executor.shutdown();
            }
        }
        this.executor.submit(new Runnable(){

            @Override
            public void run() {
                try {
                    WSClient.this.read();
                }
                catch (ConnectionException e) {
                    LOG.error(e.getMessage(), (Throwable)e);
                    WSClient.this.onClose(e.status, e.getMessage());
                }
                catch (IOException e) {
                    LOG.error(e.getMessage(), (Throwable)e);
                    WSClient.this.onClose(1002, e.getMessage());
                }
            }
        });
        this.onOpen();
    }

    public synchronized void disconnect() throws IOException {
        if (!this.connected) {
            return;
        }
        this.writeFrame((byte)-120, new byte[0]);
    }

    public synchronized void send(String message) throws IOException {
        if (!this.connected) {
            throw new IOException("Not connected. ");
        }
        if (message == null) {
            throw new IllegalArgumentException("Message may not be null. ");
        }
        this.writeFrame((byte)-127, UTF8_CS.encode(message).array());
    }

    public synchronized void ping(byte[] message) throws IOException {
        if (!this.connected) {
            throw new IOException("Not connected. ");
        }
        if (message == null) {
            message = new byte[]{};
        } else if (message.length > 125) {
            throw new IllegalArgumentException("Ping message to large, may not be greater than 125 bytes. ");
        }
        this.writeFrame((byte)-119, message);
    }

    protected String getOrigin() {
        return null;
    }

    protected String[] getSubProtocols() {
        return null;
    }

    private byte[] getHandshake() {
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        PrintWriter handshake = new PrintWriter(out);
        handshake.format("GET %s HTTP/1.1\r\n", this.target.getPath());
        int port = this.target.getPort();
        if (port == 80) {
            handshake.format("Host: %s\r\n", this.target.getHost());
        } else {
            handshake.format("Host: %s:%d\r\n", this.target.getHost(), port);
        }
        handshake.append("Upgrade: Websocket\r\n");
        handshake.append("Connection: Upgrade\r\n");
        Object[] subProtocol = this.getSubProtocols();
        if (subProtocol != null && subProtocol.length > 0) {
            handshake.format("Sec-WebSocket-Protocol: %s\r\n", Arrays.toString(subProtocol));
        }
        handshake.format("Sec-WebSocket-Key: %s\r\n", this.secWebSocketKey);
        handshake.format("Sec-WebSocket-Version: %d\r\n", 13);
        handshake.append("Sec-WebSocket-Protocol: chat\r\n");
        String origin = this.getOrigin();
        if (origin != null) {
            handshake.format("Origin: %s\r\n", origin);
        }
        handshake.append('\r');
        handshake.append('\n');
        handshake.flush();
        return out.toByteArray();
    }

    private void onOpen() {
        for (ClientMessageListener listener : this.listeners) {
            try {
                listener.onOpen(this);
            }
            catch (Exception e) {
                LOG.error(e.getMessage(), (Throwable)e);
            }
        }
    }

    private void onMessage(String message) {
        for (ClientMessageListener listener : this.listeners) {
            try {
                listener.onMessage(message);
            }
            catch (Exception e) {
                LOG.error(e.getMessage(), (Throwable)e);
            }
        }
    }

    private void onPong(byte[] message) {
        for (ClientMessageListener listener : this.listeners) {
            try {
                listener.onPong(message);
            }
            catch (Exception e) {
                LOG.error(e.getMessage(), (Throwable)e);
            }
        }
    }

    private void onClose(int status, String message) {
        try {
            this.socket.close();
        }
        catch (IOException e) {
            LOG.error(e.getMessage(), (Throwable)e);
        }
        for (ClientMessageListener listener : this.listeners) {
            try {
                listener.onClose(status, message);
            }
            catch (Exception e) {
                LOG.error(e.getMessage(), (Throwable)e);
            }
        }
        this.listeners.clear();
        this.executor.shutdown();
        this.connected = false;
    }

    private String generateSecKey() {
        int length = RANDOM.nextInt(CHARS.length);
        byte[] b = new byte[length];
        for (int i = 0; i < length; ++i) {
            b[i] = (byte)CHARS[RANDOM.nextInt(CHARS.length)];
        }
        return Base64.encodeBase64String((byte[])b);
    }

    private byte[] generateMask() {
        byte[] mask = new byte[4];
        RANDOM.nextBytes(mask);
        return mask;
    }

    private byte[] getLengthAsBytes(long length) {
        if (length <= 125L) {
            return new byte[]{(byte)length};
        }
        if (length <= 65535L) {
            byte[] bytes = new byte[]{126, (byte)(length >> 8), (byte)(length & 0xFFL)};
            return bytes;
        }
        byte[] bytes = new byte[]{127, 0, 0, 0, 0, (byte)(length >> 24), (byte)(length >> 16), (byte)(length >> 8), (byte)(length & 0xFFL)};
        return bytes;
    }

    private void validateResponseHeaders() throws IOException {
        MessageDigest md;
        BufferedReader br = new BufferedReader(new InputStreamReader(this.in));
        String line = br.readLine();
        if (!"HTTP/1.1 101 Switching Protocols".equals(line)) {
            throw new IOException("Invalid server response. Expected status is 101 'Switching Protocols'. ");
        }
        HashMap<String, String> headers = new HashMap<String, String>();
        while ((line = br.readLine()) != null && !line.isEmpty()) {
            int colon = line.indexOf(58);
            if (colon <= 0 || colon >= line.length()) continue;
            headers.put(line.substring(0, colon).trim().toLowerCase(), line.substring(colon + 1).trim());
        }
        String header = (String)headers.get("upgrade");
        if (!"websocket".equals(header)) {
            throw new IOException("Invalid 'Upgrade' response header. Returned '" + header + "' but 'websocket' expected. ");
        }
        header = (String)headers.get("connection");
        if (!"upgrade".equals(header)) {
            throw new IOException("Invalid 'Connection' response header. Returned '" + header + "' but 'upgrade' expected. ");
        }
        try {
            md = MessageDigest.getInstance("SHA-1");
        }
        catch (NoSuchAlgorithmException e) {
            throw new IllegalStateException(e.getMessage(), e);
        }
        md.reset();
        byte[] digest = md.digest((this.secWebSocketKey + GLOBAL_WS_SERVER_UUID).getBytes());
        String expectedWsSecurityAccept = Base64.encodeBase64String((byte[])digest);
        header = (String)headers.get("sec-websocket-accept");
        if (!expectedWsSecurityAccept.equals(header)) {
            throw new IOException("Invalid 'Sec-WebSocket-Accept' response header.");
        }
    }

    private void read() throws IOException {
        while (this.connected) {
            int firstByte = this.in.read();
            if (firstByte < 0) {
                throw new EOFException("Failed read next websocket frame, end of the stream was reached. ");
            }
            if ((firstByte & 0x80) == 0) {
                throw new ConnectionException(1003, "Fragmented messages is nor supported. ");
            }
            byte opCode = (byte)(firstByte & 0xF);
            switch (opCode) {
                case 0: {
                    throw new ConnectionException(1003, "Continuation frame is not supported. ");
                }
                case 1: {
                    byte[] payload = this.readFrame();
                    this.onMessage(new String(payload, UTF8_CS));
                    break;
                }
                case 2: {
                    throw new ConnectionException(1003, "Binary messages is not supported. ");
                }
                case 3: 
                case 4: 
                case 5: 
                case 6: 
                case 7: {
                    break;
                }
                case 8: {
                    int status;
                    byte[] payload = this.readFrame();
                    if (payload.length > 0) {
                        status = (payload[0] & 0xFF) << 8;
                        status += payload[1] & 0xFF;
                    } else {
                        status = 0;
                    }
                    this.onClose(status, null);
                    if (status == 0 || status == 1000) break;
                    String message = null;
                    if (payload.length > 2) {
                        message = new String(payload, 2, payload.length - 2, UTF8_CS);
                    }
                    LOG.warn("Close status: {}, message: {} ", (Object)status, message);
                    break;
                }
                case 9: {
                    byte[] payload = this.readFrame();
                    this.writeFrame((byte)-118, payload);
                    LOG.debug("Ping: {} ", (Object)new String(payload, UTF8_CS));
                    break;
                }
                case 10: {
                    byte[] payload = this.readFrame();
                    this.onPong(payload);
                    break;
                }
                case 11: 
                case 12: 
                case 13: 
                case 14: 
                case 15: {
                    break;
                }
                default: {
                    throw new ConnectionException(1003, "Invalid opcode: '" + Integer.toHexString(opCode) + "' ");
                }
            }
            if (!this.socket.isClosed()) continue;
            this.onClose(1006, null);
        }
    }

    private byte[] readFrame() throws IOException {
        byte[] block;
        int secondByte = this.in.read();
        if (secondByte < 0) {
            throw new EOFException("Failed read next websocket frame, end of the stream was reached. ");
        }
        boolean masked = (secondByte & 0x80) > 0;
        long length = secondByte & 0x7F;
        if (length == 126L) {
            block = new byte[2];
            this.readBlock(block);
            length = this.getPayloadLength(block);
        } else if (length == 127L) {
            block = new byte[8];
            this.readBlock(block);
            length = this.getPayloadLength(block);
        }
        byte[] mask = null;
        if (masked) {
            mask = new byte[4];
            this.readBlock(mask);
        }
        if (length > (long)this.maxMessagePayloadSize) {
            throw new IOException("Message payload is to large, may not be greater than " + this.maxMessagePayloadSize);
        }
        byte[] payload = new byte[(int)length];
        this.readBlock(payload);
        if (mask != null) {
            for (int i = 0; i < payload.length; ++i) {
                payload[i] = (byte)(payload[i] ^ mask[i % 4]);
            }
        }
        return payload;
    }

    private void writeFrame(byte opCode, byte[] payload) throws IOException {
        byte[] lengthBytes = this.getLengthAsBytes(payload.length);
        lengthBytes[0] = (byte)(lengthBytes[0] | 0x80);
        byte[] mask = this.generateMask();
        this.out.write(opCode);
        this.out.write(lengthBytes);
        this.out.write(mask);
        int length = payload.length;
        for (int i = 0; i < length; ++i) {
            this.out.write(payload[i] ^ mask[i % 4]);
        }
        this.out.flush();
    }

    private long getPayloadLength(byte[] bytes) throws IOException {
        if (bytes.length != 2 && bytes.length != 8) {
            throw new IOException("Unable get payload length. Invalid length bytes. Length must be represented by 2 or 8 bytes but " + bytes.length + " reached. ");
        }
        return this.getLongFromBytes(bytes);
    }

    private long getLongFromBytes(byte[] bytes) throws IOException {
        long length = 0L;
        int i = bytes.length - 1;
        int shift = 0;
        while (i >= 0) {
            length += (long)((bytes[i] & 0xFF) << shift);
            --i;
            shift += 8;
        }
        return length;
    }

    private void readBlock(byte[] buff) throws IOException {
        int r;
        int length = buff.length;
        for (int offset = 0; offset < buff.length; offset += r) {
            r = this.in.read(buff, offset, length - offset);
            if (offset >= 0) continue;
            throw new EOFException("Failed read next websocket frame, end of the stream was reached. ");
        }
    }

    static {
        int c;
        LOG = Logger.getLogger(WSClient.class);
        RANDOM = new Random();
        UTF8_CS = Charset.forName("UTF-8");
        CHARS = new char[36];
        int i = 0;
        for (c = 48; c <= 57; ++c) {
            WSClient.CHARS[i++] = (char)c;
        }
        for (c = 97; c <= 122; ++c) {
            WSClient.CHARS[i++] = (char)c;
        }
    }

    private static class ConnectionException
    extends IOException {
        private final int status;

        private ConnectionException(int status, String message) {
            super(message);
            this.status = status;
        }
    }
}

