/*
 * Decompiled with CFR 0.152.
 */
package org.everrest.websockets.client;

import com.google.common.base.Throwables;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.IOException;
import java.net.SocketTimeoutException;
import java.net.URI;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import javax.websocket.ClientEndpoint;
import javax.websocket.CloseReason;
import javax.websocket.ContainerProvider;
import javax.websocket.DeploymentException;
import javax.websocket.EncodeException;
import javax.websocket.OnClose;
import javax.websocket.OnError;
import javax.websocket.OnMessage;
import javax.websocket.OnOpen;
import javax.websocket.RemoteEndpoint;
import javax.websocket.Session;
import javax.websocket.WebSocketContainer;
import org.everrest.core.impl.provider.json.JsonException;
import org.everrest.core.impl.uri.UriComponent;
import org.everrest.websockets.client.ClientMessageListener;
import org.everrest.websockets.message.BaseTextEncoder;
import org.everrest.websockets.message.InputMessage;
import org.everrest.websockets.message.JsonMessageConverter;
import org.everrest.websockets.message.MessageSender;
import org.everrest.websockets.message.RestInputMessage;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ClientEndpoint(encoders={InputMessageEncoder.class})
public class WSClient {
    private static final Logger LOG = LoggerFactory.getLogger(WSClient.class);
    private static ExecutorService executor = Executors.newCachedThreadPool(new ThreadFactoryBuilder().setNameFormat("everrest.WSClient-%d").setDaemon(true).build());
    private final URI serverUri;
    private final List<ClientMessageListener> listeners;
    private final List<String> channels;
    private Session session;
    private MessageSender messageSender;

    public WSClient(URI serverUri, ClientMessageListener ... listeners) {
        this(Builder.create(serverUri).listeners(listeners));
    }

    public WSClient(Builder builder) {
        this.serverUri = builder.serverUri;
        this.listeners = builder.listeners;
        this.channels = builder.channels;
    }

    public void connect(int timeout) throws IOException, DeploymentException {
        if (timeout < 1) {
            throw new IllegalArgumentException(String.format("Invalid timeout: %d", timeout));
        }
        final WebSocketContainer container = ContainerProvider.getWebSocketContainer();
        container.setAsyncSendTimeout(1L);
        try {
            executor.submit(new Callable<Void>(){

                @Override
                public Void call() throws Exception {
                    Session session = container.connectToServer((Object)WSClient.this, WSClient.this.serverUri);
                    RemoteEndpoint.Basic remoteEndpoint = session.getBasicRemote();
                    for (String channel : WSClient.this.channels) {
                        remoteEndpoint.sendObject((Object)RestInputMessage.newSubscribeChannelMessage(WSClient.uuid(), channel));
                    }
                    return null;
                }
            }).get(timeout, TimeUnit.SECONDS);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        catch (TimeoutException e) {
            throw new SocketTimeoutException("Connection timeout");
        }
        catch (ExecutionException e) {
            Throwables.propagateIfPossible((Throwable)e.getCause(), IOException.class, DeploymentException.class);
            Throwables.propagate((Throwable)e);
        }
    }

    public void disconnect() throws IOException {
        if (this.isConnected()) {
            this.session.close();
        }
    }

    public URI getServerUri() {
        return this.serverUri;
    }

    public boolean isConnected() {
        return this.session != null && this.session.isOpen();
    }

    private void checkIsConnected() {
        if (!this.isConnected()) {
            throw new IllegalStateException("Unable send message because the WebSocket connection has been closed");
        }
    }

    public void send(String message) throws IOException {
        this.checkIsConnected();
        this.messageSender.send(message);
    }

    public void send(byte[] message) throws IOException {
        this.checkIsConnected();
        this.messageSender.send(message);
    }

    public void send(InputMessage message) throws IOException, EncodeException {
        this.checkIsConnected();
        this.messageSender.send(message);
    }

    public void subscribeToChannel(String channel) throws IOException {
        try {
            this.send(RestInputMessage.newSubscribeChannelMessage(WSClient.uuid(), channel));
        }
        catch (EncodeException e) {
            Throwables.propagate((Throwable)e);
        }
        LOG.debug("Subscribed to channel {}", (Object)channel);
    }

    public void unsubscribeFromChannel(String channel) throws IOException {
        try {
            this.send(RestInputMessage.newUnsubscribeChannelMessage(WSClient.uuid(), channel));
        }
        catch (EncodeException e) {
            Throwables.propagate((Throwable)e);
        }
        LOG.debug("Unsubscribed from channel {}", (Object)channel);
    }

    private static String uuid() {
        return UUID.randomUUID().toString();
    }

    @OnOpen
    public void onOpen(Session session) {
        LOG.debug("WS session {} started", (Object)session.getId());
        this.session = session;
        this.messageSender = new MessageSender(session);
        for (ClientMessageListener listener : this.listeners) {
            listener.onOpen(this);
        }
    }

    @OnMessage
    public void processTextMessage(String message) {
        for (ClientMessageListener listener : this.listeners) {
            listener.onMessage(message);
        }
    }

    @OnMessage
    public void processBinaryMessage(byte[] message) {
        for (ClientMessageListener listener : this.listeners) {
            listener.onMessage(message);
        }
    }

    @OnClose
    public void onClose(Session session, CloseReason closeReason) {
        LOG.debug("WS session {} about to be closed, {}", (Object)session.getId(), (Object)closeReason);
        for (ClientMessageListener listener : this.listeners) {
            listener.onClose(closeReason.getCloseCode().getCode(), closeReason.getReasonPhrase());
        }
    }

    @OnError
    public void onError(Throwable error) {
        LOG.warn(error.getMessage(), error);
    }

    public static class Builder {
        private final URI serverUri;
        private final List<ClientMessageListener> listeners;
        private final List<String> channels;

        public static Builder create(URI serverUri) {
            return new Builder(serverUri);
        }

        public Builder(URI serverUri) {
            if (serverUri == null) {
                throw new IllegalArgumentException("Connection URI may not be null");
            }
            this.serverUri = serverUri;
            this.listeners = new LinkedList<ClientMessageListener>();
            this.channels = new LinkedList<String>();
            List channelsFromUri = (List)UriComponent.parseQueryString((String)serverUri.getRawQuery(), (boolean)true).get((Object)"channel");
            if (channelsFromUri != null) {
                this.channels.addAll(channelsFromUri);
            }
        }

        public Builder listeners(ClientMessageListener ... listeners) {
            Collections.addAll(this.listeners, listeners);
            return this;
        }

        public Builder listeners(Collection<ClientMessageListener> listeners) {
            this.listeners.addAll(listeners);
            return this;
        }

        public Builder channels(String ... channels) {
            Collections.addAll(this.channels, channels);
            return this;
        }

        public Builder channels(Collection<String> channels) {
            this.channels.addAll(channels);
            return this;
        }

        public WSClient build() {
            return new WSClient(this);
        }
    }

    public static class InputMessageEncoder
    extends BaseTextEncoder<InputMessage> {
        private final JsonMessageConverter jsonMessageConverter = new JsonMessageConverter();

        public String encode(InputMessage output) throws EncodeException {
            try {
                return this.jsonMessageConverter.toString(output);
            }
            catch (JsonException e) {
                throw new EncodeException((Object)output, e.getMessage(), (Throwable)e);
            }
        }
    }
}

