package org.exoplatform.cs.service;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;


import javax.annotation.security.RolesAllowed;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.CacheControl;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.UriInfo;
import javax.ws.rs.ext.RuntimeDelegate;
import org.exoplatform.container.ExoContainer;
import org.exoplatform.container.ExoContainerContext;
import org.exoplatform.container.PortalContainer;
import org.exoplatform.container.RootContainer;
import org.exoplatform.cs.dao.LogDAO;
import org.exoplatform.cs.dto.TicketDTO;
import org.exoplatform.cs.entity.LogEntity;
import org.exoplatform.cs.entity.TopicEntity;
import org.exoplatform.services.log.ExoLogger;
import org.exoplatform.services.log.Log;
import org.exoplatform.services.rest.impl.RuntimeDelegateImpl;
import org.exoplatform.services.rest.resource.ResourceContainer;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * Created by eXo Platform SAS.
 *
 * @author Ali Hamdi
 * @since 03/04/17
 */
@Path("customerspace")
public class TicketUpdatesService implements ResourceContainer {

  private static final Log LOG = ExoLogger.getLogger(TicketUpdatesService.class);
  private static final CacheControl cc;
  private static final String PATTERN = "dd-MM-yyyy HH:mm:ss Z";
  private static final SimpleDateFormat sdf = new SimpleDateFormat(PATTERN);


  static {
    RuntimeDelegate.setInstance(new RuntimeDelegateImpl());
    cc = new CacheControl();
    cc.setNoCache(true);
    cc.setNoStore(true);
  }

  /**
   * This endpoint will provide the lately updated tickets
   * @param uriInfo URI info
   * @param from Date to get updates from
   * @param to Date to get updates to
   * @return Json array of updated tickets
   */
  @GET
  @RolesAllowed("users")
  @Path("cstickets/updates.json")
  @Produces(MediaType.APPLICATION_JSON)
  public Response getUpdates(@Context UriInfo uriInfo,
                             @QueryParam("from") String from,
                             @QueryParam("to") String to) {
    TicketService ticketService = getService(TicketService.class, null);
    LogDAO logDAO = getService(LogDAO.class, null);
    String DATE_FORMAT = "dd-MM-yyyy";
    DateFormat df = new SimpleDateFormat(DATE_FORMAT);
    Calendar fromDate = Calendar.getInstance();
    Calendar toDate = Calendar.getInstance();
    try {
      if(from != null && !from.isEmpty()) {
        fromDate.setTime(df.parse(from));
      } else {
        fromDate.add(Calendar.MONTH, -1); // default value for from is the value of To Date minus one month
      }
      if(to != null && !to.isEmpty()) {
        toDate.setTime(df.parse(to));
      }
    } catch (ParseException e) {
      LOG.error("Start and End Date format should be {}", DATE_FORMAT);
      return Response.status(Response.Status.INTERNAL_SERVER_ERROR).build();
    }
    List<LogEntity> updates = logDAO.getupdates(fromDate,toDate);
    String response = buildJsonArrayOfUpdates(updates);
    return Response.ok(response, MediaType.APPLICATION_JSON).cacheControl(cc).build();
  }

  private String buildJsonArrayOfUpdates(List<LogEntity> updatesList) {
    JSONArray ticketsJSON = new JSONArray();
    Map<TopicEntity,List<LogEntity>> updatesMap = new HashMap<>();
    for(LogEntity logEntity : updatesList) {
      updatesMap.computeIfAbsent(logEntity.getTopic(), k -> new ArrayList<>()).add(logEntity);
    }
    updatesMap.forEach((k,v)-> {
      JSONObject ticketJSON = new JSONObject();
        try {
          ticketJSON.put("ticket", buildJsonObject(k));
          for (LogEntity entity : v) {
            JSONObject update = new JSONObject();
            update.put("id", entity.getTopic().getId());
            update.put("topicTitle", entity.getTopic().getTitle());
            update.put("updater", entity.getUserID());
            update.put("updateType", entity.getType().name());
            update.put("oldValue", entity.getOriginalValue());
            update.put("newValue", entity.getNewValue());
            update.put("when", sdf.format(entity.getWhen().getTime()));
            ticketJSON.append("updates",update);
          }
          ticketsJSON.put(ticketJSON);
        }catch(Exception e) {
          LOG.error("A problem when building JSON for LogEntity {}", k.getId(), e);
        }
      }
    );
    return ticketsJSON.toString();
    }

  /**
   * 
   * @param clazz service class
   * @param containerName portal container name
   * @param <T> Type of class
   * @return the selected service
   */
  public static <T> T getService(Class<T> clazz, String containerName) {
    ExoContainer container = ExoContainerContext.getCurrentContainer();
    if (containerName != null) {
      container = RootContainer.getInstance().getPortalContainer(containerName);
    }
    if (container.getComponentInstanceOfType(clazz) == null) {
      containerName = PortalContainer.getCurrentPortalContainerName();
      container = RootContainer.getInstance().getPortalContainer(containerName);
    }
    return clazz.cast(container.getComponentInstanceOfType(clazz));
  }

  private JSONObject buildJsonObject(TopicEntity topicEntity) {
    JSONObject ticket = new JSONObject();
    try {
      ticket.put("id", topicEntity.getId());
      ticket.put("assignee", topicEntity.getAssignee());
      ticket.put("environmentName", topicEntity.getEnvironment() != null ? topicEntity.getEnvironment().getName():"");
      ticket.put("infoType", topicEntity.getInfoType() != null ? topicEntity.getInfoType().name() : "");
      ticket.put("link", topicEntity.getLink());
      ticket.put("severity", topicEntity.getSeverity() != null ? topicEntity.getSeverity().name() : "");
      ticket.put("spaceGroupId", topicEntity.getSpace().getGroupId());
      ticket.put("title", topicEntity.getTitle());
      ticket.put("type", topicEntity.getType() != null ? topicEntity.getType().name() : "");
      ticket.put("status", topicEntity.getStatus());
      ticket.put("updateDate", sdf.format(topicEntity.getUpdateDate().getTime()));
    } catch (JSONException jsonException) {
      LOG.error("A problem when building JSON for ticket {}", topicEntity.getId());
    }
    return ticket;
  }

  private String buildJsonArray(List<TopicEntity> ticketDTOS) {
    JSONArray array = new JSONArray();
    for (TopicEntity ticketDTO : ticketDTOS) {
      JSONObject ticket = buildJsonObject(ticketDTO);
      array.put(ticket);
    }
    return array.toString();
  }

}
