package org.exoplatform.addons.mattermost.services;

import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.servlet.http.HttpServletRequest;
import org.apache.commons.lang3.StringUtils;
import org.exoplatform.commons.utils.PropertyManager;
import org.exoplatform.container.PortalContainer;
import org.exoplatform.container.component.RequestLifeCycle;
import org.exoplatform.services.log.ExoLogger;
import org.exoplatform.services.log.Log;
import org.exoplatform.services.rest.resource.ResourceContainer;
import org.exoplatform.social.core.space.model.Space;
import org.exoplatform.social.core.space.spi.SpaceService;
import org.exoplatform.social.metadata.MetadataService;
import org.exoplatform.social.metadata.model.MetadataItem;
import org.exoplatform.social.metadata.model.MetadataKey;
import org.exoplatform.social.metadata.model.MetadataObject;

import javax.annotation.security.RolesAllowed;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.Response;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static org.exoplatform.addons.mattermost.services.MattermostSpaceListener.*;
import static org.exoplatform.addons.mattermost.services.MattermostUtils.EXO_MATTERMOST_SERVER_URL;

@Path("/mattermost")
@Tag(name = "/mattermost", description = "Managing connection between eXo and Mattermost")
public class SpaceTeamRest implements ResourceContainer {
  private static final Log LOG               = ExoLogger.getLogger(SpaceTeamRest.class);

  private SpaceService spaceService;
  private MetadataService metadataService;

  public SpaceTeamRest(SpaceService spaceService, MetadataService metadataService) {
    this.spaceService = spaceService;
    this.metadataService = metadataService;
  }

  @GET
  @Path("/teamUrl")
  @RolesAllowed("users")
  public Response getMattermostTeamUrl(@Context HttpServletRequest request, @QueryParam("spaceId") String spaceId) {
    if(StringUtils.isBlank(spaceId)) {
      LOG.error("Could not get the URL for the space, missing space ID");
      return Response.status(Response.Status.BAD_REQUEST).build();
    }
    Space space = this.spaceService.getSpaceById(spaceId);
    if(space == null) {
      LOG.error("Could not find a space with id {}", spaceId);
      return Response.status(Response.Status.NOT_FOUND).build();
    }
    String userName = request.getRemoteUser();
    if(!spaceService.isMember(space, userName) && !spaceService.isManager(space, userName) && !spaceService.isSuperManager(userName)) {
      LOG.error("User is not allowed to get the team associated with the space {}", space.getDisplayName());
      return Response.status(Response.Status.FORBIDDEN).build();
    }

    String matterMostUrl = PropertyManager.getProperty(EXO_MATTERMOST_SERVER_URL);

    MetadataKey metadataKey = new MetadataKey(MATTERMOST_METADATA_TYPE, MATTERMOST_METADATA_NAME, 0);
    MetadataObject metadataObject = new MetadataObject("Space", space.getId(), null, Long.parseLong(space.getId()));
    List<MetadataItem> metadataItems = metadataService.getMetadataItemsByMetadataTypeAndObject(metadataKey.getType(), metadataObject);
    if(metadataItems.size() > 1) {
      LOG.error("There are more than one items for type {} and space {}", metadataKey.getType(), spaceId);
      return Response.status(Response.Status.INTERNAL_SERVER_ERROR).build();
    } else if (!metadataItems.isEmpty()) {
      MetadataItem metadataItem = metadataItems.get(0);
      if (metadataItem != null) {
        matterMostUrl = matterMostUrl.lastIndexOf("/") == matterMostUrl.length() - 1 ? matterMostUrl : matterMostUrl + "/";
        matterMostUrl = matterMostUrl + metadataItem.getProperties().get(MATTERMOST_TEAM_NAME);
        return Response.ok(matterMostUrl).build();
      } else {
        return Response.ok().build();
      }
    } else {
      return Response.status(Response.Status.NOT_FOUND).build();
    }
  }

  @GET
  @Path("/linkSpaceToTeam")
  @RolesAllowed("administrators")
  public Response linkSpaceToTeam(@QueryParam("spaceName") String spaceName,
                                  @QueryParam("teamName") String teamName,
                                  @QueryParam("teamId") String teamId) {
    RequestLifeCycle.begin(PortalContainer.getInstance());
    try {
      if(StringUtils.isBlank(spaceName)) {
        LOG.error("Could not connect the space to a team, space name is missing");
        return Response.status(Response.Status.BAD_REQUEST).build();
      }

      Space space = spaceService.getSpaceByGroupId("/spaces/" + spaceName);
      MetadataKey metadataKey = new MetadataKey(MATTERMOST_METADATA_TYPE, MATTERMOST_METADATA_NAME, 0);
      MetadataObject metadataObject = new MetadataObject("Space", space.getId(), null, Long.parseLong(space.getId()));

      // check if there is already a team linked to that space
      List<MetadataItem> items = metadataService.getMetadataItemsByMetadataAndObject(metadataKey, metadataObject);
      if(!items.isEmpty()) {
        metadataService.deleteMetadataItemsByMetadataTypeAndObject(metadataKey.getType(), metadataObject);
      }

      if (!StringUtils.isBlank(teamId) && !StringUtils.isBlank(teamName)) {
        Map<String, String> properties = new HashMap<>();
        properties.put(MATTERMOST_TEAM_ID, teamId);
        properties.put(MATTERMOST_TEAM_NAME, teamName);
        metadataService.createMetadataItem(metadataObject, metadataKey, properties);
      }
      return Response.ok("Space %s was linked to team with ID %s successfully ".formatted(space.getDisplayName(), teamId)).build();
    } catch (Exception e) {
      LOG.error("Could not link space {} to Mattermost team {}", spaceName, teamId);
      return Response.serverError().entity(e.getMessage()).build();
    } finally {
      RequestLifeCycle.end();
    }
  }
}
