/*
 * Copyright (C) 2022 eXo Platform SAS
 *
 *  This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <gnu.org/licenses>.
 */
package org.exoplatform.antimalware.connector;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.jcr.ItemNotFoundException;
import javax.jcr.Node;
import javax.jcr.Property;

import org.apache.commons.lang.StringUtils;
import org.exoplatform.container.xml.InitParams;
import org.exoplatform.services.jcr.RepositoryService;
import org.exoplatform.services.jcr.core.ExtendedSession;
import org.exoplatform.services.jcr.impl.core.NodeImpl;
import org.exoplatform.services.log.ExoLogger;
import org.exoplatform.services.log.Log;
import org.exoplatform.services.wcm.core.NodetypeConstant;
import org.exoplatform.services.wcm.utils.WCMCoreUtils;

public class MalwareDetectionJcrConnector extends MalwareDetectionItemConnector {

  private static final Log LOGGER = ExoLogger.getExoLogger(MalwareDetectionJcrConnector.class);
  
  private static final String PORTAL_CONTAINER = "_portal";

  private RepositoryService repositoryService;
  
  private  static final String JCR = "jcr";
  
  private static final String VALUES = "values";
  
  private static final String MALWARE_INFECTED_JCR_NODE_DETECTION = "malware-infected-jcr-node-detection";
  
  private static final String MALWARE_INFECTED_JCR_NODE_CLEAN = "malware-infected-jcr-node-clean";
  
  private static final String INFECTED_JCR_NODE_WORKSPACE = "infectedJcrNodeWorkspace";

  public MalwareDetectionJcrConnector(InitParams initParams, RepositoryService repositoryService) {
    super(initParams);
    this.repositoryService = repositoryService;
  }

  @Override
  public List<Map<String, String>> getInfectedItems(String infectedJcrNodePath) {
    String infectedJcrNodeIdentifier = getJcrNodeIdentifier(infectedJcrNodePath);
    String jcrNodeWorkspace = getJcrNodeWorkspace(infectedJcrNodePath);
    List<Map<String, String>> infectedNodes = new ArrayList<Map<String, String>>();
    long startTime = System.currentTimeMillis();
    try {
      ExtendedSession session = (ExtendedSession) WCMCoreUtils.getSystemSessionProvider().getSession(jcrNodeWorkspace, repositoryService.getCurrentRepository());
      Property infectedJcrNodeProperty = session.getPropertyByIdentifier(infectedJcrNodeIdentifier);
      if (infectedJcrNodeProperty.getName().equals(NodetypeConstant.JCR_DATA)) {
        Node infectedJcrContentNode = infectedJcrNodeProperty.getParent();
        Node infectedJcrNode = infectedJcrContentNode.getParent();
        Map<String, String> infectedNode = new HashMap<String, String>();
        infectedNode.put(INFECTED_ITEM_ID, ((NodeImpl) infectedJcrNode).getIdentifier());
        infectedNode.put(INFECTED_JCR_NODE_WORKSPACE, jcrNodeWorkspace);
        infectedNode.put(INFECTED_ITEM_NAME, infectedJcrNode.getName());
        infectedNode.put(INFECTED_ITEM_LAST_MODIFIER, infectedJcrNode.getProperty(NodetypeConstant.EXO_LAST_MODIFIER).getString());
        infectedNodes.add(infectedNode);
        long endTime = System.currentTimeMillis();
        LOGGER.info("service={} operation={} parameters=\"jcrNodeIdentifier:{}\" \"jcrNodeName:{}\" \"jcrNodeLastModifier:{}\" status=ok " + "duration_ms={}",
                    MALWARE_DETECTION_FEATURE,
                    MALWARE_INFECTED_JCR_NODE_DETECTION,
                    ((NodeImpl) infectedJcrNode).getIdentifier(),
                    infectedJcrNode.getName(),
                    infectedJcrNode.getProperty(NodetypeConstant.EXO_LAST_MODIFIER).getString(),
                    endTime - startTime);
      }
    } catch (ItemNotFoundException e) {
      long endTime = System.currentTimeMillis();
      LOGGER.warn("service={} operation={} parameters=\"infectedJcrNodePath:{}\" status=ko duration_ms={} error_msg=\"The jcr node may be already cleaned by a previous malware detection operation:{}\"",
                   MALWARE_DETECTION_FEATURE,
                   MALWARE_INFECTED_JCR_NODE_DETECTION,
                   infectedJcrNodePath,
                   endTime - startTime);
    } catch (Exception e) {
      long endTime = System.currentTimeMillis();
      LOGGER.error("service={} operation={} parameters=\"infectedJcrNodePath:{}\" status=ko duration_ms={} error_msg=\"Error when trying to get the infected jcr node information:{}\"",
                   MALWARE_DETECTION_FEATURE,
                   MALWARE_INFECTED_JCR_NODE_DETECTION,
                   infectedJcrNodePath,
                   endTime - startTime,
                   e);
    }
    return infectedNodes;
  }

  @Override
  public boolean canProcessInfectedItem(String infectedJcrNodePath) {
    String infectedJcrNodePathSeparator = getPathSeparator(infectedJcrNodePath);
    return infectedJcrNodePath.contains(infectedJcrNodePathSeparator + JCR + infectedJcrNodePathSeparator + VALUES + infectedJcrNodePathSeparator);
  }
  
  @Override
  public void cleanInfectedItem(Map<String, String> infectedItem) {
    long startTime = System.currentTimeMillis();
    try {
      ExtendedSession session = (ExtendedSession) WCMCoreUtils.getSystemSessionProvider().getSession(infectedItem.get(INFECTED_JCR_NODE_WORKSPACE), repositoryService.getCurrentRepository());
      Node infectedJcrNode = session.getNodeByIdentifier(infectedItem.get(INFECTED_ITEM_ID));
      infectedJcrNode.remove();
      session.save();
      long endTime = System.currentTimeMillis();
      LOGGER.info("service={} operation={} parameters=\"jcrNodeIdentifier:{}\" \"jcrNodeName:{}\" status=ok " + "duration_ms={}",
                  MALWARE_DETECTION_FEATURE,
                  MALWARE_INFECTED_JCR_NODE_CLEAN,
                  infectedItem.get(INFECTED_ITEM_ID),
                  infectedItem.get(INFECTED_ITEM_NAME),
                  endTime - startTime);
    } catch (Exception e) {
      long endTime = System.currentTimeMillis();
      LOGGER.error("service={} operation={} parameters=\"jcrNodeIdentifier:{}\" \"jcrNodeName:{}\" status=ko duration_ms={} error_msg\"Error when trying to clean the infected jcr node:{}\"",
                   MALWARE_DETECTION_FEATURE,
                   MALWARE_INFECTED_JCR_NODE_CLEAN,
                   infectedItem.get(INFECTED_ITEM_ID),
                   infectedItem.get(INFECTED_ITEM_NAME),
                   endTime - startTime,
                   e);
    }
  }

  private String getJcrNodeIdentifier(String infectedJcrNodePath) {
    String infectedJcrNodeIdentifier = StringUtils.substringAfter(infectedJcrNodePath, PORTAL_CONTAINER);
    String infectedJcrNodePathSeparator = getPathSeparator(infectedJcrNodePath);
    infectedJcrNodeIdentifier = StringUtils.substringBeforeLast(infectedJcrNodeIdentifier, infectedJcrNodePathSeparator);
    return StringUtils.remove(infectedJcrNodeIdentifier, infectedJcrNodePathSeparator);

  }

  private String getJcrNodeWorkspace(String infectedJcrNodePath) {
    return StringUtils.substringBetween(infectedJcrNodePath, VALUES + getPathSeparator(infectedJcrNodePath), PORTAL_CONTAINER);
  }
}