/*
 * Copyright (C) 2022 eXo Platform SAS
 *
 *  This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <gnu.org/licenses>.
 */
package org.exoplatform.antimalware.rest;

import java.util.List;

import javax.ws.rs.*;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;

import org.exoplatform.antimalware.service.MalwareDetectionService;
import org.exoplatform.services.rest.resource.ResourceContainer;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import io.swagger.v3.oas.annotations.tags.Tag;

/**
 * @author Ayoub Zayati
 */
@Path("/malware/detection")
@Tag(name = "/malware", description = "Malware detection")
public class MalwareDetectionRestService implements ResourceContainer {

  private static final String     MALWARE_DETECTION_PASSPHRASE_PROPERTY = "exo.malwareDetection.passphrase";

  private MalwareDetectionService malwareDetectionService;

  private String                  malwareDetectionPassphrase;

  public MalwareDetectionRestService(MalwareDetectionService malwareDetectionService) {
    this.malwareDetectionService = malwareDetectionService;
    malwareDetectionPassphrase = System.getProperty(MALWARE_DETECTION_PASSPHRASE_PROPERTY);
  }

  @POST
  @Consumes(MediaType.APPLICATION_JSON)
  @Produces(MediaType.APPLICATION_JSON)
  @Operation(summary = "Process infected items", method = "POST", description = "This processes infected items")
  @ApiResponses(value = { @ApiResponse(responseCode = "200", description = "Request fulfilled"),
      @ApiResponse(responseCode = "403", description = "Unauthorized to process infected items") })
  public Response processInfectedItems(@Parameter(description = "List of infected items paths", required = true) List<String> infectedItemsPaths,
                                       @QueryParam("token") String token) throws Exception {
    if (token == null || malwareDetectionPassphrase == null || !malwareDetectionPassphrase.equals(token)) {
      throw new WebApplicationException(Response.Status.UNAUTHORIZED);
    }
    for (String infectedItemPath : infectedItemsPaths) {
      malwareDetectionService.processInfectedItem(infectedItemPath);
    }
    return Response.ok().entity("{\"result\":\"" + infectedItemsPaths + "\"}").build();
  }
}
