/*
 * Copyright (C) 2010 eXo Platform SAS.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.exoplatform.jcr.benchmark.usecases.portal;

import com.sun.japex.TestCase;

import org.exoplatform.jcr.benchmark.JCRTestBase;
import org.exoplatform.jcr.benchmark.JCRTestContext;
import org.exoplatform.services.jcr.impl.core.RepositoryImpl;
import org.exoplatform.services.jcr.util.IdGenerator;
import org.exoplatform.services.log.ExoLogger;
import org.exoplatform.services.log.Log;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Random;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.jcr.Session;

/**
 * Portal-like JCR usecase test. Originally developed for testing cluster performance and scalability.
 * 
 * @author <a href="mailto:nikolazius@gmail.com">Nikolay Zamosenchuk</a>
 * @version $Id: PageUsecases.java 34360 2009-07-22 23:58:59Z nzamosenchuk $
 *
 */
public class PageUsecasesTest extends JCRTestBase
{
   static final Log log = ExoLogger.getLogger(PageUsecasesTest.class.getName());

   // Option names
   private static final String PARAM_DEPTH = "exo.prepare.depth";

   private static final String PARAM_NODES_PER_LEVEL = "exo.prepare.nodesPerLevel";

   private static final String PARAM_STRING_LENGTH = "exo.data.stringLength";

   private static final String PARAM_MULTI_SIZE = "exo.data.multiValueSize";

   private static final String PARAM_BIN_PATH = "exo.data.binaryPath";

   private static final String PARAM_BIN_SIZE = "exo.data.binarySize";

   private static final String PARAM_SCENARIO = "exo.scenario.string";

   private static final String PARAM_CELANUP = "exo.finish.cleanup";

   private static final String PARAM_PREPARE_SYNCHRONISATION = "exo.prepare.synchronisation";

   private static final String PARAM_FINISH_SYNCHRONISATION = "exo.finish.synchronisation";

   private static final String PARAM_FINISH_SYNCHRONISATION_SLEEPTIME = "exo.finish.synchronisation.sleepTime";

   private static final String PARAM_WAIT_THREADS = "exo.synchronisation.waitThread";

   private static final String PARAM_SYNCHRONISATION_NONE = "none";

   private static final String PARAM_SYNCHRONISATION_KEYPRESS = "keypress";

   private static final String PARAM_SYNCHRONISATION_SLEEP = "sleep";

   // UseCases names
   private static final String CASE_READ_ANON = "ReadAnon";

   private static final String CASE_READ_CONN = "Read";

   private static final String CASE_WRITE_CONN = "Write";

   // Test options
   private int depth = 2;

   private int nodesPerLevel = 10;

   private int stringLength = 64;

   private int multiValueSize = 2;

   private String binaryPath = null;

   private int binarySize = 1024;

   // Fields
   private String rootNodeName;

   private Random random = new Random();

   // Default values
   private static String stringValue = null;

   private static byte[] binaryValue = null;

   // Usecases
   private List<AbstractAction> scenario = null;

   private final AtomicInteger index = new AtomicInteger();

   private static int keyPressThreadCounter = 0;

   private static int timeoutThreadCounter = 0;

   /**
    * Used to synchronize over cluster. Requires Enter pressing.
    * 
    * @param timesToWait
    * @throws IOException
    */
   public static synchronized void waitUntilKeyPressed(int timesToWait) throws IOException
   {
      keyPressThreadCounter++;
      if (keyPressThreadCounter >= timesToWait)
      {
         System.out.print("Press Enter to continue ...");
         BufferedReader in = new BufferedReader(new InputStreamReader(System.in));
         in.readLine();
         System.out.println("Continuing ...");
         keyPressThreadCounter = 0;
      }
   }

   /**
    * Used to synchronize over cluster. Sleeps for sleepTime seconds
    * 
    * @param timesToWait
    * @param sleepTime
    */
   public static synchronized void waitUntilTimeout(int timesToWait, int sleepTime)
   {
      timeoutThreadCounter++;
      if (timeoutThreadCounter >= timesToWait)
      {
         System.out.println("Waiting " + sleepTime / 1000 + "s before continue ...");
         timeoutThreadCounter = 0;
         try
         {
            Thread.sleep(sleepTime);
         }
         catch (InterruptedException e)
         {
            // skip
         }
      }
   }

   @Override
   public void doPrepare(TestCase tc, JCRTestContext context) throws Exception
   {
      super.doPrepare(tc, context);
      // get parameters from context
      if (tc.hasParam(PARAM_DEPTH))
      {
         depth = tc.getIntParam(PARAM_DEPTH);
      }
      if (tc.hasParam(PARAM_NODES_PER_LEVEL))
      {
         nodesPerLevel = tc.getIntParam(PARAM_NODES_PER_LEVEL);
      }
      if (tc.hasParam(PARAM_STRING_LENGTH))
      {
         stringLength = tc.getIntParam(PARAM_STRING_LENGTH);
      }
      if (tc.hasParam(PARAM_MULTI_SIZE))
      {
         multiValueSize = tc.getIntParam(PARAM_MULTI_SIZE);
      }
      if (tc.hasParam(PARAM_BIN_PATH))
      {
         binaryPath = tc.getParam(PARAM_BIN_PATH);
      }
      if (tc.hasParam(PARAM_BIN_SIZE))
      {
         binarySize = tc.getIntParam(PARAM_BIN_SIZE);
      }
      String scenarioString;
      if (tc.hasParam(PARAM_SCENARIO))
      {
         scenarioString = tc.getParam(PARAM_SCENARIO);
      }
      else
      {
         throw new Exception(
            "Scenario not found in configuration, but it is mandatory. Please define scenario as testCase parameter '"
            + PARAM_SCENARIO + "'.");
      }

      // initialize STATIC VALUES, synchronize all running threads
      synchronized (this)
      {
         // Store value in memory, to avoid unnecessary FS IO operations.
         // Once generate string content
         if (stringValue == null)
         {
            // not yet initialized by another thread
            byte[] bytes = new byte[stringLength];
            random.nextBytes(bytes);
            stringValue = new String(bytes);
         }

         if (binaryValue == null)
         {
            // not yet initialized by another thread
            // once read in memory FS content
            if (binaryPath == null)
            {
               binaryValue = new byte[binarySize];
               random.nextBytes(binaryValue);
            }
            else
            {
               File file = new File(binaryPath);
               FileInputStream fin = new FileInputStream(file);
               binaryValue = new byte[(int)file.length()];
               fin.read(binaryValue);
               fin.close();
            }
         }
      }

      // initialization
      Session session = context.getSession();
      RepositoryImpl repository = (RepositoryImpl)context.getSession().getRepository();

      // root node for the test
      rootNodeName = IdGenerator.generate();
      session.getRootNode().addNode(rootNodeName, "exo:genericNode");
      session.save();

      scenario = parse(tc, scenarioString, repository, session.getWorkspace().getName(), rootNodeName);
      // scenario should not be empty
      if (scenario.size() < 1)
      {
         throw new Exception("Scenario is empty. It must contain at least 1 usecase.");
      }

      // fill the repository
      InitRepositoryAction initAction =
         new InitRepositoryAction(session, rootNodeName, stringValue, binaryValue, multiValueSize, depth, nodesPerLevel);

      // perform initialize action
      initAction.perform();
      session.save();

      // synchronization
      if (tc.hasParam(PARAM_PREPARE_SYNCHRONISATION))
      {
         String synch = tc.getParam(PARAM_PREPARE_SYNCHRONISATION);

         if (synch.equalsIgnoreCase(PARAM_SYNCHRONISATION_KEYPRESS))
         {
            int thrWait = 1;
            if (tc.hasParam(PARAM_WAIT_THREADS))
            {
               thrWait = tc.getIntParam(PARAM_WAIT_THREADS);
            }
            waitUntilKeyPressed(thrWait);
         }
         else if (synch.equalsIgnoreCase(PARAM_SYNCHRONISATION_NONE))
         {
            // none used
         }
         else
         {
            log.warn("Unknown prepare synchronization method: " + synch);
         }
      }

   }

   @Override
   public void doRun(TestCase tc, JCRTestContext context) throws Exception
   {
      // get next use-case (Page) from scenario
      scenario.get(index.getAndIncrement() % scenario.size()).perform();
   }

   /**
    * Parses incoming scenario string in notation:
    * 
    * scenario ::= action {';' action}
    * action ::= fullNotation | singleNotation
    * fullNotation ::= Integer '*' singleNotation
    * singleNotation ::= Name '(' paramList ')'
    * paramList ::= Integer {',' Integer}
    * 
    * @param line
    *        Line containing the scenario
    * @param repository
    *        Repository instance
    * @param workspace
    *        Workspace name
    * @param rootNodeName
    *        Name of the root node
    * @return
    *        List of actions, defined in scenario
    * @throws Exception
    */
   private List<AbstractAction> parse(TestCase tc, String line, RepositoryImpl repository, String workspace,
      String rootNodeName) throws Exception
      {
      List<AbstractAction> actions = new ArrayList<AbstractAction>();
      // matching "22*read(2,5,4,6)"
      Pattern fullNotation = Pattern.compile("(\\d++)\\s*[*]\\s*(\\w++)\\s*[(]([,\\w.]*+)[)]");
      // matching "read(2,5,4,6)"
      Pattern singleNotation = Pattern.compile("(\\w++)\\s*[(]([,\\w.]*+)[)]");

      // split scenario into usecases, skipping whitespaces
      String[] actionNames = line.split("\\s*+;\\s*+");
      // parse each action string
      for (String actionLine : actionNames)
      {
         int times;
         String actionName;
         String[] params;

         // try parse as full notation
         Matcher fullMatcher = fullNotation.matcher(actionLine);
         if (fullMatcher.matches())
         {
            times = Integer.parseInt(fullMatcher.group(1));
            actionName = fullMatcher.group(2);
            params = fullMatcher.group(3).split("\\s*+,\\s*+");
         }
         else
         {
            // try parse as notation without multiplier
            Matcher singleMatcher = singleNotation.matcher(actionLine);
            if (singleMatcher.matches())
            {
               times = 1;
               actionName = singleMatcher.group(1);
               params = singleMatcher.group(2).split("\\s*+,\\s*+");
            }
            else
            {
               throw new Exception("Illegal scenario element:" + actionLine);
            }
         }
         // Create usecase action object
         for (int i = 0; i < times; i++)
         {
            if (CASE_READ_ANON.equalsIgnoreCase(actionName))
            {
               if (params.length == 4)
               {
                  // anonymous session
                  actions.add(new ReadPageAction(repository, workspace, rootNodeName, depth, Integer
                     .parseInt(params[0]), Integer.parseInt(params[1]), Integer.parseInt(params[2]), Integer
                     .parseInt(params[3]), true));
               }
               else if (params.length == 5)
               {
                  // last 5th param is optional. If it is present, then it must point to param in test case describing the
                  // list of desired queries (SQL ONLY)
                  // anonymous session
                  actions.add(new ReadPageAction(repository, workspace, rootNodeName, depth, Integer
                     .parseInt(params[0]), Integer.parseInt(params[1]), Integer.parseInt(params[2]), Integer
                     .parseInt(params[3]), true, Arrays.asList(tc.getParam(params[4].trim()).split(";"))));
               }
               else
               {
                  throw new Exception(
                     "Missing arguments for '"
                     + actionName
                     + "' action. Expected 4 or 5 arguments: number of JCR nodes and properties to read. Should be defined as '"
                     + actionName + "(2,5,1,1)'" + " or '" + actionName + "(2,5,1,1,exo.query.list_1)'");
               }
            }
            else if (CASE_READ_CONN.equalsIgnoreCase(actionName))
            {
               if (params.length == 4)
               {
                  // system session
                  actions.add(new ReadPageAction(repository, workspace, rootNodeName, depth, Integer
                     .parseInt(params[0]), Integer.parseInt(params[1]), Integer.parseInt(params[2]), Integer
                     .parseInt(params[3]), false));
               }
               else if (params.length == 5)
               {
                  // last 5th param is optional. If it is present, then it must point to param in test case describing the
                  // list of desired queries (SQL ONLY)
                  // anonymous session
                  actions.add(new ReadPageAction(repository, workspace, rootNodeName, depth, Integer
                     .parseInt(params[0]), Integer.parseInt(params[1]), Integer.parseInt(params[2]), Integer
                     .parseInt(params[3]), false, Arrays.asList(tc.getParam(params[4].trim()).split(";"))));
               }
               else
               {
                  throw new Exception(
                     "Missing arguments for '"
                     + actionName
                     + "' action. Expected 4 or 5 arguments: number of JCR nodes and properties to read. Should be defined as '"
                     + actionName + "(2,5,1,1)'" + " or '" + actionName + "(2,5,1,1,exo.query.list_1)'");
               }
            }
            else if (CASE_WRITE_CONN.equalsIgnoreCase(actionName))
            {
               if (params.length == 4)
               {
                  // system session

                  if (Integer.parseInt(params[0]) > Integer.parseInt(params[1]))
                  {
                     // count of removed properties must be less or equal to count of added properties
                     throw new Exception(
                        "Wrong arguments for '"
                        + actionName
                        + "' action. Count of removed properties must be less or equal to count of setted properties: '"
                        + actionName + "(" + params[0] + "," + params[1] + ",_,_)'");

                  }

                  if (Integer.parseInt(params[2]) > Integer.parseInt(params[3]))
                  {
                     // count of removed nodes must be less or equal to count of added nodes
                     throw new Exception("Wrong arguments for '" + actionName
                        + "' action. Count of removed nodes must be less or equal to count of added nodes: '"
                        + actionName + "(_,_," + params[3] + "," + params[4] + ")'");
                  }

                  actions.add(new WritePageAction(repository, workspace, rootNodeName, depth, stringValue, binaryValue,
                     multiValueSize, Integer.parseInt(params[0]), Integer.parseInt(params[1]), Integer
                     .parseInt(params[2]), Integer.parseInt(params[3])));

               }
               else
               {
                  throw new Exception(
                     "Missing arguments for '"
                     + actionName
                     + "' action. Expected 4 arguments: number of JCR nodes and properties to read. Should be defined as '"
                     + actionName + "(2,5,1,3)'");
               }
            }
            else
            {
               throw new Exception("Invalid usecase name: " + actionName);
            }
         }
      }
      return Collections.unmodifiableList(actions);
      }

   @Override
   public void doFinish(TestCase tc, JCRTestContext context) throws Exception
   {
      super.doFinish(tc, context);
      // if parameter is absent or true, then remove node
      if (!tc.hasParam(PARAM_CELANUP) || tc.getBooleanParam(PARAM_CELANUP))
      {
         Session session = context.getSession();
         session.refresh(false);
         session.getRootNode().getNode(rootNodeName).remove();
         session.save();
      }

      // Synchronization on finish to avoid any cluster-view changes during test run.
      if (tc.hasParam(PARAM_FINISH_SYNCHRONISATION))
      {
         String synch = tc.getParam(PARAM_FINISH_SYNCHRONISATION);

         if (synch.equalsIgnoreCase(PARAM_SYNCHRONISATION_KEYPRESS))
         {
            int thrWait = 1;
            if (tc.hasParam(PARAM_WAIT_THREADS))
            {
               thrWait = tc.getIntParam(PARAM_WAIT_THREADS);
            }
            waitUntilKeyPressed(thrWait);
         }
         else if (synch.equalsIgnoreCase(PARAM_SYNCHRONISATION_SLEEP))
         {
            // default sleep is 20 seconds.
            int sleepTime = 20 * 1000;
            if (tc.hasParam(PARAM_FINISH_SYNCHRONISATION_SLEEPTIME))
            {
               sleepTime = 1000 * tc.getIntParam(PARAM_FINISH_SYNCHRONISATION_SLEEPTIME);
            }
            int thrWait = 1;
            if (tc.hasParam(PARAM_WAIT_THREADS))
            {
               thrWait = tc.getIntParam(PARAM_WAIT_THREADS);
            }
            waitUntilTimeout(thrWait, sleepTime);
         }
         else if (synch.equalsIgnoreCase(PARAM_SYNCHRONISATION_NONE))
         {
            // none used
         }
         else
         {
            log.warn("Unknown finish synchronization method: " + synch);
         }
      }

   }
}
