package com.naftogaz.profile;

import org.apache.commons.lang3.StringUtils;
import org.exoplatform.container.PortalContainer;
import org.exoplatform.container.component.RequestLifeCycle;
import org.exoplatform.services.log.ExoLogger;
import org.exoplatform.services.log.Log;
import org.exoplatform.services.organization.OrganizationService;
import org.exoplatform.services.organization.UserProfile;
import org.exoplatform.services.security.ConversationState;
import org.exoplatform.social.core.identity.model.Identity;
import org.exoplatform.social.core.identity.model.Profile;
import org.exoplatform.social.core.identity.provider.OrganizationIdentityProvider;
import org.exoplatform.social.core.jpa.storage.EntityConverterUtils;
import org.exoplatform.social.core.jpa.storage.entity.IdentityEntity;
import org.exoplatform.social.core.manager.IdentityManager;

import javax.ws.rs.core.Response;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

public class BirthdayService {

    private Log log = ExoLogger.getLogger(BirthdayService.class);
    private ProfileExtraDAO profileExtraDAO;
    private IdentityManager identityManager;
    private OrganizationService organizationService;

    public static final String BIRTHDAY       = "user.bdate";
    public static final String POSITION       = "user.jobtitle";
    public static final String MOBILE       = "user.business-info.telecom.mobile.number";
    public static final String HOME_PHONE       = "user.business-info.telecom.telephone.number";

    private static final List<String> AVAILABLE_USER_ATTRIBUTES_TO_SAVE = Arrays.asList(BIRTHDAY);

    public BirthdayService(ProfileExtraDAO profileExtraDAO, IdentityManager identityManager, OrganizationService organizationService) {
        this.profileExtraDAO = profileExtraDAO;
        this.identityManager = identityManager;
        this.organizationService = organizationService;
    }

    /**
     * Returns the list of user identities having the defined birthday date
     * @param birthDate using format dd-mm-yyyy
     * @return list of user identities
     */
    public List<Identity> getIdentitiesByBirthDate(String birthDate) {
        if(StringUtils.isBlank(birthDate)) {
            throw new IllegalArgumentException("Birthdate should not be empty");
        }
        if(!birthDate.matches("\\d\\d?\\W\\d\\d\\W\\d\\d\\d\\d$")) {
            throw new IllegalArgumentException("Birthdate should be with format dd-MM-yyyy");
        }
        //remove the year from the date
        birthDate = birthDate.substring(0, birthDate.length() - 5);
        List<IdentityEntity> identityEntities = profileExtraDAO.getIdentitiesByPropertyAndValueLike(BIRTHDAY, birthDate);
        return EntityConverterUtils.convertToIdentities(identityEntities.toArray(new IdentityEntity[0]));
    }

    public boolean updateBirthday (String userName, Map<String, String> profileAttributes) {
        RequestLifeCycle.begin(PortalContainer.getInstance());
        Identity identity = identityManager.getOrCreateIdentity(OrganizationIdentityProvider.NAME, userName);
        Profile profile = identityManager.getProfile(identity);
        try {
            UserProfile userProfile = organizationService.getUserProfileHandler().findUserProfileByName(userName);
            if(userProfile == null) {
                userProfile = organizationService.getUserProfileHandler().createUserProfileInstance(userName);
            }
            Thread.sleep(500);
            for (Map.Entry<String, String> entry : profileAttributes.entrySet()) {
                String key = entry.getKey();
                String value = entry.getValue();
                if (AVAILABLE_USER_ATTRIBUTES_TO_SAVE.contains(key)) {
                    profile.setProperty(key, value);
                    userProfile.setAttribute(key, value);
                } else {
                    log.warn("Property unavailable to save: " + key);
                }
            }
            identityManager.updateProfile(profile);
            if(userProfile != null) {
                organizationService.getUserProfileHandler().saveUserProfile(userProfile, false);
            }
            return true;
        } catch (Exception e) {
            log.error("Error updating profile of {}", userName, e);
            return false;
        }
        finally {
            RequestLifeCycle.end();
        }
    }
}
