/******************************************************************************
 * JBoss, a division of Red Hat                                               *
 * Copyright 2006, Red Hat Middleware, LLC, and individual                    *
 * contributors as indicated by the @authors tag. See the                     *
 * copyright.txt in the distribution for a full listing of                    *
 * individual contributors.                                                   *
 *                                                                            *
 * This is free software; you can redistribute it and/or modify it            *
 * under the terms of the GNU Lesser General Public License as                *
 * published by the Free Software Foundation; either version 2.1 of           *
 * the License, or (at your option) any later version.                        *
 *                                                                            *
 * This software is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU           *
 * Lesser General Public License for more details.                            *
 *                                                                            *
 * You should have received a copy of the GNU Lesser General Public           *
 * License along with this software; if not, write to the Free                *
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA         *
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.                   *
 ******************************************************************************/
package javax.portlet;

import java.io.IOException;

/**
 * The <CODE>Portlet</CODE> interface is used by the portlet container to invoke the portlets. Every portlet has to
 * implement this interface, either by directly implementing it, or by using an existing class implementing the Portlet
 * interface.
 * A portlet is a Java technology-based web component. It is managed by the portlet container and processes requests and
 * generates dynamic content as response. Portlets are used by portals as pluggable user interface components.
 * The content generated by a portlet is called a fragment. A fragment is a piece of markup (e.g. HTML, XHTML, WML)
 * adhering to certain rules and can be aggregated with other fragments into a complete document. The content of a
 * portlet is normally aggregated with the content of other portlets into the portal page.
 * The portlet container instanciates portlets, manages their lifecycle and invoking them to process requests. The
 * lifecycle consists of: <ul> <li>initializing the portlet using using the <code>init</code> method <li>request
 * processsing <li>taking the portlet out of service using the <code>destroy</code> method </ul>
 * Request processing is divided into two types: <ul> <li>action requests handled through the <code>processAction</code>
 * method, to perform actions targeted to the portlet <li>render requests handled through the <code>render</code>
 * method, to perform the render operation </ul>
 *
 * @author <a href="mailto:julien@jboss.org">Julien Viet</a>
 * @version $Revision: 5441 $
 */
public interface Portlet
{
   /**
    * Called by the portlet container to indicate to a portlet that the portlet is being placed into service.
    * <p>The portlet container calls the <code>init</code> method exactly once after instantiating the portlet. The
    * <code>init</code> method must complete successfully before the portlet can receive any requests.
    * <p>The portlet container cannot place the portlet into service if the <code>init</code> method <ol> <li>Throws a
    * <code>PortletException</code> <li>Does not return within a time period defined by the portlet container. </ol>
    *
    * @param config a <code>PortletConfig</code> object containing the portlet's configuration and initialization
    *               parameters
    * @throws PortletException     if an exception has occurred that interferes with the portlet's normal operation.
    * @throws UnavailableException if the portlet cannot perform the initialization at this time.
    */
   void init(PortletConfig config) throws PortletException;

   /**
    * Called by the portlet container to allow the portlet to process an action request. This method is called if the
    * client request was originated by a URL created (by the portlet) with the <code>RenderResponse.createActionURL()</code>
    * method.
    * Typically, in response to an action request, a portlet updates state based on the information sent in the action
    * request parameters. In an action the portlet may: <ul> <li>issue a redirect <li>change its window state <li>change
    * its portlet mode <li>modify its persistent state <li>set render parameters </ul>
    * A client request triggered by an action URL translates into one action request and many render requests, one per
    * portlet in the portal page. The action processing must be finished before the render requests can be issued.
    *
    * @param request  the action request
    * @param response the action response
    * @throws PortletException         if the portlet has problems fulfilling the request
    * @throws UnavailableException     if the portlet is unavailable to process the action at this time
    * @throws PortletSecurityException if the portlet cannot fullfill this request because of security reasons
    * @throws IOException              if the streaming causes an I/O problem
    */
   void processAction(ActionRequest request, ActionResponse response) throws PortletException, PortletSecurityException, IOException;

   /**
    * Called by the portlet container to allow the portlet to generate the content of the response based on its current
    * state.
    *
    * @param request  the render request
    * @param response the render response
    * @throws PortletException         if the portlet has problems fulfilling the rendering request
    * @throws UnavailableException     if the portlet is unavailable to perform render at this time
    * @throws PortletSecurityException if the portlet cannot fullfill this request because of security reasons
    * @throws java.io.IOException      if the streaming causes an I/O problem
    */
   void render(RenderRequest request, RenderResponse response) throws PortletException, PortletSecurityException, IOException;

   /**
    * Called by the portlet container to indicate to a portlet that the portlet is being taken out of service.
    * Before the portlet container calls the destroy method, it should allow any threads that are currently processing
    * requests within the portlet object to complete execution. To avoid waiting forever, the portlet container can
    * optionally wait for a predefined time before destroying the portlet object.
    * <p>This method enables the portlet to do the following: <ul> <li>clean up any resources that it holds (for
    * example, memory, file handles, threads) <li>make sure that any persistent state is synchronized with the portlet
    * current state in memory. </ul>
    */
   void destroy();
}
