/*
 * Copyright (C) 2003-2008 eXo Platform SAS.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see<http://www.gnu.org/licenses/>.
 */
package org.exoplatform.services.jcr.ext.replication.async.storage;

import java.io.EOFException;
import java.io.IOException;
import java.util.Iterator;
import java.util.NoSuchElementException;

import org.apache.commons.logging.Log;
import org.exoplatform.services.jcr.dataflow.ItemState;
import org.exoplatform.services.jcr.dataflow.serialization.ObjectReader;
import org.exoplatform.services.jcr.impl.dataflow.serialization.ItemStateReader;
import org.exoplatform.services.jcr.impl.dataflow.serialization.ObjectReaderImpl;
import org.exoplatform.services.jcr.impl.dataflow.serialization.ReaderSpoolFileHolder;
import org.exoplatform.services.jcr.impl.util.io.FileCleaner;
import org.exoplatform.services.log.ExoLogger;

/**
 * Created by The eXo Platform SAS. <br/>Date: 10.12.2008
 * 
 * @author <a href="mailto:peter.nedonosko@exoplatform.com.ua">Peter Nedonosko</a>
 * @version $Id: ItemStatesStorage.java 30302 2009-03-30 11:15:10Z serg $
 */
public class ItemStatesStorage<T extends ItemState> extends AbstractChangesStorage<T> implements
    MemberChangesStorage<T> {

  protected static final Log LOG = ExoLogger.getLogger("ext.ItemStatesStorage");

  private final ChangesFile  storage;

  protected final Member     member;
  
  private final FileCleaner fileCleaner;
  private final int maxBufferSize;
  private final ReaderSpoolFileHolder holder;

  class ItemStateIterator<S extends ItemState> implements Iterator<S> {

    private ObjectReader in;

    private S            nextItem;

    private final FileCleaner fileCleaner;
    private final int maxBufferSize;
    
    public ItemStateIterator(FileCleaner fileCleaner, int maxBufferSize) throws IOException, ClassCastException, ClassNotFoundException {

      if (storage == null) {
        throw new NullPointerException("ChangesFile not exists.");
      }

      this.in = new ObjectReaderImpl(storage.getInputStream());
      this.nextItem = readNext();
      this.fileCleaner = fileCleaner;
      this.maxBufferSize = maxBufferSize;
    }

    /**
     * {@inheritDoc}
     */
    public boolean hasNext() {
      return nextItem != null;
    }

    /**
     * {@inheritDoc}
     */
    public S next() throws NoSuchElementException {
      if (nextItem == null)
        throw new NoSuchElementException();

      S retVal = nextItem;
      try {
        nextItem = readNext();
      } catch (IOException e) {
        throw new StorageRuntimeException(e.getMessage() + " file: " + storage, e);
      } catch (ClassNotFoundException e) {
        throw new StorageRuntimeException(e.getMessage() + " file: " + storage, e);
      } catch (ClassCastException e) {
        throw new StorageRuntimeException(e.getMessage() + " file: " + storage, e);
      }
      return retVal;
    }

    /**
     * {@inheritDoc}
     */
    public void remove() {
      throw new RuntimeException("Remove not allowed!");
    }

    @SuppressWarnings("unchecked")
    protected S readNext() throws IOException, ClassNotFoundException, ClassCastException {
      if (in != null) {
        try {

          ItemStateReader rdr = new ItemStateReader(fileCleaner, maxBufferSize, holder);
          return (S) rdr.read(in);
        } catch (EOFException e) {
          // End of list
          in.close();
          in = null;
          return null;
        } catch (Throwable e) {
          in.close();
          in = null;
          throw new StorageIOException(e.getMessage(), e);
        }
      } else
        return null;
    }
  }

  /**
   * ItemStatesStorage constructor for export.
   * 
   * @param changes ChagesFiles
   * @param member owner
   * @param holder TODO
   */
  public ItemStatesStorage(ChangesFile changes, Member member, FileCleaner fileCleaner, int maxBufferSize, ReaderSpoolFileHolder holder) {
    this.storage = changes;
    this.member = member;
    this.fileCleaner = fileCleaner;
    this.maxBufferSize = maxBufferSize;
    this.holder = holder;
  }

  /**
   * {@inheritDoc}
   */
  public int size() throws IOException, ClassNotFoundException, ClassCastException {
    Iterator<T> it = getChanges();
    int i = 0;
    while (it.hasNext()) {
      i++;
      it.next();
    }
    return i;
  }

  /**
   * {@inheritDoc}
   */
  public Member getMember() {
    return member;
  }

  /**
   * {@inheritDoc}
   */
  public void delete() throws IOException {
    storage.delete();
  }

  /**
   * {@inheritDoc}
   */
  public ChangesFile[] getChangesFile() {
    return new ChangesFile[] { storage };
  }

  /**
   * {@inheritDoc}
   */
  public Iterator<T> getChanges() throws IOException, ClassCastException, ClassNotFoundException {
    return new ItemStateIterator<T>(fileCleaner, maxBufferSize);
  }

}
