/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.exoplatform.services.jcr.rmi.api.xml;

import java.io.ByteArrayInputStream;
import java.io.IOException;

import javax.jcr.AccessDeniedException;
import javax.jcr.InvalidSerializedDataException;
import javax.jcr.ItemExistsException;
import javax.jcr.PathNotFoundException;
import javax.jcr.RepositoryException;
import javax.jcr.Workspace;
import javax.jcr.lock.LockException;
import javax.jcr.nodetype.ConstraintViolationException;
import javax.jcr.version.VersionException;

/**
 * SAX content handler for importing XML data to a JCR {@link Workspace Workspace}. This utility
 * class can be used to implement the {@link Workspace#getImportContentHandler(String, int)
 * Workspace.getImportContentHandler(String, int)} method in terms of the
 * {@link Workspace#importXML(String, java.io.InputStream, int) Workspace.importXML(String,
 * InputStream, int)} method.
 */
public class WorkspaceImportContentHandler extends ImportContentHandler {

  /** The repository workspace. */
  private Workspace workspace;

  /** The import content path. */
  private String    path;

  /** The UUID behaviour. */
  private int       uuidBehaviour;

  /**
   * Creates a SAX content handler for importing XML data to the given workspace and path using the
   * given UUID behaviour.
   * 
   * @param workspace
   *          repository workspace
   * @param path
   *          import content path
   * @param uuidBehaviour
   *          UUID behaviour
   * @throws RepositoryException
   * @throws LockException
   * @throws ConstraintViolationException
   * @throws VersionException
   */
  public WorkspaceImportContentHandler(Workspace workspace, String path, int uuidBehaviour) throws VersionException,
      ConstraintViolationException,
      LockException,
      RepositoryException {
    super(workspace.getSession(), path);
    this.workspace = workspace;
    this.path = path;
    this.uuidBehaviour = uuidBehaviour;
  }

  /**
   * Imports the serialized XML stream using the standard
   * {@link Workspace#importXML(String, java.io.InputStream, int) Workspace.importXML(String,
   * InputStream, int)} method. {@inheritDoc}
   * 
   * @throws RepositoryException
   * @throws IOException
   * @throws AccessDeniedException
   * @throws LockException
   * @throws InvalidSerializedDataException
   * @throws ConstraintViolationException
   * @throws ItemExistsException
   * @throws PathNotFoundException
   */
  @Override
  protected void importXML(byte[] xml) throws IOException,
                                      PathNotFoundException,
                                      ItemExistsException,
                                      ConstraintViolationException,
                                      InvalidSerializedDataException,
                                      RepositoryException {
    workspace.importXML(path, new ByteArrayInputStream(xml), uuidBehaviour);
  }

}
