// Copyright (C) 2011 Google Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

/**
 * @fileoverview
 * This file is concatenated into the top of the iframe scripts that caja.js
 * loads. It supplies the current build version of Caja. This is interpolated
 * into this file via build.xml rules.
 *
 * @provides cajaBuildVersion
 * @overrides window
 */

var cajaBuildVersion = '5054';

// Exports for closure compiler.
if (typeof window !== 'undefined') {
  window['cajaBuildVersion'] = cajaBuildVersion;
}
;
// Copyright (C) 2008 Google Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

/**
 * @overrides Date, String, Number, Boolean, JSON
 * @provides json_sans_eval
 */

/**
 * This file combines the JSON.parse method defined by the original
 * json_sans_eval.js with the stringify method from the original
 * json2.js. Like json2.js, it defines a JSON object if one does not
 * already exist, and it initializes its parse and stringify methods
 * only if JSON does not currently have such methods (functions at
 * those property names). Additionally, if there is no
 * <tt>Date.prototype.toJSON</tt>, this file defines an ES5 compliant
 * one as well as the <tt>toJSON</tt> methods for <tt>String</tt>,
 * <tt>Number</tt>, and <tt>Boolean</tt>. The latter three are no
 * longer part of ES5, but are expected by the parts of this file
 * derived from json2.js.
 *
 * Of course, the reason this is included in the Caja distribution is
 * so that Caja can expose an ES5 compliant but Caja-safe JSON object
 * to cajoled code. Caja's wrapping of the provided JSON therefore
 * must work with either a real ES5 JSON or the one defined in this
 * file. To do so, Caja uses the replacer and reviver
 * hooks. Fortunately, ES5 and json2.js both specify that only own
 * properties of an object are stringified, and the the replacer is
 * called on the result of a <tt>toJSON</tt> call, making it possible
 * for the replacer to do its job.
 *
 * Comment from original json2.js:
 *
    http://www.JSON.org/json2.js
    2009-08-17

    Public Domain.

    NO WARRANTY EXPRESSED OR IMPLIED. USE AT YOUR OWN RISK.

    Based on json2.js from http://www.JSON.org/js.html
    but with the parse method to be provided by json_sans_eval.js

    This file creates a global JSON object containing two methods: stringify
    and parse.

        JSON.stringify(value, replacer, space)
            value       any JavaScript value, usually an object or array.

            replacer    an optional parameter that determines how object
                        values are stringified for objects. It can be a
                        function or an array of strings.

            space       an optional parameter that specifies the indentation
                        of nested structures. If it is omitted, the text will
                        be packed without extra whitespace. If it is a number,
                        it will specify the number of spaces to indent at each
                        level. If it is a string (such as '\t' or '&nbsp;'),
                        it contains the characters used to indent at each level.

            This method produces a JSON text from a JavaScript value.

            When an object value is found, if the object contains a toJSON
            method, its toJSON method will be called and the result will be
            stringified. A toJSON method does not serialize: it returns the
            value represented by the name/value pair that should be serialized,
            or undefined if nothing should be serialized. The toJSON method
            will be passed the key associated with the value, and this will be
            bound to the value

            For example, this would serialize Dates as ISO strings.

                Date.prototype.toJSON = function (key) {
                    function f(n) {
                        // Format integers to have at least two digits.
                        return n < 10 ? '0' + n : n;
                    }

                    return this.getUTCFullYear()   + '-' +
                         f(this.getUTCMonth() + 1) + '-' +
                         f(this.getUTCDate())      + 'T' +
                         f(this.getUTCHours())     + ':' +
                         f(this.getUTCMinutes())   + ':' +
                         f(this.getUTCSeconds())   + 'Z';
                };

            You can provide an optional replacer method. It will be passed the
            key and value of each member, with this bound to the containing
            object. The value that is returned from your method will be
            serialized. If your method returns undefined, then the member will
            be excluded from the serialization.

            If the replacer parameter is an array of strings, then it will be
            used to select the members to be serialized. It filters the results
            such that only members with keys listed in the replacer array are
            stringified.

            Values that do not have JSON representations, such as undefined or
            functions, will not be serialized. Such values in objects will be
            dropped; in arrays they will be replaced with null. You can use
            a replacer function to replace those with JSON values.
            JSON.stringify(undefined) returns undefined.

            The optional space parameter produces a stringification of the
            value that is filled with line breaks and indentation to make it
            easier to read.

            If the space parameter is a non-empty string, then that string will
            be used for indentation. If the space parameter is a number, then
            the indentation will be that many spaces.

            Example:

            text = JSON.stringify(['e', {pluribus: 'unum'}]);
            // text is '["e",{"pluribus":"unum"}]'


            text = JSON.stringify(['e', {pluribus: 'unum'}], null, '\t');
            // text is '[\n\t"e",\n\t{\n\t\t"pluribus": "unum"\n\t}\n]'

            text = JSON.stringify([new Date()], function (key, value) {
                return this[key] instanceof Date ?
                    'Date(' + this[key] + ')' : value;
            });
            // text is '["Date(---current time---)"]'


        JSON.parse(text, reviver)
            This method parses a JSON text to produce an object or array.
            It can throw a SyntaxError exception.

            The optional reviver parameter is a function that can filter and
            transform the results. It receives each of the keys and values,
            and its return value is used instead of the original value.
            If it returns what it received, then the structure is not modified.
            If it returns undefined then the member is deleted.

            Example:

            // Parse the text. Values that look like ISO date strings will
            // be converted to Date objects.

            myData = JSON.parse(text, function (key, value) {
                var a;
                if (typeof value === 'string') {
                    a =
/^(\d{4})-(\d{2})-(\d{2})T(\d{2}):(\d{2}):(\d{2}(?:\.\d*)?)Z$/.exec(value);
                    if (a) {
                        return new Date(Date.UTC(+a[1], +a[2] - 1, +a[3], +a[4],
                            +a[5], +a[6]));
                    }
                }
                return value;
            });

            myData = JSON.parse('["Date(09/09/2001)"]', function (key, value) {
                var d;
                if (typeof value === 'string' &&
                        value.slice(0, 5) === 'Date(' &&
                        value.slice(-1) === ')') {
                    d = new Date(value.slice(5, -1));
                    if (d) {
                        return d;
                    }
                }
                return value;
            });


    This is a reference implementation. You are free to copy, modify, or
    redistribute.

    This code should be minified before deployment.
    See http://javascript.crockford.com/jsmin.html

    USE YOUR OWN COPY. IT IS EXTREMELY UNWISE TO LOAD CODE FROM SERVERS YOU DO
    NOT CONTROL.
 *
 * Comment from original json_sans_eval.js:
 *
 * Parses a string of well-formed JSON text.
 *
 * If the input is not well-formed, then behavior is undefined, but it is
 * deterministic and is guaranteed not to modify any object other than its
 * return value.
 *
 * This does not use `eval` so is less likely to have obscure security bugs than
 * json2.js.
 * It is optimized for speed, so is much faster than json_parse.js.
 *
 * This library should be used whenever security is a concern (when JSON may
 * come from an untrusted source), speed is a concern, and erroring on malformed
 * JSON is *not* a concern.
 *
 *                      Pros                   Cons
 *                    +-----------------------+-----------------------+
 * json_sans_eval.js  | Fast, secure          | Not validating        |
 *                    +-----------------------+-----------------------+
 * json_parse.js      | Validating, secure    | Slow                  |
 *                    +-----------------------+-----------------------+
 * json2.js           | Fast, some validation | Potentially insecure  |
 *                    +-----------------------+-----------------------+
 *
 * json2.js is very fast, but potentially insecure since it calls `eval` to
 * parse JSON data, so an attacker might be able to supply strange JS that
 * looks like JSON, but that executes arbitrary javascript.
 * If you do have to use json2.js with untrusted data, make sure you keep
 * your version of json2.js up to date so that you get patches as they're
 * released.
 *
 * @param {string} json per RFC 4627
 * @param {function} opt_reviver optional function that reworks JSON objects
 *     post-parse per Chapter 15.12 of EcmaScript3.1.
 *     If supplied, the function is called with a string key, and a value.
 *     The value is the property of 'this'.  The reviver should return
 *     the value to use in its place.  So if dates were serialized as
 *     {@code { "type": "Date", "time": 1234 }}, then a reviver might look like
 *     {@code
 *     function (key, value) {
 *       if (value && typeof value === 'object' && 'Date' === value.type) {
 *         return new Date(value.time);
 *       } else {
 *         return value;
 *       }
 *     }}.
 *     If the reviver returns {@code undefined} then the property named by key
 *     will be deleted from its container.
 *     {@code this} is bound to the object containing the specified property.
 * @return {Object|Array}
 * @author Mike Samuel <mikesamuel@gmail.com>
 */

if (typeof Date.prototype.toJSON !== 'function') {
  Date.prototype.toJSON = function (key) {
    function f(n) {
      // Format integers to have at least two digits.
      return n < 10 ? '0' + n : n;
    }

    return isFinite(this.valueOf()) ?
    this.getUTCFullYear()   + '-' +
    f(this.getUTCMonth() + 1) + '-' +
    f(this.getUTCDate())      + 'T' +
    f(this.getUTCHours())     + ':' +
    f(this.getUTCMinutes())   + ':' +
    f(this.getUTCSeconds())   + 'Z' : null;
  };

  String.prototype.toJSON =
  Number.prototype.toJSON =
  Boolean.prototype.toJSON = function (key) {
    return this.valueOf();
  };
}

var json_sans_eval = (function() {

   var hop = Object.hasOwnProperty;

   ///////////////////// from json2.js //////////////////////////


   var cx = /[\u0000\u00ad\u0600-\u0604\u070f\u17b4\u17b5\u200c-\u200f\u2028-\u202f\u2060-\u206f\ufeff\ufff0-\uffff]/g,
   escapable = /[\\\"\x00-\x1f\x7f-\x9f\u00ad\u0600-\u0604\u070f\u17b4\u17b5\u200c-\u200f\u2028-\u202f\u2060-\u206f\ufeff\ufff0-\uffff]/g,
   gap,
   indent,
   meta = {    // table of character substitutions
     '\b': '\\b',
     '\t': '\\t',
     '\n': '\\n',
     '\f': '\\f',
     '\r': '\\r',
     '"' : '\\"',
     '\\': '\\\\'
   },
   rep;


   function quote(string) {

     // If the string contains no control characters, no quote
     // characters, and no
     // backslash characters, then we can safely slap some quotes around it.
     // Otherwise we must also replace the offending characters with safe escape
     // sequences.

     escapable.lastIndex = 0;
     return escapable.test(string) ?
       '"' + string.replace(escapable, function (a) {
                              var c = meta[a];
                              return typeof c === 'string' ? c :
                                '\\u' + ('0000' + a.charCodeAt(0).toString(16)).slice(-4);
                            }) + '"' :
     '"' + string + '"';
   }


   function str(key, holder) {

     // Produce a string from holder[key].

     var i,          // The loop counter.
     k,          // The member key.
     v,          // The member value.
     length,
     mind = gap,
     partial,
     value = holder[key];

     // If the value has a toJSON method, call it to obtain a replacement value.

     if (value && typeof value === 'object' &&
         typeof value.toJSON === 'function') {
       value = value.toJSON(key);
     }

     // If we were called with a replacer function, then call the replacer to
     // obtain a replacement value.

     if (typeof rep === 'function') {
       value = rep.call(holder, key, value);
     }

     // What happens next depends on the value's type.

     switch (typeof value) {
     case 'string':
       return quote(value);

     case 'number':

       // JSON numbers must be finite. Encode non-finite numbers as null.

       return isFinite(value) ? String(value) : 'null';

     case 'boolean':
     case 'null':

       // If the value is a boolean or null, convert it to a string. Note:
       // typeof null does not produce 'null'. The case is included here in
       // the remote chance that this gets fixed someday.

       return String(value);

       // If the type is 'object', we might be dealing with an object
       // or an array or
       // null.

     case 'object':

       // Due to a specification blunder in ECMAScript, typeof null is 'object',
       // so watch out for that case.

       if (!value) {
         return 'null';
       }

       // Make an array to hold the partial results of stringifying
       // this object value.

       gap += indent;
       partial = [];

       // Is the value an array?

       if (Object.prototype.toString.apply(value) === '[object Array]') {

         // The value is an array. Stringify every element. Use null
         // as a placeholder
         // for non-JSON values.

         length = value.length;
         for (i = 0; i < length; i += 1) {
           partial[i] = str(i, value) || 'null';
         }

         // Join all of the elements together, separated with commas,
         // and wrap them in
         // brackets.

         v = partial.length === 0 ? '[]' :
           gap ? '[\n' + gap +
           partial.join(',\n' + gap) + '\n' +
           mind + ']' :
           '[' + partial.join(',') + ']';
         gap = mind;
         return v;
       }

       // If the replacer is an array, use it to select the members to
       // be stringified.

       if (rep && typeof rep === 'object') {
         length = rep.length;
         for (i = 0; i < length; i += 1) {
           k = rep[i];
           if (typeof k === 'string') {
             v = str(k, value);
             if (v) {
               partial.push(quote(k) + (gap ? ': ' : ':') + v);
             }
           }
         }
       } else {

         // Otherwise, iterate through all of the keys in the object.

         for (k in value) {
           if (hop.call(value, k)) {
             v = str(k, value);
             if (v) {
               partial.push(quote(k) + (gap ? ': ' : ':') + v);
             }
           }
         }
       }

       // Join all of the member texts together, separated with commas,
       // and wrap them in braces.

       v = partial.length === 0 ? '{}' :
         gap ? '{\n' + gap + partial.join(',\n' + gap) + '\n' +
         mind + '}' : '{' + partial.join(',') + '}';
       gap = mind;
       return v;
     }
   }

   function stringify (value, replacer, space) {

     // The stringify method takes a value and an optional replacer,
     // and an optional space parameter, and returns a JSON
     // text. The replacer can be a function that can replace
     // values, or an array of strings that will select the keys. A
     // default replacer method can be provided. Use of the space
     // parameter can produce text that is more easily readable.

     var i;
     gap = '';
     indent = '';

     // If the space parameter is a number, make an indent string
     // containing that
     // many spaces.

     if (typeof space === 'number') {
       for (i = 0; i < space; i += 1) {
         indent += ' ';
       }

       // If the space parameter is a string, it will be used as the
       // indent string.

     } else if (typeof space === 'string') {
       indent = space;
     }

     // If there is a replacer, it must be a function or an array.
     // Otherwise, throw an error.

     rep = replacer;
     if (replacer && typeof replacer !== 'function' &&
         (typeof replacer !== 'object' ||
          typeof replacer.length !== 'number')) {
       throw new Error('json_sans_eval.stringify');
     }

     // Make a fake root object containing our value under the key of ''.
     // Return the result of stringifying the value.

     return str('', {'': value});
   }

   var number
       = '(?:-?\\b(?:0|[1-9][0-9]*)(?:\\.[0-9]+)?(?:[eE][+-]?[0-9]+)?\\b)';
   var oneChar = '(?:[^\\0-\\x08\\x0a-\\x1f\"\\\\]'
       + '|\\\\(?:[\"/\\\\bfnrt]|u[0-9A-Fa-f]{4}))';
   var string = '(?:\"' + oneChar + '*\")';

   // Will match a value in a well-formed JSON file.
   // If the input is not well-formed, may match strangely, but not in an unsafe
   // way.
   // Since this only matches value tokens, it does not match
   // whitespace, colons,
   // or commas.
   var significantToken = new RegExp(
       '(?:false|true|null|[\\{\\}\\[\\]]'
       + '|' + number
       + '|' + string
       + ')', 'g');

   // Matches escape sequences in a string literal
   var escapeSequence = new RegExp('\\\\(?:([^u])|u(.{4}))', 'g');

   // Decodes escape sequences in object literals
   var escapes = {
     '"': '"',
     '/': '/',
     '\\': '\\',
     'b': '\b',
     'f': '\f',
     'n': '\n',
     'r': '\r',
     't': '\t'
   };
   function unescapeOne(_, ch, hex) {
     return ch ? escapes[ch] : String.fromCharCode(parseInt(hex, 16));
   }

   // A non-falsy value that coerces to the empty string when used as a key.
   var EMPTY_STRING = new String('');
   var SLASH = '\\';

   var completeToken = new RegExp(
       '(?:false|true|null|[ \t\r\n]+|[\\{\\}\\[\\],:]'
       + '|' + number
       + '|' + string
       + '|.)', 'g');

   function blank(arr, s, e) { while (--e >= s) { arr[e] = ''; } }

   function checkSyntax(text, keyFilter) {
     var toks = ('' + text).match(completeToken);
     var i = 0, n = toks.length;
     checkArray();
     if (i < n) {
       throw new Error('Trailing tokens ' + toks.slice(i - 1).join(''));
     }
     return toks.join('');

     function checkArray() {
       while (i < n) {
         var t = toks[i++];
         switch (t) {
           case ']': return;
           case '[': checkArray(); break;
           case '{': checkObject(); break;
         }
       }
     }
     function checkObject() {
       // For the tokens    {  "a"  :  null  ,  "b" ...
       // the state is         0    1  2     3  0
       var state = 0;
       // If we need to sanitize instead of validating, uncomment:
       // var skip = 0;  // The index of the first token to skip or 0.
       while (i < n) {
         var t = toks[i++];
         switch (t.charCodeAt(0)) {
           case 0x09: case 0x0a: case 0x0d: case 0x20: continue; // space chars
           case 0x22: // "
             var len = t.length;
             if (len === 1) { throw new Error(t); }
             if (state === 0) {
               if (keyFilter && !keyFilter(
                       t.substring(1, len - 1)
                       .replace(escapeSequence, unescapeOne))) {
                 throw new Error(t);
                 // If we need to sanitize instead of validating, uncomment:
                 // skip = i - 1;
               }
             } else if (state !== 2) { throw new Error(t); }
             break;
           case 0x27: throw new Error(t);  // '
           case 0x2c: // ,
             if (state !== 3) { throw new Error(t); }
             state = 0;
             // If we need to sanitize instead of validating, uncomment:
             // if (skip) { blank(toks, skip, i); skip = 0; }
             continue;
           case 0x3a: // :
             if (state !== 1) { throw new Error(t); }
             break;
           case 0x5b:  // [
             if (state !== 2) { throw new Error(t); }
             checkArray();
             break;
           case 0x7b:  // {
             if (state !== 2) { throw new Error(t); }
             checkObject();
             break;
           case 0x7d:  // }
             // If we need to sanitize instead of validating, uncomment:
             // if (skip) { blank(toks, skip, i - 1); skip = 0; }
             return;
           default:
             if (state !== 2) { throw new Error(t); }
             break;
         }
         ++state;
       }
     }
   };

   function parse (json, opt_reviver) {
     // Split into tokens
     var toks = json.match(significantToken);
     // Construct the object to return
     var result;
     var tok = toks[0];
     if ('{' === tok) {
       result = {};
     } else if ('[' === tok) {
       result = [];
     } else {
       throw new Error(tok);
     }

     // If undefined, the key in an object key/value record to use
     // for the next
     // value parsed.
     var key = void 0;
     // Loop over remaining tokens maintaining a stack of
     // uncompleted objects and
     // arrays.
     var stack = [result];
     for (var i = 1, n = toks.length; i < n; ++i) {
       tok = toks[i];

       var cont;
       switch (tok.charCodeAt(0)) {
       default:  // sign or digit
         cont = stack[0];
         cont[key || cont.length] = +(tok);
         key = void 0;
         break;
       case 0x22:  // '"'
         tok = tok.substring(1, tok.length - 1);
         if (tok.indexOf(SLASH) !== -1) {
           tok = tok.replace(escapeSequence, unescapeOne);
         }
         cont = stack[0];
         if (!key) {
           if (cont instanceof Array) {
             key = cont.length;
           } else {
             key = tok || EMPTY_STRING;  // Use as key for next value seen.
             break;
           }
         }
         cont[key] = tok;
         key = void 0;
         break;
       case 0x5b:  // '['
         cont = stack[0];
         stack.unshift(cont[key || cont.length] = []);
         key = void 0;
         break;
       case 0x5d:  // ']'
         stack.shift();
         break;
       case 0x66:  // 'f'
         cont = stack[0];
         cont[key || cont.length] = false;
         key = void 0;
         break;
       case 0x6e:  // 'n'
         cont = stack[0];
         cont[key || cont.length] = null;
         key = void 0;
         break;
       case 0x74:  // 't'
         cont = stack[0];
         cont[key || cont.length] = true;
         key = void 0;
         break;
       case 0x7b:  // '{'
         cont = stack[0];
         stack.unshift(cont[key || cont.length] = {});
         key = void 0;
         break;
       case 0x7d:  // '}'
         stack.shift();
         break;
       }
     }
     // Fail if we've got an uncompleted object.
     if (stack.length) { throw new Error(); }

     if (opt_reviver) {
       // Based on walk as implemented in http://www.json.org/json2.js
       var walk = function (holder, key) {
         var value = holder[key];
         if (value && typeof value === 'object') {
           var toDelete = null;
           for (var k in value) {
             if (hop.call(value, k) && value !== holder) {
               // Recurse to properties first.  This has the effect of causing
               // the reviver to be called on the object graph depth-first.

               // Since 'this' is bound to the holder of the property, the
               // reviver can access sibling properties of k including ones
               // that have not yet been revived.

               // The value returned by the reviver is used in place of the
               // current value of property k.
               // If it returns undefined then the property is deleted.
               var v = walk(value, k);
               if (v !== void 0) {
                 value[k] = v;
               } else {
                 // Deleting properties inside the loop has vaguely defined
                 // semantics in ES3.
                 if (!toDelete) { toDelete = []; }
                 toDelete.push(k);
               }
             }
           }
           if (toDelete) {
             for (var i = toDelete.length; --i >= 0;) {
               delete value[toDelete[i]];
             }
           }
         }
         return opt_reviver.call(holder, key, value);
       };
       result = walk({ '': result }, '');
     }

     return result;
   }

   return {
     checkSyntax: checkSyntax,
     parse: parse,
     stringify: stringify
   };
})();

if (typeof JSON === 'undefined') { var JSON = {}; }
if (typeof JSON.stringify !== 'function') {
  JSON.stringify = json_sans_eval.stringify;
}
if (typeof JSON.parse !== 'function') {
  JSON.parse = json_sans_eval.parse;
}
;
// Copyright (C) 2007 Google Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

/**
 * @fileoverview the ES5/3 runtime library.
 *
 * <p>It is written in Javascript, not ES5, and would be rejected by the
 * ES53Rewriter. This module exports two globals intended for normal use:<ol>
 * <li>"___" for use by the output of the ES53Rewriter and by some
 *     other untranslated Javascript code.
 * <li>"es53" providing some common services to the ES5/3 programmer.
 * </ol>
 * <p>It also exports the following additional globals:<ul>
 * <li>"safeJSON" why is this being exported?
 * </ul>
 *
 * @author metaweta@gmail.com
 * @requires json_sans_eval, cajaBuildVersion, taming, this
 * @provides ___, safeJSON, WeakMap, cajaVM
 * @overrides Error, EvalError, RangeError, ReferenceError, SyntaxError,
 *   TypeError, URIError, ArrayLike, window
 * @overrides escape, JSON, Proxy
 */

var ___, cajaVM, safeJSON, WeakMap, ArrayLike, Proxy;

(function (global) {
  var Array = global.Array,
      Boolean = global.Boolean,
      Date = global.Date,
      Function = global.Function,
      Number = global.Number,
      Object = global.Object,
      RegExp = global.RegExp,
      String = global.String;

  // For computing the [[Class]] internal property
  var classProp = Object.prototype.toString;
  var gopd = Object.getOwnPropertyDescriptor;
  var defProp = Object.defineProperty;

  // Given an object defined in an es53 frame, we can tell which
  // Object.prototype it inherits from.
  Object.prototype.baseProto___ = Object.prototype;

  var slice = Array.prototype.slice;
  var push = Array.prototype.push;
  var sort = Array.prototype.sort;
  var min = Math.min;
  var replace = String.prototype.replace;


  // Workarounds for FF2 and FF3.0 for
  // https://bugzilla.mozilla.org/show_bug.cgi?id=507453

  var antidote = function() { return void 0; };
  function deodorize(original, end) {
    if (original.__defineGetter__) {
      for (var i = end; i < 0; ++i) {
        original.__defineGetter__(i, antidote);
      }
    }
  }

  function isDeodorized(original, sprop) {
    if (original.__lookupGetter__) {
      return original.__lookupGetter__(sprop) === antidote;
    }
    return false;
  }

  // Blacklist built from:
  // http://www.thespanner.co.uk/2009/07/14/hidden-firefox-properties-revisited/
  // [args, actuals length, callee, formals length, func name, caller]
  deodorize(Function.prototype, -6);

  // [string length]
  deodorize(String.prototype, -1);

  // [source, global, ignore case, last index, multiline, sticky]
  deodorize(RegExp.prototype, -6);

  // [input, multiline, last match, last capture, lcontext, rcontext]
  deodorize(RegExp, -6);

  /**
   * Caja-specific properties
   *
   * Caja combines all numeric properties and uses the special name
   * {@code NUM___} to refer to the single property descriptor.
   * Numeric properties must be enumerable data properties.
   * If the special descriptor is absent, it defaults to
   * {writable:true, configurable:true, enumerable:true, get:void 0, set:void 0}
   *
   * Note that each of the six attributes starts with a different letter.
   * Each property has eight associated properties: six for the attributes
   * and two for writable and callable fastpath flags
   *
   * {@code obj[name + '_v___'] === obj}  means that {@code name} is
   *                                      a data property on {@code obj}.
   * {@code obj.hasOwnProperty(name + '_v___') &&
   *       obj[name + '_v___'] === false} means that {@code name} is an
   *                                      accessor property on {@code obj}.
   * {@code obj[name + '_w___'] === obj}  means that {@code name} is
   *                                      writable (fastpath).
   * {@code obj[name + '_gw___'] === obj} means that {@code name} is
   *                                      writable (grant).
   * {@code obj[name + '_c___'] === obj}  means that {@code name} is
   *                                      configurable.
   * {@code obj[name + '_e___'] === obj}  means that {@code name} is
   *                                      enumurable.
   * {@code obj[name + '_g___']}          is the getter for
   *                                      {@code name} on {@code obj}.
   * {@code obj[name + '_s___']}          is the setter for
   *                                      {@code name} on {@code obj}.
   * {@code obj[name + '_m___'] truthy}   means that {@code name} is
   *                                      callable as a method (fastpath).
   *
   * To prevent accidental misinterpretation of the above inherited
   * attribute descriptors, whenever any are defined for a given
   * {@code obj} and {@code name}, all eight must be. If {@code name}
   * is a string encoding of a number (i.e., where {@code name ===
   * String(+name)}), then all of the above attributes must not be
   * defined directly for {@code name}. Instead, the effective
   * attributes of {@code name} are covered by the actual attributes
   * of {@code 'NUM___'}.
   *
   * Another property suffix commonly used in the code is for virtualized
   * methods; since innocent code and existing host code like domita rely
   * on the original bindings of primordial methods, guest code should not
   * be allowed to change the original bindings; {@code virtualize} installs
   * ES5 getters and setters that store the guest view of the property.
   *
   * {@code obj[name + '_virt___']}       is the virtual version of a primordial
   *                                      method that's exposed to guest code.
   *
   * Per-object properties:
   *
   * {@code obj.ne___ === obj}            means that {@code obj} is not
   *                                      extensible.
   * {@code obj.z___ === obj}             means that {@code obj} is frozen.
   * {@code obj.proxy___ === obj}         means that obj is a proxy.
   * {@code '___' in obj}                 means that {@code obj} is a global
   *                                      object and shouldn't be used as
   *                                      'this'.
   *
   * {@code obj.v___(p)}                  = {@code obj.[[Get]](p)}
   * {@code obj.w___(p,v)}                = {@code obj.[[Put]](p,v)}
   * {@code obj.c___(p)}                  = {@code obj.[[Delete]](p)}
   * {@code obj.m___(p, [as])}            invokes {@code p} as a method safely;
   *                                      it may set the {@code '_m___'}
   *                                      fastpath on {@code obj}.
   * {@code obj.e___()}                   returns an object whose enumerable
   *                                      keys are the ones to iterate over.
   * {@code obj.freeze___()}              freezes the object; on proxies invokes
   *                                      the fix handler.
   * {@code obj.pe___()}                  prevents the object from being
   *                                      extended; on proxies it invokes the
   *                                      fix handler.
   * {@code obj.seal___()}                prevents the object from being
   *                                      extended and makes all own properties
   *                                      nonconfigurable; on proxies it invokes
   *                                      the fix handler.
   * {@code obj.keys___()}                returns the list of enumerable own
   *                                      properties.
   * {@code obj.ownKeys___()}             returns the list of all own
   *                                      properties.
   * {@code obj.allKeys___()}             returns the list of all properties.
   *
   * {@code g.f___(dis, [as])}            is the tamed version of {@code g},
   *                                      though it uses {@code apply}'s
   *                                      interface.
   * {@code g.i___(as)}                   = g.f___(USELESS, [as])
   * {@code g.new___(as)}                 is the tamed version of {@code g}
   *                                      used for constructing an object of
   *                                      class {@code g}.
   * {@code g.ok___ === true}             means g is non-toxic.
   *
   * Since creating function instances is a common pattern and reading
   * properties of a function instance is not, we defer whitelisting the
   * prototype, length, and name properties.
   *
   * {@code f.name___}                    holds the value of the deferred name
   *                                      property of a function instance until
   *                                      it's installed.
   */

  function notFunction(fn, opt_name) {
    throw new TypeError(
      'Expected ' +
      (opt_name ? opt_name + ' to be ' : '') +
      'a function, not ' + (typeof fn) + ': ' + fn);
  }

  function notObject(obj, opt_name) {
    throw new TypeError(
      'Expected ' +
      (opt_name ? opt_name + ' to be ' : '') +
      'an object, not ' + (typeof obj) + ': ' + obj);
  }

  // Missing on IE
  if (!Array.prototype.forEach) {
    Array.prototype.forEach = function(fun) { //, thisp
      var dis = ToObject(this);
      var len = dis.length >>> 0;
      if ('function' !== typeof fun) {
        notFunction(fun);
      }

      var thisp = arguments[1];
      for (var i = 0; i < len; i++) {
        if (i in dis) {
          fun.call(thisp, dis[i], i, dis);
        }
      }
    };
  }

  var hasOwnProperty = Object.prototype.hasOwnProperty;
  // In IE<9, this is an approximation of Object.keys, because a few own
  // properties like 'toString' are never enumerable.
  var fastOwnKeys = Object.keys || function (o) {
    var keys = [];
    for (var k in o) {
      if (hasOwnProperty.call(o, k)) { keys.push(k); }
    }
    return keys;
  };

  Object.prototype.freeze___ = function freeze___() {
      // Frozen means all the properties are neither writable nor
      // configurable, and the object itself is not extensible.
      // Cajoled setters that change properties of the object will
      // fail like any other attempt to change the properties.
      // Tamed setters should check before changing a property.
      if (this.z___ === this) { return this; }
      // Allow function instances to install their instance properties
      // before freezing them.
      if (this.v___ === deferredV) {
        this.v___('length');
      }
      var keys = fastOwnKeys(this);
      for (var k = 0, n = keys.length; k < n; k++) {
        var i = keys[k];
        if (i.length <= 5 || i.substr(i.length - 5) !== '_v___') { continue; }
        var P = i.substr(0, i.length - 5);
        this[P + '_c___'] = false;
        this[P + '_gw___'] = false;
        this[P + '_w___'] = false;
      }
      // inline this.hasNumerics___()
      if (!this.NUM____v___ === this) {
        this.NUM____v___ = this;
        this.NUM____e___ = this;
        this.NUM____g___ = void 0;
        this.NUM____s___ = void 0;
      }
      this.NUM____c___ = false;
      this.NUM____w___ = false;
      this.NUM____m___ = false;
      this.NUM____gw___ = false;
      // Make this non-extensible.
      this.ne___ = this;
      // Cache frozen state.
      this.z___ = this;
      return this;
    };

  ////////////////////////////////////////////////////////////////////////
  // Functions to walk the prototype chain
  ////////////////////////////////////////////////////////////////////////

  /**
   * An object is prototypical iff its 'constructor' property points
   * at a genuine function whose 'prototype' property points back at
   * it.
   */
  function isPrototypical(obj) {
    if ((typeof obj) !== 'object') { return false; }
    if (obj === null) { return false; }
    var constr = obj.constructor;
    if ((typeof constr) !== 'function') { return false; }
    return constr.prototype === obj;
  }

  var BASE_OBJECT_CONSTRUCTOR = {};

  /**
   * Returns the 'constructor' property of obj's prototype.
   * <p>
   * By "obj's prototype", we mean the prototypical object that obj
   * most directly inherits from, not the value of its 'prototype'
   * property. We memoize the apparent prototype into 'Prototype___' to
   * speed up future queries.
   * <p>
   * If obj is a function or not an object, return undefined.
   * <p>
   * If the object is determined to be directly constructed by the 'Object'
   * function in *some* frame, we return distinguished marker value
   * BASE_OBJECT_CONSTRUCTOR.
   */
  function directConstructor(obj) {
    if (obj === null) { return void 0; }
    if (obj === void 0) { return void 0; }
    if ((typeof obj) !== 'object') {
      // Regarding functions, since functions return undefined,
      // directConstructor() doesn't provide access to the
      // forbidden Function constructor.
      // Otherwise, we don't support finding the direct constructor
      // of a primitive.
      return void 0;
    }
    var result;
    if (obj.hasOwnProperty('Prototype___')) {
      var proto = obj.Prototype___;
      // At this point we know that (typeOf(proto) === 'object')
      if (proto === null) { return void 0; }
      result = proto.constructor;
      // rest of: if (!isPrototypical(result))
      if (result.prototype !== proto || (typeof result) !== 'function') {
        result = directConstructor(proto);
      }
    } else {
      if (!obj.hasOwnProperty('constructor')) {
        // TODO(erights): Detect whether this is a valid constructor
        // property in the sense that result is a proper answer. If
        // not, at least give a sensible error, which will be hard to
        // phrase.
        result = obj.constructor;
      } else {
        var oldConstr = obj.constructor;
        // TODO(erights): This code assumes that any 'constructor' property
        // revealed by deleting the own 'constructor' must be the constructor
        // we're interested in.
        if (delete obj.constructor) {
          result = obj.constructor;
          obj.constructor = oldConstr;
        } else if (isPrototypical(obj)) {
          log('Guessing the directConstructor of : ' + obj);
          return BASE_OBJECT_CONSTRUCTOR;
        } else {
          throw new TypeError('Discovery of direct constructors unsupported '
              + 'when the constructor property is not deletable: '
              + obj + '.constructor === ' + oldConstr);
        }
      }

      if ((typeof result) !== 'function' || !(obj instanceof result)) {
        if (obj === obj.baseProto___) {
          return void 0;
        }
        throw new TypeError('Discovery of direct constructors for foreign '
            + 'begotten objects not implemented on this platform');
      }
      if (result.prototype.constructor === result) {
        // Memoize, so it'll be faster next time.
        obj.Prototype___ = result.prototype;
      }
    }
    // If the result is marked as the 'Object' constructor from some feral
    // frame, return the distinguished marker value.
    if (result === result.FERAL_FRAME_OBJECT___) {
      return BASE_OBJECT_CONSTRUCTOR;
    }
    // If the result is the 'Object' constructor from some Caja frame,
    // return the distinguished marker value.
    if (result === obj.CAJA_FRAME_OBJECT___) {
      return BASE_OBJECT_CONSTRUCTOR;
    }
    return result;
  }

  /**
   * Initialize argument constructor <i>feralCtor</i> so that it
   * represents a "subclass" of argument constructor <i>someSuper</i>,
   * and return a non-invokable taming of <i>feralCtor</i>.
   *
   * Given:
   *
   *   function FeralFoo() { ... some uncajoled constructor ... }
   *   var Foo = extend(FeralFoo, FeralSuper, 'Foo');
   *
   * it will be the case that:
   *
   *   new FeralFoo() instanceof Foo
   *
   * however -- and this is the crucial property -- cajoled code will get an
   * error if it invokes either of:
   *
   *   new Foo()
   *   Foo()
   *
   * This allows us to expose the tame Foo to cajoled code, allowing
   * it to sense that all the FeralFoo instances we give it are
   * instanceof Foo, without granting to cajoled code the means to
   * create any new such instances.
   *
   * extend() also sets <i>feralCtor</i>.prototype to set up the
   * prototype chain so that
   *
   *   new FeralFoo() instanceof FeralSuper
   * and
   *   new FeralFoo() instanceof Super
   *
   * @param feralCtor An feral-only uncajoled constructor. This must
   *        NOT be exposed to cajoled code by any other mechanism.
   * @param someSuper Some constructor representing the
   *        superclass. This can be <ul>
   *        <li>a feralCtor that had been provided as a first argument
   *            in a previous call to extend(),
   *        <li>an inertCtor as returned by a previous call to
   *            extend(), or
   *        <li>a constructor that has been marked as such by ___.markCtor().
   *        </ul>
   *        In all cases, someSuper.prototype.constructor must be
   *        a constructor that has been marked as such by
   *        ___.markCtor().
   * @param opt_name If the returned inert constructor is made
   *        available this should be the property name used.
   *
   * @return a tame inert class constructor as described above.
   */
  function extend(feralCtor, someSuper, opt_name) {
    if ('function' !== typeof feralCtor) {
      notFunction(feralCtor, 'feral constructor');
    }
    someSuper = someSuper.prototype.constructor;
    var noop = function () {};
    if (someSuper.new___ === noop.new___) {
      throw new TypeError('Internal: toxic function encountered!');
    }
    noop.prototype = someSuper.prototype;
    feralCtor.prototype = new noop();
    feralCtor.prototype.Prototype___ = someSuper.prototype;

    var inert = function() {
        throw new TypeError('This constructor cannot be called directly.');
      };

    inert.prototype = feralCtor.prototype;
    feralCtor.prototype.constructor = inert;
    taming.tamesTo(feralCtor, inert);
    return markFuncFreeze(inert);
  }

  /**
   * A marker for all objects created within a Caja frame.
   */
  Object.prototype.CAJA_FRAME_OBJECT___ = Object;

  function isDefinedInCajaFrame(o) {
    return !!o.CAJA_FRAME_OBJECT___;
  }

  /**
   * The property descriptor for numerics
   */
  Object.prototype.NUM____v___ = Object.prototype;
  Object.prototype.NUM____gw___ = false;
  Object.prototype.NUM____w___ = false;
  Object.prototype.NUM____m___ = false;
  Object.prototype.NUM____c___ = false;
  Object.prototype.NUM____e___ = Object.prototype;
  Object.prototype.NUM____g___ = void 0;
  Object.prototype.NUM____s___ = void 0;
  Object.prototype.hasNumerics___ = function () {
      return this.NUM____v___ === this;
    };

  function isFrozen(obj) {
    return obj.z___ === obj;
  }

  /**
   * The property descriptor for array lengths
   */
  // This flag is only used when doing a dynamic lookup of the length property.
  Array.prototype.length_v___ = false;
  Array.prototype.length_gw___ = false;
  Array.prototype.length_w___ = false;
  Array.prototype.length_m___ = false;
  Array.prototype.length_c___ = false;
  Array.prototype.length_e___ = false;

  /**
   * Setter for {@code length}.  This is necessary because
   * shortening an array by setting its length may delete numeric properties.
   */
  Array.prototype.length_s___ = markFunc(function (val) {
      // Freezing an array needs to freeze the length property.
      if (this.z___ === this) {
        throw new TypeError('Cannot change the length of a frozen array.');
      }
      val = ToUint32(val);
      // Since increasing the length does not add properties,
      // we don't need to check extensibility.
      if (val >= this.length) {
        return this.length = val;
      }
      // Decreasing the length may delete properties, so
      // we need to check that numerics are configurable.
      if (!this.hasNumerics___() || this.NUM____c___ === this) {
        return this.length = val;
      }
      throw new TypeError(
          'Shortening array may delete non-configurable elements.');
    });

  /**
   * Getter for {@code length}.  Only necessary for returning
   * a property descriptor map and dynamic lookups, since reading
   * {@code length} is automatically whitelisted.
   */
  Array.prototype.length_g___ = markFunc(function () { return this.length; });

  // Replace {@code undefined} and {@code null} by
  // {@code USELESS} for use as {@code this}.  If dis is a global
  // (which we try to detect by looking for the ___ property),
  // then we throw an error (external hull breach).
  function safeDis(dis) {
    if (dis === null || dis === void 0) { return USELESS; }
    if (Type(dis) !== 'Object') { return dis; }
    if ('___' in dis) {
      throw new Error('Internal: toxic global!');
    }
    return asFirstClass(dis);
  }

  var endsWith__ = /__$/;
  var endsWith_e___ = /([\s\S]*)_e___$/;
  var endsWith_v___ = /([\s\S]*)_v___$/;
  var startsWithNUM___ = /^NUM___/;

  function assertValidPropertyName(P) {
    if (endsWith__.test(P)) {
      throw new TypeError('Properties may not end in double underscore.');
    }
  }

  function callFault(var_args) {
    throw new Error('Internal: toxic function encountered!');
  }

  /**
   * Returns the getter, if any, associated with the accessor property
   * {@code name}.
   *
   * Precondition:
   * {@code obj} must not be {@code null} or {@code undefined}.
   * {@code name} must be a string or a number.
   * Postcondition:
   * If {@code name} is a number, a string encoding of a number, or
   * the string {@code 'NUM___'}, then we must return {@code undefined}.
   */
  function getter(obj, name) {
    return obj[name + '_g___'];
  }

  /**
   * Returns the setter, if any, associated with the accessor property
   * {@code name}.
   *
   * Precondition:
   * {@code obj} must not be {@code null} or {@code undefined}.
   * {@code name} must be a string or a number.
   * Postcondition:
   * If {@code name} is a number, a string encoding of a number, or
   * the string {@code 'NUM___'}, then we must return {@code undefined}.
   */
  function setter(obj, name) {
    return obj[name + '_s___'];
  }

  /**
   * Precondition:
   * {@code obj} must not be {@code null} or {@code undefined}.
   * {@code name} must be a string or a number.
   * Postcondition:
   * If {@code name} is a number, a string encoding of a number, or
   * the string {@code 'NUM___'}, then we must return {@code false}.
   */
  function hasOwnAccessor(obj, name) {
    var valueFlag = name + '_v___';
    return obj.hasOwnProperty(valueFlag) && !obj[valueFlag];
  }

  /**
   * Precondition:
   * {@code obj} must not be {@code null} or {@code undefined}.
   * {@code name} must be a string that is not the string encoding of
   *              a number; {@code name} may be {@code 'NUM___'}.
   */
  function fastpathWrite(obj, name) {
    obj[name + '_gw___'] = obj;
    obj[name + '_m___'] = false;
    obj[name + '_w___'] = obj;
  }

  /**
   * Precondition:
   * {@code obj} must not be {@code null} or {@code undefined}.
   * {@code name} must be a string that is not the string encoding of
   *              a number; {@code name} may be {@code 'NUM___'}.
   */
  function fastpathMethod(obj, name) {
    obj[name + '_w___'] = false;
    obj[name + '_m___'] = true;
  }

  ////////////////////////////////////////////////////////////////////////
  // Creating defensible (transitively frozen) objects
  ////////////////////////////////////////////////////////////////////////

  var functionInstanceVoidNameGetter = markFunc(function() { return ''; });
  // Must freeze in a separate step to break circular dependency
  freeze(functionInstanceVoidNameGetter);

  /**
   * We defer the creation of these properties until they're asked for.
   */
  function installFunctionInstanceProps(f) {
    var name = f.name___;
    delete f.name___;
    // Object.prototype.DefineOwnProperty___ may not be defined yet
    f.prototype_v___ = f;
    f.prototype_w___ = f;
    f.prototype_gw___ = f;
    f.prototype_c___ = false;
    f.prototype_e___ = false;
    f.prototype_g___ = void 0;
    f.prototype_s___ = void 0;
    f.prototype_m___ = false;
    f.length_v___ = f;
    f.length_w___ = false;
    f.length_gw___ = false;
    f.length_c___ = false;
    f.length_e___ = false;
    f.length_g___ = void 0;
    f.length_s___ = void 0;
    f.length_m___ = false;
    // Rhino prohibits setting the name property of function instances,
    // so we install a getter instead.
    f.name_v___ = false;
    f.name_w___ = false;
    f.name_gw___ = false;
    f.name_c___ = false;
    f.name_e___ = false;
    f.name_g___ = ((name === '')
        ? functionInstanceVoidNameGetter
        : markFuncFreeze(function() {return name;}));
    f.name_s___ = void 0;
    f.name_m___ = false;
    f.caller_v___ = false;
    f.caller_w___ = false;
    f.caller_gw___ = false;
    f.caller_c___ = false;
    f.caller_e___ = false;
    f.caller_g___ = poisonFuncCaller;
    f.caller_s___ = void 0;
    f.caller_m___ = false;
    f.arguments_v___ = false;
    f.arguments_w___ = false;
    f.arguments_gw___ = false;
    f.arguments_c___ = false;
    f.arguments_e___ = false;
    f.arguments_g___ = poisonFuncArgs;
    f.arguments_s___ = void 0;
    f.arguments_m___ = false;
  }

  function deferredV(name) {
    delete this.v___;
    delete this.w___;
    delete this.c___;
    delete this.DefineOwnProperty___;
    installFunctionInstanceProps(this);
    // Object.prototype.v___ may not be defined yet
    return this.v___ ? this.v___(name) : void 0;
  }

  function deferredW(name, val) {
    delete this.v___;
    delete this.w___;
    delete this.c___;
    delete this.DefineOwnProperty___;
    installFunctionInstanceProps(this);
    return this.w___(name, val);
  }

  function deferredC(name) {
    delete this.v___;
    delete this.w___;
    delete this.c___;
    delete this.DefineOwnProperty___;
    installFunctionInstanceProps(this);
    return this.c___(name);
  }

  function deferredDOP(name, desc) {
    delete this.v___;
    delete this.w___;
    delete this.c___;
    delete this.DefineOwnProperty___;
    installFunctionInstanceProps(this);
    return this.DefineOwnProperty___(name, desc);
  }

  /**
   * For taming a simple function or a safe exophoric function (only reads
   * whitelisted properties of {@code this}).
   */
  function markFunc(fn, name) {
    if (fn.ok___) { return fn; }
    if (!isFunction(fn)) {
      notFunction(fn);
    }
    if (fn.f___ !== Function.prototype.f___ &&
        fn.f___ !== fn.apply) {
      throw new TypeError('The function is already tamed ' +
         'or not from this frame.\n' + fn.f___);
    }
    fn.f___ = fn.apply;
    fn.ok___ = true;
    fn.new___ = fn;
    // Anonymous functions get a 'name' that is the empty string
    fn.name___ = ((name === '' || name === void 0)
        ? '' : '' + name);
    fn.v___ = deferredV;
    fn.w___ = deferredW;
    fn.c___ = deferredC;
    fn.DefineOwnProperty___ = deferredDOP;
    var p = fn.prototype;
    if (p && // must be truthy
        typeof p === 'object' && // must be an object
        // must not already have constructor whitelisted.
        !p.hasOwnProperty('constructor_v___')) {
      p.constructor_v___ = p;
      p.constructor_w___ = p;
      p.constructor_gw___ = p;
      p.constructor_c___ = p;
      p.constructor_e___ = false;
      p.constructor_g___ = void 0;
      p.constructor_s___ = void 0;
      p.constructor_m___ = false;
    }
    return fn;
  }

  /**
   * Declares that it is safe for cajoled code to call this as a
   * function.
   *
   * <p>This may be because it's this-less, or because it's a cajoled
   * function that sanitizes its this on entry.
   */
  function markSafeFunc(fn, name) {
    markFunc(fn, name);
    fn.i___ = function () { return fn.f___(USELESS, arguments); };
    return fn;
  }

  function markFuncFreeze(fn, name) {
    return freeze(markFunc(fn, name));
  }

  /**
   * Is the property {@code name} whitelisted as a value on {@code obj}?
   *
   * Precondition:
   * {@code obj} must not be {@code null} or {@code undefined}.
   * {@code name} must be a string that is not the string encoding
   *              of a number; {@code name} may be {@code 'NUM___'}.
   */
  function hasValue(obj, name) {
    // This doesn't need an "|| name === 'NUM___'" since, for all obj,
    // (obj.NUM____v___) is truthy
    return !!obj[name + '_v___'];
  }

  /**
   * Is the property {@code name} whitelisted as an own value on {@code obj}?
   *
   * Precondition:
   * {@code obj} must not be {@code null} or {@code undefined}.
   * {@code name} must be a string that is not the string encoding
   *              of a number; {@code name} may be {@code 'NUM___'}.
   */
  function hasOwnValue(obj, name) {
    return obj[name + '_v___'] === obj || name === 'NUM___';
  }

  /**
   * Precondition:
   * {@code obj} must not be {@code null} or {@code undefined}.
   * {@code name} must be a string that is not the string encoding
   *              of a number; {@code name} may be {@code 'NUM___'}.
   */
  Object.prototype.HasOwnProperty___ = function (name) {
      return this.hasOwnProperty(name + '_v___') || name === 'NUM___';
    };

  /**
   * Tests whether the fast-path _w___ flag is set, or grantWrite() has been
   * called, on this object itself as an own (non-inherited) attribute.
   * Determines the value of the writable: attribute of property descriptors.
   *
   * Preconditions:
   * {@code obj} must not be {@code null} or {@code undefined}.
   * {@code name} must be a string that is not the string encoding
   *              of a number; {@code name} may be {@code 'NUM___'}.
   */
  function isWritable(obj, name) {
    if (obj[name + '_w___'] === obj) { return true; }
    if (obj[name + '_gw___'] === obj) {
      obj[name + '_m___'] = false;
      obj[name + '_w___'] = obj;
      return true;
    }
    // Frozen and preventExtensions implies hasNumerics
    if (name === 'NUM___' && !obj.hasNumerics___()) {
      obj.NUM____v___ = obj;
      obj.NUM____gw___ = obj;
      obj.NUM____w___ = false;
      obj.NUM____c___ = obj;
      obj.NUM____e___ = obj;
      obj.NUM____g___ = void 0;
      obj.NUM____s___ = void 0;
      obj.NUM____m___ = false;
      return true;
    }
    return false;
  }

  /**
   * Tests whether {@code grantEnumerate} has been called on the property
   * {@code name} of this object or one of its ancestors.
   *
   * Precondition:
   * {@code obj} must not be {@code null} or {@code undefined}.
   * {@code name} must be a string that is not the string encoding
   *              of a number; {@code name} may be {@code 'NUM___'}.
   */
  function isEnumerable(obj, name) {
    // This doesn't need an "|| name === 'NUM___'" since, for all obj,
    // (obj.NUM____e___) is truthy
    return !!obj[name + '_e___'];
  }

  /**
   * Tests whether {@code grantConfigure} has been called on the property
   * {@code name} of this object or one of its ancestors.
   *
   * Precondition:
   * {@code obj} must not be {@code null} or {@code undefined}.
   * {@code name} must be a string that is not the string encoding
   *     of a number; {@code name} may be 'NUM___'.
   */
  function isConfigurable(obj, name) {
    return obj[name + '_c___'] === obj ||
        (name === 'NUM___' && !obj.hasNumerics___());
  }

  function isExtensible(obj) {
    return Type(obj) === 'Object' && obj.ne___ !== obj;
    // TODO(felix8a): (o===Object(o)) is faster than (Type(o)==='Object')
    // except Rhino has a bug where it goes into an infinite loop if you
    // ever do Object(Object.prototype).
    // https://bugzilla.mozilla.org/show_bug.cgi?id=760973
  }

  /**
   * Tests whether an assignment to {@code obj[name]} would extend the object.
   *
   * Precondition:
   * {@code obj} must not be {@code null} or {@code undefined}.
   * {@code name} must be a string.
   */
  function wouldExtend(obj, name) {
    // Would assigning to a numeric index extend the object?
    if (isNumericName(name)) {
      return !obj.hasOwnProperty(name);
    }
    // If name is an own data property, then writing to it won't extend obj.
    if (hasOwnValue(obj, name)) { return false; }
    // If name is an inherited accessor property, invoking the
    // setter won't extend obj. (In any uncajoled setter where it
    // might, the taming must throw an error instead.)
    if (obj[name + '_s___']) { return false; }
    return true;
  }

  function isArray(obj) {
    return classProp.call(obj) === '[object Array]';
  }

  function isFunction(obj) {
    return classProp.call(obj) === '[object Function]';
  }

  function isError(obj) {
    return classProp.call(obj) === '[object Error]';
  }

  /**
   * Returns an object whose enumerable keys are the ones to iterate over.
   * For most objects, this will be the object itself; for proxies and tamings
   * it may be different.
   */
  Object.prototype.e___ = function() {
      return this;
    };

  function allKeys(obj) {
    return obj.allKeys___();
  }

  Object.prototype.allKeys___ = function () {
      var i, m, result = [];
      for (i in this) {
        if (isNumericName(i)) {
          result.push(i);
        } else {
          if (startsWithNUM___.test(i) && endsWith__.test(i)) { continue; }
          m = i.match(endsWith_v___);
          if (m) { result.push(m[1]); }
        }
      }
      return result;
    };

  function ownEnumKeys(obj) {
    return obj.keys___();
  }

  Object.prototype.keys___ = function () {
      var i, m, result = [];
      for (i in this) {
        if (!this.hasOwnProperty(i)) { continue; }
        if (isNumericName(i)) {
          result.push(i);
        } else {
          if (startsWithNUM___.test(i) && endsWith__.test(i)) { continue; }
          m = i.match(endsWith_e___);
          if (m && this[i]) { result.push(m[1]); }
        }
      }
      return result;
    };

  Object.prototype.ownKeys___ = function () {
      var i, m, result = [];
      var keys = fastOwnKeys(this);
      for (var k = 0, n = keys.length; k < n; k++) {
        i = keys[k];
        // inline isNumericName(i)
        if (typeof i === 'number' || ('' + (+i)) === i) {
          result.push(i);
        } else if (5 < i.length && i.substr(i.length - 5) === '_v___'
                   && i.substr(0, 6) !== 'NUM___') {
          result.push(i.substr(0, i.length - 5));
        }
      }
      return result;
    };

  function ownUntamedKeys(obj) {
    var i, m, result = [];
    for (i in obj) {
      if (obj.hasOwnProperty(i) && (isNumericName(i) || !endsWith__.test(i))) {
        result.push(i);
      }
    }
    return result;
  }

  /**
   * Returns a new object whose only utility is its identity and (for
   * diagnostic purposes only) its name.
   */
  function Token(name) {
    name = '' + name;
    return snowWhite({
        toString: markFuncFreeze(function tokenToString() {
            return name;
          }),
        throwable___: true
      });
  }

  /**
   * Checks if {@code n} is governed by the {@code NUM___} property descriptor.
   *
   * Preconditions:
   *   {@code typeof n === 'number'} or {@code 'string'}
   */
  function isNumericName(n) {
    return typeof n === 'number' || ('' + (+n)) === n;
  }

  ////////////////////////////////////////////////////////////////////////
  // JSON
  ////////////////////////////////////////////////////////////////////////

  // TODO: Can all this JSON stuff be moved out of the TCB?
  function jsonParseOk(json) {
    try {
      var x = json.parse('{"a":3}');
      return x.a === 3;
    } catch (e) {
      return false;
    }
  }

  function jsonStringifyOk(json) {
    try {
      var x = json.stringify({'a':3, 'b__':4}, function replacer(k, v) {
          return (/__$/.test(k) ? void 0 : v);
        });
      if (x !== '{"a":3}') {
        return false;
      }
      // ie8 has a buggy JSON unless this update has been applied:
      //   http://support.microsoft.com/kb/976662
      // test for one of the known bugs.
      x = json.stringify(void 0, 'invalid');
      return x === void 0;
    } catch (e) {
      return false;
    }
  }

  var goodJSON = {};
  var parser = jsonParseOk(JSON) ? JSON.parse : json_sans_eval.parse;
  goodJSON.parse = markFunc(function () {
      return whitelistAll(parser.apply(this, arguments), true);
    });
  goodJSON.stringify = markFunc(jsonStringifyOk(JSON) ?
      JSON.stringify : json_sans_eval.stringify);

  safeJSON = snowWhite({
      CLASS___: 'JSON',
      parse: markFunc(function (text, opt_reviver) {
          var reviver = void 0;
          if (opt_reviver) {
            reviver = markFunc(function (key, value) {
                return opt_reviver.f___(this, arguments);
              });
          }
          return goodJSON.parse(
              json_sans_eval.checkSyntax(
                  text,
                  function (key) {
                    return !endsWith__.test(key);
                  }),
              reviver);
        }),
      stringify: markFunc(function (obj, opt_replacer, opt_space) {
          switch (typeof opt_space) {
            case 'number': case 'string': case 'undefined': break;
            default:
              throw new TypeError(
                'Third arg to stringify must be a number or string');
          }
          var replacer;
          if (opt_replacer) {
            replacer = markFunc(function (key, value) {
                if (!this.HasProperty___(key)) { return void 0; }
                return opt_replacer.f___(this, arguments);
              });
          } else {
            replacer = markFunc(function (key, value) {
              if (this[key + '_g___']) {
                return this[key + '_g___'].i___();
              }
              return (this.HasProperty___(key) || key === '') ?
                  value :
                  void 0;
            });
          }
          return goodJSON.stringify(obj, replacer, opt_space);
        })
    });


  ////////////////////////////////////////////////////////////////////////
  // Diagnostics and condition enforcement
  ////////////////////////////////////////////////////////////////////////

  /**
   * The initial default logging function does nothing.
   * <p>
   * Note: JavaScript has no macros, so even in the "does nothing"
   * case, remember that the arguments are still evaluated.
   */
  function defaultLogger(str, opt_stop) {}
  var myLogFunc = markFuncFreeze(defaultLogger);

  /**
   * Gets the currently registered logging function.
   */
  function getLogFunc() { return myLogFunc; }
  markFunc(getLogFunc);

  /**
   * Register newLogFunc as the current logging function, to be called
   * by {@code ___.log(str)}.
   * <p>
   * A logging function is assumed to have the signature
   * {@code (str, opt_stop)}, where<ul>
   * <li>{@code str} is the diagnostic string to be logged, and
   * <li>{@code opt_stop}, if present and {@code true}, indicates
   *     that normal flow control is about to be terminated by a
   *     throw. This provides the logging function the opportunity to
   *     terminate normal control flow in its own way, such as by
   *     invoking an undefined method, in order to trigger a Firebug
   *     stacktrace.
   * </ul>
   */
  function setLogFunc(newLogFunc) { myLogFunc = newLogFunc; }
  markFunc(setLogFunc);

  /**
   * Calls the currently registered logging function.
   */
  function log(str) { myLogFunc('' + str); }
  markFunc(log);

  /**
   * Like an assert that can't be turned off.
   * <p>
   * Either returns true (on success) or throws (on failure). The
   * arguments starting with {@code var_args} are converted to
   * strings and appended together to make the message of the Error
   * that's thrown.
   * <p>
   * TODO(erights) We may deprecate this in favor of <pre>
   *     if (!test) { throw new Error(var_args...); }
   * </pre>
   */
  function enforce(test, var_args) {
    if (!test) { throw new Error(slice.call(arguments, 1).join('')); }
    return true;
  }

  /**
   * Enforces {@code typeof specimen === typename}, in which case
   * specimen is returned.
   * <p>
   * If not, throws an informative TypeError
   * <p>
   * opt_name, if provided, should be a name or description of the
   * specimen used only to generate friendlier error messages.
   */
  function enforceType(specimen, typename, opt_name) {
    if (typeof specimen !== typename) {
      throw new TypeError('Expected ' + typename + ', not ' +
          typeof specimen + ': ' + (opt_name || specimen));
    }
    return specimen;
  }

  /**
   * Enforces that specimen is a non-negative integer within the range
   * of exactly representable consecutive integers, in which case
   * specimen is returned.
   * <p>
   * "Nat" is short for "Natural number".
   */
  function enforceNat(specimen) {
    enforceType(specimen, 'number');
    if (specimen < 0 || specimen === specimen + 1 ||
        Math.floor(specimen) !== specimen) {
      throw new TypeError('Must be a non-negative integer: ' + specimen);
    }
    // Could pre-compute precision limit, but probably not faster
    // enough to be worth it.
    if (Math.floor(specimen - 1) !== specimen - 1) {
      throw new TypeError('Integer too large: ' + specimen);
    }
    return specimen;
  }

  /**
   * Returns a function that can typically be used in lieu of
   * {@code func}, but that logs a deprecation warning on first use.
   * <p>
   * Currently for internal use only, though may make this available
   * on {@code ___} or even {@code es53} at a later time, after
   * making it safe for such use. Forwards only arguments to
   * {@code func} and returns results back, without forwarding
   * {@code this}. If you want to deprecate an exophoric function,
   * deprecate a bind()ing of that function instead.
   */
  function deprecate(func, badName, advice) {
    var warningNeeded = true;
    return function() {
        if (warningNeeded) {
          log('"' + badName + '" is deprecated.\n' + advice);
          warningNeeded = false;
        }
        return func.apply(USELESS, arguments);
      };
  }

  /**
   * Create a unique identification of a given table identity that can
   * be used to invisibly (to cajoled code) annotate a key object to
   * index into a table.
   * <p>
   * magicCount and MAGIC_TOKEN together represent a
   * unique-across-frames value safe against collisions, under the
   * normal Caja threat model assumptions. magicCount and
   * MAGIC_NAME together represent a probably unique across frames
   * value, with which can generate strings in which collision is
   * unlikely but possible.
   * <p>
   * The MAGIC_TOKEN is a unique unforgeable per-ES53 runtime
   * value. magicCount is a per-ES53 counter, which increments each
   * time a new one is needed.
   */
  var magicCount = 0;
  var MAGIC_NUM = Math.random();
  var MAGIC_TOKEN = Token('MAGIC_TOKEN_FOR:' + MAGIC_NUM);
  // Using colons in the below causes it to fail on IE since getting a
  // property whose name contains a colon on a DOM table element causes
  // an exception.
  var MAGIC_NAME = '_index;'+ MAGIC_NUM + ';';

  /**
   *
   * Creates a new mutable associative table mapping from the
   * identity of arbitrary keys (as defined by <tt>identical()</tt>) to
   * arbitrary values.
   *
   * <p>Operates as specified by <a href=
   * "http://wiki.ecmascript.org/doku.php?id=harmony:weak_maps"
   * >weak maps</a>, including the optional parameters of the old
   * <a href=
   * "http://wiki.ecmascript.org/doku.php?id=strawman:ephemeron_tables&rev=1269457867#implementation_considerations"
   * >Implementation Considerations</a> section regarding emulation on
   * ES3, except that, when {@code opt_useKeyLifetime} is falsy or
   * absent, the keys here may be primitive types as well.
   *
   * <p> To support Domita, the keys might be host objects.
   */
  function newTable(opt_useKeyLifetime, opt_expectedSize) {
    magicCount++;
    var myMagicIndexName = MAGIC_NAME + magicCount + '___';

    function setOnKey(key, value) {
      var ktype = typeof key;
      if (!key || (ktype !== 'function' && ktype !== 'object')) {
        notObject(key, 'WeakMap key');
      }
      var list = key[myMagicIndexName];
      // To distinguish key from objects that derive from it,
      //    list[0] should be === key
      // For odd positive i,
      //    list[i] is the MAGIC_TOKEN for an ES53 runtime (i.e., a
      //            browser frame in which the ES53 runtime has been
      //            loaded). The myMagicName and the MAGIC_TOKEN
      //            together uniquely identify a table.
      //    list[i+1] is the value stored in that table under this key.
      if (!list || list[0] !== key) {
        key[myMagicIndexName] = [key, MAGIC_TOKEN, value];
      } else {
        var i;
        for (i = 1; i < list.length; i += 2) {
          if (list[i] === MAGIC_TOKEN) { break; }
        }
        list[i] = MAGIC_TOKEN;
        list[i + 1] = value;
      }
    }

    function getOnKey(key) {
      var ktype = typeof key;
      if (!key || (ktype !== 'function' && ktype !== 'object')) {
        notObject(key, 'WeakMap key');
      }
      var list = key[myMagicIndexName];
      if (!list || list[0] !== key) {
        return void 0;
      } else {
        for (var i = 1; i < list.length; i += 2) {
          if (list[i] === MAGIC_TOKEN) { return list[i + 1]; }
        }
        return void 0;
      }
    }

    function hasOnKey(key) {
      var ktype = typeof key;
      if (!key || (ktype !== 'function' && ktype !== 'object')) {
        notObject(key, 'WeakMap key');
      }
      var list = key[myMagicIndexName];
      if (!list || list[0] !== key) {
        return false;
      } else {
        for (var i = 1; i < list.length; i += 2) {
          if (list[i] === MAGIC_TOKEN) { return true; }
        }
        return false;
      }
    }

    if (opt_useKeyLifetime) {
      return snowWhite({
          set: markFuncFreeze(setOnKey),
          get: markFuncFreeze(getOnKey),
          has: markFuncFreeze(hasOnKey)
        });
    }

    var myValues = [];

    function setOnTable(key, value) {
      var index;
      switch (typeof key) {
        case 'object':
        case 'function': {
          if (null === key) { myValues.prim_null = value; return; }
          index = getOnKey(key);
          if (value === void 0) {
            if (index === void 0) {
              return;
            } else {
              setOnKey(key, void 0);
            }
          } else {
            if (index === void 0) {
              index = myValues.length;
              setOnKey(key, index);
            }
          }
          break;
        }
        case 'string': {
          index = 'str_' + key;
          break;
        }
        default: {
          index = 'prim_' + key;
          break;
        }
      }
      if (value === void 0) {
        // TODO(erights): Not clear that this is the performant
        // thing to do when index is numeric and < length-1.
        delete myValues[index];
      } else {
        myValues[index] = value;
      }
    }

    /**
     * If the key is absent, returns {@code undefined}.
     * <p>
     * Users of this table cannot distinguish an {@code undefined}
     * value from an absent key.
     */
    function getOnTable(key) {
      switch (typeof key) {
        case 'object':
        case 'function': {
          if (null === key) { return myValues.prim_null; }
          var index = getOnKey(key);
          if (void 0 === index) { return void 0; }
          return myValues[index];
        }
        case 'string': { return myValues['str_' + key]; }
        default: { return myValues['prim_' + key]; }
      }
    }

    /**
     * Returns whether this table contains the {@code key}.
     */
    function hasOnTable(key) {
      switch (typeof key) {
        case 'object':
        case 'function': {
          if (null === key) { return 'prim_null' in myValues; }
          var index = getOnKey(key);
          return void 0 !== index;
        }
        case 'string': { return ('str_' + key) in myValues; }
        default: { return ('prim_' + key) in myValues; }
      }
    }

    return snowWhite({
        set: markFuncFreeze(setOnTable),
        get: markFuncFreeze(getOnTable),
        has: markFuncFreeze(hasOnTable)
      });
  }

  WeakMap = WeakMap ?
      (function (WeakMap) {
        return markFunc(function () {
          var result = WeakMap();
          // DefineOwnProperty___ may not be defined yet.
          markFunc(result.get);
          result.get_v___ = result;
          result.get_c___ = false;
          result.get_w___ = false;
          result.get_gw___ = result;
          result.get_e___ = result;
          result.get_m___ = false;
          result.get_g___ = false;
          result.get_s___ = false;

          markFunc(result.set);
          result.set_v___ = result;
          result.set_c___ = false;
          result.set_w___ = false;
          result.set_gw___ = result;
          result.set_e___ = result;
          result.set_m___ = false;
          result.set_g___ = false;
          result.set_s___ = false;

          markFunc(result.has);
          result.has_v___ = result;
          result.has_c___ = false;
          result.has_w___ = false;
          result.has_gw___ = result;
          result.has_e___ = result;
          result.has_m___ = false;
          result.has_g___ = false;
          result.has_s___ = false;
          return result;
        });
      })(WeakMap) :
      markFunc(function () { return newTable(true); });

  var registeredImports = [];

  /**
   * Gets or assigns the id associated with this (assumed to be)
   * imports object, registering it so that
   * <tt>getImports(getId(imports)) === imports</tt>.
   * <p>
   * This system of registration and identification allows us to
   * cajole html such as
   * <pre>&lt;a onmouseover="alert(1)"&gt;Mouse here&lt;/a&gt;</pre>
   * into html-writing JavaScript such as<pre>
   * IMPORTS___.document.innerHTML = "
   *  &lt;a onmouseover=\"
   *    (function(IMPORTS___) {
   *      IMPORTS___.alert(1);
   *    })(___.getImports(" + ___.getId(IMPORTS___) + "))
   *  \"&gt;Mouse here&lt;/a&gt;
   * ";
   * </pre>
   * If this is executed by a plugin whose imports is assigned id 42,
   * it generates html with the same meaning as<pre>
   * &lt;a onmouseover="___.getImports(42).alert(1)"&gt;Mouse here&lt;/a&gt;
   * </pre>
   * <p>
   * An imports is not registered and no id is assigned to it until the
   * first call to <tt>getId</tt>. This way, an imports that is never
   * registered, or that has been <tt>unregister</tt>ed since the last
   * time it was registered, will still be garbage collectable.
   */
  function getId(imports) {
    enforceType(imports, 'object', 'imports');
    var id;
    if ('id___' in imports) {
      id = enforceType(imports.id___, 'number', 'id');
    } else {
      id = imports.id___ = registeredImports.length;
    }
    registeredImports[id] = imports;
    return id;
  }

  /**
   * Gets the imports object registered under this id.
   * <p>
   * If it has been <tt>unregistered</tt> since the last
   * <tt>getId</tt> on it, then <tt>getImports</tt> will fail.
   */
  function getImports(id) {
    var result = registeredImports[enforceType(id, 'number', 'id')];
    if (result === void 0) {
      throw new Error('Internal: imports#', id, ' unregistered');
    }
    return result;
  }

  /**
   * If you know that this <tt>imports</tt> no longer needs to be
   * accessed by <tt>getImports</tt>, then you should
   * <tt>unregister</tt> it so it can be garbage collected.
   * <p>
   * After unregister()ing, the id is not reassigned, and the imports
   * remembers its id. If asked for another <tt>getId</tt>, it
   * reregisters itself at its old id.
   */
  function unregister(imports) {
    enforceType(imports, 'object', 'imports');
    if ('id___' in imports) {
      var id = enforceType(imports.id___, 'number', 'id');
      registeredImports[id] = void 0;
    }
  }

  ////////////////////////////////////////////////////////////////////////
  // Creating defensible (transitively frozen) objects.
  ////////////////////////////////////////////////////////////////////////

  var defended = newTable();
  /**
   * To define a defended object is to freeze it and all objects
   * transitively reachable from it via transitive reflective property
   * traversal.  Note, this is only subjective freezing; uncajoled code can
   * still modify the object, and can make a previously-defended object
   * into a subjectively-mutable object.
   */

  var def = markFuncFreeze(function (root) {

    // 'defended' is a table of objects known to be deep-frozen.  In the
    // loop below, we add objects to the table before they're completely
    // deep-frozen, because that's much faster than making another table
    // for pending-deep-frozen.  But def can throw an error before it
    // completes a deep freeze, so to maintain the contract of 'defended',
    // we set it to null here, and restore its value when we know that the
    // deep-freeze succeeded.  If we do error out, we lose our existing
    // knowledge of what's already deep-frozen, and will have to
    // re-traverse the object tree on the next call to def(), but that
    // should be rare in practice.
    var defending = defended || newTable();
    defended = null;

    var i, n;
    var todo = [root];
    while (todo.length) {
      var val = todo.pop();
      // TODO(felix8a): Rhino bug; see isExtensible.
      if (val !== Object.prototype && val !== Object(val)) { continue; }
      if (defending.get(val)) { continue; }
      defending.set(freeze(val), true);
      todo.push(stdGetPrototypeOf(val));
      // Note, only subjectively-visible properties.
      var keys = val.ownKeys___();
      for (i = 0, n = keys.length; i < n; i++) {
        var desc = val.GetOwnProperty___(keys[i]);
        desc.value && todo.push(desc.value);
        desc.get && todo.push(desc.get);
        desc.set && todo.push(desc.set);
      }
    }
    defended = defending;
    return root;
  });

  ////////////////////////////////////////////////////////////////////////
  // Tokens
  ////////////////////////////////////////////////////////////////////////

  /**
   * When a {@code this} value must be provided but nothing is
   * suitable, provide this useless object instead.
   */
  var USELESS = Token('USELESS');

  /**
   * A unique value that should never be made accessible to untrusted
   * code, for distinguishing the absence of a result from any
   * returnable result.
   * <p>
   * See makeNewModuleHandler's getLastOutcome().
   */
  var NO_RESULT = Token('NO_RESULT');

  ////////////////////////////////////////////////////////////////////////
  // Guards and Trademarks
  ////////////////////////////////////////////////////////////////////////

  /**
   * The identity function just returns its argument.
   */
  function identity(x) { return x; }

  /**
   * One-arg form is known in scheme as "call with escape
   * continuation" (call/ec).
   *
   * <p>In this analogy, a call to {@code callWithEjector} emulates a
   * labeled statement. The ejector passed to the {@code attemptFunc}
   * emulates the label part. The {@code attemptFunc} itself emulates
   * the statement being labeled. And a call to {@code eject} with
   * this ejector emulates the return-to-label statement.
   *
   * <p>We extend the normal notion of call/ec with an
   * {@code opt_failFunc} in order to give more the sense of a
   * {@code try/catch} (or similarly, the {@code escape} special
   * form in E). The {@code attemptFunc} is like the {@code try}
   * clause and the {@code opt_failFunc} is like the {@code catch}
   * clause. If omitted, {@code opt_failFunc} defaults to the
   * {@code identity} function.
   *
   * <p>{@code callWithEjector} creates a fresh ejector -- a one
   * argument function -- for exiting from this attempt. It then calls
   * {@code attemptFunc} passing that ejector as argument. If
   * {@code attemptFunc} completes without calling the ejector, then
   * this call to {@code callWithEjector} completes
   * likewise. Otherwise, if the ejector is called with an argument,
   * then {@code opt_failFunc} is called with that argument. The
   * completion of {@code opt_failFunc} is then the completion of the
   * {@code callWithEjector} as a whole.
   *
   * <p>The ejector stays live until {@code attemptFunc} is exited,
   * at which point the ejector is disabled. Calling a disabled
   * ejector throws.
   *
   * <p>Historic note: This was first invented by John C. Reynolds in
   * <a href="http://doi.acm.org/10.1145/800194.805852"
   * >Definitional interpreters for higher-order programming
   * languages</a>. Reynold's invention was a special form as in E,
   * rather than a higher order function as here and in call/ec.
   */
  function callWithEjector(attemptFunc, opt_failFunc) {
    var failFunc = opt_failFunc || identity;
    var disabled = false;
    var token = new Token('ejection');
    var stash = void 0;
    function ejector(result) {
      if (disabled) {
        throw new Error('ejector disabled');
      } else {
        // don't disable here.
        stash = result;
        throw token;
      }
    }
    markFuncFreeze(ejector);
    try {
      try {
        return attemptFunc.m___('call', [USELESS, ejector]);
      } finally {
        disabled = true;
      }
    } catch (e) {
      if (e === token) {
        return failFunc.m___('call', [USELESS, stash]);
      } else {
        throw e;
      }
    }
  }

  /**
   * Safely invokes {@code opt_ejector} with {@code result}.
   * <p>
   * If {@code opt_ejector} is falsy, disabled, or returns
   * normally, then {@code eject} throws. Under no conditions does
   * {@code eject} return normally.
   */
  function eject(opt_ejector, result) {
    if (opt_ejector) {
      opt_ejector.m___('call', [USELESS, result]);
      throw new Error('Ejector did not exit: ', opt_ejector);
    } else {
      throw new Error(result);
    }
  }

  /**
   * Internal routine for making a trademark from a table.
   * <p>
   * To untangle a cycle, the guard made by {@code makeTrademark} is
   * not yet either stamped or frozen. The caller of
   * {@code makeTrademark} must do both before allowing it to
   * escape.
   */
  function makeTrademark(typename, table) {
    typename = '' + typename;
    return snowWhite({
        toString: markFuncFreeze(function() { return typename + 'Mark'; }),

        stamp: snowWhite({
          toString: markFuncFreeze(function() { return typename + 'Stamp'; }),
          mark___: markFuncFreeze(function(obj) {
            table.set(obj, true);
            return obj;
          })
        }),

        guard: snowWhite({
          toString: markFuncFreeze(function() { return typename + 'T'; }),
          coerce: markFuncFreeze(function(specimen, opt_ejector) {
            if (table.get(specimen)) { return specimen; }
            eject(opt_ejector,
                  'Specimen does not have the "' + typename + '" trademark');
          })
        })
      });
  }

  /**
   * Objects representing guards should be marked as such, so that
   * they will pass the {@code GuardT} guard.
   * <p>
   * {@code GuardT} is generally accessible as
   * {@code cajaVM.GuardT}. However, {@code GuardStamp} must not be
   * made generally accessible, but rather only given to code trusted
   * to use it to deem as guards things that act in a guard-like
   * manner: A guard MUST be immutable and SHOULD be idempotent. By
   * "idempotent", we mean that<pre>
   *     var x = g(specimen, ej); // may fail
   *     // if we're still here, then without further failure
   *     g(x) === x
   * </pre>
   */
  var GuardMark = makeTrademark('Guard', newTable(true));
  var GuardT = GuardMark.guard;
  var GuardStamp = GuardMark.stamp;
  freeze(GuardStamp.mark___(GuardT));

  /**
   * The {@code Trademark} constructor makes a trademark, which is a
   * guard/stamp pair, where the stamp marks and freezes unfrozen
   * records as carrying that trademark and the corresponding guard
   * cerifies objects as carrying that trademark (and therefore as
   * having been marked by that stamp).
   * <p>
   * By convention, a guard representing the type-like concept 'Foo'
   * is named 'FooT'. The corresponding stamp is 'FooStamp'. And the
   * record holding both is 'FooMark'. Many guards also have
   * {@code of} methods for making guards like themselves but
   * parameterized by further constraints, which are usually other
   * guards. For example, {@code T.ListT} is the guard representing
   * frozen array, whereas {@code T.ListT.of(cajaVM.GuardT)}
   * represents frozen arrays of guards.
   */
  function Trademark(typename) {
    var result = makeTrademark(typename, newTable(true));
    freeze(GuardStamp.mark___(result.guard));
    return result;
  }
  markFuncFreeze(Trademark);

  /**
   * First ensures that g is a guard; then does
   * {@code g.coerce(specimen, opt_ejector)}.
   */
  function guard(g, specimen, opt_ejector) {
    g = GuardT.coerce(g); // failure throws rather than ejects
    return g.coerce(specimen, opt_ejector);
  }

  /**
   * First ensures that g is a guard; then checks whether the specimen
   * passes that guard.
   * <p>
   * If g is a coercing guard, this only checks that g coerces the
   * specimen to something rather than failing. Note that trademark
   * guards are non-coercing, so if specimen passes a trademark guard,
   * then specimen itself has been marked with that trademark.
   */
  function passesGuard(g, specimen) {
    g = GuardT.coerce(g); // failure throws rather than ejects
    return callWithEjector(
      markFuncFreeze(function(opt_ejector) {
        g.coerce(specimen, opt_ejector);
        return true;
      }),
      markFuncFreeze(function(ignored) {
        return false;
      })
    );
  }

  /**
   * Given that {@code stamps} is a list of stamps and
   * {@code record} is a non-frozen object, this marks record with
   * the trademarks of all of these stamps, and then freezes and
   * returns the record.
   * <p>
   * If any of these conditions do not hold, this throws.
   */
  function stamp(stamps, record) {
    // TODO: Should nonextensible objects be stampable?
    if (isFrozen(record)) {
      throw new TypeError("Can't stamp frozen objects: " + record);
    }
    stamps = slice.call(stamps, 0);
    var numStamps = stamps.length;
    // First ensure that we will succeed before applying any stamps to
    // the record.
    var i;
    for (i = 0; i < numStamps; i++) {
      if (!('mark___' in stamps[i])) {
        throw new TypeError("Can't stamp with a non-stamp: " + stamps[i]);
      }
    }
    for (i = 0; i < numStamps; i++) {
      // Only works for real stamps, postponing the need for a
      // user-implementable auditing protocol.
      stamps[i].mark___(record);
    }
    return freeze(record);
  }

  /**
   * Create a guard which passes all objects present in {@code table}.
   * This may be used to define trademark-like systems which do not require
   * the object to be frozen.
   *
   * @param {string} typename Used for toString results.
   * @param {string} errorMessage Used when an object does not pass the guard.
   */
  function makeTableGuard(table, typename, errorMessage) {
    var g = whitelistAll({
      toString: markFuncFreeze(function() { return typename + 'T'; }),
      coerce: markFuncFreeze(function(specimen, opt_ejector) {
        if (Object(specimen) === specimen && table.get(specimen)) {
          return specimen;
        }
        eject(opt_ejector, errorMessage);
      })
    });
    stamp([GuardStamp], g);
    return g;
  }

  ////////////////////////////////////////////////////////////////////////
  // Sealing and Unsealing
  ////////////////////////////////////////////////////////////////////////

  /**
   * Returns a pair of functions such that the seal(x) wraps x in an object
   * so that only unseal can get x back from the object.
   * <p>
   * TODO(erights): The only remaining use as of this writing is
   * in domado for css. Perhaps a refactoring is in order.
   *
   * @return {object} of the form
   *     { seal: function seal(x) { return Token('(box)'); },
   *       unseal: function unseal(box) { return x; } }.
   */
  function makeSealerUnsealerPair() {
    var table = newTable(true);
    var undefinedStandin = {};
    function seal(payload) {
      if (payload === void 0) {
        payload = undefinedStandin;
      }
      var box = Token('(box)');
      table.set(box, payload);
      return box;
    }
    function unseal(box) {
      var payload = table.get(box);
      if (payload === void 0) {
        throw new TypeError('Sealer/Unsealer mismatch');
      } else if (payload === undefinedStandin) {
        return void 0;
      } else {
        return payload;
      }
    }
    return snowWhite({
        seal: markFuncFreeze(seal),
        unseal: markFuncFreeze(unseal)
      });
  }

  /**
   * A call to cajaVM.manifest(data) is dynamically ignored, but if the
   * data expression is valid static JSON text, its value is made
   * statically available to the module loader.
   * <p>
   * TODO(erights): Find out if this is still the plan.
   */
  function manifest(ignored) {}

  ///////////////////////////////////////////////////////////////////
  // Specification
  ///////////////////////////////////////////////////////////////////

  /**
   * 4. Overview
   */

  /**
   * 4.2 Language Overview
   */

  /**
   * 8. Types
   */

  function Type(x) {
    switch (typeof x) {
      case 'undefined': { return 'Undefined'; }
      case 'boolean': { return 'Boolean'; }
      case 'string': { return 'String'; }
      case 'number': { return 'Number'; }
      default: { return x ? 'Object' : 'Null'; }
    }
  }

  /**
   * 8.6 Object type
   */

  // 8.6.1
  var attributeDefaults = {
      value: void 0,
      get: void 0,
      set: void 0,
      writable: false,
      enumerable: false,
      configurable: false
    };

  // 8.6.2
  function isPrimitive(x) {
    return Type(x) !== 'Object';
  }

  /**
   * 8.10 The Property Descriptor and Property Identifier Specification Types
   */

  // 8.10.1
  function IsAccessorDescriptor(Desc) {
    if (Desc === void 0) { return false; }
    if ('get' in Desc) { return true; }
    if ('set' in Desc) { return true; }
    return false;
  }

  // 8.10.2
  function IsDataDescriptor(Desc) {
    if (Desc === void 0) { return false; }
    if ('value' in Desc) { return true; }
    if ('writable' in Desc) { return true; }
    return false;
  }

  // 8.10.3
  function IsGenericDescriptor(Desc) {
    if (Desc === void 0) { return false; }
    if (!IsAccessorDescriptor(Desc) && !IsDataDescriptor(Desc)) {
      return true;
    }
    return false;
  }

  // 8.10.4
  // Converts an internal property descriptor to an external one.
  function FromPropertyDescriptor(Desc) {
    function copyProp(Desc, obj, name) {
      obj.DefineOwnProperty___(name, {
          value: Desc[name],
          writable: true,
          enumerable: true,
          configurable: true
        });
    }

    // 1. If Desc is undefined, then return undefined.
    if (Desc === void 0) { return void 0; }
    // 2. Let obj be the result of creating a new object
    //    as if by the expression new Object() where Object is the standard
    //    built-in constructor with that name.
    var obj = {};
    // 3. If IsDataDescriptor(Desc) is true, then
    if (IsDataDescriptor(Desc)) {
      // a. Call the [[DefineOwnProperty]] internal method of obj
      //    with arguments "value", Property Descriptor {
      //      [[Value]]:Desc.[[Value]],
      //      [[Writable]]: true,
      //      [[Enumerable]]: true,
      //      [[Configurable]]: true
      //    }, and false.
      copyProp(Desc, obj, 'value');
      // b. Call the [[DefineOwnProperty]] internal method of obj
      //    with arguments "writable", Property Descriptor {[[Value]]:
      //    Desc.[[Writable]], [[Writable]]: true, [[Enumerable]]:
      //    true, [[Configurable]]: true}, and false.
      copyProp(Desc, obj, 'writable');
    }
    // 4. Else, IsAccessorDescriptor(Desc) must be true, so
    else {
      // a. Call the [[DefineOwnProperty]] internal method of obj
      //    with arguments "get", Property Descriptor {[[Value]]:
      //    Desc.[[Get]], [[Writable]]: true, [[Enumerable]]: true,
      //    [[Configurable]]: true}, and false.
      copyProp(Desc, obj, 'get');
      // b. Call the [[DefineOwnProperty]] internal method of obj
      //    with arguments "set", Property Descriptor {[[Value]]:
      //    Desc.[[Set]], [[Writable]]: true, [[Enumerable]]: true,
      //    [[Configurable]]: true}, and false.
      copyProp(Desc, obj, 'set');
    }
    // 5. Call the [[DefineOwnProperty]] internal method of obj with
    //    arguments "enumerable", Property Descriptor {[[Value]]:
    //    Desc.[[Enumerable]], [[Writable]]: true, [[Enumerable]]: true,
    //    [[Configurable]]: true}, and false.
    copyProp(Desc, obj, 'enumerable');
    // 6. Call the [[DefineOwnProperty]] internal method of obj with
    //    arguments "configurable", Property Descriptor {[[Value]]:
    //    Desc.[[Configurable]], [[Writable]]: true, [[Enumerable]]:
    //    true, [[Configurable]]: true}, and false.
    copyProp(Desc, obj, 'configurable');
    // 7. Return obj.
    return obj;
  }

  // 8.10.5
  // Converts an external property descriptor to an internal one.
  function ToPropertyDescriptor(Obj) {
    // 1. If Type(Obj) is not Object throw a TypeError exception.
    if (Type(Obj) !== 'Object') {
      notObject(Obj);
    }
    // 2. Let desc be the result of creating a new Property
    //    Descriptor that initially has no fields.
    var desc = {};
    // 3. If the result of calling the [[HasProperty]]
    //    internal method of Obj with argument "enumerable" is true, then
    //   a. Let enum be the result of calling the [[Get]]
    //      internal method of Obj with "enumerable".
    //   b. Set the [[Enumerable]] field of desc to ToBoolean(enum).
    if (Obj.HasProperty___('enumerable')) {
      desc.enumerable = !!Obj.v___('enumerable');
    }
    // 4. If the result of calling the [[HasProperty]]
    //    internal method of Obj with argument "configurable" is true, then
    //   a. Let conf  be the result of calling the [[Get]]
    //      internal method of Obj with argument "configurable".
    //   b. Set the [[Configurable]] field of desc to ToBoolean(conf).
    if (Obj.HasProperty___('configurable')) {
      desc.configurable = !!Obj.v___('configurable');
    }
    // 5. If the result of calling the [[HasProperty]]
    //    internal method of Obj with argument "value" is true, then
    //   a. Let value be the result of calling the [[Get]]
    //      internal method of Obj with argument "value".
    //   b. Set the [[Value]] field of desc to value.
    if (Obj.HasProperty___('value')) {
      desc.value = Obj.v___('value');
    }
    // 6. If the result of calling the [[HasProperty]]
    //    internal method of Obj with argument "writable" is true, then
    // a. Let writable be the result of calling the [[Get]]
    //    internal method of Obj with argument "writable".
    // b. Set the [[Writable]] field of desc to ToBoolean(writable).
    if (Obj.HasProperty___('writable')) {
      desc.writable = !!Obj.v___('writable');
    }
    // 7. If the result of calling the [[HasProperty]]
    //    internal method of Obj with argument "get" is true, then
    if (Obj.HasProperty___('get')) {
      // a. Let getter be the result of calling the [[Get]]
      //    internal method of Obj with argument "get".
      var getter = Obj.v___('get');
      // b. If IsCallable(getter) is false and getter is not
      //    undefined, then throw a TypeError exception.
      if (!isFunction(getter) && getter !== void 0) {
        notFunction(getter, 'getter');
      }
      // c. Set the [[Get]] field of desc to getter.
      desc.get = asFirstClass(getter);
    }
    // 8. If the result of calling the [[HasProperty]]
    //    internal method of Obj with argument "set" is true, then
    if (Obj.HasProperty___('set')) {
      // a. Let setter be the result of calling the [[Get]]
      //    internal method of Obj with argument "set".
      var setter = Obj.v___('set');
      // b. If IsCallable(setter) is false and setter is not
      //    undefined, then throw a TypeError exception.
      if (!isFunction(setter) && setter !== void 0) {
        notFunction(setter, 'setter');
      }
      // c. Set the [[Set]] field of desc to setter.
      desc.set = asFirstClass(setter);
    }
    // 9. If either desc.[[Get]] or desc.[[Set]] are present, then
    if ('set' in desc || 'get' in desc) {
      // a. If either desc.[[Value]] or desc.[[Writable]] are present,
      //    then throw a TypeError exception.
      if ('value' in desc) {
        throw new TypeError('Property with get/set cannot have value.');
      }
      if ('writable' in desc) {
        throw new TypeError('Property with get/set cannot have writable.');
      }
    }
    // 10. Return desc.
    return desc;
  }

  /**
   * 8.12 Algorithms for Object Internal Methods
   */
  // 8.12.1
  // Returns internal property descriptor or undefined.
  Object.prototype.GetOwnProperty___ = function GetOwnProperty___(P) {
      var O = this;
      //inline if (isNumericName(P)) {
      if (typeof P === 'number' || ('' + (+P)) === P) {
        if (O.hasOwnProperty(P)) {
          return {
              value: O[P],
              writable: isWritable(O, 'NUM___'),
              configurable: isConfigurable(O, 'NUM___'),
              enumerable: true
            };
        } else {
          return void 0;
        }
      }
      P = '' + P;
      // inline assertValidPropertyName(P);
      if (endsWith__.test(P)) {
        throw new TypeError('Property name may not end in double underscore.');
      }

      // 1. If O doesn't have an own property with name P, return undefined.
      // Inline HasOwnProperty___.  Works with proxies because GetOwnProperty___
      // also gets overridden.
      if (!O.hasOwnProperty(P + '_v___')) {
        return void 0;
      }

      // 2. Let D be a newly created Property Descriptor with no fields.
      var D = {};
      // 3. Let X be O's own property named P.
      // 4. If X is a data property, then
      if (hasValue(O, P)) {
        // a. Set D.[[Value]] to the value of X's [[Value]] attribute.
        D.value = O[P];
        // b. Set D.[[Writable]] to the value of X's [[Writable]] attribute
        D.writable = isWritable(O, P);
      } else {
        // 5. Else X is an accessor property, so
        // a. Set D.[[Get]] to the value of X's [[Get]] attribute.
        D.get = getter(O, P);
        // b. Set D.[[Set]] to the value of X's [[Set]] attribute.
        D.set = setter(O, P);
      }
      // 6. Set D.[[Enumerable]] to the value of X's [[Enumerable]] attribute.
      D.enumerable = isEnumerable(O, P);
      // 7. Set D.[[Configurable]] to the value of X's
      // [[Configurable]] attribute.
      D.configurable = isConfigurable(O, P);
      // 8. Return D.
      return D;
    };

  // 8.12.3
  Object.prototype.v___ = function v___(P) {
      P = '' + P;
      if (isNumericName(P)) { return this[P]; }
      assertValidPropertyName(P);
      // Is P an accessor property on this?
      var g = getter(this, P);
      if (g) { return g.f___(this); }
      // Is it whitelisted as a value?
      if (hasValue(this, P)) { return this[P]; }
      return void 0;
    };

  // 8.12.5
  // This follows the philosophy that the ES5 spec was mistaken to prohibit
  // overriding read-only properties.  Chrome also follows this pholosophy.
  // http://wiki.ecmascript.org/doku.php?id=strawman:fixing_override_mistake
  Object.prototype.w___ = function w___(P, V) {
      var thisExtensible = isExtensible(this);
      P = '' + P;
      assertValidPropertyName(P);
      if (!thisExtensible) {
        if (wouldExtend(this, P)) {
          throw new TypeError("Could not create the property '" +
              P + "': " + this + " is not extensible.");
        }
      }

      // At this point, obj is either (extensible) or
      // (non-extensible but already has the property in question).

      V = asFirstClass(V);

      // Numeric names can't be emulated accessors.
      if(isNumericName(P)) {
        if (isWritable(this, 'NUM___')) {
          return this[P] = V;
        } else {
          throw new TypeError("The property '" + P + "' is not writable.");
        }
      }
      // Is P an accessor property on obj?
      if (this[P + '_v___'] === false) {
        var s = setter(this, P);
        if (s) { s.f___(this, [V]); return V; }
        throw new TypeError("The property '" + P + "' has no setter.");
      }

      // If P is an own data property,
      if (this[P + '_v___'] === this) {
        // and it's writable, then write;
        if (isWritable(this, P)) {
          fastpathWrite(this, P);
          return this[P] = V;
        }
        // otherwise throw.
        throw new TypeError("The property '" + P + "' is not writable.");
      }

      // At this point, the object is known to be extensible and not to have the
      // property whitelisted.  We need to check if the property exists but
      // is purposely not whitelisted.

      // If it doesn't exist,
      if (!this.hasOwnProperty(P)) {
        // then create it;
        this.DefineOwnProperty___(P, {
            value: V,
            writable: true,
            configurable: true,
            enumerable: true
          });
        return V;
      }
      // otherwise throw.
      throw new TypeError("The property '" + P + "' is not writable.");
    };

  // 8.12.6
  /**
   * Precondition: P is a number or string; this is to prevent testing
   * P and the string coersion having side effects.
   */
  Object.prototype.HasProperty___ = function (P) {
      if (isNumericName(P)) { return P in this; }
      return (P + '_v___' in this);
    };

  // Delete the own property P from O without refusing non-configurability.
  function rawDelete(O, P) {
    return delete O[P]
        && delete O[P + '_v___']
        && delete O[P + '_w___']
        && delete O[P + '_gw___']
        && delete O[P + '_g___']
        && delete O[P + '_s___']
        && delete O[P + '_c___']
        && delete O[P + '_e___']
        && delete O[P + '_m___'];
  }

  // 8.12.7
  Object.prototype.c___ = function (P) {
      var O = this;
      P = '' + P;
      // 1. Let desc be the result of calling the [[GetOwnProperty]]
      //    internal method of O with property name P.
      var desc = O.GetOwnProperty___(P);
      // 2. If desc is undefined, then return true.
      if (!desc) {
        return true;
      }
      // 3. If desc.[[Configurable]] is true, then
      if (desc.configurable) {
        if (isNumericName(P)) {
          if (isDeodorized(O, P)) {
            throw new TypeError("Cannot delete Firefox-specific antidote '"
                + P + "' on " + O);
          } else {
            delete O[P];
            return true;
          }
        }
        // a. Remove the own property with name P from O.
        rawDelete(O, P);
        // b. Return true.
        return true;
      }
      // 4. Else if Throw, then throw a TypeError exception.
      // [This is strict mode, so Throw is always true.]
      throw new TypeError("Cannot delete '" + P + "' on " + O);
      // 5. Return false.
    };

  // 8.12.9
  // Preconditions:
  //   Desc is an internal property descriptor.
  //   P is a valid property name.
  Object.prototype.DefineOwnProperty___ =
    function DefineOwnProperty___(P, Desc) {
      //inline if (isNumericName(P)) {
      if (typeof P === 'number' || ('' + (+P)) === P) {
        throw new TypeError('Cannot define numeric properties.');
      }
      var O = this;
      P = '' + P;
      // 1. Let current be the result of calling the [[GetOwnProperty]]
      //    internal method of O with property name P.
      var current = O.GetOwnProperty___(P);
      // 2. Let extensible be the value of the [[Extensible]] internal
      //    property of O.

      //inline var extensible = isExtensible(O);
      var extensible = Type(O) === 'Object' && O.ne___ !== O;
      // 3. If current is undefined and extensible is false, then Reject.
      if (!current && !extensible) {
        throw new TypeError('This object is not extensible.');
      }
      // 4. If current is undefined and extensible is true, then
      if (!current && extensible) {
        // a. If  IsGenericDescriptor(Desc) or IsDataDescriptor(Desc)
        //    is true, then
        if (IsDataDescriptor(Desc) || IsGenericDescriptor(Desc)) {
          // i. Create an own data property named P of object O whose
          //    [[Value]], [[Writable]], [[Enumerable]] and
          //    [[Configurable]] attribute values are described by
          //    Desc. If the value of an attribute field of Desc is
          //    absent, the attribute of the newly created property is
          //    set to its default value.
          O[P] = Desc.value;
          O[P + '_v___'] = O;
          O[P + '_w___'] = false;
          O[P + '_gw___'] = Desc.writable ? O : false;
          O[P + '_e___'] = Desc.enumerable ? O : false;
          O[P + '_c___'] = Desc.configurable ? O : false;
          O[P + '_g___'] = void 0;
          O[P + '_s___'] = void 0;
          O[P + '_m___'] = false;
        }
        // b. Else, Desc must be an accessor Property Descriptor so,
        else {
          // i. Create an own accessor property named P of object O
          //    whose [[Get]], [[Set]], [[Enumerable]] and
          //    [[Configurable]] attribute values are described by
          //    Desc. If the value of an attribute field of Desc is
          //    absent, the attribute of the newly created property is
          //    set to its default value.
          O[P] = Desc.configurable ? void 0 : O[P];
          O[P + '_v___'] = false;
          O[P + '_w___'] =  O[P + '_gw___'] = false;
          O[P + '_e___'] = Desc.enumerable ? O : false;
          O[P + '_c___'] = Desc.configurable ? O : false;
          O[P + '_g___'] = Desc.get;
          O[P + '_s___'] = Desc.set;
          O[P + '_m___'] = false;
        }
        // c. Return true.
        return true;
      }
      // 5. Return true, if every field in Desc is absent.
      if (!('value' in Desc ||
          'writable' in Desc ||
          'enumerable' in Desc ||
          'configurable' in Desc ||
          'get' in Desc ||
          'set' in Desc)) {
        return true;
      }
      // 6. Return true, if every field in Desc also occurs in current
      //    and the value of every field in Desc is the same value as the
      //    corresponding field in current when compared using the
      //    SameValue algorithm (9.12).
      var allHaveAppearedAndAreTheSame = true;
      for (var i in Desc) {
        if (!Desc.hasOwnProperty(i)) { continue; }
        if (!SameValue(current[i], Desc[i])) {
          allHaveAppearedAndAreTheSame = false;
          break;
        }
      }
      if (allHaveAppearedAndAreTheSame) { return true; }
      // 7. If the [[Configurable]] field of current is false then
      if (!current.configurable) {
        // a. Reject, if the [Configurable]] field of Desc is true.
        if (Desc.configurable) {
          throw new TypeError("The property '" + P +
              "' is not configurable.");
        }
        // b. Reject, if the [[Enumerable]] field of Desc is present
        //    and the [[Enumerable]] fields of current and Desc are
        //    the Boolean negation of each other.
        if ('enumerable' in Desc && Desc.enumerable !== current.enumerable) {
          throw new TypeError("The property '" + P +
              "' is not configurable.");
        }
      }
      var iddCurrent = IsDataDescriptor(current);
      var iddDesc = IsDataDescriptor(Desc);
      // 8. If IsGenericDescriptor(Desc) is true, then no further
      //    validation is required.
      if (IsGenericDescriptor(Desc)) {
        // Do nothing
      }
      // 9. Else, if IsDataDescriptor(current) and IsDataDescriptor(Desc)
      //    have different results, then
      else if (iddCurrent !== iddDesc) {
        // a. Reject, if the [[Configurable]] field of current is false.
        if (!current.configurable) {
          throw new TypeError("The property '" + P +
              "' is not configurable.");
        }
        // b. If IsDataDescriptor(current) is true, then
        if (iddCurrent) {
          // i. Convert the property named P of object O from a data
          //    property to an accessor property. Preserve the existing
          //    values of the converted property's [[Configurable]] and
          //    [[Enumerable]] attributes and set the rest of the
          //    property's attributes to their default values.
          O[P] = void 0;
          O[P + '_v___'] = false;
          O[P + '_w___'] =  O[P + '_gw___'] = false;
          // O[P + '_e___'] = O[P + '_e___'];
          // O[P + '_c___'] = O[P + '_c___'];
          O[P + '_g___'] = void 0;
          O[P + '_s___'] = void 0;
          O[P + '_m___'] = false;
        }
        // c. Else,
        else {
          // i. Convert the property named P of object O from an
          //    accessor property to a data property. Preserve the
          //    existing values of the converted property's
          //    [[Configurable]] and [[Enumerable]] attributes and set
          //    the rest of the property's attributes to their default
          //    values.
          O[P] = Desc.value;
          O[P + '_v___'] = O;
          O[P + '_w___'] = O[P + '_gw___'] = false;
          // O[P + '_e___'] = O[P + '_e___'];
          // O[P + '_c___'] = O[P + '_c___'];
          O[P + '_g___'] = void 0;
          O[P + '_s___'] = void 0;
          O[P + '_m___'] = false;
        }
      }
      // 10. Else, if IsDataDescriptor(current) and
      //     IsDataDescriptor(Desc) are both true, then
      else if (iddCurrent && iddDesc) {
        // a. If the [[Configurable]] field of current is false, then
        if (!current.configurable) {
          // i. Reject, if the [[Writable]] field of current is false
          //    and the [[Writable]] field of Desc is true.
          if (!current.writable && Desc.writable) {
            throw new TypeError("The property '" + P +
                "' is not configurable.");
          }
          // ii. If the [[Writable]] field of current is false, then
          if (!current.writable) {
            // 1. Reject, if the [[Value]] field of Desc is present and
            //    SameValue(Desc.[[Value]], current.[[Value]]) is false.
            if ('value' in Desc && !SameValue(Desc.value, current.value)) {
              throw new TypeError("The property '" + P +
                  "' is not writable.");
            }
          }
        }
        // b. else, the [[Configurable]] field of current is true, so
        //    any change is acceptable. (Skip to 12)
      }
      // 11. Else, IsAccessorDescriptor(current) and
      //     IsAccessorDescriptor(Desc) are both true so,
      else {
        // a. If the [[Configurable]] field of current is false, then
        if (!current.configurable) {
          // i. Reject, if the [[Set]] field of Desc is present and
          //    SameValue(Desc.[[Set]], current.[[Set]]) is false.
          // ii. Reject, if the [[Get]] field of Desc is present and
          //     SameValue(Desc.[[Get]], current.[[Get]]) is false.
          if (('set' in Desc && !SameValue(Desc.set, current.set)) ||
              ('get' in Desc && !SameValue(Desc.get, current.get))) {
            throw new TypeError("The property '" + P +
                "' is not configurable.");
          }
        }
      }
      // 12. For each attribute field of Desc that is present,
      //     set the correspondingly named attribute of the property
      //     named P of object O to the value of the field.
      if (iddDesc) {
        O[P] = Desc.value;
        O[P + '_v___'] = O;
        O[P + '_gw___'] = Desc.writable ? O : false;
        O[P + '_g___'] = O[P + '_s___'] = void 0;
      } else {
        // Create the property if it's not there so that JSON.stringify
        // can see the property.
        O[P] = O[P];
        O[P + '_v___'] = false;
        O[P + '_gw___'] = false;
        O[P + '_g___'] = Desc.get;
        O[P + '_s___'] = Desc.set;
      }
      O[P + '_e___'] = Desc.enumerable ? O : false;
      O[P + '_c___'] = Desc.configurable ? O : false;
      O[P + '_m___'] = false;
      O[P + '_w___'] = false;
      // 13. Return true.
      return true;
    };

  /**
   * 9 Type Conversion and Testing
   */

  // 9.6
  function ToUint32(input) {
    return input >>> 0;
  }

  // 9.9
  function ToObject(input) {
    if (input === void 0 || input === null) {
      throw new TypeError('Cannot convert ' + input + ' to Object.');
    }
    return Object(input);
  }

  // 9.12
  /**
   * Are x and y not observably distinguishable?
   */
  function SameValue(x, y) {
    if (x === y) {
      // 0 === -0, but they are not identical
      return x !== 0 || 1/x === 1/y;
    } else {
      // NaN !== NaN, but they are identical.
      // NaNs are the only non-reflexive value, i.e., if x !== x,
      // then x is a NaN.
      return x !== x && y !== y;
    }
  }


  /**
   * 11 Expressions
   */

  /**
   * Throws an exception if the value is an unmarked function.
   */
  function asFirstClass(value) {
    if (isFunction(value) && !value.ok___) {
      if (value === Function.prototype) {
        throw new Error('Cannot invoke Function.prototype.');
      }
      // webkit js debuggers rely on ambient Function.bind
      // http://code.google.com/p/chromium/issues/detail?id=145871
      if (!___.DISABLE_SECURITY_FOR_DEBUGGER) {
        throw new Error('Internal: toxic function encountered!\n' + value);
      }
      log('Internal: toxic function encountered!\n' + value);
    }
    return value;
  }

  // 11.1.5
  /**
   * Creates a well-formed ES5 record from a list of alternating
   * keys and values.
   */
  function initializeMap(list) {
    var result = {};
    var accessors = {};
    var hasAccessors = false;
    var i;
    for (i = 0; i < list.length; i += 2) {
      if (typeof list[i] === 'string') {
        if (result.hasOwnProperty(list[i])) {
          throw new SyntaxError('Duplicate keys: ' + list[i]);
        }
        if (isNumericName(list[i])) {
          result[list[i]] = asFirstClass(list[i + 1]);
        } else {
          result.DefineOwnProperty___(
              list[i],
              {
                value: asFirstClass(list[i + 1]),
                writable: true,
                enumerable: true,
                configurable: true
              });
        }
      } else {
        hasAccessors = true;
        var name = list[i][0];
        if (isNumericName(name)) {
          throw new TypeError('Accessors not supported for numerics.');
        }
        var type = list[i][1];
        accessors[name] = accessors[name] || {};
        if (accessors[name].hasOwnProperty(type)) {
          throw new SyntaxError('Duplicate accessor keys: ' +
              type + ' ' + list[i]);
        }
        accessors[name][type] = asFirstClass(list[i + 1]);
      }
    }
    if (hasAccessors) {
      for (i in accessors) {
        if (!accessors.hasOwnProperty(i)) { continue; }
        if (endsWith__.test(i)) { continue; }
        result.DefineOwnProperty___(i, {
            get: accessors[i].get,
            set: accessors[i].set,
            enumerable: true,
            configurable: true
          });
      }
    }
    return result;
  }

  // 11.2.3
  /**
   * Makes a [[ThrowTypeError]] function, as defined in section 13.2.3
   * of the ES5 spec.
   *
   * <p>The informal name for the [[ThrowTypeError]] function, defined
   * in section 13.2.3 of the ES5 spec, is the "poison pill". The poison
   * pill is simply a no-argument function that, when called, always
   * throws a TypeError. Since we wish this TypeError to carry useful
   * diagnostic info, we violate the ES5 spec by defining 4 poison
   * pills with 4 distinct identities.
   *
   * <p>A poison pill is installed as the getter & setter of the
   * de-jure (arguments.callee) and de-facto non-strict magic stack
   * inspection properties, which no longer work in ES5/strict, since
   * they violate encapsulation. Rather than simply remove them,
   * access to these properties is poisoned in order to catch errors
   * earlier when porting old non-strict code.
   */
  function makePoisonPill(badThing) {
    function poisonPill() {
      throw new TypeError('' + badThing + ' is forbidden by ES5/strict');
    }
    return markFunc(poisonPill);
  }
  var poisonFuncCaller = makePoisonPill("A function's .caller");
  var poisonFuncArgs = makePoisonPill("A function's .arguments");

  /**
   * Function calls g(args) get translated to g.f___(USELESS, args)
   * Tamed functions and cajoled functions install an overriding fastpath f___
   * to apply, the original Function.prototype.apply.
   */
  Function.prototype.f___ = callFault;
  Function.prototype.i___ = function(var_args) {
      return this.f___(USELESS, arguments);
    };
  Function.prototype.new___ = callFault;
  Function.prototype.DefineOwnProperty___('arguments', {
      enumerable: false,
      configurable: false,
      get: poisonFuncArgs,
      set: poisonFuncArgs
    });
  Function.prototype.DefineOwnProperty___('caller', {
      enumerable: false,
      configurable: false,
      get: poisonFuncCaller,
      set: poisonFuncCaller
    });

  // 11.2.4
  var poisonArgsCallee = makePoisonPill('arguments.callee');
  var poisonArgsCaller = makePoisonPill('arguments.caller');

  /**
   * Given either an array or an actual arguments object, return
   * ES53's emulation of an ES5/strict arguments object.
   */
  function args(original) {
    var result = initializeMap(['length', 0]);
    push.apply(result, original);
    result.CLASS___ = 'Arguments';
    result.DefineOwnProperty___(
        'callee',
        {
          enumerable: false,
          configurable: false,
          get: poisonArgsCallee,
          set: poisonArgsCallee
        });
    result.DefineOwnProperty___(
        'caller',
        {
          enumerable: false,
          configurable: false,
          get: poisonArgsCaller,
          set: poisonArgsCaller
        });
    return result;
  }

  // 11.8.7
  /**
   * Implements ES5's {@code <i>name</i> in <i>obj</i>}
   *
   * Precondition: name is a string
   */
  function isIn(name, obj) {
    if (Type(obj) !== 'Object') {
      notObject(obj);
    }
    return obj.HasProperty___(name);
  }

  /**
   * 15 Standard Built-in ECMAScript Objects
   */

  // Sets up a per-object getter and setter.  Necessary to prevent
  // guest code from messing with expectations of host and innocent code.
  // If innocent code needs access to the guest properties, explicitly tame
  // it that way.
  function virtualize(obj, name, fun) {
    var vname = name + '_virt___';
    obj[vname] = fun ? markFunc(fun) : obj[name] ? markFunc(obj[name]) : void 0;
    obj.DefineOwnProperty___(name, {
        get: markFunc(function () {
            return this[vname];
          }),
        set: markFunc(function (val) {
            if (!isFunction(val)) {
              notFunction(val);
            }
            if (isFrozen(this)) {
              throw new TypeError('This object is frozen.');
            }
            if (!isExtensible(this) &&
                !this.hasOwnProperty(vname)) {
              throw new TypeError('This object is not extensible.');
            }
            this[vname] = asFirstClass(val);
          }),
        enumerable: false,
        configurable: false
      });
  }

  // 15.1.3.1--4
  markFunc(decodeURI);
  markFunc(decodeURIComponent);
  markFunc(encodeURI);
  markFunc(encodeURIComponent);

  // 15.2.1.1
  Object.f___ = markFunc(function (dis, as) {
      var len = as.length;
      if (len === 0 || as[0] === null || as[0] === void 0) {
        return {};
      }
      return ToObject(as[0]);
    });

  // 15.2.2.1
  Object.new___ = markFunc(function (value) {
      return Object.f___(USELESS, [value]);
    });

  // 15.2.3.1
  Object.DefineOwnProperty___('prototype', {
      value: Object.prototype,
      writable: false,
      enumerable: false,
      configurable: false
    });

  // 15.2.3.2
  // Prefer the browser's built-in version.
  if (!Object.getPrototypeOf) {
    Object.getPrototypeOf = function (obj) {
        if (Type(obj) !== 'Object') {
          notObject(obj);
        }
        if (!obj.hasOwnProperty('Prototype___')) {
          // If there's no built-in version, fall back to __proto__.
          if ({}.__proto__ === Object.prototype) {
            obj.Prototype___ = obj.__proto__;
          } else {
            // If that fails, use directConstructor to give our best guess.
            var constr = directConstructor(obj);
            if (constr === BASE_OBJECT_CONSTRUCTOR) {
              obj.Prototype___ = obj.baseProto___;
            } else if (constr === void 0) {
              obj.Prototype___ = null;
            } else {
              obj.Prototype___ = constr.prototype;
            }
          }
        }
        return obj.Prototype___;
      };
  }
  var stdGetPrototypeOf = Object.getPrototypeOf;

  // The Chrome/Safari/Webkit debugger injects a script that expects to
  // call Object.getOwnPropertyNames and Object.getOwnPropertyDescriptor on
  // arbitrary objects from any frame.

  // The es53 implementations below only work on objects that inherit from
  // es53's Object, so we virtualize instead of overriding.

  // Note, the es5 spec says these should be configurable:true, but if we
  // do that, DefineOwnProperty___ will set the real value to undefined.

  // 15.2.3.3
  // This is the original implementation exposed to guest code,
  // which may change it.
  var stdGetOwnPropertyDescriptor = function (obj, P) {
      // 1. If Type(object) is not Object throw a TypeError exception.
      if (Type(obj) !== 'Object') {
        notObject(obj);
      }
      // 2. Let name be ToString(P).
      var name = '' + P;
      // 3. Let desc be the result of calling the [[GetOwnProperty]]
      //    internal method of obj with argument name.
      var desc = obj.GetOwnProperty___(name);
      // 4. Return the result of calling FromPropertyDescriptor(desc).
      return desc ? FromPropertyDescriptor(desc) : void 0;
    };
  virtualize(Object, 'getOwnPropertyDescriptor', stdGetOwnPropertyDescriptor);

  // 15.2.3.4
  virtualize(Object, 'getOwnPropertyNames',
      function (obj) { return obj.ownKeys___(); });

  // 15.2.3.5
  /**
   * Makes a new empty object that directly inherits from {@code proto}.
   */
  function beget(proto) {
    if (proto === null) {
      throw new TypeError('Cannot beget from null.');
    }
    if (proto === (void 0)) {
      throw new TypeError('Cannot beget from undefined.');
    }
    function F() {}
    F.prototype = proto;
    var result = new F();
    return result;
  }

  // The algorithm below doesn't care whether Properties is absent
  // or undefined, so we can simplify.
  Object.create = function (O, opt_Properties) {
      // 1. If Type(O) is not Object or Null throw a TypeError exception.
      // (ES3 doesn't support null prototypes.)
      if (Type(O) !== 'Object') {
        notObject(O);
      }
      // 2. Let obj be the result of creating a new object
      //    as if by the expression new Object() where Object
      //    is the standard built-in constructor with that name
      // 3. Set the [[Prototype]] internal property of obj to O.
      var obj = beget(O);
      // 4. If the argument Properties is present
      // and not undefined, add own properties to obj
      // as if by calling the standard built-in function
      // Object.defineProperties with arguments obj and Properties.
      if (opt_Properties !== void 0) {
        DefineProperties(obj, opt_Properties);
      }
      // 5. Return obj.
      return obj;
    };

  // 15.2.3.6
  Object.defineProperty = function (O, P, Attributes) {
      // 1. If Type(O) is not Object throw a TypeError exception.
      if (Type(O) !== 'Object') {
        notObject(O);
      }
      // 2. Let name be ToString(P).
      var name = '' + P;
      // 3. Let desc be the result of calling
      //    ToPropertyDescriptor with Attributes as the argument.
      var desc = ToPropertyDescriptor(Attributes);
      // 4. Call the [[DefineOwnProperty]] internal method of O
      //    with arguments name, desc, and true.
      // (We don't need 'true' because we always throw in strict mode.)
      O.DefineOwnProperty___(name, desc);
      // 5. Return O.
      return O;
    };

  // 15.2.3.7
  function DefineProperties(O, Properties) {
    // 1. If Type(O) is not Object throw a TypeError exception.
    if (Type(O) !== 'Object') {
      notObject(O);
    }
    // 2. Let props be ToObject(Properties).
    var props = ToObject(Properties);
    // 3. Let names be an internal list containing
    //    the names of each enumerable own property of props.
    var names = ownEnumKeys(props);
    // 4. Let descriptors be an empty internal List.
    var descriptors = [];
    // 5. For each element P of names in list order,
    var len = names.length;
    var i, P, desc;
    for (i = 0; i < len; ++i) {
      P = names[i];
      // a. Let descObj be the result of calling the [[Get]]
      //    internal method of props with P as the argument.
      var descObj = props.v___(P);
      // b. Let desc be the result of calling ToPropertyDescriptor
      //    with descObj as the argument.
      desc = ToPropertyDescriptor(descObj);
      // c. Append desc to the end of descriptors.
      descriptors.push(desc);
    }
    // 6. For each element desc of descriptors in list order,
      // a. Call the [[DefineOwnProperty]] internal method
      //    of O with arguments P, desc, and true.
    // This part of the spec is nonsense.  I'm following Besen's
    // interpretation: see line 31479 of
    // http://besen.svn.sourceforge.net/viewvc/besen/trunk/src/BESEN.pas?revision=27&view=markup

    // TODO: The latest draft errata fixes this. We'll be ratifying
    // these errata at the upcoming EcmaScript meeting on 7/28 &
    // 7/29. Watch this space.
    for (i = 0; i < len; ++i) {
      P = names[i];
      desc = descriptors[i];
      O.DefineOwnProperty___(P, desc);
    }
    // 7. Return O.
    return O;
  }
  Object.defineProperties = DefineProperties;

  // 15.2.3.8
  Object.seal = function (O) {
      // 1. If Type(O) is not Object throw a TypeError exception.
      if (Type(O) !== 'Object') {
        notObject(O);
      }
      return O.seal___();
    };

  Object.prototype.seal___ = function () {
      // Allow function instances to install their instance properties
      // before sealing them.
      if (this.v___ === deferredV) {
        this.v___('length');
      }
      // 2. For each own property name P of this,
      for (var i in this) {
        if (!this.hasOwnProperty(i)) { continue; }
        if (isNumericName(i)) { continue; }
        var m = i.match(endsWith_v___);
        if (!m) { continue; }
        var P = m[1];
        // a. Let desc be the result of calling the [[GetOwnProperty]]
        //    internal method of this with P.
        // b. If desc.[[Configurable]] is true, set
        //    desc.[[Configurable]] to false.
        // c. Call the [[DefineOwnProperty]] internal method of this with P,
        //    desc, and true as arguments.
        this[P + '_c___'] = false;
      }
      if (!this.hasNumerics___()) {
        this.NUM____v___ = this;
        this.NUM____gw___ = this;
        this.NUM____w___ = this;
        this.NUM____m___ = false;
        this.NUM____e___ = this;
        this.NUM____g___ = void 0;
        this.NUM____s___ = void 0;
      }
      this.NUM____c___ = false;
      // 3. Set the [[Extensible]] internal property of this to false.
      this.ne___ = this;
      // 4. Return this.
      return this;
    };

  // 15.2.3.9
  /**
   * Whitelists all the object's own properties that do not
   * end in __ and have not already been whitelisted.
   * If opt_deep is true, recurses on objects and
   * assumes the object has no cycles through accessible keys.
   */
  function whitelistAll(obj, opt_deep) {
    var keys = fastOwnKeys(obj);
    for (var k = 0, n = keys.length; k < n; k++) {
      var i = keys[k], val = obj[i];
      if (i.substr(i.length - 2) !== '__' && !((i + '_v___') in obj)) {
        if (opt_deep && val && typeof val === 'object') {
          whitelistAll(val, true);
        }
        obj[i + '_v___'] = obj;
        obj[i + '_w___'] = false;
        obj[i + '_gw___'] = false;
        obj[i + '_e___'] = obj;
        obj[i + '_c___'] = false;
        obj[i + '_g___'] = void 0;
        obj[i + '_s___'] = void 0;
        obj[i + '_m___'] = false;
        if (val && val.f___ === Function.prototype.f___) {
          // inline isFunction(val)
          if (classProp.call(val) === '[object Function]') {
            markFunc(val);
          }
        } else {
          // inline isFunction(val)
          if (classProp.call(val) === '[object Function]') {
            val.ok___ = true;
          }
        }
      }
    }
    return obj;
  }

  // TODO: Where this is used, do we really want frozenness
  // that is transitive across property traversals?
  function snowWhite(obj) {
    return freeze(whitelistAll(obj));
  }

  function makeDefensibleFunction(f) {
    return markFuncFreeze(function(_) {
      return f.apply(USELESS, arguments);
    });
  }

  function makeDefensibleObject(descriptors) {
    var td = {};
    for (var k in descriptors) {
      if (!descriptors.hasOwnProperty(k)) { continue; }
      if (isNumericName(k)) { throw 'Cannot define numeric property: ' + k; }
      td.DefineOwnProperty___(k, {
        value: FromPropertyDescriptor(descriptors[k]),
        enumerable: true,
        writable: false
      });
    }
    return Object.seal(Object.create(Object.prototype, td));
  }

  function freeze(obj) {
      if (Type(obj) !== 'Object') {
        notObject(obj);
      }
      return obj.freeze___();
  }
  Object.freeze = freeze;

  // 15.2.3.10
  Object.preventExtensions = function (obj) {
      if (Type(obj) !== 'Object') {
        notObject(obj);
      }
      return obj.pe___();
    };

  Object.prototype.pe___ = function () {
      if (!this.hasNumerics___()) {
        this.NUM____v___ = this;
        this.NUM____e___ = this;
        this.NUM____g___ = void 0;
        this.NUM____s___ = void 0;
        this.NUM____c___ = this;
        this.NUM____gw___ = this;
        this.NUM____w___ = this;
        this.NUM____m___ = false;
      }
      this.ne___ = this;
      return this;
    };

  // 15.2.3.11
  Object.isSealed = function (O) {
      // 1. If Type(O) is not Object throw a TypeError exception.
      if (Type(O) !== 'Object') {
        notObject(O);
      }
      // 2. For each named own property name P of O,
      // a. Let desc be the result of calling the [[GetOwnProperty]]
      //    internal method of O with P.
      // b. If desc.[[Configurable]] is true, then return false.
      for (var i in O) {
        if (endsWith__.test(i)) { continue; }
        if (!O.hasOwnProperty(i)) { continue; }
        if (isNumericName(i)) { continue; }
        if (O[i + '_c___']) { return false; }
      }
      // 3. If the [[Extensible]] internal property of O is false, then
      //    return true.
      if (O.ne___ === O) { return true; }
      // 4. Otherwise, return false.
      return false;
    };

  // 15.2.3.12
  Object.isFrozen = isFrozen;

  // 15.2.3.13
  Object.isExtensible = isExtensible;

  // 15.2.3.14
  // virtualized to avoid confusing the webkit/safari/chrome debugger
  virtualize(Object, 'keys', ownEnumKeys);
  // TODO(felix8a): ES5 says this should be configurable: true

  (function () {
    var objectStaticMethods = [
        'getPrototypeOf',
        // getOwnPropertyDescriptor is virtual
        // getOwnPropertyNames is virtual
        'create',
        'defineProperty',
        'defineProperties',
        'seal',
        'freeze',
        'preventExtensions',
        'isSealed',
        'isFrozen',
        'isExtensible'
        // keys is virtual
      ];
    var i, len = objectStaticMethods.length;
    for (i = 0; i < len; ++i) {
      var name = objectStaticMethods[i];
      Object.DefineOwnProperty___(name, {
          value: markFunc(Object[name]),
          writable: true,
          enumerable: false,
          configurable: true
        });
    }
  })();

  // 15.2.4.1
  Object.DefineOwnProperty___('constructor', {
      value: Object,
      writable: false,
      enumerable: false,
      configurable: false
    });

  // 15.2.4.2
  Object.prototype.toString = markFunc(function() {
      if (this.CLASS___) { return '[object ' + this.CLASS___ + ']'; }
      return classProp.call(this);
    });
  Object.prototype.DefineOwnProperty___('toString', {
      get: markFunc(function () {
        return this.toString.orig___ ? this.toString.orig___ : this.toString;
      }),
      set: markFunc(function (val) {
        if (!isFunction(val)) {
          notFunction(val);
        }
        if (isFrozen(this)) {
          throw new TypeError("Won't set toString on a frozen object.");
        }
        val = asFirstClass(val);
        this.toString = markFunc(function (var_args) {
            return val.f___(safeDis(this), arguments);
          });
        this.toString.orig___ = val;
      }),
      enumerable: false,
      configurable: false
    });

  // 15.2.4.4
  markFunc(Object.prototype.valueOf);
  Object.prototype.DefineOwnProperty___('valueOf', {
      get: markFunc(function () {
          return this.valueOf.orig___ ? this.valueOf.orig___ : this.valueOf;
        }),
      set: markFunc(function (val) {
          if (!isFunction(val)) {
            notFunction(val);
          }
          if (isFrozen(this)) {
            throw new TypeError("Won't set valueOf on a frozen object.");
          }
          val = asFirstClass(val);
          this.valueOf = markFunc(function (var_args) {
              return val.f___(safeDis(this), arguments);
            });
          this.valueOf.orig___ = val;
        }),
      enumerable: false,
      configurable: false
    });

  // 15.2.4.5
  virtualize(Object.prototype, 'hasOwnProperty', function (P) {
      if (isNumericName(P)) { return this.hasOwnProperty(P); }
      return this.HasOwnProperty___(P);
    });

  // 15.2.4.7
  virtualize(Object.prototype, 'propertyIsEnumerable', function (V) {
      return isEnumerable(this, '' + V);
    });

  // 15.2.4.3, 5--7
  virtualize(Object.prototype, 'toLocaleString');
  virtualize(Object.prototype, 'isPrototypeOf');

  // 15.2.4
  // NOT extensible under ES5/3
  freeze(Object.prototype);

  // 15.3 Function
  var FakeFunction = function () {
      throw new
          Error('Internal: FakeFunction should not be directly invocable.');
    };

  FakeFunction.toString = (function (str) {
      return markFunc(function () {
          return str;
        });
    })(Function.toString());

  // 15.3.1
  Function.f___ = FakeFunction.f___ = markFunc(function() {
      throw new Error('Invoking the Function constructor is unsupported.');
    });

  // 15.3.2
  Function.new___ = FakeFunction.new___ = markFunc(function () {
      throw new Error('Constructing functions dynamically is unsupported.');
    });

  // 15.3.3.1
  FakeFunction.DefineOwnProperty___('prototype', {
      value: Function.prototype,
      writable: false,
      enumerable: false,
      configurable: false
    });

  // 15.3.4.1
  Function.prototype.DefineOwnProperty___('constructor', {
      value: FakeFunction,
      writable: true,
      enumerable: false,
      configurable: false
    });

  // 15.3.4.2
  (function () {
    var orig = Function.prototype.toString;
    Function.prototype.toString = markFunc(function () {
        if (this.toString___) { return this.toString___(); };
        return orig.call(this);
      });
    })();

  // 15.3.4.3--5
  virtualize(Function.prototype, 'call', function (dis, var_args) {
      return this.apply(safeDis(dis), slice.call(arguments, 1));
    });
  virtualize(Function.prototype, 'apply', function (dis, as) {
      // The arguments object exposed to guest code is neither a real
      // arguments object nor an array, so if they call apply on that,
      // we have to turn it into an array.
      return this.apply(safeDis(dis), as ? slice.call(as, 0) : void 0);
    });
  /**
   * Bind this function to <tt>self</tt>, which will serve
   * as the value of <tt>this</tt> during invocation. Curry on a
   * partial set of arguments in <tt>var_args</tt>. Return the curried
   * result as a new function object.
   */
  Function.prototype.bind = function(self, var_args) {
      var thisFunc = safeDis(this);
      var leftArgs = slice.call(arguments, 1);
      function funcBound(var_args) {
        var args = leftArgs.concat(slice.call(arguments, 0));
        return thisFunc.apply(safeDis(self), args);
      }
      markFunc(funcBound);
      funcBound.f___ = funcBound.apply;
      funcBound.ok___ = true;
      // 15.3.5.2
      rawDelete(funcBound, 'prototype');
      funcBound.new___ = function () {
          throw "Constructing the result of a bind() not yet implemented.";
        };
      return funcBound;
    };
  virtualize(Function.prototype, 'bind');

  // 15.4 Array

  // 15.4.1--2
  markFunc(Array);

  // 15.4.3.1
  Array.DefineOwnProperty___('prototype', {
      value: Array.prototype,
      writable: false,
      enumerable: false,
      configurable: false
    });

  // 15.4.3.2
  Array.isArray = markFunc(isArray);
  Array.DefineOwnProperty___('isArray', {
      value: Array.isArray,
      writable: true,
      enumerable: false,
      configurable: true
    });

  // 15.4.4.1
  Array.prototype.DefineOwnProperty___('constructor', {
      value: Array,
      writable: true,
      enumerable: false,
      configurable: false
    });

  // 15.4.4.2
  markFunc(Array.prototype.toString);

  // 15.4.4.3--6
  (function () {
    var methods = [
        'toLocaleString',
        'concat',
        'join',
        'pop'
      ];
    for (var i = 0, len = methods.length; i < len; ++i) {
      virtualize(Array.prototype, methods[i]);
    }
  })();

  // 15.4.4.7--9

  // Array generics can add a length property; static accesses are
  // whitelisted by the cajoler, but dynamic ones need this.
  function whitelistLengthIfItExists(dis) {
    if (('length' in dis) && !('length_v___' in dis)) {
      dis.DefineOwnProperty___('length', {
          value: dis.length,
          writable: true,
          configurable: true,
          enumerable: true
        });
    }
  }

  function guardedVirtualize(obj, name) {
    var orig = obj[name];
    virtualize(obj, name, function (var_args) {
        if (!isExtensible(this)) {
          throw new TypeError("This object is not extensible.");
        }
        var dis = safeDis(this);
        var result = orig.apply(dis, arguments);
        whitelistLengthIfItExists(dis);
        return result;
      });
  }

  (function () {
    // Reverse can create own numeric properties.
    var methods = [
        'push',
        'reverse',
        'shift',
        'splice',
        'unshift'
      ];
    for (var i = 0, len = methods.length; i < len; ++i) {
      guardedVirtualize(Array.prototype, methods[i]);
    }
  })();

  // 15.4.4.10
  virtualize(Array.prototype, 'slice');

  // 15.4.4.11
  virtualize(Array.prototype, 'sort', function (comparefn) {
      // This taming assumes that sort only modifies {@code this},
      // even though it may read numeric properties on the prototype chain.
      if (!isWritable(this, 'NUM___')) {
        throw new TypeError(
            'Cannot sort an object whose ' +
            'numeric properties are not writable.');
      }
      if (!isExtensible(this)) {
        throw new TypeError(
            'Cannot sort an object that is not extensible.');
      }
      var result = (comparefn ?
          sort.call(
              this,
              markFunc(function(var_args){
                return comparefn.f___(this, arguments);
              })
            ) :
          sort.call(this));
      whitelistLengthIfItExists(this);
      return result;
    });

  // 15.4.4.14
  Array.prototype.indexOf = markFunc(function (value, fromIndex) {
      // length is always readable
      var len = this.length >>> 0;
      if (!len) { return -1; }
      var i = fromIndex || 0;
      if (i >= len) { return -1; }
      if (i < 0) { i += len; }
      for (; i < len; i++) {
        if (!this.hasOwnProperty(i)) {
          continue;
        }
        // Numerics are always readable
        if (value === this[i]) { return i; }
      }
      return -1;
    });
  virtualize(Array.prototype, 'indexOf');

  // 15.4.4.15
  Array.prototype.lastIndexOf = function (value, fromIndex) {
      // length is always readable
      var len = this.length;
      if (!len) { return -1; }
      var i = arguments[1] || len;
      if (i < 0) { i += len; }
      i = min(i, len - 1);
      for (; i >= 0; i--) {
        if (!this.hasOwnProperty(i)) {
          continue;
        }
        if (value === this[i]) { return i; }
      }
      return -1;
    };
  virtualize(Array.prototype, 'lastIndexOf');

  // For protecting methods that use the map-reduce API against
  // inner hull breaches. For example, we don't want cajoled code
  // to be able to use {@code every} to invoke a toxic function as
  // a filter, for instance.
  //
  // {@code fun} must not be marked as callable.
  // {@code fun} expects
  // - a function {@code block} to use (like the filter in {@code every})
  // - an optional object {@code thisp} to use as {@code this}
  // It wraps {@code block} in a function that invokes its taming.
  function createOrWrap(obj, name, fun) {
    virtualize(obj, name);
    var vname = name + '_virt___';
    if (!obj[name]) {
      // Create
      obj[vname] = fun;
    } else {
      // Wrap
      obj[vname] = (function (orig) {
          return function (block) { //, thisp
              // We have to create a copy of arguments because
              // modifying arguments[0] breaks lexical scoping.
              var a = slice.call(arguments, 0);
              // Replace block with the taming of block
              a[0] = markFunc(function(var_args) {
                  return block.f___(this, arguments);
                });
              // Invoke the original function on the tamed
              // {@code block} and optional {@code thisp}.
              return orig.apply(this, a);
            };
        })(obj[name]);
    }
    markFunc(obj[vname]);
  }

  // 15.4.4.16
  createOrWrap(Array.prototype, 'every', function (block, thisp) {
      var len = this.length >>> 0;
      for (var i = 0; i < len; i++) {
        if (!block.f___(thisp, [this[i]])) { return false; }
      }
      return true;
    });

  // 15.4.4.17
  createOrWrap(Array.prototype, 'some', function (block, thisp) {
      var len = this.length >>> 0;
      for (var i = 0; i < this.length; i++) {
        if (block.f___(thisp, [this[i]])) { return true; }
      }
      return false;
    });

  // 15.4.4.18
  virtualize(Array.prototype, 'forEach', function (block, thisp) {
      var len = this.length >>> 0;
      for (var i = 0; i < len; i++) {
        if (i in this) {
          block.f___(thisp, [this[i], i, this]);
        }
      }
    });

  // 15.4.4.19
  // https://developer.mozilla.org/en/Core_JavaScript_1.5_Reference/Objects/Array/map
  createOrWrap(Array.prototype, 'map', function (fun, thisp) {
      var len = this.length >>> 0;
      if (!isFunction(fun)) {
        notFunction(fun);
      }
      var res = new Array(len);
      for (var i = 0; i < len; i++) {
        if (i in this) {
          res[i] = fun.f___(thisp, [this[i], i, this]);
        }
      }
      return res;
    });

  // 15.4.4.20
  createOrWrap(Array.prototype, 'filter', function (block, thisp) {
      var values = [];
      var len = this.length >>> 0;
      for (var i = 0; i < len; i++) {
        if (block.f___(thisp, [this[i]])) {
          values.push(this[i]);
        }
      }
      return values;
    });

  // 15.4.4.21
  // https://developer.mozilla.org/en/Core_JavaScript_1.5_Reference/Objects/Array/reduce
  createOrWrap(Array.prototype, 'reduce', function(fun) { // , initial
      // {@code fun} is of the form
      // function(previousValue, currentValue, index, array) { ... }
      var len = this.length >>> 0;
      if (!isFunction(fun)) {
        notFunction(fun);
      }
      // no value to return if no initial value and an empty array
      if (len === 0 && arguments.length === 1) {
        throw new TypeError('Expected an initial value or a non-empty array.');
      }
      var i = 0;
      var rv = void 0;
      if (arguments.length >= 2) {
        rv = arguments[1];
      } else {
        do {
          if (i in this) {
            rv = this[i++];
            break;
          }
          // if array contains no values, no initial value to return
          if (++i >= len) {
            throw new TypeError('Expected non-empty array.');
          }
        } while (true);
      }
      for (; i < len; i++) {
        if (i in this) {
          rv = fun.f___(USELESS, [rv, this[i], i, this]);
        }
      }
      return rv;
    });

  // 15.4.4.22
  // https://developer.mozilla.org/en/Core_JavaScript_1.5_Reference/Objects/Array/reduceRight
  createOrWrap(Array.prototype, 'reduceRight', function(fun) { // , initial
      var len = this.length >>> 0;
      if (!isFunction(fun)) {
        notFunction(fun);
      }
      // no value to return if no initial value, empty array
      if (len === 0 && arguments.length === 1) {
        throw new TypeError('Expected an initial value or a non-empty array.');
      }
      var i = len - 1;
      var rv = void 0;
      if (arguments.length >= 2) {
        rv = arguments[1];
      } else {
        do {
          if (i in this) {
            rv = this[i--];
            break;
          }
          // if array contains no values, no initial value to return
          if (--i < 0) {
            throw new TypeError('Expected a non-empty array.');
          }
        } while (true);
      }
      for (; i >= 0; i--) {
        if (i in this) {
          rv = fun.f___(USELESS, [rv, this[i], i, this]);
        }
      }
      return rv;
    });

  // 15.5 String

  // 15.5.1--2
  markFunc(String);

  // 15.5.3.1
  String.DefineOwnProperty___('prototype', {
      value: String.prototype,
      writable: false,
      enumerable: false,
      configurable: false
    });

  // 15.5.3.2
  virtualize(String, 'fromCharCode');

  // 15.5.4.1
  String.prototype.DefineOwnProperty___('constructor', {
      value: String.prototype.constructor,
      writable: true,
      enumerable: false,
      configurable: false
    });

  // 15.5.4.2
  markFunc(String.prototype.toString);

  // 15.5.4.3
  markFunc(String.prototype.valueOf);

  // 15.5.4.4--9, 13, 15--20
  // and the nonstandard but universally implemented substr.
  (function () {
    var methods = [
        'charAt',
        'charCodeAt',
        'concat',
        'indexOf',
        'lastIndexOf',
        'localeCompare',
        'slice',
        'substring',
        'toLowerCase',
        'toLocaleLowerCase',
        'toUpperCase',
        'toLocaleUpperCase',
        'substr'
      ];
    var i, len = methods.length;
    for (i = 0; i < len; ++i) {
      virtualize(String.prototype, methods[i]);
    }
  })();

  // 15.5.4.10, 12, 14
  /**
   * Verifies that regexp is something that can appear as a
   * parameter to a Javascript method that would use it in a match.
   * <p>
   * If it is a RegExp, then this match might mutate it, which must
   * not be allowed if regexp is frozen.
   *
   * Returns: a boolean indicating whether {@code regexp} should be
   * cast to a String
   */
  function enforceMatchable(regexp) {
    if (regexp instanceof RegExp) {
      if (isFrozen(regexp)) {
        throw new Error("Can't match with frozen RegExp: " + regexp);
      }
      return false;
    }
    return true;
  }

  function tameStringRegExp(orig) {
    return markFunc(function (regexp) {
        var cast = enforceMatchable(regexp);
        return orig.call(this, cast ? ('' + regexp) : regexp);
      });
  }

  (function () {
    var methods = [
        'match',
        'search',
        'split'
      ];
    for (var i = 0, len = methods.length; i < len; ++i) {
      virtualize(
          String.prototype,
          methods[i],
          tameStringRegExp(String.prototype[methods[i]]));
    }
  })();

  // 15.5.4.11
  virtualize(String.prototype, 'replace', function (searcher, replacement) {
      var cast = enforceMatchable(searcher);
      if (isFunction(replacement)) {
        replacement = asFirstClass(replacement);
      } else {
        replacement = '' + replacement;
      }
      return replace.call(
          this,
          cast ? ('' + searcher) : searcher,
          replacement
        );
    });

  // 15.5.4.20
  // http://blog.stevenlevithan.com/archives/faster-trim-javascript
  var trimBeginRegexp = /^\s\s*/;
  var trimEndRegexp = /\s\s*$/;
  virtualize(String.prototype, 'trim', function () {
      return ('' + this).
          replace(trimBeginRegexp, '').
          replace(trimEndRegexp, '');
    });

  // 15.6 Boolean

  // 15.6.1--2
  markFunc(Boolean);

  // 15.6.3.1
  Boolean.DefineOwnProperty___('prototype', {
      value: Boolean.prototype,
      writable: false,
      enumerable: false,
      configurable: false
    });

  // 15.6.4.1
  Boolean.prototype.DefineOwnProperty___('constructor', {
      value: Boolean.prototype.constructor,
      writable: true,
      enumerable: false,
      configurable: false
    });

    // 15.6.4.2
    markFunc(Boolean.prototype.toString);

    // 15.6.4.3
    markFunc(Boolean.prototype.valueOf);

  // 15.7 Number

  // 15.7.1--2
  markFunc(Number);

  // 15.7.3.1--6
  (function () {
    var props = [
        'prototype',
        'MAX_VALUE',
        'MIN_VALUE',
        // 'NaN' is automatically readable since it's a numeric property
        'NEGATIVE_INFINITY',
        'POSITIVE_INFINITY'
      ];
    var i, len = props.length;
    for (i = 0; i < len; ++i) {
      Number.DefineOwnProperty___(props[i], {
          value: Number[props[i]],
          writable: false,
          enumerable: false,
          configurable: false
        });
    }
  })();

  // 15.7.4.1
  Number.prototype.DefineOwnProperty___('constructor', {
      value: Number.prototype.constructor,
      writable: true,
      enumerable: false,
      configurable: false
    });

  // 15.7.4.2
  markFunc(Number.prototype.toString);

  // 15.7.4.4
  markFunc(Number.prototype.valueOf);

  // 15.7.4.3, 5--7
  (function (){
    var methods = [
        'toLocaleString',
        'toFixed',
        'toExponential',
        'toPrecision'
      ];
    var i, len = methods.length;
    for (i = 0; i < len; ++i) {
      virtualize(Number.prototype, methods[i]);
    }
  })();

  // 15.8 Math

  // 15.8.1.1--8
  (function (){
    var props = [
        'E',
        'LN10',
        'LN2',
        'LOG2E',
        'LOG10E',
        'PI',
        'SQRT1_2',
        'SQRT2'
      ];
    var i, len = props.length;
    for (i = 0; i < len; ++i) {
      Math.DefineOwnProperty___(props[i], {
          value: Math[props[i]],
          writable: false,
          enumerable: false,
          configurable: false
        });
    }
  })();

  // 15.8.2.1--18
  (function (){
    var methods = [
        'abs',
        'acos',
        'asin',
        'atan',
        'atan2',
        'ceil',
        'cos',
        'exp',
        'floor',
        'log',
        'max',
        'min',
        'pow',
        'random',
        'round',
        'sin',
        'sqrt',
        'tan'
      ];
    var i, len = methods.length;
    for (i = 0; i < len; ++i) {
      virtualize(Math, methods[i]);
    }
  })();

  // 15.9 Date

  // 15.9.1--3
  markFunc(Date);

  // 15.9.4.1
  Date.DefineOwnProperty___('prototype', {
      value: Date.prototype,
      writable: false,
      enumerable: false,
      configurable: false
    });

  // 15.9.4.2--4
  (function () {
    var staticMethods = [
        'parse',
        'UTC',
        'now'
      ];
    var i, len = staticMethods.length;
    for (i = 0; i < len; ++i) {
      virtualize(Date, staticMethods[i]);
    }
  })();

  // 15.9.5.1
  Date.prototype.DefineOwnProperty___('constructor', {
      value: Date.prototype.constructor,
      writable: true,
      enumerable: false,
      configurable: false
    });

  // 15.9.5.2
  markFunc(Date.prototype.toString);

  // 15.9.5.8
  markFunc(Date.prototype.valueOf);

  // 15.9.5.3--7, 9--44
  (function () {
    var methods = [
        'toDateString',
        'toTimeString',
        'toLocaleString',
        'toLocaleDateString',
        'toLocaleTimeString',
        'getTime',
        'getFullYear',
        'getMonth',
        'getDate',
        'getDay',
        'getHours',
        'getMinutes',
        'getSeconds',
        'getUTCSeconds',
        'getUTCMinutes',
        'getUTCHours',
        'getUTCDay',
        'getUTCDate',
        'getUTCMonth',
        'getUTCFullYear',
        'getMilliseconds',
        'getTimezoneOffset',
        'toUTCString',
        'toISOString',
        'toJSON'
      ];
    for (var i = 0; i < methods.length; ++i) {
      virtualize(Date.prototype, methods[i]);
    }
  })();

  (function () {
    var methods = [
        'setTime',
        'setMilliseconds',
        'setUTCMilliseconds',
        'setSeconds',
        'setUTCSeconds',
        'setMinutes',
        'setUTCMinutes',
        'setHours',
        'setUTCHours',
        'setDate',
        'setUTCDate',
        'setMonth',
        'setUTCMonth',
        'setFullYear',
        'setUTCFullYear'
      ];
    var proto = Date.prototype;
    for (var i = 0; i < methods.length; ++i) {
      virtualize(proto, methods[i], (function (method) {
        return function (varArgs) {
            if (this === proto) {
              throw new TypeError(
                  'Cannot change the internal state of Date.prototype');
            }
            if (!(this instanceof Date)) {
              throw new TypeError(
                  'Cannot call this method on a non-Date or ' +
                  'a cross-frame Date object.');
            }
            return method.apply(this, arguments);
        };
      })(proto[methods[i]]));
    }
  })();

  // 15.10 RegExp

  // 15.10.5
  RegExp.f___ = markFunc(function (dis___, as) {
      var pattern = as[0], flags = as[1];
      if (classProp.call(pattern) === '[object RegExp]'
          && flags === void 0) {
        return pattern;
      }
      switch (as.length) {
        case 0:
          return new RegExp.new___();
        case 1:
          return new RegExp.new___(pattern);
        default:
          return new RegExp.new___(pattern, flags);
      }
    });

  RegExp.new___ = markFunc(function (pattern, flags){
      var re;
      switch (arguments.length) {
        case 0:
          re = new RegExp();
          break;
        case 1:
          re = new RegExp(pattern);
          break;
        default:
          re = new RegExp(pattern, flags);
      }
      var instanceProps = [
          'source',
          'global',
          'ignoreCase',
          'multiline'
        ];
      for (var i = 0; i < instanceProps.length; ++i) {
        re.DefineOwnProperty___(instanceProps[i], {
            value: re[instanceProps[i]],
            writable: false,
            enumerable: false,
            configurable: false
          });
      }
      re.DefineOwnProperty___('lastIndex', {
          value: re.lastIndex,
          writable: true,
          enumerable: false,
          configurable: false
        });
      return re;
    });

  // 15.10.5.1
  RegExp.DefineOwnProperty___('prototype', {
      value: RegExp.prototype,
      writable: false,
      enumerable: false,
      configurable: false
    });

  RegExp.prototype.DefineOwnProperty___('constructor', {
      value: RegExp,
      writable: true,
      enumerable: false,
      configurable: false
    });

  // Invoking exec and test with no arguments uses ambient data,
  // so we force them to be called with an argument, even if undefined.

  // 15.10.6.2
  virtualize(RegExp.prototype, 'exec', function (specimen) {
      return RegExp.prototype.exec.call(safeDis(this), specimen);
    });

  // 15.10.6.3
  virtualize(RegExp.prototype, 'test', function (specimen) {
      return RegExp.prototype.test.call(safeDis(this), specimen);
    });


  // 15.11 Error

  // 15.11.1, 2
  markFunc(Error);

  // 15.11.3.1
  Error.DefineOwnProperty___('prototype', {
      value: Error.prototype,
      enumerable: false,
      configurable: false,
      writable: true
    });

  // 15.11.4.1
  Error.prototype.DefineOwnProperty___('constructor', {
      value: Error,
      enumerable: false,
      configurable: false,
      writable: true
    });

  // 15.11.4.2
  Error.prototype.DefineOwnProperty___('name', {
      value: 'Error',
      enumerable: false,
      configurable: false,
      writable: true
    });

  // 15.11.4.3
  Error.prototype.DefineOwnProperty___('message', {
      value: '',
      enumerable: false,
      configurable: false,
      writable: true
    });

  // 15.11.4.4
  markFunc(Error.prototype.toString);

  // 15.11.6
  markFunc(EvalError);
  markFunc(RangeError);
  markFunc(ReferenceError);
  markFunc(SyntaxError);
  markFunc(TypeError);
  markFunc(URIError);

  ////////////////////////////////////////////////////////////////////////
  // ArrayLike
  ////////////////////////////////////////////////////////////////////////

  // makeArrayLike() produces a constructor for the purpose of taming
  // things like nodeLists.  The result, ArrayLike, takes an instance of
  // ArrayLike and two functions, getItem and getLength, which put
  // it in a position to do taming on demand.
  var makeArrayLike, itemMap = WeakMap(), lengthMap = WeakMap();
  var lengthGetter = markFuncFreeze(function () {
      var getter = lengthMap.get(this);
      return getter ? getter.i___() : void 0;
    });
  freeze(lengthGetter.prototype);

  var nativeProxies = Proxy && (function () {
      var obj = {0: 'hi'};
      var p = Proxy.create({
          get: function () {
            var P = arguments[0];
            if (typeof P !== 'string') { P = arguments[1]; }
            return obj[P];
          }
        });
      return p[0] === 'hi';
    })();
  var numericGetters = (function () {
      var obj = {};
      try {
        defProp(obj, 0, {
            get: function () { return obj; }
          });
        if (obj[0] !== obj) { return false; }
      }
      catch (_) { return false; }
      return true;
    })();

  if (nativeProxies) { (function () {
    // Make ArrayLike.prototype be a native proxy object that intercepts
    // lookups of numeric properties and redirects them to getItem, and
    // similarly for length.
    // TODO: provide ArrayLike.prototype.constructor
    ArrayLike = markFunc(function(proto, getItem, getLength) {
        if (Type(proto) !== 'Object') {
          notObject(proto, 'proto');
        }
        if (!(proto instanceof ArrayLike)) {
          throw new TypeError('Expected proto to be instanceof ArrayLike.');
        }
        var obj = beget(proto);
        itemMap.set(obj, getItem);
        lengthMap.set(obj, getLength);
        return obj;
      });

    // These are the handler methods for the proxy.
    var propDesc = function (P) {
        var opd = ownPropDesc(P);
        if (opd) {
          return opd;
        } else {
          return gopd(Object.prototype, P);
        }
      };
    var ownPropDesc = function (P) {
        // If P is 'length' or a number, handle the lookup; otherwise
        // pass it on to Object.prototype.
        P = '' + P;
        if (P === 'length') {
          return { get: lengthGetter };
        } else if (isNumericName(P)) {
          var get = markFuncFreeze(function () {
              var getter = itemMap.get(this);
              return getter ? getter.i___(+P) : void 0;
            });
          freeze(get.prototype);
          return {
              get: get,
              enumerable: true,
              configurable: true
            };
        }
        return void 0;
      };
    var has = function (P) {
        // The proxy has a length, numeric indices, and behaves
        // as though it inherits from Object.prototype.
        P = '' + P;
        return (P === 'length') ||
            isNumericName(P) ||
            P in Object.prototype;
      };
    var hasOwn = function (P) {
        // The proxy has a length and numeric indices.
        P = '' + P;
        return (P === 'length') ||
            isNumericName(P);
      };
    var gpn = function () {
        var result = gopn ();
        var objPropNames = Object.getOwnPropertyNames(Object.prototype);
        result.push.apply(result, objPropNames);
        return result;
      };
    var gopn = function () {
        var lenGetter = lengthMap.get(this);
        if (!lenGetter) { return void 0; }
        var len = lenGetter.i___();
        var result = ['length'];
        for (var i = 0; i < len; ++i) {
          result.push('' + i);
        }
        return result;
      };
    var del = function (P) {
        P = '' + P;
        if ((P === 'length') || ('' + +P === P)) { return false; }
        return true;
      };

    ArrayLike.prototype = Proxy.create({
        getPropertyDescriptor: propDesc,
        getOwnPropertyDescriptor: ownPropDesc,
        has: has,
        hasOwn: hasOwn,
        getPropertyNames: gpn,
        getOwnPropertyNames: gopn,
        'delete': del,
        fix: function() { return void 0; }
      },
      Object.prototype);
    ArrayLike.DefineOwnProperty___('prototype', {
        value: ArrayLike.prototype
      });
    freeze(ArrayLike);
    makeArrayLike = markFunc(function () { return ArrayLike; });
  })();} else if (numericGetters) { (function () {
    // Make ArrayLike.prototype be an object with a fixed set of numeric
    // getters.  To tame larger lists, replace ArrayLike and its prototype
    // using makeArrayLike(newLength).

    // See http://graphics.stanford.edu/~seander/bithacks.html#RoundUpPowerOf2
    function nextUInt31PowerOf2(v) {
      v &= 0x7fffffff;
      v |= v >> 1;
      v |= v >> 2;
      v |= v >> 4;
      v |= v >> 8;
      v |= v >> 16;
      return v + 1;
    }

    // The current function whose prototype has the most numeric getters.
    var BiggestArrayLike = void 0;
    var maxLen = 0;
    makeArrayLike = markFunc(function(length) {
        if (!BiggestArrayLike || length > maxLen) {
          var len = nextUInt31PowerOf2(length);
          // Create a new ArrayLike constructor to replace the old one.
          var BAL = markFunc(function(proto, getItem, getLength) {
              if (Type(proto) !== 'Object') {
                notObject(proto, 'proto');
              }
              if (!(proto instanceof BAL)) {
                throw new TypeError(
                    'Expected proto to be instanceof ArrayLike.');
              }
              var obj = beget(proto);
              itemMap.set(obj, getItem);
              lengthMap.set(obj, getLength);
              return obj;
            });
          // Install native numeric getters.
          for (var i = 0; i < len; i++) {
            (function(j) {
              defProp(BAL.prototype, j, {
                  get: markFuncFreeze(function() {
                    var itemGetter = itemMap.get(this);
                    return itemGetter ? itemGetter.i___(j) : void 0;
                  }),
                  enumerable: true
                });
            })(i);
          }
          // Install native length getter.
          defProp(BAL.prototype, 'length', { get: lengthGetter });
          // Whitelist prototype and prototype.constructor for ES5/3.
          BAL.DefineOwnProperty___('prototype', { value: BAL.prototype });
          BAL.prototype.DefineOwnProperty___('constructor', { value: BAL });
          // Freeze and cache the result
          freeze(BAL);
          freeze(BAL.prototype);
          BiggestArrayLike = BAL;
          maxLen = len;
        }
        return BiggestArrayLike;
      });
  })(); } else {
    // ArrayLike constructs a frozen array in the absence of better support.
    ArrayLike = markFunc(function(proto, getItem, getLength) {
        if (Type(proto) !== 'Object') {
          notObject(proto, 'proto');
        }
        var obj = beget(proto);
        var len = +getLength.i___();
        obj.DefineOwnProperty___('length', {
            value: len
          });
        for (var i = 0; i < len; ++i) {
          obj[i] = getItem.i___(i);
        }
        // Install the NUM___ flags
        isWritable(obj, 'NUM___');
        // Make numeric indices read-only
        obj.NUM____w___ = false;
        return obj;
      });
    ArrayLike.DefineOwnProperty___('prototype', {
        value: ArrayLike.prototype
      });
    ArrayLike.prototype.DefineOwnProperty___('constructor', {
        value: ArrayLike
      });
    freeze(ArrayLike);
    freeze(ArrayLike.prototype);
    makeArrayLike = markFunc(function () { return ArrayLike; });
  }

  ////////////////////////////////////////////////////////////////////////
  // Proxies
  ////////////////////////////////////////////////////////////////////////

  // Default implementations for derived traps invoke code supplied by
  // the guest on objects supplied by the guest, so we have to be careful.
  var defaultDerivedTraps = {
      has: function(name, proxy) {
          var dis = safeDis(this);
          return !!(dis.getPropertyDescriptor_m___ ?
              dis.getPropertyDescriptor(name, proxy) :
              dis.m___('getPropertyDescriptor', [name, proxy]));
        },
      hasOwn: function(name, proxy) {
          var dis = safeDis(this);
          return !!(dis.getOwnPropertyDescriptor_m___ ?
              dis.getOwnPropertyDescriptor(name, proxy) :
              dis.m___('getOwnPropertyDescriptor', [name, proxy]));
        },
      get: function(name, proxy) {
          var dis = safeDis(this);
          var desc = dis.getPropertyDescriptor_m___ ?
              dis.getPropertyDescriptor(name) :
              dis.m___('getPropertyDescriptor', [name]);
          if (desc === void 0) { return void 0; }
          if ('value_v___' in desc) {
            return desc.value_v___ ? desc.value : desc.v___('value');
          } else {
            var get = desc.get_v___ ? desc.get : desc.v___('get');
            if (!isFunction(get)) {
              return void 0;
            }
            return get.f___(safeDis(proxy), []);
          }
        },
      set: function(name, val, proxy) {
          var dis = safeDis(this);
          var desc = dis.getOwnPropertyDescriptor_m___ ?
              dis.getOwnPropertyDescriptor(name) :
              dis.m___('getOwnPropertyDescriptor', [name]);
          var set;
          if (desc) {
            if ('writable_v___' in desc) { // data
              if (desc.writable_v___ ?
                  desc.writable :
                  desc.v___('writable')) {
                desc.value_w___ === desc ?
                    desc.value = val :
                    desc.w___('value', val);
                dis.defineProperty_m___ ?
                    dis.defineProperty(name, desc) :
                    dis.m___('defineProperty', [name, desc]);
                return true;
              } else {
                return false;
              }
            } else { // accessor
              set = desc.set_v___ ? desc.set : desc.v___('set');
              if (isFunction(set)) {
                set.f___(safeDis(proxy), [val]);
                return true;
              } else {
                return false;
              }
            }
          }
          desc = dis.getPropertyDescriptor_m___ ?
              dis.getPropertyDescriptor(name) :
              dis.m___('getPropertyDescriptor', [name]);
          if (desc) {
            if ('writable_v___' in desc) { // data
              if (desc.writable_v___ ?
                  desc.writable :
                  desc.v___('writable')) {
                // fall through
              } else {
                return false;
              }
            } else { // accessor
              set = desc.set_v___ ? desc.set : desc.v___('set');
              if (isFunction(set)) {
                set.f___(safeDis(proxy), [val]);
                return true;
              } else {
                return false;
              }
            }
          }
          desc = initializeMap([
              'value', val,
              'writable', true,
              'enumerable', true,
              'configurable', true]);
          dis.defineProperty_m___ ?
              dis.defineProperty(name, desc) :
              dis.m___('defineProperty', [name, desc]);
          return true;
        },
      enumerate: function(proxy) {
          var dis = safeDis(this);
          var names = dis.getPropertyNames_m___ ?
              dis.getPropertyNames(proxy) :
              dis.m___('getPropertyNames', [proxy]);
          var filter = markFunc(function (name) {
                var desc = dis.getPropertyDescriptor_m___ ?
                    dis.getPropertyDescriptor(name, proxy) :
                    dis.m___('getPropertyDescriptor', [name, proxy]);
                return desc.enumerable_v___ ?
                    desc.enumerable :
                    desc.v___('enumerable');
            });
          return names.filter_m___ ?
              names.filter(filter) :
              names.m___('filter', [filter]);
        },
      keys: function(proxy) {
          var dis = safeDis(this);
          var names = dis.getOwnPropertyNames_m___ ?
              dis.getOwnPropertyNames(proxy) :
              dis.m___('getOwnPropertyNames', [proxy]);
          var filter = markFunc(function (name) {
                var desc = dis.getOwnPropertyDescriptor_m___ ?
                    dis.getOwnPropertyDescriptor(name, proxy) :
                    dis.m___('getOwnPropertyDescriptor', [name, proxy]);
                return desc.enumerable_v___ ?
                    desc.enumerable :
                    desc.v___('enumerable');
            });
          return names.filter_m___ ?
              names.filter(filter) :
              names.m___('filter', [filter]);
        }
    };

  function prepareProxy(proxy, handler) {
    proxy.proxy___ = proxy;
    proxy.v___ = function (P) {
        if (this !== proxy) {
          throw new TypeError(
              'Inheritance from proxies not implemented yet.');
        }
        P = '' + P;
        // Inline numeric name check
        if ('' + +P === P) { return proxy[P]; }
        assertValidPropertyName(P);
        var get = handler.get_v___ ? handler.get : handler.v___('get');
        if (isFunction(get)) {
          return get.f___(handler, [P, proxy]);
        } else {
          return defaultDerivedTraps.get.apply(handler, [P, proxy]);
        }
      };
    proxy.w___ = function (P, V) {
        if (this !== proxy) {
          throw new TypeError(
              'Inheritance from proxies not implemented yet.');
        }
        P = '' + P;
        V = asFirstClass(V);
        // Inline numeric name check
        if ('' + +P === P) { return proxy[P] = V; }
        assertValidPropertyName(P);
        var result;
        var set = handler.set_v___ ? handler.set : handler.v___('set');
        if (isFunction(set)) {
          result = set.f___(handler, [P, V, proxy]);
        } else {
          result = defaultDerivedTraps.set.apply(handler, [P, V, proxy]);
        }
        if (!result) { throw new TypeError('Failed to set ' + P); }
        return V;
      };
    proxy.c___ = function (P) {
        if (this !== proxy) {
          throw new TypeError(
              'Inheritance from proxies not implemented yet.');
        }
        P = '' + P;
        if ('' + +P === P) {
          // Allow deleting numeric properties since we allow get & set.
          delete proxy[P];
        }
        assertValidPropertyName(P);
        var deleter = handler.delete_v___ ? handler['delete'] :
            handler.v___('delete');
        var result = deleter.f___(handler, [P, proxy]);
        if (!result) { throw new TypeError('Unable to delete ' + P); }
        return true;
      };
    proxy.m___ = function (P, args) {
        if (this !== proxy) {
          throw new TypeError(
              'Inheritance from proxies not implemented yet.');
        }
        // First look it up, then call it.
        P = '' + P;
        var method;
        // Inline numeric name check
        if ('' + +P === P) { method = proxy[P]; }
        else {
          assertValidPropertyName(P);
          var get = handler.get_v___ ? handler.get : handler.v___('get');
          if (!isFunction(get)) {
            method = defaultDerivedTraps.get.apply(handler, [P, proxy]);
          } else {
            method = get.f___(handler, [P, proxy]);
          }
        }
        return method.f___(proxy, args);
      };
    proxy.e___ = function () {
        if (this !== proxy) {
          throw new TypeError(
              'Inheritance from proxies not implemented yet.');
        }
        var names;
        var enumerate = handler.enumerate___ ? handler.enumerate :
            handler.v___('enumerate');
        if (isFunction(enumerate)) {
          names = enumerate.f___(handler, [proxy]);
        } else {
          names = defaultDerivedTraps.enumerate.apply(handler, [proxy]);
        }
        var result = {};
        for (var i = 0, len = names.length; i < len; ++i) {
          var name = '' + names[i];
          assertValidPropertyName(name);
          result.DefineOwnProperty___(name, { enumerable: true });
        }
        return result;
      };
    proxy.keys___ = function () {
        if (this !== proxy) {
          throw new TypeError(
              'Inheritance from proxies not implemented yet.');
        }
        var names;
        var keys = handler.keys_v___ ? handler.keys : handler.v___('keys');
        if (isFunction(keys)) {
          names = keys.f___(handler, [proxy]);
        } else {
          names = defaultDerivedTraps.keys.apply(handler, [proxy]);
        }
        var result = [];
        var i, len = names.length, seen = {};
        for (i = 0; i < len; ++i) {
          var name = '' + names[i];
          assertValidPropertyName(name);
          if (seen[name]) { continue; }
          result.push('' + name);
          seen[name] = 1;
        }
        return result;
      };
    proxy.ownKeys___ = function () {
        if (this !== proxy) {
          throw new TypeError(
              'Inheritance from proxies not implemented yet.');
        }
        var names;
        var keys = handler.getOwnPropertyNames_v___ ?
            handler.getOwnPropertyNames :
            handler.v___('getOwnPropertyNames');
        names = keys.f___(handler, [proxy]);
        var result = [];
        var i, len = names.length, seen = {};
        for (i = 0; i < len; ++i) {
          var name = '' + names[i];
          assertValidPropertyName(name);
          if (seen[name]) { continue; }
          result.push(name);
          seen[name] = 1;
        }
        return result;
      };
    proxy.allKeys___ = function () {
        if (this !== proxy) {
          throw new TypeError(
              'Inheritance from proxies not implemented yet.');
        }
        var names;
        var keys = handler.getPropertyNames_v___ ?
            handler.getPropertyNames :
            handler.v___('getPropertyNames');
        names = keys.f___(handler, [proxy]);
        var result = [];
        var i, len = names.length, seen = {};
        for (i = 0; i < len; ++i) {
          var name = '' + names[i];
          assertValidPropertyName(name);
          if (seen[name]) { continue; }
          result.push('' + name);
          seen[name] = 1;
        }
        return result;
      };
    // TODO: If we need to let the fix trap know about numeric properties
    // set on the proxy object itself (since the proxy wasn't handling them),
    // we can pass a map of {numeric: descriptor} pairs as a second
    // argument to to handler.fix(), which can echo them back if it chooses to.
    function fix() {
      if (this !== proxy) {
        throw new TypeError(
            'Inheritance from proxies not implemented yet.');
      }
      if (proxy.fixing___) {
        throw new TypeError('Recursive fixing prohibited.');
      }
      var descMap = void 0;
      try {
        proxy.fixing___ = true;
        var fixer = handler.fix_v___ ? handler.fix : handler.v___('fix');
        descMap = fixer.f___(handler, [proxy]);
      } finally {
        delete proxy.fixing___;
      }
      if (!descMap) {
        throw new TypeError('Unable to fix the proxy.');
      }
      var isSafeFunc = isFunction(proxy) && proxy.ok___;
      var constructTrap = proxy.new___;
      var i;
      for (i in proxy) {
        if (proxy.hasOwnProperty(i)) { delete proxy[i]; }
      }
      if (isSafeFunc) {
        markFunc(proxy);
        proxy.new___ = constructTrap;
      }
      var keys = descMap.keys___();
      var len = keys.length;
      for (i = 0; i < len; ++i) {
        if (isNumericName(keys[i])) {
          var desc = ToPropertyDescriptor(descMap.v___(keys[i]));
          if (!IsDataDescriptor(desc) ||
              !desc.writable ||
              !desc.configurable ||
              !desc.enumerable) {
            throw new TypeError('Numeric properties returned from a fix trap ' +
                'must be writable, enumerable, configurable data properties.');
          }
          proxy[keys[i]] = desc.value;
        } else {
          proxy.DefineOwnProperty___(keys[i], descMap.v___(keys[i]));
        }
      }
    }
    proxy.freeze___ = function () {
        fix.call(this);
        // The fix function above deletes proxy.freeze___, so this call
        // goes to Object.prototype.freeze___
        return this.freeze___();
      };
    proxy.pe___ = function () {
        fix.call(this);
        // The fix function above deletes proxy.pe___, so this call
        // goes to Object.prototype.pe___
        return this.pe___();
      };
    proxy.seal___ = function () {
        fix.call(this);
        // The fix function above deletes proxy.seal___, so this call
        // goes to Object.prototype.seal___
        return this.seal___();
      };
    // desc is an internal property descriptor
    proxy.DefineOwnProperty___ = function (P, desc) {
        if (this !== proxy) {
          throw new TypeError(
              'Inheritance from proxies not implemented yet.');
        }
        P = '' + P;
        var extDesc = desc ? FromPropertyDescriptor(desc) : void 0;
        return (handler.defineProperty_m___ ?
            handler.defineProperty(P, extDesc, proxy) :
            handler.m___('defineProperty', [P, extDesc, proxy]));
      };
    proxy.GetOwnProperty___ = function (P) {
        if (this !== proxy) {
          throw new TypeError(
              'Inheritance from proxies not implemented yet.');
        }
        P = '' + P;
        var desc = (handler.getOwnPropertyDescriptor_m___ ?
            handler.getOwnPropertyDescriptor(P, proxy) :
            handler.m___('getOwnPropertyDescriptor', [P, proxy]));
        var intDesc = desc ? ToPropertyDescriptor(desc) : void 0;
        if (intDesc && !intDesc.configurable) {
          throw new TypeError('Proxy properties must be configurable.');
        }
        return intDesc;
     };
    proxy.HasProperty___ = function (P) {
        if (this !== proxy) {
          throw new TypeError(
              'Inheritance from proxies not implemented yet.');
        }
        P = '' + P;
        var has = handler.has_v___ ? handler.has : handler.v___('has');
        if (isFunction(has)) {
          return !!(has.f___(handler, [P, proxy]));
        } else {
          return defaultDerivedTraps.has.apply(handler, [P, proxy]);
        }
      };
    proxy.HasOwnProperty___ = function (P) {
        if (this !== proxy) {
          throw new TypeError(
              'Inheritance from proxies not implemented yet.');
        }
        P = '' + P;
        var hasOwn = handler.hasOwn_v___ ? handler.hasOwn :
            handler.v___('hasOwn');
        if (isFunction(hasOwn)) {
          return !!(hasOwn.f___(handler, [P, proxy]));
        } else {
          return defaultDerivedTraps.hasOwn.apply(handler, [P, proxy]);
        }
      };
  }

  var CajaProxy = {};
  CajaProxy.DefineOwnProperty___('create', {
      value: markFuncFreeze(function CajaProxy_create(handler, proto) {
          if (Type(handler) !== 'Object') {
            notObject(handler, 'handler');
          }
          var proxy = void 0;
          if (proto === void 0 || proxy === null) {
            proxy = {};
          } else {
            if (Type(proto) !== 'Object') {
              notObject(proto, 'proto');
            }
            if (proto.proxy___) {
              throw new TypeError('Proxies cannot inherit from proxies.');
            }
            var p = proto;
            while (p !== p.baseProto___) {
              if (p.ne___ !== p) {
                throw new TypeError(
                    'Proxies cannot inherit from extensible objects.');
              }
              p = stdGetPrototypeOf(p);
            }
            proxy = beget(proto);
            // Override all the fastpath properties so a fastpath on proto
            // doesn't give a fastpath on proxy, which must always use
            // the handlers.
            for (var i in proto) {
              var m = endsWith_v___.test(i);
              if (!m) { continue; }
              var P = m[1];
              proxy[P + '_v___'] = false;
              proxy[P + '_w___'] = false;
              proxy[P + '_gw___'] = false;
              proxy[P + '_c___'] = false;
              proxy[P + '_e___'] = false;
              proxy[P + '_m___'] = false;
              proxy[P + '_g___'] = void 0;
              proxy[P + '_s___'] = void 0;
            }
          }
          prepareProxy(proxy, handler);
          return proxy;
        }),
      enumerable: true
    });

  CajaProxy.DefineOwnProperty___('createFunction', {
      value: markFuncFreeze(function (handler, callTrap, constructTrap) {
          if (Type(handler) !== 'Object') {
            notObject(handler, 'handler');
          }
          if (!isFunction(callTrap)) {
            notFunction(callTrap, 'callTrap');
          }
          if (constructTrap === void 0) {
            constructTrap = callTrap;
          }
          if (!isFunction(constructTrap)) {
            notFunction(constructTrap, 'constructTrap');
          }
          var proto = Function.prototype;
          // Here we know the prototype chain, so we can optimize.
          if (proto.ne___ !== proto) {
            throw new TypeError(
                  'Function.prototype must not be extensible to create ' +
                  'function proxies.');
          }
          var proxy = markFunc(function (var_args) {
              return callTrap.f___(safeDis(this), arguments);
            });
          // Install deferred handlers
          proxy.v___('length');
          // Override any fastpathed properties on Function.prototype
          for (var i in proto) {
            var m = endsWith_v___.test(i);
            if (!m) { continue; }
            var P = m[1];
            proxy[P + '_v___'] = false;
            proxy[P + '_w___'] = false;
            proxy[P + '_gw___'] = false;
            proxy[P + '_c___'] = false;
            proxy[P + '_e___'] = false;
            proxy[P + '_m___'] = false;
            proxy[P + '_g___'] = void 0;
            proxy[P + '_s___'] = void 0;
          }
          prepareProxy(proxy, handler);
          proxy.new___ = function (var_args) {
              return constructTrap.apply(this, arguments);
            };
          if (constructTrap) { proxy.prototype = constructTrap.prototype; }
          return proxy;
        }),
      enumerable: true
    });

  ////////////////////////////////////////////////////////////////////////
  // Module loading
  ////////////////////////////////////////////////////////////////////////

  var myNewModuleHandler;

  /**
   * Gets the current module handler.
   */
  function getNewModuleHandler() {
    return myNewModuleHandler;
  }

  /**
   * Registers a new-module-handler, to be called back when a new
   * module is loaded.
   * <p>
   * This callback mechanism is provided so that cajoled
   * modules can be loaded from a trusted site with the
   * &lt;script&gt; tag, which runs its script as a statement, not
   * an expression. The callback is of the form
   * <tt>newModuleHandler.handle(newModule)</tt>.
   */
  function setNewModuleHandler(newModuleHandler) {
    myNewModuleHandler = newModuleHandler;
  }

  /**
   * A new-module-handler which returns the new module without
   * instantiating it.
   */
  var obtainNewModule = snowWhite({
    handle: markFuncFreeze(function handleOnly(newModule){ return newModule; })
  });

  /**
   * Enable the use of Closure Inspector, nee LavaBug
   */
  function registerClosureInspector(module) {
    if (this && this.CLOSURE_INSPECTOR___
        && this.CLOSURE_INSPECTOR___.supportsCajaDebugging) {
      this.CLOSURE_INSPECTOR___.registerCajaModule(module);
    }
  }

  /**
   * Makes a mutable copy of an object.
   * <p>
   * Even if the original is frozen, the copy will still be mutable.
   * It does a shallow copy, i.e., if record y inherits from record x,
   * ___.copy(y) will also inherit from x.
   * Copies all whitelisted properties, not just enumerable ones.
   * All resulting properties are writable, enumerable, and configurable.
   */
  function copy(obj) {
    // TODO(ihab.awad): Primordials may not be frozen; is this safe?
    var result = Array.isArray(obj) ? [] : {};
    var keys = obj.ownKeys___(), len = keys.length;
    for (var i = 0; i < len; ++i) {
      var k = keys[i], v = obj[k];
      if (isNumericName(k)) { result[k] = v; }
      else {
        result.DefineOwnProperty___(k, {
            value: v,
            writable: true,
            enumerable: true,
            configurable: true
          });
      }
    }
    return result;
  }

  /**
   * Makes and returns a fresh "normal" module handler whose imports
   * are initialized to a copy of the sharedImports.
   * <p>
   * This handles a new module by calling it, passing it the imports
   * object held in this handler. Successive modules handled by the
   * same "normal" handler thereby see a simulation of successive
   * updates to a shared global scope.
   */
  function makeNormalNewModuleHandler() {
    var imports = void 0;
    var lastOutcome = void 0;
    function getImports() {
      if (!imports) { imports = copy(sharedImports); }
      return imports;
    }
    return snowWhite({
      getImports: markFuncFreeze(getImports),
      setImports: markFuncFreeze(function setImports(newImports) {
          imports = newImports;
        }),

      /**
       * An outcome is a pair of a success flag and a value.
       * <p>
       * If the success flag is true, then the value is the normal
       * result of calling the module function. If the success flag is
       * false, then the value is the thrown error by which the module
       * abruptly terminated.
       * <p>
       * An html page is cajoled to a module that runs to completion,
       * but which reports as its outcome the outcome of its last
       * script block. In order to reify that outcome and report it
       * later, the html page initializes moduleResult___ to
       * NO_RESULT, the last script block is cajoled to set
       * moduleResult___ to something other than NO_RESULT on success
       * but to call handleUncaughtException() on
       * failure, and the html page returns moduleResult___ on
       * completion. handleUncaughtException() records a failed
       * outcome. This newModuleHandler's handle() method will not
       * overwrite an already reported outcome with NO_RESULT, so the
       * last script-block's outcome will be preserved.
       */
      getLastOutcome: markFuncFreeze(function getLastOutcome() {
          return lastOutcome;
        }),

      /**
       * If the last outcome is a success, returns its value;
       * otherwise <tt>undefined</tt>.
       */
      getLastValue: markFuncFreeze(function getLastValue() {
          if (lastOutcome && lastOutcome[0]) {
            return lastOutcome[1];
          } else {
            return void 0;
          }
        }),

      /**
       * Runs the newModule's module function.
       * <p>
       * Updates the last outcome to report the module function's
       * reported outcome. Propagate this outcome by terminating in
       * the same manner.
       */
      handle: markFuncFreeze(function handle(newModule) {
          registerClosureInspector(newModule);
          var outcome = void 0;
          try {
            var result = newModule.instantiate(___, getImports());
            if (result !== NO_RESULT) {
              outcome = [true, result];
            }
          } catch (ex) {
            outcome = [false, ex];
          }
          lastOutcome = outcome;
          if (outcome) {
            if (outcome[0]) {
              return outcome[1];
            } else {
              log(outcome[1]);
              if (outcome[1].stack) { log(outcome[1].stack); }
              throw outcome[1];
            }
          } else {
            return void 0;
          }
        }),

      /**
       * This emulates HTML5 exception handling for scripts as discussed at
       * http://code.google.com/p/google-caja/wiki/UncaughtExceptionHandling
       * and see HtmlCompiler.java for the code that calls this.
       * @param exception a raw exception.  Since {@code throw} can raise any
       *   value, exception could be any value accessible to cajoled code, or
       *   any value thrown by an API imported by cajoled code.
       * @param onerror the value of the raw reference "onerror" in top level
       *   cajoled code.  This will likely be undefined much of the time, but
       *   could be anything.  If it is a func, it can be called with
       *   three strings (message, source, lineNum) as the
       *   {@code window.onerror} event handler.
       * @param {string} source a URI describing the source file from which the
       *   error originated.
       * @param {string} lineNum the approximate line number in source at which
       *   the error originated.
       */
      handleUncaughtException: markFuncFreeze(
          function handleUncaughtException(exception,
                                           onerror,
                                           source,
                                           lineNum) {
            lastOutcome = [false, exception];

            // Cause exception to be rethrown if it is uncatchable.
            var message = exception;
            if ('object' === typeof exception && exception !== null) {
              message = '' + (exception.message || exception.desc ||
                exception.description || message);
            }

            // If we wanted to provide a hook for containers to get uncaught
            // exceptions, it would go here before onerror is invoked.

            // See the HTML5 discussion for the reasons behind this rule.
            if (!isFunction(onerror)) {
              notFunction(onerror, 'onerror');
            }
            var shouldReport = onerror.i___(
                message,
                '' + source,
                '' + lineNum);
            if (shouldReport !== false) {
              log(source + ':' + lineNum + ': ' + message);
            }
          })
    });
  }

  function isFlag(name) {
    return /_v___$/ .test(name)
        || /_w___$/ .test(name)
        || /_gw___$/.test(name)
        || /_c___$/ .test(name)
        || /_e___$/ .test(name)
        || /_g___$/ .test(name)
        || /_s___$/ .test(name)
        || /_m___$/ .test(name);
  }

  function copyToImports(imports, source) {
    for (var p in source) {
      if (source.hasOwnProperty(p)) {
        if (/__$/.test(p)) {
          if (!isFlag(p)) {
            // Caja hidden property on IMPORTS -- these are used by Domita
            imports[p] = source[p];
          }
        } else if (isNumericName(p)) {
          // Set directly
          imports[p] = source[p];
        } else {
          imports.DefineOwnProperty___(p, {
            value: source[p],
            writable: true,
            enumerable: true,
            configurable: true
          });
        }
      }
    }
  }

  /**
   * Produces a function module given an object literal module
   */
  function prepareModule(module, load) {
    if (cajaBuildVersion !== module.cajolerVersion) {
      throw new TypeError(
          "Version error: es53 expects " + cajaBuildVersion +
          " but module is " + module.cajolerVersion);
    }
    registerClosureInspector(module);
    function theModule(extraImports) {
      var imports = copy(sharedImports);
      copyToImports({
        load: load,
        cajaVM: cajaVM
      });
      copyToImports(imports, extraImports);
      return module.instantiate(___, imports);
    }

    // Whitelist certain module properties as visible to guest code. These
    // are all primitive values that do not allow two guest entities with
    // access to the same module object to communicate.
    var props = ['cajolerName', 'cajolerVersion', 'cajoledDate', 'moduleURL'];
    for (var i = 0; i < props.length; ++i) {
      theModule.DefineOwnProperty___(props[i], {
          value: module[props[i]],
          writable: false,
          enumerable: true,
          configurable: false
        });
    }
    // The below is a transitive freeze because includedModules is an array
    // of strings.
    if (!!module.includedModules) {
      theModule.DefineOwnProperty___('includedModules', {
          value: freeze(module.includedModules),
          writable: false,
          enumerable: true,
          configurable: false
        });
    }

    // Provide direct access to 'instantiate' for privileged use
    theModule.instantiate___ = function(___, IMPORTS___) {
      return module.instantiate(___, IMPORTS___);
    };

    return markFuncFreeze(theModule);
  }

  /**
   * A module is an object literal containing metadata and an
   * <code>instantiate</code> member, which is a plugin-maker function.
   * <p>
   * loadModule(module) marks module's <code>instantiate</code> member as a
   * func, freezes the module, asks the current new-module-handler to handle it
   * (thereby notifying the handler), and returns the new module.
   */
  function loadModule(module) {
    freeze(module);
    markFuncFreeze(module.instantiate);
    return myNewModuleHandler.m___('handle', [module]);
  }

  // *********************************************************************
  // * Cajita Taming API
  // * Reproduced here for Domita's and Shindig's use; new
  // * tamings should be done with the ES5 API.
  // *********************************************************************

  function grantFunc(obj, name) {
    obj.DefineOwnProperty___(name, {
        value: markFuncFreeze(obj[name]),
        writable: false,
        enumerable: false,
        configurable: false
      });
  }

  function grantRead(obj, name) {
    obj.DefineOwnProperty___(name, {
        value: obj[name],
        writable: false,
        enumerable: false,
        configurable: false
      });
  }

  /**
   * Install a getter for proto[name] that returns a wrapper that
   * first verifies that {@code this} inherits from proto.
   * <p>
   * When a pre-existing Javascript method may do something unsafe
   * when applied to a {@code this} of the wrong type, we need to
   * prevent such mis-application.
   */
  function grantTypedMethod(proto, name) {
    name = '' + name;
    var original = proto[name];
    var f = function () {};
    f.prototype = proto;
    proto.DefineOwnProperty___(name, {
        value: markFunc(function guardedApplier(var_args) {
            if (!(this instanceof f)) {
              throw new TypeError(
                  'Tamed method applied to the wrong class of object.');
            }
            return original.apply(this, arguments);
          }),
        enumerable: false,
        configurable: true,
        writable: true
      });
  }

  /**
   * Install a getter for proto[name] under the assumption
   * that the original is a generic innocent method.
   * <p>
   * As an innocent method, we assume it is exophoric (uses its
   * <tt>this</tt> parameter), requires a feral <tt>this</tt> and
   * arguments, and returns a feral result. As a generic method, we
   * assume that its <tt>this</tt> may be bound to objects that do not
   * inherit from <tt>proto</tt>.
   * <p>
   * The wrapper will untame <tt>this</tt>. Note that typically
   * <tt>this</tt> will be a constructed object and so will untame to
   * itself. The wrapper will also untame the arguments and tame and
   * return the result.
   */
  function grantInnocentMethod(proto, name) {
    var original = proto[name];
    proto.DefineOwnProperty___(name, {
        enumerable: false,
        configurable: false,
        get: markFunc(function () {
            return function guardedApplier(var_args) {
                var feralThis = safeDis(taming.untame(this));
                // The function untame expects an actual array.
                var feralArgs = taming.untame(slice.call(arguments, 0));
                var feralResult = original.apply(feralThis, feralArgs);
                return taming.tame(feralResult);
              };
          }),
        set: void 0
      });
  }

  /**
   * A shorthand that happens to be useful here.
   * <p>
   * For all i in arg2s: func2(arg1,arg2s[i]).
   */
  function all2(func2, arg1, arg2s) {
    var len = arg2s.length;
    for (var i = 0; i < len; i += 1) {
      func2(arg1, arg2s[i]);
    }
  }

  /**
   * Inside a <tt>___.forOwnKeys()</tt> or <tt>___.forAllKeys()</tt>, the
   * body function can terminate early, as if with a conventional
   * <tt>break;</tt>, by doing a <pre>return ___.BREAK;</pre>
   */
  var BREAK = Token('BREAK');

  /**
   * Used in domita with the name "forOwnKeys" for iterating over
   * JSON containers.
   */
  function forOwnNonCajaKeys(obj, fn) {
    for (var i in obj) {
      if (!obj.hasOwnProperty(i)) { continue; }
      if (endsWith__.test(i)) { continue; }
      if (fn(i, obj[i]) === BREAK) {
        return;
      }
    }
  }

  // TODO(metaweta): Deprecate this API, since it requires that we leave
  // configurable set to true in order to use both a getter and a setter.
  function useGetHandler(obj, name, getHandler) {
    getHandler = markFunc(getHandler);
    name = '' + name;
    var desc = obj.GetOwnProperty___(name);
    if (!desc || !IsAccessorDescriptor(desc)) {
      desc = {
          enumerable: false,
          configurable: true,
          get: getHandler,
          set: void 0
        };
    } else {
      desc.get = getHandler;
    }
    obj.DefineOwnProperty___(name, desc);
  }

  function useSetHandler(obj, name, setHandler) {
    setHandler = markFunc(setHandler);
    name = '' + name;
    var desc = obj.GetOwnProperty___(name);
    if (!IsAccessorDescriptor(desc)) {
      desc = {
          enumerable: false,
          configurable: true,
          get: void 0,
          set: setHandler
        };
    } else {
      desc.set = setHandler;
    }
    obj.DefineOwnProperty___(name, desc);
  }

  function hasOwnProp(obj, name) {
    return obj && obj.hasOwnProperty(name);
  }

  // *********************************************************************
  // * Exports
  // *********************************************************************

  cajaVM = whitelistAll({
      // Diagnostics and condition enforcement
      log: log,
      enforce: enforce,
      enforceType: enforceType,
      enforceNat: enforceNat,

      // Object indistinguishability
      Token: Token,

      // Guards and Trademarks
      identity: identity,
      callWithEjector: callWithEjector,
      eject: eject,
      GuardT: GuardT,
      Trademark: Trademark,
      guard: guard,
      passesGuard: passesGuard,
      stamp: stamp,
      makeTableGuard: makeTableGuard,

      // Sealing & Unsealing
      makeSealerUnsealerPair: makeSealerUnsealerPair,

      // Defensible objects
      def: def,

      // Other
      makeArrayLike: makeArrayLike,
      isFunction: isFunction,
      USELESS: USELESS,
      manifest: manifest,
      allKeys: allKeys
    });

  function readImport(imports, name) {
    name = '' + name;
    if (imports.HasProperty___(name)) {
      return imports.v___(name);
    }
    throw new ReferenceError(name + ' is not defined.');
  }

  function declareImport(imports, name) {
    if (imports.HasProperty___(name)) {
      return;
    }
    imports.w___(name, void 0);
  }

  function writeImport(imports, name, value) {
    if (imports.HasProperty___(name)) {
      imports.w___(name, value);
      return value;
    }
    throw new ReferenceError(name + ' is not defined.');
  }

  function goodParseInt(n, radix) {
    n = '' + n;
    // This turns an undefined radix into a NaN but is ok since NaN
    // is treated as undefined by parseInt
    radix = +radix;
    var isHexOrOctal = /^\s*[+-]?\s*0(x?)/.exec(n);
    var isOct = isHexOrOctal ? isHexOrOctal[1] !== 'x' : false;

    if (isOct && (radix !== radix || 0 === radix)) {
      return parseInt(n, 10);
    }
    return parseInt(n, radix);
  }

  var sharedImports = whitelistAll({
      cajaVM: cajaVM,

      'null': null,
      'false': false,
      'true': true,
      'NaN': NaN,
      'Infinity': Infinity,
      'undefined': void 0,
      parseInt: markFunc(goodParseInt),
      parseFloat: markFunc(parseFloat),
      isNaN: markFunc(isNaN),
      isFinite: markFunc(isFinite),
      decodeURI: markFunc(decodeURI),
      decodeURIComponent: markFunc(decodeURIComponent),
      encodeURI: markFunc(encodeURI),
      encodeURIComponent: markFunc(encodeURIComponent),
      escape: escape ? markFunc(escape) : (void 0),
      Math: Math,
      JSON: safeJSON,

      Object: Object,
      Array: Array,
      String: String,
      Boolean: Boolean,
      Number: Number,
      Date: Date,
      RegExp: RegExp,
      Function: FakeFunction,
      Proxy: CajaProxy,

      Error: Error,
      EvalError: EvalError,
      RangeError: RangeError,
      ReferenceError: ReferenceError,
      SyntaxError: SyntaxError,
      TypeError: TypeError,
      URIError: URIError,

      // ES-Harmony future features
      WeakMap: WeakMap
    });

  Object.prototype.m___ = function (name, as) {
      name = '' + name;
      if (this[name + '_m___']) {
        return this[name].f___(this, as);
      }
      var m = this.v___(name);
      if (typeof m !== 'function') {
        notFunction(m, 'property "' + name + '"');
      }
      // Fastpath the method on the object pointed to by name_v___
      // which is truthy iff it's a data property
      var ownerObj = this[name + '_v___'];
      if (ownerObj && ownerObj !== Function.prototype) {
        fastpathMethod(ownerObj, name);
      }
      return m.f___(this, as);
    };

  ___ = {
      sharedImports: sharedImports,
      USELESS: USELESS,
      BREAK: BREAK,
      args: args,
      deodorize: deodorize,
      copy: copy,
      i: isIn,
      iM: initializeMap,
      f: markSafeFunc,
      markFunc: markFunc,
      markFuncFreeze: markFuncFreeze,
      Trademark: Trademark,
      makeSealerUnsealerPair: makeSealerUnsealerPair,
      getId: getId,
      getImports: getImports,
      unregister: unregister,
      newTable: newTable,
      whitelistAll: whitelistAll,
      snowWhite: snowWhite,
      makeDefensibleFunction: makeDefensibleFunction,
      makeDefensibleObject: makeDefensibleObject,
      ri: readImport,
      di: declareImport,
      wi: writeImport,
      copyToImports: copyToImports,
      // Cajita API
      grantRead: grantRead,
      grantFunc: grantFunc,
      grantTypedMethod: grantTypedMethod,
      grantInnocentMethod: grantInnocentMethod,
      all2: all2,
      hasOwnProp: hasOwnProp,
      forOwnKeys: forOwnNonCajaKeys,
      markCtor: markFuncFreeze,
      useGetHandler: useGetHandler,
      useSetHandler: useSetHandler,
      primFreeze: snowWhite,
      isJSONContainer: isExtensible,
      getLogFunc: getLogFunc,
      setLogFunc: setLogFunc,
      callPub: function (obj, name, args) { return obj.m___(name, args); },
      readPub: function (obj, name) { return obj.v___(name); },
      canRead: function (obj, name) { return (name + '_v___') in obj; },
      freeze: freeze,
      // Module loading
      getNewModuleHandler: getNewModuleHandler,
      setNewModuleHandler: setNewModuleHandler,
      obtainNewModule: obtainNewModule,
      makeNormalNewModuleHandler: makeNormalNewModuleHandler,
      prepareModule: prepareModule,
      loadModule: loadModule,
      NO_RESULT: NO_RESULT,
      // Defensible objects
      def: def,
      // Taming
      extend: extend,
      isDefinedInCajaFrame: isDefinedInCajaFrame,
      rawDelete: rawDelete,
      getter: getter,
      setter: setter,
      directConstructor: directConstructor,
      BASE_OBJECT_CONSTRUCTOR: BASE_OBJECT_CONSTRUCTOR,
      DISABLE_SECURITY_FOR_DEBUGGER: false
    };
  var cajaVMKeys = ownEnumKeys(cajaVM);
  for (var i = 0; i < cajaVMKeys.length; ++i) {
    ___[cajaVMKeys[i]] = cajaVM[cajaVMKeys[i]];
  }
  setNewModuleHandler(makeNormalNewModuleHandler());
})(this);

// Exports for closure compiler.
if (typeof window !== 'undefined') {
  window['___'] = ___;
  window['safeJSON'] = safeJSON;
  window['WeakMap'] = WeakMap;
  window['cajaVM' ] = cajaVM;
}
;
/**
 * @fileoverview
 * Capture the results of a cajoled module that is included by script
 * concatenation.
  *
 * @author jasvir@gmail.com
 * @author kpreid@switchb.org
 * @requires ___, console
 * @overrides window
 * @provides CaptureCajoledModule
 */

function CaptureCajoledModule(importsCallback) {
  // Save and restore
  var originalNewModuleHandler = ___.getNewModuleHandler();

  // Set up a fresh handler
  var ourHandler = ___.makeNormalNewModuleHandler();

  // ... which grabs Domado and removes itself.
  var normalHandle = ourHandler.handle;
  ourHandler.handle = ___.markFuncFreeze(function (module) {
    ___.setNewModuleHandler(originalNewModuleHandler);

    // TODO(kpreid): This useful-for-debugging code ought to be available
    // somewhere, perhaps as part of the ES53 runtime. cajaVM.log is similar,
    // but is not a de facto standard for non-Caja environments.
    //
    // Firebug console object and its methods have no prototype methods so we
    // cannot expose them.
    if (typeof console !== 'undefined') {
      var saneConsole = {};
      ["log", "warn", "error", "debug"].forEach(function (v) {
        var f = console[v];
        var logFunc;
        if (console.firebug) {
          // On Firebug, there is a hazard of the inspector invoking
          // the objects it inspects, so we replace them with text
          logFunc = function () {
            var a = [];
            for (var i = 0; i < arguments.length; i++) {
              a[i] = "" + arguments[i];
            }
            Function.prototype.apply.call(f, console, a);
          };
        } else {
          logFunc = function () {
            Function.prototype.apply.call(f, console, arguments);
          };
        }
        saneConsole.DefineOwnProperty___(v, {
          value: ___.markFuncFreeze(logFunc),
          enumerable: true
        });
      });
      ourHandler.getImports().DefineOwnProperty___("console", {
        value: ___.freeze(saneConsole),
        enumerable: true
      });
    }

    // Load module
    normalHandle.call(ourHandler, module);

    var imports = ourHandler.getImports();

    importsCallback(imports);
  });

  ___.setNewModuleHandler(ourHandler);
}

if (typeof window !== 'undefined') {
  window['CaptureCajoledModule'] = CaptureCajoledModule;
}
;
/**
 * @fileoverview
 * Given that the next script executed is the cajoled Domado module, make
 * the Domado entry points available in the global scope.
 *
 * @author ihab.awad@gmail.com
 * @requires CaptureCajoledModule
 * @overrides window
 * @provides Domado, HtmlEmitter
 */

var Domado = undefined;
var HtmlEmitter = undefined;

CaptureCajoledModule(function(imports) {
  Domado = imports.Domado;
  HtmlEmitter = imports.HtmlEmitter;
});

// Exports for closure compiler.
if (typeof window !== 'undefined') {
  window['Domado'] = Domado;
  window['HtmlEmitter'] = HtmlEmitter;
}
;
{
  ___.loadModule({
      'instantiate': function (___, IMPORTS___) {
        return ___.prepareModule({
            'instantiate': function (___, IMPORTS___) {
              var dis___ = IMPORTS___;
              var moduleResult___;
              moduleResult___ = ___.NO_RESULT;
              ___.di(IMPORTS___, 'URI'), IMPORTS___.w___('URI', ___.f(function
                  () {
                    var parse, create, encodeIfExists, encodeIfExists2,
                    encodeOne, normPath, PARENT_DIRECTORY_HANDLER,
                    PARENT_DIRECTORY_HANDLER_RE, EXTRA_PARENT_PATHS_RE,
                    collapse_dots, resolve, URI, x0___, x1___, x2___, x3___,
                    x4___, x5___, x6___, x7___, x8___, x9___, x10___, x11___,
                    x12___, x13___, x14___, x15___, x16___, x17___, x18___,
                    x19___, x20___, x21___, x22___, x23___, x24___, x25___,
                    x26___, x27___, x28___, x29___, x30___, x31___, x32___,
                    x33___, x34___, x35___, x36___, x37___, x38___, x39___,
                    x40___, x41___, x42___, x43___, x44___, x45___, x46___,
                    x47___, x48___, x49___, x50___, x51___, x52___, x53___,
                    x54___, x55___, x56___, x57___, x58___, x59___, x60___,
                    x61___, x62___, x63___, x64___, x65___, x66___, x67___,
                    x68___, x69___, x70___, x71___, x72___, x73___, x74___,
                    x75___, x76___, x77___, x78___, x79___, x80___, x81___,
                    x82___, x83___, x84___, x85___, nullIfAbsent, URI_RE_,
                    URI_DISALLOWED_IN_SCHEME_OR_CREDENTIALS_,
                    URI_DISALLOWED_IN_PATH_, x86___, x87___;
                    {
                      function parse(uriStr) {
                        var m, x0___, x1___;
                        m = (x0___ = '' + uriStr, x1___ = URI_RE_,
                          x0___.match_m___? x0___.match(x1___):
                          x0___.m___('match', [ x1___ ]));
                        if (!m) { return null; }
                        return new URI.new___((nullIfAbsent.i___(m[ +1 ])),
                          (nullIfAbsent.i___(m[ +2 ])), (nullIfAbsent.i___(m[
                                +3 ])), (nullIfAbsent.i___(m[ +4 ])),
                          (nullIfAbsent.i___(m[ +5 ])), (nullIfAbsent.i___(m[
                                +6 ])), (nullIfAbsent.i___(m[ +7 ])));
                      }
                      ___.f(parse, 'parse');
                    }
                    {
                      function create(scheme, credentials, domain, port, path,
                        query, fragment) {
                        var uri, x0___, x1___, x2___, x3___, x4___;
                        uri = new URI.new___((encodeIfExists2.i___(scheme,
                              URI_DISALLOWED_IN_SCHEME_OR_CREDENTIALS_)),
                          (encodeIfExists2.i___(credentials,
                             URI_DISALLOWED_IN_SCHEME_OR_CREDENTIALS_)),
                          (encodeIfExists.i___(domain)), (port > 0?
                            port.toString_m___? port.toString():
                            port.m___('toString', [ ]): null),
                          (encodeIfExists2.i___(path, URI_DISALLOWED_IN_PATH_))
                          , null, (encodeIfExists.i___(fragment)));
                        if (query) {
                          if ('string' === typeof query) {
                            x4___ = uri, x3___ = (x1___ = query, x0___ = new
                              RegExp.new___('[^?\x26=0-9A-Za-z_\\-~.%]', 'g'),
                              x2___ = encodeOne, x1___.replace_m___?
                              x1___.replace(x0___, x2___):
                              x1___.m___('replace', [ x0___, x2___ ])),
                            x4___.setRawQuery_m___? x4___.setRawQuery(x3___):
                            x4___.m___('setRawQuery', [ x3___ ]);
                          } else {
                            uri.setAllParameters_m___?
                              uri.setAllParameters(query):
                            uri.m___('setAllParameters', [ query ]);
                          }
                        }
                        return uri;
                      }
                      ___.f(create, 'create');
                    }
                    {
                      function encodeIfExists(unescapedPart) {
                        if ('string' == typeof unescapedPart) {
                          return (IMPORTS___.encodeURIComponent_v___?
                            IMPORTS___.encodeURIComponent: ___.ri(IMPORTS___,
                              'encodeURIComponent')).i___(unescapedPart);
                        }
                        return null;
                      }
                      ___.f(encodeIfExists, 'encodeIfExists');
                    }
                    {
                      function encodeIfExists2(unescapedPart, extra) {
                        var x0___, x1___, x2___;
                        if ('string' == typeof unescapedPart) {
                          return x0___ = (IMPORTS___.encodeURI_v___?
                            IMPORTS___.encodeURI: ___.ri(IMPORTS___,
                              'encodeURI')).i___(unescapedPart), x1___ = extra,
                          x2___ = encodeOne, x0___.replace_m___?
                            x0___.replace(x1___, x2___): x0___.m___('replace',
                            [ x1___, x2___ ]);
                        }
                        return null;
                      }
                      ___.f(encodeIfExists2, 'encodeIfExists2');
                    }
                    {
                      function encodeOne(ch) {
                        var n, x0___, x1___;
                        n = ch.charCodeAt_m___? ch.charCodeAt(0):
                        ch.m___('charCodeAt', [ 0 ]);
                        return '%' + (x0___ = n >> 4 & 15,
                          '0123456789ABCDEF'.charAt_m___?
                          '0123456789ABCDEF'.charAt(x0___):
                          '0123456789ABCDEF'.m___('charAt', [ x0___ ])) +
                          (x1___ = n & 15, '0123456789ABCDEF'.charAt_m___?
                          '0123456789ABCDEF'.charAt(x1___):
                          '0123456789ABCDEF'.m___('charAt', [ x1___ ]));
                      }
                      ___.f(encodeOne, 'encodeOne');
                    }
                    {
                      function normPath(path) {
                        var x0___, x1___, x2___, x3___;
                        return x2___ = (x1___ = path, x0___ = new
                          RegExp.new___('(^|\\/)\\.(?:\\/|$)', 'g'),
                          x1___.replace_m___? x1___.replace(x0___, '$1'):
                          x1___.m___('replace', [ x0___, '$1' ])), x3___ = new
                          RegExp.new___('\\/{2,}', 'g'), x2___.replace_m___?
                          x2___.replace(x3___, '/'): x2___.m___('replace', [
                            x3___, '/' ]);
                      }
                      ___.f(normPath, 'normPath');
                    }
                    {
                      function collapse_dots(path) {
                        var p, r, q;
                        if (path === null) { return null; }
                        p = normPath.i___(path);
                        r = PARENT_DIRECTORY_HANDLER_RE;
                        for (; (q = p.replace_m___? p.replace(r, '$1'):
                            p.m___('replace', [ r, '$1' ])) != p; p = q) {}
                        return p;
                      }
                      ___.f(collapse_dots, 'collapse_dots');
                    }
                    {
                      function resolve(baseUri, relativeUri) {
                        var absoluteUri, overridden, x0___, x1___, x2___,
                        x3___, x4___, x5___, rawPath, simplifiedPath, x6___,
                        x7___, absRawPath, x8___, x9___, slash, x10___, x11___,
                        x12___, x13___, x14___, x15___;
                        absoluteUri = baseUri.clone_m___? baseUri.clone():
                        baseUri.m___('clone', [ ]);
                        overridden = relativeUri.hasScheme_m___?
                          relativeUri.hasScheme():
                        relativeUri.m___('hasScheme', [ ]);
                        if (overridden) {
                          x1___ = absoluteUri, x0___ =
                            relativeUri.getRawScheme_m___?
                            relativeUri.getRawScheme():
                          relativeUri.m___('getRawScheme', [ ]),
                          x1___.setRawScheme_m___? x1___.setRawScheme(x0___):
                          x1___.m___('setRawScheme', [ x0___ ]);
                        } else {
                          overridden = relativeUri.hasCredentials_m___?
                            relativeUri.hasCredentials():
                          relativeUri.m___('hasCredentials', [ ]);
                        }
                        if (overridden) {
                          x3___ = absoluteUri, x2___ =
                            relativeUri.getRawCredentials_m___?
                            relativeUri.getRawCredentials():
                          relativeUri.m___('getRawCredentials', [ ]),
                          x3___.setRawCredentials_m___?
                            x3___.setRawCredentials(x2___):
                          x3___.m___('setRawCredentials', [ x2___ ]);
                        } else {
                          overridden = relativeUri.hasDomain_m___?
                            relativeUri.hasDomain():
                          relativeUri.m___('hasDomain', [ ]);
                        }
                        if (overridden) {
                          x5___ = absoluteUri, x4___ =
                            relativeUri.getRawDomain_m___?
                            relativeUri.getRawDomain():
                          relativeUri.m___('getRawDomain', [ ]),
                          x5___.setRawDomain_m___? x5___.setRawDomain(x4___):
                          x5___.m___('setRawDomain', [ x4___ ]);
                        } else {
                          overridden = relativeUri.hasPort_m___?
                            relativeUri.hasPort(): relativeUri.m___('hasPort',
                            [ ]);
                        }
                        rawPath = relativeUri.getRawPath_m___?
                          relativeUri.getRawPath():
                        relativeUri.m___('getRawPath', [ ]);
                        simplifiedPath = collapse_dots.i___(rawPath);
                        if (overridden) {
                          x7___ = absoluteUri, x6___ =
                            relativeUri.getPort_m___? relativeUri.getPort():
                          relativeUri.m___('getPort', [ ]), x7___.setPort_m___?
                            x7___.setPort(x6___): x7___.m___('setPort', [ x6___
                            ]);
                          simplifiedPath = simplifiedPath &&
                            (simplifiedPath.replace_m___?
                            simplifiedPath.replace(EXTRA_PARENT_PATHS_RE, ''):
                            simplifiedPath.m___('replace', [
                                EXTRA_PARENT_PATHS_RE, '' ]));
                        } else {
                          overridden = ! !rawPath;
                          if (overridden) {
                            if ((simplifiedPath.charCodeAt_m___?
                                simplifiedPath.charCodeAt(0):
                                simplifiedPath.m___('charCodeAt', [ 0 ])) !==
                              47) {
                              absRawPath = (x8___ =
                                collapse_dots.i___((absoluteUri.getRawPath_m___?
                                    absoluteUri.getRawPath():
                                    absoluteUri.m___('getRawPath', [ ])) || '')
                                , x9___ = EXTRA_PARENT_PATHS_RE,
                                x8___.replace_m___? x8___.replace(x9___, ''):
                                x8___.m___('replace', [ x9___, '' ]));
                              slash = (absRawPath.lastIndexOf_m___?
                                absRawPath.lastIndexOf('/'):
                                absRawPath.m___('lastIndexOf', [ '/' ])) + 1;
                              simplifiedPath = (x10___ =
                                collapse_dots.i___((slash?
                                    absRawPath.substring_m___?
                                    absRawPath.substring(0, slash):
                                    absRawPath.m___('substring', [ 0, slash ]):
                                    '') + collapse_dots.i___(rawPath)), x11___
                                = EXTRA_PARENT_PATHS_RE, x10___.replace_m___?
                                x10___.replace(x11___, ''):
                                x10___.m___('replace', [ x11___, '' ]));
                            }
                          } else {
                            simplifiedPath = simplifiedPath &&
                              (simplifiedPath.replace_m___?
                              simplifiedPath.replace(EXTRA_PARENT_PATHS_RE, '')
                              : simplifiedPath.m___('replace', [
                                  EXTRA_PARENT_PATHS_RE, '' ]));
                            if (simplifiedPath !== rawPath) {
                              absoluteUri.setRawPath_m___?
                                absoluteUri.setRawPath(simplifiedPath):
                              absoluteUri.m___('setRawPath', [ simplifiedPath ]
                              );
                            }
                          }
                        }
                        if (overridden) {
                          absoluteUri.setRawPath_m___?
                            absoluteUri.setRawPath(simplifiedPath):
                          absoluteUri.m___('setRawPath', [ simplifiedPath ]);
                        } else {
                          overridden = relativeUri.hasQuery_m___?
                            relativeUri.hasQuery():
                          relativeUri.m___('hasQuery', [ ]);
                        }
                        if (overridden) {
                          x13___ = absoluteUri, x12___ =
                            relativeUri.getRawQuery_m___?
                            relativeUri.getRawQuery():
                          relativeUri.m___('getRawQuery', [ ]),
                          x13___.setRawQuery_m___? x13___.setRawQuery(x12___):
                          x13___.m___('setRawQuery', [ x12___ ]);
                        } else {
                          overridden = relativeUri.hasFragment_m___?
                            relativeUri.hasFragment():
                          relativeUri.m___('hasFragment', [ ]);
                        }
                        if (overridden) {
                          x15___ = absoluteUri, x14___ =
                            relativeUri.getRawFragment_m___?
                            relativeUri.getRawFragment():
                          relativeUri.m___('getRawFragment', [ ]),
                          x15___.setRawFragment_m___?
                            x15___.setRawFragment(x14___):
                          x15___.m___('setRawFragment', [ x14___ ]);
                        }
                        return absoluteUri;
                      }
                      ___.f(resolve, 'resolve');
                    }
                    {
                      function URI(rawScheme, rawCredentials, rawDomain, port,
                        rawPath, rawQuery, rawFragment) {
                        var dis___ = this && this.___? void 0: this;
                        dis___.scheme__w___ === dis___? (dis___.scheme_ =
                          rawScheme): dis___.w___('scheme_', rawScheme);
                        dis___.credentials__w___ === dis___?
                          (dis___.credentials_ = rawCredentials):
                        dis___.w___('credentials_', rawCredentials);
                        dis___.domain__w___ === dis___? (dis___.domain_ =
                          rawDomain): dis___.w___('domain_', rawDomain);
                        dis___.port__w___ === dis___? (dis___.port_ = port):
                        dis___.w___('port_', port);
                        dis___.path__w___ === dis___? (dis___.path_ = rawPath):
                        dis___.w___('path_', rawPath);
                        dis___.query__w___ === dis___? (dis___.query_ =
                          rawQuery): dis___.w___('query_', rawQuery);
                        dis___.fragment__w___ === dis___? (dis___.fragment_ =
                          rawFragment): dis___.w___('fragment_', rawFragment);
                        dis___.paramCache__w___ === dis___? (dis___.paramCache_
                          = null): dis___.w___('paramCache_', null);
                      }
                      ___.f(URI, 'URI');
                    }
                    {
                      function nullIfAbsent(matchPart) {
                        return 'string' == typeof matchPart && matchPart.length
                          > 0? matchPart: null;
                      }
                      ___.f(nullIfAbsent, 'nullIfAbsent');
                    }
                    PARENT_DIRECTORY_HANDLER = new (IMPORTS___.RegExp_v___?
                      IMPORTS___.RegExp: ___.ri(IMPORTS___, 'RegExp'))
                      .new___(('' + '(/|^)' +
                        '(?:[^./][^/]*|\\.{2,}(?:[^./][^/]*)|\\.{3,}[^/]*)' +
                        '/\\.\\.(?:/|$)'));
                    PARENT_DIRECTORY_HANDLER_RE = new (IMPORTS___.RegExp_v___?
                      IMPORTS___.RegExp: ___.ri(IMPORTS___, 'RegExp'))
                      .new___(PARENT_DIRECTORY_HANDLER);
                    EXTRA_PARENT_PATHS_RE = new
                      RegExp.new___('^(?:\\.\\.\\/)*(?:\\.\\.$)?');
                    x0___ = URI.prototype_v___? URI.prototype:
                    URI.v___('prototype'), x1___ = (function () {
                        function toString$_meth() {
                          var dis___ = this && this.___? void 0: this;
                          var out, x0___, x1___, x2___, x3___, x4___, x5___,
                          x6___, x7___, x8___, x9___, x10___, x11___, x12___,
                          x13___, x14___;
                          out = [ ];
                          if (null !== (dis___.scheme__v___? dis___.scheme_:
                              dis___.v___('scheme_'))) {
                            x1___ = out, x0___ = dis___.scheme__v___?
                              dis___.scheme_: dis___.v___('scheme_'),
                            x1___.push_m___? x1___.push(x0___, ':'):
                            x1___.m___('push', [ x0___, ':' ]);
                          }
                          if (null !== (dis___.domain__v___? dis___.domain_:
                              dis___.v___('domain_'))) {
                            out.push_m___? out.push('//'): out.m___('push', [
                                '//' ]);
                            if (null !== (dis___.credentials__v___?
                                dis___.credentials_:
                                dis___.v___('credentials_'))) {
                              x3___ = out, x2___ = dis___.credentials__v___?
                                dis___.credentials_:
                              dis___.v___('credentials_'), x3___.push_m___?
                                x3___.push(x2___, '@'): x3___.m___('push', [
                                  x2___, '@' ]);
                            }
                            x5___ = out, x4___ = dis___.domain__v___?
                              dis___.domain_: dis___.v___('domain_'),
                            x5___.push_m___? x5___.push(x4___):
                            x5___.m___('push', [ x4___ ]);
                            if (null !== (dis___.port__v___? dis___.port_:
                                dis___.v___('port_'))) {
                              x8___ = out, x7___ = (x6___ = dis___.port__v___?
                                dis___.port_: dis___.v___('port_'),
                                x6___.toString_m___? x6___.toString():
                                x6___.m___('toString', [ ])), x8___.push_m___?
                                x8___.push(':', x7___): x8___.m___('push', [
                                  ':', x7___ ]);
                            }
                          }
                          if (null !== (dis___.path__v___? dis___.path_:
                              dis___.v___('path_'))) {
                            x10___ = out, x9___ = dis___.path__v___?
                              dis___.path_: dis___.v___('path_'),
                            x10___.push_m___? x10___.push(x9___):
                            x10___.m___('push', [ x9___ ]);
                          }
                          if (null !== (dis___.query__v___? dis___.query_:
                              dis___.v___('query_'))) {
                            x12___ = out, x11___ = dis___.query__v___?
                              dis___.query_: dis___.v___('query_'),
                            x12___.push_m___? x12___.push('?', x11___):
                            x12___.m___('push', [ '?', x11___ ]);
                          }
                          if (null !== (dis___.fragment__v___?
                              dis___.fragment_: dis___.v___('fragment_'))) {
                            x14___ = out, x13___ = dis___.fragment__v___?
                              dis___.fragment_: dis___.v___('fragment_'),
                            x14___.push_m___? x14___.push('#', x13___):
                            x14___.m___('push', [ '#', x13___ ]);
                          }
                          return out.join_m___? out.join(''): out.m___('join',
                            [ '' ]);
                        }
                        return ___.f(toString$_meth, 'toString$_meth');
                      })(), x0___.toString_w___ === x0___? (x0___.toString =
                      x1___): x0___.w___('toString', x1___);
                    x2___ = URI.prototype_v___? URI.prototype:
                    URI.v___('prototype'), x3___ = (function () {
                        function clone$_meth() {
                          var dis___ = this && this.___? void 0: this;
                          return new URI.new___((dis___.scheme__v___?
                              dis___.scheme_: dis___.v___('scheme_')),
                            (dis___.credentials__v___? dis___.credentials_:
                             dis___.v___('credentials_')),
                            (dis___.domain__v___? dis___.domain_:
                             dis___.v___('domain_')), (dis___.port__v___?
                              dis___.port_: dis___.v___('port_')),
                            (dis___.path__v___? dis___.path_:
                             dis___.v___('path_')), (dis___.query__v___?
                              dis___.query_: dis___.v___('query_')),
                            (dis___.fragment__v___? dis___.fragment_:
                             dis___.v___('fragment_')));
                        }
                        return ___.f(clone$_meth, 'clone$_meth');
                      })(), x2___.clone_w___ === x2___? (x2___.clone = x3___):
                    x2___.w___('clone', x3___);
                    x4___ = URI.prototype_v___? URI.prototype:
                    URI.v___('prototype'), x5___ = (function () {
                        function getScheme$_meth() {
                          var dis___ = this && this.___? void 0: this;
                          return (dis___.scheme__v___? dis___.scheme_:
                            dis___.v___('scheme_')) &&
                            (IMPORTS___.decodeURIComponent_v___?
                            IMPORTS___.decodeURIComponent: ___.ri(IMPORTS___,
                              'decodeURIComponent')).i___(dis___.scheme__v___?
                            dis___.scheme_: dis___.v___('scheme_'));
                        }
                        return ___.f(getScheme$_meth, 'getScheme$_meth');
                      })(), x4___.getScheme_w___ === x4___? (x4___.getScheme =
                      x5___): x4___.w___('getScheme', x5___);
                    x6___ = URI.prototype_v___? URI.prototype:
                    URI.v___('prototype'), x7___ = (function () {
                        function getRawScheme$_meth() {
                          var dis___ = this && this.___? void 0: this;
                          return dis___.scheme__v___? dis___.scheme_:
                          dis___.v___('scheme_');
                        }
                        return ___.f(getRawScheme$_meth, 'getRawScheme$_meth');
                      })(), x6___.getRawScheme_w___ === x6___?
                      (x6___.getRawScheme = x7___): x6___.w___('getRawScheme',
                      x7___);
                    x8___ = URI.prototype_v___? URI.prototype:
                    URI.v___('prototype'), x9___ = (function () {
                        function setScheme$_meth(newScheme) {
                          var dis___ = this && this.___? void 0: this;
                          var x0___, x1___;
                          x1___ = dis___, x0___ =
                            encodeIfExists2.i___(newScheme,
                            URI_DISALLOWED_IN_SCHEME_OR_CREDENTIALS_),
                          x1___.scheme__w___ === x1___? (x1___.scheme_ = x0___)
                            : x1___.w___('scheme_', x0___);
                          return dis___;
                        }
                        return ___.f(setScheme$_meth, 'setScheme$_meth');
                      })(), x8___.setScheme_w___ === x8___? (x8___.setScheme =
                      x9___): x8___.w___('setScheme', x9___);
                    x10___ = URI.prototype_v___? URI.prototype:
                    URI.v___('prototype'), x11___ = (function () {
                        function setRawScheme$_meth(newScheme) {
                          var dis___ = this && this.___? void 0: this;
                          var x0___, x1___;
                          x1___ = dis___, x0___ = newScheme? newScheme: null,
                          x1___.scheme__w___ === x1___? (x1___.scheme_ = x0___)
                            : x1___.w___('scheme_', x0___);
                          return dis___;
                        }
                        return ___.f(setRawScheme$_meth, 'setRawScheme$_meth');
                      })(), x10___.setRawScheme_w___ === x10___?
                      (x10___.setRawScheme = x11___):
                    x10___.w___('setRawScheme', x11___);
                    x12___ = URI.prototype_v___? URI.prototype:
                    URI.v___('prototype'), x13___ = (function () {
                        function hasScheme$_meth() {
                          var dis___ = this && this.___? void 0: this;
                          return null !== (dis___.scheme__v___? dis___.scheme_:
                            dis___.v___('scheme_'));
                        }
                        return ___.f(hasScheme$_meth, 'hasScheme$_meth');
                      })(), x12___.hasScheme_w___ === x12___? (x12___.hasScheme
                      = x13___): x12___.w___('hasScheme', x13___);
                    x14___ = URI.prototype_v___? URI.prototype:
                    URI.v___('prototype'), x15___ = (function () {
                        function getCredentials$_meth() {
                          var dis___ = this && this.___? void 0: this;
                          return (dis___.credentials__v___?
                            dis___.credentials_: dis___.v___('credentials_'))
                            && (IMPORTS___.decodeURIComponent_v___?
                            IMPORTS___.decodeURIComponent: ___.ri(IMPORTS___,
                              'decodeURIComponent'))
                            .i___(dis___.credentials__v___?
                            dis___.credentials_: dis___.v___('credentials_'));
                        }
                        return ___.f(getCredentials$_meth,
                          'getCredentials$_meth');
                      })(), x14___.getCredentials_w___ === x14___?
                      (x14___.getCredentials = x15___):
                    x14___.w___('getCredentials', x15___);
                    x16___ = URI.prototype_v___? URI.prototype:
                    URI.v___('prototype'), x17___ = (function () {
                        function getRawCredentials$_meth() {
                          var dis___ = this && this.___? void 0: this;
                          return dis___.credentials__v___? dis___.credentials_:
                          dis___.v___('credentials_');
                        }
                        return ___.f(getRawCredentials$_meth,
                          'getRawCredentials$_meth');
                      })(), x16___.getRawCredentials_w___ === x16___?
                      (x16___.getRawCredentials = x17___):
                    x16___.w___('getRawCredentials', x17___);
                    x18___ = URI.prototype_v___? URI.prototype:
                    URI.v___('prototype'), x19___ = (function () {
                        function setCredentials$_meth(newCredentials) {
                          var dis___ = this && this.___? void 0: this;
                          var x0___, x1___;
                          x1___ = dis___, x0___ =
                            encodeIfExists2.i___(newCredentials,
                            URI_DISALLOWED_IN_SCHEME_OR_CREDENTIALS_),
                          x1___.credentials__w___ === x1___?
                            (x1___.credentials_ = x0___):
                          x1___.w___('credentials_', x0___);
                          return dis___;
                        }
                        return ___.f(setCredentials$_meth,
                          'setCredentials$_meth');
                      })(), x18___.setCredentials_w___ === x18___?
                      (x18___.setCredentials = x19___):
                    x18___.w___('setCredentials', x19___);
                    x20___ = URI.prototype_v___? URI.prototype:
                    URI.v___('prototype'), x21___ = (function () {
                        function setRawCredentials$_meth(newCredentials) {
                          var dis___ = this && this.___? void 0: this;
                          var x0___, x1___;
                          x1___ = dis___, x0___ = newCredentials?
                            newCredentials: null, x1___.credentials__w___ ===
                            x1___? (x1___.credentials_ = x0___):
                          x1___.w___('credentials_', x0___);
                          return dis___;
                        }
                        return ___.f(setRawCredentials$_meth,
                          'setRawCredentials$_meth');
                      })(), x20___.setRawCredentials_w___ === x20___?
                      (x20___.setRawCredentials = x21___):
                    x20___.w___('setRawCredentials', x21___);
                    x22___ = URI.prototype_v___? URI.prototype:
                    URI.v___('prototype'), x23___ = (function () {
                        function hasCredentials$_meth() {
                          var dis___ = this && this.___? void 0: this;
                          return null !== (dis___.credentials__v___?
                            dis___.credentials_: dis___.v___('credentials_'));
                        }
                        return ___.f(hasCredentials$_meth,
                          'hasCredentials$_meth');
                      })(), x22___.hasCredentials_w___ === x22___?
                      (x22___.hasCredentials = x23___):
                    x22___.w___('hasCredentials', x23___);
                    x24___ = URI.prototype_v___? URI.prototype:
                    URI.v___('prototype'), x25___ = (function () {
                        function getDomain$_meth() {
                          var dis___ = this && this.___? void 0: this;
                          return (dis___.domain__v___? dis___.domain_:
                            dis___.v___('domain_')) &&
                            (IMPORTS___.decodeURIComponent_v___?
                            IMPORTS___.decodeURIComponent: ___.ri(IMPORTS___,
                              'decodeURIComponent')).i___(dis___.domain__v___?
                            dis___.domain_: dis___.v___('domain_'));
                        }
                        return ___.f(getDomain$_meth, 'getDomain$_meth');
                      })(), x24___.getDomain_w___ === x24___? (x24___.getDomain
                        = x25___): x24___.w___('getDomain', x25___);
                    x26___ = URI.prototype_v___? URI.prototype:
                    URI.v___('prototype'), x27___ = (function () {
                        function getRawDomain$_meth() {
                          var dis___ = this && this.___? void 0: this;
                          return dis___.domain__v___? dis___.domain_:
                          dis___.v___('domain_');
                        }
                        return ___.f(getRawDomain$_meth, 'getRawDomain$_meth');
                      })(), x26___.getRawDomain_w___ === x26___?
                      (x26___.getRawDomain = x27___):
                    x26___.w___('getRawDomain', x27___);
                    x28___ = URI.prototype_v___? URI.prototype:
                    URI.v___('prototype'), x29___ = (function () {
                        function setDomain$_meth(newDomain) {
                          var dis___ = this && this.___? void 0: this;
                          var x0___, x1___;
                          return x1___ = dis___, x0___ = newDomain &&
                            (IMPORTS___.encodeURIComponent_v___?
                            IMPORTS___.encodeURIComponent: ___.ri(IMPORTS___,
                              'encodeURIComponent')).i___(newDomain),
                          x1___.setRawDomain_m___? x1___.setRawDomain(x0___):
                          x1___.m___('setRawDomain', [ x0___ ]);
                        }
                        return ___.f(setDomain$_meth, 'setDomain$_meth');
                      })(), x28___.setDomain_w___ === x28___? (x28___.setDomain
                        = x29___): x28___.w___('setDomain', x29___);
                    x30___ = URI.prototype_v___? URI.prototype:
                    URI.v___('prototype'), x31___ = (function () {
                        function setRawDomain$_meth(newDomain) {
                          var dis___ = this && this.___? void 0: this;
                          var x0___, x1___, x2___, x3___;
                          x1___ = dis___, x0___ = newDomain? newDomain: null,
                          x1___.domain__w___ === x1___? (x1___.domain_ = x0___)
                            : x1___.w___('domain_', x0___);
                          return x3___ = dis___, x2___ = dis___.path__v___?
                            dis___.path_: dis___.v___('path_'),
                          x3___.setRawPath_m___? x3___.setRawPath(x2___):
                          x3___.m___('setRawPath', [ x2___ ]);
                        }
                        return ___.f(setRawDomain$_meth, 'setRawDomain$_meth');
                      })(), x30___.setRawDomain_w___ === x30___?
                      (x30___.setRawDomain = x31___):
                    x30___.w___('setRawDomain', x31___);
                    x32___ = URI.prototype_v___? URI.prototype:
                    URI.v___('prototype'), x33___ = (function () {
                        function hasDomain$_meth() {
                          var dis___ = this && this.___? void 0: this;
                          return null !== (dis___.domain__v___? dis___.domain_:
                            dis___.v___('domain_'));
                        }
                        return ___.f(hasDomain$_meth, 'hasDomain$_meth');
                      })(), x32___.hasDomain_w___ === x32___? (x32___.hasDomain
                        = x33___): x32___.w___('hasDomain', x33___);
                    x34___ = URI.prototype_v___? URI.prototype:
                    URI.v___('prototype'), x35___ = (function () {
                        function getPort$_meth() {
                          var dis___ = this && this.___? void 0: this;
                          return (dis___.port__v___? dis___.port_:
                            dis___.v___('port_')) &&
                            (IMPORTS___.decodeURIComponent_v___?
                            IMPORTS___.decodeURIComponent: ___.ri(IMPORTS___,
                              'decodeURIComponent')).i___(dis___.port__v___?
                            dis___.port_: dis___.v___('port_'));
                        }
                        return ___.f(getPort$_meth, 'getPort$_meth');
                      })(), x34___.getPort_w___ === x34___? (x34___.getPort =
                        x35___): x34___.w___('getPort', x35___);
                    x36___ = URI.prototype_v___? URI.prototype:
                    URI.v___('prototype'), x37___ = (function () {
                        function setPort$_meth(newPort) {
                          var dis___ = this && this.___? void 0: this;
                          var x0___, x1___;
                          if (newPort) {
                            newPort = (IMPORTS___.Number_v___?
                              IMPORTS___.Number: ___.ri(IMPORTS___, 'Number'))
                              .i___(newPort);
                            if (newPort !== (newPort & 65535)) {
                              throw new (IMPORTS___.Error_v___?
                                IMPORTS___.Error: ___.ri(IMPORTS___, 'Error'))
                                .new___(('Bad port number ' + newPort));
                            }
                            x1___ = dis___, x0___ = '' + newPort,
                            x1___.port__w___ === x1___? (x1___.port_ = x0___):
                            x1___.w___('port_', x0___);
                          } else {
                            dis___.port__w___ === dis___? (dis___.port_ = null)
                              : dis___.w___('port_', null);
                          }
                          return dis___;
                        }
                        return ___.f(setPort$_meth, 'setPort$_meth');
                      })(), x36___.setPort_w___ === x36___? (x36___.setPort =
                        x37___): x36___.w___('setPort', x37___);
                    x38___ = URI.prototype_v___? URI.prototype:
                    URI.v___('prototype'), x39___ = (function () {
                        function hasPort$_meth() {
                          var dis___ = this && this.___? void 0: this;
                          return null !== (dis___.port__v___? dis___.port_:
                            dis___.v___('port_'));
                        }
                        return ___.f(hasPort$_meth, 'hasPort$_meth');
                      })(), x38___.hasPort_w___ === x38___? (x38___.hasPort =
                        x39___): x38___.w___('hasPort', x39___);
                    x40___ = URI.prototype_v___? URI.prototype:
                    URI.v___('prototype'), x41___ = (function () {
                        function getPath$_meth() {
                          var dis___ = this && this.___? void 0: this;
                          return (dis___.path__v___? dis___.path_:
                            dis___.v___('path_')) &&
                            (IMPORTS___.decodeURIComponent_v___?
                            IMPORTS___.decodeURIComponent: ___.ri(IMPORTS___,
                              'decodeURIComponent')).i___(dis___.path__v___?
                            dis___.path_: dis___.v___('path_'));
                        }
                        return ___.f(getPath$_meth, 'getPath$_meth');
                      })(), x40___.getPath_w___ === x40___? (x40___.getPath =
                        x41___): x40___.w___('getPath', x41___);
                    x42___ = URI.prototype_v___? URI.prototype:
                    URI.v___('prototype'), x43___ = (function () {
                        function getRawPath$_meth() {
                          var dis___ = this && this.___? void 0: this;
                          return dis___.path__v___? dis___.path_:
                          dis___.v___('path_');
                        }
                        return ___.f(getRawPath$_meth, 'getRawPath$_meth');
                      })(), x42___.getRawPath_w___ === x42___?
                      (x42___.getRawPath = x43___): x42___.w___('getRawPath',
                      x43___);
                    x44___ = URI.prototype_v___? URI.prototype:
                    URI.v___('prototype'), x45___ = (function () {
                        function setPath$_meth(newPath) {
                          var dis___ = this && this.___? void 0: this;
                          var x0___, x1___;
                          return x1___ = dis___, x0___ =
                            encodeIfExists2.i___(newPath,
                            URI_DISALLOWED_IN_PATH_), x1___.setRawPath_m___?
                            x1___.setRawPath(x0___): x1___.m___('setRawPath', [
                              x0___ ]);
                        }
                        return ___.f(setPath$_meth, 'setPath$_meth');
                      })(), x44___.setPath_w___ === x44___? (x44___.setPath =
                        x45___): x44___.w___('setPath', x45___);
                    x46___ = URI.prototype_v___? URI.prototype:
                    URI.v___('prototype'), x47___ = (function () {
                        function setRawPath$_meth(newPath) {
                          var dis___ = this && this.___? void 0: this;
                          var x0___, x1___, x2___, x3___;
                          if (newPath) {
                            newPath = (IMPORTS___.String_v___?
                              IMPORTS___.String: ___.ri(IMPORTS___, 'String'))
                              .i___(newPath);
                            x3___ = dis___, x2___ = ! (dis___.domain__v___?
                              dis___.domain_: dis___.v___('domain_')) || (x0___
                              = new RegExp.new___('^\\/'), x1___ = newPath,
                              x0___.test_m___? x0___.test(x1___):
                              x0___.m___('test', [ x1___ ])) ? newPath: '/' +
                              newPath, x3___.path__w___ === x3___? (x3___.path_
                              = x2___): x3___.w___('path_', x2___);
                          } else {
                            dis___.path__w___ === dis___? (dis___.path_ = null)
                              : dis___.w___('path_', null);
                          }
                          return dis___;
                        }
                        return ___.f(setRawPath$_meth, 'setRawPath$_meth');
                      })(), x46___.setRawPath_w___ === x46___?
                      (x46___.setRawPath = x47___): x46___.w___('setRawPath',
                      x47___);
                    x48___ = URI.prototype_v___? URI.prototype:
                    URI.v___('prototype'), x49___ = (function () {
                        function hasPath$_meth() {
                          var dis___ = this && this.___? void 0: this;
                          return null !== (dis___.path__v___? dis___.path_:
                            dis___.v___('path_'));
                        }
                        return ___.f(hasPath$_meth, 'hasPath$_meth');
                      })(), x48___.hasPath_w___ === x48___? (x48___.hasPath =
                      x49___): x48___.w___('hasPath', x49___);
                    x50___ = URI.prototype_v___? URI.prototype:
                    URI.v___('prototype'), x51___ = (function () {
                        function getQuery$_meth() {
                          var dis___ = this && this.___? void 0: this;
                          var x0___, x1___;
                          return (dis___.query__v___? dis___.query_:
                            dis___.v___('query_')) && (x0___ =
                            (IMPORTS___.decodeURIComponent_v___?
                             IMPORTS___.decodeURIComponent: ___.ri(IMPORTS___,
                               'decodeURIComponent')).i___(dis___.query__v___?
                              dis___.query_: dis___.v___('query_')), x1___ =
                            new RegExp.new___('\\+', 'g'), x0___.replace_m___?
                            x0___.replace(x1___, ' '): x0___.m___('replace', [
                                x1___, ' ' ]));
                        }
                        return ___.f(getQuery$_meth, 'getQuery$_meth');
                      })(), x50___.getQuery_w___ === x50___? (x50___.getQuery =
                      x51___): x50___.w___('getQuery', x51___);
                    x52___ = URI.prototype_v___? URI.prototype:
                    URI.v___('prototype'), x53___ = (function () {
                        function getRawQuery$_meth() {
                          var dis___ = this && this.___? void 0: this;
                          return dis___.query__v___? dis___.query_:
                          dis___.v___('query_');
                        }
                        return ___.f(getRawQuery$_meth, 'getRawQuery$_meth');
                      })(), x52___.getRawQuery_w___ === x52___?
                      (x52___.getRawQuery = x53___): x52___.w___('getRawQuery',
                      x53___);
                    x54___ = URI.prototype_v___? URI.prototype:
                    URI.v___('prototype'), x55___ = (function () {
                        function setQuery$_meth(newQuery) {
                          var dis___ = this && this.___? void 0: this;
                          var x0___, x1___;
                          dis___.paramCache__w___ === dis___?
                            (dis___.paramCache_ = null):
                          dis___.w___('paramCache_', null);
                          x1___ = dis___, x0___ = encodeIfExists.i___(newQuery)
                            , x1___.query__w___ === x1___? (x1___.query_ =
                            x0___): x1___.w___('query_', x0___);
                          return dis___;
                        }
                        return ___.f(setQuery$_meth, 'setQuery$_meth');
                      })(), x54___.setQuery_w___ === x54___? (x54___.setQuery =
                      x55___): x54___.w___('setQuery', x55___);
                    x56___ = URI.prototype_v___? URI.prototype:
                    URI.v___('prototype'), x57___ = (function () {
                        function setRawQuery$_meth(newQuery) {
                          var dis___ = this && this.___? void 0: this;
                          var x0___, x1___;
                          dis___.paramCache__w___ === dis___?
                            (dis___.paramCache_ = null):
                          dis___.w___('paramCache_', null);
                          x1___ = dis___, x0___ = newQuery? newQuery: null,
                          x1___.query__w___ === x1___? (x1___.query_ = x0___):
                          x1___.w___('query_', x0___);
                          return dis___;
                        }
                        return ___.f(setRawQuery$_meth, 'setRawQuery$_meth');
                      })(), x56___.setRawQuery_w___ === x56___?
                      (x56___.setRawQuery = x57___): x56___.w___('setRawQuery',
                      x57___);
                    x58___ = URI.prototype_v___? URI.prototype:
                    URI.v___('prototype'), x59___ = (function () {
                        function hasQuery$_meth() {
                          var dis___ = this && this.___? void 0: this;
                          return null !== (dis___.query__v___? dis___.query_:
                            dis___.v___('query_'));
                        }
                        return ___.f(hasQuery$_meth, 'hasQuery$_meth');
                      })(), x58___.hasQuery_w___ === x58___? (x58___.hasQuery =
                      x59___): x58___.w___('hasQuery', x59___);
                    x60___ = URI.prototype_v___? URI.prototype:
                    URI.v___('prototype'), x61___ = (function () {
                        function setAllParameters$_meth(params) {
                          var dis___ = this && this.___? void 0: this;
                          var x0___, x1___, x2___, x3___, newParams, i, k,
                          x4___, x5___, x6___, v, queryBuf, separator, j, k, v,
                          x7___, x8___, x9___, x10___, x11___, x12___, x13___;;
                          if (typeof params === 'object') {
                            if (! (params instanceof (IMPORTS___.Array_v___?
                                  IMPORTS___.Array: ___.ri(IMPORTS___, 'Array')
                                )) && (params instanceof
                                (IMPORTS___.Object_v___? IMPORTS___.Object:
                                 ___.ri(IMPORTS___, 'Object')) || (x2___ =
                                  (x1___ = (x0___ = IMPORTS___.Object_v___?
                                     IMPORTS___.Object: ___.ri(IMPORTS___,
                                       'Object'), x0___.prototype_v___?
                                     x0___.prototype: x0___.v___('prototype')),
                                   x1___.toString_v___? x1___.toString:
                                   x1___.v___('toString')), x3___ = params,
                                  x2___.call_m___? x2___.call(x3___):
                                  x2___.m___('call', [ x3___ ])) !==
                                '[object Array]')) {
                              newParams = [ ];
                              i = -1;
                              x6___ = Object(params).e___();
                              for (x5___ in x6___) {
                                if (typeof x5___ === 'number' || '' + +x5___
                                  === x5___) { k = x5___; } else {
                                  if (/^NUM___/.test(x5___) &&
                                      /__$/.test(x5___)) { continue; }
                                  x4___ = x5___.match(/([\s\S]*)_e___$/);
                                  if (!x4___ || !x6___[ x5___ ]) { continue; }
                                  k = x4___[ 1 ];
                                }
                                {
                                  v = params.v___(k);
                                  if ('string' === typeof v) {
                                    newParams.NUM____w___ === newParams?
                                      (newParams[ + ++i ] = k):
                                    newParams.w___(+ ++i, k);
                                    newParams.NUM____w___ === newParams?
                                      (newParams[ + ++i ] = v):
                                    newParams.w___(+ ++i, v);
                                  }
                                }
                              }
                              params = newParams;
                            }
                          }
                          dis___.paramCache__w___ === dis___?
                            (dis___.paramCache_ = null):
                          dis___.w___('paramCache_', null);
                          queryBuf = [ ];
                          separator = '';
                          for (j = 0; j < params.length;) {
                            k = params.v___(j++);
                            v = params.v___(j++);
                            x8___ = queryBuf, x9___ = separator, x7___ =
                              (IMPORTS___.encodeURIComponent_v___?
                              IMPORTS___.encodeURIComponent: ___.ri(IMPORTS___,
                                'encodeURIComponent')).i___(k.toString_m___?
                              k.toString(): k.m___('toString', [ ])),
                            x8___.push_m___? x8___.push(x9___, x7___):
                            x8___.m___('push', [ x9___, x7___ ]);
                            separator = '\x26';
                            if (v) {
                              x11___ = queryBuf, x10___ =
                                (IMPORTS___.encodeURIComponent_v___?
                                IMPORTS___.encodeURIComponent:
                                ___.ri(IMPORTS___, 'encodeURIComponent'))
                                .i___(v.toString_m___? v.toString():
                                v.m___('toString', [ ])), x11___.push_m___?
                                x11___.push('=', x10___): x11___.m___('push', [
                                  '=', x10___ ]);
                            }
                          }
                          x13___ = dis___, x12___ = queryBuf.join_m___?
                            queryBuf.join(''): queryBuf.m___('join', [ '' ]),
                          x13___.query__w___ === x13___? (x13___.query_ =
                            x12___): x13___.w___('query_', x12___);
                          return dis___;
                        }
                        return ___.f(setAllParameters$_meth,
                          'setAllParameters$_meth');
                      })(), x60___.setAllParameters_w___ === x60___?
                      (x60___.setAllParameters = x61___):
                    x60___.w___('setAllParameters', x61___);
                    x62___ = URI.prototype_v___? URI.prototype:
                    URI.v___('prototype'), x63___ = (function () {
                        function checkParameterCache_$_meth() {
                          var dis___ = this && this.___? void 0: this;
                          var q, x0___, x1___, cgiParams, x2___, x3___, out, k,
                          i, m, x4___, x5___, x6___, x7___, x8___, x9___,
                          x10___, x11___, x12___, x13___;
                          if (! (dis___.paramCache__v___? dis___.paramCache_:
                              dis___.v___('paramCache_'))) {
                            q = dis___.query__v___? dis___.query_:
                            dis___.v___('query_');
                            if (!q) {
                              x1___ = dis___, x0___ = [ ],
                              x1___.paramCache__w___ === x1___?
                                (x1___.paramCache_ = x0___):
                              x1___.w___('paramCache_', x0___);
                            } else {
                              cgiParams = (x3___ = q, x2___ = new
                                RegExp.new___('[\x26\\?]'), x3___.split_m___?
                                x3___.split(x2___): x3___.m___('split', [ x2___
                                  ]));
                              out = [ ];
                              k = -1;
                              for (i = 0; i < cgiParams.length; ++i) {
                                m = (x4___ = cgiParams[ +i ], x5___ = new
                                  RegExp.new___('^([^=]*)(?:=(.*))?$'),
                                  x4___.match_m___? x4___.match(x5___):
                                  x4___.m___('match', [ x5___ ]));
                                x9___ = out, x8___ = (x6___ =
                                  (IMPORTS___.decodeURIComponent_v___?
                                   IMPORTS___.decodeURIComponent:
                                   ___.ri(IMPORTS___, 'decodeURIComponent'))
                                  .i___(m[ +1 ]), x7___ = new
                                  RegExp.new___('\\+', 'g'),
                                  x6___.replace_m___? x6___.replace(x7___, ' ')
                                  : x6___.m___('replace', [ x7___, ' ' ])),
                                x9___.NUM____w___ === x9___? (x9___[ + ++k ] =
                                  x8___): x9___.w___(+ ++k, x8___);
                                x13___ = out, x12___ = (x10___ =
                                  (IMPORTS___.decodeURIComponent_v___?
                                   IMPORTS___.decodeURIComponent:
                                   ___.ri(IMPORTS___, 'decodeURIComponent'))
                                  .i___(m[ +2 ] || ''), x11___ = new
                                  RegExp.new___('\\+', 'g'),
                                  x10___.replace_m___? x10___.replace(x11___,
                                    ' '): x10___.m___('replace', [ x11___, ' '
                                    ])), x13___.NUM____w___ === x13___?
                                  (x13___[ + ++k ] = x12___): x13___.w___(+ ++k, x12___);
                              }
                              dis___.paramCache__w___ === dis___?
                                (dis___.paramCache_ = out):
                              dis___.w___('paramCache_', out);
                            }
                          }
                        }
                        return ___.f(checkParameterCache_$_meth,
                          'checkParameterCache_$_meth');
                      })(), x62___.checkParameterCache__w___ === x62___?
                      (x62___.checkParameterCache_ = x63___):
                    x62___.w___('checkParameterCache_', x63___);
                    x64___ = URI.prototype_v___? URI.prototype:
                    URI.v___('prototype'), x65___ = (function () {
                        function setParameterValues$_meth(key, values) {
                          var dis___ = this && this.___? void 0: this;
                          var newValueIndex, pc, params, i, k, x0___, x1___,
                          x2___, x3___, x4___, x5___, x6___, x7___, x8___;
                          if (typeof values === 'string') {
                            values = [ values ];
                          }
                          dis___.checkParameterCache__m___?
                            dis___.checkParameterCache_():
                          dis___.m___('checkParameterCache_', [ ]);
                          newValueIndex = 0;
                          pc = dis___.paramCache__v___? dis___.paramCache_:
                          dis___.v___('paramCache_');
                          params = [ ];
                          for (i = 0, k = 0; i < pc.length; i = i + 2) {
                            if (key === pc.v___(i)) {
                              if (newValueIndex < values.length) {
                                x1___ = params, x2___ = key, x0___ =
                                  values.v___(newValueIndex++),
                                x1___.push_m___? x1___.push(x2___, x0___):
                                x1___.m___('push', [ x2___, x0___ ]);
                              }
                            } else {
                              x5___ = params, x3___ = pc.v___(i), x4___ =
                                pc.v___(i + 1), x5___.push_m___?
                                x5___.push(x3___, x4___): x5___.m___('push', [
                                  x3___, x4___ ]);
                            }
                          }
                          while (newValueIndex < values.length) {
                            x7___ = params, x8___ = key, x6___ =
                              values.v___(newValueIndex++), x7___.push_m___?
                              x7___.push(x8___, x6___): x7___.m___('push', [
                                x8___, x6___ ]);
                          }
                          dis___.setAllParameters_m___?
                            dis___.setAllParameters(params):
                          dis___.m___('setAllParameters', [ params ]);
                          return dis___;
                        }
                        return ___.f(setParameterValues$_meth,
                          'setParameterValues$_meth');
                      })(), x64___.setParameterValues_w___ === x64___?
                      (x64___.setParameterValues = x65___):
                    x64___.w___('setParameterValues', x65___);
                    x66___ = URI.prototype_v___? URI.prototype:
                    URI.v___('prototype'), x67___ = (function () {
                        function removeParameter$_meth(key) {
                          var dis___ = this && this.___? void 0: this;
                          var x0___, x1___, x2___;
                          return x1___ = dis___, x2___ = key, x0___ = [ ],
                          x1___.setParameterValues_m___?
                            x1___.setParameterValues(x2___, x0___):
                          x1___.m___('setParameterValues', [ x2___, x0___ ]);
                        }
                        return ___.f(removeParameter$_meth,
                          'removeParameter$_meth');
                      })(), x66___.removeParameter_w___ === x66___?
                      (x66___.removeParameter = x67___):
                    x66___.w___('removeParameter', x67___);
                    x68___ = URI.prototype_v___? URI.prototype:
                    URI.v___('prototype'), x69___ = (function () {
                        function getAllParameters$_meth() {
                          var dis___ = this && this.___? void 0: this;
                          var x0___, x1___;
                          dis___.checkParameterCache__m___?
                            dis___.checkParameterCache_():
                          dis___.m___('checkParameterCache_', [ ]);
                          return x0___ = dis___.paramCache__v___?
                            dis___.paramCache_: dis___.v___('paramCache_'),
                          x1___ = (dis___.paramCache__v___? dis___.paramCache_:
                            dis___.v___('paramCache_')).length,
                          x0___.slice_m___? x0___.slice(0, x1___):
                          x0___.m___('slice', [ 0, x1___ ]);
                        }
                        return ___.f(getAllParameters$_meth,
                          'getAllParameters$_meth');
                      })(), x68___.getAllParameters_w___ === x68___?
                      (x68___.getAllParameters = x69___):
                    x68___.w___('getAllParameters', x69___);
                    x70___ = URI.prototype_v___? URI.prototype:
                    URI.v___('prototype'), x71___ = (function () {
                        function getParameterValues$_meth(paramNameUnescaped) {
                          var dis___ = this && this.___? void 0: this;
                          var values, i, x0___, x1___;
                          dis___.checkParameterCache__m___?
                            dis___.checkParameterCache_():
                          dis___.m___('checkParameterCache_', [ ]);
                          values = [ ];
                          for (i = 0; i < (dis___.paramCache__v___?
                              dis___.paramCache_: dis___.v___('paramCache_'))
                            .length; i = i + 2) {
                            if (paramNameUnescaped ===
                              (dis___.paramCache__v___? dis___.paramCache_:
                               dis___.v___('paramCache_')).v___(i)) {
                              x1___ = values, x0___ = (dis___.paramCache__v___?
                                dis___.paramCache_: dis___.v___('paramCache_'))
                                .v___(i + 1), x1___.push_m___?
                                x1___.push(x0___): x1___.m___('push', [ x0___ ]
                              );
                            }
                          }
                          return values;
                        }
                        return ___.f(getParameterValues$_meth,
                          'getParameterValues$_meth');
                      })(), x70___.getParameterValues_w___ === x70___?
                      (x70___.getParameterValues = x71___):
                    x70___.w___('getParameterValues', x71___);
                    x72___ = URI.prototype_v___? URI.prototype:
                    URI.v___('prototype'), x73___ = (function () {
                        function getParameterMap$_meth(paramNameUnescaped) {
                          var dis___ = this && this.___? void 0: this;
                          var paramMap, i, key, value, x0___, x1___;
                          dis___.checkParameterCache__m___?
                            dis___.checkParameterCache_():
                          dis___.m___('checkParameterCache_', [ ]);
                          paramMap = ___.iM([ ]);
                          for (i = 0; i < (dis___.paramCache__v___?
                              dis___.paramCache_: dis___.v___('paramCache_'))
                            .length; i = i + 2) {
                            key = (dis___.paramCache__v___? dis___.paramCache_:
                              dis___.v___('paramCache_')).v___(i++), value =
                              (dis___.paramCache__v___? dis___.paramCache_:
                              dis___.v___('paramCache_')).v___(i++);
                            if (!___.i('' + key, paramMap)) {
                              paramMap.w___(key, [ value ]);
                            } else {
                              x0___ = paramMap.v___(key), x1___ = value,
                              x0___.push_m___? x0___.push(x1___):
                              x0___.m___('push', [ x1___ ]);
                            }
                          }
                          return paramMap;
                        }
                        return ___.f(getParameterMap$_meth,
                          'getParameterMap$_meth');
                      })(), x72___.getParameterMap_w___ === x72___?
                      (x72___.getParameterMap = x73___):
                    x72___.w___('getParameterMap', x73___);
                    x74___ = URI.prototype_v___? URI.prototype:
                    URI.v___('prototype'), x75___ = (function () {
                        function getParameterValue$_meth(paramNameUnescaped) {
                          var dis___ = this && this.___? void 0: this;
                          var i;
                          dis___.checkParameterCache__m___?
                            dis___.checkParameterCache_():
                          dis___.m___('checkParameterCache_', [ ]);
                          for (i = 0; i < (dis___.paramCache__v___?
                              dis___.paramCache_: dis___.v___('paramCache_'))
                            .length; i = i + 2) {
                            if (paramNameUnescaped ===
                              (dis___.paramCache__v___? dis___.paramCache_:
                               dis___.v___('paramCache_')).v___(i)) {
                              return (dis___.paramCache__v___?
                                dis___.paramCache_: dis___.v___('paramCache_'))
                                .v___(i + 1);
                            }
                          }
                          return null;
                        }
                        return ___.f(getParameterValue$_meth,
                          'getParameterValue$_meth');
                      })(), x74___.getParameterValue_w___ === x74___?
                      (x74___.getParameterValue = x75___):
                    x74___.w___('getParameterValue', x75___);
                    x76___ = URI.prototype_v___? URI.prototype:
                    URI.v___('prototype'), x77___ = (function () {
                        function getFragment$_meth() {
                          var dis___ = this && this.___? void 0: this;
                          return (dis___.fragment__v___? dis___.fragment_:
                            dis___.v___('fragment_')) &&
                            (IMPORTS___.decodeURIComponent_v___?
                            IMPORTS___.decodeURIComponent: ___.ri(IMPORTS___,
                              'decodeURIComponent'))
                            .i___(dis___.fragment__v___? dis___.fragment_:
                            dis___.v___('fragment_'));
                        }
                        return ___.f(getFragment$_meth, 'getFragment$_meth');
                      })(), x76___.getFragment_w___ === x76___?
                      (x76___.getFragment = x77___): x76___.w___('getFragment',
                      x77___);
                    x78___ = URI.prototype_v___? URI.prototype:
                    URI.v___('prototype'), x79___ = (function () {
                        function getRawFragment$_meth() {
                          var dis___ = this && this.___? void 0: this;
                          return dis___.fragment__v___? dis___.fragment_:
                          dis___.v___('fragment_');
                        }
                        return ___.f(getRawFragment$_meth,
                          'getRawFragment$_meth');
                      })(), x78___.getRawFragment_w___ === x78___?
                      (x78___.getRawFragment = x79___):
                    x78___.w___('getRawFragment', x79___);
                    x80___ = URI.prototype_v___? URI.prototype:
                    URI.v___('prototype'), x81___ = (function () {
                        function setFragment$_meth(newFragment) {
                          var dis___ = this && this.___? void 0: this;
                          var x0___, x1___;
                          x1___ = dis___, x0___ = newFragment?
                            (IMPORTS___.encodeURIComponent_v___?
                              IMPORTS___.encodeURIComponent: ___.ri(IMPORTS___,
                              'encodeURIComponent')).i___(newFragment): null,
                          x1___.fragment__w___ === x1___? (x1___.fragment_ =
                              x0___): x1___.w___('fragment_', x0___);
                          return dis___;
                        }
                        return ___.f(setFragment$_meth, 'setFragment$_meth');
                      })(), x80___.setFragment_w___ === x80___?
                      (x80___.setFragment = x81___): x80___.w___('setFragment',
                      x81___);
                    x82___ = URI.prototype_v___? URI.prototype:
                    URI.v___('prototype'), x83___ = (function () {
                        function setRawFragment$_meth(newFragment) {
                          var dis___ = this && this.___? void 0: this;
                          var x0___, x1___;
                          x1___ = dis___, x0___ = newFragment? newFragment:
                          null, x1___.fragment__w___ === x1___?
                            (x1___.fragment_ = x0___): x1___.w___('fragment_',
                            x0___);
                          return dis___;
                        }
                        return ___.f(setRawFragment$_meth,
                          'setRawFragment$_meth');
                      })(), x82___.setRawFragment_w___ === x82___?
                      (x82___.setRawFragment = x83___):
                    x82___.w___('setRawFragment', x83___);
                    x84___ = URI.prototype_v___? URI.prototype:
                    URI.v___('prototype'), x85___ = (function () {
                        function hasFragment$_meth() {
                          var dis___ = this && this.___? void 0: this;
                          return null !== (dis___.fragment__v___?
                            dis___.fragment_: dis___.v___('fragment_'));
                        }
                        return ___.f(hasFragment$_meth, 'hasFragment$_meth');
                      })(), x84___.hasFragment_w___ === x84___?
                      (x84___.hasFragment = x85___): x84___.w___('hasFragment',
                      x85___);
                    URI_RE_ = new (IMPORTS___.RegExp_v___? IMPORTS___.RegExp:
                      ___.ri(IMPORTS___, 'RegExp')).new___(('^' + '(?:' +
                        '([^:/?#]+)' + ':)?' + '(?://' + '(?:([^/?#]*)@)?' +
                        '([^/?#:@]*)' + '(?::([0-9]+))?' + ')?' + '([^?#]+)?' +
                        '(?:\\?([^#]*))?' + '(?:#(.*))?' + '$'));
                    URI_DISALLOWED_IN_SCHEME_OR_CREDENTIALS_ = new
                      RegExp.new___('[#\\/\\?@]', 'g');
                    URI_DISALLOWED_IN_PATH_ = new RegExp.new___('[\\#\\?]',
                      'g');
                    URI.parse_w___ === URI? (URI.parse = parse):
                    URI.w___('parse', parse);
                    URI.create_w___ === URI? (URI.create = create):
                    URI.w___('create', create);
                    URI.resolve_w___ === URI? (URI.resolve = resolve):
                    URI.w___('resolve', resolve);
                    URI.collapse_dots_w___ === URI? (URI.collapse_dots =
                        collapse_dots): URI.w___('collapse_dots',
                      collapse_dots);
                    x87___ = URI, x86___ = ___.iM([ 'mimeTypeOf', (function ()
                            {
                            function mimeTypeOf$_lit(uri) {
                              var uriObj, x0___, x1___;
                              uriObj = parse.i___(uri);
                              if (x0___ = new RegExp.new___('\\.html$'), x1___
                                = uriObj.getPath_m___? uriObj.getPath():
                                uriObj.m___('getPath', [ ]), x0___.test_m___?
                                x0___.test(x1___): x0___.m___('test', [ x1___ ]
                                )) {
                                return 'text/html';
                              } else {
                                return 'application/javascript';
                              }
                            }
                            return ___.f(mimeTypeOf$_lit, 'mimeTypeOf$_lit');
                          })(), 'resolve', (function () {
                            function resolve$_lit(base, uri) {
                              var x0___;
                              if (base) {
                                return x0___ = resolve.i___(parse.i___(base),
                                  parse.i___(uri)), x0___.toString_m___?
                                  x0___.toString(): x0___.m___('toString', [ ])
                                  ;
                              } else {
                                return '' + uri;
                              }
                            }
                            return ___.f(resolve$_lit, 'resolve$_lit');
                          })() ]), x87___.utils_w___ === x87___? (x87___.utils
                      = x86___): x87___.w___('utils', x86___);
                    return URI;
                  }).i___());
              if (typeof IMPORTS___.v___('window') !== 'undefined') {
                moduleResult___ = (IMPORTS___.window_v___? IMPORTS___.window:
                  ___.ri(IMPORTS___, 'window')).w___('URI',
                  IMPORTS___.URI_v___? IMPORTS___.URI: ___.ri(IMPORTS___,
                    'URI'));
              }
              return moduleResult___;
            },
            'cajolerName': 'com.google.caja',
            'cajolerVersion': '5054',
            'cajoledDate': 1347987482308
          }).instantiate___(___, IMPORTS___), ___.prepareModule({
            'instantiate': function (___, IMPORTS___) {
              var dis___ = IMPORTS___;
              var moduleResult___;
              moduleResult___ = ___.NO_RESULT;
              ___.di(IMPORTS___, 'Q');
              ___.f(function () {
                  var promises, reject, ref, enqueue, forward, promised, defer,
                  x0___, x1___;
                  {
                    function reject(reason) {
                      var rejected;
                      {
                        function rejected(op, arg1, arg2, arg3) {
                          if ((IMPORTS___.undefined_v___? IMPORTS___.undefined:
                              ___.ri(IMPORTS___, 'undefined')) === op) { return rejected; }
                          if ('WHEN' === op) {
                            return arg2? arg2.i___(reason): reject.i___(reason)
                              ;
                          }
                          return arg1? arg1.i___(reject.i___(reason)):
                          reject.i___(reason);
                        }
                        ___.f(rejected, 'rejected');
                      }
                      rejected.reason_w___ === rejected? (rejected.reason =
                        reason): rejected.w___('reason', reason);
                      promises.set_m___? promises.set(rejected, true):
                      promises.m___('set', [ rejected, true ]);
                      return rejected;
                    }
                    ___.f(reject, 'reject');
                  }
                  {
                    function ref(value) {
                      var fulfilled;
                      {
                        function fulfilled(op, arg1, arg2, arg3) {
                          var r, x0___, x1___, x2___, x3___, x4___;
                          if ((IMPORTS___.undefined_v___? IMPORTS___.undefined:
                              ___.ri(IMPORTS___, 'undefined')) === op) { return value; }
                          switch (op) {
                          case 'WHEN':
                            r = value;
                            break;
                          case 'GET':
                            if ((IMPORTS___.undefined_v___?
                                IMPORTS___.undefined: ___.ri(IMPORTS___,
                                  'undefined')) === arg2 || null === arg2) { r
                                = value; } else {
                              r = value.v___(arg2);
                            }
                            break;
                          case 'POST':
                            if ((IMPORTS___.undefined_v___?
                                IMPORTS___.undefined: ___.ri(IMPORTS___,
                                  'undefined')) === arg2 || null === arg2) {
                              r = reject.i___(___.iM([ ]));
                            } else {
                              r = (x0___ = value.v___(arg2), x1___ = value,
                                x2___ = arg3, x0___.apply_m___?
                                x0___.apply(x1___, x2___): x0___.m___('apply',
                                  [ x1___, x2___ ]));
                            }
                            break;
                          case 'PUT':
                            if ((IMPORTS___.undefined_v___?
                                IMPORTS___.undefined: ___.ri(IMPORTS___,
                                  'undefined')) === arg2 || null === arg2) {
                              r = reject.i___(___.iM([ ]));
                            } else {
                              value.w___(arg2, arg3);
                              r = ___.iM([ ]);
                            }
                            break;
                          case 'DELETE':
                            if ((IMPORTS___.undefined_v___?
                                IMPORTS___.undefined: ___.ri(IMPORTS___,
                                  'undefined')) === arg2 || null === arg2) {
                              r = reject.i___(___.iM([ ]));
                            } else {
                              value.c___(arg2);
                              r = ___.iM([ ]);
                            }
                            break;
                          default:
                            r = reject.i___(___.iM([ ]));
                          }
                          return arg1? (x4___ = arg1, x3___ = [ r ],
                            x4___.apply_m___? x4___.apply(null, x3___):
                            x4___.m___('apply', [ null, x3___ ])): r;
                        }
                        ___.f(fulfilled, 'fulfilled');
                      }
                      if (null === value || (IMPORTS___.undefined_v___?
                            IMPORTS___.undefined: ___.ri(IMPORTS___,
                            'undefined')) === value) {
                        return reject.i___(___.iM([ 'class', [ 'NaO' ] ]));
                      }
                      if ('number' === typeof value && !
                        (IMPORTS___.isFinite_v___? IMPORTS___.isFinite:
                         ___.ri(IMPORTS___, 'isFinite')).i___(value)) {
                        return reject.i___(___.iM([ 'class', [ 'NaN' ] ]));
                      }
                      promises.set_m___? promises.set(fulfilled, true):
                      promises.m___('set', [ fulfilled, true ]);
                      return fulfilled;
                    }
                    ___.f(ref, 'ref');
                  }
                  {
                    function forward(p, op, arg1, arg2, arg3) {
                      enqueue.i___(___.f(function () {
                            p.i___(op, arg1, arg2, arg3);
                          }));
                    }
                    ___.f(forward, 'forward');
                  }
                  {
                    function promised(value) {
                      return 'function' === typeof value && (promises.get_m___?
                          promises.get(value): promises.m___('get', [ value ]))
                        ? value: ref.i___(value);
                    }
                    ___.f(promised, 'promised');
                  }
                  {
                    function defer() {
                      var value, pending, promise, x0___, x1___;
                      {
                        function promise(op, arg1, arg2, arg3) {
                          var x0___, x1___;
                          if ((IMPORTS___.undefined_v___? IMPORTS___.undefined:
                              ___.ri(IMPORTS___, 'undefined')) === op) {
                            return pending? promise: value.i___();
                          }
                          if (pending) {
                            x1___ = pending, x0___ = ___.iM([ 'op', op, 'arg1',
                                arg1, 'arg2', arg2, 'arg3', arg3 ]),
                            x1___.push_m___? x1___.push(x0___):
                            x1___.m___('push', [ x0___ ]);
                          } else {
                            forward.i___(value, op, arg1, arg2, arg3);
                          }
                        }
                        ___.f(promise, 'promise');
                      }
                      pending = [ ];
                      promises.set_m___? promises.set(promise, true):
                      promises.m___('set', [ promise, true ]);
                      return x0___ = IMPORTS___.cajaVM_v___? IMPORTS___.cajaVM:
                      ___.ri(IMPORTS___, 'cajaVM'), x1___ = ___.iM([ 'promise',
                          promise, 'resolve', (function () {
                              function resolve$_lit(p) {
                                var todo, i, x;
                                if (!pending) { return; }
                                todo = pending;
                                pending = null;
                                value = promised.i___(p);
                                for (i = 0; i !== todo.length; i = i + 1) {
                                  x = todo[ + +i ];
                                  forward.i___(value, x.op_v___? x.op:
                                    x.v___('op'), x.arg1_v___? x.arg1:
                                    x.v___('arg1'), x.arg2_v___? x.arg2:
                                    x.v___('arg2'), x.arg3_v___? x.arg3:
                                    x.v___('arg3'));
                                }
                              }
                              return ___.f(resolve$_lit, 'resolve$_lit');
                            })() ]), x0___.def_m___? x0___.def(x1___):
                      x0___.m___('def', [ x1___ ]);
                    }
                    ___.f(defer, 'defer');
                  }
                  promises = new (IMPORTS___.WeakMap_v___? IMPORTS___.WeakMap:
                    ___.ri(IMPORTS___, 'WeakMap')).new___(true);
                  enqueue = ___.f(function () {
                      var active, pending, run;
                      active = false;
                      pending = [ ];
                      run = (function () {
                          function run$_var() {
                            var task;
                            task = pending.shift_m___? pending.shift():
                            pending.m___('shift', [ ]);
                            if (0 === pending.length) { active = false; } else
                              {
                              (IMPORTS___.setTimeout_v___?
                                 IMPORTS___.setTimeout: ___.ri(IMPORTS___,
                                 'setTimeout')).i___(run, 0);
                            }
                            task.i___();
                          }
                          return ___.f(run$_var, 'run$_var');
                        })();
                      return ___.f(function (task) {
                          pending.push_m___? pending.push(task):
                          pending.m___('push', [ task ]);
                          if (!active) {
                            (IMPORTS___.setTimeout_v___? IMPORTS___.setTimeout:
                             ___.ri(IMPORTS___, 'setTimeout')).i___(run, 0);
                            active = true;
                          }
                        });
                    }).i___();
                  IMPORTS___.Q_w___? (IMPORTS___.Q = (x0___ =
                      IMPORTS___.cajaVM_v___? IMPORTS___.cajaVM:
                      ___.ri(IMPORTS___, 'cajaVM'), x1___ = ___.iM([ 'run',
                          enqueue, 'reject', reject, 'ref', ref, 'defer',
                          defer, 'near', (function () {
                              function near$_lit(value) {
                                return 'function' === typeof value &&
                                  (promises.get_m___? promises.get(value):
                                  promises.m___('get', [ value ])) ?
                                  value.i___(): value;
                              }
                              return ___.f(near$_lit, 'near$_lit');
                            })(), 'when', (function () {
                              function when$_lit(value, fulfilled, rejected) {
                                var r, done;
                                r = defer.i___();
                                done = false;
                                forward.i___(promised.i___(value), 'WHEN',
                                  ___.f(function (x) {
                                      var x0___, x1___;
                                      if (done) {
                                        throw new (IMPORTS___.Error_v___?
                                            IMPORTS___.Error:
                                          ___.ri(IMPORTS___, 'Error')).new___;
                                      }
                                      done = true;
                                      x1___ = r, x0___ = ref.i___(x)
                                        .i___('WHEN', fulfilled, rejected),
                                      x1___.resolve_m___? x1___.resolve(x0___):
                                      x1___.m___('resolve', [ x0___ ]);
                                    }), ___.f(function (reason) {
                                      var x0___, x1___, x2___, x3___;
                                      if (done) {
                                        throw new (IMPORTS___.Error_v___?
                                            IMPORTS___.Error:
                                          ___.ri(IMPORTS___, 'Error')).new___;
                                      }
                                      done = true;
                                      x3___ = r, x2___ = rejected? (x1___ =
                                        rejected, x0___ = [ reason ],
                                        x1___.apply_m___? x1___.apply(null,
                                          x0___): x1___.m___('apply', [ null,
                                            x0___ ])): reject.i___(reason),
                                      x3___.resolve_m___? x3___.resolve(x2___):
                                      x3___.m___('resolve', [ x2___ ]);
                                    }));
                                return r.promise_v___? r.promise:
                                r.v___('promise');
                              }
                              return ___.f(when$_lit, 'when$_lit');
                            })(), 'get', (function () {
                              function get$_lit(target, noun) {
                                var r;
                                r = defer.i___();
                                forward.i___(promised.i___(target), 'GET',
                                  r.resolve_v___? r.resolve: r.v___('resolve'),
                                  noun);
                                return r.promise_v___? r.promise:
                                r.v___('promise');
                              }
                              return ___.f(get$_lit, 'get$_lit');
                            })(), 'post', (function () {
                              function post$_lit(target, verb, argv) {
                                var r;
                                r = defer.i___();
                                forward.i___(promised.i___(target), 'POST',
                                  r.resolve_v___? r.resolve: r.v___('resolve'),
                                  verb, argv);
                                return r.promise_v___? r.promise:
                                r.v___('promise');
                              }
                              return ___.f(post$_lit, 'post$_lit');
                            })(), 'put', (function () {
                              function put$_lit(target, noun, value) {
                                var r;
                                r = defer.i___();
                                forward.i___(promised.i___(target), 'PUT',
                                  r.resolve_v___? r.resolve: r.v___('resolve'),
                                  noun, value);
                                return r.promise_v___? r.promise:
                                r.v___('promise');
                              }
                              return ___.f(put$_lit, 'put$_lit');
                            })(), 'remove', (function () {
                              function remove$_lit(target, noun) {
                                var r;
                                r = defer.i___();
                                forward.i___(promised.i___(target), 'DELETE',
                                  r.resolve_v___? r.resolve: r.v___('resolve'),
                                  noun);
                                return r.promise_v___? r.promise:
                                r.v___('promise');
                              }
                              return ___.f(remove$_lit, 'remove$_lit');
                            })() ]), x0___.def_m___? x0___.def(x1___):
                      x0___.m___('def', [ x1___ ]))): ___.wi(IMPORTS___, 'Q',
                    (x0___ = IMPORTS___.cajaVM_v___? IMPORTS___.cajaVM:
                     ___.ri(IMPORTS___, 'cajaVM'), x1___ = ___.iM([ 'run',
                         enqueue, 'reject', reject, 'ref', ref, 'defer', defer,
                         'near', (function () {
                             function near$_lit(value) {
                               return 'function' === typeof value &&
                                 (promises.get_m___? promises.get(value):
                                 promises.m___('get', [ value ])) ?
                                 value.i___(): value;
                             }
                             return ___.f(near$_lit, 'near$_lit');
                           })(), 'when', (function () {
                             function when$_lit(value, fulfilled, rejected) {
                               var r, done;
                               r = defer.i___();
                               done = false;
                               forward.i___(promised.i___(value), 'WHEN',
                                 ___.f(function (x) {
                                     var x0___, x1___;
                                     if (done) {
                                       throw new (IMPORTS___.Error_v___?
                                           IMPORTS___.Error: ___.ri(IMPORTS___,
                                           'Error')).new___;
                                     }
                                     done = true;
                                     x1___ = r, x0___ = ref.i___(x)
                                       .i___('WHEN', fulfilled, rejected),
                                     x1___.resolve_m___? x1___.resolve(x0___):
                                     x1___.m___('resolve', [ x0___ ]);
                                   }), ___.f(function (reason) {
                                     var x0___, x1___, x2___, x3___;
                                     if (done) {
                                       throw new (IMPORTS___.Error_v___?
                                           IMPORTS___.Error: ___.ri(IMPORTS___,
                                           'Error')).new___;
                                     }
                                     done = true;
                                     x3___ = r, x2___ = rejected? (x1___ =
                                       rejected, x0___ = [ reason ],
                                       x1___.apply_m___? x1___.apply(null,
                                         x0___): x1___.m___('apply', [ null,
                                           x0___ ])): reject.i___(reason),
                                     x3___.resolve_m___? x3___.resolve(x2___):
                                     x3___.m___('resolve', [ x2___ ]);
                                   }));
                               return r.promise_v___? r.promise:
                               r.v___('promise');
                             }
                             return ___.f(when$_lit, 'when$_lit');
                           })(), 'get', (function () {
                             function get$_lit(target, noun) {
                               var r;
                               r = defer.i___();
                               forward.i___(promised.i___(target), 'GET',
                                 r.resolve_v___? r.resolve: r.v___('resolve'),
                                 noun);
                               return r.promise_v___? r.promise:
                               r.v___('promise');
                             }
                             return ___.f(get$_lit, 'get$_lit');
                           })(), 'post', (function () {
                             function post$_lit(target, verb, argv) {
                               var r;
                               r = defer.i___();
                               forward.i___(promised.i___(target), 'POST',
                                 r.resolve_v___? r.resolve: r.v___('resolve'),
                                 verb, argv);
                               return r.promise_v___? r.promise:
                               r.v___('promise');
                             }
                             return ___.f(post$_lit, 'post$_lit');
                           })(), 'put', (function () {
                             function put$_lit(target, noun, value) {
                               var r;
                               r = defer.i___();
                               forward.i___(promised.i___(target), 'PUT',
                                 r.resolve_v___? r.resolve: r.v___('resolve'),
                                 noun, value);
                               return r.promise_v___? r.promise:
                               r.v___('promise');
                             }
                             return ___.f(put$_lit, 'put$_lit');
                           })(), 'remove', (function () {
                             function remove$_lit(target, noun) {
                               var r;
                               r = defer.i___();
                               forward.i___(promised.i___(target), 'DELETE',
                                 r.resolve_v___? r.resolve: r.v___('resolve'),
                                 noun);
                               return r.promise_v___? r.promise:
                               r.v___('promise');
                             }
                             return ___.f(remove$_lit, 'remove$_lit');
                           })() ]), x0___.def_m___? x0___.def(x1___):
                     x0___.m___('def', [ x1___ ])));
                }).i___();
              if (typeof IMPORTS___.v___('window') !== 'undefined') {
                moduleResult___ = (IMPORTS___.window_v___? IMPORTS___.window:
                  ___.ri(IMPORTS___, 'window')).w___('Q', IMPORTS___.Q_v___?
                    IMPORTS___.Q: ___.ri(IMPORTS___, 'Q'));
              }
              return moduleResult___;
            },
            'cajolerName': 'com.google.caja',
            'cajolerVersion': '5054',
            'cajoledDate': 1347987482731
          }).instantiate___(___, IMPORTS___), ___.prepareModule({
            'instantiate': function (___, IMPORTS___) {
              var dis___ = IMPORTS___;
              var moduleResult___, x0___, x1___, x2___, x3___, x4___, x5___,
              x6___, x7___, x8___, x9___, x10___, x11___, x12___;
              moduleResult___ = ___.NO_RESULT;
              ___.di(IMPORTS___, 'unicode'), IMPORTS___.w___('unicode',
                ___.iM([ ]));
              x0___ = IMPORTS___.unicode_v___? IMPORTS___.unicode:
              ___.ri(IMPORTS___, 'unicode'), x1___ =
                'A-Za-z\xc0-\xd6\xd8-\xf6\xf8-\xff' +
                '\u0100-\u0131\u0134-\u013e\u0141-\u0148\u014a-\u017e\u0180-\u01c3'
                +
                '\u01cd-\u01f0\u01f4-\u01f5\u01fa-\u0217\u0250-\u02a8\u02bb-\u02c1'
                +
                '\u0386\u0388-\u038a\u038c\u038e-\u03a1\u03a3-\u03ce\u03d0-\u03d6'
                +
                '\u03da\u03dc\u03de\u03e0\u03e2-\u03f3\u0401-\u040c\u040e-\u044f'
                +
                '\u0451-\u045c\u045e-\u0481\u0490-\u04c4\u04c7-\u04c8\u04cb-\u04cc'
                + '\u04d0-\u04eb\u04ee-\u04f5\u04f8-\u04f9\u0531-\u0556\u0559'
                +
                '\u0561-\u0586\u05d0-\u05ea\u05f0-\u05f2\u0621-\u063a\u0641-\u064a'
                + '\u0671-\u06b7\u06ba-\u06be\u06c0-\u06ce\u06d0-\u06d3\u06d5'
                + '\u06e5-\u06e6\u0905-\u0939\u093d\u0958-\u0961\u0985-\u098c'
                + '\u098f-\u0990\u0993-\u09a8\u09aa-\u09b0\u09b2\u09b6-\u09b9'
                +
                '\u09dc-\u09dd\u09df-\u09e1\u09f0-\u09f1\u0a05-\u0a0a\u0a0f-\u0a10'
                +
                '\u0a13-\u0a28\u0a2a-\u0a30\u0a32-\u0a33\u0a35-\u0a36\u0a38-\u0a39'
                +
                '\u0a59-\u0a5c\u0a5e\u0a72-\u0a74\u0a85-\u0a8b\u0a8d\u0a8f-\u0a91'
                +
                '\u0a93-\u0aa8\u0aaa-\u0ab0\u0ab2-\u0ab3\u0ab5-\u0ab9\u0abd\u0ae0'
                +
                '\u0b05-\u0b0c\u0b0f-\u0b10\u0b13-\u0b28\u0b2a-\u0b30\u0b32-\u0b33'
                + '\u0b36-\u0b39\u0b3d\u0b5c-\u0b5d\u0b5f-\u0b61\u0b85-\u0b8a'
                + '\u0b8e-\u0b90\u0b92-\u0b95\u0b99-\u0b9a\u0b9c\u0b9e-\u0b9f'
                +
                '\u0ba3-\u0ba4\u0ba8-\u0baa\u0bae-\u0bb5\u0bb7-\u0bb9\u0c05-\u0c0c'
                +
                '\u0c0e-\u0c10\u0c12-\u0c28\u0c2a-\u0c33\u0c35-\u0c39\u0c60-\u0c61'
                +
                '\u0c85-\u0c8c\u0c8e-\u0c90\u0c92-\u0ca8\u0caa-\u0cb3\u0cb5-\u0cb9'
                + '\u0cde\u0ce0-\u0ce1\u0d05-\u0d0c\u0d0e-\u0d10\u0d12-\u0d28'
                + '\u0d2a-\u0d39\u0d60-\u0d61\u0e01-\u0e2e\u0e30\u0e32-\u0e33'
                + '\u0e40-\u0e45\u0e81-\u0e82\u0e84\u0e87-\u0e88\u0e8a\u0e8d' +
                '\u0e94-\u0e97\u0e99-\u0e9f\u0ea1-\u0ea3\u0ea5\u0ea7\u0eaa-\u0eab'
                +
                '\u0ead-\u0eae\u0eb0\u0eb2-\u0eb3\u0ebd\u0ec0-\u0ec4\u0f40-\u0f47'
                + '\u0f49-\u0f69\u10a0-\u10c5\u10d0-\u10f6\u1100\u1102-\u1103'
                +
                '\u1105-\u1107\u1109\u110b-\u110c\u110e-\u1112\u113c\u113e\u1140'
                +
                '\u114c\u114e\u1150\u1154-\u1155\u1159\u115f-\u1161\u1163\u1165'
                +
                '\u1167\u1169\u116d-\u116e\u1172-\u1173\u1175\u119e\u11a8\u11ab'
                +
                '\u11ae-\u11af\u11b7-\u11b8\u11ba\u11bc-\u11c2\u11eb\u11f0\u11f9'
                +
                '\u1e00-\u1e9b\u1ea0-\u1ef9\u1f00-\u1f15\u1f18-\u1f1d\u1f20-\u1f45'
                + '\u1f48-\u1f4d\u1f50-\u1f57\u1f59\u1f5b\u1f5d\u1f5f-\u1f7d' +
                '\u1f80-\u1fb4\u1fb6-\u1fbc\u1fbe\u1fc2-\u1fc4\u1fc6-\u1fcc' +
                '\u1fd0-\u1fd3\u1fd6-\u1fdb\u1fe0-\u1fec\u1ff2-\u1ff4\u1ff6-\u1ffc'
                +
                '\u2126\u212a-\u212b\u212e\u2180-\u2182\u3041-\u3094\u30a1-\u30fa'
                + '\u3105-\u312c\uac00-\ud7a3', x0___.BASE_CHAR_w___ === x0___?
                (x0___.BASE_CHAR = x1___): x0___.w___('BASE_CHAR', x1___);
              x2___ = IMPORTS___.unicode_v___? IMPORTS___.unicode:
              ___.ri(IMPORTS___, 'unicode'), x2___.IDEOGRAPHIC_w___ === x2___?
                (x2___.IDEOGRAPHIC = '\u4e00-\u9fa5\u3007\u3021-\u3029'):
              x2___.w___('IDEOGRAPHIC', '\u4e00-\u9fa5\u3007\u3021-\u3029');
              x3___ = IMPORTS___.unicode_v___? IMPORTS___.unicode:
              ___.ri(IMPORTS___, 'unicode'), x6___ = (x4___ =
                IMPORTS___.unicode_v___? IMPORTS___.unicode: ___.ri(IMPORTS___,
                  'unicode'), x4___.BASE_CHAR_v___? x4___.BASE_CHAR:
                x4___.v___('BASE_CHAR')) + (x5___ = IMPORTS___.unicode_v___?
                IMPORTS___.unicode: ___.ri(IMPORTS___, 'unicode'),
                x5___.IDEOGRAPHIC_v___? x5___.IDEOGRAPHIC:
                x5___.v___('IDEOGRAPHIC')), x3___.LETTER_w___ === x3___?
                (x3___.LETTER = x6___): x3___.w___('LETTER', x6___);
              x7___ = IMPORTS___.unicode_v___? IMPORTS___.unicode:
              ___.ri(IMPORTS___, 'unicode'), x8___ =
                '\u0300-\u0345\u0360-\u0361\u0483-\u0486\u0591-\u05a1\u05a3-\u05b9'
                + '\u05bb-\u05bd\u05bf\u05c1-\u05c2\u05c4\u064b-\u0652\u0670' +
                '\u06d6-\u06dc\u06dd-\u06df\u06e0-\u06e4\u06e7-\u06e8\u06ea-\u06ed'
                +
                '\u0901-\u0903\u093c\u093e-\u094c\u094d\u0951-\u0954\u0962-\u0963'
                + '\u0981-\u0983\u09bc\u09be\u09bf\u09c0-\u09c4\u09c7-\u09c8' +
                '\u09cb-\u09cd\u09d7\u09e2-\u09e3\u0a02\u0a3c\u0a3e\u0a3f' +
                '\u0a40-\u0a42\u0a47-\u0a48\u0a4b-\u0a4d\u0a70-\u0a71\u0a81-\u0a83'
                +
                '\u0abc\u0abe-\u0ac5\u0ac7-\u0ac9\u0acb-\u0acd\u0b01-\u0b03\u0b3c'
                +
                '\u0b3e-\u0b43\u0b47-\u0b48\u0b4b-\u0b4d\u0b56-\u0b57\u0b82-\u0b83'
                + '\u0bbe-\u0bc2\u0bc6-\u0bc8\u0bca-\u0bcd\u0bd7\u0c01-\u0c03'
                +
                '\u0c3e-\u0c44\u0c46-\u0c48\u0c4a-\u0c4d\u0c55-\u0c56\u0c82-\u0c83'
                +
                '\u0cbe-\u0cc4\u0cc6-\u0cc8\u0cca-\u0ccd\u0cd5-\u0cd6\u0d02-\u0d03'
                +
                '\u0d3e-\u0d43\u0d46-\u0d48\u0d4a-\u0d4d\u0d57\u0e31\u0e34-\u0e3a'
                + '\u0e47-\u0e4e\u0eb1\u0eb4-\u0eb9\u0ebb-\u0ebc\u0ec8-\u0ecd'
                + '\u0f18-\u0f19\u0f35\u0f37\u0f39\u0f3e\u0f3f\u0f71-\u0f84' +
                '\u0f86-\u0f8b\u0f90-\u0f95\u0f97\u0f99-\u0fad\u0fb1-\u0fb7\u0fb9'
                + '\u20d0-\u20dc\u20e1\u302a-\u302f\u3099\u309a',
              x7___.COMBINING_CHAR_w___ === x7___? (x7___.COMBINING_CHAR =
                  x8___): x7___.w___('COMBINING_CHAR', x8___);
              x9___ = IMPORTS___.unicode_v___? IMPORTS___.unicode:
              ___.ri(IMPORTS___, 'unicode'), x10___ =
                '0-9\u0660-\u0669\u06f0-\u06f9\u0966-\u096f\u09e6-\u09ef' +
                '\u0a66-\u0a6f\u0ae6-\u0aef\u0b66-\u0b6f\u0be7-\u0bef\u0c66-\u0c6f'
                +
                '\u0ce6-\u0cef\u0d66-\u0d6f\u0e50-\u0e59\u0ed0-\u0ed9\u0f20-\u0f29',
              x9___.DIGIT_w___ === x9___? (x9___.DIGIT = x10___):
              x9___.w___('DIGIT', x10___);
              x11___ = IMPORTS___.unicode_v___? IMPORTS___.unicode:
              ___.ri(IMPORTS___, 'unicode'), x12___ =
                '\xb7\u02d0\u02d1\u0387\u0640\u0e46\u0ec6\u3005\u3031-\u3035' +
                '\u309d-\u309e\u30fc-\u30fe', x11___.EXTENDER_w___ === x11___?
                (x11___.EXTENDER = x12___): x11___.w___('EXTENDER', x12___);
              if (typeof IMPORTS___.v___('window') !== 'undefined') {
                moduleResult___ = (IMPORTS___.window_v___? IMPORTS___.window:
                  ___.ri(IMPORTS___, 'window')).w___('unicode',
                  IMPORTS___.unicode_v___? IMPORTS___.unicode:
                  ___.ri(IMPORTS___, 'unicode'));
              }
              return moduleResult___;
            },
            'cajolerName': 'com.google.caja',
            'cajolerVersion': '5054',
            'cajoledDate': 1347987482750
          }).instantiate___(___, IMPORTS___), ___.prepareModule({
            'instantiate': function (___, IMPORTS___) {
              var dis___ = IMPORTS___;
              var moduleResult___;
              moduleResult___ = ___.NO_RESULT;
              ___.di(IMPORTS___, 'CSS_PROP_BIT_QUANTITY'),
              IMPORTS___.w___('CSS_PROP_BIT_QUANTITY', 1);
              ___.di(IMPORTS___, 'CSS_PROP_BIT_HASH_VALUE'),
              IMPORTS___.w___('CSS_PROP_BIT_HASH_VALUE', 2);
              ___.di(IMPORTS___, 'CSS_PROP_BIT_NEGATIVE_QUANTITY'),
              IMPORTS___.w___('CSS_PROP_BIT_NEGATIVE_QUANTITY', 4);
              ___.di(IMPORTS___, 'CSS_PROP_BIT_QSTRING_CONTENT'),
              IMPORTS___.w___('CSS_PROP_BIT_QSTRING_CONTENT', 8);
              ___.di(IMPORTS___, 'CSS_PROP_BIT_QSTRING_URL'),
              IMPORTS___.w___('CSS_PROP_BIT_QSTRING_URL', 16);
              ___.di(IMPORTS___, 'CSS_PROP_BIT_HISTORY_INSENSITIVE'),
              IMPORTS___.w___('CSS_PROP_BIT_HISTORY_INSENSITIVE', 32);
              ___.di(IMPORTS___, 'CSS_PROP_BIT_Z_INDEX'),
              IMPORTS___.w___('CSS_PROP_BIT_Z_INDEX', 64);
              ___.di(IMPORTS___, 'CSS_PROP_BIT_ALLOWED_IN_LINK'),
              IMPORTS___.w___('CSS_PROP_BIT_ALLOWED_IN_LINK', 128);
              ___.di(IMPORTS___, 'cssSchema'), IMPORTS___.w___('cssSchema',
                ___.f(function () {
                    var s, c, L;
                    s = [
                      'rgb(?:\\(\\s*(?:\\d+|0|\\d+(?:\\.\\d+)?%)\\s*,\\s*(?:\\d+|0|\\d+(?:\\.\\d+)?%)\\s*,\\s*(?:\\d+|0|\\d+(?:\\.\\d+)?%)|a\\(\\s*(?:\\d+|0|\\d+(?:\\.\\d+)?%)\\s*,\\s*(?:\\d+|0|\\d+(?:\\.\\d+)?%)\\s*,\\s*(?:\\d+|0|\\d+(?:\\.\\d+)?%)\\s*,\\s*(?:\\d+|0(?:\\.\\d+)?|\\.\\d+|1(?:\\.0+)?|0|\\d+(?:\\.\\d+)?%)) *\\)'
                    ], c = [ new RegExp.new___('^ *$', 'i'),
                      (IMPORTS___.RegExp_v___? IMPORTS___.RegExp:
                       ___.ri(IMPORTS___, 'RegExp')).i___('^ *\\s*' + s[ +0 ] +
                        ' *$', 'i'), (IMPORTS___.RegExp_v___?
                          IMPORTS___.RegExp: ___.ri(IMPORTS___, 'RegExp'))
                      .i___('^ *(?:\\s*' + s[ +0 ] + '|(?:\\s*' + s[ +0 ] +
                          ')?)+ *$', 'i') ], L = [ [ 'aliceblue',
                        'antiquewhite', 'aqua', 'aquamarine', 'azure', 'beige',
                        'bisque', 'black', 'blanchedalmond', 'blue',
                        'blueviolet', 'brown', 'burlywood', 'cadetblue',
                        'chartreuse', 'chocolate', 'coral', 'cornflowerblue',
                        'cornsilk', 'crimson', 'cyan', 'darkblue', 'darkcyan',
                        'darkgoldenrod', 'darkgray', 'darkgreen', 'darkkhaki',
                        'darkmagenta', 'darkolivegreen', 'darkorange',
                        'darkorchid', 'darkred', 'darksalmon', 'darkseagreen',
                        'darkslateblue', 'darkslategray', 'darkturquoise',
                        'darkviolet', 'deeppink', 'deepskyblue', 'dimgray',
                        'dodgerblue', 'firebrick', 'floralwhite',
                        'forestgreen', 'fuchsia', 'gainsboro', 'ghostwhite',
                        'gold', 'goldenrod', 'gray', 'green', 'greenyellow',
                        'honeydew', 'hotpink', 'indianred', 'indigo', 'ivory',
                        'khaki', 'lavender', 'lavenderblush', 'lawngreen',
                        'lemonchiffon', 'lightblue', 'lightcoral', 'lightcyan',
                        'lightgoldenrodyellow', 'lightgreen', 'lightgrey',
                        'lightpink', 'lightsalmon', 'lightseagreen',
                        'lightskyblue', 'lightslategray', 'lightsteelblue',
                        'lightyellow', 'lime', 'limegreen', 'linen', 'magenta',
                        'maroon', 'mediumaquamarine', 'mediumblue',
                        'mediumorchid', 'mediumpurple', 'mediumseagreen',
                        'mediumslateblue', 'mediumspringgreen',
                        'mediumturquoise', 'mediumvioletred', 'midnightblue',
                        'mintcream', 'mistyrose', 'moccasin', 'navajowhite',
                        'navy', 'oldlace', 'olive', 'olivedrab', 'orange',
                        'orangered', 'orchid', 'palegoldenrod', 'palegreen',
                        'paleturquoise', 'palevioletred', 'papayawhip',
                        'peachpuff', 'peru', 'pink', 'plum', 'powderblue',
                        'purple', 'red', 'rosybrown', 'royalblue',
                        'saddlebrown', 'salmon', 'sandybrown', 'seagreen',
                        'seashell', 'sienna', 'silver', 'skyblue', 'slateblue',
                        'slategray', 'snow', 'springgreen', 'steelblue', 'tan',
                        'teal', 'thistle', 'tomato', 'turquoise', 'violet',
                        'wheat', 'white', 'whitesmoke', 'yellow', 'yellowgreen'
                      ], [ 'all-scroll', 'col-resize', 'crosshair', 'default',
                        'e-resize', 'hand', 'help', 'move', 'n-resize',
                        'ne-resize', 'no-drop', 'not-allowed', 'nw-resize',
                        'pointer', 'progress', 'row-resize', 's-resize',
                        'se-resize', 'sw-resize', 'text', 'vertical-text',
                        'w-resize', 'wait' ], [ '-moz-inline-box',
                        '-moz-inline-stack', 'block', 'inline', 'inline-block',
                        'inline-table', 'list-item', 'run-in', 'table',
                        'table-caption', 'table-cell', 'table-column',
                        'table-column-group', 'table-footer-group',
                        'table-header-group', 'table-row', 'table-row-group' ],
                      [ 'armenian', 'circle', 'decimal',
                       'decimal-leading-zero', 'disc', 'georgian',
                       'lower-alpha', 'lower-greek', 'lower-latin',
                       'lower-roman', 'square', 'upper-alpha', 'upper-latin',
                       'upper-roman' ], [ '100', '200', '300', '400', '500',
                        '600', '700', '800', '900', 'bold', 'bolder', 'lighter'
                      ], [ 'condensed', 'expanded', 'extra-condensed',
                        'extra-expanded', 'narrower', 'semi-condensed',
                        'semi-expanded', 'ultra-condensed', 'ultra-expanded',
                        'wider' ], [ 'behind', 'center-left', 'center-right',
                        'far-left', 'far-right', 'left-side', 'leftwards',
                        'right-side', 'rightwards' ], [ 'large', 'larger',
                        'small', 'smaller', 'x-large', 'x-small', 'xx-large',
                        'xx-small' ], [ '-moz-pre-wrap', '-o-pre-wrap',
                        '-pre-wrap', 'nowrap', 'pre', 'pre-line', 'pre-wrap' ],
                      [ 'dashed', 'dotted', 'double', 'groove', 'outset',
                       'ridge', 'solid' ], [ 'baseline', 'middle', 'sub',
                        'super', 'text-bottom', 'text-top' ], [ 'caption',
                        'icon', 'menu', 'message-box', 'small-caption',
                        'status-bar' ], [ 'fast', 'faster', 'slow', 'slower',
                        'x-fast', 'x-slow' ], [ 'above', 'below', 'higher',
                        'level', 'lower' ], [ 'border-box', 'contain',
                        'content-box', 'cover', 'padding-box' ], [ 'cursive',
                        'fantasy', 'monospace', 'sans-serif', 'serif' ], [
                        'loud', 'silent', 'soft', 'x-loud', 'x-soft' ], [
                        'no-repeat', 'repeat-x', 'repeat-y', 'round', 'space' ]
                      , [ 'blink', 'line-through', 'overline', 'underline' ], [
                        'high', 'low', 'x-high', 'x-low' ], [ 'absolute',
                        'relative', 'static' ], [ 'capitalize', 'lowercase',
                        'uppercase' ], [ 'child', 'female', 'male' ], [
                        'bidi-override', 'embed' ], [ 'bottom', 'top' ], [
                        'clip', 'ellipsis' ], [ 'continuous', 'digits' ], [
                        'hide', 'show' ], [ 'inside', 'outside' ], [ 'italic',
                        'oblique' ], [ 'left', 'right' ], [ 'ltr', 'rtl' ], [
                        'no-content', 'no-display' ], [ 'suppress',
                        'unrestricted' ], [ 'thick', 'thin' ], [ ',' ], [ '/' ]
                      , [ 'always' ], [ 'auto' ], [ 'avoid' ], [ 'both' ], [
                        'break-word' ], [ 'center' ], [ 'code' ], [ 'collapse'
                      ], [ 'fixed' ], [ 'hidden' ], [ 'inherit' ], [ 'inset' ],
                      [ 'invert' ], [ 'justify' ], [ 'local' ], [ 'medium' ], [
                        'mix' ], [ 'none' ], [ 'normal' ], [ 'once' ], [
                        'repeat' ], [ 'scroll' ], [ 'separate' ], [
                        'small-caps' ], [ 'spell-out' ], [ 'transparent' ], [
                        'visible' ] ];
                    return ___.iM([ '-moz-border-radius', ___.iM([ 'cssExtra',
                            c[ +0 ], 'cssPropBits', 5, 'cssLitGroup', [ L[ +36
                              ] ] ]), '-moz-border-radius-bottomleft', ___.iM([
                            'cssExtra', c[ +0 ], 'cssPropBits', 5 ]),
                        '-moz-border-radius-bottomright', ___.iM([ 'cssExtra',
                            c[ +0 ], 'cssPropBits', 5 ]),
                        '-moz-border-radius-topleft', ___.iM([ 'cssExtra', c[
                              +0 ], 'cssPropBits', 5 ]),
                        '-moz-border-radius-topright', ___.iM([ 'cssExtra', c[
                              +0 ], 'cssPropBits', 5 ]), '-moz-box-shadow',
                        ___.iM([ 'cssExtra', c[ +2 ], 'cssAlternates', [
                              'boxShadow' ], 'cssPropBits', 7, 'cssLitGroup', [
                              L[ +0 ], L[ +35 ], L[ +48 ], L[ +54 ] ] ]),
                        '-moz-opacity', ___.iM([ 'cssPropBits', 1,
                            'cssLitGroup', [ L[ +47 ] ] ]), '-moz-outline',
                        ___.iM([ 'cssExtra', c[ +1 ], 'cssPropBits', 7,
                            'cssLitGroup', [ L[ +0 ], L[ +9 ], L[ +34 ], L[ +46
                              ], L[ +47 ], L[ +48 ], L[ +49 ], L[ +52 ], L[ +54
                              ] ] ]), '-moz-outline-color', ___.iM([
                            'cssExtra', c[ +1 ], 'cssPropBits', 2,
                            'cssLitGroup', [ L[ +0 ], L[ +47 ], L[ +49 ] ] ]),
                        '-moz-outline-style', ___.iM([ 'cssPropBits', 0,
                            'cssLitGroup', [ L[ +9 ], L[ +46 ], L[ +47 ], L[
                                +48 ], L[ +54 ] ] ]), '-moz-outline-width',
                        ___.iM([ 'cssPropBits', 5, 'cssLitGroup', [ L[ +34 ],
                              L[ +47 ], L[ +52 ] ] ]), '-o-text-overflow',
                        ___.iM([ 'cssPropBits', 0, 'cssLitGroup', [ L[ +25 ] ]
                          ]), '-webkit-border-bottom-left-radius', ___.iM([
                            'cssExtra', c[ +0 ], 'cssPropBits', 5 ]),
                        '-webkit-border-bottom-right-radius', ___.iM([
                            'cssExtra', c[ +0 ], 'cssPropBits', 5 ]),
                        '-webkit-border-radius', ___.iM([ 'cssExtra', c[ +0 ],
                            'cssPropBits', 5, 'cssLitGroup', [ L[ +36 ] ] ]),
                        '-webkit-border-radius-bottom-left', ___.iM([
                            'cssExtra', c[ +0 ], 'cssPropBits', 5 ]),
                        '-webkit-border-radius-bottom-right', ___.iM([
                            'cssExtra', c[ +0 ], 'cssPropBits', 5 ]),
                        '-webkit-border-radius-top-left', ___.iM([ 'cssExtra',
                            c[ +0 ], 'cssPropBits', 5 ]),
                        '-webkit-border-radius-top-right', ___.iM([ 'cssExtra',
                            c[ +0 ], 'cssPropBits', 5 ]),
                        '-webkit-border-top-left-radius', ___.iM([ 'cssExtra',
                            c[ +0 ], 'cssPropBits', 5 ]),
                        '-webkit-border-top-right-radius', ___.iM([ 'cssExtra',
                            c[ +0 ], 'cssPropBits', 5 ]), '-webkit-box-shadow',
                        ___.iM([ 'cssExtra', c[ +2 ], 'cssAlternates', [
                              'boxShadow' ], 'cssPropBits', 7, 'cssLitGroup', [
                              L[ +0 ], L[ +35 ], L[ +48 ], L[ +54 ] ] ]),
                        'azimuth', ___.iM([ 'cssPropBits', 5, 'cssLitGroup', [
                              L[ +6 ], L[ +30 ], L[ +42 ], L[ +47 ] ] ]),
                        'background', ___.iM([ 'cssExtra',
                            (IMPORTS___.RegExp_v___? IMPORTS___.RegExp:
                             ___.ri(IMPORTS___, 'RegExp')).i___('^ *(?:\\s*' +
                                s[ +0 ] + '){0,2} *$', 'i'), 'cssPropBits', 23,
                            'cssLitGroup', [ L[ +0 ], L[ +14 ], L[ +17 ], L[
                                +24 ], L[ +30 ], L[ +35 ], L[ +36 ], L[ +38 ],
                              L[ +42 ], L[ +45 ], L[ +47 ], L[ +51 ], L[ +54 ],
                              L[ +57 ], L[ +58 ], L[ +62 ] ] ]),
                        'background-attachment', ___.iM([ 'cssExtra', c[ +0 ],
                            'cssPropBits', 0, 'cssLitGroup', [ L[ +35 ], L[ +45
                              ], L[ +51 ], L[ +58 ] ] ]), 'background-color',
                        ___.iM([ 'cssExtra', c[ +1 ], 'cssPropBits', 130,
                            'cssLitGroup', [ L[ +0 ], L[ +47 ], L[ +62 ] ] ]),
                        'background-image', ___.iM([ 'cssExtra', c[ +0 ],
                            'cssPropBits', 16, 'cssLitGroup', [ L[ +35 ], L[
                                +54 ] ] ]), 'background-position', ___.iM([
                            'cssExtra', c[ +0 ], 'cssPropBits', 5,
                            'cssLitGroup', [ L[ +24 ], L[ +30 ], L[ +35 ], L[
                                +42 ] ] ]), 'background-repeat', ___.iM([
                            'cssExtra', c[ +0 ], 'cssPropBits', 0,
                            'cssLitGroup', [ L[ +17 ], L[ +35 ], L[ +57 ] ] ]),
                        'border', ___.iM([ 'cssExtra', c[ +1 ], 'cssPropBits',
                            7, 'cssLitGroup', [ L[ +0 ], L[ +9 ], L[ +34 ], L[
                                +46 ], L[ +47 ], L[ +48 ], L[ +52 ], L[ +54 ],
                              L[ +62 ] ] ]), 'border-bottom', ___.iM([
                            'cssExtra', c[ +1 ], 'cssPropBits', 7,
                            'cssLitGroup', [ L[ +0 ], L[ +9 ], L[ +34 ], L[ +46
                              ], L[ +47 ], L[ +48 ], L[ +52 ], L[ +54 ], L[ +62
                              ] ] ]), 'border-bottom-color', ___.iM([
                            'cssExtra', c[ +1 ], 'cssPropBits', 2,
                            'cssLitGroup', [ L[ +0 ], L[ +47 ], L[ +62 ] ] ]),
                        'border-bottom-left-radius', ___.iM([ 'cssExtra', c[ +0
                            ], 'cssPropBits', 5 ]),
                        'border-bottom-right-radius', ___.iM([ 'cssExtra', c[
                              +0 ], 'cssPropBits', 5 ]), 'border-bottom-style',
                        ___.iM([ 'cssPropBits', 0, 'cssLitGroup', [ L[ +9 ], L[
                                +46 ], L[ +47 ], L[ +48 ], L[ +54 ] ] ]),
                        'border-bottom-width', ___.iM([ 'cssPropBits', 5,
                            'cssLitGroup', [ L[ +34 ], L[ +47 ], L[ +52 ] ] ]),
                        'border-collapse', ___.iM([ 'cssPropBits', 0,
                            'cssLitGroup', [ L[ +44 ], L[ +47 ], L[ +59 ] ] ]),
                        'border-color', ___.iM([ 'cssExtra',
                            (IMPORTS___.RegExp_v___? IMPORTS___.RegExp:
                             ___.ri(IMPORTS___, 'RegExp')).i___('^ *(?:\\s*' +
                                s[ +0 ] + '){1,4} *$', 'i'), 'cssPropBits', 2,
                            'cssLitGroup', [ L[ +0 ], L[ +47 ], L[ +62 ] ] ]),
                        'border-left', ___.iM([ 'cssExtra', c[ +1 ],
                            'cssPropBits', 7, 'cssLitGroup', [ L[ +0 ], L[ +9 ]
                                , L[ +34 ], L[ +46 ], L[ +47 ], L[ +48 ], L[
                                +52 ], L[ +54 ], L[ +62 ] ] ]),
                        'border-left-color', ___.iM([ 'cssExtra', c[ +1 ],
                            'cssPropBits', 2, 'cssLitGroup', [ L[ +0 ], L[ +47
                              ], L[ +62 ] ] ]), 'border-left-style', ___.iM([
                            'cssPropBits', 0, 'cssLitGroup', [ L[ +9 ], L[ +46
                              ], L[ +47 ], L[ +48 ], L[ +54 ] ] ]),
                        'border-left-width', ___.iM([ 'cssPropBits', 5,
                            'cssLitGroup', [ L[ +34 ], L[ +47 ], L[ +52 ] ] ]),
                        'border-radius', ___.iM([ 'cssExtra', c[ +0 ],
                            'cssPropBits', 5, 'cssLitGroup', [ L[ +36 ] ] ]),
                        'border-right', ___.iM([ 'cssExtra', c[ +1 ],
                            'cssPropBits', 7, 'cssLitGroup', [ L[ +0 ], L[ +9 ]
                                , L[ +34 ], L[ +46 ], L[ +47 ], L[ +48 ], L[
                                +52 ], L[ +54 ], L[ +62 ] ] ]),
                        'border-right-color', ___.iM([ 'cssExtra', c[ +1 ],
                            'cssPropBits', 2, 'cssLitGroup', [ L[ +0 ], L[ +47
                              ], L[ +62 ] ] ]), 'border-right-style', ___.iM([
                            'cssPropBits', 0, 'cssLitGroup', [ L[ +9 ], L[ +46
                              ], L[ +47 ], L[ +48 ], L[ +54 ] ] ]),
                        'border-right-width', ___.iM([ 'cssPropBits', 5,
                            'cssLitGroup', [ L[ +34 ], L[ +47 ], L[ +52 ] ] ]),
                        'border-spacing', ___.iM([ 'cssExtra', c[ +0 ],
                            'cssPropBits', 5, 'cssLitGroup', [ L[ +47 ] ] ]),
                        'border-style', ___.iM([ 'cssPropBits', 0,
                            'cssLitGroup', [ L[ +9 ], L[ +46 ], L[ +47 ], L[
                                +48 ], L[ +54 ] ] ]), 'border-top', ___.iM([
                            'cssExtra', c[ +1 ], 'cssPropBits', 7,
                            'cssLitGroup', [ L[ +0 ], L[ +9 ], L[ +34 ], L[ +46
                              ], L[ +47 ], L[ +48 ], L[ +52 ], L[ +54 ], L[ +62
                              ] ] ]), 'border-top-color', ___.iM([ 'cssExtra',
                            c[ +1 ], 'cssPropBits', 2, 'cssLitGroup', [ L[ +0 ]
                                , L[ +47 ], L[ +62 ] ] ]),
                        'border-top-left-radius', ___.iM([ 'cssExtra', c[ +0 ],
                            'cssPropBits', 5 ]), 'border-top-right-radius',
                        ___.iM([ 'cssExtra', c[ +0 ], 'cssPropBits', 5 ]),
                        'border-top-style', ___.iM([ 'cssPropBits', 0,
                            'cssLitGroup', [ L[ +9 ], L[ +46 ], L[ +47 ], L[
                                +48 ], L[ +54 ] ] ]), 'border-top-width',
                        ___.iM([ 'cssPropBits', 5, 'cssLitGroup', [ L[ +34 ],
                              L[ +47 ], L[ +52 ] ] ]), 'border-width', ___.iM([
                            'cssPropBits', 5, 'cssLitGroup', [ L[ +34 ], L[ +47
                              ], L[ +52 ] ] ]), 'bottom', ___.iM([
                            'cssPropBits', 5, 'cssLitGroup', [ L[ +38 ], L[ +47
                              ] ] ]), 'box-shadow', ___.iM([ 'cssExtra', c[ +2
                            ], 'cssPropBits', 7, 'cssLitGroup', [ L[ +0 ], L[
                                +35 ], L[ +48 ], L[ +54 ] ] ]), 'caption-side',
                        ___.iM([ 'cssPropBits', 0, 'cssLitGroup', [ L[ +24 ],
                              L[ +47 ] ] ]), 'clear', ___.iM([ 'cssPropBits',
                            0, 'cssLitGroup', [ L[ +30 ], L[ +40 ], L[ +47 ],
                              L[ +54 ] ] ]), 'clip', ___.iM([ 'cssExtra', new
                              RegExp.new___('^ *\\s*rect\\(\\s*(?:0|[+\\-]?\\d+(?:\\.\\d+)?(?:[cem]m|ex|in|p[ctx])|auto)\\s*,\\s*(?:0|[+\\-]?\\d+(?:\\.\\d+)?(?:[cem]m|ex|in|p[ctx])|auto)\\s*,\\s*(?:0|[+\\-]?\\d+(?:\\.\\d+)?(?:[cem]m|ex|in|p[ctx])|auto)\\s*,\\s*(?:0|[+\\-]?\\d+(?:\\.\\d+)?(?:[cem]m|ex|in|p[ctx])|auto) *\\) *$',
                              'i'), 'cssPropBits', 0, 'cssLitGroup', [ L[ +38 ]
                                , L[ +47 ] ] ]), 'color', ___.iM([ 'cssExtra',
                            c[ +1 ], 'cssPropBits', 130, 'cssLitGroup', [ L[ +0
                              ], L[ +47 ] ] ]), 'content', ___.iM([
                            'cssPropBits', 0 ]), 'counter-increment', ___.iM([
                            'cssExtra', c[ +0 ], 'cssPropBits', 5,
                            'cssLitGroup', [ L[ +47 ], L[ +54 ] ] ]),
                        'counter-reset', ___.iM([ 'cssExtra', c[ +0 ],
                            'cssPropBits', 5, 'cssLitGroup', [ L[ +47 ], L[ +54
                              ] ] ]), 'cue', ___.iM([ 'cssPropBits', 16,
                            'cssLitGroup', [ L[ +47 ], L[ +54 ] ] ]),
                        'cue-after', ___.iM([ 'cssPropBits', 16, 'cssLitGroup',
                            [ L[ +47 ], L[ +54 ] ] ]), 'cue-before', ___.iM([
                            'cssPropBits', 16, 'cssLitGroup', [ L[ +47 ], L[
                                +54 ] ] ]), 'cursor', ___.iM([ 'cssExtra', c[
                              +0 ], 'cssPropBits', 144, 'cssLitGroup', [ L[ +1
                              ], L[ +35 ], L[ +38 ], L[ +47 ] ] ]),
                        'direction', ___.iM([ 'cssPropBits', 0, 'cssLitGroup',
                            [ L[ +31 ], L[ +47 ] ] ]), 'display', ___.iM([
                            'cssPropBits', 32, 'cssLitGroup', [ L[ +2 ], L[ +47
                              ], L[ +54 ] ] ]), 'elevation', ___.iM([
                            'cssPropBits', 5, 'cssLitGroup', [ L[ +13 ], L[ +47
                              ] ] ]), 'empty-cells', ___.iM([ 'cssPropBits', 0,
                            'cssLitGroup', [ L[ +27 ], L[ +47 ] ] ]), 'filter',
                        ___.iM([ 'cssExtra', new
                              RegExp.new___('^ *(?:\\s*alpha\\(\\s*opacity\\s*=\\s*(?:0|\\d+(?:\\.\\d+)?%|[+\\-]?\\d+(?:\\.\\d+)?) *\\))+ *$',
                              'i'), 'cssPropBits', 32 ]), 'float', ___.iM([
                            'cssAlternates', [ 'cssFloat', 'styleFloat' ],
                            'cssPropBits', 32, 'cssLitGroup', [ L[ +30 ], L[
                                +47 ], L[ +54 ] ] ]), 'font', ___.iM([
                            'cssExtra', c[ +0 ], 'cssPropBits', 9,
                            'cssLitGroup', [ L[ +4 ], L[ +7 ], L[ +11 ], L[ +15
                              ], L[ +29 ], L[ +35 ], L[ +36 ], L[ +47 ], L[ +52
                              ], L[ +55 ], L[ +60 ] ] ]), 'font-family',
                        ___.iM([ 'cssExtra', c[ +0 ], 'cssPropBits', 8,
                            'cssLitGroup', [ L[ +15 ], L[ +35 ], L[ +47 ] ] ]),
                        'font-size', ___.iM([ 'cssPropBits', 1, 'cssLitGroup',
                            [ L[ +7 ], L[ +47 ], L[ +52 ] ] ]), 'font-stretch',
                        ___.iM([ 'cssPropBits', 0, 'cssLitGroup', [ L[ +5 ], L[
                                +55 ] ] ]), 'font-style', ___.iM([
                            'cssPropBits', 0, 'cssLitGroup', [ L[ +29 ], L[ +47
                              ], L[ +55 ] ] ]), 'font-variant', ___.iM([
                            'cssPropBits', 0, 'cssLitGroup', [ L[ +47 ], L[ +55
                              ], L[ +60 ] ] ]), 'font-weight', ___.iM([
                            'cssPropBits', 0, 'cssLitGroup', [ L[ +4 ], L[ +47
                              ], L[ +55 ] ] ]), 'height', ___.iM([
                            'cssPropBits', 37, 'cssLitGroup', [ L[ +38 ], L[
                                +47 ] ] ]), 'left', ___.iM([ 'cssPropBits', 37,
                            'cssLitGroup', [ L[ +38 ], L[ +47 ] ] ]),
                        'letter-spacing', ___.iM([ 'cssPropBits', 5,
                            'cssLitGroup', [ L[ +47 ], L[ +55 ] ] ]),
                        'line-height', ___.iM([ 'cssPropBits', 1,
                            'cssLitGroup', [ L[ +47 ], L[ +55 ] ] ]),
                        'list-style', ___.iM([ 'cssPropBits', 16,
                            'cssLitGroup', [ L[ +3 ], L[ +28 ], L[ +47 ], L[
                                +54 ] ] ]), 'list-style-image', ___.iM([
                            'cssPropBits', 16, 'cssLitGroup', [ L[ +47 ], L[
                                +54 ] ] ]), 'list-style-position', ___.iM([
                            'cssPropBits', 0, 'cssLitGroup', [ L[ +28 ], L[ +47
                              ] ] ]), 'list-style-type', ___.iM([
                            'cssPropBits', 0, 'cssLitGroup', [ L[ +3 ], L[ +47
                              ], L[ +54 ] ] ]), 'margin', ___.iM([
                            'cssPropBits', 5, 'cssLitGroup', [ L[ +38 ], L[ +47
                              ] ] ]), 'margin-bottom', ___.iM([ 'cssPropBits',
                            5, 'cssLitGroup', [ L[ +38 ], L[ +47 ] ] ]),
                        'margin-left', ___.iM([ 'cssPropBits', 5,
                            'cssLitGroup', [ L[ +38 ], L[ +47 ] ] ]),
                        'margin-right', ___.iM([ 'cssPropBits', 5,
                            'cssLitGroup', [ L[ +38 ], L[ +47 ] ] ]),
                        'margin-top', ___.iM([ 'cssPropBits', 5, 'cssLitGroup',
                            [ L[ +38 ], L[ +47 ] ] ]), 'max-height', ___.iM([
                            'cssPropBits', 1, 'cssLitGroup', [ L[ +38 ], L[ +47
                              ], L[ +54 ] ] ]), 'max-width', ___.iM([
                            'cssPropBits', 1, 'cssLitGroup', [ L[ +38 ], L[ +47
                              ], L[ +54 ] ] ]), 'min-height', ___.iM([
                            'cssPropBits', 1, 'cssLitGroup', [ L[ +38 ], L[ +47
                              ] ] ]), 'min-width', ___.iM([ 'cssPropBits', 1,
                            'cssLitGroup', [ L[ +38 ], L[ +47 ] ] ]),
                        'opacity', ___.iM([ 'cssPropBits', 33, 'cssLitGroup', [
                              L[ +47 ] ] ]), 'outline', ___.iM([ 'cssExtra', c[
                              +1 ], 'cssPropBits', 7, 'cssLitGroup', [ L[ +0 ],
                              L[ +9 ], L[ +34 ], L[ +46 ], L[ +47 ], L[ +48 ],
                              L[ +49 ], L[ +52 ], L[ +54 ] ] ]),
                        'outline-color', ___.iM([ 'cssExtra', c[ +1 ],
                            'cssPropBits', 2, 'cssLitGroup', [ L[ +0 ], L[ +47
                              ], L[ +49 ] ] ]), 'outline-style', ___.iM([
                            'cssPropBits', 0, 'cssLitGroup', [ L[ +9 ], L[ +46
                              ], L[ +47 ], L[ +48 ], L[ +54 ] ] ]),
                        'outline-width', ___.iM([ 'cssPropBits', 5,
                            'cssLitGroup', [ L[ +34 ], L[ +47 ], L[ +52 ] ] ]),
                        'overflow', ___.iM([ 'cssPropBits', 32, 'cssLitGroup',
                            [ L[ +38 ], L[ +46 ], L[ +47 ], L[ +58 ], L[ +63 ]
                            ] ]), 'overflow-x', ___.iM([ 'cssPropBits', 0,
                            'cssLitGroup', [ L[ +32 ], L[ +38 ], L[ +46 ], L[
                                +58 ], L[ +63 ] ] ]), 'overflow-y', ___.iM([
                            'cssPropBits', 0, 'cssLitGroup', [ L[ +32 ], L[ +38
                              ], L[ +46 ], L[ +58 ], L[ +63 ] ] ]), 'padding',
                        ___.iM([ 'cssPropBits', 1, 'cssLitGroup', [ L[ +47 ] ]
                          ]), 'padding-bottom', ___.iM([ 'cssPropBits', 33,
                            'cssLitGroup', [ L[ +47 ] ] ]), 'padding-left',
                        ___.iM([ 'cssPropBits', 33, 'cssLitGroup', [ L[ +47 ] ]
                          ]), 'padding-right', ___.iM([ 'cssPropBits', 33,
                            'cssLitGroup', [ L[ +47 ] ] ]), 'padding-top',
                        ___.iM([ 'cssPropBits', 33, 'cssLitGroup', [ L[ +47 ] ]
                          ]), 'page-break-after', ___.iM([ 'cssPropBits', 0,
                            'cssLitGroup', [ L[ +30 ], L[ +37 ], L[ +38 ], L[
                                +39 ], L[ +47 ] ] ]), 'page-break-before',
                        ___.iM([ 'cssPropBits', 0, 'cssLitGroup', [ L[ +30 ],
                              L[ +37 ], L[ +38 ], L[ +39 ], L[ +47 ] ] ]),
                        'page-break-inside', ___.iM([ 'cssPropBits', 0,
                            'cssLitGroup', [ L[ +38 ], L[ +39 ], L[ +47 ] ] ]),
                        'pause', ___.iM([ 'cssPropBits', 5, 'cssLitGroup', [ L[
                                +47 ] ] ]), 'pause-after', ___.iM([
                            'cssPropBits', 5, 'cssLitGroup', [ L[ +47 ] ] ]),
                        'pause-before', ___.iM([ 'cssPropBits', 5,
                            'cssLitGroup', [ L[ +47 ] ] ]), 'pitch', ___.iM([
                            'cssPropBits', 5, 'cssLitGroup', [ L[ +19 ], L[ +47
                              ], L[ +52 ] ] ]), 'pitch-range', ___.iM([
                            'cssPropBits', 5, 'cssLitGroup', [ L[ +47 ] ] ]),
                        'play-during', ___.iM([ 'cssExtra', c[ +0 ],
                            'cssPropBits', 16, 'cssLitGroup', [ L[ +38 ], L[
                                +47 ], L[ +53 ], L[ +54 ], L[ +57 ] ] ]),
                        'position', ___.iM([ 'cssPropBits', 32, 'cssLitGroup',
                            [ L[ +20 ], L[ +47 ] ] ]), 'quotes', ___.iM([
                            'cssExtra', c[ +0 ], 'cssPropBits', 0,
                            'cssLitGroup', [ L[ +47 ], L[ +54 ] ] ]),
                        'richness', ___.iM([ 'cssPropBits', 5, 'cssLitGroup', [
                              L[ +47 ] ] ]), 'right', ___.iM([ 'cssPropBits',
                            37, 'cssLitGroup', [ L[ +38 ], L[ +47 ] ] ]),
                        'speak', ___.iM([ 'cssPropBits', 0, 'cssLitGroup', [ L[
                                +47 ], L[ +54 ], L[ +55 ], L[ +61 ] ] ]),
                        'speak-header', ___.iM([ 'cssPropBits', 0,
                            'cssLitGroup', [ L[ +37 ], L[ +47 ], L[ +56 ] ] ]),
                        'speak-numeral', ___.iM([ 'cssPropBits', 0,
                            'cssLitGroup', [ L[ +26 ], L[ +47 ] ] ]),
                        'speak-punctuation', ___.iM([ 'cssPropBits', 0,
                            'cssLitGroup', [ L[ +43 ], L[ +47 ], L[ +54 ] ] ]),
                        'speech-rate', ___.iM([ 'cssPropBits', 5,
                            'cssLitGroup', [ L[ +12 ], L[ +47 ], L[ +52 ] ] ]),
                        'stress', ___.iM([ 'cssPropBits', 5, 'cssLitGroup', [
                              L[ +47 ] ] ]), 'table-layout', ___.iM([
                            'cssPropBits', 0, 'cssLitGroup', [ L[ +38 ], L[ +45
                              ], L[ +47 ] ] ]), 'text-align', ___.iM([
                            'cssPropBits', 0, 'cssLitGroup', [ L[ +30 ], L[ +42
                              ], L[ +47 ], L[ +50 ] ] ]), 'text-decoration',
                        ___.iM([ 'cssPropBits', 0, 'cssLitGroup', [ L[ +18 ],
                              L[ +47 ], L[ +54 ] ] ]), 'text-indent', ___.iM([
                            'cssPropBits', 5, 'cssLitGroup', [ L[ +47 ] ] ]),
                        'text-overflow', ___.iM([ 'cssPropBits', 0,
                            'cssLitGroup', [ L[ +25 ] ] ]), 'text-shadow',
                        ___.iM([ 'cssExtra', c[ +2 ], 'cssPropBits', 7,
                            'cssLitGroup', [ L[ +0 ], L[ +35 ], L[ +48 ], L[
                                +54 ] ] ]), 'text-transform', ___.iM([
                            'cssPropBits', 0, 'cssLitGroup', [ L[ +21 ], L[ +47
                              ], L[ +54 ] ] ]), 'text-wrap', ___.iM([
                            'cssPropBits', 0, 'cssLitGroup', [ L[ +33 ], L[ +54
                              ], L[ +55 ] ] ]), 'top', ___.iM([ 'cssPropBits',
                            37, 'cssLitGroup', [ L[ +38 ], L[ +47 ] ] ]),
                        'unicode-bidi', ___.iM([ 'cssPropBits', 0,
                            'cssLitGroup', [ L[ +23 ], L[ +47 ], L[ +55 ] ] ]),
                        'vertical-align', ___.iM([ 'cssPropBits', 5,
                            'cssLitGroup', [ L[ +10 ], L[ +24 ], L[ +47 ] ] ]),
                        'visibility', ___.iM([ 'cssPropBits', 32,
                            'cssLitGroup', [ L[ +44 ], L[ +46 ], L[ +47 ], L[
                                +63 ] ] ]), 'voice-family', ___.iM([
                            'cssExtra', c[ +0 ], 'cssPropBits', 8,
                            'cssLitGroup', [ L[ +22 ], L[ +35 ], L[ +47 ] ] ]),
                        'volume', ___.iM([ 'cssPropBits', 1, 'cssLitGroup', [
                              L[ +16 ], L[ +47 ], L[ +52 ] ] ]), 'white-space',
                        ___.iM([ 'cssPropBits', 0, 'cssLitGroup', [ L[ +8 ], L[
                                +47 ], L[ +55 ] ] ]), 'width', ___.iM([
                            'cssPropBits', 33, 'cssLitGroup', [ L[ +38 ], L[
                                +47 ] ] ]), 'word-spacing', ___.iM([
                            'cssPropBits', 5, 'cssLitGroup', [ L[ +47 ], L[ +55
                              ] ] ]), 'word-wrap', ___.iM([ 'cssPropBits', 0,
                            'cssLitGroup', [ L[ +41 ], L[ +55 ] ] ]),
                        'z-index', ___.iM([ 'cssPropBits', 69, 'cssLitGroup', [
                              L[ +38 ], L[ +47 ] ] ]), 'zoom', ___.iM([
                            'cssPropBits', 1, 'cssLitGroup', [ L[ +55 ] ] ]) ])
                      ;
                  }).i___());
              if (typeof IMPORTS___.v___('window') !== 'undefined') {
                moduleResult___ = (IMPORTS___.window_v___? IMPORTS___.window:
                  ___.ri(IMPORTS___, 'window')).w___('cssSchema',
                  IMPORTS___.cssSchema_v___? IMPORTS___.cssSchema:
                  ___.ri(IMPORTS___, 'cssSchema'));
              }
              return moduleResult___;
            },
            'cajolerName': 'com.google.caja',
            'cajolerVersion': '5054',
            'cajoledDate': 1347987482968
          }).instantiate___(___, IMPORTS___), ___.prepareModule({
            'instantiate': function (___, IMPORTS___) {
              var dis___ = IMPORTS___;
              var moduleResult___, x0___, x1___, x2___, x3___, x4___, x5___,
              x6___, x7___, x8___, x9___, x10___, x11___, x12___, x13___,
              x14___, x15___, x16___, x17___, x18___, x19___, x20___, x21___,
              x22___, x23___;
              moduleResult___ = ___.NO_RESULT;
              ___.di(IMPORTS___, 'html4'), IMPORTS___.w___('html4', ___.iM([ ])
              );
              x0___ = IMPORTS___.html4_v___? IMPORTS___.html4:
              ___.ri(IMPORTS___, 'html4'), x1___ = ___.iM([ 'NONE', 0, 'URI',
                  1, 'URI_FRAGMENT', 11, 'SCRIPT', 2, 'STYLE', 3, 'ID', 4,
                  'IDREF', 5, 'IDREFS', 6, 'GLOBAL_NAME', 7, 'LOCAL_NAME', 8,
                  'CLASSES', 9, 'FRAME_TARGET', 10 ]), x0___.atype_w___ ===
                x0___? (x0___.atype = x1___): x0___.w___('atype', x1___);
              x2___ = IMPORTS___.html4_v___? IMPORTS___.html4:
              ___.ri(IMPORTS___, 'html4'), x3___ = ___.iM([ '*::class', 9,
                  '*::dir', 0, '*::id', 4, '*::lang', 0, '*::onclick', 2,
                  '*::ondblclick', 2, '*::onkeydown', 2, '*::onkeypress', 2,
                  '*::onkeyup', 2, '*::onload', 2, '*::onmousedown', 2,
                  '*::onmousemove', 2, '*::onmouseout', 2, '*::onmouseover', 2,
                  '*::onmouseup', 2, '*::onscroll', 2, '*::style', 3,
                  '*::title', 0, 'a::accesskey', 0, 'a::coords', 0, 'a::href',
                  1, 'a::hreflang', 0, 'a::name', 7, 'a::onblur', 2,
                  'a::onfocus', 2, 'a::rel', 0, 'a::rev', 0, 'a::shape', 0,
                  'a::tabindex', 0, 'a::target', 10, 'a::type', 0,
                  'area::accesskey', 0, 'area::alt', 0, 'area::coords', 0,
                  'area::href', 1, 'area::nohref', 0, 'area::onblur', 2,
                  'area::onfocus', 2, 'area::shape', 0, 'area::tabindex', 0,
                  'area::target', 10, 'bdo::dir', 0, 'blockquote::cite', 1,
                  'br::clear', 0, 'button::accesskey', 0, 'button::disabled',
                  0, 'button::name', 8, 'button::onblur', 2, 'button::onfocus',
                  2, 'button::tabindex', 0, 'button::type', 0, 'button::value',
                  0, 'caja-v-body::alink', 0, 'caja-v-body::background', 1,
                  'caja-v-body::bgcolor', 0, 'caja-v-body::link', 0,
                  'caja-v-body::onload', 2, 'caja-v-body::onunload', 2,
                  'caja-v-body::text', 0, 'caja-v-body::vlink', 0,
                  'caja-v-html::version', 0, 'canvas::height', 0,
                  'canvas::width', 0, 'caption::align', 0, 'col::align', 0,
                  'col::char', 0, 'col::charoff', 0, 'col::span', 0,
                  'col::valign', 0, 'col::width', 0, 'colgroup::align', 0,
                  'colgroup::char', 0, 'colgroup::charoff', 0,
                  'colgroup::span', 0, 'colgroup::valign', 0,
                  'colgroup::width', 0, 'del::cite', 1, 'del::datetime', 0,
                  'dir::compact', 0, 'div::align', 0, 'dl::compact', 0,
                  'font::color', 0, 'font::face', 0, 'font::size', 0,
                  'form::accept', 0, 'form::action', 1, 'form::autocomplete',
                  0, 'form::enctype', 0, 'form::method', 0, 'form::name', 7,
                  'form::onreset', 2, 'form::onsubmit', 2, 'form::target', 10,
                  'h1::align', 0, 'h2::align', 0, 'h3::align', 0, 'h4::align',
                  0, 'h5::align', 0, 'h6::align', 0, 'hr::align', 0,
                  'hr::noshade', 0, 'hr::size', 0, 'hr::width', 0,
                  'iframe::align', 0, 'iframe::frameborder', 0,
                  'iframe::height', 0, 'iframe::marginheight', 0,
                  'iframe::marginwidth', 0, 'iframe::width', 0, 'img::align',
                  0, 'img::alt', 0, 'img::border', 0, 'img::height', 0,
                  'img::hspace', 0, 'img::ismap', 0, 'img::name', 7,
                  'img::src', 1, 'img::usemap', 11, 'img::vspace', 0,
                  'img::width', 0, 'input::accept', 0, 'input::accesskey', 0,
                  'input::align', 0, 'input::alt', 0, 'input::autocomplete', 0,
                  'input::checked', 0, 'input::disabled', 0, 'input::ismap', 0,
                  'input::maxlength', 0, 'input::name', 8, 'input::onblur', 2,
                  'input::onchange', 2, 'input::onfocus', 2, 'input::onselect',
                  2, 'input::readonly', 0, 'input::size', 0, 'input::src', 1,
                  'input::tabindex', 0, 'input::type', 0, 'input::usemap', 11,
                  'input::value', 0, 'ins::cite', 1, 'ins::datetime', 0,
                  'label::accesskey', 0, 'label::for', 5, 'label::onblur', 2,
                  'label::onfocus', 2, 'legend::accesskey', 0, 'legend::align',
                  0, 'li::type', 0, 'li::value', 0, 'map::name', 7,
                  'menu::compact', 0, 'ol::compact', 0, 'ol::start', 0,
                  'ol::type', 0, 'optgroup::disabled', 0, 'optgroup::label', 0,
                  'option::disabled', 0, 'option::label', 0,
                  'option::selected', 0, 'option::value', 0, 'p::align', 0,
                  'pre::width', 0, 'q::cite', 1, 'select::disabled', 0,
                  'select::multiple', 0, 'select::name', 8, 'select::onblur',
                  2, 'select::onchange', 2, 'select::onfocus', 2,
                  'select::size', 0, 'select::tabindex', 0, 'table::align', 0,
                  'table::bgcolor', 0, 'table::border', 0,
                  'table::cellpadding', 0, 'table::cellspacing', 0,
                  'table::frame', 0, 'table::rules', 0, 'table::summary', 0,
                  'table::width', 0, 'tbody::align', 0, 'tbody::char', 0,
                  'tbody::charoff', 0, 'tbody::valign', 0, 'td::abbr', 0,
                  'td::align', 0, 'td::axis', 0, 'td::bgcolor', 0, 'td::char',
                  0, 'td::charoff', 0, 'td::colspan', 0, 'td::headers', 6,
                  'td::height', 0, 'td::nowrap', 0, 'td::rowspan', 0,
                  'td::scope', 0, 'td::valign', 0, 'td::width', 0,
                  'textarea::accesskey', 0, 'textarea::cols', 0,
                  'textarea::disabled', 0, 'textarea::name', 8,
                  'textarea::onblur', 2, 'textarea::onchange', 2,
                  'textarea::onfocus', 2, 'textarea::onselect', 2,
                  'textarea::readonly', 0, 'textarea::rows', 0,
                  'textarea::tabindex', 0, 'tfoot::align', 0, 'tfoot::char', 0,
                  'tfoot::charoff', 0, 'tfoot::valign', 0, 'th::abbr', 0,
                  'th::align', 0, 'th::axis', 0, 'th::bgcolor', 0, 'th::char',
                  0, 'th::charoff', 0, 'th::colspan', 0, 'th::headers', 6,
                  'th::height', 0, 'th::nowrap', 0, 'th::rowspan', 0,
                  'th::scope', 0, 'th::valign', 0, 'th::width', 0,
                  'thead::align', 0, 'thead::char', 0, 'thead::charoff', 0,
                  'thead::valign', 0, 'tr::align', 0, 'tr::bgcolor', 0,
                  'tr::char', 0, 'tr::charoff', 0, 'tr::valign', 0,
                  'ul::compact', 0, 'ul::type', 0 ]), x2___.ATTRIBS_w___ ===
                x2___? (x2___.ATTRIBS = x3___): x2___.w___('ATTRIBS', x3___);
              x4___ = IMPORTS___.html4_v___? IMPORTS___.html4:
              ___.ri(IMPORTS___, 'html4'), x5___ = ___.iM([ 'OPTIONAL_ENDTAG',
                  1, 'EMPTY', 2, 'CDATA', 4, 'RCDATA', 8, 'UNSAFE', 16,
                  'FOLDABLE', 32, 'SCRIPT', 64, 'STYLE', 128, 'VIRTUALIZED',
                  256 ]), x4___.eflags_w___ === x4___? (x4___.eflags = x5___):
              x4___.w___('eflags', x5___);
              x6___ = IMPORTS___.html4_v___? IMPORTS___.html4:
              ___.ri(IMPORTS___, 'html4'), x7___ = ___.iM([ 'a', 0, 'abbr', 0,
                  'acronym', 0, 'address', 0, 'applet', 16, 'area', 2,
                  'article', 16, 'aside', 16, 'audio', 16, 'b', 0, 'base', 18,
                  'basefont', 18, 'bdi', 16, 'bdo', 0, 'big', 0, 'blockquote',
                  0, 'body', 305, 'br', 2, 'button', 0, 'caja-v-body', 1,
                  'caja-v-head', 1, 'caja-v-html', 1, 'caja-v-title', 0,
                  'canvas', 0, 'caption', 0, 'center', 0, 'cite', 0, 'code', 0,
                  'col', 2, 'colgroup', 1, 'data', 16, 'datalist', 16, 'dd', 1,
                  'del', 0, 'details', 16, 'dfn', 0, 'dir', 0, 'div', 0, 'dl',
                  0, 'dt', 1, 'em', 0, 'fieldset', 0, 'figcaption', 16,
                  'figure', 16, 'font', 0, 'footer', 16, 'form', 0, 'frame',
                  18, 'frameset', 16, 'h1', 0, 'h2', 0, 'h3', 0, 'h4', 0, 'h5',
                  0, 'h6', 0, 'head', 305, 'header', 16, 'hgroup', 16, 'hr', 2,
                  'html', 305, 'i', 0, 'iframe', 4, 'img', 2, 'input', 2,
                  'ins', 0, 'isindex', 18, 'kbd', 0, 'label', 0, 'legend', 0,
                  'li', 1, 'link', 18, 'map', 0, 'mark', 16, 'menu', 0, 'meta',
                  18, 'meter', 16, 'nav', 0, 'nobr', 0, 'noembed', 4,
                  'noframes', 20, 'noscript', 20, 'object', 16, 'ol', 0,
                  'optgroup', 0, 'option', 1, 'output', 16, 'p', 1, 'param',
                  18, 'pre', 0, 'progress', 16, 'q', 0, 's', 0, 'samp', 0,
                  'script', 84, 'section', 16, 'select', 0, 'small', 0, 'span',
                  0, 'strike', 0, 'strong', 0, 'style', 148, 'sub', 0,
                  'summary', 16, 'sup', 0, 'table', 0, 'tbody', 1, 'td', 1,
                  'textarea', 8, 'tfoot', 1, 'th', 1, 'thead', 1, 'time', 16,
                  'title', 280, 'tr', 1, 'tt', 0, 'u', 0, 'ul', 0, 'var', 0,
                  'video', 16 ]), x6___.ELEMENTS_w___ === x6___?
                (x6___.ELEMENTS = x7___): x6___.w___('ELEMENTS', x7___);
              x8___ = IMPORTS___.html4_v___? IMPORTS___.html4:
              ___.ri(IMPORTS___, 'html4'), x9___ = ___.iM([ 'NOT_LOADED', 0,
                  'SAME_DOCUMENT', 1, 'NEW_DOCUMENT', 2 ]), x8___.ueffects_w___
                === x8___? (x8___.ueffects = x9___): x8___.w___('ueffects',
                x9___);
              x10___ = IMPORTS___.html4_v___? IMPORTS___.html4:
              ___.ri(IMPORTS___, 'html4'), x11___ = ___.iM([ 'a::href', 2,
                  'area::href', 2, 'blockquote::cite', 0, 'body::background',
                  1, 'caja-v-body::background', 1, 'del::cite', 0,
                  'form::action', 2, 'img::src', 1, 'input::src', 1,
                  'ins::cite', 0, 'q::cite', 0 ]), x10___.URIEFFECTS_w___ ===
                x10___? (x10___.URIEFFECTS = x11___): x10___.w___('URIEFFECTS',
                x11___);
              x12___ = IMPORTS___.html4_v___? IMPORTS___.html4:
              ___.ri(IMPORTS___, 'html4'), x13___ = ___.iM([ 'UNSANDBOXED', 2,
                  'SANDBOXED', 1, 'DATA', 0 ]), x12___.ltypes_w___ === x12___?
                (x12___.ltypes = x13___): x12___.w___('ltypes', x13___);
              x14___ = IMPORTS___.html4_v___? IMPORTS___.html4:
              ___.ri(IMPORTS___, 'html4'), x15___ = ___.iM([ 'a::href', 2,
                  'area::href', 2, 'blockquote::cite', 2, 'body::background',
                  1, 'caja-v-body::background', 1, 'del::cite', 2,
                  'form::action', 2, 'img::src', 1, 'input::src', 1,
                  'ins::cite', 2, 'q::cite', 2 ]), x14___.LOADERTYPES_w___ ===
                x14___? (x14___.LOADERTYPES = x15___):
              x14___.w___('LOADERTYPES', x15___);
              (IMPORTS___.html4_v___? IMPORTS___.html4: ___.ri(IMPORTS___,
                 'html4')).w___('ATTRIBS', (x16___ = IMPORTS___.html4_v___?
                  IMPORTS___.html4: ___.ri(IMPORTS___, 'html4'),
                  x16___.ATTRIBS_v___? x16___.ATTRIBS: x16___.v___('ATTRIBS')))
                ;
              (IMPORTS___.html4_v___? IMPORTS___.html4: ___.ri(IMPORTS___,
                 'html4')).w___('ELEMENTS', (x17___ = IMPORTS___.html4_v___?
                  IMPORTS___.html4: ___.ri(IMPORTS___, 'html4'),
                  x17___.ELEMENTS_v___? x17___.ELEMENTS:
                  x17___.v___('ELEMENTS')));
              (IMPORTS___.html4_v___? IMPORTS___.html4: ___.ri(IMPORTS___,
                 'html4')).w___('URIEFFECTS', (x18___ = IMPORTS___.html4_v___?
                  IMPORTS___.html4: ___.ri(IMPORTS___, 'html4'),
                  x18___.URIEFFECTS_v___? x18___.URIEFFECTS:
                  x18___.v___('URIEFFECTS')));
              (IMPORTS___.html4_v___? IMPORTS___.html4: ___.ri(IMPORTS___,
                 'html4')).w___('LOADERTYPES', (x19___ = IMPORTS___.html4_v___?
                  IMPORTS___.html4: ___.ri(IMPORTS___, 'html4'),
                  x19___.LOADERTYPES_v___? x19___.LOADERTYPES:
                  x19___.v___('LOADERTYPES')));
              (IMPORTS___.html4_v___? IMPORTS___.html4: ___.ri(IMPORTS___,
                 'html4')).w___('atype', (x20___ = IMPORTS___.html4_v___?
                  IMPORTS___.html4: ___.ri(IMPORTS___, 'html4'),
                  x20___.atype_v___? x20___.atype: x20___.v___('atype')));
              (IMPORTS___.html4_v___? IMPORTS___.html4: ___.ri(IMPORTS___,
                 'html4')).w___('eflags', (x21___ = IMPORTS___.html4_v___?
                  IMPORTS___.html4: ___.ri(IMPORTS___, 'html4'),
                  x21___.eflags_v___? x21___.eflags: x21___.v___('eflags')));
              (IMPORTS___.html4_v___? IMPORTS___.html4: ___.ri(IMPORTS___,
                 'html4')).w___('ltypes', (x22___ = IMPORTS___.html4_v___?
                  IMPORTS___.html4: ___.ri(IMPORTS___, 'html4'),
                  x22___.ltypes_v___? x22___.ltypes: x22___.v___('ltypes')));
              (IMPORTS___.html4_v___? IMPORTS___.html4: ___.ri(IMPORTS___,
                 'html4')).w___('ueffects', (x23___ = IMPORTS___.html4_v___?
                  IMPORTS___.html4: ___.ri(IMPORTS___, 'html4'),
                  x23___.ueffects_v___? x23___.ueffects:
                  x23___.v___('ueffects')));
              if (typeof IMPORTS___.v___('window') !== 'undefined') {
                moduleResult___ = (IMPORTS___.window_v___? IMPORTS___.window:
                  ___.ri(IMPORTS___, 'window')).w___('html4',
                  IMPORTS___.html4_v___? IMPORTS___.html4: ___.ri(IMPORTS___,
                    'html4'));
              }
              return moduleResult___;
            },
            'cajolerName': 'com.google.caja',
            'cajolerVersion': '5054',
            'cajoledDate': 1347987483086
          }).instantiate___(___, IMPORTS___), ___.prepareModule({
            'instantiate': function (___, IMPORTS___) {
              var dis___ = IMPORTS___;
              var moduleResult___;
              moduleResult___ = ___.NO_RESULT;
              if (('I'.toLowerCase_m___? 'I'.toLowerCase():
                  'I'.m___('toLowerCase', [ ])) !== 'i') {
                throw 'I/i problem';
              }
              ___.di(IMPORTS___, 'html'), IMPORTS___.w___('html',
                ___.f(function (html4) {
                    var parseCssDeclarations, sanitizeCssProperty, cssSchema,
                    ENTITIES, decimalEscapeRe, hexEscapeRe, lookupEntity,
                    decodeOneEntity, nulRe, stripNULs, ENTITY_RE_1,
                    ENTITY_RE_2, unescapeEntities, ampRe, looseAmpRe, ltRe,
                    gtRe, quotRe, escapeAttrib, normalizeRCData, ATTR_RE,
                    splitWillCapture, x0___, EFLAGS_TEXT, makeSaxParser,
                    continuationMarker, parse, continuationMaker, parseCPS,
                    htmlSplit, parseEndTag, parseStartTag, endTagRe, parseText,
                    parseTagAndAttrs, decodeValue, VIRTUALIZED_ELEMENT_NAME_RE,
                    VIRTUALIZED_ELEMENT_PREFIX, isVirtualizedElementName,
                    realToVirtualElementName, virtualToRealElementName,
                    makeHtmlSanitizer, URI_SCHEME_RE, ALLOWED_URI_SCHEMES,
                    safeUri, log, lookupAttribute, getAttributeType,
                    getLoaderType, getUriEffect, sanitizeAttribs,
                    makeTagPolicy, sanitizeWithPolicy, sanitize, html, x1___,
                    x2___, x3___, x4___, x5___, x6___, x7___, x8___, x9___,
                    x10___, x11___, x12___, x13___, x14___, x15___, x16___,
                    x17___, x18___, x19___, x20___, x21___, x22___, x23___,
                    x24___;
                    {
                      function lookupEntity(name) {
                        var m, x0___, x1___, x2___, x3___;
                        name = name.toLowerCase_m___? name.toLowerCase():
                        name.m___('toLowerCase', [ ]);
                        if (ENTITIES.hasOwnProperty_m___?
                            ENTITIES.hasOwnProperty(name):
                          ENTITIES.m___('hasOwnProperty', [ name ])) {
                          return ENTITIES.v___(name);
                        }
                        m = name.match_m___? name.match(decimalEscapeRe):
                        name.m___('match', [ decimalEscapeRe ]);
                        if (m) {
                          return x0___ = IMPORTS___.String_v___?
                            IMPORTS___.String: ___.ri(IMPORTS___, 'String'),
                          x1___ = (IMPORTS___.parseInt_v___?
                              IMPORTS___.parseInt: ___.ri(IMPORTS___,
                              'parseInt')).i___(m[ +1 ], 10),
                          x0___.fromCharCode_m___? x0___.fromCharCode(x1___):
                          x0___.m___('fromCharCode', [ x1___ ]);
                        } else if (! ! (m = name.match_m___?
                              name.match(hexEscapeRe): name.m___('match', [
                                hexEscapeRe ]))) {
                          return x2___ = IMPORTS___.String_v___?
                            IMPORTS___.String: ___.ri(IMPORTS___, 'String'),
                          x3___ = (IMPORTS___.parseInt_v___?
                              IMPORTS___.parseInt: ___.ri(IMPORTS___,
                              'parseInt')).i___(m[ +1 ], 16),
                          x2___.fromCharCode_m___? x2___.fromCharCode(x3___):
                          x2___.m___('fromCharCode', [ x3___ ]);
                        }
                        return '';
                      }
                      ___.f(lookupEntity, 'lookupEntity');
                    }
                    {
                      function decodeOneEntity(_, name) {
                        return lookupEntity.i___(name);
                      }
                      ___.f(decodeOneEntity, 'decodeOneEntity');
                    }
                    {
                      function stripNULs(s) {
                        return s.replace_m___? s.replace(nulRe, ''):
                        s.m___('replace', [ nulRe, '' ]);
                      }
                      ___.f(stripNULs, 'stripNULs');
                    }
                    {
                      function unescapeEntities(s) {
                        return s.replace_m___? s.replace(ENTITY_RE_1,
                          decodeOneEntity): s.m___('replace', [ ENTITY_RE_1,
                            decodeOneEntity ]);
                      }
                      ___.f(unescapeEntities, 'unescapeEntities');
                    }
                    {
                      function escapeAttrib(s) {
                        var x0___, x1___, x2___, x3___, x4___, x5___, x6___,
                        x7___;
                        return x6___ = (x4___ = (x2___ = (x0___ = '' + s, x1___
                                = ampRe, x0___.replace_m___?
                                x0___.replace(x1___, '\x26amp;'):
                              x0___.m___('replace', [ x1___, '\x26amp;' ])),
                            x3___ = ltRe, x2___.replace_m___?
                            x2___.replace(x3___, '\x26lt;'):
                            x2___.m___('replace', [ x3___, '\x26lt;' ])), x5___
                          = gtRe, x4___.replace_m___? x4___.replace(x5___,
                            '\x26gt;'): x4___.m___('replace', [ x5___,
                              '\x26gt;' ])), x7___ = quotRe,
                        x6___.replace_m___? x6___.replace(x7___, '\x26#34;'):
                        x6___.m___('replace', [ x7___, '\x26#34;' ]);
                      }
                      ___.f(escapeAttrib, 'escapeAttrib');
                    }
                    {
                      function normalizeRCData(rcdata) {
                        var x0___, x1___, x2___, x3___;
                        return x2___ = (x0___ = rcdata.replace_m___?
                          rcdata.replace(looseAmpRe, '\x26amp;$1'):
                          rcdata.m___('replace', [ looseAmpRe, '\x26amp;$1' ]),
                          x1___ = ltRe, x0___.replace_m___?
                          x0___.replace(x1___, '\x26lt;'):
                          x0___.m___('replace', [ x1___, '\x26lt;' ])), x3___ =
                          gtRe, x2___.replace_m___? x2___.replace(x3___,
                          '\x26gt;'): x2___.m___('replace', [ x3___, '\x26gt;'
                          ]);
                      }
                      ___.f(normalizeRCData, 'normalizeRCData');
                    }
                    {
                      function makeSaxParser(handler) {
                        var hcopy;
                        hcopy = ___.iM([ 'cdata', (handler.cdata_v___?
                              handler.cdata: handler.v___('cdata')) ||
                            handler.v___('cdata'), 'comment',
                            (handler.comment_v___? handler.comment:
                             handler.v___('comment')) ||
                            handler.v___('comment'), 'endDoc',
                            (handler.endDoc_v___? handler.endDoc:
                             handler.v___('endDoc')) || handler.v___('endDoc'),
                            'endTag', (handler.endTag_v___? handler.endTag:
                              handler.v___('endTag')) || handler.v___('endTag')
                            , 'pcdata', (handler.pcdata_v___? handler.pcdata:
                              handler.v___('pcdata')) || handler.v___('pcdata')
                            , 'rcdata', (handler.rcdata_v___? handler.rcdata:
                              handler.v___('rcdata')) || handler.v___('rcdata')
                            , 'startDoc', (handler.startDoc_v___?
                              handler.startDoc: handler.v___('startDoc')) ||
                            handler.v___('startDoc'), 'startTag',
                            (handler.startTag_v___? handler.startTag:
                             handler.v___('startTag')) ||
                            handler.v___('startTag') ]);
                        return ___.f(function (htmlText, param) {
                            return parse.i___(htmlText, hcopy, param);
                          });
                      }
                      ___.f(makeSaxParser, 'makeSaxParser');
                    }
                    {
                      function parse(htmlText, handler, param) {
                        var m, p, tagName, parts, state;
                        parts = htmlSplit.i___(htmlText);
                        state = ___.iM([ 'noMoreGT', false,
                            'noMoreEndComments', false ]);
                        parseCPS.i___(handler, parts, 0, state, param);
                      }
                      ___.f(parse, 'parse');
                    }
                    {
                      function continuationMaker(h, parts, initial, state,
                        param) {
                        return ___.f(function () {
                            parseCPS.i___(h, parts, initial, state, param);
                          });
                      }
                      ___.f(continuationMaker, 'continuationMaker');
                    }
                    {
                      function parseCPS(h, parts, initial, state, param) {
                        var m, p, tagName, pos, end, current, next, x0___,
                        x1___, x2___, x3___, x4___, x5___, x6___, x7___, x8___,
                        x9___, x10___, x11___, x12___, x13___, x14___, x15___,
                        x16___, x17___, x18___, x19___, x20___, x21___, x22___,
                        x23___, x24___, x25___, x26___, x27___, x28___, x29___,
                        eflags, tag, x30___, x31___, x32___, x33___, x34___,
                        x35___, comment, x36___, x37___, x38___, x39___,
                        x40___, x41___, x42___, x43___, x44___, x45___, x46___,
                        x47___, x48___, x49___, x50___, x51___, x52___, x53___,
                        x54___, x55___, x56___, x57___, x58___, x59___, x60___,
                        x61___, x62___, x63___, x64___, x65___, x66___, x67___,
                        x68___, x69___, x70___;
                        try {
                          if ((h.startDoc_v___? h.startDoc: h.v___('startDoc'))
                            && initial == 0) {
                            h.startDoc_m___? h.startDoc(param):
                            h.m___('startDoc', [ param ]);
                          }
                          for (pos = initial, end = parts.length; pos < end;) {
                            current = parts.v___(pos++);
                            next = parts.v___(pos);
                            switch (current) {
                            case '\x26':
                              if (ENTITY_RE_2.test_m___? ENTITY_RE_2.test(next)
                                : ENTITY_RE_2.m___('test', [ next ])) {
                                if (h.pcdata_v___? h.pcdata: h.v___('pcdata'))
                                  {
                                  x2___ = h, x0___ = '\x26' + next, x3___ =
                                    param, x4___ = continuationMarker, x1___ =
                                    continuationMaker.i___(h, parts, pos,
                                    state, param), x2___.pcdata_m___?
                                    x2___.pcdata(x0___, x3___, x4___, x1___):
                                  x2___.m___('pcdata', [ x0___, x3___, x4___,
                                      x1___ ]);
                                }
                                pos++;
                              } else {
                                if (h.pcdata_v___? h.pcdata: h.v___('pcdata'))
                                  {
                                  x6___ = h, x7___ = param, x8___ =
                                    continuationMarker, x5___ =
                                    continuationMaker.i___(h, parts, pos,
                                    state, param), x6___.pcdata_m___?
                                    x6___.pcdata('\x26amp;', x7___, x8___,
                                    x5___): x6___.m___('pcdata', [ '\x26amp;',
                                      x7___, x8___, x5___ ]);
                                }
                              }
                              break;
                            case '\x3c/':
                              if (m = (x9___ = new
                                  RegExp.new___('^([-\\w:]+)[^\\\'\\\"]*'),
                                  x10___ = next, x9___.exec_m___?
                                  x9___.exec(x10___): x9___.m___('exec', [
                                      x10___ ]))) {
                                if (m[ +0 ].length === next.length &&
                                    parts.v___(pos + 1) === '\x3e') {
                                  pos = pos + 2;
                                  tagName = (x11___ = m[ +1 ],
                                    x11___.toLowerCase_m___?
                                    x11___.toLowerCase():
                                    x11___.m___('toLowerCase', [ ]));
                                  if (h.endTag_v___? h.endTag: h.v___('endTag')
                                  ) {
                                    x13___ = h, x14___ = tagName, x15___ =
                                      param, x16___ = continuationMarker,
                                    x12___ = continuationMaker.i___(h, parts,
                                      pos, state, param), x13___.endTag_m___?
                                      x13___.endTag(x14___, x15___, x16___,
                                      x12___): x13___.m___('endTag', [ x14___,
                                        x15___, x16___, x12___ ]);
                                  }
                                } else {
                                  pos = parseEndTag.i___(parts, pos, h, param,
                                    continuationMarker, state);
                                }
                              } else {
                                if (h.pcdata_v___? h.pcdata: h.v___('pcdata'))
                                  {
                                  x18___ = h, x19___ = param, x20___ =
                                    continuationMarker, x17___ =
                                    continuationMaker.i___(h, parts, pos,
                                    state, param), x18___.pcdata_m___?
                                    x18___.pcdata('\x26lt;/', x19___, x20___,
                                    x17___): x18___.m___('pcdata', [
                                      '\x26lt;/', x19___, x20___, x17___ ]);
                                }
                              }
                              break;
                            case '\x3c':
                              if (m = (x21___ = new
                                  RegExp.new___('^([-\\w:]+)\\s*\\/?'), x22___
                                  = next, x21___.exec_m___? x21___.exec(x22___)
                                  : x21___.m___('exec', [ x22___ ]))) {
                                if (m[ +0 ].length === next.length &&
                                    parts.v___(pos + 1) === '\x3e') {
                                  pos = pos + 2;
                                  tagName = (x23___ = m[ +1 ],
                                    x23___.toLowerCase_m___?
                                    x23___.toLowerCase():
                                    x23___.m___('toLowerCase', [ ]));
                                  if (h.startTag_v___? h.startTag:
                                    h.v___('startTag')) {
                                    x26___ = h, x27___ = tagName, x24___ = [ ],
                                    x28___ = param, x29___ =
                                      continuationMarker, x25___ =
                                      continuationMaker.i___(h, parts, pos,
                                      state, param), x26___.startTag_m___?
                                      x26___.startTag(x27___, x24___, x28___,
                                      x29___, x25___): x26___.m___('startTag',
                                      [ x27___, x24___, x28___, x29___, x25___
                                      ]);
                                  }
                                  eflags = (html4.ELEMENTS_v___?
                                      html4.ELEMENTS: html4.v___('ELEMENTS'))
                                    .v___(tagName);
                                  if (eflags & EFLAGS_TEXT) {
                                    tag = ___.iM([ 'name', tagName, 'next',
                                        pos, 'eflags', eflags ]);
                                    pos = parseText.i___(parts, tag, h, param,
                                      continuationMarker, state);
                                  }
                                } else {
                                  pos = parseStartTag.i___(parts, pos, h,
                                    param, continuationMarker, state);
                                }
                              } else {
                                if (h.pcdata_v___? h.pcdata: h.v___('pcdata'))
                                  {
                                  x31___ = h, x32___ = param, x33___ =
                                    continuationMarker, x30___ =
                                    continuationMaker.i___(h, parts, pos,
                                    state, param), x31___.pcdata_m___?
                                    x31___.pcdata('\x26lt;', x32___, x33___,
                                    x30___): x31___.m___('pcdata', [ '\x26lt;',
                                      x32___, x33___, x30___ ]);
                                }
                              }
                              break;
                            case '\x3c!--':
                              if (! (state.noMoreEndComments_v___?
                                    state.noMoreEndComments:
                                  state.v___('noMoreEndComments'))) {
                                for (p = pos + 1; p < end; p++) {
                                  if (parts.v___(p) === '\x3e' && (x34___ = new
                                      RegExp.new___('--$'), x35___ = parts[ +
                                          (p - 1) ], x34___.test_m___?
                                      x34___.test(x35___): x34___.m___('test',
                                        [ x35___ ]))) { break; }
                                }
                                if (p < end) {
                                  if (h.comment_v___? h.comment:
                                    h.v___('comment')) {
                                    comment = (x36___ = parts.slice_m___?
                                      parts.slice(pos, p): parts.m___('slice',
                                        [ pos, p ]), x36___.join_m___?
                                      x36___.join(''): x36___.m___('join', [ ''
                                        ]));
                                    x41___ = h, x39___ = (x38___ = comment,
                                      x37___ = comment.length - 2,
                                      x38___.substr_m___? x38___.substr(0,
                                        x37___): x38___.m___('substr', [ 0,
                                          x37___ ])), x42___ = param, x43___ =
                                      continuationMarker, x40___ =
                                      continuationMaker.i___(h, parts, p + 1,
                                      state, param), x41___.comment_m___?
                                      x41___.comment(x39___, x42___, x43___,
                                      x40___): x41___.m___('comment', [ x39___,
                                        x42___, x43___, x40___ ]);
                                  }
                                  pos = p + 1;
                                } else {
                                  state.noMoreEndComments_w___ === state?
                                    (state.noMoreEndComments = true):
                                  state.w___('noMoreEndComments', true);
                                }
                              }
                              if (state.noMoreEndComments_v___?
                                  state.noMoreEndComments:
                                state.v___('noMoreEndComments')) {
                                if (h.pcdata_v___? h.pcdata: h.v___('pcdata'))
                                  {
                                  x45___ = h, x46___ = param, x47___ =
                                    continuationMarker, x44___ =
                                    continuationMaker.i___(h, parts, pos,
                                    state, param), x45___.pcdata_m___?
                                    x45___.pcdata('\x26lt;!--', x46___, x47___,
                                    x44___): x45___.m___('pcdata', [
                                      '\x26lt;!--', x46___, x47___, x44___ ]);
                                }
                              }
                              break;
                            case '\x3c!':
                              if (! (x48___ = new RegExp.new___('^\\w'), x49___
                                    = next, x48___.test_m___?
                                    x48___.test(x49___): x48___.m___('test', [
                                      x49___ ]))) {
                                if (h.pcdata_v___? h.pcdata: h.v___('pcdata'))
                                  {
                                  x51___ = h, x52___ = param, x53___ =
                                    continuationMarker, x50___ =
                                    continuationMaker.i___(h, parts, pos,
                                    state, param), x51___.pcdata_m___?
                                    x51___.pcdata('\x26lt;!', x52___, x53___,
                                    x50___): x51___.m___('pcdata', [
                                      '\x26lt;!', x52___, x53___, x50___ ]);
                                }
                              } else {
                                if (! (state.noMoreGT_v___? state.noMoreGT:
                                    state.v___('noMoreGT'))) {
                                  for (p = pos + 1; p < end; p++) {
                                    if (parts.v___(p) === '\x3e') { break; }
                                  }
                                  if (p < end) {
                                    pos = p + 1;
                                  } else {
                                    state.noMoreGT_w___ === state?
                                      (state.noMoreGT = true):
                                    state.w___('noMoreGT', true);
                                  }
                                }
                                if (state.noMoreGT_v___? state.noMoreGT:
                                  state.v___('noMoreGT')) {
                                  if (h.pcdata_v___? h.pcdata: h.v___('pcdata')
                                  ) {
                                    x55___ = h, x56___ = param, x57___ =
                                      continuationMarker, x54___ =
                                      continuationMaker.i___(h, parts, pos,
                                      state, param), x55___.pcdata_m___?
                                      x55___.pcdata('\x26lt;!', x56___, x57___,
                                      x54___): x55___.m___('pcdata', [
                                        '\x26lt;!', x56___, x57___, x54___ ]);
                                  }
                                }
                              }
                              break;
                            case '\x3c?':
                              if (! (state.noMoreGT_v___? state.noMoreGT:
                                  state.v___('noMoreGT'))) {
                                for (p = pos + 1; p < end; p++) {
                                  if (parts.v___(p) === '\x3e') { break; }
                                }
                                if (p < end) {
                                  pos = p + 1;
                                } else {
                                  state.noMoreGT_w___ === state?
                                    (state.noMoreGT = true):
                                  state.w___('noMoreGT', true);
                                }
                              }
                              if (state.noMoreGT_v___? state.noMoreGT:
                                state.v___('noMoreGT')) {
                                if (h.pcdata_v___? h.pcdata: h.v___('pcdata'))
                                  {
                                  x59___ = h, x60___ = param, x61___ =
                                    continuationMarker, x58___ =
                                    continuationMaker.i___(h, parts, pos,
                                    state, param), x59___.pcdata_m___?
                                    x59___.pcdata('\x26lt;?', x60___, x61___,
                                    x58___): x59___.m___('pcdata', [
                                      '\x26lt;?', x60___, x61___, x58___ ]);
                                }
                              }
                              break;
                            case '\x3e':
                              if (h.pcdata_v___? h.pcdata: h.v___('pcdata')) {
                                x63___ = h, x64___ = param, x65___ =
                                  continuationMarker, x62___ =
                                  continuationMaker.i___(h, parts, pos, state,
                                  param), x63___.pcdata_m___?
                                  x63___.pcdata('\x26gt;', x64___, x65___,
                                  x62___): x63___.m___('pcdata', [ '\x26gt;',
                                    x64___, x65___, x62___ ]);
                              }
                              break;
                            case '':
                              break;
                            default:
                              if (h.pcdata_v___? h.pcdata: h.v___('pcdata')) {
                                x67___ = h, x68___ = current, x69___ = param,
                                x70___ = continuationMarker, x66___ =
                                  continuationMaker.i___(h, parts, pos, state,
                                  param), x67___.pcdata_m___?
                                  x67___.pcdata(x68___, x69___, x70___, x66___)
                                  : x67___.m___('pcdata', [ x68___, x69___,
                                    x70___, x66___ ]);
                              }
                              break;
                            }
                          }
                          if (h.endDoc_v___? h.endDoc: h.v___('endDoc')) {
                            h.endDoc_m___? h.endDoc(param): h.m___('endDoc', [
                                param ]);
                          }
                        } catch (e) {
                          if (e !== continuationMarker) { throw e; }
                        }
                      }
                      ___.f(parseCPS, 'parseCPS');
                    }
                    {
                      function htmlSplit(str) {
                        var re, parts, lastPos, m, x0___, x1___, x2___, x3___,
                        x4___, x5___, x6___, x7___, x8___;
                        re = new
                          RegExp.new___('(\x3c\\/|\x3c\\!--|\x3c[!?]|[\x26\x3c\x3e])',
                          'g');
                        str = str + '';
                        if (splitWillCapture) {
                          return str.split_m___? str.split(re):
                          str.m___('split', [ re ]);
                        } else {
                          parts = [ ];
                          lastPos = 0;
                          while ((m = re.exec_m___? re.exec(str):
                              re.m___('exec', [ str ])) !== null) {
                            x4___ = parts, x3___ = (x1___ = str, x2___ =
                              lastPos, x0___ = m.index_v___? m.index:
                              m.v___('index'), x1___.substring_m___?
                              x1___.substring(x2___, x0___):
                              x1___.m___('substring', [ x2___, x0___ ])),
                            x4___.push_m___? x4___.push(x3___):
                            x4___.m___('push', [ x3___ ]);
                            x6___ = parts, x5___ = m[ +0 ], x6___.push_m___?
                              x6___.push(x5___): x6___.m___('push', [ x5___ ]);
                            lastPos = (m.index_v___? m.index: m.v___('index'))
                              + m[ +0 ].length;
                          }
                          x8___ = parts, x7___ = str.substring_m___?
                            str.substring(lastPos): str.m___('substring', [
                              lastPos ]), x8___.push_m___? x8___.push(x7___):
                          x8___.m___('push', [ x7___ ]);
                          return parts;
                        }
                      }
                      ___.f(htmlSplit, 'htmlSplit');
                    }
                    {
                      function parseEndTag(parts, pos, h, param,
                        continuationMarker, state) {
                        var tag, x0___, x1___, x2___, x3___, x4___;
                        tag = parseTagAndAttrs.i___(parts, pos);
                        if (!tag) {
                          return parts.length;
                        }
                        if (h.endTag_v___? h.endTag: h.v___('endTag')) {
                          x2___ = h, x0___ = tag.name_v___? tag.name:
                          tag.v___('name'), x3___ = param, x4___ =
                            continuationMarker, x1___ =
                            continuationMaker.i___(h, parts, pos, state, param)
                            , x2___.endTag_m___? x2___.endTag(x0___, x3___,
                            x4___, x1___): x2___.m___('endTag', [ x0___, x3___,
                              x4___, x1___ ]);
                        }
                        return tag.next_v___? tag.next: tag.v___('next');
                      }
                      ___.f(parseEndTag, 'parseEndTag');
                    }
                    {
                      function parseStartTag(parts, pos, h, param,
                        continuationMarker, state) {
                        var tag, x0___, x1___, x2___, x3___, x4___, x5___;
                        tag = parseTagAndAttrs.i___(parts, pos);
                        if (!tag) {
                          return parts.length;
                        }
                        if (h.startTag_v___? h.startTag: h.v___('startTag')) {
                          x3___ = h, x0___ = tag.name_v___? tag.name:
                          tag.v___('name'), x1___ = tag.attrs_v___? tag.attrs:
                          tag.v___('attrs'), x4___ = param, x5___ =
                            continuationMarker, x2___ =
                            continuationMaker.i___(h, parts, tag.next_v___?
                            tag.next: tag.v___('next'), state, param),
                          x3___.startTag_m___? x3___.startTag(x0___, x1___,
                            x4___, x5___, x2___): x3___.m___('startTag', [
                              x0___, x1___, x4___, x5___, x2___ ]);
                        }
                        if ((tag.eflags_v___? tag.eflags: tag.v___('eflags')) &
                            EFLAGS_TEXT) {
                          return parseText.i___(parts, tag, h, param,
                            continuationMarker, state);
                        } else {
                          return tag.next_v___? tag.next: tag.v___('next');
                        }
                      }
                      ___.f(parseStartTag, 'parseStartTag');
                    }
                    {
                      function parseText(parts, tag, h, param,
                        continuationMarker, state) {
                        var end, x0___, x1___, re, first, p, x2___, x3___, buf,
                        x4___, x5___, x6___, x7___, x8___, x9___, x10___,
                        x11___, x12___, x13___, x14___;
                        end = parts.length;
                        if (! (x1___ = endTagRe, x0___ = tag.name_v___?
                              tag.name: tag.v___('name'),
                            x1___.hasOwnProperty_m___?
                              x1___.hasOwnProperty(x0___):
                            x1___.m___('hasOwnProperty', [ x0___ ]))) {
                          endTagRe.w___(tag.name_v___? tag.name:
                            tag.v___('name'), new (IMPORTS___.RegExp_v___?
                              IMPORTS___.RegExp: ___.ri(IMPORTS___, 'RegExp'))
                            .new___(('^' + (tag.name_v___? tag.name:
                                  tag.v___('name')) + '(?:[\\s\\/]|$)'), 'i'));
                        }
                        re = endTagRe.v___(tag.name_v___? tag.name:
                          tag.v___('name'));
                        first = tag.next_v___? tag.next: tag.v___('next');
                        p = (tag.next_v___? tag.next: tag.v___('next')) + 1;
                        for (; p < end; p++) {
                          if (parts[ + (p - 1) ] === '\x3c/' && (x3___ = re,
                              x2___ = parts.v___(p), x3___.test_m___?
                                x3___.test(x2___): x3___.m___('test', [ x2___ ]
                              ))) { break; }
                        }
                        if (p < end) {
                          p = p - 1;
                        }
                        buf = (x4___ = parts.slice_m___? parts.slice(first, p):
                          parts.m___('slice', [ first, p ]), x4___.join_m___?
                          x4___.join(''): x4___.m___('join', [ '' ]));
                        if ((tag.eflags_v___? tag.eflags: tag.v___('eflags')) &
                          (html4.eflags_v___? html4.eflags:
                           html4.v___('eflags')).v___('CDATA')) {
                          if (h.cdata_v___? h.cdata: h.v___('cdata')) {
                            x6___ = h, x7___ = buf, x8___ = param, x9___ =
                              continuationMarker, x5___ =
                              continuationMaker.i___(h, parts, p, state, param)
                              , x6___.cdata_m___? x6___.cdata(x7___, x8___,
                              x9___, x5___): x6___.m___('cdata', [ x7___,
                                x8___, x9___, x5___ ]);
                          }
                        } else if ((tag.eflags_v___? tag.eflags:
                            tag.v___('eflags')) & (html4.eflags_v___?
                              html4.eflags: html4.v___('eflags'))
                          .v___('RCDATA')) {
                          if (h.rcdata_v___? h.rcdata: h.v___('rcdata')) {
                            x12___ = h, x10___ = normalizeRCData.i___(buf),
                            x13___ = param, x14___ = continuationMarker, x11___
                              = continuationMaker.i___(h, parts, p, state,
                              param), x12___.rcdata_m___? x12___.rcdata(x10___,
                              x13___, x14___, x11___): x12___.m___('rcdata', [
                                x10___, x13___, x14___, x11___ ]);
                          }
                        } else {
                          throw new (IMPORTS___.Error_v___? IMPORTS___.Error:
                            ___.ri(IMPORTS___, 'Error')).new___('bug');
                        }
                        return p;
                      }
                      ___.f(parseText, 'parseText');
                    }
                    {
                      function parseTagAndAttrs(parts, pos) {
                        var m, x0___, x1___, tag, x2___, x3___, x4___, x5___,
                        x6___, buf, x7___, x8___, p, end, attrs, x9___, x10___,
                        quote, sawQuote, abuf, x11___, x12___, x13___, x14___,
                        aName, x15___, aValue, x16___, x17___, x18___, x19___;
                        m = (x0___ = new RegExp.new___('^([-\\w:]+)'), x1___ =
                            parts.v___(pos), x0___.exec_m___? x0___.exec(x1___)
                            : x0___.m___('exec', [ x1___ ]));
                        tag = ___.iM([ ]);
                        x4___ = tag, x3___ = (x2___ = m[ +1 ],
                          x2___.toLowerCase_m___? x2___.toLowerCase():
                          x2___.m___('toLowerCase', [ ])), x4___.name_w___ ===
                          x4___? (x4___.name = x3___): x4___.w___('name',
                          x3___);
                        x6___ = tag, x5___ = (html4.ELEMENTS_v___?
                            html4.ELEMENTS: html4.v___('ELEMENTS'))
                          .v___(tag.name_v___? tag.name: tag.v___('name')),
                        x6___.eflags_w___ === x6___? (x6___.eflags = x5___):
                        x6___.w___('eflags', x5___);
                        buf = (x7___ = parts.v___(pos), x8___ = m[ +0 ].length,
                          x7___.substr_m___? x7___.substr(x8___):
                          x7___.m___('substr', [ x8___ ]));
                        p = pos + 1;
                        end = parts.length;
                        for (; p < end; p++) {
                          if (parts.v___(p) === '\x3e') { break; }
                          buf = buf + parts.v___(p);
                        }
                        if (end <= p) { return void 0; }
                        attrs = [ ];
                        while (buf !== '') {
                          m = ATTR_RE.exec_m___? ATTR_RE.exec(buf):
                          ATTR_RE.m___('exec', [ buf ]);
                          if (!m) {
                            buf = (x10___ = buf, x9___ = new
                                RegExp.new___('^[\\s\\S][^a-z\\s]*'),
                              x10___.replace_m___? x10___.replace(x9___, ''):
                              x10___.m___('replace', [ x9___, '' ]));
                          } else if (m[ +4 ] && !m[ +5 ] || m[ +6 ] && !m[ +7 ]
                          ) {
                            quote = m[ +4 ] || m[ +6 ];
                            sawQuote = false;
                            abuf = [ buf, parts.v___(p++) ];
                            for (; p < end; p++) {
                              if (sawQuote) {
                                if (parts.v___(p) === '\x3e') { break; }
                              } else if (0 <= (x11___ = parts.v___(p), x12___ =
                                    quote, x11___.indexOf_m___?
                                    x11___.indexOf(x12___):
                                  x11___.m___('indexOf', [ x12___ ]))) {
                                sawQuote = true; }
                              x14___ = abuf, x13___ = parts.v___(p),
                              x14___.push_m___? x14___.push(x13___):
                              x14___.m___('push', [ x13___ ]);
                            }
                            if (end <= p) { break; }
                            buf = abuf.join_m___? abuf.join(''):
                            abuf.m___('join', [ '' ]);
                            continue;
                          } else {
                            aName = (x15___ = m[ +1 ], x15___.toLowerCase_m___?
                                x15___.toLowerCase():
                              x15___.m___('toLowerCase', [ ]));
                            aValue = m[ +2 ] ? decodeValue.i___(m[ +3 ]):
                            aName;
                            attrs.push_m___? attrs.push(aName, aValue):
                            attrs.m___('push', [ aName, aValue ]);
                            buf = (x17___ = buf, x16___ = m[ +0 ].length,
                              x17___.substr_m___? x17___.substr(x16___):
                              x17___.m___('substr', [ x16___ ]));
                          }
                        }
                        tag.attrs_w___ === tag? (tag.attrs = attrs):
                        tag.w___('attrs', attrs);
                        x19___ = tag, x18___ = p + 1, x19___.next_w___ ===
                          x19___? (x19___.next = x18___): x19___.w___('next',
                          x18___);
                        return tag;
                      }
                      ___.f(parseTagAndAttrs, 'parseTagAndAttrs');
                    }
                    {
                      function decodeValue(v) {
                        var q, x0___, x1___;
                        q = v.charCodeAt_m___? v.charCodeAt(0):
                        v.m___('charCodeAt', [ 0 ]);
                        if (q === 34 || q === 39) {
                          v = (x1___ = v, x0___ = v.length - 2,
                            x1___.substr_m___? x1___.substr(1, x0___):
                            x1___.m___('substr', [ 1, x0___ ]));
                        }
                        return unescapeEntities.i___(stripNULs.i___(v));
                      }
                      ___.f(decodeValue, 'decodeValue');
                    }
                    {
                      function isVirtualizedElementName(elementName) {
                        return VIRTUALIZED_ELEMENT_NAME_RE.test_m___?
                          VIRTUALIZED_ELEMENT_NAME_RE.test(elementName):
                        VIRTUALIZED_ELEMENT_NAME_RE.m___('test', [ elementName
                          ]);
                      }
                      ___.f(isVirtualizedElementName,
                        'isVirtualizedElementName');
                    }
                    {
                      function realToVirtualElementName(elementName) {
                        var match;
                        match = VIRTUALIZED_ELEMENT_NAME_RE.exec_m___?
                          VIRTUALIZED_ELEMENT_NAME_RE.exec(elementName):
                        VIRTUALIZED_ELEMENT_NAME_RE.m___('exec', [ elementName
                          ]);
                        return match? match[ +1 ]: elementName;
                      }
                      ___.f(realToVirtualElementName,
                        'realToVirtualElementName');
                    }
                    {
                      function virtualToRealElementName(elementName) {
                        var x0___, x1___, x2___, x3___, x4___;
                        if ((x2___ = (x1___ = (x0___ = IMPORTS___.Object_v___?
                                IMPORTS___.Object: ___.ri(IMPORTS___, 'Object')
                                , x0___.prototype_v___? x0___.prototype:
                                x0___.v___('prototype')),
                              x1___.hasOwnProperty_v___? x1___.hasOwnProperty:
                              x1___.v___('hasOwnProperty')), x3___ =
                            html4.ELEMENTS_v___? html4.ELEMENTS:
                            html4.v___('ELEMENTS'), x4___ = elementName,
                            x2___.call_m___? x2___.call(x3___, x4___):
                            x2___.m___('call', [ x3___, x4___ ])) &&
                          (html4.ELEMENTS_v___? html4.ELEMENTS:
                           html4.v___('ELEMENTS')).v___(elementName) &
                          (html4.eflags_v___? html4.eflags:
                           html4.v___('eflags')).v___('VIRTUALIZED')) {
                          return VIRTUALIZED_ELEMENT_PREFIX + elementName;
                        } else {
                          return elementName;
                        }
                      }
                      ___.f(virtualToRealElementName,
                        'virtualToRealElementName');
                    }
                    {
                      function makeHtmlSanitizer(tagPolicy) {
                        var stack, ignoring, emit;
                        emit = (function () {
                            function emit$_var(text, out) {
                              if (!ignoring) {
                                out.push_m___? out.push(text): out.m___('push',
                                  [ text ]);
                              }
                            }
                            return ___.f(emit$_var, 'emit$_var');
                          })();
                        return makeSaxParser.i___(___.iM([ 'startDoc',
                              (function () {
                                 function startDoc$_lit(_) {
                                   stack = [ ];
                                   ignoring = false;
                                 }
                                 return ___.f(startDoc$_lit, 'startDoc$_lit');
                               })(), 'startTag', (function () {
                                  function startTag$_lit(tagName, attribs, out)
                                    {
                                    var x0___, x1___, eflags, i, n, attribName,
                                    value, x2___, x3___, x4___;
                                    if (ignoring) { return; }
                                    if (! (x0___ = html4.ELEMENTS_v___?
                                          html4.ELEMENTS:
                                        html4.v___('ELEMENTS'), x1___ =
                                          tagName, x0___.hasOwnProperty_m___?
                                          x0___.hasOwnProperty(x1___):
                                        x0___.m___('hasOwnProperty', [ x1___ ])
                                      )) { return; }
                                    eflags = (html4.ELEMENTS_v___?
                                        html4.ELEMENTS: html4.v___('ELEMENTS'))
                                      .v___(tagName);
                                    if (eflags & (html4.eflags_v___?
                                        html4.eflags: html4.v___('eflags'))
                                        .v___('FOLDABLE')) { return; }
                                    attribs = tagPolicy.i___(tagName, attribs);
                                    if (!attribs) {
                                      ignoring = ! (eflags &
                                        (html4.eflags_v___? html4.eflags:
                                         html4.v___('eflags')).v___('EMPTY'));
                                      return;
                                    }
                                    if (! (eflags & (html4.eflags_v___?
                                            html4.eflags: html4.v___('eflags'))
                                          .v___('EMPTY'))) {
                                      stack.push_m___? stack.push(tagName):
                                      stack.m___('push', [ tagName ]);
                                    }
                                    out.push_m___? out.push('\x3c', tagName):
                                    out.m___('push', [ '\x3c', tagName ]);
                                    for (i = 0, n = attribs.length; i < n; i =
                                        i + 2) {
                                      attribName = attribs.v___(i), value =
                                        attribs.v___(i + 1);
                                      if (value !== null && value !== void 0) {
                                        x3___ = out, x4___ = attribName, x2___
                                          = escapeAttrib.i___(value),
                                        x3___.push_m___? x3___.push(' ', x4___,
                                          '=\"', x2___, '\"'):
                                        x3___.m___('push', [ ' ', x4___, '=\"',
                                            x2___, '\"' ]);
                                      }
                                    }
                                    out.push_m___? out.push('\x3e'):
                                    out.m___('push', [ '\x3e' ]);
                                  }
                                  return ___.f(startTag$_lit, 'startTag$_lit');
                                })(), 'endTag', (function () {
                                  function endTag$_lit(tagName, out) {
                                    var x0___, x1___, eflags, index, stackEl,
                                    i, stackEl;
                                    if (ignoring) {
                                      ignoring = false;
                                      return;
                                    }
                                    if (! (x0___ = html4.ELEMENTS_v___?
                                          html4.ELEMENTS:
                                        html4.v___('ELEMENTS'), x1___ =
                                          tagName, x0___.hasOwnProperty_m___?
                                          x0___.hasOwnProperty(x1___):
                                        x0___.m___('hasOwnProperty', [ x1___ ])
                                      )) { return; }
                                    eflags = (html4.ELEMENTS_v___?
                                      html4.ELEMENTS: html4.v___('ELEMENTS'))
                                      .v___(tagName);
                                    if (! (eflags & ((html4.eflags_v___?
                                            html4.eflags: html4.v___('eflags'))
                                          .v___('EMPTY') | (html4.eflags_v___?
                                            html4.eflags: html4.v___('eflags'))
                                          .v___('FOLDABLE')))) {
                                      if (eflags & (html4.eflags_v___?
                                          html4.eflags: html4.v___('eflags'))
                                        .v___('OPTIONAL_ENDTAG')) {
                                        for (index = stack.length; --index >=
                                          0;) {
                                          stackEl = stack.v___(index);
                                          if (stackEl === tagName) { break; }
                                          if (! ((html4.ELEMENTS_v___?
                                                html4.ELEMENTS:
                                                html4.v___('ELEMENTS'))
                                              .v___(stackEl) &
                                              (html4.eflags_v___? html4.eflags:
                                               html4.v___('eflags'))
                                              .v___('OPTIONAL_ENDTAG'))) {
                                            return; }
                                        }
                                      } else {
                                        for (index = stack.length; --index >=
                                          0;) {
                                          if (stack.v___(index) === tagName) {
                                            break; }
                                        }
                                      }
                                      if (index < 0) { return; }
                                      for (i = stack.length; --i > index;) {
                                        stackEl = stack.v___(i);
                                        if (! ((html4.ELEMENTS_v___?
                                                html4.ELEMENTS:
                                              html4.v___('ELEMENTS'))
                                            .v___(stackEl) &
                                              (html4.eflags_v___? html4.eflags:
                                              html4.v___('eflags'))
                                              .v___('OPTIONAL_ENDTAG'))) {
                                          out.push_m___? out.push('\x3c/',
                                            stackEl, '\x3e'): out.m___('push',
                                            [ '\x3c/', stackEl, '\x3e' ]);
                                        }
                                      }
                                      stack.length_w___ === stack?
                                        (stack.length = index):
                                      stack.w___('length', index);
                                      out.push_m___? out.push('\x3c/', tagName,
                                        '\x3e'): out.m___('push', [ '\x3c/',
                                          tagName, '\x3e' ]);
                                    }
                                  }
                                  return ___.f(endTag$_lit, 'endTag$_lit');
                                })(), 'pcdata', emit, 'rcdata', emit, 'cdata',
                              emit, 'endDoc', (function () {
                                  function endDoc$_lit(out) {
                                    var x0___, x1___, x2___, x3___, x4___;
                                    for (; stack.length; void 0, x0___ =
                                      +stack.length, (x2___ = stack, x1___ =
                                        x0___ - 1, x2___.length_w___ === x2___?
                                        (x2___.length = x1___):
                                        x2___.w___('length', x1___)), x0___) {
                                      x4___ = out, x3___ = stack[ +
                                          (stack.length - 1) ],
                                      x4___.push_m___? x4___.push('\x3c/',
                                        x3___, '\x3e'): x4___.m___('push', [
                                          '\x3c/', x3___, '\x3e' ]);
                                    }
                                  }
                                  return ___.f(endDoc$_lit, 'endDoc$_lit');
                                })() ]));
                      }
                      ___.f(makeHtmlSanitizer, 'makeHtmlSanitizer');
                    }
                    {
                      function safeUri(uri, effect, ltype, hints,
                        naiveUriRewriter) {
                        var parsed, x0___, x1___, x2___, x3___;
                        if (!naiveUriRewriter) { return null; }
                        parsed = (x0___ = '' + uri, x1___ = URI_SCHEME_RE,
                          x0___.match_m___? x0___.match(x1___):
                          x0___.m___('match', [ x1___ ]));
                        if (parsed && (!parsed[ +1 ] || (x3___ =
                                ALLOWED_URI_SCHEMES, x2___ = parsed[ +1 ],
                              x3___.test_m___? x3___.test(x2___):
                              x3___.m___('test', [ x2___ ])))) {
                          return naiveUriRewriter.i___(uri, effect, ltype,
                            hints);
                        } else { return null; }
                      }
                      ___.f(safeUri, 'safeUri');
                    }
                    {
                      function log(logger, tagName, attribName, oldValue,
                        newValue) {
                        var changed;
                        if (!attribName) {
                          logger.i___(tagName + ' removed', ___.iM([ 'change',
                                'removed', 'tagName', tagName ]));
                        }
                        if (oldValue !== newValue) {
                          changed = 'changed';
                          if (oldValue && !newValue) {
                            changed = 'removed';
                          } else if (!oldValue && newValue) { changed =
                              'added'; }
                          logger.i___(tagName + '.' + attribName + ' ' +
                              changed, ___.iM([ 'change', changed, 'tagName',
                                tagName, 'attribName', attribName, 'oldValue',
                                oldValue, 'newValue', newValue ]));
                        }
                      }
                      ___.f(log, 'log');
                    }
                    {
                      function lookupAttribute(map, tagName, attribName) {
                        var attribKey;
                        attribKey = tagName + '::' + attribName;
                        if (map.hasOwnProperty_m___?
                          map.hasOwnProperty(attribKey):
                          map.m___('hasOwnProperty', [ attribKey ])) {
                          return map.v___(attribKey);
                        }
                        attribKey = '*::' + attribName;
                        if (map.hasOwnProperty_m___?
                          map.hasOwnProperty(attribKey):
                          map.m___('hasOwnProperty', [ attribKey ])) {
                          return map.v___(attribKey);
                        }
                        return void 0;
                      }
                      ___.f(lookupAttribute, 'lookupAttribute');
                    }
                    {
                      function getAttributeType(tagName, attribName) {
                        return lookupAttribute.i___(html4.ATTRIBS_v___?
                          html4.ATTRIBS: html4.v___('ATTRIBS'), tagName,
                          attribName);
                      }
                      ___.f(getAttributeType, 'getAttributeType');
                    }
                    {
                      function getLoaderType(tagName, attribName) {
                        return lookupAttribute.i___(html4.LOADERTYPES_v___?
                          html4.LOADERTYPES: html4.v___('LOADERTYPES'),
                          tagName, attribName);
                      }
                      ___.f(getLoaderType, 'getLoaderType');
                    }
                    {
                      function getUriEffect(tagName, attribName) {
                        return lookupAttribute.i___(html4.URIEFFECTS_v___?
                            html4.URIEFFECTS: html4.v___('URIEFFECTS'),
                          tagName, attribName);
                      }
                      ___.f(getUriEffect, 'getUriEffect');
                    }
                    {
                      function sanitizeAttribs(tagName, attribs,
                        opt_naiveUriRewriter, opt_nmTokenPolicy, opt_logger) {
                        var i, attribName, value, oldValue, atype, attribKey,
                        x0___, x1___, x2___, x3___, sanitizedDeclarations;
                        for (i = 0; i < attribs.length; i = i + 2) {
                          attribName = attribs.v___(i);
                          value = attribs.v___(i + 1);
                          oldValue = value;
                          atype = null;
                          if ((attribKey = tagName + '::' + attribName, (x0___
                                = html4.ATTRIBS_v___? html4.ATTRIBS:
                                html4.v___('ATTRIBS'), x1___ = attribKey,
                                x0___.hasOwnProperty_m___?
                                x0___.hasOwnProperty(x1___):
                                x0___.m___('hasOwnProperty', [ x1___ ]))) ||
                            (attribKey = '*::' + attribName, (x2___ =
                               html4.ATTRIBS_v___? html4.ATTRIBS:
                               html4.v___('ATTRIBS'), x3___ = attribKey,
                               x2___.hasOwnProperty_m___?
                               x2___.hasOwnProperty(x3___):
                               x2___.m___('hasOwnProperty', [ x3___ ])))) {
                            atype = (html4.ATTRIBS_v___? html4.ATTRIBS:
                              html4.v___('ATTRIBS')).v___(attribKey);
                          }
                          if (atype !== null) {
                            switch (atype) {
                            case (html4.atype_v___? html4.atype:
                                html4.v___('atype')).v___('NONE'):
                              break;
                            case (html4.atype_v___? html4.atype:
                                html4.v___('atype')).v___('SCRIPT'):
                              value = null;
                              if (opt_logger) {
                                log.i___(opt_logger, tagName, attribName,
                                  oldValue, value);
                              }
                              break;
                            case (html4.atype_v___? html4.atype:
                                html4.v___('atype')).v___('STYLE'):
                              if ('undefined' === typeof parseCssDeclarations)
                                {
                                value = null;
                                if (opt_logger) {
                                  log.i___(opt_logger, tagName, attribName,
                                    oldValue, value);
                                }
                                break;
                              }
                              sanitizedDeclarations = [ ];
                              parseCssDeclarations.i___(value, ___.iM([
                                    'declaration', (function () {
                                        function declaration$_lit(property,
                                          tokens) {
                                          var normProp, schema, x0___, x1___;
                                          normProp = property.toLowerCase_m___?
                                            property.toLowerCase():
                                          property.m___('toLowerCase', [ ]);
                                          schema = cssSchema.v___(normProp);
                                          if (!schema) { return; }
                                          sanitizeCssProperty.i___(normProp,
                                            schema, tokens,
                                            opt_naiveUriRewriter?
                                            ___.f(function (url) {
                                                var x0___, x1___;
                                                return safeUri.i___(url, (x0___
                                                    = html4.ueffects_v___?
                                                    html4.ueffects:
                                                    html4.v___('ueffects'),
                                                    x0___.SAME_DOCUMENT_v___?
                                                    x0___.SAME_DOCUMENT:
                                                    x0___.v___('SAME_DOCUMENT')
                                                  ), (x1___ =
                                                      html4.ltypes_v___?
                                                      html4.ltypes:
                                                    html4.v___('ltypes'),
                                                    x1___.SANDBOXED_v___?
                                                      x1___.SANDBOXED:
                                                    x1___.v___('SANDBOXED')),
                                                  ___.iM([ 'TYPE', 'CSS',
                                                      'CSS_PROP', normProp ]),
                                                  opt_naiveUriRewriter);
                                              }): null);
                                          x1___ = sanitizedDeclarations, x0___
                                            = property + ': ' +
                                            (tokens.join_m___? tokens.join(' ')
                                            : tokens.m___('join', [ ' ' ])),
                                          x1___.push_m___? x1___.push(x0___):
                                          x1___.m___('push', [ x0___ ]);
                                        }
                                        return ___.f(declaration$_lit,
                                          'declaration$_lit');
                                      })() ]));
                              value = sanitizedDeclarations.length > 0?
                                sanitizedDeclarations.join_m___?
                                sanitizedDeclarations.join(' ; '):
                              sanitizedDeclarations.m___('join', [ ' ; ' ]):
                              null;
                              if (opt_logger) {
                                log.i___(opt_logger, tagName, attribName,
                                  oldValue, value);
                              }
                              break;
                            case (html4.atype_v___? html4.atype:
                                html4.v___('atype')).v___('ID'):
                            case (html4.atype_v___? html4.atype:
                                html4.v___('atype')).v___('IDREF'):
                            case (html4.atype_v___? html4.atype:
                                html4.v___('atype')).v___('IDREFS'):
                            case (html4.atype_v___? html4.atype:
                                html4.v___('atype')).v___('GLOBAL_NAME'):
                            case (html4.atype_v___? html4.atype:
                                html4.v___('atype')).v___('LOCAL_NAME'):
                            case (html4.atype_v___? html4.atype:
                                html4.v___('atype')).v___('CLASSES'):
                              value = opt_nmTokenPolicy?
                                opt_nmTokenPolicy.i___(value): value;
                              if (opt_logger) {
                                log.i___(opt_logger, tagName, attribName,
                                  oldValue, value);
                              }
                              break;
                            case (html4.atype_v___? html4.atype:
                                html4.v___('atype')).v___('URI'):
                              value = safeUri.i___(value,
                                getUriEffect.i___(tagName, attribName),
                                getLoaderType.i___(tagName, attribName),
                                ___.iM([ 'TYPE', 'MARKUP', 'XML_ATTR',
                                    attribName, 'XML_TAG', tagName ]),
                                opt_naiveUriRewriter);
                              if (opt_logger) {
                                log.i___(opt_logger, tagName, attribName,
                                  oldValue, value);
                              }
                              break;
                            case (html4.atype_v___? html4.atype:
                                html4.v___('atype')).v___('URI_FRAGMENT'):
                              if (value && '#' === (value.charAt_m___?
                                  value.charAt(0): value.m___('charAt', [ 0 ]))
                              ) {
                                value = value.substring_m___?
                                  value.substring(1): value.m___('substring', [
                                    1 ]);
                                value = opt_nmTokenPolicy?
                                  opt_nmTokenPolicy.i___(value): value;
                                if (value !== null && value !== void 0) {
                                  value = '#' + value;
                                }
                              } else { value = null; }
                              if (opt_logger) {
                                log.i___(opt_logger, tagName, attribName,
                                  oldValue, value);
                              }
                              break;
                            default:
                              value = null;
                              if (opt_logger) {
                                log.i___(opt_logger, tagName, attribName,
                                  oldValue, value);
                              }
                              break;
                            }
                          } else {
                            value = null;
                            if (opt_logger) {
                              log.i___(opt_logger, tagName, attribName,
                                oldValue, value);
                            }
                          }
                          attribs.w___(i + 1, value);
                        }
                        return attribs;
                      }
                      ___.f(sanitizeAttribs, 'sanitizeAttribs');
                    }
                    {
                      function makeTagPolicy(opt_naiveUriRewriter,
                        opt_nmTokenPolicy, opt_logger) {
                        return ___.f(function (tagName, attribs) {
                            if (! ((html4.ELEMENTS_v___? html4.ELEMENTS:
                                  html4.v___('ELEMENTS')).v___(tagName) &
                                  (html4.eflags_v___? html4.eflags:
                                  html4.v___('eflags')).v___('UNSAFE'))) {
                              return sanitizeAttribs.i___(tagName, attribs,
                                opt_naiveUriRewriter, opt_nmTokenPolicy,
                                opt_logger);
                            } else {
                              if (opt_logger) {
                                log.i___(opt_logger, tagName,
                                  IMPORTS___.undefined_v___?
                                    IMPORTS___.undefined: ___.ri(IMPORTS___,
                                    'undefined'), IMPORTS___.undefined_v___?
                                    IMPORTS___.undefined: ___.ri(IMPORTS___,
                                    'undefined'), IMPORTS___.undefined_v___?
                                    IMPORTS___.undefined: ___.ri(IMPORTS___,
                                    'undefined'));
                              }
                            }
                          });
                      }
                      ___.f(makeTagPolicy, 'makeTagPolicy');
                    }
                    {
                      function sanitizeWithPolicy(inputHtml, tagPolicy) {
                        var outputArray;
                        outputArray = [ ];
                        makeHtmlSanitizer.i___(tagPolicy).i___(inputHtml,
                          outputArray);
                        return outputArray.join_m___? outputArray.join(''):
                        outputArray.m___('join', [ '' ]);
                      }
                      ___.f(sanitizeWithPolicy, 'sanitizeWithPolicy');
                    }
                    {
                      function sanitize(inputHtml, opt_naiveUriRewriter,
                        opt_nmTokenPolicy, opt_logger) {
                        var tagPolicy;
                        tagPolicy = makeTagPolicy.i___(opt_naiveUriRewriter,
                          opt_nmTokenPolicy, opt_logger);
                        return sanitizeWithPolicy.i___(inputHtml, tagPolicy);
                      }
                      ___.f(sanitize, 'sanitize');
                    }
                    if ('undefined' !== typeof IMPORTS___.v___('window')) {
                      parseCssDeclarations = (IMPORTS___.window_v___?
                        IMPORTS___.window: ___.ri(IMPORTS___, 'window'))
                        .v___('parseCssDeclarations');
                      sanitizeCssProperty = (IMPORTS___.window_v___?
                          IMPORTS___.window: ___.ri(IMPORTS___, 'window'))
                        .v___('sanitizeCssProperty');
                      cssSchema = (IMPORTS___.window_v___? IMPORTS___.window:
                        ___.ri(IMPORTS___, 'window')).v___('cssSchema');
                    }
                    ENTITIES = ___.iM([ 'lt', '\x3c', 'gt', '\x3e', 'amp',
                        '\x26', 'nbsp', '\xa0', 'quot', '\"', 'apos', '\'' ]);
                    decimalEscapeRe = new RegExp.new___('^#(\\d+)$');
                    hexEscapeRe = new RegExp.new___('^#x([0-9A-Fa-f]+)$');
                    nulRe = new RegExp.new___('\\0', 'g');
                    ENTITY_RE_1 = new
                      RegExp.new___('\x26(#[0-9]+|#[xX][0-9A-Fa-f]+|\\w+);',
                      'g');
                    ENTITY_RE_2 = new
                      RegExp.new___('^(#[0-9]+|#[xX][0-9A-Fa-f]+|\\w+);');
                    ampRe = new RegExp.new___('\x26', 'g');
                    looseAmpRe = new
                      RegExp.new___('\x26([^a-z#]|#(?:[^0-9x]|x(?:[^0-9a-f]|$)|$)|$)',
                      'gi');
                    ltRe = new RegExp.new___('[\x3c]', 'g');
                    gtRe = new RegExp.new___('\x3e', 'g');
                    quotRe = new RegExp.new___('\\\"', 'g');
                    ATTR_RE = new (IMPORTS___.RegExp_v___? IMPORTS___.RegExp:
                      ___.ri(IMPORTS___, 'RegExp')).new___(('^\\s*' +
                        '([-.:\\w]+)' + '(?:' + ('\\s*(=)\\s*' + '(' +
                          ('(\")[^\"]*(\"|$)' + '|' + '(\')[^\']*(\'|$)' + '|'
                           + '(?=[a-z][-\\w]*\\s*=)' + '|' + '[^\"\'\\s]*') +
                          ')') + ')?'), 'i');
                    splitWillCapture = (x0___ = new RegExp.new___('(,)'),
                      'a,b'.split_m___? 'a,b'.split(x0___): 'a,b'.m___('split',
                        [ x0___ ])).length === 3;
                    EFLAGS_TEXT = (html4.eflags_v___? html4.eflags:
                      html4.v___('eflags')).v___('CDATA') | (html4.eflags_v___?
                        html4.eflags: html4.v___('eflags')).v___('RCDATA');
                    continuationMarker = ___.iM([ ]);
                    endTagRe = ___.iM([ ]);
                    VIRTUALIZED_ELEMENT_NAME_RE = new
                      RegExp.new___('^caja-v-(.*)$', 'i');
                    VIRTUALIZED_ELEMENT_PREFIX = 'caja-v-';
                    URI_SCHEME_RE = new (IMPORTS___.RegExp_v___?
                        IMPORTS___.RegExp: ___.ri(IMPORTS___, 'RegExp'))
                      .new___(('^' + '(?:' + '([^:/?# ]+)' + ':)?'));
                    ALLOWED_URI_SCHEMES = new
                      RegExp.new___('^(?:https?|mailto)$', 'i');
                    html = ___.iM([ ]);
                    x2___ = html, x1___ = html.w___('escapeAttrib',
                      escapeAttrib), x2___.escapeAttrib_w___ === x2___?
                      (x2___.escapeAttrib = x1___): x2___.w___('escapeAttrib',
                      x1___);
                    x4___ = html, x3___ = html.w___('makeHtmlSanitizer',
                      makeHtmlSanitizer), x4___.makeHtmlSanitizer_w___ ===
                      x4___? (x4___.makeHtmlSanitizer = x3___):
                    x4___.w___('makeHtmlSanitizer', x3___);
                    x6___ = html, x5___ = html.w___('makeSaxParser',
                      makeSaxParser), x6___.makeSaxParser_w___ === x6___?
                      (x6___.makeSaxParser = x5___):
                    x6___.w___('makeSaxParser', x5___);
                    x8___ = html, x7___ = html.w___('makeTagPolicy',
                      makeTagPolicy), x8___.makeTagPolicy_w___ === x8___?
                      (x8___.makeTagPolicy = x7___):
                    x8___.w___('makeTagPolicy', x7___);
                    x10___ = html, x9___ = html.w___('normalizeRCData',
                      normalizeRCData), x10___.normalizeRCData_w___ === x10___?
                      (x10___.normalizeRCData = x9___):
                    x10___.w___('normalizeRCData', x9___);
                    x12___ = html, x11___ = html.w___('sanitize', sanitize),
                    x12___.sanitize_w___ === x12___? (x12___.sanitize = x11___)
                      : x12___.w___('sanitize', x11___);
                    x14___ = html, x13___ = html.w___('sanitizeAttribs',
                      sanitizeAttribs), x14___.sanitizeAttribs_w___ === x14___?
                      (x14___.sanitizeAttribs = x13___):
                    x14___.w___('sanitizeAttribs', x13___);
                    x16___ = html, x15___ = html.w___('sanitizeWithPolicy',
                      sanitizeWithPolicy), x16___.sanitizeWithPolicy_w___ ===
                      x16___? (x16___.sanitizeWithPolicy = x15___):
                    x16___.w___('sanitizeWithPolicy', x15___);
                    x18___ = html, x17___ = html.w___('unescapeEntities',
                      unescapeEntities), x18___.unescapeEntities_w___ ===
                      x18___? (x18___.unescapeEntities = x17___):
                    x18___.w___('unescapeEntities', x17___);
                    x20___ = html, x19___ =
                      html.w___('isVirtualizedElementName',
                      isVirtualizedElementName),
                    x20___.isVirtualizedElementName_w___ === x20___?
                      (x20___.isVirtualizedElementName = x19___):
                    x20___.w___('isVirtualizedElementName', x19___);
                    x22___ = html, x21___ =
                      html.w___('realToVirtualElementName',
                      realToVirtualElementName),
                    x22___.realToVirtualElementName_w___ === x22___?
                      (x22___.realToVirtualElementName = x21___):
                    x22___.w___('realToVirtualElementName', x21___);
                    x24___ = html, x23___ =
                      html.w___('virtualToRealElementName',
                      virtualToRealElementName),
                    x24___.virtualToRealElementName_w___ === x24___?
                      (x24___.virtualToRealElementName = x23___):
                    x24___.w___('virtualToRealElementName', x23___);
                    return html;
                  }).i___(IMPORTS___.html4_v___? IMPORTS___.html4:
                  ___.ri(IMPORTS___, 'html4')));
              ___.di(IMPORTS___, 'html_sanitize'),
              IMPORTS___.w___('html_sanitize', (IMPORTS___.html_v___?
                  IMPORTS___.html: ___.ri(IMPORTS___, 'html')).v___('sanitize')
              );
              if (typeof IMPORTS___.v___('window') !== 'undefined') {
                (IMPORTS___.window_v___? IMPORTS___.window: ___.ri(IMPORTS___,
                   'window')).w___('html', IMPORTS___.html_v___?
                    IMPORTS___.html: ___.ri(IMPORTS___, 'html'));
                moduleResult___ = (IMPORTS___.window_v___? IMPORTS___.window:
                  ___.ri(IMPORTS___, 'window')).w___('html_sanitize',
                  IMPORTS___.html_sanitize_v___? IMPORTS___.html_sanitize:
                  ___.ri(IMPORTS___, 'html_sanitize'));
              }
              return moduleResult___;
            },
            'cajolerName': 'com.google.caja',
            'cajolerVersion': '5054',
            'cajoledDate': 1347987483345
          }).instantiate___(___, IMPORTS___), ___.prepareModule({
            'instantiate': function (___, IMPORTS___) {
              var dis___ = IMPORTS___;
              var moduleResult___;
              {
                function HtmlEmitter(makeDOMAccessible, base, opt_domicile,
                  opt_guestGlobal) {
                  var dis___ = this && this.___? void 0: this;
                  var insertionPoint, bridal, detached, idMap,
                  updateInsertionMode, arraySplice, x0___, x1___,
                  HTML5_WHITESPACE_RE, buildIdMap, byId, emitStatic,
                  detachOnto, attach, discard, finish, signalLoaded,
                  handleEmbed, x2___, x3___, x4___, x5___;
                  {
                    function buildIdMap() {
                      var descs, i, desc, id;
                      idMap = ___.iM([ ]);
                      descs = base.getElementsByTagName_m___?
                        base.getElementsByTagName('*'):
                      base.m___('getElementsByTagName', [ '*' ]);
                      for (i = 0; desc = descs[ +i ]; ++i) {
                        desc = makeDOMAccessible.i___(desc);
                        id = desc.getAttributeNode_m___?
                          desc.getAttributeNode('id'):
                        desc.m___('getAttributeNode', [ 'id' ]);
                        id = makeDOMAccessible.i___(id);
                        if (id && (id.value_v___? id.value: id.v___('value')))
                          {
                          idMap.w___((id.value_v___? id.value: id.v___('value')
                            ) + ' map entry', desc);
                        }
                      }
                    }
                    ___.f(buildIdMap, 'buildIdMap');
                  }
                  {
                    function byId(id) {
                      var node, x0___, x1___;
                      if (!idMap) { buildIdMap.i___(); }
                      node = idMap.v___(id + ' map entry');
                      if (node) { return node; }
                      for (; node = (x0___ = base.ownerDocument_v___?
                          base.ownerDocument: base.v___('ownerDocument'), x1___
                          = id, x0___.getElementById_m___?
                          x0___.getElementById(x1___):
                          x0___.m___('getElementById', [ x1___ ]));) {
                        if ((base.contains_v___? base.contains:
                            base.v___('contains')) ? base.contains_m___?
                            base.contains(node): base.m___('contains', [ node ]
                          ): (base.compareDocumentPosition_m___?
                              base.compareDocumentPosition(node):
                            base.m___('compareDocumentPosition', [ node ])) &
                            16) {
                          idMap.w___(id + ' map entry', node);
                          return node;
                        } else {
                          node.id_w___ === node? (node.id = ''):
                          node.w___('id', '');
                        }
                      }
                      return null;
                    }
                    ___.f(byId, 'byId');
                  }
                  {
                    function emitStatic(htmlString) {
                      var x0___, x1___;
                      if (!base) {
                        throw new (IMPORTS___.Error_v___? IMPORTS___.Error:
                          ___.ri(IMPORTS___, 'Error'))
                          .new___(('Host page error: HtmlEmitter.emitStatic called after'
                            + ' document finish()ed'));
                      }
                      x1___ = base, x0___ = (base.innerHTML_v___?
                          base.innerHTML: base.v___('innerHTML')) + htmlString,
                      x1___.innerHTML_w___ === x1___? (x1___.innerHTML = x0___)
                        : x1___.w___('innerHTML', x0___);
                    }
                    ___.f(emitStatic, 'emitStatic');
                  }
                  {
                    function detachOnto(limit, out) {
                      var child, next, anc, greatAnc, sibling, next;
                      for (child = limit.firstChild_v___? limit.firstChild:
                        limit.v___('firstChild'); child; child = next) {
                        child = makeDOMAccessible.i___(child);
                        next = child.nextSibling_v___? child.nextSibling:
                        child.v___('nextSibling');
                        out.push_m___? out.push(child, limit): out.m___('push',
                          [ child, limit ]);
                        limit.removeChild_m___? limit.removeChild(child):
                        limit.m___('removeChild', [ child ]);
                      }
                      for (anc = limit; anc && anc !== base; anc = greatAnc) {
                        greatAnc = anc.parentNode_v___? anc.parentNode:
                        anc.v___('parentNode');
                        greatAnc = makeDOMAccessible.i___(greatAnc);
                        for (sibling = anc.nextSibling_v___? anc.nextSibling:
                          anc.v___('nextSibling'); sibling; sibling = next) {
                          sibling = makeDOMAccessible.i___(sibling);
                          next = sibling.nextSibling_v___? sibling.nextSibling:
                          sibling.v___('nextSibling');
                          out.push_m___? out.push(sibling, greatAnc):
                          out.m___('push', [ sibling, greatAnc ]);
                          greatAnc.removeChild_m___?
                            greatAnc.removeChild(sibling):
                          greatAnc.m___('removeChild', [ sibling ]);
                        }
                      }
                    }
                    ___.f(detachOnto, 'detachOnto');
                  }
                  {
                    function attach(id) {
                      var limit, newDetached, limitAnc, parent, nConsumed,
                      reattach, reattAnc, x0___, x1___, isLimitClosed;
                      limit = byId.i___(id);
                      if (detached) {
                        newDetached = [ 0, 0 ];
                        detachOnto.i___(limit, newDetached);
                        limitAnc = limit;
                        for (; parent = limitAnc.parentNode_v___?
                          limitAnc.parentNode: limitAnc.v___('parentNode');) {
                          limitAnc = parent; }
                        nConsumed = 0;
                        while (nConsumed < detached.length) {
                          reattach = detached.v___(nConsumed);
                          reattAnc = reattach;
                          for (; reattAnc.parentNode_v___? reattAnc.parentNode:
                            reattAnc.v___('parentNode'); reattAnc =
                            reattAnc.parentNode_v___? reattAnc.parentNode:
                            reattAnc.v___('parentNode')) {}
                          x0___ = detached.v___(nConsumed + 1), x1___ =
                            reattach, x0___.appendChild_m___?
                            x0___.appendChild(x1___): x0___.m___('appendChild',
                            [ x1___ ]);
                          nConsumed = nConsumed + 2;
                          if (reattAnc === limitAnc) { break; }
                        }
                        newDetached.NUM____w___ === newDetached? (newDetached[
                            +1 ] = nConsumed): newDetached.w___(+1, nConsumed);
                        arraySplice.apply_m___? arraySplice.apply(detached,
                          newDetached): arraySplice.m___('apply', [ detached,
                            newDetached ]);
                      } else {
                        detached = [ ];
                        detachOnto.i___(limit, detached);
                      }
                      isLimitClosed = detached[ +1 ] !== limit;
                      insertionPoint = isLimitClosed? limit.parentNode_v___?
                        limit.parentNode: limit.v___('parentNode'): limit;
                      updateInsertionMode.i___();
                      return limit;
                    }
                    ___.f(attach, 'attach');
                  }
                  {
                    function discard(placeholder) {
                      var x0___, x1___;
                      x0___ = placeholder.parentNode_v___?
                        placeholder.parentNode: placeholder.v___('parentNode'),
                      x1___ = placeholder, x0___.removeChild_m___?
                        x0___.removeChild(x1___): x0___.m___('removeChild', [
                          x1___ ]);
                    }
                    ___.f(discard, 'discard');
                  }
                  {
                    function finish() {
                      var dis___ = this && this.___? void 0: this;
                      var i, n, x0___, x1___;
                      insertionPoint = null;
                      if (detached) {
                        for (i = 0, n = detached.length; i < n; i = i + 2) {
                          x0___ = detached.v___(i + 1), x1___ =
                            detached.v___(i), x0___.appendChild_m___?
                            x0___.appendChild(x1___): x0___.m___('appendChild',
                            [ x1___ ]);
                        }
                      }
                      idMap = detached = base = null;
                      return dis___;
                    }
                    ___.f(finish, 'finish');
                  }
                  {
                    function signalLoaded() {
                      var dis___ = this && this.___? void 0: this;
                      var domicile;
                      domicile = opt_domicile;
                      if (domicile) {
                        domicile.signalLoaded_m___? domicile.signalLoaded():
                        domicile.m___('signalLoaded', [ ]);
                      }
                      return dis___;
                    }
                    ___.f(signalLoaded, 'signalLoaded');
                  }
                  {
                    function handleEmbed(params) {
                      if (!opt_guestGlobal) { return; }
                      if (! (opt_guestGlobal.cajaHandleEmbed_v___?
                          opt_guestGlobal.cajaHandleEmbed:
                          opt_guestGlobal.v___('cajaHandleEmbed'))) { return; }
                      opt_guestGlobal.cajaHandleEmbed_m___?
                        opt_guestGlobal.cajaHandleEmbed(params):
                      opt_guestGlobal.m___('cajaHandleEmbed', [ params ]);
                    }
                    ___.f(handleEmbed, 'handleEmbed');
                  }
                  if (!base) {
                    throw new (IMPORTS___.Error_v___? IMPORTS___.Error:
                      ___.ri(IMPORTS___, 'Error'))
                      .new___('Host page error: Virtual document element was not provided')
                      ;
                  }
                  base = makeDOMAccessible.i___(base);
                  insertionPoint = base;
                  bridal = (IMPORTS___.bridalMaker_v___?
                    IMPORTS___.bridalMaker: ___.ri(IMPORTS___, 'bridalMaker'))
                    .i___(makeDOMAccessible, base.ownerDocument_v___?
                    base.ownerDocument: base.v___('ownerDocument'));
                  detached = null;
                  idMap = null;
                  arraySplice = (x1___ = (x0___ = IMPORTS___.Array_v___?
                      IMPORTS___.Array: ___.ri(IMPORTS___, 'Array'),
                      x0___.prototype_v___? x0___.prototype:
                      x0___.v___('prototype')), x1___.splice_v___?
                      x1___.splice: x1___.v___('splice'));
                  HTML5_WHITESPACE_RE = new
                    RegExp.new___('^[\\u0009\\u000a\\u000c\\u000d\\u0020]*$');
                  dis___.byId_w___ === dis___? (dis___.byId = byId):
                  dis___.w___('byId', byId);
                  dis___.attach_w___ === dis___? (dis___.attach = attach):
                  dis___.w___('attach', attach);
                  dis___.discard_w___ === dis___? (dis___.discard = discard):
                  dis___.w___('discard', discard);
                  dis___.emitStatic_w___ === dis___? (dis___.emitStatic =
                      emitStatic): dis___.w___('emitStatic', emitStatic);
                  dis___.finish_w___ === dis___? (dis___.finish = finish):
                  dis___.w___('finish', finish);
                  dis___.signalLoaded_w___ === dis___? (dis___.signalLoaded =
                      signalLoaded): dis___.w___('signalLoaded', signalLoaded);
                  x3___ = dis___, x2___ = bridal.setAttribute_v___?
                    bridal.setAttribute: bridal.v___('setAttribute'),
                  x3___.setAttr_w___ === x3___? (x3___.setAttr = x2___):
                  x3___.w___('setAttr', x2___);
                  x5___ = dis___, x4___ = (function () {
                      function rmAttr$_meth(el, attr) {
                        return el.removeAttribute_m___?
                          el.removeAttribute(attr): el.m___('removeAttribute',
                          [ attr ]);
                      }
                      return ___.f(rmAttr$_meth, 'rmAttr$_meth');
                    })(), x5___.rmAttr_w___ === x5___? (x5___.rmAttr = x4___):
                  x5___.w___('rmAttr', x4___);
                  dis___.handleEmbed_w___ === dis___? (dis___.handleEmbed =
                      handleEmbed): dis___.w___('handleEmbed', handleEmbed);
                  ___.f(function (domicile) {
                      var concat, evaluateUntrustedScript, makeCssUriSanitizer,
                      defineUntrustedStylesheet, resolveUntrustedExternal,
                      defineUntrustedExternalStylesheet,
                      evaluateUntrustedExternalScript, lookupAttr,
                      resolveUriRelativeToDocument, cdataContentType,
                      cdataContent, pendingExternal, documentLoaded, depth,
                      normalInsert, normalEndTag, normalText,
                      isHtml5NonWhitespace, insertionModes, insertionMode,
                      originalInsertionMode, documentWriter, x0___, x1___,
                      htmlParser, x2___, x3___, tameDocWrite, x4___, x5___,
                      x6___, x7___, x8___, x9___, x10___, x11___;
                      {
                        function concat(items) {
                          var x0___, x1___, x2___, x3___;
                          return x2___ = (x1___ = (x0___ =
                                IMPORTS___.Array_v___? IMPORTS___.Array:
                              ___.ri(IMPORTS___, 'Array'),
                              x0___.prototype_v___? x0___.prototype:
                              x0___.v___('prototype')), x1___.join_v___?
                            x1___.join: x1___.v___('join')), x3___ = items,
                          x2___.call_m___? x2___.call(x3___, ''):
                          x2___.m___('call', [ x3___, '' ]);
                        }
                        ___.f(concat, 'concat');
                      }
                      {
                        function evaluateUntrustedScript(scriptBaseUri,
                          scriptInnerText) {
                          var errorMessage, cajaVM, compileModule,
                          compiledModule, x0___, x1___, x2___, x3___;
                          if (!opt_guestGlobal) { return; }
                          errorMessage = 'SCRIPT element evaluation failed';
                          cajaVM = opt_guestGlobal.cajaVM_v___?
                            opt_guestGlobal.cajaVM:
                          opt_guestGlobal.v___('cajaVM');
                          if (cajaVM) {
                            compileModule = cajaVM.compileModule_v___?
                              cajaVM.compileModule:
                            cajaVM.v___('compileModule');
                            if (compileModule) {
                              try {
                                compiledModule =
                                  compileModule.i___(scriptInnerText);
                                try {
                                  compiledModule.i___(opt_domicile.window_v___?
                                    opt_domicile.window:
                                    opt_domicile.v___('window'));
                                  return;
                                } catch (runningEx) {
                                  errorMessage = (IMPORTS___.String_v___?
                                    IMPORTS___.String: ___.ri(IMPORTS___,
                                      'String')).i___(runningEx);
                                }
                              } catch (compileEx) {
                                errorMessage = (IMPORTS___.String_v___?
                                    IMPORTS___.String: ___.ri(IMPORTS___,
                                    'String')).i___(compileEx &&
                                    ((compileEx.message_v___?
                                        compileEx.message:
                                      compileEx.v___('message')) ||
                                      (compileEx.description_v___?
                                      compileEx.description:
                                      compileEx.v___('description')))) ||
                                  errorMessage;
                              }
                            }
                          }
                          try {
                            x0___ = opt_domicile.window_v___?
                              opt_domicile.window: opt_domicile.v___('window'),
                            x3___ = errorMessage, x2___ = opt_guestGlobal?
                              (x1___ = opt_guestGlobal.location_v___?
                              opt_guestGlobal.location:
                              opt_guestGlobal.v___('location'),
                              x1___.href_v___? x1___.href: x1___.v___('href')):
                            '', x0___.onerror_m___? x0___.onerror(x3___, x2___,
                              1): x0___.m___('onerror', [ x3___, x2___, 1 ]);
                          } catch (_) {}
                        }
                        ___.f(evaluateUntrustedScript,
                          'evaluateUntrustedScript');
                      }
                      {
                        function makeCssUriSanitizer(baseUri) {
                          return ___.f(function (uri, prop) {
                              var x0___, x1___, x2___, x3___, x4___, x5___,
                              x6___;
                              return domicile && (domicile.cssUri_v___?
                                domicile.cssUri: domicile.v___('cssUri')) ?
                                (x5___ = domicile, x4___ = (x1___ = (x0___ =
                                      IMPORTS___.URI_v___? IMPORTS___.URI:
                                    ___.ri(IMPORTS___, 'URI'),
                                    x0___.utils_v___? x0___.utils:
                                    x0___.v___('utils')), x2___ = baseUri,
                                  x3___ = uri, x1___.resolve_m___?
                                    x1___.resolve(x2___, x3___):
                                  x1___.m___('resolve', [ x2___, x3___ ])),
                                x6___ = prop, x5___.cssUri_m___?
                                x5___.cssUri(x4___, 'image/*', x6___):
                                x5___.m___('cssUri', [ x4___, 'image/*', x6___
                                  ])): void 0;
                            });
                        }
                        ___.f(makeCssUriSanitizer, 'makeCssUriSanitizer');
                      }
                      {
                        function defineUntrustedStylesheet(styleBaseUri,
                          cssText) {
                          var x0___, x1___, x2___, x3___;
                          if (domicile && (domicile.emitCss_v___?
                              domicile.emitCss: domicile.v___('emitCss'))) {
                            x3___ = domicile, x2___ =
                              (IMPORTS___.sanitizeStylesheet_v___?
                              IMPORTS___.sanitizeStylesheet: ___.ri(IMPORTS___,
                                'sanitizeStylesheet')).i___(styleBaseUri,
                              cssText, (x0___ = domicile.suffixStr_v___?
                                domicile.suffixStr: domicile.v___('suffixStr'),
                                x1___ = new RegExp.new___('^-'),
                                x0___.replace_m___? x0___.replace(x1___, ''):
                                x0___.m___('replace', [ x1___, '' ])),
                              makeCssUriSanitizer.i___(styleBaseUri)),
                            x3___.emitCss_m___? x3___.emitCss(x2___):
                            x3___.m___('emitCss', [ x2___ ]);
                          }
                        }
                        ___.f(defineUntrustedStylesheet,
                          'defineUntrustedStylesheet');
                      }
                      {
                        function resolveUntrustedExternal(func, url, mime,
                          marker, continuation) {
                          var x0___, x1___, x2___, x3___;
                          if (domicile && (domicile.fetchUri_v___?
                                domicile.fetchUri: domicile.v___('fetchUri')))
                            {
                            x1___ = domicile, x2___ = url, x3___ = mime, x0___
                              = ___.f(function (result) {
                                if (result && (result.html_v___? result.html:
                                    result.v___('html'))) {
                                  func.i___(url, result.html_v___? result.html:
                                    result.v___('html'));
                                } else {}
                                if (continuation) {
                                  continuation.i___();
                                }
                              }), x1___.fetchUri_m___? x1___.fetchUri(x2___,
                              x3___, x0___): x1___.m___('fetchUri', [ x2___,
                                x3___, x0___ ]);
                            if (marker) { throw marker; }
                          }
                        }
                        ___.f(resolveUntrustedExternal,
                          'resolveUntrustedExternal');
                      }
                      {
                        function defineUntrustedExternalStylesheet(url, marker,
                          continuation) {
                          resolveUntrustedExternal.i___(defineUntrustedStylesheet,
                            url, 'text/css', marker, continuation);
                        }
                        ___.f(defineUntrustedExternalStylesheet,
                          'defineUntrustedExternalStylesheet');
                      }
                      {
                        function evaluateUntrustedExternalScript(url, marker,
                          continuation) {
                          resolveUntrustedExternal.i___(evaluateUntrustedScript,
                            url, 'text/javascript', marker, continuation);
                        }
                        ___.f(evaluateUntrustedExternalScript,
                          'evaluateUntrustedExternalScript');
                      }
                      {
                        function lookupAttr(attribs, attr) {
                          var srcIndex;
                          srcIndex = 0;
                          do {
                            srcIndex = (attribs.indexOf_m___?
                              attribs.indexOf(attr, srcIndex):
                              attribs.m___('indexOf', [ attr, srcIndex ])) + 1;
                          } while (srcIndex && ! (srcIndex & 1));
                          return srcIndex? attribs.v___(srcIndex):
                          IMPORTS___.undefined_v___? IMPORTS___.undefined:
                          ___.ri(IMPORTS___, 'undefined');
                        }
                        ___.f(lookupAttr, 'lookupAttr');
                      }
                      {
                        function resolveUriRelativeToDocument(href) {
                          var x0___, x1___, x2___, x3___, x4___, x5___;
                          if (domicile && (domicile.pseudoLocation_v___?
                              domicile.pseudoLocation:
                              domicile.v___('pseudoLocation')) && (x0___ =
                              domicile.pseudoLocation_v___?
                              domicile.pseudoLocation:
                              domicile.v___('pseudoLocation'), x0___.href_v___?
                              x0___.href: x0___.v___('href'))) {
                            return x2___ = (x1___ = IMPORTS___.URI_v___?
                                IMPORTS___.URI: ___.ri(IMPORTS___, 'URI'),
                              x1___.utils_v___? x1___.utils:
                              x1___.v___('utils')), x4___ = (x3___ =
                                domicile.pseudoLocation_v___?
                                domicile.pseudoLocation:
                              domicile.v___('pseudoLocation'), x3___.href_v___?
                                x3___.href: x3___.v___('href')), x5___ = href,
                            x2___.resolve_m___? x2___.resolve(x4___, x5___):
                            x2___.m___('resolve', [ x4___, x5___ ]);
                          }
                          return href;
                        }
                        ___.f(resolveUriRelativeToDocument,
                          'resolveUriRelativeToDocument');
                      }
                      {
                        function normalInsert(tagName, attribs) {
                          var x0___, x1___, eltype, x2___, x3___, x4___, el,
                          x5___, x6___, x7___, x8___, x9___, x10___;
                          tagName = (x0___ = IMPORTS___.html_v___?
                            IMPORTS___.html: ___.ri(IMPORTS___, 'html'), x1___
                            = tagName, x0___.virtualToRealElementName_m___?
                            x0___.virtualToRealElementName(x1___):
                            x0___.m___('virtualToRealElementName', [ x1___ ]));
                          eltype = (x2___ = IMPORTS___.html4_v___?
                            IMPORTS___.html4: ___.ri(IMPORTS___, 'html4'),
                            x2___.ELEMENTS_v___? x2___.ELEMENTS:
                            x2___.v___('ELEMENTS')).v___(tagName);
                          domicile.sanitizeAttrs_m___?
                            domicile.sanitizeAttrs(tagName, attribs):
                          domicile.m___('sanitizeAttrs', [ tagName, attribs ]);
                          if ((eltype & (x4___ = (x3___ =
                                  IMPORTS___.html4_v___? IMPORTS___.html4:
                                  ___.ri(IMPORTS___, 'html4'),
                                  x3___.eflags_v___? x3___.eflags:
                                  x3___.v___('eflags')), x4___.UNSAFE_v___?
                                  x4___.UNSAFE: x4___.v___('UNSAFE'))) !== 0) {
                            return; }
                          el = bridal.createElement_m___?
                            bridal.createElement(tagName, attribs):
                          bridal.m___('createElement', [ tagName, attribs ]);
                          if (eltype & (x6___ = (x5___ = IMPORTS___.html4_v___?
                                IMPORTS___.html4: ___.ri(IMPORTS___, 'html4'),
                                x5___.eflags_v___? x5___.eflags:
                                x5___.v___('eflags')),
                              x6___.OPTIONAL_ENDTAG_v___?
                              x6___.OPTIONAL_ENDTAG:
                              x6___.v___('OPTIONAL_ENDTAG')) &&
                            (el.tagName_v___? el.tagName: el.v___('tagName'))
                            === (insertionPoint.tagName_v___?
                                insertionPoint.tagName:
                              insertionPoint.v___('tagName'))) {
                            x8___ = documentWriter, x7___ = el.tagName_v___?
                              el.tagName: el.v___('tagName'),
                            x8___.endTag_m___? x8___.endTag(x7___, true):
                            x8___.m___('endTag', [ x7___, true ]);
                          }
                          insertionPoint.appendChild_m___?
                            insertionPoint.appendChild(el):
                          insertionPoint.m___('appendChild', [ el ]);
                          if (! (eltype & (x10___ = (x9___ =
                                    IMPORTS___.html4_v___? IMPORTS___.html4:
                                  ___.ri(IMPORTS___, 'html4'),
                                  x9___.eflags_v___? x9___.eflags:
                                  x9___.v___('eflags')), x10___.EMPTY_v___?
                                x10___.EMPTY: x10___.v___('EMPTY')))) {
                            insertionPoint = el;
                          }
                        }
                        ___.f(normalInsert, 'normalInsert');
                      }
                      {
                        function normalEndTag(tagName) {
                          var x0___, x1___, x2___, anc, x3___, x4___, p;
                          tagName = (x2___ = (x0___ = IMPORTS___.html_v___?
                                IMPORTS___.html: ___.ri(IMPORTS___, 'html'),
                              x1___ = tagName,
                              x0___.virtualToRealElementName_m___?
                                x0___.virtualToRealElementName(x1___):
                              x0___.m___('virtualToRealElementName', [ x1___ ])
                            ), x2___.toUpperCase_m___? x2___.toUpperCase():
                            x2___.m___('toUpperCase', [ ]));
                          anc = insertionPoint;
                          while (anc !== base && ! (x3___ = new
                                RegExp.new___('\\bvdoc-container___\\b'), x4___
                                = anc.className_v___? anc.className:
                              anc.v___('className'), x3___.test_m___?
                                x3___.test(x4___): x3___.m___('test', [ x4___ ]
                              ))) {
                            p = anc.parentNode_v___? anc.parentNode:
                            anc.v___('parentNode');
                            if ((anc.tagName_v___? anc.tagName:
                                anc.v___('tagName')) === tagName) {
                              insertionPoint = p;
                              return;
                            }
                            anc = p;
                          }
                        }
                        ___.f(normalEndTag, 'normalEndTag');
                      }
                      {
                        function normalText(text) {
                          var x0___, x1___, x2___, x3___, x4___, x5___;
                          x5___ = insertionPoint, x4___ = (x0___ =
                            insertionPoint.ownerDocument_v___?
                            insertionPoint.ownerDocument:
                            insertionPoint.v___('ownerDocument'), x3___ =
                            (x1___ = IMPORTS___.html_v___? IMPORTS___.html:
                             ___.ri(IMPORTS___, 'html'), x2___ = text,
                             x1___.unescapeEntities_m___?
                               x1___.unescapeEntities(x2___):
                             x1___.m___('unescapeEntities', [ x2___ ])),
                            x0___.createTextNode_m___?
                            x0___.createTextNode(x3___):
                            x0___.m___('createTextNode', [ x3___ ])),
                          x5___.appendChild_m___? x5___.appendChild(x4___):
                          x5___.m___('appendChild', [ x4___ ]);
                        }
                        ___.f(normalText, 'normalText');
                      }
                      {
                        function isHtml5NonWhitespace(text) {
                          return ! (HTML5_WHITESPACE_RE.test_m___?
                            HTML5_WHITESPACE_RE.test(text):
                            HTML5_WHITESPACE_RE.m___('test', [ text ]));
                        }
                        ___.f(isHtml5NonWhitespace, 'isHtml5NonWhitespace');
                      }
                      if (!domicile || (domicile.writeHook_v___?
                            domicile.writeHook: domicile.v___('writeHook'))) {
                        updateInsertionMode = (function () {
                            function updateInsertionMode$_var() {}
                            return ___.f(updateInsertionMode$_var,
                              'updateInsertionMode$_var');
                          })();
                        return;
                      }
                      cdataContentType = 0;
                      cdataContent = [ ];
                      pendingExternal = IMPORTS___.undefined_v___?
                        IMPORTS___.undefined: ___.ri(IMPORTS___, 'undefined');
                      documentLoaded = IMPORTS___.undefined_v___?
                        IMPORTS___.undefined: ___.ri(IMPORTS___, 'undefined');
                      depth = 0;
                      insertionModes = ___.iM([ 'initial', ___.iM([ 'toString',
                              (function () {
                                 function toString$_lit() { return 'initial'; }
                                 return ___.f(toString$_lit, 'toString$_lit');
                               })(), 'startTag', (function () {
                                  function startTag$_lit(tagName, attribs) {
                                    {
                                      ___.deodorize(arguments, -6);
                                      var a___ = ___.args(arguments);
                                    }
                                    var x0___, x1___, x2___;
                                    insertionMode =
                                      insertionModes.beforeHtml_v___?
                                      insertionModes.beforeHtml:
                                    insertionModes.v___('beforeHtml');
                                    x0___ = insertionMode.startTag_v___?
                                      insertionMode.startTag:
                                    insertionMode.v___('startTag'), x1___ =
                                      IMPORTS___.undefined_v___?
                                      IMPORTS___.undefined: ___.ri(IMPORTS___,
                                      'undefined'), x2___ = a___,
                                    x0___.apply_m___? x0___.apply(x1___, x2___)
                                      : x0___.m___('apply', [ x1___, x2___ ]);
                                  }
                                  return ___.f(startTag$_lit, 'startTag$_lit');
                                })(), 'endTag', (function () {
                                  function endTag$_lit(tagName) {
                                    {
                                      ___.deodorize(arguments, -6);
                                      var a___ = ___.args(arguments);
                                    }
                                    var x0___, x1___, x2___;
                                    insertionMode =
                                      insertionModes.beforeHtml_v___?
                                      insertionModes.beforeHtml:
                                    insertionModes.v___('beforeHtml');
                                    x0___ = insertionMode.endTag_v___?
                                      insertionMode.endTag:
                                    insertionMode.v___('endTag'), x1___ =
                                      IMPORTS___.undefined_v___?
                                      IMPORTS___.undefined: ___.ri(IMPORTS___,
                                      'undefined'), x2___ = a___,
                                    x0___.apply_m___? x0___.apply(x1___, x2___)
                                      : x0___.m___('apply', [ x1___, x2___ ]);
                                  }
                                  return ___.f(endTag$_lit, 'endTag$_lit');
                                })(), 'text', (function () {
                                  function text$_lit(text) {
                                    {
                                      ___.deodorize(arguments, -6);
                                      var a___ = ___.args(arguments);
                                    }
                                    var x0___, x1___, x2___;
                                    if (isHtml5NonWhitespace.i___(text)) {
                                      insertionMode =
                                        insertionModes.beforeHtml_v___?
                                        insertionModes.beforeHtml:
                                      insertionModes.v___('beforeHtml');
                                      x0___ = insertionMode.text_v___?
                                        insertionMode.text:
                                      insertionMode.v___('text'), x1___ =
                                        IMPORTS___.undefined_v___?
                                        IMPORTS___.undefined:
                                      ___.ri(IMPORTS___, 'undefined'), x2___ =
                                        a___, x0___.apply_m___?
                                        x0___.apply(x1___, x2___):
                                      x0___.m___('apply', [ x1___, x2___ ]);
                                    }
                                  }
                                  return ___.f(text$_lit, 'text$_lit');
                                })() ]), 'beforeHtml', ___.iM([ 'toString',
                              (function () {
                                 function toString$_lit() {
                                   return 'before html';
                                 }
                                 return ___.f(toString$_lit, 'toString$_lit');
                               })(), 'startTag', (function () {
                                  function startTag$_lit(tagName, attribs) {
                                    {
                                      ___.deodorize(arguments, -6);
                                      var a___ = ___.args(arguments);
                                    }
                                    var x0___, x1___, x2___;
                                    if (tagName === 'html') {
                                      normalInsert.i___(tagName, attribs);
                                      insertionMode =
                                        insertionModes.beforeHead_v___?
                                        insertionModes.beforeHead:
                                      insertionModes.v___('beforeHead');
                                    } else {
                                      normalInsert.i___('html', [ ]);
                                      insertionMode =
                                        insertionModes.beforeHead_v___?
                                        insertionModes.beforeHead:
                                      insertionModes.v___('beforeHead');
                                      x0___ = insertionMode.startTag_v___?
                                        insertionMode.startTag:
                                      insertionMode.v___('startTag'), x1___ =
                                        IMPORTS___.undefined_v___?
                                        IMPORTS___.undefined:
                                      ___.ri(IMPORTS___, 'undefined'), x2___ =
                                        a___, x0___.apply_m___?
                                        x0___.apply(x1___, x2___):
                                      x0___.m___('apply', [ x1___, x2___ ]);
                                    }
                                  }
                                  return ___.f(startTag$_lit, 'startTag$_lit');
                                })(), 'endTag', (function () {
                                  function endTag$_lit(tagName) {
                                    {
                                      ___.deodorize(arguments, -6);
                                      var a___ = ___.args(arguments);
                                    }
                                    var x0___, x1___, x2___;
                                    if (tagName === 'head' || tagName ===
                                        'body' || tagName === 'html' || tagName
                                        === 'br') {
                                      normalInsert.i___('html', [ ]);
                                      insertionMode =
                                        insertionModes.beforeHead_v___?
                                        insertionModes.beforeHead:
                                      insertionModes.v___('beforeHead');
                                      x0___ = insertionMode.endTag_v___?
                                        insertionMode.endTag:
                                      insertionMode.v___('endTag'), x1___ =
                                        IMPORTS___.undefined_v___?
                                        IMPORTS___.undefined:
                                      ___.ri(IMPORTS___, 'undefined'), x2___ =
                                        a___, x0___.apply_m___?
                                        x0___.apply(x1___, x2___):
                                      x0___.m___('apply', [ x1___, x2___ ]);
                                    } else {}
                                  }
                                  return ___.f(endTag$_lit, 'endTag$_lit');
                                })(), 'text', (function () {
                                  function text$_lit(text) {
                                    {
                                      ___.deodorize(arguments, -6);
                                      var a___ = ___.args(arguments);
                                    }
                                    var x0___, x1___, x2___;
                                    if (isHtml5NonWhitespace.i___(text)) {
                                      normalInsert.i___('html', [ ]);
                                      insertionMode =
                                        insertionModes.beforeHead_v___?
                                        insertionModes.beforeHead:
                                      insertionModes.v___('beforeHead');
                                      x0___ = insertionMode.text_v___?
                                        insertionMode.text:
                                      insertionMode.v___('text'), x1___ =
                                        IMPORTS___.undefined_v___?
                                        IMPORTS___.undefined:
                                      ___.ri(IMPORTS___, 'undefined'), x2___ =
                                        a___, x0___.apply_m___?
                                        x0___.apply(x1___, x2___):
                                      x0___.m___('apply', [ x1___, x2___ ]);
                                    }
                                  }
                                  return ___.f(text$_lit, 'text$_lit');
                                })() ]), 'beforeHead', ___.iM([ 'toString',
                              (function () {
                                 function toString$_lit() {
                                   return 'before head';
                                 }
                                 return ___.f(toString$_lit, 'toString$_lit');
                               })(), 'startTag', (function () {
                                  function startTag$_lit(tagName, attribs) {
                                    {
                                      ___.deodorize(arguments, -6);
                                      var a___ = ___.args(arguments);
                                    }
                                    var x0___, x1___, x2___, x3___, x4___,
                                    x5___, x6___, x7___, x8___;
                                    if (tagName === 'html') {
                                      x1___ = (x0___ =
                                        insertionModes.inBody_v___?
                                        insertionModes.inBody:
                                        insertionModes.v___('inBody'),
                                        x0___.startTag_v___? x0___.startTag:
                                        x0___.v___('startTag')), x2___ =
                                        IMPORTS___.undefined_v___?
                                        IMPORTS___.undefined:
                                      ___.ri(IMPORTS___, 'undefined'), x3___ =
                                        a___, x1___.apply_m___?
                                        x1___.apply(x2___, x3___):
                                      x1___.m___('apply', [ x2___, x3___ ]);
                                    } else if (tagName === 'head') {
                                      normalInsert.i___(tagName, attribs);
                                      insertionMode =
                                        insertionModes.inHead_v___?
                                        insertionModes.inHead:
                                      insertionModes.v___('inHead');
                                    } else {
                                      x5___ = insertionMode, x4___ = [ ],
                                      x5___.startTag_m___?
                                        x5___.startTag('head', x4___):
                                      x5___.m___('startTag', [ 'head', x4___ ])
                                        ;
                                      x6___ = insertionMode.startTag_v___?
                                        insertionMode.startTag:
                                      insertionMode.v___('startTag'), x7___ =
                                        IMPORTS___.undefined_v___?
                                        IMPORTS___.undefined:
                                      ___.ri(IMPORTS___, 'undefined'), x8___ =
                                        a___, x6___.apply_m___?
                                        x6___.apply(x7___, x8___):
                                      x6___.m___('apply', [ x7___, x8___ ]);
                                    }
                                  }
                                  return ___.f(startTag$_lit, 'startTag$_lit');
                                })(), 'endTag', (function () {
                                  function endTag$_lit(tagName) {
                                    {
                                      ___.deodorize(arguments, -6);
                                      var a___ = ___.args(arguments);
                                    }
                                    var x0___, x1___, x2___, x3___, x4___;
                                    if (tagName === 'head' || tagName ===
                                        'body' || tagName === 'html' || tagName
                                        === 'br') {
                                      x1___ = insertionMode, x0___ = [ ],
                                      x1___.startTag_m___?
                                        x1___.startTag('head', x0___):
                                      x1___.m___('startTag', [ 'head', x0___ ])
                                        ;
                                      x2___ = insertionMode.endTag_v___?
                                        insertionMode.endTag:
                                      insertionMode.v___('endTag'), x3___ =
                                        IMPORTS___.undefined_v___?
                                        IMPORTS___.undefined:
                                      ___.ri(IMPORTS___, 'undefined'), x4___ =
                                        a___, x2___.apply_m___?
                                        x2___.apply(x3___, x4___):
                                      x2___.m___('apply', [ x3___, x4___ ]);
                                    } else {}
                                  }
                                  return ___.f(endTag$_lit, 'endTag$_lit');
                                })(), 'text', (function () {
                                  function text$_lit(text) {
                                    {
                                      ___.deodorize(arguments, -6);
                                      var a___ = ___.args(arguments);
                                    }
                                    var x0___, x1___, x2___, x3___, x4___;
                                    if (isHtml5NonWhitespace.i___(text)) {
                                      x1___ = insertionMode, x0___ = [ ],
                                      x1___.startTag_m___?
                                        x1___.startTag('head', x0___):
                                      x1___.m___('startTag', [ 'head', x0___ ])
                                        ;
                                      x2___ = insertionMode.text_v___?
                                        insertionMode.text:
                                      insertionMode.v___('text'), x3___ =
                                        IMPORTS___.undefined_v___?
                                        IMPORTS___.undefined:
                                      ___.ri(IMPORTS___, 'undefined'), x4___ =
                                        a___, x2___.apply_m___?
                                        x2___.apply(x3___, x4___):
                                      x2___.m___('apply', [ x3___, x4___ ]);
                                    }
                                  }
                                  return ___.f(text$_lit, 'text$_lit');
                                })() ]), 'inHead', ___.iM([ 'toString',
                              (function () {
                                 function toString$_lit() { return 'in head'; }
                                 return ___.f(toString$_lit, 'toString$_lit');
                               })(), 'startTag', (function () {
                                  function startTag$_lit(tagName, attribs) {
                                    {
                                      ___.deodorize(arguments, -6);
                                      var a___ = ___.args(arguments);
                                    }
                                    var x0___, x1___, x2___, x3___, x4___,
                                    x5___, x6___;
                                    if (tagName === 'html') {
                                      x1___ = (x0___ =
                                        insertionModes.inBody_v___?
                                        insertionModes.inBody:
                                        insertionModes.v___('inBody'),
                                        x0___.startTag_v___? x0___.startTag:
                                        x0___.v___('startTag')), x2___ =
                                        IMPORTS___.undefined_v___?
                                        IMPORTS___.undefined:
                                      ___.ri(IMPORTS___, 'undefined'), x3___ =
                                        a___, x1___.apply_m___?
                                        x1___.apply(x2___, x3___):
                                      x1___.m___('apply', [ x2___, x3___ ]);
                                    } else if (tagName === 'base' || tagName
                                        === 'basefont' || tagName === 'bgsound'
                                        || tagName === 'command' || tagName ===
                                        'link' || tagName === 'meta' || tagName
                                        === 'noscript' || tagName ===
                                        'noframes' || tagName === 'style' ||
                                        tagName === 'script') {
                                      normalInsert.i___(tagName, attribs);
                                    } else if (tagName === 'title') {
                                      normalInsert.i___(tagName, attribs);
                                      originalInsertionMode = insertionMode;
                                      insertionMode = insertionModes.text_v___?
                                        insertionModes.text:
                                      insertionModes.v___('text');
                                    } else if (tagName === 'head') {} else {
                                      insertionMode.endTag_m___?
                                        insertionMode.endTag('head'):
                                      insertionMode.m___('endTag', [ 'head' ]);
                                      x4___ = insertionMode.startTag_v___?
                                        insertionMode.startTag:
                                      insertionMode.v___('startTag'), x5___ =
                                        IMPORTS___.undefined_v___?
                                        IMPORTS___.undefined:
                                      ___.ri(IMPORTS___, 'undefined'), x6___ =
                                        a___, x4___.apply_m___?
                                        x4___.apply(x5___, x6___):
                                      x4___.m___('apply', [ x5___, x6___ ]);
                                    }
                                  }
                                  return ___.f(startTag$_lit, 'startTag$_lit');
                                })(), 'endTag', (function () {
                                  function endTag$_lit(tagName) {
                                    {
                                      ___.deodorize(arguments, -6);
                                      var a___ = ___.args(arguments);
                                    }
                                    var x0___, x1___, x2___;
                                    if (tagName === 'head') {
                                      insertionPoint =
                                        insertionPoint.parentElement_v___?
                                        insertionPoint.parentElement:
                                      insertionPoint.v___('parentElement');
                                      insertionMode =
                                        insertionModes.afterHead_v___?
                                        insertionModes.afterHead:
                                      insertionModes.v___('afterHead');
                                    } else if (tagName === 'body' || tagName
                                      === 'html' || tagName === 'br') {
                                      insertionMode.endTag_m___?
                                        insertionMode.endTag('head'):
                                      insertionMode.m___('endTag', [ 'head' ]);
                                      x0___ = insertionMode.endTag_v___?
                                        insertionMode.endTag:
                                      insertionMode.v___('endTag'), x1___ =
                                        IMPORTS___.undefined_v___?
                                        IMPORTS___.undefined:
                                      ___.ri(IMPORTS___, 'undefined'), x2___ =
                                        a___, x0___.apply_m___?
                                        x0___.apply(x1___, x2___):
                                      x0___.m___('apply', [ x1___, x2___ ]);
                                    } else {}
                                  }
                                  return ___.f(endTag$_lit, 'endTag$_lit');
                                })(), 'text', (function () {
                                  function text$_lit(text) {
                                    {
                                      ___.deodorize(arguments, -6);
                                      var a___ = ___.args(arguments);
                                    }
                                    var x0___, x1___, x2___;
                                    if (isHtml5NonWhitespace.i___(text)) {
                                      insertionMode.endTag_m___?
                                        insertionMode.endTag('head'):
                                      insertionMode.m___('endTag', [ 'head' ]);
                                      x0___ = insertionMode.text_v___?
                                        insertionMode.text:
                                      insertionMode.v___('text'), x1___ =
                                        IMPORTS___.undefined_v___?
                                        IMPORTS___.undefined:
                                      ___.ri(IMPORTS___, 'undefined'), x2___ =
                                        a___, x0___.apply_m___?
                                        x0___.apply(x1___, x2___):
                                      x0___.m___('apply', [ x1___, x2___ ]);
                                    }
                                  }
                                  return ___.f(text$_lit, 'text$_lit');
                                })() ]), 'afterHead', ___.iM([ 'toString',
                              (function () {
                                 function toString$_lit() {
                                   return 'after head';
                                 }
                                 return ___.f(toString$_lit, 'toString$_lit');
                               })(), 'startTag', (function () {
                                  function startTag$_lit(tagName, attribs) {
                                    {
                                      ___.deodorize(arguments, -6);
                                      var a___ = ___.args(arguments);
                                    }
                                    var x0___, x1___, x2___, x3___, x4___,
                                    x5___, x6___, x7___, x8___;
                                    if (tagName === 'html') {
                                      x1___ = (x0___ =
                                        insertionModes.inBody_v___?
                                        insertionModes.inBody:
                                        insertionModes.v___('inBody'),
                                        x0___.startTag_v___? x0___.startTag:
                                        x0___.v___('startTag')), x2___ =
                                        IMPORTS___.undefined_v___?
                                        IMPORTS___.undefined:
                                      ___.ri(IMPORTS___, 'undefined'), x3___ =
                                        a___, x1___.apply_m___?
                                        x1___.apply(x2___, x3___):
                                      x1___.m___('apply', [ x2___, x3___ ]);
                                    } else if (tagName === 'body') {
                                      normalInsert.i___(tagName, attribs);
                                      insertionMode =
                                        insertionModes.inBody_v___?
                                        insertionModes.inBody:
                                      insertionModes.v___('inBody');
                                    } else if (tagName === 'head') {} else {
                                      x5___ = insertionMode, x4___ = [ ],
                                      x5___.startTag_m___?
                                        x5___.startTag('body', x4___):
                                      x5___.m___('startTag', [ 'body', x4___ ])
                                        ;
                                      x6___ = insertionMode.startTag_v___?
                                        insertionMode.startTag:
                                      insertionMode.v___('startTag'), x7___ =
                                        IMPORTS___.undefined_v___?
                                        IMPORTS___.undefined:
                                      ___.ri(IMPORTS___, 'undefined'), x8___ =
                                        a___, x6___.apply_m___?
                                        x6___.apply(x7___, x8___):
                                      x6___.m___('apply', [ x7___, x8___ ]);
                                    }
                                  }
                                  return ___.f(startTag$_lit, 'startTag$_lit');
                                })(), 'endTag', (function () {
                                  function endTag$_lit(tagName) {
                                    {
                                      ___.deodorize(arguments, -6);
                                      var a___ = ___.args(arguments);
                                    }
                                    var x0___, x1___, x2___, x3___, x4___;
                                    if (tagName === 'body' || tagName ===
                                        'html' || tagName === 'br') {
                                      x1___ = insertionMode, x0___ = [ ],
                                      x1___.startTag_m___?
                                        x1___.startTag('body', x0___):
                                      x1___.m___('startTag', [ 'body', x0___ ])
                                        ;
                                      x2___ = insertionMode.endTag_v___?
                                        insertionMode.endTag:
                                      insertionMode.v___('endTag'), x3___ =
                                        IMPORTS___.undefined_v___?
                                        IMPORTS___.undefined:
                                      ___.ri(IMPORTS___, 'undefined'), x4___ =
                                        a___, x2___.apply_m___?
                                        x2___.apply(x3___, x4___):
                                      x2___.m___('apply', [ x3___, x4___ ]);
                                    } else {}
                                  }
                                  return ___.f(endTag$_lit, 'endTag$_lit');
                                })(), 'text', (function () {
                                  function text$_lit(text) {
                                    {
                                      ___.deodorize(arguments, -6);
                                      var a___ = ___.args(arguments);
                                    }
                                    var x0___, x1___, x2___, x3___, x4___;
                                    if (isHtml5NonWhitespace.i___(text)) {
                                      x1___ = insertionMode, x0___ = [ ],
                                      x1___.startTag_m___?
                                        x1___.startTag('body', x0___):
                                      x1___.m___('startTag', [ 'body', x0___ ])
                                        ;
                                      x2___ = insertionMode.text_v___?
                                        insertionMode.text:
                                      insertionMode.v___('text'), x3___ =
                                        IMPORTS___.undefined_v___?
                                        IMPORTS___.undefined:
                                      ___.ri(IMPORTS___, 'undefined'), x4___ =
                                        a___, x2___.apply_m___?
                                        x2___.apply(x3___, x4___):
                                      x2___.m___('apply', [ x3___, x4___ ]);
                                    } else {
                                      normalText.i___(text);
                                    }
                                  }
                                  return ___.f(text$_lit, 'text$_lit');
                                })() ]), 'inBody', ___.iM([ 'toString',
                              (function () {
                                 function toString$_lit() { return 'in body'; }
                                 return ___.f(toString$_lit, 'toString$_lit');
                               })(), 'startTag', (function () {
                                  function startTag$_lit(tagName, attribs) {
                                    {
                                      ___.deodorize(arguments, -6);
                                      var a___ = ___.args(arguments);
                                    }
                                    var x0___, x1___, x2___, x3___;
                                    if (tagName === 'html') {} else if (tagName
                                      === 'base' || tagName === 'basefont' ||
                                      tagName === 'bgsound' || tagName ===
                                      'command' || tagName === 'link' ||
                                      tagName === 'meta' || tagName ===
                                      'noframes' || tagName === 'script' ||
                                      tagName === 'style' || tagName ===
                                      'title') {
                                      x1___ = (x0___ =
                                          insertionModes.inHead_v___?
                                          insertionModes.inHead:
                                        insertionModes.v___('inHead'),
                                        x0___.startTag_v___? x0___.startTag:
                                        x0___.v___('startTag')), x2___ =
                                        IMPORTS___.undefined_v___?
                                        IMPORTS___.undefined:
                                      ___.ri(IMPORTS___, 'undefined'), x3___ =
                                        a___, x1___.apply_m___?
                                        x1___.apply(x2___, x3___):
                                      x1___.m___('apply', [ x2___, x3___ ]);
                                    } else if (tagName === 'body') {} else {
                                      normalInsert.i___(tagName, attribs);
                                    }
                                  }
                                  return ___.f(startTag$_lit, 'startTag$_lit');
                                })(), 'endTag', (function () {
                                  function endTag$_lit(tagName) {
                                    {
                                      ___.deodorize(arguments, -6);
                                      var a___ = ___.args(arguments);
                                    }
                                    var x0___, x1___, x2___;
                                    if (tagName === 'body') {
                                      insertionMode =
                                        insertionModes.afterBody_v___?
                                        insertionModes.afterBody:
                                      insertionModes.v___('afterBody');
                                    } else if (tagName === 'html') {
                                      insertionMode.endTag_m___?
                                        insertionMode.endTag('body'):
                                      insertionMode.m___('endTag', [ 'body' ]);
                                      x0___ = insertionMode.endTag_v___?
                                        insertionMode.endTag:
                                      insertionMode.v___('endTag'), x1___ =
                                        IMPORTS___.undefined_v___?
                                        IMPORTS___.undefined:
                                      ___.ri(IMPORTS___, 'undefined'), x2___ =
                                        a___, x0___.apply_m___?
                                        x0___.apply(x1___, x2___):
                                      x0___.m___('apply', [ x1___, x2___ ]);
                                    } else {
                                      normalEndTag.i___(tagName);
                                    }
                                  }
                                  return ___.f(endTag$_lit, 'endTag$_lit');
                                })(), 'text', (function () {
                                  function text$_lit(text) {
                                    normalText.i___(text);
                                  }
                                  return ___.f(text$_lit, 'text$_lit');
                                })() ]), 'text', ___.iM([ 'toString', (function
                                () {
                                  function toString$_lit() { return 'text'; }
                                  return ___.f(toString$_lit, 'toString$_lit');
                                })(), 'startTag', (function () {
                                  function startTag$_lit(tagName, attribs) {
                                    throw new (IMPORTS___.Error_v___?
                                        IMPORTS___.Error: ___.ri(IMPORTS___,
                                        'Error'))
                                      .new___(('shouldn\'t happen: start tag \x3c'
                                        + tagName +
                                        '...\x3e while in text insertion mode for '
                                        + (insertionPoint.tagName_v___?
                                            insertionPoint.tagName:
                                          insertionPoint.v___('tagName'))));
                                  }
                                  return ___.f(startTag$_lit, 'startTag$_lit');
                                })(), 'endTag', (function () {
                                  function endTag$_lit(tagName) {
                                    normalEndTag.i___(tagName);
                                    insertionMode = originalInsertionMode;
                                  }
                                  return ___.f(endTag$_lit, 'endTag$_lit');
                                })(), 'text', (function () {
                                  function text$_lit(text) {
                                    normalText.i___(text);
                                  }
                                  return ___.f(text$_lit, 'text$_lit');
                                })() ]), 'afterBody', ___.iM([ 'toString',
                              (function () {
                                 function toString$_lit() {
                                   return 'after body';
                                 }
                                 return ___.f(toString$_lit, 'toString$_lit');
                               })(), 'startTag', (function () {
                                  function startTag$_lit(tagName, attribs) {
                                    {
                                      ___.deodorize(arguments, -6);
                                      var a___ = ___.args(arguments);
                                    }
                                    var x0___, x1___, x2___, x3___, x4___,
                                    x5___, x6___;
                                    if (tagName === 'html') {
                                      x1___ = (x0___ =
                                          insertionModes.inBody_v___?
                                          insertionModes.inBody:
                                        insertionModes.v___('inBody'),
                                        x0___.startTag_v___? x0___.startTag:
                                        x0___.v___('startTag')), x2___ =
                                        IMPORTS___.undefined_v___?
                                        IMPORTS___.undefined:
                                      ___.ri(IMPORTS___, 'undefined'), x3___ =
                                        a___, x1___.apply_m___?
                                        x1___.apply(x2___, x3___):
                                      x1___.m___('apply', [ x2___, x3___ ]);
                                    } else {
                                      insertionMode =
                                        insertionModes.inBody_v___?
                                        insertionModes.inBody:
                                      insertionModes.v___('inBody');
                                      x4___ = insertionMode.startTag_v___?
                                        insertionMode.startTag:
                                      insertionMode.v___('startTag'), x5___ =
                                        IMPORTS___.undefined_v___?
                                        IMPORTS___.undefined:
                                      ___.ri(IMPORTS___, 'undefined'), x6___ =
                                        a___, x4___.apply_m___?
                                        x4___.apply(x5___, x6___):
                                      x4___.m___('apply', [ x5___, x6___ ]);
                                    }
                                  }
                                  return ___.f(startTag$_lit, 'startTag$_lit');
                                })(), 'endTag', (function () {
                                  function endTag$_lit(tagName) {
                                    {
                                      ___.deodorize(arguments, -6);
                                      var a___ = ___.args(arguments);
                                    }
                                    var x0___, x1___, x2___;
                                    if (tagName === 'html') {
                                      insertionMode =
                                        insertionModes.afterAfterBody_v___?
                                        insertionModes.afterAfterBody:
                                      insertionModes.v___('afterAfterBody');
                                    } else {
                                      insertionMode =
                                        insertionModes.inBody_v___?
                                        insertionModes.inBody:
                                      insertionModes.v___('inBody');
                                      x0___ = insertionMode.endTag_v___?
                                        insertionMode.endTag:
                                      insertionMode.v___('endTag'), x1___ =
                                        IMPORTS___.undefined_v___?
                                        IMPORTS___.undefined:
                                      ___.ri(IMPORTS___, 'undefined'), x2___ =
                                        a___, x0___.apply_m___?
                                        x0___.apply(x1___, x2___):
                                      x0___.m___('apply', [ x1___, x2___ ]);
                                    }
                                  }
                                  return ___.f(endTag$_lit, 'endTag$_lit');
                                })(), 'text', (function () {
                                  function text$_lit(text) {
                                    {
                                      ___.deodorize(arguments, -6);
                                      var a___ = ___.args(arguments);
                                    }
                                    var x0___, x1___, x2___, x3___;
                                    if (isHtml5NonWhitespace.i___(text)) {
                                      insertionMode =
                                        insertionModes.inBody_v___?
                                        insertionModes.inBody:
                                      insertionModes.v___('inBody');
                                    }
                                    x1___ = (x0___ =
                                        insertionModes.inBody_v___?
                                        insertionModes.inBody:
                                      insertionModes.v___('inBody'),
                                      x0___.text_v___? x0___.text:
                                      x0___.v___('text')), x2___ =
                                      IMPORTS___.undefined_v___?
                                      IMPORTS___.undefined: ___.ri(IMPORTS___,
                                      'undefined'), x3___ = a___,
                                    x1___.apply_m___? x1___.apply(x2___, x3___)
                                      : x1___.m___('apply', [ x2___, x3___ ]);
                                  }
                                  return ___.f(text$_lit, 'text$_lit');
                                })() ]), 'afterAfterBody', ___.iM([ 'toString',
                              (function () {
                                 function toString$_lit() {
                                   return 'after after body';
                                 }
                                 return ___.f(toString$_lit, 'toString$_lit');
                               })(), 'startTag', (function () {
                                  function startTag$_lit(tagName, attribs) {
                                    {
                                      ___.deodorize(arguments, -6);
                                      var a___ = ___.args(arguments);
                                    }
                                    var x0___, x1___, x2___, x3___, x4___,
                                    x5___, x6___;
                                    if (tagName === 'html') {
                                      x1___ = (x0___ =
                                          insertionModes.inBody_v___?
                                          insertionModes.inBody:
                                        insertionModes.v___('inBody'),
                                        x0___.startTag_v___? x0___.startTag:
                                        x0___.v___('startTag')), x2___ =
                                        IMPORTS___.undefined_v___?
                                        IMPORTS___.undefined:
                                      ___.ri(IMPORTS___, 'undefined'), x3___ =
                                        a___, x1___.apply_m___?
                                        x1___.apply(x2___, x3___):
                                      x1___.m___('apply', [ x2___, x3___ ]);
                                    } else {
                                      insertionMode =
                                        insertionModes.inBody_v___?
                                        insertionModes.inBody:
                                      insertionModes.v___('inBody');
                                      x4___ = insertionMode.startTag_v___?
                                        insertionMode.startTag:
                                      insertionMode.v___('startTag'), x5___ =
                                        IMPORTS___.undefined_v___?
                                        IMPORTS___.undefined:
                                      ___.ri(IMPORTS___, 'undefined'), x6___ =
                                        a___, x4___.apply_m___?
                                        x4___.apply(x5___, x6___):
                                      x4___.m___('apply', [ x5___, x6___ ]);
                                    }
                                  }
                                  return ___.f(startTag$_lit, 'startTag$_lit');
                                })(), 'endTag', (function () {
                                  function endTag$_lit(tagName) {
                                    {
                                      ___.deodorize(arguments, -6);
                                      var a___ = ___.args(arguments);
                                    }
                                    var x0___, x1___, x2___;
                                    insertionMode = insertionModes.inBody_v___?
                                      insertionModes.inBody:
                                    insertionModes.v___('inBody');
                                    x0___ = insertionMode.endTag_v___?
                                      insertionMode.endTag:
                                    insertionMode.v___('endTag'), x1___ =
                                      IMPORTS___.undefined_v___?
                                      IMPORTS___.undefined: ___.ri(IMPORTS___,
                                      'undefined'), x2___ = a___,
                                    x0___.apply_m___? x0___.apply(x1___, x2___)
                                      : x0___.m___('apply', [ x1___, x2___ ]);
                                  }
                                  return ___.f(endTag$_lit, 'endTag$_lit');
                                })(), 'text', (function () {
                                  function text$_lit(text) {
                                    {
                                      ___.deodorize(arguments, -6);
                                      var a___ = ___.args(arguments);
                                    }
                                    var x0___, x1___, x2___, x3___, x4___,
                                    x5___, x6___;
                                    if (isHtml5NonWhitespace.i___(text)) {
                                      insertionMode =
                                        insertionModes.inBody_v___?
                                        insertionModes.inBody:
                                      insertionModes.v___('inBody');
                                      x0___ = insertionMode.text_v___?
                                        insertionMode.text:
                                      insertionMode.v___('text'), x1___ =
                                        IMPORTS___.undefined_v___?
                                        IMPORTS___.undefined:
                                      ___.ri(IMPORTS___, 'undefined'), x2___ =
                                        a___, x0___.apply_m___?
                                        x0___.apply(x1___, x2___):
                                      x0___.m___('apply', [ x1___, x2___ ]);
                                    } else {
                                      x4___ = (x3___ =
                                          insertionModes.inBody_v___?
                                          insertionModes.inBody:
                                        insertionModes.v___('inBody'),
                                        x3___.text_v___? x3___.text:
                                        x3___.v___('text')), x5___ =
                                        IMPORTS___.undefined_v___?
                                        IMPORTS___.undefined:
                                      ___.ri(IMPORTS___, 'undefined'), x6___ =
                                        a___, x4___.apply_m___?
                                        x4___.apply(x5___, x6___):
                                      x4___.m___('apply', [ x5___, x6___ ]);
                                    }
                                  }
                                  return ___.f(text$_lit, 'text$_lit');
                                })() ]) ]);
                      insertionMode = insertionModes.initial_v___?
                        insertionModes.initial: insertionModes.v___('initial');
                      originalInsertionMode = null;
                      updateInsertionMode = (function () {
                          function updateInsertionMode() {
                            var anc, tn, x0___, x1___, x2___, prevtn, x3___,
                            x4___, x5___, x6___;
                            if (insertionPoint === base) {
                              if (insertionPoint.lastChild_v___?
                                insertionPoint.lastChild:
                                insertionPoint.v___('lastChild')) {
                                insertionMode =
                                  insertionModes.afterAfterBody_v___?
                                  insertionModes.afterAfterBody:
                                insertionModes.v___('afterAfterBody');
                              } else {
                                insertionMode = insertionModes.beforeHtml_v___?
                                  insertionModes.beforeHtml:
                                insertionModes.v___('beforeHtml');
                              }
                            } else {
                              for (anc = insertionPoint; anc !== base; anc =
                                anc.parentNode_v___? anc.parentNode:
                                anc.v___('parentNode')) {
                                tn = (x2___ = (x0___ = IMPORTS___.html_v___?
                                      IMPORTS___.html: ___.ri(IMPORTS___,
                                      'html'), x1___ = anc.tagName_v___?
                                      anc.tagName: anc.v___('tagName'),
                                    x0___.realToVirtualElementName_m___?
                                      x0___.realToVirtualElementName(x1___):
                                    x0___.m___('realToVirtualElementName', [
                                        x1___ ])), x2___.toLowerCase_m___?
                                    x2___.toLowerCase():
                                  x2___.m___('toLowerCase', [ ]));
                                switch (tn) {
                                case 'head':
                                  insertionMode = insertionModes.inHead_v___?
                                    insertionModes.inHead:
                                  insertionModes.v___('inHead');
                                  break;
                                case 'body':
                                  insertionMode = insertionModes.inBody_v___?
                                    insertionModes.inBody:
                                  insertionModes.v___('inBody');
                                  break;
                                case 'html':
                                  prevtn = (x6___ = (x3___ =
                                      IMPORTS___.html_v___? IMPORTS___.html:
                                      ___.ri(IMPORTS___, 'html'), x5___ =
                                      (x4___ = (anc.lastChild_v___?
                                         anc.lastChild: anc.v___('lastChild'))
                                       || ___.iM([ ]), x4___.tagName_v___?
                                       x4___.tagName: x4___.v___('tagName')),
                                      x3___.realToVirtualElementName_m___?
                                      x3___.realToVirtualElementName(x5___):
                                      x3___.m___('realToVirtualElementName', [
                                          x5___ ])), x6___.toLowerCase_m___?
                                    x6___.toLowerCase():
                                    x6___.m___('toLowerCase', [ ]));
                                  if (prevtn === (IMPORTS___.undefined_v___?
                                        IMPORTS___.undefined:
                                      ___.ri(IMPORTS___, 'undefined'))) {
                                    insertionMode =
                                      insertionModes.beforeHead_v___?
                                      insertionModes.beforeHead:
                                    insertionModes.v___('beforeHead');
                                  } else {
                                    switch (prevtn) {
                                    case 'head':
                                      insertionMode =
                                        insertionModes.afterHead_v___?
                                        insertionModes.afterHead:
                                      insertionModes.v___('afterHead');
                                      break;
                                    case 'body':
                                      insertionMode =
                                        insertionModes.afterBody_v___?
                                        insertionModes.afterBody:
                                      insertionModes.v___('afterBody');
                                      break;
                                    }
                                  }
                                  break;
                                default:
                                  break;
                                }
                              }
                            }
                          }
                          return ___.f(updateInsertionMode,
                            'updateInsertionMode');
                        })();
                      documentWriter = ___.iM([ 'startDoc', (function () {
                              function startDoc$_lit() {
                                var x0___;
                                if (depth == 0) {
                                  documentLoaded = (x0___ = IMPORTS___.Q_v___?
                                      IMPORTS___.Q: ___.ri(IMPORTS___, 'Q'),
                                    x0___.defer_m___? x0___.defer():
                                    x0___.m___('defer', [ ]));
                                }
                                depth++;
                              }
                              return ___.f(startDoc$_lit, 'startDoc$_lit');
                            })(), 'endDoc', (function () {
                              function endDoc$_lit() {
                                depth--;
                                if (depth == 0) {
                                  documentLoaded.resolve_m___?
                                    documentLoaded.resolve(true):
                                  documentLoaded.m___('resolve', [ true ]);
                                }
                              }
                              return ___.f(endDoc$_lit, 'endDoc$_lit');
                            })(), 'startTag', (function () {
                              function startTag$_lit(tagName, attribs, params,
                                marker, continuation) {
                                var eltype, x0___, x1___, x2___, x3___, x4___,
                                x5___, x6___, x7___, scriptSrc, x8___, x9___,
                                x10___, x11___, x12___, x13___, rel, href,
                                rels, x14___, x15___, res, baseHref, x16___,
                                x17___;
                                eltype = (x0___ = IMPORTS___.html4_v___?
                                  IMPORTS___.html4: ___.ri(IMPORTS___, 'html4')
                                  , x0___.ELEMENTS_v___? x0___.ELEMENTS:
                                  x0___.v___('ELEMENTS')).v___(tagName);
                                if (! (x2___ = (x1___ = IMPORTS___.html4_v___?
                                      IMPORTS___.html4: ___.ri(IMPORTS___,
                                        'html4'), x1___.ELEMENTS_v___?
                                      x1___.ELEMENTS: x1___.v___('ELEMENTS')),
                                    x3___ = tagName, x2___.hasOwnProperty_m___?
                                    x2___.hasOwnProperty(x3___):
                                    x2___.m___('hasOwnProperty', [ x3___ ])) ||
                                  (eltype & (x5___ = (x4___ =
                                       IMPORTS___.html4_v___? IMPORTS___.html4:
                                       ___.ri(IMPORTS___, 'html4'),
                                       x4___.eflags_v___? x4___.eflags:
                                       x4___.v___('eflags')),
                                     x5___.UNSAFE_v___? x5___.UNSAFE:
                                     x5___.v___('UNSAFE'))) !== 0 && ! (eltype
                                      & (x7___ = (x6___ =
                                          IMPORTS___.html4_v___?
                                          IMPORTS___.html4: ___.ri(IMPORTS___,
                                          'html4'), x6___.eflags_v___?
                                          x6___.eflags: x6___.v___('eflags')),
                                      x7___.VIRTUALIZED_v___?
                                      x7___.VIRTUALIZED:
                                      x7___.v___('VIRTUALIZED')))) {
                                  if (tagName === 'script') {
                                    scriptSrc = lookupAttr.i___(attribs, 'src')
                                      ;
                                    if (!scriptSrc) {
                                      cdataContentType = (x9___ = (x8___ =
                                          IMPORTS___.html4_v___?
                                          IMPORTS___.html4: ___.ri(IMPORTS___,
                                            'html4'), x8___.eflags_v___?
                                          x8___.eflags: x8___.v___('eflags')),
                                        x9___.SCRIPT_v___? x9___.SCRIPT:
                                        x9___.v___('SCRIPT'));
                                      pendingExternal =
                                        IMPORTS___.undefined_v___?
                                        IMPORTS___.undefined:
                                      ___.ri(IMPORTS___, 'undefined');
                                    } else {
                                      cdataContentType = (x11___ = (x10___ =
                                          IMPORTS___.html4_v___?
                                          IMPORTS___.html4: ___.ri(IMPORTS___,
                                            'html4'), x10___.eflags_v___?
                                          x10___.eflags: x10___.v___('eflags'))
                                        , x11___.SCRIPT_v___? x11___.SCRIPT:
                                        x11___.v___('SCRIPT'));
                                      pendingExternal = scriptSrc;
                                    }
                                  } else if (tagName === 'style') {
                                    cdataContentType = (x13___ = (x12___ =
                                          IMPORTS___.html4_v___?
                                          IMPORTS___.html4: ___.ri(IMPORTS___,
                                          'html4'), x12___.eflags_v___?
                                          x12___.eflags: x12___.v___('eflags'))
                                      , x13___.STYLE_v___? x13___.STYLE:
                                      x13___.v___('STYLE'));
                                    pendingExternal =
                                      IMPORTS___.undefined_v___?
                                      IMPORTS___.undefined: ___.ri(IMPORTS___,
                                      'undefined');
                                  } else if ('link' === tagName) {
                                    rel = lookupAttr.i___(attribs, 'rel');
                                    href = lookupAttr.i___(attribs, 'href');
                                    rels = rel? (x15___ = (x14___ =
                                        (IMPORTS___.String_v___?
                                         IMPORTS___.String: ___.ri(IMPORTS___,
                                           'String')).i___(rel),
                                        x14___.toLowerCase_m___?
                                        x14___.toLowerCase():
                                        x14___.m___('toLowerCase', [ ])),
                                      x15___.split_m___? x15___.split(' '):
                                      x15___.m___('split', [ ' ' ])): [ ];
                                    if (href && (rels.indexOf_m___?
                                          rels.indexOf('stylesheet'):
                                        rels.m___('indexOf', [ 'stylesheet' ]))
                                      >= 0) {
                                      res =
                                        resolveUriRelativeToDocument.i___(href)
                                        ;
                                      defineUntrustedExternalStylesheet.i___(res,
                                        marker, continuation);
                                    }
                                  } else if (tagName === 'base') {
                                    baseHref = lookupAttr.i___(attribs, 'href')
                                      ;
                                    if (baseHref && domicile) {
                                      x17___ = domicile, x16___ =
                                        resolveUriRelativeToDocument.i___(baseHref)
                                        , x17___.setBaseUri_m___?
                                        x17___.setBaseUri(x16___):
                                      x17___.m___('setBaseUri', [ x16___ ]);
                                    }
                                  }
                                  return;
                                }
                                insertionMode.startTag_m___?
                                  insertionMode.startTag(tagName, attribs):
                                insertionMode.m___('startTag', [ tagName,
                                    attribs ]);
                              }
                              return ___.f(startTag$_lit, 'startTag$_lit');
                            })(), 'endTag', (function () {
                              function endTag$_lit(tagName, optional, marker,
                                continuation) {
                                var eltype, x0___, isScript, x1___, x2___, res,
                                content, x3___, x4___;
                                eltype = (x0___ = IMPORTS___.html4_v___?
                                  IMPORTS___.html4: ___.ri(IMPORTS___, 'html4')
                                  , x0___.ELEMENTS_v___? x0___.ELEMENTS:
                                  x0___.v___('ELEMENTS')).v___(tagName);
                                if (cdataContentType) {
                                  isScript = cdataContentType === (x2___ =
                                    (x1___ = IMPORTS___.html4_v___?
                                     IMPORTS___.html4: ___.ri(IMPORTS___,
                                       'html4'), x1___.eflags_v___?
                                     x1___.eflags: x1___.v___('eflags')),
                                    x2___.SCRIPT_v___? x2___.SCRIPT:
                                    x2___.v___('SCRIPT'));
                                  cdataContentType = 0;
                                  if (pendingExternal) {
                                    if (isScript) {
                                      res =
                                        resolveUriRelativeToDocument.i___(pendingExternal)
                                        ;
                                      evaluateUntrustedExternalScript.i___(res,
                                        marker, continuation);
                                    }
                                    pendingExternal =
                                      IMPORTS___.undefined_v___?
                                      IMPORTS___.undefined: ___.ri(IMPORTS___,
                                      'undefined');
                                  } else {
                                    content = cdataContent.join_m___?
                                      cdataContent.join(''):
                                    cdataContent.m___('join', [ '' ]);
                                    cdataContent.length_w___ === cdataContent?
                                      (cdataContent.length = 0):
                                    cdataContent.w___('length', 0);
                                    if (isScript) {
                                      evaluateUntrustedScript.i___((x3___ =
                                            domicile.pseudoLocation_v___?
                                            domicile.pseudoLocation:
                                          domicile.v___('pseudoLocation'),
                                          x3___.href_v___? x3___.href:
                                          x3___.v___('href')), content);
                                    } else {
                                      defineUntrustedStylesheet.i___((x4___ =
                                            domicile.pseudoLocation_v___?
                                            domicile.pseudoLocation:
                                          domicile.v___('pseudoLocation'),
                                          x4___.href_v___? x4___.href:
                                          x4___.v___('href')), content);
                                    }
                                  }
                                }
                                insertionMode.endTag_m___?
                                  insertionMode.endTag(tagName):
                                insertionMode.m___('endTag', [ tagName ]);
                              }
                              return ___.f(endTag$_lit, 'endTag$_lit');
                            })(), 'pcdata', (function () {
                              function pcdata$_lit(text) {
                                insertionMode.text_m___?
                                  insertionMode.text(text):
                                insertionMode.m___('text', [ text ]);
                              }
                              return ___.f(pcdata$_lit, 'pcdata$_lit');
                            })(), 'cdata', (function () {
                              function cdata$_lit(text) {
                                if (cdataContentType) {
                                  cdataContent.push_m___?
                                    cdataContent.push(text):
                                  cdataContent.m___('push', [ text ]);
                                } else {
                                  documentWriter.pcdata_m___?
                                    documentWriter.pcdata(text):
                                  documentWriter.m___('pcdata', [ text ]);
                                }
                              }
                              return ___.f(cdata$_lit, 'cdata$_lit');
                            })() ]);
                      x1___ = documentWriter, x0___ =
                        documentWriter.pcdata_v___? documentWriter.pcdata:
                      documentWriter.v___('pcdata'), x1___.rcdata_w___ ===
                        x1___? (x1___.rcdata = x0___): x1___.w___('rcdata',
                        x0___);
                      htmlParser = (x2___ = IMPORTS___.html_v___?
                          IMPORTS___.html: ___.ri(IMPORTS___, 'html'), x3___ =
                          documentWriter, x2___.makeSaxParser_m___?
                          x2___.makeSaxParser(x3___):
                        x2___.m___('makeSaxParser', [ x3___ ]));
                      tameDocWrite = (function () {
                          function write(html_varargs) {
                            {
                              ___.deodorize(arguments, -6);
                              var a___ = ___.args(arguments);
                            }
                            var htmlText, x0___, x1___;
                            htmlText = concat.i___(a___);
                            if (!insertionPoint) {
                              insertionPoint = base;
                            }
                            if (cdataContentType) {
                              htmlText = (cdataContentType === (x1___ = (x0___
                                      = IMPORTS___.html4_v___?
                                      IMPORTS___.html4: ___.ri(IMPORTS___,
                                      'html4'), x0___.eflags_v___?
                                      x0___.eflags: x0___.v___('eflags')),
                                  x1___.SCRIPT_v___? x1___.SCRIPT:
                                  x1___.v___('SCRIPT')) ? '\x3cscript\x3e':
                                '\x3cstyle\x3e') + htmlText;
                            }
                            htmlParser.i___(htmlText);
                            return documentLoaded.promise_v___?
                              documentLoaded.promise:
                            documentLoaded.v___('promise');
                          }
                          return ___.f(write, 'write');
                        })();
                      x7___ = domicile, x6___ = (x4___ =
                        IMPORTS___.cajaVM_v___? IMPORTS___.cajaVM:
                        ___.ri(IMPORTS___, 'cajaVM'), x5___ = tameDocWrite,
                        x4___.def_m___? x4___.def(x5___): x4___.m___('def', [
                            x5___ ])), x7___.writeHook_w___ === x7___?
                        (x7___.writeHook = x6___): x7___.w___('writeHook',
                        x6___);
                      x11___ = domicile, x10___ = (x8___ =
                        IMPORTS___.cajaVM_v___? IMPORTS___.cajaVM:
                        ___.ri(IMPORTS___, 'cajaVM'), x9___ =
                        evaluateUntrustedExternalScript, x8___.def_m___?
                        x8___.def(x9___): x8___.m___('def', [ x9___ ])),
                      x11___.evaluateUntrustedExternalScript_w___ === x11___?
                        (x11___.evaluateUntrustedExternalScript = x10___):
                      x11___.w___('evaluateUntrustedExternalScript', x10___);
                    }).i___(opt_domicile);
                }
                IMPORTS___.w___('HtmlEmitter', ___.f(HtmlEmitter,
                    'HtmlEmitter'));
              }
              moduleResult___ = ___.NO_RESULT;
              if (('I'.toLowerCase_m___? 'I'.toLowerCase():
                  'I'.m___('toLowerCase', [ ])) !== 'i') {
                throw 'I/i problem';
              }
              if (typeof IMPORTS___.v___('window') !== 'undefined') {
                moduleResult___ = (IMPORTS___.window_v___? IMPORTS___.window:
                  ___.ri(IMPORTS___, 'window')).w___('HtmlEmitter',
                  IMPORTS___.HtmlEmitter_v___? IMPORTS___.HtmlEmitter:
                  ___.ri(IMPORTS___, 'HtmlEmitter'));
              }
              return moduleResult___;
            },
            'cajolerName': 'com.google.caja',
            'cajolerVersion': '5054',
            'cajoledDate': 1347987483586
          }).instantiate___(___, IMPORTS___), ___.prepareModule({
            'instantiate': function (___, IMPORTS___) {
              var dis___ = IMPORTS___;
              var moduleResult___;
              moduleResult___ = ___.NO_RESULT;
              ___.di(IMPORTS___, 'lexCss');
              ___.di(IMPORTS___, 'decodeCss');
              ___.f(function () {
                  var decodeCssEscape, escapeCssString, escapeCssStrChar,
                  escapeCssUrlChar, cssStrChars, cssUrlChars, WC, W, NL,
                  SURROGATE_PAIR, NONASCII, UNICODE_TAIL, UNICODE, ESCAPE_TAIL,
                  ESCAPE, URLCHAR, STRINGCHAR, STRING, NUM, NMSTART, NMCHAR,
                  NAME, IDENT, ATKEYWORD, HASH, NUMBER, WORD_TERM, PERCENTAGE,
                  DIMENSION, NUMERIC_VALUE, URI, UNICODE_RANGE, CDO, CDC, S,
                  COMMENT, FUNCTION, INCLUDES, DASHMATCH, PREFIXMATCH,
                  SUFFIXMATCH, SUBSTRINGMATCH, CMP_OPS, CHAR, BOM, CSS_TOKEN,
                  x0___;
                  {
                    function decodeCssEscape(s) {
                      var i, x0___, x1___, x2___, x3___, x4___;
                      i = (IMPORTS___.parseInt_v___? IMPORTS___.parseInt:
                        ___.ri(IMPORTS___, 'parseInt')).i___(s.substring_m___?
                        s.substring(1): s.m___('substring', [ 1 ]), 16);
                      if (i > 65535) {
                        return i = i - 65536, (x0___ = IMPORTS___.String_v___?
                            IMPORTS___.String: ___.ri(IMPORTS___, 'String'),
                          x1___ = 55296 + (i >> 10), x2___ = 56320 + (i & 1023)
                            , x0___.fromCharCode_m___?
                            x0___.fromCharCode(x1___, x2___):
                          x0___.m___('fromCharCode', [ x1___, x2___ ]));
                      } else if (i == i) {
                        return x3___ = IMPORTS___.String_v___?
                          IMPORTS___.String: ___.ri(IMPORTS___, 'String'),
                        x4___ = i, x3___.fromCharCode_m___?
                          x3___.fromCharCode(x4___): x3___.m___('fromCharCode',
                          [ x4___ ]);
                      } else if (s[ +1 ] < ' ') { return ''; } else {
                        return s[ +1 ];
                      }
                    }
                    ___.f(decodeCssEscape, 'decodeCssEscape');
                  }
                  {
                    function escapeCssString(s, replacer) {
                      var x0___, x1___, x2___;
                      return '\"' + (x1___ = s, x0___ = new
                          RegExp.new___('[\\u0000-\\u001f\\\\\\\"\x3c\x3e]',
                          'g'), x2___ = replacer, x1___.replace_m___?
                          x1___.replace(x0___, x2___): x1___.m___('replace', [
                            x0___, x2___ ])) + '\"';
                    }
                    ___.f(escapeCssString, 'escapeCssString');
                  }
                  {
                    function escapeCssStrChar(ch) {
                      var x0___;
                      return cssStrChars.v___(ch) || cssStrChars.w___(ch, '\\'
                        + (x0___ = ch.charCodeAt_m___? ch.charCodeAt(0):
                          ch.m___('charCodeAt', [ 0 ]), x0___.toString_m___?
                          x0___.toString(16): x0___.m___('toString', [ 16 ])) +
                        ' ');
                    }
                    ___.f(escapeCssStrChar, 'escapeCssStrChar');
                  }
                  {
                    function escapeCssUrlChar(ch) {
                      var x0___;
                      return cssUrlChars.v___(ch) || cssUrlChars.w___(ch, (ch <
                            '\x10'? '%0': '%') + (x0___ = ch.charCodeAt_m___?
                            ch.charCodeAt(0): ch.m___('charCodeAt', [ 0 ]),
                          x0___.toString_m___? x0___.toString(16):
                          x0___.m___('toString', [ 16 ])));
                    }
                    ___.f(escapeCssUrlChar, 'escapeCssUrlChar');
                  }
                  cssStrChars = ___.iM([ '\\', '\\\\' ]);
                  cssUrlChars = ___.iM([ '\\', '%5c' ]);
                  WC = '[\\t\\n\\f ]';
                  W = WC + '*';
                  NL = '[\\n\\f]';
                  SURROGATE_PAIR = '[\\ud800-\\udbff][\\udc00-\\udfff]';
                  NONASCII = '[\\u0080-\\ud7ff\\ue000-\\ufffd]|' +
                    SURROGATE_PAIR;
                  UNICODE_TAIL = '[0-9a-fA-F]{1,6}' + WC + '?';
                  UNICODE = '\\\\' + UNICODE_TAIL;
                  ESCAPE_TAIL = '(?:' + UNICODE_TAIL +
                    '|[\\u0020-\\u007e\\u0080-\\ud7ff\\ue000\\ufffd]|' +
                    SURROGATE_PAIR + ')';
                  ESCAPE = '\\\\' + ESCAPE_TAIL;
                  URLCHAR = '(?:[\\t\\x21\\x23-\\x26\\x28-\\x5b\\x5d-\\x7e]|' +
                    NONASCII + '|' + ESCAPE + ')';
                  STRINGCHAR = '[^\'\"\\n\\f\\\\]|\\\\[\\s\\S]';
                  STRING = '\"(?:\'|' + STRINGCHAR + ')*\"' + '|\'(?:\"|' +
                    STRINGCHAR + ')*\'';
                  NUM = '[-+]?(?:[0-9]+(?:[.][0-9]+)?|[.][0-9]+)';
                  NMSTART = '(?:[a-zA-Z_]|' + NONASCII + '|' + ESCAPE + ')';
                  NMCHAR = '(?:[a-zA-Z0-9_-]|' + NONASCII + '|' + ESCAPE + ')';
                  NAME = NMCHAR + '+';
                  IDENT = '-?' + NMSTART + NMCHAR + '*';
                  ATKEYWORD = '@' + IDENT;
                  HASH = '#' + NAME;
                  NUMBER = NUM;
                  WORD_TERM = '(?:@?-?' + NMSTART + '|#)' + NMCHAR + '*';
                  PERCENTAGE = NUM + '%';
                  DIMENSION = NUM + IDENT;
                  NUMERIC_VALUE = NUM + '(?:%|' + IDENT + ')?';
                  URI = 'url[(]' + W + '(?:' + STRING + '|' + URLCHAR + '*)' +
                    W + '[)]';
                  UNICODE_RANGE = 'U[+][0-9A-F?]{1,6}(?:-[0-9A-F]{1,6})?';
                  CDO = '\x3c!--';
                  CDC = '--\x3e';
                  S = WC + '+';
                  COMMENT = '/(?:[*][^*]*[*]+(?:[^/][^*]*[*]+)*/|/[^\\n\\f]*)';
                  FUNCTION = '(?!url[(])' + IDENT + '[(]';
                  INCLUDES = '~=';
                  DASHMATCH = '[|]=';
                  PREFIXMATCH = '[^]=';
                  SUFFIXMATCH = '[$]=';
                  SUBSTRINGMATCH = '[*]=';
                  CMP_OPS = '[~|^$*]=';
                  CHAR = '[^\"\'\\\\/]|/(?![/*])';
                  BOM = '\\uFEFF';
                  CSS_TOKEN = new (IMPORTS___.RegExp_v___? IMPORTS___.RegExp:
                    ___.ri(IMPORTS___, 'RegExp')).new___((x0___ = [ BOM,
                        UNICODE_RANGE, URI, FUNCTION, WORD_TERM, STRING,
                        NUMERIC_VALUE, CDO, CDC, S, COMMENT, CMP_OPS, CHAR ],
                      x0___.join_m___? x0___.join('|'): x0___.m___('join', [
                          '|' ])), 'gi');
                  IMPORTS___.decodeCss_w___? (IMPORTS___.decodeCss = (function
                      () {
                        function decodeCss$_var(css) {
                          var x0___, x1___, x2___;
                          return x1___ = css, x0___ = new
                            (IMPORTS___.RegExp_v___? IMPORTS___.RegExp:
                            ___.ri(IMPORTS___, 'RegExp')).new___(('\\\\(?:' +
                                ESCAPE_TAIL + '|' + NL + ')'), 'g'), x2___ =
                            decodeCssEscape, x1___.replace_m___?
                            x1___.replace(x0___, x2___): x1___.m___('replace',
                            [ x0___, x2___ ]);
                        }
                        return ___.f(decodeCss$_var, 'decodeCss$_var');
                      })()): ___.wi(IMPORTS___, 'decodeCss', (function () {
                        function decodeCss$_var(css) {
                          var x0___, x1___, x2___;
                          return x1___ = css, x0___ = new
                            (IMPORTS___.RegExp_v___? IMPORTS___.RegExp:
                            ___.ri(IMPORTS___, 'RegExp')).new___(('\\\\(?:' +
                              ESCAPE_TAIL + '|' + NL + ')'), 'g'), x2___ =
                            decodeCssEscape, x1___.replace_m___?
                            x1___.replace(x0___, x2___): x1___.m___('replace',
                            [ x0___, x2___ ]);
                        }
                        return ___.f(decodeCss$_var, 'decodeCss$_var');
                      })());
                  IMPORTS___.lexCss_w___? (IMPORTS___.lexCss = (function () {
                        function lexCss$_var(cssText) {
                          var tokens, x0___, x1___, x2___, x3___, j, last, i,
                          n, tok, len, cc, x4___, x5___, x6___, x7___, x8___,
                          x9___;
                          cssText = '' + cssText;
                          tokens = (x2___ = (x1___ = cssText, x0___ = new
                                RegExp.new___('\\r\\n?', 'g'),
                              x1___.replace_m___? x1___.replace(x0___, '\n'):
                              x1___.m___('replace', [ x0___, '\n' ])), x3___ =
                              CSS_TOKEN, x2___.match_m___? x2___.match(x3___):
                            x2___.m___('match', [ x3___ ])) || [ ];
                          j = 0;
                          last = ' ';
                          for (i = 0, n = tokens.length; i < n; ++i) {
                            tok = (IMPORTS___.decodeCss_v___?
                                IMPORTS___.decodeCss: ___.ri(IMPORTS___,
                                'decodeCss')).i___(tokens[ +i ]);
                            len = tok.length;
                            cc = tok.charCodeAt_m___? tok.charCodeAt(0):
                            tok.m___('charCodeAt', [ 0 ]);
                            tok = cc == ('\"'.charCodeAt_m___?
                              '\"'.charCodeAt(0): '\"'.m___('charCodeAt', [ 0 ]
                              )) || cc == ('\''.charCodeAt_m___?
                              '\''.charCodeAt(0): '\''.m___('charCodeAt', [ 0 ]
                              )) ? escapeCssString.i___((x5___ = tok, x4___ =
                                  len - 1, x5___.substring_m___?
                                  x5___.substring(1, x4___):
                                x5___.m___('substring', [ 1, x4___ ])),
                              escapeCssStrChar): cc == ('/'.charCodeAt_m___?
                              '/'.charCodeAt(0): '/'.m___('charCodeAt', [ 0 ]))
                              && len > 1 || tok == '\\' || tok == CDC || tok ==
                              CDO || tok == '\ufeff' || cc <=
                              (' '.charCodeAt_m___? ' '.charCodeAt(0):
                              ' '.m___('charCodeAt', [ 0 ])) ? ' ': (x6___ =
                              new RegExp.new___('url\\(', 'i'), x7___ = tok,
                              x6___.test_m___? x6___.test(x7___):
                              x6___.m___('test', [ x7___ ])) ? 'url(' +
                              escapeCssString.i___((x9___ = tok, x8___ = new
                                (IMPORTS___.RegExp_v___? IMPORTS___.RegExp:
                                 ___.ri(IMPORTS___, 'RegExp'))
                                .new___(('^url\\(' + W + '[\"\']?|[\"\']?' + W
                                      + '\\)$'), 'gi'), x9___.replace_m___?
                                x9___.replace(x8___, ''): x9___.m___('replace',
                                  [ x8___, '' ])), escapeCssUrlChar) + ')':
                            tok;
                            if (last != tok || tok != ' ') {
                              tokens.w___(j++, last = tok);
                            }
                          }
                          tokens.length_w___ === tokens? (tokens.length = j):
                          tokens.w___('length', j);
                          return tokens;
                        }
                        return ___.f(lexCss$_var, 'lexCss$_var');
                      })()): ___.wi(IMPORTS___, 'lexCss', (function () {
                        function lexCss$_var(cssText) {
                          var tokens, x0___, x1___, x2___, x3___, j, last, i,
                          n, tok, len, cc, x4___, x5___, x6___, x7___, x8___,
                          x9___;
                          cssText = '' + cssText;
                          tokens = (x2___ = (x1___ = cssText, x0___ = new
                                RegExp.new___('\\r\\n?', 'g'),
                              x1___.replace_m___? x1___.replace(x0___, '\n'):
                              x1___.m___('replace', [ x0___, '\n' ])), x3___ =
                              CSS_TOKEN, x2___.match_m___? x2___.match(x3___):
                            x2___.m___('match', [ x3___ ])) || [ ];
                          j = 0;
                          last = ' ';
                          for (i = 0, n = tokens.length; i < n; ++i) {
                            tok = (IMPORTS___.decodeCss_v___?
                                IMPORTS___.decodeCss: ___.ri(IMPORTS___,
                                'decodeCss')).i___(tokens[ +i ]);
                            len = tok.length;
                            cc = tok.charCodeAt_m___? tok.charCodeAt(0):
                            tok.m___('charCodeAt', [ 0 ]);
                            tok = cc == ('\"'.charCodeAt_m___?
                                '\"'.charCodeAt(0): '\"'.m___('charCodeAt', [ 0
                                ])) || cc == ('\''.charCodeAt_m___?
                              '\''.charCodeAt(0): '\''.m___('charCodeAt', [ 0 ]
                              )) ? escapeCssString.i___((x5___ = tok, x4___ =
                                  len - 1, x5___.substring_m___?
                                  x5___.substring(1, x4___):
                                x5___.m___('substring', [ 1, x4___ ])),
                              escapeCssStrChar): cc == ('/'.charCodeAt_m___?
                              '/'.charCodeAt(0): '/'.m___('charCodeAt', [ 0 ]))
                              && len > 1 || tok == '\\' || tok == CDC || tok ==
                              CDO || tok == '\ufeff' || cc <=
                              (' '.charCodeAt_m___? ' '.charCodeAt(0):
                              ' '.m___('charCodeAt', [ 0 ])) ? ' ': (x6___ =
                              new RegExp.new___('url\\(', 'i'), x7___ = tok,
                              x6___.test_m___? x6___.test(x7___):
                              x6___.m___('test', [ x7___ ])) ? 'url(' +
                              escapeCssString.i___((x9___ = tok, x8___ = new
                                (IMPORTS___.RegExp_v___? IMPORTS___.RegExp:
                                 ___.ri(IMPORTS___, 'RegExp'))
                                .new___(('^url\\(' + W + '[\"\']?|[\"\']?' + W
                                      + '\\)$'), 'gi'), x9___.replace_m___?
                                x9___.replace(x8___, ''): x9___.m___('replace',
                                  [ x8___, '' ])), escapeCssUrlChar) + ')':
                            tok;
                            if (last != tok || tok != ' ') {
                              tokens.w___(j++, last = tok);
                            }
                          }
                          tokens.length_w___ === tokens? (tokens.length = j):
                          tokens.w___('length', j);
                          return tokens;
                        }
                        return ___.f(lexCss$_var, 'lexCss$_var');
                      })());
                }).i___();
              if (typeof IMPORTS___.v___('window') !== 'undefined') {
                (IMPORTS___.window_v___? IMPORTS___.window: ___.ri(IMPORTS___,
                   'window')).w___('lexCss', IMPORTS___.lexCss_v___?
                    IMPORTS___.lexCss: ___.ri(IMPORTS___, 'lexCss'));
                moduleResult___ = (IMPORTS___.window_v___? IMPORTS___.window:
                  ___.ri(IMPORTS___, 'window')).w___('decodeCss',
                  IMPORTS___.decodeCss_v___? IMPORTS___.decodeCss:
                  ___.ri(IMPORTS___, 'decodeCss'));
              }
              return moduleResult___;
            },
            'cajolerName': 'com.google.caja',
            'cajolerVersion': '5054',
            'cajoledDate': 1347987483653
          }).instantiate___(___, IMPORTS___), ___.prepareModule({
            'instantiate': function (___, IMPORTS___) {
              var dis___ = IMPORTS___;
              var moduleResult___;
              {
                function sanitizeCssSelectors(selectors, suffix) {
                  var historySensitiveSelectors, historyInsensitiveSelectors,
                  k, i, n, start, processSelector;
                  {
                    function processSelector(start, end) {
                      var historySensitive, out, lastOperator, elSelector, i,
                      tok, isChild, x0___, x1___, x2___, x3___, x4___,
                      processElementSelector, safeSelector, x5___, x6___;
                      {
                        function processElementSelector(start, end, last) {
                          var debugStart, debugEnd, element, classId,
                          pseudoSelector, tok, x0___, x1___, x2___, elType,
                          x3___, x4___, x5___, x6___, x7___, x8___, x9___,
                          x10___, x11___, x12___;
                          debugStart = start, debugEnd = end;
                          element = '';
                          if (start < end) {
                            tok = (x0___ = IMPORTS___.html_v___?
                              IMPORTS___.html: ___.ri(IMPORTS___, 'html'),
                              x2___ = (x1___ = selectors.v___(start),
                                x1___.toLowerCase_m___? x1___.toLowerCase():
                                x1___.m___('toLowerCase', [ ])),
                              x0___.virtualToRealElementName_m___?
                              x0___.virtualToRealElementName(x2___):
                              x0___.m___('virtualToRealElementName', [ x2___ ])
                            );
                            elType = (x3___ = IMPORTS___.html4_v___?
                              IMPORTS___.html4: ___.ri(IMPORTS___, 'html4'),
                              x3___.ELEMENTS_v___? x3___.ELEMENTS:
                              x3___.v___('ELEMENTS')).v___(tok);
                            if (tok === '*' || 'number' === typeof elType && !
                                (elType & (x4___ = IMPORTS___.html4_v___?
                                  IMPORTS___.html4: ___.ri(IMPORTS___, 'html4')
                                  , x4___.eflags_v___? x4___.eflags:
                                  x4___.v___('eflags')).v___('UNSAFE'))) { ++start;
                              element = tok;
                            }
                          }
                          classId = '';
                          while (start < end) {
                            tok = selectors.v___(start);
                            if ((tok.charAt_m___? tok.charAt(0):
                                tok.m___('charAt', [ 0 ])) === '#') {
                              if (x5___ = new
                                RegExp.new___('^#_|__$|[^#0-9A-Za-z:_\\-]'),
                                x6___ = tok, x5___.test_m___? x5___.test(x6___)
                                : x5___.m___('test', [ x6___ ])) { return null;
                              }
                              classId = classId + (tok + '-' + suffix);
                            } else if (tok === '.') {
                              if (++start < end && (x7___ = new
                                    RegExp.new___('^[0-9A-Za-z:_\\-]+$'), x8___
                                    = tok = selectors.v___(start),
                                  x7___.test_m___? x7___.test(x8___):
                                  x7___.m___('test', [ x8___ ])) && ! (x9___ =
                                  new RegExp.new___('^_|__$'), x10___ = tok,
                                  x9___.test_m___? x9___.test(x10___):
                                  x9___.m___('test', [ x10___ ]))) {
                                classId = classId + ('.' + tok);
                              } else { return null; }
                            } else { break; }
                            ++start;
                          }
                          pseudoSelector = '';
                          if (start < end && selectors.v___(start) === ':') {
                            tok = selectors[ + ++start ];
                            if (tok === 'visited' || tok === 'link') {
                              if (! (x11___ = new RegExp.new___('^[a*]?$'),
                                  x12___ = element, x11___.test_m___?
                                    x11___.test(x12___): x11___.m___('test', [
                                      x12___ ]))) { return null; }
                              historySensitive = true;
                              pseudoSelector = ':' + tok;
                              element = 'a';
                              ++start;
                            }
                          }
                          if (start === end) {
                            return element + classId + pseudoSelector;
                          }
                          return null;
                        }
                        ___.f(processElementSelector, 'processElementSelector')
                          ;
                      }
                      historySensitive = false;
                      if (selectors.v___(start) === ' ') { ++start; }
                      if (end - 1 !== start && selectors.v___(end) === ' ') { --end; }
                      out = [ ];
                      lastOperator = start;
                      elSelector = '';
                      for (i = start; i < end; ++i) {
                        tok = selectors.v___(i);
                        isChild = tok === '\x3e';
                        if (isChild || tok === ' ') {
                          elSelector =
                            processElementSelector.i___(lastOperator, i, false)
                            ;
                          if (!elSelector || isChild && (x0___ = new
                              RegExp.new___('^html', 'i'), x1___ = elSelector,
                              x0___.test_m___? x0___.test(x1___):
                              x0___.m___('test', [ x1___ ]))) { return; }
                          lastOperator = i + 1;
                          x3___ = out, x4___ = elSelector, x2___ = isChild?
                            ' \x3e ': ' ', x3___.push_m___? x3___.push(x4___,
                            x2___): x3___.m___('push', [ x4___, x2___ ]);
                        }
                      }
                      elSelector = processElementSelector.i___(lastOperator,
                        end, true);
                      if (!elSelector) { return; }
                      out.push_m___? out.push(elSelector): out.m___('push', [
                          elSelector ]);
                      safeSelector = out.join_m___? out.join(''):
                      out.m___('join', [ '' ]);
                      safeSelector = '.' + suffix + ' ' + safeSelector;
                      x5___ = historySensitive? historySensitiveSelectors:
                      historyInsensitiveSelectors, x6___ = safeSelector,
                      x5___.push_m___? x5___.push(x6___): x5___.m___('push', [
                          x6___ ]);
                    }
                    ___.f(processSelector, 'processSelector');
                  }
                  historySensitiveSelectors = [ ];
                  historyInsensitiveSelectors = [ ];
                  k = 0;
                  for (i = 0; i < selectors.length; ++i) {
                    if (! (selectors[ +i ] == ' ' && (selectors[ + (i - 1) ] ==
                          '\x3e' || selectors[ + (i + 1) ] == '\x3e'))) {
                      selectors.w___(k++, selectors[ +i ]);
                    }
                  }
                  selectors.length_w___ === selectors? (selectors.length = k):
                  selectors.w___('length', k);
                  n = selectors.length, start = 0;
                  for (i = 0; i < n; ++i) {
                    if (selectors[ +i ] == ',') {
                      processSelector.i___(start, i);
                      start = i + 1;
                    }
                  }
                  processSelector.i___(start, n);
                  return [ historyInsensitiveSelectors,
                    historySensitiveSelectors ];
                }
                IMPORTS___.w___('sanitizeCssSelectors',
                  ___.f(sanitizeCssSelectors, 'sanitizeCssSelectors'));
              }
              moduleResult___ = ___.NO_RESULT;
              ___.di(IMPORTS___, 'sanitizeCssProperty'),
              IMPORTS___.w___('sanitizeCssProperty', ___.f(function () {
                    var NOEFFECT_URL, NORM_URL_REGEXP, NORM_URL_REPLACEMENTS,
                    normalizeUrl, normalizeUrlChar, URI_SCHEME_RE,
                    ALLOWED_URI_SCHEMES, resolveUri, safeUri, unionArrays,
                    normalizeFunctionCall, ALLOWED_LITERAL;
                    {
                      function normalizeUrl(s) {
                        if ('string' === typeof s) {
                          return 'url(\"' + (s.replace_m___?
                              s.replace(NORM_URL_REGEXP, normalizeUrlChar):
                            s.m___('replace', [ NORM_URL_REGEXP,
                                normalizeUrlChar ])) + '\")';
                        } else {
                          return NOEFFECT_URL;
                        }
                      }
                      ___.f(normalizeUrl, 'normalizeUrl');
                    }
                    {
                      function normalizeUrlChar(ch) {
                        return NORM_URL_REPLACEMENTS.v___(ch);
                      }
                      ___.f(normalizeUrlChar, 'normalizeUrlChar');
                    }
                    {
                      function resolveUri(baseUri, uri) {
                        var x0___, x1___, x2___, x3___;
                        if (baseUri) {
                          return x1___ = (x0___ = IMPORTS___.URI_v___?
                            IMPORTS___.URI: ___.ri(IMPORTS___, 'URI'),
                            x0___.utils_v___? x0___.utils: x0___.v___('utils'))
                            , x2___ = baseUri, x3___ = uri, x1___.resolve_m___?
                            x1___.resolve(x2___, x3___): x1___.m___('resolve',
                            [ x2___, x3___ ]);
                        }
                        return uri;
                      }
                      ___.f(resolveUri, 'resolveUri');
                    }
                    {
                      function safeUri(uri, prop, naiveUriRewriter) {
                        var parsed, x0___, x1___, x2___, x3___;
                        if (!naiveUriRewriter) { return null; }
                        parsed = (x0___ = '' + uri, x1___ = URI_SCHEME_RE,
                          x0___.match_m___? x0___.match(x1___):
                          x0___.m___('match', [ x1___ ]));
                        if (parsed && (!parsed[ +1 ] || (x3___ =
                                ALLOWED_URI_SCHEMES, x2___ = parsed[ +1 ],
                              x3___.test_m___? x3___.test(x2___):
                              x3___.m___('test', [ x2___ ])))) {
                          return naiveUriRewriter.i___(uri, prop);
                        } else { return null; }
                      }
                      ___.f(safeUri, 'safeUri');
                    }
                    {
                      function unionArrays(arrs) {
                        var map, i, arr, j;
                        map = ___.iM([ ]);
                        for (i = arrs.length; --i >= 0;) {
                          arr = arrs.v___(i);
                          for (j = arr.length; --j >= 0;) {
                            map.w___(arr.v___(j), ALLOWED_LITERAL);
                          }
                        }
                        return map;
                      }
                      ___.f(unionArrays, 'unionArrays');
                    }
                    {
                      function normalizeFunctionCall(tokens, start) {
                        var parenDepth, end, n, token;
                        parenDepth = 1, end = start + 1, n = tokens.length;
                        while (end < n && parenDepth) {
                          token = tokens.v___(end++);
                          parenDepth = parenDepth + (token === '('? 1: token
                              === ')'? -1: 0);
                        }
                        return end;
                      }
                      ___.f(normalizeFunctionCall, 'normalizeFunctionCall');
                    }
                    NOEFFECT_URL = 'url(\"about:blank\")';
                    NORM_URL_REGEXP = new
                      RegExp.new___('[\\n\\f\\r\\\"\\\'()*\x3c\x3e]', 'g');
                    NORM_URL_REPLACEMENTS = ___.iM([ '\n', '%0a', '\f', '%0c',
                        '\r', '%0d', '\"', '%22', '\'', '%27', '(', '%28', ')',
                        '%29', '*', '%2a', '\x3c', '%3c', '\x3e', '%3e' ]);
                    URI_SCHEME_RE = new (IMPORTS___.RegExp_v___?
                        IMPORTS___.RegExp: ___.ri(IMPORTS___, 'RegExp'))
                      .new___(('^' + '(?:' + '([^:/?# ]+)' + ':)?'));
                    ALLOWED_URI_SCHEMES = new
                      RegExp.new___('^(?:https?|mailto)$', 'i');
                    ALLOWED_LITERAL = ___.iM([ ]);
                    return ___.f(function (property, propertySchema, tokens,
                        opt_naiveUriRewriter, opt_baseUri) {
                        var propBits, qstringBits, lastQuoted, i, k, token,
                        x0___, cc, cc1, cc2, isnum1, isnum2, end, litGroup,
                        litMap, x1___, x2___, x3___, x4___, x5___, x6___,
                        x7___, x8___, x9___, x10___, x11___, x12___, x13___,
                        x14___, x15___, x16___, x17___, x18___, x19___, x20___,
                        x21___, x22___, x23___, x24___, x25___, x26___, x27___;
                        propBits = propertySchema.cssPropBits_v___?
                          propertySchema.cssPropBits:
                        propertySchema.v___('cssPropBits');
                        qstringBits = propBits &
                          ((IMPORTS___.CSS_PROP_BIT_QSTRING_CONTENT_v___?
                            IMPORTS___.CSS_PROP_BIT_QSTRING_CONTENT:
                            ___.ri(IMPORTS___, 'CSS_PROP_BIT_QSTRING_CONTENT'))
                          | (IMPORTS___.CSS_PROP_BIT_QSTRING_URL_v___?
                            IMPORTS___.CSS_PROP_BIT_QSTRING_URL:
                            ___.ri(IMPORTS___, 'CSS_PROP_BIT_QSTRING_URL')));
                        lastQuoted = IMPORTS___.NaN_v___? IMPORTS___.NaN:
                        ___.ri(IMPORTS___, 'NaN');
                        i = 0, k = 0;
                        for (; i < tokens.length; ++i) {
                          token = (x0___ = tokens[ +i ],
                            x0___.toLowerCase_m___? x0___.toLowerCase():
                            x0___.m___('toLowerCase', [ ]));
                          cc = token.charCodeAt_m___? token.charCodeAt(0):
                          token.m___('charCodeAt', [ 0 ]);
                          token = cc === (' '.charCodeAt_m___?
                            ' '.charCodeAt(0): ' '.m___('charCodeAt', [ 0 ])) ?
                            '': cc === ('\"'.charCodeAt_m___?
                            '\"'.charCodeAt(0): '\"'.m___('charCodeAt', [ 0 ]))
                            ? qstringBits ===
                            (IMPORTS___.CSS_PROP_BIT_QSTRING_URL_v___?
                            IMPORTS___.CSS_PROP_BIT_QSTRING_URL:
                            ___.ri(IMPORTS___, 'CSS_PROP_BIT_QSTRING_URL')) &&
                            opt_naiveUriRewriter?
                            normalizeUrl.i___(safeUri.i___(resolveUri.i___(opt_baseUri,
                                (IMPORTS___.decodeCss_v___?
                                   IMPORTS___.decodeCss: ___.ri(IMPORTS___,
                                   'decodeCss')).i___((x1___ = tokens[ +i ],
                                    x2___ = token.length - 1,
                                    x1___.substring_m___? x1___.substring(1,
                                      x2___): x1___.m___('substring', [ 1,
                                        x2___ ])))), property,
                              opt_naiveUriRewriter)): qstringBits ===
                            (IMPORTS___.CSS_PROP_BIT_QSTRING_CONTENT_v___?
                            IMPORTS___.CSS_PROP_BIT_QSTRING_CONTENT:
                            ___.ri(IMPORTS___, 'CSS_PROP_BIT_QSTRING_CONTENT'))
                            ? token: '': cc === ('#'.charCodeAt_m___?
                            '#'.charCodeAt(0): '#'.m___('charCodeAt', [ 0 ]))
                            && (x3___ = new
                              RegExp.new___('^#(?:[0-9a-f]{3}){1,2}$'), x4___ =
                              token, x3___.test_m___? x3___.test(x4___):
                            x3___.m___('test', [ x4___ ])) ? propBits &
                            (IMPORTS___.CSS_PROP_BIT_HASH_VALUE_v___?
                            IMPORTS___.CSS_PROP_BIT_HASH_VALUE:
                            ___.ri(IMPORTS___, 'CSS_PROP_BIT_HASH_VALUE')) ?
                            token: '': ('0'.charCodeAt_m___? '0'.charCodeAt(0):
                            '0'.m___('charCodeAt', [ 0 ])) <= cc && cc <=
                            ('9'.charCodeAt_m___? '9'.charCodeAt(0):
                            '9'.m___('charCodeAt', [ 0 ])) ? propBits &
                            (IMPORTS___.CSS_PROP_BIT_QUANTITY_v___?
                            IMPORTS___.CSS_PROP_BIT_QUANTITY:
                            ___.ri(IMPORTS___, 'CSS_PROP_BIT_QUANTITY')) ?
                            propBits & (IMPORTS___.CSS_PROP_BIT_Z_INDEX_v___?
                            IMPORTS___.CSS_PROP_BIT_Z_INDEX: ___.ri(IMPORTS___,
                              'CSS_PROP_BIT_Z_INDEX')) ? (x6___ = token, x5___
                              = new RegExp.new___('^\\d{1,7}$'),
                            x6___.match_m___? x6___.match(x5___):
                            x6___.m___('match', [ x5___ ])) ? token: '': token:
                          '': (cc1 = token.charCodeAt_m___? token.charCodeAt(1)
                            : token.m___('charCodeAt', [ 1 ]), cc2 =
                            token.charCodeAt_m___? token.charCodeAt(2):
                            token.m___('charCodeAt', [ 2 ]), isnum1 =
                            ('0'.charCodeAt_m___? '0'.charCodeAt(0):
                             '0'.m___('charCodeAt', [ 0 ])) <= cc1 && cc1 <=
                            ('9'.charCodeAt_m___? '9'.charCodeAt(0):
                             '9'.m___('charCodeAt', [ 0 ])), isnum2 =
                            ('0'.charCodeAt_m___? '0'.charCodeAt(0):
                             '0'.m___('charCodeAt', [ 0 ])) <= cc2 && cc2 <=
                            ('9'.charCodeAt_m___? '9'.charCodeAt(0):
                             '9'.m___('charCodeAt', [ 0 ])), cc ===
                            ('+'.charCodeAt_m___? '+'.charCodeAt(0):
                             '+'.m___('charCodeAt', [ 0 ])) && (isnum1 || cc1
                              === ('.'.charCodeAt_m___? '.'.charCodeAt(0):
                                '.'.m___('charCodeAt', [ 0 ])) && isnum2)) ?
                            propBits & (IMPORTS___.CSS_PROP_BIT_QUANTITY_v___?
                            IMPORTS___.CSS_PROP_BIT_QUANTITY:
                            ___.ri(IMPORTS___, 'CSS_PROP_BIT_QUANTITY')) ?
                            propBits & (IMPORTS___.CSS_PROP_BIT_Z_INDEX_v___?
                              IMPORTS___.CSS_PROP_BIT_Z_INDEX:
                            ___.ri(IMPORTS___, 'CSS_PROP_BIT_Z_INDEX')) ?
                            (x8___ = token, x7___ = new
                            RegExp.new___('^\\+\\d{1,7}$'), x8___.match_m___?
                            x8___.match(x7___): x8___.m___('match', [ x7___ ]))
                            ? token: '': (isnum1? '': '0') +
                            (token.substring_m___? token.substring(1):
                            token.m___('substring', [ 1 ])): '': cc ===
                            ('-'.charCodeAt_m___? '-'.charCodeAt(0):
                            '-'.m___('charCodeAt', [ 0 ])) && (isnum1 || cc1
                            === ('.'.charCodeAt_m___? '.'.charCodeAt(0):
                              '.'.m___('charCodeAt', [ 0 ])) && isnum2) ?
                            propBits &
                            (IMPORTS___.CSS_PROP_BIT_NEGATIVE_QUANTITY_v___?
                            IMPORTS___.CSS_PROP_BIT_NEGATIVE_QUANTITY:
                            ___.ri(IMPORTS___,
                              'CSS_PROP_BIT_NEGATIVE_QUANTITY')) ? propBits &
                            (IMPORTS___.CSS_PROP_BIT_Z_INDEX_v___?
                              IMPORTS___.CSS_PROP_BIT_Z_INDEX:
                            ___.ri(IMPORTS___, 'CSS_PROP_BIT_Z_INDEX')) ?
                            (x10___ = token, x9___ = new
                              RegExp.new___('^\\-\\d{1,7}$'),
                            x10___.match_m___? x10___.match(x9___):
                            x10___.m___('match', [ x9___ ])) ? token: '':
                          (isnum1? '-': '-0') + (token.substring_m___?
                            token.substring(1): token.m___('substring', [ 1 ]))
                            : propBits &
                            (IMPORTS___.CSS_PROP_BIT_QUANTITY_v___?
                            IMPORTS___.CSS_PROP_BIT_QUANTITY:
                            ___.ri(IMPORTS___, 'CSS_PROP_BIT_QUANTITY')) ? '0':
                          '': cc === ('.'.charCodeAt_m___? '.'.charCodeAt(0):
                            '.'.m___('charCodeAt', [ 0 ])) && isnum1? propBits
                            & (IMPORTS___.CSS_PROP_BIT_QUANTITY_v___?
                            IMPORTS___.CSS_PROP_BIT_QUANTITY:
                            ___.ri(IMPORTS___, 'CSS_PROP_BIT_QUANTITY')) ? '0'
                            + token: '': 'url(' === (token.substring_m___?
                            token.substring(0, 4): token.m___('substring', [ 0,
                                4 ])) ? opt_naiveUriRewriter && qstringBits &
                            (IMPORTS___.CSS_PROP_BIT_QSTRING_URL_v___?
                            IMPORTS___.CSS_PROP_BIT_QSTRING_URL:
                            ___.ri(IMPORTS___, 'CSS_PROP_BIT_QSTRING_URL')) ?
                            normalizeUrl.i___(safeUri.i___(resolveUri.i___(opt_baseUri,
                                (x11___ = tokens[ +i ], x12___ = token.length -
                                 2, x11___.substring_m___? x11___.substring(5,
                                   x12___): x11___.m___('substring', [ 5,
                                     x12___ ]))), property,
                              opt_naiveUriRewriter)): '': ((x14___ = token,
                              x13___ = token.length - 1, x14___.charAt_m___?
                              x14___.charAt(x13___): x14___.m___('charAt', [
                                  x13___ ])) === '(' && (end =
                              normalizeFunctionCall.i___(tokens, i), (x18___ =
                                tokens, x19___ = i, x15___ = end - i, x17___ =
                                token = (x16___ = tokens.slice_m___?
                                  tokens.slice(i, end): tokens.m___('slice', [
                                      i, end ]), x16___.join_m___?
                                  x16___.join(' '): x16___.m___('join', [ ' ' ]
                                  )), x18___.splice_m___? x18___.splice(x19___,
                                  x15___, x17___): x18___.m___('splice', [
                                    x19___, x15___, x17___ ]))), litGroup =
                            propertySchema.cssLitGroup_v___?
                            propertySchema.cssLitGroup:
                            propertySchema.v___('cssLitGroup'), litMap =
                            litGroup? (propertySchema.cssLitMap_v___?
                              propertySchema.cssLitMap:
                              propertySchema.v___('cssLitMap')) || (x21___ =
                              propertySchema, x20___ =
                              unionArrays.i___(litGroup), x21___.cssLitMap_w___
                              === x21___? (x21___.cssLitMap = x20___):
                              x21___.w___('cssLitMap', x20___)):
                            ALLOWED_LITERAL, litMap.v___(token) ===
                            ALLOWED_LITERAL || (propertySchema.cssExtra_v___?
                              propertySchema.cssExtra:
                              propertySchema.v___('cssExtra')) && (x22___ =
                              propertySchema.cssExtra_v___?
                              propertySchema.cssExtra:
                              propertySchema.v___('cssExtra'), x23___ = token,
                              x22___.test_m___? x22___.test(x23___):
                              x22___.m___('test', [ x23___ ]))) ? token:
                          (x24___ = new RegExp.new___('^\\w+$'), x25___ =
                           token, x24___.test_m___? x24___.test(x25___):
                           x24___.m___('test', [ x25___ ])) && qstringBits ===
                            (IMPORTS___.CSS_PROP_BIT_QSTRING_CONTENT_v___?
                            IMPORTS___.CSS_PROP_BIT_QSTRING_CONTENT:
                            ___.ri(IMPORTS___, 'CSS_PROP_BIT_QSTRING_CONTENT'))
                            ? lastQuoted + 1 === k? (tokens.w___(lastQuoted,
                              (x26___ = tokens.v___(lastQuoted), x27___ =
                               tokens.v___(lastQuoted).length - 1,
                               x26___.substring_m___? x26___.substring(0,
                                 x27___): x26___.m___('substring', [ 0, x27___
                                 ])) + ' ' + token + '\"'), token = ''):
                          (lastQuoted = k, '\"' + token + '\"'): '';
                          if (token) {
                            tokens.w___(k++, token);
                          }
                        }
                        if (k === 1 && tokens[ +0 ] === NOEFFECT_URL) { k = 0;
                        }
                        tokens.length_w___ === tokens? (tokens.length = k):
                        tokens.w___('length', k);
                      });
                  }).i___());
              ___.di(IMPORTS___, 'sanitizeStylesheet'),
              IMPORTS___.w___('sanitizeStylesheet', ___.f(function () {
                    var allowed, cssMediaTypeWhitelist,
                    sanitizeHistorySensitive;
                    {
                      function sanitizeHistorySensitive(blockOfProperties) {
                        var elide, i, n, token, x0___;
                        elide = false;
                        for (i = 0, n = blockOfProperties.length; i < n - 1;
                          ++i) {
                          token = blockOfProperties[ +i ];
                          if (':' === blockOfProperties[ + (i + 1) ]) {
                            elide = ! ((x0___ = (IMPORTS___.cssSchema_v___?
                                  IMPORTS___.cssSchema: ___.ri(IMPORTS___,
                                    'cssSchema')).v___(token),
                                x0___.cssPropBits_v___? x0___.cssPropBits:
                                x0___.v___('cssPropBits')) &
                                (IMPORTS___.CSS_PROP_BIT_ALLOWED_IN_LINK_v___?
                                  IMPORTS___.CSS_PROP_BIT_ALLOWED_IN_LINK:
                                ___.ri(IMPORTS___,
                                  'CSS_PROP_BIT_ALLOWED_IN_LINK')));
                          }
                          if (elide) {
                            blockOfProperties.NUM____w___ ===
                              blockOfProperties? (blockOfProperties[ +i ] = '')
                              : blockOfProperties.w___(+i, '');
                          }
                          if (';' === token) { elide = false; }
                        }
                        return blockOfProperties.join_m___?
                          blockOfProperties.join(''):
                        blockOfProperties.m___('join', [ '' ]);
                      }
                      ___.f(sanitizeHistorySensitive,
                        'sanitizeHistorySensitive');
                    }
                    allowed = ___.iM([ ]);
                    cssMediaTypeWhitelist = ___.iM([ 'braille', allowed,
                        'embossed', allowed, 'handheld', allowed, 'print',
                        allowed, 'projection', allowed, 'screen', allowed,
                        'speech', allowed, 'tty', allowed, 'tv', allowed ]);
                    return ___.f(function (baseUri, cssText, suffix,
                        opt_naiveUriRewriter) {
                        var safeCss, blockStack, elide, checkElide;
                        {
                          function checkElide() {
                            elide = blockStack.length !== 0 && blockStack[ +
                                (blockStack.length - 1) ] !== null;
                          }
                          ___.f(checkElide, 'checkElide');
                        }
                        safeCss = void 0;
                        blockStack = [ ];
                        elide = false;
                        (IMPORTS___.parseCssStylesheet_v___?
                         IMPORTS___.parseCssStylesheet: ___.ri(IMPORTS___,
                           'parseCssStylesheet')).i___(cssText, ___.iM([
                              'startStylesheet', (function () {
                                  function startStylesheet$_lit() {
                                    safeCss = [ ];
                                  }
                                  return ___.f(startStylesheet$_lit,
                                    'startStylesheet$_lit');
                                })(), 'endStylesheet', (function () {
                                  function endStylesheet$_lit() {}
                                  return ___.f(endStylesheet$_lit,
                                    'endStylesheet$_lit');
                                })(), 'startAtrule', (function () {
                                  function startAtrule$_lit(atIdent,
                                    headerArray) {
                                    var x0___, x1___, x2___, x3___, x4___,
                                    x5___, x6___, x7___, x8___;
                                    if (elide) { atIdent = null; } else if
                                      (atIdent === '@media') {
                                      headerArray = (x1___ = headerArray, x0___
                                          = ___.f(function (mediaType) {
                                            return cssMediaTypeWhitelist.v___(mediaType)
                                              == allowed;
                                          }), x1___.filter_m___?
                                          x1___.filter(x0___):
                                        x1___.m___('filter', [ x0___ ]));
                                      if (headerArray.length) {
                                        x3___ = safeCss, x4___ = atIdent, x2___
                                          = headerArray.join_m___?
                                          headerArray.join(','):
                                        headerArray.m___('join', [ ',' ]),
                                        x3___.push_m___? x3___.push(x4___,
                                          x2___, '{'): x3___.m___('push', [
                                            x4___, x2___, '{' ]);
                                      } else { atIdent = null; }
                                    } else {
                                      if (atIdent === '@import') {
                                        if (x5___ = IMPORTS___.window_v___?
                                            IMPORTS___.window:
                                          ___.ri(IMPORTS___, 'window'),
                                          x5___.console_v___? x5___.console:
                                          x5___.v___('console')) {
                                          x7___ = (x6___ =
                                              IMPORTS___.window_v___?
                                              IMPORTS___.window:
                                            ___.ri(IMPORTS___, 'window'),
                                            x6___.console_v___? x6___.console:
                                            x6___.v___('console')), x8___ =
                                            '@import ' +
                                            (headerArray.join_m___?
                                            headerArray.join(' '):
                                            headerArray.m___('join', [ ' ' ]))
                                            + ' elided', x7___.log_m___?
                                            x7___.log(x8___): x7___.m___('log',
                                            [ x8___ ]);
                                        }
                                      }
                                      atIdent = null;
                                    }
                                    elide = !atIdent;
                                    blockStack.push_m___?
                                      blockStack.push(atIdent):
                                    blockStack.m___('push', [ atIdent ]);
                                  }
                                  return ___.f(startAtrule$_lit,
                                    'startAtrule$_lit');
                                })(), 'endAtrule', (function () {
                                  function endAtrule$_lit() {
                                    var atIdent;
                                    atIdent = blockStack.pop_m___?
                                      blockStack.pop(): blockStack.m___('pop',
                                      [ ]);
                                    if (!elide) {
                                      safeCss.push_m___? safeCss.push(';'):
                                      safeCss.m___('push', [ ';' ]);
                                    }
                                    checkElide.i___();
                                  }
                                  return ___.f(endAtrule$_lit,
                                    'endAtrule$_lit');
                                })(), 'startBlock', (function () {
                                  function startBlock$_lit() {
                                    if (!elide) {
                                      safeCss.push_m___? safeCss.push('{'):
                                      safeCss.m___('push', [ '{' ]);
                                    }
                                  }
                                  return ___.f(startBlock$_lit,
                                    'startBlock$_lit');
                                })(), 'endBlock', (function () {
                                  function endBlock$_lit() {
                                    if (!elide) {
                                      safeCss.push_m___? safeCss.push('}'):
                                      safeCss.m___('push', [ '}' ]);
                                      elide = true;
                                    }
                                  }
                                  return ___.f(endBlock$_lit, 'endBlock$_lit');
                                })(), 'startRuleset', (function () {
                                  function startRuleset$_lit(selectorArray) {
                                    var historySensitiveSelectors,
                                    removeHistoryInsensitiveSelectors,
                                    selectors, historyInsensitiveSelectors,
                                    selector, x0___, x1___;
                                    historySensitiveSelectors = void 0;
                                    removeHistoryInsensitiveSelectors = false;
                                    if (!elide) {
                                      selectors =
                                        (IMPORTS___.sanitizeCssSelectors_v___?
                                          IMPORTS___.sanitizeCssSelectors:
                                        ___.ri(IMPORTS___,
                                          'sanitizeCssSelectors'))
                                        .i___(selectorArray, suffix);
                                      historyInsensitiveSelectors = selectors[
                                        +0 ];
                                      historySensitiveSelectors = selectors[ +1
                                      ];
                                      if (!historyInsensitiveSelectors.length
                                          && !historySensitiveSelectors.length)
                                        { elide = true; } else {
                                        selector =
                                          historyInsensitiveSelectors.join_m___?
                                          historyInsensitiveSelectors.join(', ')
                                          :
                                        historyInsensitiveSelectors.m___('join',
                                          [ ', ' ]);
                                        if (!selector) {
                                          selector = 'head \x3e html';
                                          removeHistoryInsensitiveSelectors =
                                            true;
                                        }
                                        safeCss.push_m___?
                                          safeCss.push(selector, '{'):
                                        safeCss.m___('push', [ selector, '{' ])
                                          ;
                                      }
                                    }
                                    x1___ = blockStack, x0___ = elide? null:
                                    ___.iM([ 'historySensitiveSelectors',
                                        historySensitiveSelectors,
                                        'endOfSelectors', safeCss.length - 1,
                                        'removeHistoryInsensitiveSelectors',
                                        removeHistoryInsensitiveSelectors ]),
                                    x1___.push_m___? x1___.push(x0___):
                                    x1___.m___('push', [ x0___ ]);
                                  }
                                  return ___.f(startRuleset$_lit,
                                    'startRuleset$_lit');
                                })(), 'endRuleset', (function () {
                                  function endRuleset$_lit() {
                                    var rules, propertiesEnd, extraSelectors,
                                    propertyGroupTokens, x0___, x1___, x2___,
                                    x3___, x4___, x5___, x6___, x7___;
                                    rules = blockStack.pop_m___?
                                      blockStack.pop(): blockStack.m___('pop',
                                      [ ]);
                                    propertiesEnd = safeCss.length;
                                    if (!elide) {
                                      safeCss.push_m___? safeCss.push('}'):
                                      safeCss.m___('push', [ '}' ]);
                                      if (rules) {
                                        extraSelectors =
                                          rules.historySensitiveSelectors_v___?
                                          rules.historySensitiveSelectors:
                                        rules.v___('historySensitiveSelectors')
                                          ;
                                        if (extraSelectors.length) {
                                          propertyGroupTokens = (x1___ =
                                            safeCss, x0___ =
                                            rules.endOfSelectors_v___?
                                            rules.endOfSelectors:
                                            rules.v___('endOfSelectors'),
                                            x1___.slice_m___?
                                            x1___.slice(x0___):
                                            x1___.m___('slice', [ x0___ ]));
                                          x4___ = safeCss, x2___ =
                                            extraSelectors.join_m___?
                                            extraSelectors.join(', '):
                                          extraSelectors.m___('join', [ ', ' ])
                                            , x3___ =
                                            sanitizeHistorySensitive.i___(propertyGroupTokens)
                                            , x4___.push_m___?
                                            x4___.push(x2___, x3___):
                                          x4___.m___('push', [ x2___, x3___ ]);
                                        }
                                      }
                                    }
                                    if (rules &&
                                      (rules.removeHistoryInsensitiveSelectors_v___?
                                         rules.removeHistoryInsensitiveSelectors:
                                       rules.v___('removeHistoryInsensitiveSelectors')
                                      )) {
                                      x7___ = safeCss, x5___ =
                                        (rules.endOfSelectors_v___?
                                          rules.endOfSelectors:
                                        rules.v___('endOfSelectors')) - 1,
                                      x6___ = propertiesEnd + 1,
                                      x7___.splice_m___? x7___.splice(x5___,
                                        x6___): x7___.m___('splice', [ x5___,
                                          x6___ ]);
                                    }
                                    checkElide.i___();
                                  }
                                  return ___.f(endRuleset$_lit,
                                    'endRuleset$_lit');
                                })(), 'declaration', (function () {
                                  function declaration$_lit(property,
                                    valueArray) {
                                    var schema, x0___, x1___, x2___;
                                    if (!elide) {
                                      schema = (IMPORTS___.cssSchema_v___?
                                          IMPORTS___.cssSchema:
                                        ___.ri(IMPORTS___, 'cssSchema'))
                                        .v___(property);
                                      if (schema) {
                                        (IMPORTS___.sanitizeCssProperty_v___?
                                           IMPORTS___.sanitizeCssProperty:
                                         ___.ri(IMPORTS___,
                                           'sanitizeCssProperty'))
                                        .i___(property, schema, valueArray,
                                          opt_naiveUriRewriter, baseUri);
                                        if (valueArray.length) {
                                          x1___ = safeCss, x2___ = property,
                                          x0___ = valueArray.join_m___?
                                            valueArray.join(' '):
                                          valueArray.m___('join', [ ' ' ]),
                                          x1___.push_m___? x1___.push(x2___,
                                            ':', x0___, ';'):
                                          x1___.m___('push', [ x2___, ':',
                                              x0___, ';' ]);
                                        }
                                      }
                                    }
                                  }
                                  return ___.f(declaration$_lit,
                                    'declaration$_lit');
                                })() ]));
                        return safeCss.join_m___? safeCss.join(''):
                        safeCss.m___('join', [ '' ]);
                      });
                  }).i___());
              if (typeof IMPORTS___.v___('window') !== 'undefined') {
                (IMPORTS___.window_v___? IMPORTS___.window: ___.ri(IMPORTS___,
                   'window')).w___('sanitizeCssProperty',
                  IMPORTS___.sanitizeCssProperty_v___?
                    IMPORTS___.sanitizeCssProperty: ___.ri(IMPORTS___,
                    'sanitizeCssProperty'));
                (IMPORTS___.window_v___? IMPORTS___.window: ___.ri(IMPORTS___,
                   'window')).w___('sanitizeCssSelectors',
                  IMPORTS___.sanitizeCssSelectors_v___?
                    IMPORTS___.sanitizeCssSelectors: ___.ri(IMPORTS___,
                    'sanitizeCssSelectors'));
                moduleResult___ = (IMPORTS___.window_v___? IMPORTS___.window:
                  ___.ri(IMPORTS___, 'window')).w___('sanitizeStylesheet',
                  IMPORTS___.sanitizeStylesheet_v___?
                  IMPORTS___.sanitizeStylesheet: ___.ri(IMPORTS___,
                    'sanitizeStylesheet'));
              }
              return moduleResult___;
            },
            'cajolerName': 'com.google.caja',
            'cajolerVersion': '5054',
            'cajoledDate': 1347987483746
          }).instantiate___(___, IMPORTS___), ___.prepareModule({
            'instantiate': function (___, IMPORTS___) {
              var dis___ = IMPORTS___;
              var moduleResult___;
              moduleResult___ = ___.NO_RESULT;
              if (('I'.toLowerCase_m___? 'I'.toLowerCase():
                  'I'.m___('toLowerCase', [ ])) !== 'i') {
                throw 'I/i problem';
              }
              ___.di(IMPORTS___, 'parseCssStylesheet');
              ___.di(IMPORTS___, 'parseCssDeclarations');
              ___.f(function () {
                  var statement, atrule, block, ruleset, selector, ident,
                  declaration;
                  {
                    function statement(toks, i, n, handler) {
                      var tok;
                      if (i < n) {
                        tok = toks.v___(i);
                        if ((tok.charAt_m___? tok.charAt(0): tok.m___('charAt',
                              [ 0 ])) === '@') {
                          return atrule.i___(toks, i, n, handler, true);
                        } else {
                          return ruleset.i___(toks, i, n, handler);
                        }
                      } else { return i; }
                    }
                    ___.f(statement, 'statement');
                  }
                  {
                    function atrule(toks, i, n, handler, blockok) {
                      var start, s, e, x0___, x1___, x2___, x3___;
                      start = i++;
                      while (i < n && toks.v___(i) !== '{' && toks.v___(i) !==
                          ';') { ++i; }
                      if (i < n && (blockok || toks.v___(i) === ';')) {
                        s = start + 1, e = i;
                        if (s < n && toks.v___(s) === ' ') { ++s; }
                        if (e > s && toks[ + (e - 1) ] === ' ') { --e; }
                        if (handler.startAtrule_v___? handler.startAtrule:
                          handler.v___('startAtrule')) {
                          x3___ = handler, x1___ = (x0___ = toks.v___(start),
                            x0___.toLowerCase_m___? x0___.toLowerCase():
                            x0___.m___('toLowerCase', [ ])), x2___ =
                            toks.slice_m___? toks.slice(s, e):
                          toks.m___('slice', [ s, e ]), x3___.startAtrule_m___?
                            x3___.startAtrule(x1___, x2___):
                          x3___.m___('startAtrule', [ x1___, x2___ ]);
                        }
                        i = toks.v___(i) === '{'? block.i___(toks, i, n,
                          handler): i + 1;
                        if (handler.endAtrule_v___? handler.endAtrule:
                          handler.v___('endAtrule')) {
                          handler.endAtrule_m___? handler.endAtrule():
                          handler.m___('endAtrule', [ ]);
                        }
                      }
                      return i;
                    }
                    ___.f(atrule, 'atrule');
                  }
                  {
                    function block(toks, i, n, handler) {
                      var ch, x0___;
                      ++i;
                      if (handler.startBlock_v___? handler.startBlock:
                        handler.v___('startBlock')) {
                        handler.startBlock_m___? handler.startBlock():
                        handler.m___('startBlock', [ ]);
                      }
                      while (i < n) {
                        ch = (x0___ = toks.v___(i), x0___.charAt_m___?
                            x0___.charAt(0): x0___.m___('charAt', [ 0 ]));
                        if (ch == '}') { ++i;
                          break; }
                        if (ch === ' ' || ch === ';') {
                          i = i + 1;
                        } else if (ch === '@') {
                          i = atrule.i___(toks, i, n, handler, false);
                        } else if (ch === '{') {
                          i = block.i___(toks, i, n, handler);
                        } else {
                          i = ruleset.i___(toks, i, n, handler);
                        }
                      }
                      if (handler.endBlock_v___? handler.endBlock:
                        handler.v___('endBlock')) {
                        handler.endBlock_m___? handler.endBlock():
                        handler.m___('endBlock', [ ]);
                      }
                      return i;
                    }
                    ___.f(block, 'block');
                  }
                  {
                    function ruleset(toks, i, n, handler) {
                      var s, e, tok, x0___, x1___;
                      s = i, e = selector.i___(toks, i, n, true);
                      if (e < 0) {
                        e = ~e;
                        return i === e? e + 1: e;
                      }
                      i = e;
                      if (e > s && toks[ + (e - 1) ] === ' ') { --e; }
                      tok = toks.v___(i);
                      ++i;
                      if (tok !== '{') { return i; }
                      if (handler.startRuleset_v___? handler.startRuleset:
                        handler.v___('startRuleset')) {
                        x1___ = handler, x0___ = toks.slice_m___? toks.slice(s,
                          e): toks.m___('slice', [ s, e ]),
                        x1___.startRuleset_m___? x1___.startRuleset(x0___):
                        x1___.m___('startRuleset', [ x0___ ]);
                      }
                      while (i < n) {
                        tok = toks.v___(i);
                        if (tok === '}') { ++i;
                          break; }
                        if (tok === ' ') {
                          i = i + 1;
                        } else {
                          i = declaration.i___(toks, i, n, handler);
                        }
                      }
                      if (handler.endRuleset_v___? handler.endRuleset:
                        handler.v___('endRuleset')) {
                        handler.endRuleset_m___? handler.endRuleset():
                        handler.m___('endRuleset', [ ]);
                      }
                      return i < n? i + 1: i;
                    }
                    ___.f(ruleset, 'ruleset');
                  }
                  {
                    function selector(toks, i, n, allowSemi) {
                      var s, tok, brackets, stackLast, x0___;
                      s = i;
                      brackets = [ ], stackLast = -1;
                      for (; i < n; ++i) {
                        tok = (x0___ = toks.v___(i), x0___.charAt_m___?
                          x0___.charAt(0): x0___.m___('charAt', [ 0 ]));
                        if (tok === '[' || tok === '(') {
                          brackets.NUM____w___ === brackets? (brackets[ + ++stackLast ] = tok): brackets.w___(+ ++stackLast, tok);
                        } else if (tok === ']' && brackets[ +stackLast ] ===
                            '[' || tok === ')' && brackets[ +stackLast ] ===
                            '(') { --stackLast; } else if (tok === '{' || tok
                            === '}' || tok === ';' || tok === '@' || tok ===
                            ':' && !allowSemi) { break; }
                      }
                      if (stackLast >= 0) {
                        i = ~ (i + 1);
                      }
                      return i;
                    }
                    ___.f(selector, 'selector');
                  }
                  {
                    function declaration(toks, i, n, handler) {
                      var property, tok, s, e, value, valuelen, j, x0___,
                      x1___, x2___;
                      property = toks.v___(i++);
                      if (! (ident.test_m___? ident.test(property):
                          ident.m___('test', [ property ]))) {
                        return i + 1;
                      }
                      if (i < n && toks.v___(i) === ' ') { ++i; }
                      if (i == n || toks.v___(i) !== ':') {
                        while (i < n && (tok = toks.v___(i)) !== ';' && tok !==
                            '}') { ++i; }
                        return i;
                      }
                      ++i;
                      if (i < n && toks.v___(i) === ' ') { ++i; }
                      s = i, e = selector.i___(toks, i, n, false);
                      if (e < 0) {
                        e = ~e;
                      } else {
                        value = [ ], valuelen = 0;
                        for (j = s; j < e; ++j) {
                          tok = toks.v___(j);
                          if (tok !== ' ') {
                            value.w___(valuelen++, tok);
                          }
                        }
                        if (e < n) {
                          do {
                            tok = toks.v___(e);
                            if (tok === ';' || tok === '}') { break; }
                            valuelen = 0;
                          } while (++e < n);
                          if (tok === ';') { ++e; }
                        }
                        if (valuelen && (handler.declaration_v___?
                            handler.declaration: handler.v___('declaration')))
                          {
                          x1___ = handler, x0___ = property.toLowerCase_m___?
                            property.toLowerCase():
                          property.m___('toLowerCase', [ ]), x2___ = value,
                          x1___.declaration_m___? x1___.declaration(x0___,
                            x2___): x1___.m___('declaration', [ x0___, x2___ ])
                            ;
                        }
                      }
                      return e;
                    }
                    ___.f(declaration, 'declaration');
                  }
                  IMPORTS___.parseCssStylesheet_w___?
                    (IMPORTS___.parseCssStylesheet = (function () {
                        function parseCssStylesheet$_var(cssText, handler) {
                          var toks, i, n;
                          toks = (IMPORTS___.lexCss_v___? IMPORTS___.lexCss:
                            ___.ri(IMPORTS___, 'lexCss')).i___(cssText);
                          if (handler.startStylesheet_v___?
                            handler.startStylesheet:
                            handler.v___('startStylesheet')) {
                            handler.startStylesheet_m___?
                              handler.startStylesheet():
                            handler.m___('startStylesheet', [ ]);
                          }
                          for (i = 0, n = toks.length; i < n;) {
                            i = toks.v___(i) === ' '? i + 1:
                            statement.i___(toks, i, n, handler);
                          }
                          if (handler.endStylesheet_v___?
                            handler.endStylesheet:
                            handler.v___('endStylesheet')) {
                            handler.endStylesheet_m___? handler.endStylesheet()
                              : handler.m___('endStylesheet', [ ]);
                          }
                        }
                        return ___.f(parseCssStylesheet$_var,
                          'parseCssStylesheet$_var');
                      })()): ___.wi(IMPORTS___, 'parseCssStylesheet', (function
                        () {
                        function parseCssStylesheet$_var(cssText, handler) {
                          var toks, i, n;
                          toks = (IMPORTS___.lexCss_v___? IMPORTS___.lexCss:
                            ___.ri(IMPORTS___, 'lexCss')).i___(cssText);
                          if (handler.startStylesheet_v___?
                              handler.startStylesheet:
                            handler.v___('startStylesheet')) {
                            handler.startStylesheet_m___?
                              handler.startStylesheet():
                            handler.m___('startStylesheet', [ ]);
                          }
                          for (i = 0, n = toks.length; i < n;) {
                            i = toks.v___(i) === ' '? i + 1:
                            statement.i___(toks, i, n, handler);
                          }
                          if (handler.endStylesheet_v___?
                            handler.endStylesheet:
                            handler.v___('endStylesheet')) {
                            handler.endStylesheet_m___? handler.endStylesheet()
                              : handler.m___('endStylesheet', [ ]);
                          }
                        }
                        return ___.f(parseCssStylesheet$_var,
                          'parseCssStylesheet$_var');
                      })());
                  ident = new RegExp.new___('^-?[a-z]', 'i');
                  IMPORTS___.parseCssDeclarations_w___?
                    (IMPORTS___.parseCssDeclarations = (function () {
                        function parseCssDeclarations$_var(cssText, handler) {
                          var toks, i, n;
                          toks = (IMPORTS___.lexCss_v___? IMPORTS___.lexCss:
                            ___.ri(IMPORTS___, 'lexCss')).i___(cssText);
                          for (i = 0, n = toks.length; i < n;) {
                            i = toks.v___(i) !== ' '? declaration.i___(toks, i,
                              n, handler): i + 1;
                          }
                        }
                        return ___.f(parseCssDeclarations$_var,
                          'parseCssDeclarations$_var');
                      })()): ___.wi(IMPORTS___, 'parseCssDeclarations',
                    (function () {
                       function parseCssDeclarations$_var(cssText, handler) {
                         var toks, i, n;
                         toks = (IMPORTS___.lexCss_v___? IMPORTS___.lexCss:
                           ___.ri(IMPORTS___, 'lexCss')).i___(cssText);
                         for (i = 0, n = toks.length; i < n;) {
                           i = toks.v___(i) !== ' '? declaration.i___(toks, i,
                             n, handler): i + 1;
                         }
                       }
                       return ___.f(parseCssDeclarations$_var,
                         'parseCssDeclarations$_var');
                     })());
                }).i___();
              if (typeof IMPORTS___.v___('window') !== 'undefined') {
                (IMPORTS___.window_v___? IMPORTS___.window: ___.ri(IMPORTS___,
                   'window')).w___('parseCssStylesheet',
                  IMPORTS___.parseCssStylesheet_v___?
                    IMPORTS___.parseCssStylesheet: ___.ri(IMPORTS___,
                    'parseCssStylesheet'));
                moduleResult___ = (IMPORTS___.window_v___? IMPORTS___.window:
                  ___.ri(IMPORTS___, 'window')).w___('parseCssDeclarations',
                  IMPORTS___.parseCssDeclarations_v___?
                  IMPORTS___.parseCssDeclarations: ___.ri(IMPORTS___,
                    'parseCssDeclarations'));
              }
              return moduleResult___;
            },
            'cajolerName': 'com.google.caja',
            'cajolerVersion': '5054',
            'cajoledDate': 1347987483805
          }).instantiate___(___, IMPORTS___), ___.prepareModule({
            'instantiate': function (___, IMPORTS___) {
              var dis___ = IMPORTS___;
              var moduleResult___, x0___, x1___;
              moduleResult___ = ___.NO_RESULT;
              if (('I'.toLowerCase_m___? 'I'.toLowerCase():
                  'I'.m___('toLowerCase', [ ])) !== 'i') {
                throw 'I/i problem';
              }
              ___.di(IMPORTS___, 'bridalMaker'), IMPORTS___.w___('bridalMaker',
                (function () {
                   function bridalMaker$_var(makeDOMAccessible, document) {
                     var docEl, window, x0___, x1___, x2___, navigator,
                     XMLHttpRequest, ActiveXObject, isOpera, x3___, isIE,
                     x4___, isWebkit, x5___, featureAttachEvent,
                     featureExtendedCreateElement, html5shim, hiddenEventTypes,
                     CUSTOM_EVENT_TYPE_SUFFIX, tameEventType,
                     eventHandlerTypeFilter, endsWithUnderbars, escapeAttrib,
                     x6___, constructClone, fixupClone, untameEventType,
                     initEvent, dispatchEvent, addEventListener,
                     removeEventListener, cloneNode, initCanvasElements,
                     initCanvasElement, createElement, createStylesheet,
                     hiddenStoredTarget, setAttribute, getBoundingClientRect,
                     getAttribute, hasAttribute, getComputedStyle, makeXhr;
                     {
                       function html5shim() {
                         var html5_elements, documentFragment, i, x0___, x1___,
                         x2___, x3___;
                         html5_elements = [ 'abbr', 'article', 'aside',
                           'audio', 'canvas', 'details', 'figcaption',
                           'figure', 'footer', 'header', 'hgroup', 'mark',
                           'meter', 'nav', 'output', 'progress', 'section',
                           'summary', 'time', 'video' ];
                         documentFragment =
                           makeDOMAccessible.i___(document.createDocumentFragment_m___?
                           document.createDocumentFragment():
                           document.m___('createDocumentFragment', [ ]));
                         for (i = 0; i < html5_elements.length; i++) {
                           try {
                             x1___ = document, x0___ = html5_elements[ +i ],
                             x1___.createElement_m___?
                               x1___.createElement(x0___):
                             x1___.m___('createElement', [ x0___ ]);
                             x3___ = documentFragment, x2___ = html5_elements[
                               +i ], x3___.createElement_m___?
                               x3___.createElement(x2___):
                             x3___.m___('createElement', [ x2___ ]);
                           } catch (e) {}
                         }
                       }
                       ___.f(html5shim, 'html5shim');
                     }
                     {
                       function tameEventType(type, opt_isCustom, opt_tagName)
                         {
                         var tagAttr, x0___, x1___, x2___, x3___, x4___, x5___,
                         x6___;
                         type = (IMPORTS___.String_v___? IMPORTS___.String:
                           ___.ri(IMPORTS___, 'String')).i___(type);
                         if (endsWithUnderbars.test_m___?
                           endsWithUnderbars.test(type):
                           endsWithUnderbars.m___('test', [ type ])) {
                           throw new (IMPORTS___.Error_v___? IMPORTS___.Error:
                             ___.ri(IMPORTS___, 'Error'))
                             .new___(('Invalid event type ' + type));
                         }
                         tagAttr = false;
                         if (opt_tagName) {
                           tagAttr = (x0___ = (IMPORTS___.String_v___?
                               IMPORTS___.String: ___.ri(IMPORTS___, 'String'))
                             .i___(opt_tagName), x0___.toLowerCase_m___?
                             x0___.toLowerCase(): x0___.m___('toLowerCase', [ ]
                             )) + '::on' + type;
                         }
                         if (!opt_isCustom && (tagAttr && (x2___ = (x1___ =
                                 IMPORTS___.html4_v___? IMPORTS___.html4:
                                 ___.ri(IMPORTS___, 'html4'), x1___.atype_v___?
                                 x1___.atype: x1___.v___('atype')),
                               x2___.SCRIPT_v___? x2___.SCRIPT:
                               x2___.v___('SCRIPT')) === (x3___ =
                               IMPORTS___.html4_v___? IMPORTS___.html4:
                               ___.ri(IMPORTS___, 'html4'), x3___.ATTRIBS_v___?
                               x3___.ATTRIBS: x3___.v___('ATTRIBS'))
                             .v___(tagAttr) || (x5___ = (x4___ =
                                 IMPORTS___.html4_v___? IMPORTS___.html4:
                                 ___.ri(IMPORTS___, 'html4'), x4___.atype_v___?
                                 x4___.atype: x4___.v___('atype')),
                               x5___.SCRIPT_v___? x5___.SCRIPT:
                               x5___.v___('SCRIPT')) === (x6___ =
                               IMPORTS___.html4_v___? IMPORTS___.html4:
                               ___.ri(IMPORTS___, 'html4'), x6___.ATTRIBS_v___?
                               x6___.ATTRIBS: x6___.v___('ATTRIBS'))
                             .v___('*::on' + type))) { return type; }
                         return type + CUSTOM_EVENT_TYPE_SUFFIX;
                       }
                       ___.f(tameEventType, 'tameEventType');
                     }
                     {
                       function eventHandlerTypeFilter(handler, tameType) {
                         return ___.f(function (event) {
                             var dis___ = this && this.___? void 0: this;
                             if (hiddenEventTypes && tameType ===
                                 (hiddenEventTypes.get_m___?
                                   hiddenEventTypes.get(event):
                                 hiddenEventTypes.m___('get', [ event ]))) {
                               return handler.call_m___? handler.call(dis___,
                                 event): handler.m___('call', [ dis___, event ]
                               );
                             }
                           });
                       }
                       ___.f(eventHandlerTypeFilter, 'eventHandlerTypeFilter');
                     }
                     {
                       function constructClone(node, deep) {
                         var clone, tagDesc, attrs, i, attr, x0___, x1___,
                         child, cloneChild;
                         node = makeDOMAccessible.i___(node);
                         if ((node.nodeType_v___? node.nodeType:
                             node.v___('nodeType')) === 1 &&
                             featureExtendedCreateElement) {
                           tagDesc = node.tagName_v___? node.tagName:
                           node.v___('tagName');
                           switch (node.tagName_v___? node.tagName:
                             node.v___('tagName')) {
                           case 'INPUT':
                             tagDesc = '\x3cinput name=\"' +
                               escapeAttrib.i___(node.name_v___? node.name:
                               node.v___('name')) + '\" type=\"' +
                               escapeAttrib.i___(node.type_v___? node.type:
                               node.v___('type')) + '\" value=\"' +
                               escapeAttrib.i___(node.defaultValue_v___?
                               node.defaultValue: node.v___('defaultValue')) +
                               '\"' + ((node.defaultChecked_v___?
                                 node.defaultChecked:
                                 node.v___('defaultChecked')) ?
                               ' checked=\"checked\"\x3e': '\x3e');
                             break;
                           case 'BUTTON':
                             tagDesc = '\x3cbutton name=\"' +
                               escapeAttrib.i___(node.name_v___? node.name:
                               node.v___('name')) + '\" type=\"' +
                               escapeAttrib.i___(node.type_v___? node.type:
                               node.v___('type')) + '\" value=\"' +
                               escapeAttrib.i___(node.value_v___? node.value:
                               node.v___('value')) + '\"\x3e';
                             break;
                           case 'OPTION':
                             tagDesc = '\x3coption ' +
                               ((node.defaultSelected_v___?
                                 node.defaultSelected:
                                 node.v___('defaultSelected')) ?
                               ' selected=\"selected\"\x3e': '\x3e');
                             break;
                           case 'TEXTAREA':
                             tagDesc = '\x3ctextarea value=\"' +
                               escapeAttrib.i___(node.defaultValue_v___?
                               node.defaultValue: node.v___('defaultValue')) +
                               '\"\x3e';
                             break;
                           }
                           clone =
                             makeDOMAccessible.i___(document.createElement_m___?
                             document.createElement(tagDesc):
                             document.m___('createElement', [ tagDesc ]));
                           attrs = makeDOMAccessible.i___(node.attributes_v___?
                               node.attributes: node.v___('attributes'));
                           for (i = 0; attr = attrs[ +i ]; ++i) {
                             attr = makeDOMAccessible.i___(attr);
                             if ((attr.specified_v___? attr.specified:
                                 attr.v___('specified')) && ! (x1___ =
                                   endsWithUnderbars, x0___ = attr.name_v___?
                                   attr.name: attr.v___('name'),
                                 x1___.test_m___? x1___.test(x0___):
                                 x1___.m___('test', [ x0___ ]))) {
                               setAttribute.i___(clone, attr.nodeName_v___?
                                 attr.nodeName: attr.v___('nodeName'),
                                 attr.nodeValue_v___? attr.nodeValue:
                                 attr.v___('nodeValue'));
                             }
                           }
                         } else {
                           clone = makeDOMAccessible.i___(node.cloneNode_m___?
                               node.cloneNode(false): node.m___('cloneNode', [
                                 false ]));
                         }
                         if (deep) {
                           for (child = node.firstChild_v___? node.firstChild:
                             node.v___('firstChild'); child; child =
                               child.nextSibling_v___? child.nextSibling:
                             child.v___('nextSibling')) {
                             cloneChild = constructClone.i___(child, deep);
                             clone.appendChild_m___?
                               clone.appendChild(cloneChild):
                             clone.m___('appendChild', [ cloneChild ]);
                           }
                         }
                         return clone;
                       }
                       ___.f(constructClone, 'constructClone');
                     }
                     {
                       function fixupClone(node, clone) {
                         var child, cloneChild, x0___, x1___, x2___, x3___,
                         x4___, x5___, x6___, x7___, x8___, x9___,
                         originalAttribs, attribs, k, v, x10___, x11___,
                         x12___, x13___, x14___;
                         node = makeDOMAccessible.i___(node);
                         clone = makeDOMAccessible.i___(clone);
                         for (child = node.firstChild_v___? node.firstChild:
                           node.v___('firstChild'), cloneChild =
                           clone.firstChild_v___? clone.firstChild:
                           clone.v___('firstChild'); cloneChild; child =
                           child.nextSibling_v___? child.nextSibling:
                           child.v___('nextSibling'), cloneChild =
                           cloneChild.nextSibling_v___? cloneChild.nextSibling:
                           cloneChild.v___('nextSibling')) {
                           fixupClone.i___(child, cloneChild);
                         }
                         if ((node.nodeType_v___? node.nodeType:
                             node.v___('nodeType')) === 1) {
                           switch (node.tagName_v___? node.tagName:
                             node.v___('tagName')) {
                           case 'INPUT':
                             x1___ = clone, x0___ = node.value_v___?
                               node.value: node.v___('value'), x1___.value_w___
                               === x1___? (x1___.value = x0___):
                             x1___.w___('value', x0___);
                             x3___ = clone, x2___ = node.checked_v___?
                               node.checked: node.v___('checked'),
                             x3___.checked_w___ === x3___? (x3___.checked =
                               x2___): x3___.w___('checked', x2___);
                             break;
                           case 'OPTION':
                             x5___ = clone, x4___ = node.selected_v___?
                               node.selected: node.v___('selected'),
                             x5___.selected_w___ === x5___? (x5___.selected =
                               x4___): x5___.w___('selected', x4___);
                             x7___ = clone, x6___ = node.value_v___?
                               node.value: node.v___('value'), x7___.value_w___
                               === x7___? (x7___.value = x6___):
                             x7___.w___('value', x6___);
                             break;
                           case 'TEXTAREA':
                             x9___ = clone, x8___ = node.value_v___?
                               node.value: node.v___('value'), x9___.value_w___
                               === x9___? (x9___.value = x8___):
                             x9___.w___('value', x8___);
                             break;
                           }
                         }
                         originalAttribs = node._d_attributes_v___?
                           node._d_attributes: node.v___('_d_attributes');
                         if (originalAttribs) {
                           attribs = ___.iM([ ]);
                           clone._d_attributes_w___ === clone?
                             (clone._d_attributes = attribs):
                           clone.w___('_d_attributes', attribs);
                           x12___ = Object(originalAttribs).e___();
                           for (x11___ in x12___) {
                             if (typeof x11___ === 'number' || '' + +x11___ ===
                               x11___) { k = x11___; } else {
                               if (/^NUM___/.test(x11___) && /__$/.test(x11___)
                               ) { continue; }
                               x10___ = x11___.match(/([\s\S]*)_e___$/);
                               if (!x10___ || !x12___[ x11___ ]) { continue; }
                               k = x10___[ 1 ];
                             }
                             {
                               if (x13___ = new RegExp.new___('__$'), x14___ =
                                   k, x13___.test_m___? x13___.test(x14___):
                                 x13___.m___('test', [ x14___ ])) { continue; }
                               v = originalAttribs.v___(k);
                               switch (typeof v) {
                               case 'string':
                               case 'number':
                               case 'boolean':
                                 attribs.w___(k, v);
                                 break;
                               }
                             }
                           }
                         }
                       }
                       ___.f(fixupClone, 'fixupClone');
                     }
                     {
                       function untameEventType(type) {
                         var suffix, tlen, slen, end;
                         suffix = CUSTOM_EVENT_TYPE_SUFFIX;
                         tlen = type.length, slen = suffix.length;
                         end = tlen - slen;
                         if (end >= 0 && suffix === (type.substring_m___?
                             type.substring(end): type.m___('substring', [ end
                               ]))) {
                           type = type.substring_m___? type.substring(0, end):
                           type.m___('substring', [ 0, end ]);
                         }
                         return type;
                       }
                       ___.f(untameEventType, 'untameEventType');
                     }
                     {
                       function initEvent(event, type, bubbles, cancelable) {
                         event = makeDOMAccessible.i___(event);
                         type = tameEventType.i___(type, true);
                         bubbles = (IMPORTS___.Boolean_v___?
                             IMPORTS___.Boolean: ___.ri(IMPORTS___, 'Boolean'))
                           .i___(bubbles);
                         cancelable = (IMPORTS___.Boolean_v___?
                             IMPORTS___.Boolean: ___.ri(IMPORTS___, 'Boolean'))
                           .i___(cancelable);
                         if (event.initEvent_v___? event.initEvent:
                           event.v___('initEvent')) {
                           event.initEvent_m___? event.initEvent(type, bubbles,
                             cancelable): event.m___('initEvent', [ type,
                               bubbles, cancelable ]);
                         } else if (bubbles && cancelable) {
                           if (!hiddenEventTypes) {
                             hiddenEventTypes = new (IMPORTS___.WeakMap_v___?
                               IMPORTS___.WeakMap: ___.ri(IMPORTS___,
                                 'WeakMap')).new___;
                           }
                           hiddenEventTypes.set_m___?
                             hiddenEventTypes.set(event, type):
                           hiddenEventTypes.m___('set', [ event, type ]);
                         } else {
                           throw new (IMPORTS___.Error_v___? IMPORTS___.Error:
                             ___.ri(IMPORTS___, 'Error'))
                             .new___('Browser does not support non-bubbling/uncanceleable events')
                             ;
                         }
                       }
                       ___.f(initEvent, 'initEvent');
                     }
                     {
                       function dispatchEvent(element, event) {
                         element = makeDOMAccessible.i___(element);
                         event = makeDOMAccessible.i___(event);
                         if (element.dispatchEvent_v___? element.dispatchEvent:
                           element.v___('dispatchEvent')) {
                           return (IMPORTS___.Boolean_v___? IMPORTS___.Boolean:
                             ___.ri(IMPORTS___, 'Boolean'))
                             .i___(element.dispatchEvent_m___?
                               element.dispatchEvent(event):
                             element.m___('dispatchEvent', [ event ]));
                         } else {
                           element.fireEvent_m___?
                             element.fireEvent('ondataavailable', event):
                           element.m___('fireEvent', [ 'ondataavailable', event
                             ]);
                           return (IMPORTS___.Boolean_v___? IMPORTS___.Boolean:
                             ___.ri(IMPORTS___, 'Boolean'))
                             .i___(event.returnValue_v___? event.returnValue:
                             event.v___('returnValue'));
                         }
                       }
                       ___.f(dispatchEvent, 'dispatchEvent');
                     }
                     {
                       function addEventListener(element, type, handler,
                         useCapture) {
                         var tameType, wrapper, x0___, x1___, x2___;
                         element = makeDOMAccessible.i___(element);
                         type = (IMPORTS___.String_v___? IMPORTS___.String:
                           ___.ri(IMPORTS___, 'String')).i___(type);
                         tameType = tameEventType.i___(type, false,
                           element.tagName_v___? element.tagName:
                           element.v___('tagName'));
                         if (featureAttachEvent) {
                           if (type !== tameType) {
                             wrapper = eventHandlerTypeFilter.i___(handler,
                               tameType);
                             element.attachEvent_m___?
                               element.attachEvent('ondataavailable', wrapper):
                             element.m___('attachEvent', [ 'ondataavailable',
                                 wrapper ]);
                             return wrapper;
                           } else {
                             x1___ = element, x0___ = 'on' + type, x2___ =
                               handler, x1___.attachEvent_m___?
                               x1___.attachEvent(x0___, x2___):
                             x1___.m___('attachEvent', [ x0___, x2___ ]);
                             return handler;
                           }
                         } else {
                           element.addEventListener_m___?
                             element.addEventListener(tameType, handler,
                             useCapture): element.m___('addEventListener', [
                               tameType, handler, useCapture ]);
                           return handler;
                         }
                       }
                       ___.f(addEventListener, 'addEventListener');
                     }
                     {
                       function removeEventListener(element, type, handler,
                         useCapture) {
                         var tameType, x0___, x1___, x2___;
                         element = makeDOMAccessible.i___(element);
                         type = (IMPORTS___.String_v___? IMPORTS___.String:
                           ___.ri(IMPORTS___, 'String')).i___(type);
                         tameType = tameEventType.i___(type, false,
                           element.tagName_v___? element.tagName:
                           element.v___('tagName'));
                         if (featureAttachEvent) {
                           if (tameType !== type) {
                             element.detachEvent_m___?
                               element.detachEvent('ondataavailable', handler):
                             element.m___('detachEvent', [ 'ondataavailable',
                                 handler ]);
                           } else {
                             x1___ = element, x0___ = 'on' + type, x2___ =
                               handler, x1___.detachEvent_m___?
                               x1___.detachEvent(x0___, x2___):
                             x1___.m___('detachEvent', [ x0___, x2___ ]);
                           }
                         } else {
                           element.removeEventListener_m___?
                             element.removeEventListener(tameType, handler,
                             useCapture): element.m___('removeEventListener', [
                               tameType, handler, useCapture ]);
                         }
                       }
                       ___.f(removeEventListener, 'removeEventListener');
                     }
                     {
                       function cloneNode(node, deep) {
                         var clone;
                         node = makeDOMAccessible.i___(node);
                         if (! (document.all_v___? document.all:
                             document.v___('all'))) {
                           clone = node.cloneNode_m___? node.cloneNode(deep):
                           node.m___('cloneNode', [ deep ]);
                         } else {
                           clone = constructClone.i___(node, deep);
                         }
                         fixupClone.i___(node, clone);
                         return clone;
                       }
                       ___.f(cloneNode, 'cloneNode');
                     }
                     {
                       function initCanvasElements(doc) {
                         var els, x0___, i;
                         els = (x0___ = makeDOMAccessible.i___(doc),
                           x0___.getElementsByTagName_m___?
                             x0___.getElementsByTagName('canvas'):
                           x0___.m___('getElementsByTagName', [ 'canvas' ]));
                         for (i = 0; i < els.length; i++) {
                           initCanvasElement.i___(makeDOMAccessible.i___(els[
                                 +i ]));
                         }
                       }
                       ___.f(initCanvasElements, 'initCanvasElements');
                     }
                     {
                       function initCanvasElement(el) {
                         var x0___, x1___;
                         if (window.G_vmlCanvasManager_v___?
                             window.G_vmlCanvasManager:
                           window.v___('G_vmlCanvasManager')) {
                           x0___ = window.G_vmlCanvasManager_v___?
                             window.G_vmlCanvasManager:
                           window.v___('G_vmlCanvasManager'), x1___ = el,
                           x0___.initElement_m___? x0___.initElement(x1___):
                           x0___.m___('initElement', [ x1___ ]);
                         }
                       }
                       ___.f(initCanvasElement, 'initCanvasElement');
                     }
                     {
                       function createElement(tagName, attribs) {
                         var tag, i, n, x0___, x1___, x2___, x3___, x4___, el,
                         i, n;
                         if (featureExtendedCreateElement) {
                           tag = [ '\x3c', tagName ];
                           for (i = 0, n = attribs.length; i < n; i = i + 2) {
                             x2___ = tag, x0___ = attribs.v___(i), x1___ =
                               escapeAttrib.i___(attribs.v___(i + 1)),
                             x2___.push_m___? x2___.push(' ', x0___, '=\"',
                               x1___, '\"'): x2___.m___('push', [ ' ', x0___,
                                 '=\"', x1___, '\"' ]);
                           }
                           tag.push_m___? tag.push('\x3e'): tag.m___('push', [
                               '\x3e' ]);
                           return makeDOMAccessible.i___((x4___ = document,
                               x3___ = tag.join_m___? tag.join(''):
                               tag.m___('join', [ '' ]),
                               x4___.createElement_m___?
                                 x4___.createElement(x3___):
                               x4___.m___('createElement', [ x3___ ])));
                         } else {
                           el =
                             makeDOMAccessible.i___(document.createElement_m___?
                               document.createElement(tagName):
                             document.m___('createElement', [ tagName ]));
                           for (i = 0, n = attribs.length; i < n; i = i + 2) {
                             setAttribute.i___(el, attribs.v___(i),
                               attribs.v___(i + 1));
                           }
                           return el;
                         }
                       }
                       ___.f(createElement, 'createElement');
                     }
                     {
                       function createStylesheet(document, cssText) {
                         var styleSheet, ssss, x0___, x1___;
                         styleSheet =
                           makeDOMAccessible.i___(document.createElement_m___?
                             document.createElement('style'):
                           document.m___('createElement', [ 'style' ]));
                         styleSheet.setAttribute_m___?
                           styleSheet.setAttribute('type', 'text/css'):
                         styleSheet.m___('setAttribute', [ 'type', 'text/css' ]
                         );
                         ssss =
                           makeDOMAccessible.i___(styleSheet.styleSheet_v___?
                             styleSheet.styleSheet:
                           styleSheet.v___('styleSheet'));
                         if (ssss) {
                           ssss.cssText_w___ === ssss? (ssss.cssText = cssText)
                             : ssss.w___('cssText', cssText);
                         } else {
                           x1___ = styleSheet, x0___ =
                             document.createTextNode_m___?
                             document.createTextNode(cssText):
                           document.m___('createTextNode', [ cssText ]),
                           x1___.appendChild_m___? x1___.appendChild(x0___):
                           x1___.m___('appendChild', [ x0___ ]);
                         }
                         return styleSheet;
                       }
                       ___.f(createStylesheet, 'createStylesheet');
                     }
                     {
                       function setAttribute(element, name, value) {
                         var x0___, x1___, x2___, x3___, x4___, x5___, x6___,
                         x7___, x8___, x9___, x10___, attr, x11___, x12___;
                         element = makeDOMAccessible.i___(element);
                         switch (name) {
                         case 'style':
                           x0___ = makeDOMAccessible.i___(element.style_v___?
                             element.style: element.v___('style')), x1___ =
                             value, x0___.cssText_w___ === x0___?
                             (x0___.cssText = x1___): x0___.w___('cssText',
                             x1___);
                           return value;
                         case 'href':
                           if (x2___ = new RegExp.new___('^javascript:', 'i'),
                             x3___ = value, x2___.test_m___? x2___.test(x3___):
                             x2___.m___('test', [ x3___ ])) {
                             if (!hiddenStoredTarget) {
                               hiddenStoredTarget = new
                                 (IMPORTS___.WeakMap_v___? IMPORTS___.WeakMap:
                                 ___.ri(IMPORTS___, 'WeakMap')).new___;
                             }
                             x5___ = hiddenStoredTarget, x6___ = element, x4___
                               = element.target_v___? element.target:
                             element.v___('target'), x5___.set_m___?
                               x5___.set(x6___, x4___): x5___.m___('set', [
                                 x6___, x4___ ]);
                             element.target_w___ === element? (element.target =
                                 ''): element.w___('target', '');
                           } else if (hiddenStoredTarget &&
                             (hiddenStoredTarget.has_m___?
                              hiddenStoredTarget.has(element):
                              hiddenStoredTarget.m___('has', [ element ]))) {
                             x8___ = element, x7___ =
                               hiddenStoredTarget.get_m___?
                               hiddenStoredTarget.get(element):
                             hiddenStoredTarget.m___('get', [ element ]),
                             x8___.target_w___ === x8___? (x8___.target =
                                 x7___): x8___.w___('target', x7___);
                             hiddenStoredTarget.m___('delete', [ element ]);
                           }
                           break;
                         case 'target':
                           if ((element.href_v___? element.href:
                               element.v___('href')) && (x9___ = new
                                 RegExp.new___('^javascript:', 'i'), x10___ =
                                 element.href_v___? element.href:
                               element.v___('href'), x9___.test_m___?
                                 x9___.test(x10___): x9___.m___('test', [
                                   x10___ ]))) {
                             if (!hiddenStoredTarget) {
                               hiddenStoredTarget = new
                                 (IMPORTS___.WeakMap_v___? IMPORTS___.WeakMap:
                                 ___.ri(IMPORTS___, 'WeakMap')).new___;
                             }
                             hiddenStoredTarget.set_m___?
                               hiddenStoredTarget.set(element, value):
                             hiddenStoredTarget.m___('set', [ element, value ])
                               ;
                             return value;
                           }
                           break;
                         }
                         try {
                           attr = makeDOMAccessible.i___((x11___ =
                                 makeDOMAccessible.i___(element.ownerDocument_v___?
                                   element.ownerDocument:
                                 element.v___('ownerDocument')), x12___ = name,
                               x11___.createAttribute_m___?
                                 x11___.createAttribute(x12___):
                               x11___.m___('createAttribute', [ x12___ ])));
                           attr.value_w___ === attr? (attr.value = value):
                           attr.w___('value', value);
                           element.setAttributeNode_m___?
                             element.setAttributeNode(attr):
                           element.m___('setAttributeNode', [ attr ]);
                         } catch (e) {
                           return element.setAttribute_m___?
                             element.setAttribute(name, value, 0):
                           element.m___('setAttribute', [ name, value, 0 ]);
                         }
                         return value;
                       }
                       ___.f(setAttribute, 'setAttribute');
                     }
                     {
                       function getBoundingClientRect(el) {
                         var doc, cRect, fixupLeft, x0___, x1___, x2___, x3___,
                         x4___, x5___, fixupTop, x6___, x7___, x8___, x9___,
                         x10___, x11___, viewport, pageX, pageY, elBoxObject,
                         viewPortBoxObject, op, opPosition, x12___, x13___,
                         x14___, x15___, x16___, x17___, x18___, op, scrollEl,
                         left, top;
                         makeDOMAccessible.i___(el);
                         doc = el.ownerDocument_v___? el.ownerDocument:
                         el.v___('ownerDocument');
                         if (el.getBoundingClientRect_v___?
                             el.getBoundingClientRect:
                           el.v___('getBoundingClientRect')) {
                           cRect = el.getBoundingClientRect_m___?
                             el.getBoundingClientRect():
                           el.m___('getBoundingClientRect', [ ]);
                           makeDOMAccessible.i___(cRect);
                           if (isIE) {
                             fixupLeft = (x0___ = doc.documentElement_v___?
                                 doc.documentElement:
                               doc.v___('documentElement'),
                               x0___.clientLeft_v___? x0___.clientLeft:
                               x0___.v___('clientLeft')) + (x1___ =
                               doc.body_v___? doc.body: doc.v___('body'),
                               x1___.clientLeft_v___? x1___.clientLeft:
                               x1___.v___('clientLeft'));
                             x3___ = cRect, x2___ = (cRect.left_v___?
                               cRect.left: cRect.v___('left')) - fixupLeft,
                             x3___.left_w___ === x3___? (x3___.left = x2___):
                             x3___.w___('left', x2___);
                             x5___ = cRect, x4___ = (cRect.right_v___?
                               cRect.right: cRect.v___('right')) - fixupLeft,
                             x5___.right_w___ === x5___? (x5___.right = x4___):
                             x5___.w___('right', x4___);
                             fixupTop = (x6___ = doc.documentElement_v___?
                                 doc.documentElement:
                               doc.v___('documentElement'),
                               x6___.clientTop_v___? x6___.clientTop:
                               x6___.v___('clientTop')) + (x7___ =
                               doc.body_v___? doc.body: doc.v___('body'),
                               x7___.clientTop_v___? x7___.clientTop:
                               x7___.v___('clientTop'));
                             x9___ = cRect, x8___ = (cRect.top_v___? cRect.top:
                               cRect.v___('top')) - fixupTop, x9___.top_w___
                               === x9___? (x9___.top = x8___):
                             x9___.w___('top', x8___);
                             x11___ = cRect, x10___ = (cRect.bottom_v___?
                               cRect.bottom: cRect.v___('bottom')) - fixupTop,
                             x11___.bottom_w___ === x11___? (x11___.bottom =
                               x10___): x11___.w___('bottom', x10___);
                           }
                           return ___.iM([ 'top', + (cRect.top_v___? cRect.top:
                                 cRect.v___('top')), 'left', +
                                 (cRect.left_v___? cRect.left:
                                 cRect.v___('left')), 'right', +
                                 (cRect.right_v___? cRect.right:
                                 cRect.v___('right')), 'bottom', +
                                 (cRect.bottom_v___? cRect.bottom:
                                 cRect.v___('bottom')) ]);
                         }
                         viewport = isIE && (doc.compatMode_v___?
                           doc.compatMode: doc.v___('compatMode')) ===
                           'CSS1Compat'? doc.body_v___? doc.body:
                         doc.v___('body'): doc.documentElement_v___?
                           doc.documentElement: doc.v___('documentElement');
                         pageX = 0, pageY = 0;
                         if (el === viewport) {} else if
                           (doc.getBoxObjectFor_v___? doc.getBoxObjectFor:
                           doc.v___('getBoxObjectFor')) {
                           elBoxObject = doc.getBoxObjectFor_m___?
                             doc.getBoxObjectFor(el):
                           doc.m___('getBoxObjectFor', [ el ]);
                           viewPortBoxObject = doc.getBoxObjectFor_m___?
                             doc.getBoxObjectFor(viewport):
                           doc.m___('getBoxObjectFor', [ viewport ]);
                           pageX = (elBoxObject.screenX_v___?
                             elBoxObject.screenX: elBoxObject.v___('screenX'))
                             - (viewPortBoxObject.screenX_v___?
                             viewPortBoxObject.screenX:
                             viewPortBoxObject.v___('screenX'));
                           pageY = (elBoxObject.screenY_v___?
                             elBoxObject.screenY: elBoxObject.v___('screenY'))
                             - (viewPortBoxObject.screenY_v___?
                             viewPortBoxObject.screenY:
                             viewPortBoxObject.v___('screenY'));
                         } else {
                           for (op = el; op && op !== el; op =
                             op.offsetParent_v___? op.offsetParent:
                             op.v___('offsetParent')) {
                             pageX = pageX + (op.offsetLeft_v___?
                               op.offsetLeft: op.v___('offsetLeft'));
                             pageY = pageY + (op.offsetTop_v___? op.offsetTop:
                               op.v___('offsetTop'));
                             if (op !== el) {
                               pageX = pageX + ((op.clientLeft_v___?
                                     op.clientLeft: op.v___('clientLeft')) ||
                                   0);
                               pageY = pageY + ((op.clientTop_v___?
                                   op.clientTop: op.v___('clientTop')) || 0);
                             }
                             if (isWebkit) {
                               opPosition = (x12___ = doc.defaultView_v___?
                                 doc.defaultView: doc.v___('defaultView'),
                                 x13___ = op, x12___.getComputedStyle_m___?
                                 x12___.getComputedStyle(x13___, 'position'):
                                 x12___.m___('getComputedStyle', [ x13___,
                                     'position' ]));
                               if (opPosition === 'fixed') {
                                 pageX = pageX + (x14___ = doc.body_v___?
                                   doc.body: doc.v___('body'),
                                   x14___.scrollLeft_v___? x14___.scrollLeft:
                                   x14___.v___('scrollLeft'));
                                 pageY = pageY + (x15___ = doc.body_v___?
                                   doc.body: doc.v___('body'),
                                   x15___.scrollTop_v___? x15___.scrollTop:
                                   x15___.v___('scrollTop'));
                               }
                               break;
                             }
                           }
                           if (isWebkit && (x16___ = doc.defaultView_v___?
                               doc.defaultView: doc.v___('defaultView'), x17___
                               = el, x16___.getComputedStyle_m___?
                               x16___.getComputedStyle(x17___, 'position'):
                               x16___.m___('getComputedStyle', [ x17___,
                                   'position' ])) === 'absolute' || isOpera) {
                             pageY = pageY - (x18___ = doc.body_v___? doc.body:
                               doc.v___('body'), x18___.offsetTop_v___?
                               x18___.offsetTop: x18___.v___('offsetTop'));
                           }
                           for (op = el; (op = op.offsetParent_v___?
                                 op.offsetParent: op.v___('offsetParent')) &&
                               op !== (doc.body_v___? doc.body:
                               doc.v___('body'));) {
                             pageX = pageX - (op.scrollLeft_v___?
                                 op.scrollLeft: op.v___('scrollLeft'));
                             if (!isOpera || (op.tagName_v___? op.tagName:
                                 op.v___('tagName')) !== 'TR') {
                               pageY = pageY - (op.scrollTop_v___?
                                 op.scrollTop: op.v___('scrollTop'));
                             }
                           }
                         }
                         scrollEl = !isWebkit && (doc.compatMode_v___?
                           doc.compatMode: doc.v___('compatMode')) ===
                           'CSS1Compat'? doc.documentElement_v___?
                           doc.documentElement: doc.v___('documentElement'):
                         doc.body_v___? doc.body: doc.v___('body');
                         left = pageX - (scrollEl.scrollLeft_v___?
                             scrollEl.scrollLeft: scrollEl.v___('scrollLeft')),
                         top = pageY - (scrollEl.scrollTop_v___?
                           scrollEl.scrollTop: scrollEl.v___('scrollTop'));
                         return ___.iM([ 'top', top, 'left', left, 'right',
                             left + (el.clientWidth_v___? el.clientWidth:
                               el.v___('clientWidth')), 'bottom', top +
                             (el.clientHeight_v___? el.clientHeight:
                              el.v___('clientHeight')) ]);
                       }
                       ___.f(getBoundingClientRect, 'getBoundingClientRect');
                     }
                     {
                       function getAttribute(element, name) {
                         var style, attr;
                         if (name === 'style') {
                           style = makeDOMAccessible.i___(element.style_v___?
                             element.style: element.v___('style'));
                           if (typeof (style.cssText_v___? style.cssText:
                               style.v___('cssText')) === 'string') {
                             return style.cssText_v___? style.cssText:
                             style.v___('cssText');
                           }
                         }
                         attr =
                           makeDOMAccessible.i___(element.getAttributeNode_m___?
                             element.getAttributeNode(name):
                           element.m___('getAttributeNode', [ name ]));
                         if (attr && (attr.specified_v___? attr.specified:
                             attr.v___('specified'))) {
                           return attr.value_v___? attr.value:
                           attr.v___('value');
                         } else { return null; }
                       }
                       ___.f(getAttribute, 'getAttribute');
                     }
                     {
                       function hasAttribute(element, name) {
                         var attr;
                         if (element.hasAttribute_v___? element.hasAttribute:
                           element.v___('hasAttribute')) {
                           return element.hasAttribute_m___?
                             element.hasAttribute(name):
                           element.m___('hasAttribute', [ name ]);
                         } else {
                           attr =
                             makeDOMAccessible.i___(element.getAttributeNode_m___?
                             element.getAttributeNode(name):
                             element.m___('getAttributeNode', [ name ]));
                           return attr !== null && (attr.specified_v___?
                               attr.specified: attr.v___('specified'));
                         }
                       }
                       ___.f(hasAttribute, 'hasAttribute');
                     }
                     {
                       function getComputedStyle(element, pseudoElement) {
                         var x0___;
                         if ((x0___ = makeDOMAccessible.i___(element),
                             x0___.currentStyle_v___? x0___.currentStyle:
                             x0___.v___('currentStyle')) && pseudoElement ===
                           void 0) {
                           return makeDOMAccessible.i___(element.currentStyle_v___?
                               element.currentStyle:
                             element.v___('currentStyle'));
                         } else if (window.getComputedStyle_v___?
                           window.getComputedStyle:
                           window.v___('getComputedStyle')) {
                           return makeDOMAccessible.i___(window.getComputedStyle_m___?
                               window.getComputedStyle(element, pseudoElement):
                             window.m___('getComputedStyle', [ element,
                                 pseudoElement ]));
                         } else {
                           throw new (IMPORTS___.Error_v___? IMPORTS___.Error:
                             ___.ri(IMPORTS___, 'Error'))
                             .new___(('Computed style not available for pseudo element '
                               + pseudoElement));
                         }
                       }
                       ___.f(getComputedStyle, 'getComputedStyle');
                     }
                     {
                       function makeXhr() {
                         var activeXClassIds, i, n, candidate;
                         if (typeof XMLHttpRequest === 'undefined') {
                           activeXClassIds = [ 'MSXML2.XMLHTTP.5.0',
                             'MSXML2.XMLHTTP.4.0', 'MSXML2.XMLHTTP.3.0',
                             'MSXML2.XMLHTTP', 'MICROSOFT.XMLHTTP.1.0',
                             'MICROSOFT.XMLHTTP.1', 'MICROSOFT.XMLHTTP' ];
                           for (i = 0, n = activeXClassIds.length; i < n; i++)
                             {
                             candidate = activeXClassIds[ +i ];
                             try {
                               return makeDOMAccessible.i___(new
                                 ActiveXObject.new___(candidate));
                             } catch (e) {}
                           }
                         }
                         return makeDOMAccessible.i___(new
                           XMLHttpRequest.new___);
                       }
                       ___.f(makeXhr, 'makeXhr');
                     }
                     document = makeDOMAccessible.i___(document);
                     docEl =
                       makeDOMAccessible.i___(document.documentElement_v___?
                         document.documentElement:
                       document.v___('documentElement'));
                     window = makeDOMAccessible.i___((x0___ =
                           IMPORTS___.bridalMaker_v___? IMPORTS___.bridalMaker:
                         ___.ri(IMPORTS___, 'bridalMaker'), x1___ = docEl,
                         x2___ = makeDOMAccessible, x0___.getWindow_m___?
                           x0___.getWindow(x1___, x2___):
                         x0___.m___('getWindow', [ x1___, x2___ ])));
                     navigator = makeDOMAccessible.i___(window.navigator_v___?
                       window.navigator: window.v___('navigator'));
                     XMLHttpRequest =
                       makeDOMAccessible.i___(window.XMLHttpRequest_v___?
                       window.XMLHttpRequest: window.v___('XMLHttpRequest'));
                     ActiveXObject =
                       makeDOMAccessible.i___(window.ActiveXObject_v___?
                       window.ActiveXObject: window.v___('ActiveXObject'));
                     isOpera = (x3___ = navigator.userAgent_v___?
                       navigator.userAgent: navigator.v___('userAgent'),
                       x3___.indexOf_m___? x3___.indexOf('Opera'):
                       x3___.m___('indexOf', [ 'Opera' ])) === 0;
                     isIE = !isOpera && (x4___ = navigator.userAgent_v___?
                       navigator.userAgent: navigator.v___('userAgent'),
                       x4___.indexOf_m___? x4___.indexOf('MSIE'):
                       x4___.m___('indexOf', [ 'MSIE' ])) !== -1;
                     isWebkit = !isOpera && (x5___ = navigator.userAgent_v___?
                       navigator.userAgent: navigator.v___('userAgent'),
                       x5___.indexOf_m___? x5___.indexOf('WebKit'):
                       x5___.m___('indexOf', [ 'WebKit' ])) !== -1;
                     featureAttachEvent = ! ! ((window.attachEvent_v___?
                           window.attachEvent: window.v___('attachEvent')) && !
                         (window.addEventListener_v___?
                           window.addEventListener:
                         window.v___('addEventListener')));
                     featureExtendedCreateElement = ___.f(function () {
                         var x0___;
                         try {
                           return (x0___ = document.createElement_m___?
                             document.createElement('\x3cinput type=\"radio\"\x3e')
                             : document.m___('createElement', [
                                 '\x3cinput type=\"radio\"\x3e' ]),
                             x0___.type_v___? x0___.type: x0___.v___('type'))
                             === 'radio';
                         } catch (e) { return false; }
                       }).i___();
                     if (isIE) { html5shim.i___(); }
                     CUSTOM_EVENT_TYPE_SUFFIX = '_custom___';
                     endsWithUnderbars = new RegExp.new___('__$');
                     escapeAttrib = (x6___ = IMPORTS___.html_v___?
                         IMPORTS___.html: ___.ri(IMPORTS___, 'html'),
                       x6___.escapeAttrib_v___? x6___.escapeAttrib:
                       x6___.v___('escapeAttrib'));
                     return ___.iM([ 'addEventListener', addEventListener,
                         'removeEventListener', removeEventListener,
                         'initEvent', initEvent, 'dispatchEvent',
                         dispatchEvent, 'cloneNode', cloneNode,
                         'createElement', createElement, 'createStylesheet',
                         createStylesheet, 'setAttribute', setAttribute,
                         'getAttribute', getAttribute, 'hasAttribute',
                         hasAttribute, 'getBoundingClientRect',
                         getBoundingClientRect, 'untameEventType',
                         untameEventType, 'extendedCreateElementFeature',
                         featureExtendedCreateElement, 'getComputedStyle',
                         getComputedStyle, 'makeXhr', makeXhr,
                         'initCanvasElement', initCanvasElement,
                         'initCanvasElements', initCanvasElements ]);
                   }
                   return ___.f(bridalMaker$_var, 'bridalMaker$_var');
                 })());
              x0___ = IMPORTS___.bridalMaker_v___? IMPORTS___.bridalMaker:
              ___.ri(IMPORTS___, 'bridalMaker'), x1___ = (function () {
                  function getWindow$_meth(element, mda) {
                    var doc, x0___, s, body;
                    doc = mda.i___((x0___ = mda.i___(element),
                        x0___.ownerDocument_v___? x0___.ownerDocument:
                        x0___.v___('ownerDocument')));
                    if (doc.parentWindow_v___? doc.parentWindow:
                      doc.v___('parentWindow')) {
                      return doc.parentWindow_v___? doc.parentWindow:
                      doc.v___('parentWindow');
                    }
                    if (doc.defaultView_v___? doc.defaultView:
                      doc.v___('defaultView')) {
                      return doc.defaultView_v___? doc.defaultView:
                      doc.v___('defaultView');
                    }
                    s = doc.createElement_m___? doc.createElement('script'):
                    doc.m___('createElement', [ 'script' ]);
                    s.innerHTML_w___ === s? (s.innerHTML =
                      'document.parentWindow = window;'): s.w___('innerHTML',
                      'document.parentWindow = window;');
                    body = mda.i___(doc.body_v___? doc.body: doc.v___('body'));
                    body.appendChild_m___? body.appendChild(s):
                    body.m___('appendChild', [ s ]);
                    body.removeChild_m___? body.removeChild(s):
                    body.m___('removeChild', [ s ]);
                    return doc.parentWindow_v___? doc.parentWindow:
                    doc.v___('parentWindow');
                  }
                  return ___.f(getWindow$_meth, 'getWindow$_meth');
                })(), x0___.getWindow_w___ === x0___? (x0___.getWindow = x1___)
                : x0___.w___('getWindow', x1___);
              if (typeof IMPORTS___.v___('window') !== 'undefined') {
                moduleResult___ = (IMPORTS___.window_v___? IMPORTS___.window:
                  ___.ri(IMPORTS___, 'window')).w___('bridalMaker',
                  IMPORTS___.bridalMaker_v___? IMPORTS___.bridalMaker:
                  ___.ri(IMPORTS___, 'bridalMaker'));
              }
              return moduleResult___;
            },
            'cajolerName': 'com.google.caja',
            'cajolerVersion': '5054',
            'cajoledDate': 1347987483907
          }).instantiate___(___, IMPORTS___), ___.prepareModule({
            'instantiate': function (___, IMPORTS___) {
              var dis___ = IMPORTS___;
              var moduleResult___;
              moduleResult___ = ___.NO_RESULT;
              if (('I'.toLowerCase_m___? 'I'.toLowerCase():
                  'I'.m___('toLowerCase', [ ])) !== 'i') {
                throw 'I/i problem';
              }
              ___.di(IMPORTS___, 'domitaModules');
              ___.di(IMPORTS___, 'Domado'), IMPORTS___.w___('Domado',
                ___.f(function () {
                    var isVirtualizedElementName, x0___,
                    realToVirtualElementName, x1___, virtualToRealElementName,
                    x2___, cajaPrefix, cajaPrefRe, URI_SCHEME_RE,
                    ALLOWED_URI_SCHEMES, allowedUriScheme, uriFetch,
                    uriRewrite, x3___, x4___, x5___, x6___, x7___, x8___,
                    x9___, x10___, x11___, x12___, x13___, x14___, x15___,
                    x16___, x17___, x18___, x19___, x20___, x21___, x22___,
                    x23___, x24___, x25___, x26___, x27___, x28___, x29___,
                    x30___, x31___, x32___, x33___, x34___, x35___, x36___,
                    x37___, x38___, x39___, x40___;
                    {
                      function allowedUriScheme(uri) {
                        var parsed, x0___, x1___, x2___, x3___;
                        parsed = (x0___ = '' + uri, x1___ = URI_SCHEME_RE,
                          x0___.match_m___? x0___.match(x1___):
                          x0___.m___('match', [ x1___ ]));
                        return parsed && (!parsed[ +1 ] || (x3___ =
                              ALLOWED_URI_SCHEMES, x2___ = parsed[ +1 ],
                            x3___.test_m___? x3___.test(x2___):
                            x3___.m___('test', [ x2___ ])));
                      }
                      ___.f(allowedUriScheme, 'allowedUriScheme');
                    }
                    {
                      function uriFetch(naiveUriPolicy, uri, mime, callback) {
                        var x0___, x1___, x2___, x3___;
                        if (!naiveUriPolicy || !callback || 'function' !==
                          typeof (naiveUriPolicy.fetch_v___?
                              naiveUriPolicy.fetch:
                            naiveUriPolicy.v___('fetch'))) { return; }
                        if (allowedUriScheme.i___(uri)) {
                          naiveUriPolicy.fetch_m___? naiveUriPolicy.fetch(uri,
                            mime, callback): naiveUriPolicy.m___('fetch', [
                              uri, mime, callback ]);
                        } else {
                          x1___ = naiveUriPolicy, x0___ =
                            IMPORTS___.undefined_v___? IMPORTS___.undefined:
                          ___.ri(IMPORTS___, 'undefined'), x2___ = mime, x3___
                            = callback, x1___.fetch_m___? x1___.fetch(x0___,
                            x2___, x3___): x1___.m___('fetch', [ x0___, x2___,
                              x3___ ]);
                        }
                      }
                      ___.f(uriFetch, 'uriFetch');
                    }
                    {
                      function uriRewrite(naiveUriPolicy, uri, effects, ltype,
                        hints) {
                        if (!naiveUriPolicy) { return null; }
                        return allowedUriScheme.i___(uri) ? 'function' ===
                          typeof (naiveUriPolicy.rewrite_v___?
                          naiveUriPolicy.rewrite:
                          naiveUriPolicy.v___('rewrite')) ?
                          naiveUriPolicy.rewrite_m___?
                          naiveUriPolicy.rewrite(uri, effects, ltype, hints):
                        naiveUriPolicy.m___('rewrite', [ uri, effects, ltype,
                            hints ]): null: null;
                      }
                      ___.f(uriRewrite, 'uriRewrite');
                    }
                    isVirtualizedElementName = (x0___ = IMPORTS___.html_v___?
                      IMPORTS___.html: ___.ri(IMPORTS___, 'html'),
                      x0___.isVirtualizedElementName_v___?
                      x0___.isVirtualizedElementName:
                      x0___.v___('isVirtualizedElementName'));
                    realToVirtualElementName = (x1___ = IMPORTS___.html_v___?
                      IMPORTS___.html: ___.ri(IMPORTS___, 'html'),
                      x1___.realToVirtualElementName_v___?
                      x1___.realToVirtualElementName:
                      x1___.v___('realToVirtualElementName'));
                    virtualToRealElementName = (x2___ = IMPORTS___.html_v___?
                      IMPORTS___.html: ___.ri(IMPORTS___, 'html'),
                      x2___.virtualToRealElementName_v___?
                      x2___.virtualToRealElementName:
                      x2___.v___('virtualToRealElementName'));
                    cajaPrefix = 'data-caja-';
                    cajaPrefRe = new (IMPORTS___.RegExp_v___?
                      IMPORTS___.RegExp: ___.ri(IMPORTS___, 'RegExp'))
                      .new___(('^' + cajaPrefix));
                    URI_SCHEME_RE = new (IMPORTS___.RegExp_v___?
                        IMPORTS___.RegExp: ___.ri(IMPORTS___, 'RegExp'))
                      .new___(('^' + '(?:' + '([^:/?# ]+)' + ':)?'));
                    ALLOWED_URI_SCHEMES = new
                      RegExp.new___('^(?:https?|mailto)$', 'i');
                    if (! (IMPORTS___.domitaModules_v___?
                          IMPORTS___.domitaModules: ___.ri(IMPORTS___,
                          'domitaModules'))) {
                      IMPORTS___.domitaModules_w___? (IMPORTS___.domitaModules
                          = ___.iM([ ])): ___.wi(IMPORTS___, 'domitaModules',
                        ___.iM([ ]));
                    }
                    x3___ = IMPORTS___.domitaModules_v___?
                      IMPORTS___.domitaModules: ___.ri(IMPORTS___,
                      'domitaModules'), x4___ = typeof IMPORTS___.v___('Proxy')
                      !== 'undefined', x3___.proxiesAvailable_w___ === x3___?
                      (x3___.proxiesAvailable = x4___):
                    x3___.w___('proxiesAvailable', x4___);
                    x5___ = IMPORTS___.domitaModules_v___?
                      IMPORTS___.domitaModules: ___.ri(IMPORTS___,
                      'domitaModules'), x6___ = ___.f(function () {
                        var needToSwap, x0___, testHandler, proxy, x1___,
                        x2___;
                        needToSwap = false;
                        if (x0___ = IMPORTS___.domitaModules_v___?
                          IMPORTS___.domitaModules: ___.ri(IMPORTS___,
                            'domitaModules'), x0___.proxiesAvailable_v___?
                          x0___.proxiesAvailable:
                          x0___.v___('proxiesAvailable')) {
                          testHandler = ___.iM([ 'set', (function () {
                                  function set$_lit(a, b, c) {
                                    {
                                      ___.deodorize(arguments, -6);
                                      var a___ = ___.args(arguments);
                                    }
                                    var x0___, x1___, x2___, x3___, x4___,
                                    x5___, x6___, x7___;
                                    if (a === proxy && b === 'foo' && c === 1)
                                      { needToSwap = true; } else if (a ===
                                      'foo' && b === 1 && c === proxy) {} else
                                      if (a === 'foo' && b === 1 && c ===
                                      (IMPORTS___.undefined_v___?
                                       IMPORTS___.undefined: ___.ri(IMPORTS___,
                                         'undefined'))) {
                                      throw new (IMPORTS___.Error_v___?
                                        IMPORTS___.Error: ___.ri(IMPORTS___,
                                          'Error'))
                                        .new___(('Proxy implementation does not provide proxy '
                                          + 'parameter: ' + (x2___ = (x1___ =
                                              (x0___ = IMPORTS___.Array_v___?
                                               IMPORTS___.Array:
                                               ___.ri(IMPORTS___, 'Array'),
                                               x0___.prototype_v___?
                                               x0___.prototype:
                                               x0___.v___('prototype')),
                                              x1___.slice_v___? x1___.slice:
                                              x1___.v___('slice')), x3___ =
                                            a___, x2___.call_m___?
                                            x2___.call(x3___, 0):
                                            x2___.m___('call', [ x3___, 0 ]))))
                                        ;
                                    } else {
                                      throw new (IMPORTS___.Error_v___?
                                        IMPORTS___.Error: ___.ri(IMPORTS___,
                                          'Error'))
                                        .new___(('internal: Failed to understand proxy arguments: '
                                          + (x6___ = (x5___ = (x4___ =
                                                IMPORTS___.Array_v___?
                                                IMPORTS___.Array:
                                                ___.ri(IMPORTS___, 'Array'),
                                                x4___.prototype_v___?
                                                x4___.prototype:
                                                x4___.v___('prototype')),
                                              x5___.slice_v___? x5___.slice:
                                              x5___.v___('slice')), x7___ =
                                            a___, x6___.call_m___?
                                            x6___.call(x7___, 0):
                                            x6___.m___('call', [ x7___, 0 ]))))
                                        ;
                                    }
                                    return true;
                                  }
                                  return ___.f(set$_lit, 'set$_lit');
                                })() ]);
                          proxy = (x1___ = IMPORTS___.Proxy_v___?
                              IMPORTS___.Proxy: ___.ri(IMPORTS___, 'Proxy'),
                            x2___ = testHandler, x1___.create_m___?
                              x1___.create(x2___): x1___.m___('create', [ x2___
                              ]));
                          proxy.foo_w___ === proxy? (proxy.foo = 1):
                          proxy.w___('foo', 1);
                        }
                        if (needToSwap) {
                          return ___.iM([ 'getter', (function () {
                                  function getter$_lit(getFunc) {
                                    var permutedGetter;
                                    {
                                      function permutedGetter(proxy, name) {
                                        var dis___ = this && this.___? void 0:
                                        this;
                                        return getFunc.call_m___?
                                          getFunc.call(dis___, name, proxy):
                                        getFunc.m___('call', [ dis___, name,
                                            proxy ]);
                                      }
                                      ___.f(permutedGetter, 'permutedGetter');
                                    }
                                    permutedGetter.unpermuted_w___ ===
                                      permutedGetter?
                                      (permutedGetter.unpermuted = getFunc):
                                    permutedGetter.w___('unpermuted', getFunc);
                                    return permutedGetter;
                                  }
                                  return ___.f(getter$_lit, 'getter$_lit');
                                })(), 'setter', (function () {
                                  function setter$_lit(setFunc) {
                                    var permutedSetter;
                                    {
                                      function permutedSetter(proxy, name,
                                        value) {
                                        var dis___ = this && this.___? void 0:
                                        this;
                                        return setFunc.call_m___?
                                          setFunc.call(dis___, name, value,
                                          proxy): setFunc.m___('call', [
                                            dis___, name, value, proxy ]);
                                      }
                                      ___.f(permutedSetter, 'permutedSetter');
                                    }
                                    permutedSetter.unpermuted_w___ ===
                                      permutedSetter?
                                      (permutedSetter.unpermuted = setFunc):
                                    permutedSetter.w___('unpermuted', setFunc);
                                    return permutedSetter;
                                  }
                                  return ___.f(setter$_lit, 'setter$_lit');
                                })() ]);
                        } else {
                          return ___.iM([ 'getter', (function () {
                                  function getter$_lit(getFunc) {
                                    getFunc.unpermuted_w___ === getFunc?
                                      (getFunc.unpermuted = getFunc):
                                    getFunc.w___('unpermuted', getFunc);
                                    return getFunc;
                                  }
                                  return ___.f(getter$_lit, 'getter$_lit');
                                })(), 'setter', (function () {
                                  function setter$_lit(setFunc) {
                                    setFunc.unpermuted_w___ === setFunc?
                                      (setFunc.unpermuted = setFunc):
                                    setFunc.w___('unpermuted', setFunc);
                                    return setFunc;
                                  }
                                  return ___.f(setter$_lit, 'setter$_lit');
                                })() ]);
                        }
                      }).i___(), x5___.permuteProxyGetSet_w___ === x5___?
                      (x5___.permuteProxyGetSet = x6___):
                    x5___.w___('permuteProxyGetSet', x6___);
                    x7___ = IMPORTS___.domitaModules_v___?
                      IMPORTS___.domitaModules: ___.ri(IMPORTS___,
                      'domitaModules'), x8___ = ___.f(function () {
                        var x0___, x1___, x2___;
                        try {
                          x0___ = IMPORTS___.Object_v___? IMPORTS___.Object:
                          ___.ri(IMPORTS___, 'Object'), x1___ = ___.iM([ ]),
                          x2___ = ___.iM([ 'enumerable', true, 'configurable',
                              false, 'get', (function () {
                                  function get$_lit() {}
                                  return ___.f(get$_lit, 'get$_lit');
                                })() ]), x0___.defineProperty_m___?
                            x0___.defineProperty(x1___, 'foo', x2___):
                          x0___.m___('defineProperty', [ x1___, 'foo', x2___ ])
                            ;
                          return true;
                        } catch (e) { return false; }
                      }).i___(), x7___.canHaveEnumerableAccessors_w___ ===
                      x7___? (x7___.canHaveEnumerableAccessors = x8___):
                    x7___.w___('canHaveEnumerableAccessors', x8___);
                    x9___ = IMPORTS___.domitaModules_v___?
                      IMPORTS___.domitaModules: ___.ri(IMPORTS___,
                      'domitaModules'), x10___ = (function () {
                        function getPropertyDescriptor$_meth(o, n) {
                          var x0___, x1___, x2___, x3___, x4___, x5___, x6___,
                          x7___;
                          if (o === null || o === (IMPORTS___.undefined_v___?
                              IMPORTS___.undefined: ___.ri(IMPORTS___,
                                'undefined'))) {
                            return IMPORTS___.undefined_v___?
                              IMPORTS___.undefined: ___.ri(IMPORTS___,
                              'undefined');
                          } else {
                            return (x0___ = IMPORTS___.Object_v___?
                              IMPORTS___.Object: ___.ri(IMPORTS___, 'Object'),
                              x1___ = o, x2___ = n,
                              x0___.getOwnPropertyDescriptor_m___?
                              x0___.getOwnPropertyDescriptor(x1___, x2___):
                              x0___.m___('getOwnPropertyDescriptor', [ x1___,
                                  x2___ ])) || (x3___ =
                              IMPORTS___.domitaModules_v___?
                              IMPORTS___.domitaModules: ___.ri(IMPORTS___,
                                'domitaModules'), x6___ = (x4___ =
                                  IMPORTS___.Object_v___? IMPORTS___.Object:
                                ___.ri(IMPORTS___, 'Object'), x5___ = o,
                                x4___.getPrototypeOf_m___?
                                  x4___.getPrototypeOf(x5___):
                                x4___.m___('getPrototypeOf', [ x5___ ])), x7___
                              = n, x3___.getPropertyDescriptor_m___?
                              x3___.getPropertyDescriptor(x6___, x7___):
                              x3___.m___('getPropertyDescriptor', [ x6___,
                                  x7___ ]));
                          }
                        }
                        return ___.f(getPropertyDescriptor$_meth,
                          'getPropertyDescriptor$_meth');
                      })(), x9___.getPropertyDescriptor_w___ === x9___?
                      (x9___.getPropertyDescriptor = x10___):
                    x9___.w___('getPropertyDescriptor', x10___);
                    x11___ = IMPORTS___.domitaModules_v___?
                      IMPORTS___.domitaModules: ___.ri(IMPORTS___,
                      'domitaModules'), x12___ = (function () {
                        function ProxyHandler$_meth(target) {
                          var dis___ = this && this.___? void 0: this;
                          dis___.target_w___ === dis___? (dis___.target =
                              target): dis___.w___('target', target);
                        }
                        return ___.f(ProxyHandler$_meth, 'ProxyHandler$_meth');
                      })(), x11___.ProxyHandler_w___ === x11___?
                      (x11___.ProxyHandler = x12___):
                    x11___.w___('ProxyHandler', x12___);
                    x14___ = (x13___ = IMPORTS___.domitaModules_v___?
                      IMPORTS___.domitaModules: ___.ri(IMPORTS___,
                        'domitaModules'), x13___.ProxyHandler_v___?
                      x13___.ProxyHandler: x13___.v___('ProxyHandler')), x21___
                      = ___.iM([ 'getOwnPropertyDescriptor', (function () {
                            function getOwnPropertyDescriptor$_lit(name) {
                              var dis___ = this && this.___? void 0: this;
                              var desc, x0___, x1___, x2___;
                              desc = (x0___ = IMPORTS___.Object_v___?
                                  IMPORTS___.Object: ___.ri(IMPORTS___,
                                  'Object'), x1___ = dis___.target_v___?
                                  dis___.target: dis___.v___('target'), x2___ =
                                  name, x0___.getOwnPropertyDescriptor_m___?
                                  x0___.getOwnPropertyDescriptor(x1___, x2___):
                                x0___.m___('getOwnPropertyDescriptor', [ x1___,
                                    x2___ ]));
                              if (desc !== (IMPORTS___.undefined_v___?
                                  IMPORTS___.undefined: ___.ri(IMPORTS___,
                                    'undefined'))) {
                                desc.configurable_w___ === desc?
                                  (desc.configurable = true):
                                desc.w___('configurable', true);
                              }
                              return desc;
                            }
                            return ___.f(getOwnPropertyDescriptor$_lit,
                              'getOwnPropertyDescriptor$_lit');
                          })(), 'getPropertyDescriptor', (function () {
                            function getPropertyDescriptor$_lit(name) {
                              var dis___ = this && this.___? void 0: this;
                              var desc, x0___, x1___, x2___;
                              desc = (x0___ = IMPORTS___.Object_v___?
                                  IMPORTS___.Object: ___.ri(IMPORTS___,
                                  'Object'), x1___ = dis___.target_v___?
                                  dis___.target: dis___.v___('target'), x2___ =
                                  name, x0___.getPropertyDescriptor_m___?
                                  x0___.getPropertyDescriptor(x1___, x2___):
                                x0___.m___('getPropertyDescriptor', [ x1___,
                                    x2___ ]));
                              if (desc !== (IMPORTS___.undefined_v___?
                                    IMPORTS___.undefined: ___.ri(IMPORTS___,
                                    'undefined'))) {
                                desc.configurable_w___ === desc?
                                  (desc.configurable = true):
                                desc.w___('configurable', true);
                              }
                              return desc;
                            }
                            return ___.f(getPropertyDescriptor$_lit,
                              'getPropertyDescriptor$_lit');
                          })(), 'getOwnPropertyNames', (function () {
                            function getOwnPropertyNames$_lit() {
                              var dis___ = this && this.___? void 0: this;
                              var x0___, x1___;
                              return x0___ = IMPORTS___.Object_v___?
                                IMPORTS___.Object: ___.ri(IMPORTS___, 'Object')
                                , x1___ = dis___.target_v___? dis___.target:
                              dis___.v___('target'),
                              x0___.getOwnPropertyNames_m___?
                                x0___.getOwnPropertyNames(x1___):
                              x0___.m___('getOwnPropertyNames', [ x1___ ]);
                            }
                            return ___.f(getOwnPropertyNames$_lit,
                              'getOwnPropertyNames$_lit');
                          })(), 'getPropertyNames', (function () {
                            function getPropertyNames$_lit() {
                              var dis___ = this && this.___? void 0: this;
                              var x0___, x1___;
                              return x0___ = IMPORTS___.Object_v___?
                                IMPORTS___.Object: ___.ri(IMPORTS___, 'Object')
                                , x1___ = dis___.target_v___? dis___.target:
                              dis___.v___('target'),
                              x0___.getPropertyNames_m___?
                                x0___.getPropertyNames(x1___):
                              x0___.m___('getPropertyNames', [ x1___ ]);
                            }
                            return ___.f(getPropertyNames$_lit,
                              'getPropertyNames$_lit');
                          })(), 'defineProperty', (function () {
                            function defineProperty$_lit(name, desc) {
                              var dis___ = this && this.___? void 0: this;
                              var x0___, x1___, x2___, x3___;
                              return x0___ = IMPORTS___.Object_v___?
                                IMPORTS___.Object: ___.ri(IMPORTS___, 'Object')
                                , x1___ = dis___.target_v___? dis___.target:
                              dis___.v___('target'), x2___ = name, x3___ =
                                desc, x0___.defineProperty_m___?
                                x0___.defineProperty(x1___, x2___, x3___):
                              x0___.m___('defineProperty', [ x1___, x2___,
                                  x3___ ]);
                            }
                            return ___.f(defineProperty$_lit,
                              'defineProperty$_lit');
                          })(), 'delete', (function () {
                            function delete$_lit(name) {
                              var dis___ = this && this.___? void 0: this;
                              return (dis___.target_v___? dis___.target:
                                dis___.v___('target')).c___(name);
                            }
                            return ___.f(delete$_lit, 'delete$_lit');
                          })(), 'fix', (function () {
                            function fix$_lit() {
                              var dis___ = this && this.___? void 0: this;
                              var x0___, x1___, props, name, x2___, x3___,
                              x4___, x5___, x6___, x7___;;
                              if (! (x0___ = IMPORTS___.Object_v___?
                                  IMPORTS___.Object: ___.ri(IMPORTS___,
                                    'Object'), x1___ = dis___.target_v___?
                                  dis___.target: dis___.v___('target'),
                                  x0___.isFrozen_m___? x0___.isFrozen(x1___):
                                  x0___.m___('isFrozen', [ x1___ ]))) {
                                return IMPORTS___.undefined_v___?
                                  IMPORTS___.undefined: ___.ri(IMPORTS___,
                                  'undefined');
                              }
                              props = ___.iM([ ]);
                              {
                                x4___ = Object(dis___.target_v___?
                                  dis___.target: dis___.v___('target')).e___();
                                for (x3___ in x4___) {
                                  if (typeof x3___ === 'number' || '' + +x3___
                                    === x3___) { name = x3___; } else {
                                    if (/^NUM___/.test(x3___) &&
                                        /__$/.test(x3___)) { continue; }
                                    x2___ = x3___.match(/([\s\S]*)_e___$/);
                                    if (!x2___ || !x4___[ x3___ ]) { continue;
                                    }
                                    name = x2___[ 1 ];
                                  }
                                  {
                                    props.w___(name, (x5___ =
                                          IMPORTS___.Object_v___?
                                          IMPORTS___.Object: ___.ri(IMPORTS___,
                                          'Object'), x6___ =
                                          dis___.target_v___? dis___.target:
                                        dis___.v___('target'), x7___ = name,
                                        x5___.getOwnPropertyDescriptor_m___?
                                          x5___.getOwnPropertyDescriptor(x6___,
                                          x7___):
                                        x5___.m___('getOwnPropertyDescriptor',
                                          [ x6___, x7___ ])));
                                  }
                                }
                              }
                              return props;
                            }
                            return ___.f(fix$_lit, 'fix$_lit');
                          })(), 'has', (function () {
                            function has$_lit(name) {
                              var dis___ = this && this.___? void 0: this;
                              return ___.i('' + name, dis___.target_v___?
                                  dis___.target: dis___.v___('target'));
                            }
                            return ___.f(has$_lit, 'has$_lit');
                          })(), 'hasOwn', (function () {
                            function hasOwn$_lit(name) {
                              var dis___ = this && this.___? void 0: this;
                              var x0___, x1___, x2___, x3___;
                              return x1___ = (x0___ = ___.iM([ ]),
                                x0___.hasOwnProperty_v___?
                                x0___.hasOwnProperty:
                                x0___.v___('hasOwnProperty')), x2___ =
                                dis___.target_v___? dis___.target:
                              dis___.v___('target'), x3___ = name,
                              x1___.call_m___? x1___.call(x2___, x3___):
                              x1___.m___('call', [ x2___, x3___ ]);
                            }
                            return ___.f(hasOwn$_lit, 'hasOwn$_lit');
                          })(), 'get', (x16___ = (x15___ =
                            IMPORTS___.domitaModules_v___?
                            IMPORTS___.domitaModules: ___.ri(IMPORTS___,
                              'domitaModules'), x15___.permuteProxyGetSet_v___?
                            x15___.permuteProxyGetSet:
                            x15___.v___('permuteProxyGetSet')), x17___ =
                          ___.f(function (name, proxy) {
                              var dis___ = this && this.___? void 0: this;
                              return (dis___.target_v___? dis___.target:
                                dis___.v___('target')).v___(name);
                            }), x16___.getter_m___? x16___.getter(x17___):
                          x16___.m___('getter', [ x17___ ])), 'set', (x19___ =
                            (x18___ = IMPORTS___.domitaModules_v___?
                              IMPORTS___.domitaModules: ___.ri(IMPORTS___,
                              'domitaModules'), x18___.permuteProxyGetSet_v___?
                              x18___.permuteProxyGetSet:
                            x18___.v___('permuteProxyGetSet')), x20___ =
                            ___.f(function (name, value, proxy) {
                              var dis___ = this && this.___? void 0: this;
                              (dis___.target_v___? dis___.target:
                               dis___.v___('target')).w___(name, value);
                            }), x19___.setter_m___? x19___.setter(x20___):
                          x19___.m___('setter', [ x20___ ])), 'enumerate',
                        (function () {
                           function enumerate$_lit() {
                             var dis___ = this && this.___? void 0: this;
                             var result, name, x0___, x1___, x2___;;
                             result = [ ];
                             {
                               x2___ = Object(dis___.target_v___?
                                   dis___.target: dis___.v___('target')).e___()
                                 ;
                               for (x1___ in x2___) {
                                 if (typeof x1___ === 'number' || '' + +x1___
                                     === x1___) { name = x1___; } else {
                                   if (/^NUM___/.test(x1___) &&
                                       /__$/.test(x1___)) { continue; }
                                   x0___ = x1___.match(/([\s\S]*)_e___$/);
                                   if (!x0___ || !x2___[ x1___ ]) { continue; }
                                   name = x0___[ 1 ];
                                 }
                                 {
                                   result.push_m___? result.push(name):
                                   result.m___('push', [ name ]);
                                 }
                               }
                             }
                             return result;
                           }
                           return ___.f(enumerate$_lit, 'enumerate$_lit');
                         })(), 'keys', (function () {
                            function keys$_lit() {
                              var dis___ = this && this.___? void 0: this;
                              var x0___, x1___;
                              return x0___ = IMPORTS___.Object_v___?
                                IMPORTS___.Object: ___.ri(IMPORTS___, 'Object')
                                , x1___ = dis___.target_v___? dis___.target:
                              dis___.v___('target'), x0___.keys_m___?
                                x0___.keys(x1___): x0___.m___('keys', [ x1___ ]
                              );
                            }
                            return ___.f(keys$_lit, 'keys$_lit');
                          })() ]), x14___.prototype_w___ === x14___?
                      (x14___.prototype = x21___): x14___.w___('prototype',
                      x21___);
                    x22___ = IMPORTS___.cajaVM_v___? IMPORTS___.cajaVM:
                    ___.ri(IMPORTS___, 'cajaVM'), x24___ = (x23___ =
                      IMPORTS___.domitaModules_v___? IMPORTS___.domitaModules:
                      ___.ri(IMPORTS___, 'domitaModules'),
                      x23___.ProxyHandler_v___? x23___.ProxyHandler:
                      x23___.v___('ProxyHandler')), x22___.def_m___?
                      x22___.def(x24___): x22___.m___('def', [ x24___ ]);
                    x25___ = IMPORTS___.domitaModules_v___?
                      IMPORTS___.domitaModules: ___.ri(IMPORTS___,
                      'domitaModules'), x26___ = (function () {
                        function setOwn$_meth(object, propName, value) {
                          var desc, x0___, x1___, x2___, x3___, x4___, x5___,
                          x6___;
                          propName = propName + '';
                          desc = (x0___ = IMPORTS___.domitaModules_v___?
                            IMPORTS___.domitaModules: ___.ri(IMPORTS___,
                              'domitaModules'), x1___ = object, x2___ =
                            propName, x0___.getPropertyDescriptor_m___?
                            x0___.getPropertyDescriptor(x1___, x2___):
                            x0___.m___('getPropertyDescriptor', [ x1___, x2___
                              ]));
                          x3___ = IMPORTS___.Object_v___? IMPORTS___.Object:
                          ___.ri(IMPORTS___, 'Object'), x5___ = object, x6___ =
                            propName, x4___ = ___.iM([ 'enumerable', desc?
                              desc.enumerable_v___? desc.enumerable:
                              desc.v___('enumerable'): false, 'value', value ])
                            , x3___.defineProperty_m___?
                            x3___.defineProperty(x5___, x6___, x4___):
                          x3___.m___('defineProperty', [ x5___, x6___, x4___ ])
                            ;
                        }
                        return ___.f(setOwn$_meth, 'setOwn$_meth');
                      })(), x25___.setOwn_w___ === x25___? (x25___.setOwn =
                        x26___): x25___.w___('setOwn', x26___);
                    x27___ = IMPORTS___.domitaModules_v___?
                      IMPORTS___.domitaModules: ___.ri(IMPORTS___,
                      'domitaModules'), x28___ = (function () {
                        function ensureValidCallback$_meth(aCallback) {
                          if ('function' !== typeof aCallback) {
                            throw new (IMPORTS___.Error_v___? IMPORTS___.Error:
                              ___.ri(IMPORTS___, 'Error'))
                              .new___(('Expected function not ' + typeof
                                  aCallback));
                          }
                        }
                        return ___.f(ensureValidCallback$_meth,
                          'ensureValidCallback$_meth');
                      })(), x27___.ensureValidCallback_w___ === x27___?
                      (x27___.ensureValidCallback = x28___):
                    x27___.w___('ensureValidCallback', x28___);
                    x29___ = IMPORTS___.domitaModules_v___?
                      IMPORTS___.domitaModules: ___.ri(IMPORTS___,
                      'domitaModules'), x30___ = ___.f(function () {
                        var setOwn, x0___, Confidence, x1___, x2___, x3___,
                        x4___;
                        {
                          function Confidence(typename) {
                            var dis___ = this && this.___? void 0: this;
                            var table, x0___, x1___, x2___, x3___, guard,
                            x4___, x5___, x6___, x7___, x8___, x9___, x10___,
                            x11___, x12___, x13___, x14___, x15___;
                            table = new (IMPORTS___.WeakMap_v___?
                              IMPORTS___.WeakMap: ___.ri(IMPORTS___, 'WeakMap')
                            ).new___;
                            x3___ = dis___, x2___ = (x0___ =
                                IMPORTS___.cajaVM_v___? IMPORTS___.cajaVM:
                              ___.ri(IMPORTS___, 'cajaVM'), x1___ =
                                ___.f(function (object, opt_sameAs) {
                                  var dis___ = this && this.___? void 0: this;
                                  var x0___, x1___, privates, proto, x2___,
                                  x3___;
                                  if ((table.get_m___? table.get(object):
                                      table.m___('get', [ object ])) !==
                                    (IMPORTS___.undefined_v___?
                                     IMPORTS___.undefined: ___.ri(IMPORTS___,
                                       'undefined'))) {
                                    if ((x0___ = table.get_m___?
                                        table.get(object): table.m___('get', [
                                            object ]), x0___._obj_v___?
                                        x0___._obj: x0___.v___('_obj')) !==
                                      object) {
                                      throw new (IMPORTS___.Error_v___?
                                        IMPORTS___.Error: ___.ri(IMPORTS___,
                                          'Error')).new___(('WeakMap broke! ' +
                                          object + ' vs. ' + (x1___ =
                                            table.get_m___? table.get(object):
                                            table.m___('get', [ object ]),
                                            x1___._obj_v___? x1___._obj:
                                            x1___.v___('_obj'))));
                                    }
                                    throw new (IMPORTS___.Error_v___?
                                        IMPORTS___.Error: ___.ri(IMPORTS___,
                                        'Error')).new___((typename +
                                          ' has already confided in ' + object)
                                    );
                                  }
                                  proto = (x2___ = IMPORTS___.Object_v___?
                                      IMPORTS___.Object: ___.ri(IMPORTS___,
                                      'Object'), x3___ = object,
                                    x2___.getPrototypeOf_m___?
                                      x2___.getPrototypeOf(x3___):
                                    x2___.m___('getPrototypeOf', [ x3___ ]));
                                  if (opt_sameAs !==
                                      (IMPORTS___.undefined_v___?
                                      IMPORTS___.undefined: ___.ri(IMPORTS___,
                                        'undefined'))) {
                                    privates = dis___.p_m___?
                                      dis___.p(opt_sameAs): dis___.m___('p', [
                                        opt_sameAs ]);
                                  } else {
                                    privates = ___.iM([ '_obj', object ]);
                                  }
                                  table.set_m___? table.set(object, privates):
                                  table.m___('set', [ object, privates ]);
                                }), x0___.def_m___? x0___.def(x1___):
                              x0___.m___('def', [ x1___ ])), x3___.confide_w___
                              === x3___? (x3___.confide = x2___):
                            x3___.w___('confide', x2___);
                            guard = (x9___ = dis___, x8___ = (x4___ =
                                  IMPORTS___.cajaVM_v___? IMPORTS___.cajaVM:
                                ___.ri(IMPORTS___, 'cajaVM'), x6___ = table,
                                x7___ = typename, x5___ =
                                  'This operation requires a ' + typename,
                                x4___.makeTableGuard_m___?
                                  x4___.makeTableGuard(x6___, x7___, x5___):
                                x4___.m___('makeTableGuard', [ x6___, x7___,
                                    x5___ ])), x9___.guard_w___ === x9___?
                                (x9___.guard = x8___): x9___.w___('guard',
                                x8___));
                            x11___ = dis___, x10___ = (function () {
                                function protectMethod$_meth(method) {
                                  var protectedMethod, x0___, x1___, x2___,
                                  x3___;
                                  {
                                    function protectedMethod(var_args) {
                                      {
                                        ___.deodorize(arguments, -6);
                                        var a___ = ___.args(arguments);
                                      }
                                      var dis___ = this && this.___? void 0:
                                      this;
                                      var x0___, x1___, x2___;
                                      return x1___ = method, x0___ =
                                        guard.coerce_m___? guard.coerce(dis___)
                                        : guard.m___('coerce', [ dis___ ]),
                                      x2___ = a___, x1___.apply_m___?
                                        x1___.apply(x0___, x2___):
                                      x1___.m___('apply', [ x0___, x2___ ]);
                                    }
                                    ___.f(protectedMethod, 'protectedMethod');
                                  }
                                  setOwn.i___(protectedMethod, 'toString',
                                    (x0___ = IMPORTS___.cajaVM_v___?
                                       IMPORTS___.cajaVM: ___.ri(IMPORTS___,
                                       'cajaVM'), x1___ = ___.f(function () {
                                         return '[' + typename + ']' +
                                           (method.toString_m___?
                                             method.toString():
                                           method.m___('toString', [ ]));
                                       }), x0___.def_m___? x0___.def(x1___):
                                     x0___.m___('def', [ x1___ ])));
                                  return x2___ = IMPORTS___.cajaVM_v___?
                                    IMPORTS___.cajaVM: ___.ri(IMPORTS___,
                                    'cajaVM'), x3___ = protectedMethod,
                                  x2___.def_m___? x2___.def(x3___):
                                  x2___.m___('def', [ x3___ ]);
                                }
                                return ___.f(protectMethod$_meth,
                                  'protectMethod$_meth');
                              })(), x11___.protectMethod_w___ === x11___?
                              (x11___.protectMethod = x10___):
                            x11___.w___('protectMethod', x10___);
                            x15___ = dis___, x14___ = (x12___ =
                                IMPORTS___.cajaVM_v___? IMPORTS___.cajaVM:
                              ___.ri(IMPORTS___, 'cajaVM'), x13___ =
                                ___.f(function (object) {
                                  var p;
                                  p = table.get_m___? table.get(object):
                                  table.m___('get', [ object ]);
                                  if (p === (IMPORTS___.undefined_v___?
                                      IMPORTS___.undefined: ___.ri(IMPORTS___,
                                        'undefined'))) {
                                    guard.coerce_m___? guard.coerce(object):
                                    guard.m___('coerce', [ object ]);
                                    throw new (IMPORTS___.Error_v___?
                                      IMPORTS___.Error: ___.ri(IMPORTS___,
                                        'Error')).new___('can\'t happen');
                                  } else { return p; }
                                }), x12___.def_m___? x12___.def(x13___):
                              x12___.m___('def', [ x13___ ])), x15___.p_w___
                              === x15___? (x15___.p = x14___): x15___.w___('p',
                              x14___);
                            dis___.typename_w___ === dis___? (dis___.typename =
                              typename): dis___.w___('typename', typename);
                          }
                          ___.f(Confidence, 'Confidence');
                        }
                        setOwn = (x0___ = IMPORTS___.domitaModules_v___?
                          IMPORTS___.domitaModules: ___.ri(IMPORTS___,
                            'domitaModules'), x0___.setOwn_v___? x0___.setOwn:
                          x0___.v___('setOwn'));
                        setOwn.i___(Confidence.prototype_v___?
                            Confidence.prototype: Confidence.v___('prototype'),
                          'toString', (x1___ = IMPORTS___.Object_v___?
                            IMPORTS___.Object: ___.ri(IMPORTS___, 'Object'),
                            x2___ = ___.f(function () {
                                var dis___ = this && this.___? void 0: this;
                                return (dis___.typename_v___? dis___.typename:
                                  dis___.v___('typename')) + 'Confidence';
                              }), x1___.freeze_m___? x1___.freeze(x2___):
                            x1___.m___('freeze', [ x2___ ])));
                        return x3___ = IMPORTS___.cajaVM_v___?
                          IMPORTS___.cajaVM: ___.ri(IMPORTS___, 'cajaVM'),
                        x4___ = Confidence, x3___.def_m___? x3___.def(x4___):
                        x3___.m___('def', [ x4___ ]);
                      }).i___(), x29___.Confidence_w___ === x29___?
                      (x29___.Confidence = x30___): x29___.w___('Confidence',
                      x30___);
                    x31___ = IMPORTS___.domitaModules_v___?
                      IMPORTS___.domitaModules: ___.ri(IMPORTS___,
                      'domitaModules'), x32___ = ___.f(function () {
                        var getPropertyDescriptor, x0___, ProxyHandler, x1___,
                        ExpandoProxyHandler, expandoProxyTargets, x2___, x3___,
                        x4___, x5___, x6___, x7___, x8___, x9___, x10___,
                        x11___, x12___, x13___, x14___, x15___, x16___, x17___,
                        x18___, x19___, x20___, x21___, x22___, x23___, x24___,
                        x25___, x26___, x27___, x28___, x29___;
                        {
                          function ExpandoProxyHandler(target, editable,
                            storage) {
                            var dis___ = this && this.___? void 0: this;
                            dis___.editable_w___ === dis___? (dis___.editable =
                              editable): dis___.w___('editable', editable);
                            dis___.storage_w___ === dis___? (dis___.storage =
                              storage): dis___.w___('storage', storage);
                            dis___.target_w___ === dis___? (dis___.target =
                              target): dis___.w___('target', target);
                          }
                          ___.f(ExpandoProxyHandler, 'ExpandoProxyHandler');
                        }
                        getPropertyDescriptor = (x0___ =
                          IMPORTS___.domitaModules_v___?
                          IMPORTS___.domitaModules: ___.ri(IMPORTS___,
                            'domitaModules'), x0___.getPropertyDescriptor_v___?
                          x0___.getPropertyDescriptor:
                          x0___.v___('getPropertyDescriptor'));
                        ProxyHandler = (x1___ = IMPORTS___.domitaModules_v___?
                          IMPORTS___.domitaModules: ___.ri(IMPORTS___,
                            'domitaModules'), x1___.ProxyHandler_v___?
                          x1___.ProxyHandler: x1___.v___('ProxyHandler'));
                        expandoProxyTargets = new (IMPORTS___.WeakMap_v___?
                          IMPORTS___.WeakMap: ___.ri(IMPORTS___, 'WeakMap'))
                          .new___;
                        x3___ = ExpandoProxyHandler, x2___ = (function () {
                            function register$_meth(proxy, handler) {
                              expandoProxyTargets.set_m___?
                                expandoProxyTargets.set(proxy, handler):
                              expandoProxyTargets.m___('set', [ proxy, handler
                                ]);
                            }
                            return ___.f(register$_meth, 'register$_meth');
                          })(), x3___.register_w___ === x3___? (x3___.register
                            = x2___): x3___.w___('register', x2___);
                        x5___ = ExpandoProxyHandler, x4___ = (function () {
                            function unwrap$_meth(obj) {
                              return (expandoProxyTargets.has_m___?
                                  expandoProxyTargets.has(obj):
                                expandoProxyTargets.m___('has', [ obj ])) ?
                                expandoProxyTargets.get_m___?
                                expandoProxyTargets.get(obj):
                              expandoProxyTargets.m___('get', [ obj ]): obj;
                            }
                            return ___.f(unwrap$_meth, 'unwrap$_meth');
                          })(), x5___.unwrap_w___ === x5___? (x5___.unwrap =
                            x4___): x5___.w___('unwrap', x4___);
                        x6___ = ExpandoProxyHandler.prototype_v___?
                          ExpandoProxyHandler.prototype:
                        ExpandoProxyHandler.v___('prototype'), x7___ =
                          (function () {
                            function getOwnPropertyDescriptor$_meth(name) {
                              var dis___ = this && this.___? void 0: this;
                              var x0___, x1___, x2___, x3___, x4___;
                              if (name === 'ident___') {
                                return x0___ = IMPORTS___.Object_v___?
                                  IMPORTS___.Object: ___.ri(IMPORTS___,
                                  'Object'), x1___ = dis___,
                                x0___.getOwnPropertyDescriptor_m___?
                                  x0___.getOwnPropertyDescriptor(x1___,
                                  'ident'):
                                x0___.m___('getOwnPropertyDescriptor', [ x1___,
                                    'ident' ]);
                              } else {
                                return x2___ = IMPORTS___.Object_v___?
                                  IMPORTS___.Object: ___.ri(IMPORTS___,
                                  'Object'), x3___ = dis___.storage_v___?
                                  dis___.storage: dis___.v___('storage'), x4___
                                  = name, x2___.getOwnPropertyDescriptor_m___?
                                  x2___.getOwnPropertyDescriptor(x3___, x4___):
                                x2___.m___('getOwnPropertyDescriptor', [ x3___,
                                    x4___ ]);
                              }
                            }
                            return ___.f(getOwnPropertyDescriptor$_meth,
                              'getOwnPropertyDescriptor$_meth');
                          })(), x6___.getOwnPropertyDescriptor_w___ === x6___?
                          (x6___.getOwnPropertyDescriptor = x7___):
                        x6___.w___('getOwnPropertyDescriptor', x7___);
                        x8___ = ExpandoProxyHandler.prototype_v___?
                          ExpandoProxyHandler.prototype:
                        ExpandoProxyHandler.v___('prototype'), x9___ =
                          (function () {
                            function getPropertyDescriptor$_meth(name) {
                              var dis___ = this && this.___? void 0: this;
                              var desc;
                              desc = (dis___.getOwnPropertyDescriptor_m___?
                                  dis___.getOwnPropertyDescriptor(name):
                                dis___.m___('getOwnPropertyDescriptor', [ name
                                  ])) ||
                                getPropertyDescriptor.i___(dis___.target_v___?
                                  dis___.target: dis___.v___('target'), name);
                              return desc;
                            }
                            return ___.f(getPropertyDescriptor$_meth,
                              'getPropertyDescriptor$_meth');
                          })(), x8___.getPropertyDescriptor_w___ === x8___?
                          (x8___.getPropertyDescriptor = x9___):
                        x8___.w___('getPropertyDescriptor', x9___);
                        x10___ = ExpandoProxyHandler.prototype_v___?
                          ExpandoProxyHandler.prototype:
                        ExpandoProxyHandler.v___('prototype'), x11___ =
                          (function () {
                            function getOwnPropertyNames$_meth() {
                              var dis___ = this && this.___? void 0: this;
                              var x0___, x1___;
                              return x0___ = IMPORTS___.Object_v___?
                                IMPORTS___.Object: ___.ri(IMPORTS___, 'Object')
                                , x1___ = (dis___.storage_v___? dis___.storage:
                                dis___.v___('storage')) || ___.iM([ ]),
                              x0___.getOwnPropertyNames_m___?
                                x0___.getOwnPropertyNames(x1___):
                              x0___.m___('getOwnPropertyNames', [ x1___ ]);
                            }
                            return ___.f(getOwnPropertyNames$_meth,
                              'getOwnPropertyNames$_meth');
                          })(), x10___.getOwnPropertyNames_w___ === x10___?
                          (x10___.getOwnPropertyNames = x11___):
                        x10___.w___('getOwnPropertyNames', x11___);
                        x12___ = ExpandoProxyHandler.prototype_v___?
                          ExpandoProxyHandler.prototype:
                        ExpandoProxyHandler.v___('prototype'), x13___ =
                          (function () {
                            function defineProperty$_meth(name, descriptor) {
                              var dis___ = this && this.___? void 0: this;
                              var x0___, x1___, x2___, x3___, x4___, x5___,
                              x6___, x7___, x8___, x9___, x10___, x11___,
                              x12___, x13___;
                              if (name === 'ident___') {
                                if ((descriptor.set_v___? descriptor.set:
                                    descriptor.v___('set')) === null) x1___ =
                                  descriptor, x0___ =
                                  IMPORTS___.undefined_v___?
                                  IMPORTS___.undefined: ___.ri(IMPORTS___,
                                  'undefined'), x1___.set_w___ === x1___?
                                  (x1___.set = x0___): x1___.w___('set', x0___)
                                  ;
                                x2___ = IMPORTS___.Object_v___?
                                  IMPORTS___.Object: ___.ri(IMPORTS___,
                                  'Object'), x3___ = dis___, x4___ =
                                  descriptor, x2___.defineProperty_m___?
                                  x2___.defineProperty(x3___, 'ident', x4___):
                                x2___.m___('defineProperty', [ x3___, 'ident',
                                    x4___ ]);
                              } else if (___.i('' + name, dis___.target_v___?
                                  dis___.target: dis___.v___('target'))) {
                                return x6___ = (x5___ =
                                    ProxyHandler.prototype_v___?
                                    ProxyHandler.prototype:
                                  ProxyHandler.v___('prototype'),
                                  x5___.defineProperty_v___?
                                    x5___.defineProperty:
                                  x5___.v___('defineProperty')), x7___ =
                                  dis___, x8___ = name, x9___ = descriptor,
                                x6___.call_m___? x6___.call(x7___, x8___,
                                  x9___): x6___.m___('call', [ x7___, x8___,
                                    x9___ ]);
                              } else {
                                if (! (dis___.editable_v___? dis___.editable:
                                    dis___.v___('editable'))) {
                                  throw new (IMPORTS___.Error_v___?
                                    IMPORTS___.Error: ___.ri(IMPORTS___,
                                      'Error')).new___('Not editable');
                                }
                                x10___ = IMPORTS___.Object_v___?
                                  IMPORTS___.Object: ___.ri(IMPORTS___,
                                  'Object'), x11___ = dis___.storage_v___?
                                  dis___.storage: dis___.v___('storage'),
                                x12___ = name, x13___ = descriptor,
                                x10___.defineProperty_m___?
                                  x10___.defineProperty(x11___, x12___, x13___)
                                  : x10___.m___('defineProperty', [ x11___,
                                    x12___, x13___ ]);
                                return true;
                              }
                              return false;
                            }
                            return ___.f(defineProperty$_meth,
                              'defineProperty$_meth');
                          })(), x12___.defineProperty_w___ === x12___?
                          (x12___.defineProperty = x13___):
                        x12___.w___('defineProperty', x13___);
                        (ExpandoProxyHandler.prototype_v___?
                         ExpandoProxyHandler.prototype:
                         ExpandoProxyHandler.v___('prototype')).w___('delete',
                          ___.f(function (name) {
                              var dis___ = this && this.___? void 0: this;
                              var x0___, x1___, x2___;
                              if (name === 'ident___') { return false; } else
                                if (___.i('' + name, dis___.target_v___?
                                  dis___.target: dis___.v___('target'))) {
                                return x0___ = (ProxyHandler.prototype_v___?
                                    ProxyHandler.prototype:
                                  ProxyHandler.v___('prototype'))
                                  .v___('delete'), x1___ = dis___, x2___ =
                                  name, x0___.call_m___? x0___.call(x1___,
                                  x2___): x0___.m___('call', [ x1___, x2___ ]);
                              } else {
                                if (! (dis___.editable_v___? dis___.editable:
                                    dis___.v___('editable'))) {
                                  throw new (IMPORTS___.Error_v___?
                                    IMPORTS___.Error: ___.ri(IMPORTS___,
                                      'Error')).new___('Not editable');
                                }
                                return (dis___.storage_v___? dis___.storage:
                                  dis___.v___('storage')).c___(name);
                              }
                              return false;
                            }));
                        x14___ = ExpandoProxyHandler.prototype_v___?
                          ExpandoProxyHandler.prototype:
                        ExpandoProxyHandler.v___('prototype'), x15___ =
                          (function () {
                            function fix$_meth() {
                              return IMPORTS___.undefined_v___?
                                IMPORTS___.undefined: ___.ri(IMPORTS___,
                                'undefined');
                            }
                            return ___.f(fix$_meth, 'fix$_meth');
                          })(), x14___.fix_w___ === x14___? (x14___.fix =
                          x15___): x14___.w___('fix', x15___);
                        x16___ = ExpandoProxyHandler.prototype_v___?
                          ExpandoProxyHandler.prototype:
                        ExpandoProxyHandler.v___('prototype'), x20___ = (x18___
                          = (x17___ = IMPORTS___.domitaModules_v___?
                            IMPORTS___.domitaModules: ___.ri(IMPORTS___,
                              'domitaModules'), x17___.permuteProxyGetSet_v___?
                            x17___.permuteProxyGetSet:
                            x17___.v___('permuteProxyGetSet')), x19___ =
                          ___.f(function (name) {
                              var dis___ = this && this.___? void 0: this;
                              var x0___, x1___, x2___, x3___, x4___;
                              if (name === 'ident___') {
                                return dis___.ident_v___? dis___.ident:
                                dis___.v___('ident');
                              } else if (x2___ = (x1___ = (x0___ =
                                      IMPORTS___.Object_v___?
                                      IMPORTS___.Object: ___.ri(IMPORTS___,
                                      'Object'), x0___.prototype_v___?
                                      x0___.prototype: x0___.v___('prototype'))
                                  , x1___.hasOwnProperty_v___?
                                  x1___.hasOwnProperty:
                                  x1___.v___('hasOwnProperty')), x3___ =
                                dis___.storage_v___? dis___.storage:
                                dis___.v___('storage'), x4___ = name,
                                x2___.call_m___? x2___.call(x3___, x4___):
                                x2___.m___('call', [ x3___, x4___ ])) {
                                return (dis___.storage_v___? dis___.storage:
                                  dis___.v___('storage')).v___(name);
                              } else {
                                return (dis___.target_v___? dis___.target:
                                  dis___.v___('target')).v___(name);
                              }
                            }), x18___.getter_m___? x18___.getter(x19___):
                          x18___.m___('getter', [ x19___ ])), x16___.get_w___
                          === x16___? (x16___.get = x20___): x16___.w___('get',
                          x20___);
                        x21___ = ExpandoProxyHandler.prototype_v___?
                          ExpandoProxyHandler.prototype:
                        ExpandoProxyHandler.v___('prototype'), x22___ =
                          (function () {
                            function enumerate$_meth() {
                              var dis___ = this && this.___? void 0: this;
                              var names, k, x0___, x1___, x2___, k, x3___,
                              x4___, x5___;;;
                              names = [ ];
                              {
                                x2___ = Object(dis___.target_v___?
                                    dis___.target: dis___.v___('target'))
                                  .e___();
                                for (x1___ in x2___) {
                                  if (typeof x1___ === 'number' || '' + +x1___
                                    === x1___) { k = x1___; } else {
                                    if (/^NUM___/.test(x1___) &&
                                        /__$/.test(x1___)) { continue; }
                                    x0___ = x1___.match(/([\s\S]*)_e___$/);
                                    if (!x0___ || !x2___[ x1___ ]) { continue;
                                    }
                                    k = x0___[ 1 ];
                                  }
                                  names.push_m___? names.push(k):
                                  names.m___('push', [ k ]);
                                }
                              }
                              {
                                x5___ = Object(dis___.storage_v___?
                                    dis___.storage: dis___.v___('storage'))
                                  .e___();
                                for (x4___ in x5___) {
                                  if (typeof x4___ === 'number' || '' + +x4___
                                    === x4___) { k = x4___; } else {
                                    if (/^NUM___/.test(x4___) &&
                                        /__$/.test(x4___)) { continue; }
                                    x3___ = x4___.match(/([\s\S]*)_e___$/);
                                    if (!x3___ || !x5___[ x4___ ]) { continue;
                                    }
                                    k = x3___[ 1 ];
                                  }
                                  names.push_m___? names.push(k):
                                  names.m___('push', [ k ]);
                                }
                              }
                              return names;
                            }
                            return ___.f(enumerate$_meth, 'enumerate$_meth');
                          })(), x21___.enumerate_w___ === x21___?
                          (x21___.enumerate = x22___): x21___.w___('enumerate',
                          x22___);
                        x23___ = ExpandoProxyHandler.prototype_v___?
                          ExpandoProxyHandler.prototype:
                        ExpandoProxyHandler.v___('prototype'), x27___ = (x25___
                            = (x24___ = IMPORTS___.domitaModules_v___?
                            IMPORTS___.domitaModules: ___.ri(IMPORTS___,
                              'domitaModules'), x24___.permuteProxyGetSet_v___?
                            x24___.permuteProxyGetSet:
                            x24___.v___('permuteProxyGetSet')), x26___ =
                            ___.f(function (name, val, receiver) {
                              var dis___ = this && this.___? void 0: this;
                              var desc, x0___, x1___, x2___, x3___, x4___,
                              x5___, x6___, x7___, x8___;
                              desc = dis___.getOwnPropertyDescriptor_m___?
                                dis___.getOwnPropertyDescriptor(name):
                              dis___.m___('getOwnPropertyDescriptor', [ name ])
                                ;
                              if (desc) {
                                if (___.i('' + 'writable', desc)) {
                                  if (desc.writable_v___? desc.writable:
                                    desc.v___('writable')) {
                                    desc.value_w___ === desc? (desc.value =
                                        val): desc.w___('value', val);
                                    dis___.defineProperty_m___?
                                      dis___.defineProperty(name, desc):
                                    dis___.m___('defineProperty', [ name, desc
                                      ]);
                                    return true;
                                  } else { return false; }
                                } else {
                                  if (desc.set_v___? desc.set: desc.v___('set')
                                  ) {
                                    x0___ = desc.set_v___? desc.set:
                                    desc.v___('set'), x1___ = receiver, x2___ =
                                      val, x0___.call_m___? x0___.call(x1___,
                                      x2___): x0___.m___('call', [ x1___, x2___
                                      ]);
                                    return true;
                                  } else { return false; }
                                }
                              }
                              desc = dis___.getPropertyDescriptor_m___?
                                dis___.getPropertyDescriptor(name):
                              dis___.m___('getPropertyDescriptor', [ name ]);
                              if (desc) {
                                if (___.i('' + 'writable', desc)) {
                                  if (desc.writable_v___? desc.writable:
                                    desc.v___('writable')) {} else { return false; }
                                } else {
                                  if (desc.set_v___? desc.set: desc.v___('set')
                                  ) {
                                    x3___ = desc.set_v___? desc.set:
                                    desc.v___('set'), x4___ = receiver, x5___ =
                                      val, x3___.call_m___? x3___.call(x4___,
                                      x5___): x3___.m___('call', [ x4___, x5___
                                      ]);
                                    return true;
                                  } else { return false; }
                                }
                              }
                              x7___ = dis___, x8___ = name, x6___ = ___.iM([
                                  'value', val, 'writable', true, 'enumerable',
                                  true, 'configurable', true ]),
                              x7___.defineProperty_m___?
                                x7___.defineProperty(x8___, x6___):
                              x7___.m___('defineProperty', [ x8___, x6___ ]);
                              return true;
                            }), x25___.setter_m___? x25___.setter(x26___):
                          x25___.m___('setter', [ x26___ ])), x23___.set_w___
                          === x23___? (x23___.set = x27___): x23___.w___('set',
                          x27___);
                        return x28___ = IMPORTS___.cajaVM_v___?
                          IMPORTS___.cajaVM: ___.ri(IMPORTS___, 'cajaVM'),
                        x29___ = ExpandoProxyHandler, x28___.def_m___?
                          x28___.def(x29___): x28___.m___('def', [ x29___ ]);
                      }).i___(), x31___.ExpandoProxyHandler_w___ === x31___?
                      (x31___.ExpandoProxyHandler = x32___):
                    x31___.w___('ExpandoProxyHandler', x32___);
                    x33___ = IMPORTS___.domitaModules_v___?
                      IMPORTS___.domitaModules: ___.ri(IMPORTS___,
                      'domitaModules'), x34___ = (function () {
                        function XMLHttpRequestCtor$_meth(XMLHttpRequest,
                          ActiveXObject) {
                          var activeXClassId;
                          if (XMLHttpRequest) {
                            return XMLHttpRequest;
                          } else if (ActiveXObject) {
                            return (function () {
                                function ActiveXObjectForIE() {
                                  var activeXClassIds, i, n, candidate;
                                  if (activeXClassId === void 0) {
                                    activeXClassId = null;
                                    activeXClassIds = [ 'MSXML2.XMLHTTP.5.0',
                                      'MSXML2.XMLHTTP.4.0',
                                      'MSXML2.XMLHTTP.3.0', 'MSXML2.XMLHTTP',
                                      'MICROSOFT.XMLHTTP.1.0',
                                      'MICROSOFT.XMLHTTP.1',
                                      'MICROSOFT.XMLHTTP' ];
                                    for (i = 0, n = activeXClassIds.length; i <
                                        n; i++) {
                                      candidate = activeXClassIds[ + +i ];
                                      try {
                                        void new
                                          ActiveXObject.new___(candidate);
                                        activeXClassId = candidate;
                                        break;
                                      } catch (e) {}
                                    }
                                    activeXClassIds = null;
                                  }
                                  return new
                                    ActiveXObject.new___(activeXClassId);
                                }
                                return ___.f(ActiveXObjectForIE,
                                  'ActiveXObjectForIE');
                              })();
                          } else {
                            throw new (IMPORTS___.Error_v___? IMPORTS___.Error:
                              ___.ri(IMPORTS___, 'Error'))
                              .new___('ActiveXObject not available');
                          }
                        }
                        return ___.f(XMLHttpRequestCtor$_meth,
                          'XMLHttpRequestCtor$_meth');
                      })(), x33___.XMLHttpRequestCtor_w___ === x33___?
                      (x33___.XMLHttpRequestCtor = x34___):
                    x33___.w___('XMLHttpRequestCtor', x34___);
                    x35___ = IMPORTS___.domitaModules_v___?
                      IMPORTS___.domitaModules: ___.ri(IMPORTS___,
                      'domitaModules'), x36___ = (function () {
                        function TameXMLHttpRequest$_meth(taming, rulebreaker,
                          xmlHttpRequestMaker, naiveUriPolicy) {
                          var Confidence, x0___, setOwn, x1___,
                          canHaveEnumerableAccessors, x2___, TameXHRConf, p,
                          x3___, x4___, method, TameXMLHttpRequest, x5___,
                          x6___, x7___, x8___, x9___, x10___, x11___, x12___,
                          x13___, x14___, x15___, x16___, x17___, x18___,
                          x19___, x20___, x21___;
                          {
                            function TameXMLHttpRequest() {
                              var dis___ = this && this.___? void 0: this;
                              var xhr, x0___, x1___, x2___, x3___;
                              TameXHRConf.confide_m___?
                                TameXHRConf.confide(dis___):
                              TameXHRConf.m___('confide', [ dis___ ]);
                              xhr = (x0___ = p.i___(dis___), x3___ = (x2___ =
                                  rulebreaker, x1___ = new
                                  xmlHttpRequestMaker.new___,
                                  x2___.makeDOMAccessible_m___?
                                  x2___.makeDOMAccessible(x1___):
                                  x2___.m___('makeDOMAccessible', [ x1___ ])),
                                x0___.feral_w___ === x0___? (x0___.feral =
                                  x3___): x0___.w___('feral', x3___));
                              taming.tamesTo_m___? taming.tamesTo(xhr, dis___):
                              taming.m___('tamesTo', [ xhr, dis___ ]);
                            }
                            ___.f(TameXMLHttpRequest, 'TameXMLHttpRequest');
                          }
                          Confidence = (x0___ = IMPORTS___.domitaModules_v___?
                            IMPORTS___.domitaModules: ___.ri(IMPORTS___,
                              'domitaModules'), x0___.Confidence_v___?
                            x0___.Confidence: x0___.v___('Confidence'));
                          setOwn = (x1___ = IMPORTS___.domitaModules_v___?
                            IMPORTS___.domitaModules: ___.ri(IMPORTS___,
                              'domitaModules'), x1___.setOwn_v___?
                            x1___.setOwn: x1___.v___('setOwn'));
                          canHaveEnumerableAccessors = (x2___ =
                              IMPORTS___.domitaModules_v___?
                              IMPORTS___.domitaModules: ___.ri(IMPORTS___,
                              'domitaModules'),
                            x2___.canHaveEnumerableAccessors_v___?
                              x2___.canHaveEnumerableAccessors:
                            x2___.v___('canHaveEnumerableAccessors'));
                          TameXHRConf = new
                            Confidence.new___('TameXMLHttpRequest');
                          p = (x3___ = TameXHRConf.p_v___? TameXHRConf.p:
                            TameXHRConf.v___('p'), x4___ = TameXHRConf,
                            x3___.bind_m___? x3___.bind(x4___):
                            x3___.m___('bind', [ x4___ ]));
                          method = TameXHRConf.protectMethod_v___?
                            TameXHRConf.protectMethod:
                          TameXHRConf.v___('protectMethod');
                          x5___ = IMPORTS___.Object_v___? IMPORTS___.Object:
                          ___.ri(IMPORTS___, 'Object'), x6___ =
                            TameXMLHttpRequest.prototype_v___?
                            TameXMLHttpRequest.prototype:
                          TameXMLHttpRequest.v___('prototype'), x7___ =
                            ___.iM([ 'onreadystatechange', ___.iM([
                                  'enumerable', canHaveEnumerableAccessors,
                                  'set', method.i___(___.f(function (handler) {
                                        var dis___ = this && this.___? void 0:
                                        this;
                                        var self, x0___, x1___, x2___, x3___,
                                        x4___;
                                        self = dis___;
                                        x1___ = (x0___ = p.i___(dis___),
                                          x0___.feral_v___? x0___.feral:
                                          x0___.v___('feral')), x2___ =
                                          (function () {
                                            function
                                              onreadystatechange$_meth(event) {
                                              var evt, x0___, x1___, x2___;
                                              evt = ___.iM([ 'target', self ]);
                                              return x1___ = handler, x0___ =
                                                void 0, x2___ = evt,
                                              x1___.call_m___?
                                                x1___.call(x0___, x2___):
                                              x1___.m___('call', [ x0___, x2___
                                                ]);
                                            }
                                            return ___.f(onreadystatechange$_meth,
                                              'onreadystatechange$_meth');
                                          })(), x1___.onreadystatechange_w___
                                          === x1___? (x1___.onreadystatechange
                                          = x2___):
                                        x1___.w___('onreadystatechange', x2___)
                                          ;
                                        x3___ = p.i___(dis___), x4___ =
                                          handler, x3___.handler_w___ ===
                                          x3___? (x3___.handler = x4___):
                                        x3___.w___('handler', x4___);
                                      })) ]), 'readyState', ___.iM([
                                  'enumerable', canHaveEnumerableAccessors,
                                  'get', method.i___(___.f(function () {
                                        var dis___ = this && this.___? void 0:
                                        this;
                                        var x0___, x1___;
                                        return (IMPORTS___.Number_v___?
                                          IMPORTS___.Number: ___.ri(IMPORTS___,
                                            'Number')).i___((x1___ = (x0___ =
                                                p.i___(dis___),
                                              x0___.feral_v___? x0___.feral:
                                              x0___.v___('feral')),
                                            x1___.readyState_v___?
                                              x1___.readyState:
                                            x1___.v___('readyState')));
                                      })) ]), 'responseText', ___.iM([
                                  'enumerable', canHaveEnumerableAccessors,
                                  'get', method.i___(___.f(function () {
                                        var dis___ = this && this.___? void 0:
                                        this;
                                        var result, x0___, x1___;
                                        result = (x1___ = (x0___ =
                                              p.i___(dis___), x0___.feral_v___?
                                              x0___.feral: x0___.v___('feral'))
                                          , x1___.responseText_v___?
                                          x1___.responseText:
                                          x1___.v___('responseText'));
                                        return result ===
                                          (IMPORTS___.undefined_v___?
                                            IMPORTS___.undefined:
                                          ___.ri(IMPORTS___, 'undefined')) ||
                                          result === null ? result:
                                        (IMPORTS___.String_v___?
                                           IMPORTS___.String:
                                         ___.ri(IMPORTS___, 'String'))
                                        .i___(result);
                                      })) ]), 'responseXML', ___.iM([
                                  'enumerable', canHaveEnumerableAccessors,
                                  'get', method.i___(___.f(function () {
                                        var node, x0___, x1___, x2___, x3___;
                                        node = ___.iM([ 'nodeName', '#document'
                                          ]);
                                        x1___ = node, x0___ = (function () {
                                            function cloneNode$_meth() { return node; }
                                            return ___.f(cloneNode$_meth,
                                              'cloneNode$_meth');
                                          })(), x1___.cloneNode_w___ === x1___?
                                          (x1___.cloneNode = x0___):
                                        x1___.w___('cloneNode', x0___);
                                        x3___ = node, x2___ = (function () {
                                            function toString$_meth() {
                                              return 'Caja does not support XML.';
                                            }
                                            return ___.f(toString$_meth,
                                              'toString$_meth');
                                          })(), x3___.toString_w___ === x3___?
                                          (x3___.toString = x2___):
                                        x3___.w___('toString', x2___);
                                        return ___.iM([ 'documentElement', node
                                          ]);
                                      })) ]), 'status', ___.iM([ 'enumerable',
                                  canHaveEnumerableAccessors, 'get',
                                  method.i___(___.f(function () {
                                        var dis___ = this && this.___? void 0:
                                        this;
                                        var result, x0___, x1___;
                                        result = (x1___ = (x0___ =
                                              p.i___(dis___), x0___.feral_v___?
                                              x0___.feral: x0___.v___('feral'))
                                            , x1___.status_v___? x1___.status:
                                          x1___.v___('status'));
                                        return result ===
                                          (IMPORTS___.undefined_v___?
                                          IMPORTS___.undefined:
                                          ___.ri(IMPORTS___, 'undefined')) ||
                                          result === null ? result:
                                        (IMPORTS___.Number_v___?
                                           IMPORTS___.Number:
                                         ___.ri(IMPORTS___, 'Number'))
                                        .i___(result);
                                      })) ]), 'statusText', ___.iM([
                                  'enumerable', canHaveEnumerableAccessors,
                                  'get', method.i___(___.f(function () {
                                        var dis___ = this && this.___? void 0:
                                        this;
                                        var result, x0___, x1___;
                                        result = (x1___ = (x0___ =
                                            p.i___(dis___), x0___.feral_v___?
                                            x0___.feral: x0___.v___('feral')),
                                          x1___.statusText_v___?
                                            x1___.statusText:
                                          x1___.v___('statusText'));
                                        return result ===
                                          (IMPORTS___.undefined_v___?
                                            IMPORTS___.undefined:
                                          ___.ri(IMPORTS___, 'undefined')) ||
                                          result === null ? result:
                                        (IMPORTS___.String_v___?
                                           IMPORTS___.String:
                                         ___.ri(IMPORTS___, 'String'))
                                        .i___(result);
                                      })) ]) ]), x5___.defineProperties_m___?
                            x5___.defineProperties(x6___, x7___):
                          x5___.m___('defineProperties', [ x6___, x7___ ]);
                          x8___ = TameXMLHttpRequest.prototype_v___?
                            TameXMLHttpRequest.prototype:
                          TameXMLHttpRequest.v___('prototype'), x9___ =
                            method.i___(___.f(function (method, URL, opt_async,
                                opt_userName, opt_password) {
                                {
                                  ___.deodorize(arguments, -6);
                                  var a___ = ___.args(arguments);
                                }
                                var dis___ = this && this.___? void 0: this;
                                var safeUri, x0___, x1___, x2___, x3___, x4___,
                                x5___, x6___, x7___, x8___, x9___, x10___,
                                x11___, x12___, x13___, x14___, x15___, x16___,
                                x17___, x18___, x19___, x20___, x21___, x22___,
                                x23___, x24___, x25___, x26___, x27___, x28___,
                                x29___, x30___, x31___, x32___;
                                method = (IMPORTS___.String_v___?
                                  IMPORTS___.String: ___.ri(IMPORTS___,
                                    'String')).i___(method);
                                safeUri = uriRewrite.i___(naiveUriPolicy,
                                  (IMPORTS___.String_v___? IMPORTS___.String:
                                   ___.ri(IMPORTS___, 'String')).i___(URL),
                                  (x1___ = (x0___ = IMPORTS___.html4_v___?
                                       IMPORTS___.html4: ___.ri(IMPORTS___,
                                       'html4'), x0___.ueffects_v___?
                                       x0___.ueffects: x0___.v___('ueffects')),
                                   x1___.SAME_DOCUMENT_v___?
                                     x1___.SAME_DOCUMENT:
                                   x1___.v___('SAME_DOCUMENT')), (x3___ =
                                    (x2___ = IMPORTS___.html4_v___?
                                     IMPORTS___.html4: ___.ri(IMPORTS___,
                                       'html4'), x2___.ltypes_v___?
                                     x2___.ltypes: x2___.v___('ltypes')),
                                    x3___.DATA_v___? x3___.DATA:
                                    x3___.v___('DATA')), ___.iM([ 'TYPE',
                                      'XHR', 'XHR_METHOD', method, 'XHR', true
                                    ]));
                                if ('string' !== typeof safeUri) {
                                  throw 'URI violates security policy';
                                }
                                switch (a___.length) {
                                case 2:
                                  x4___ = p.i___(dis___), x4___.async_w___ ===
                                    x4___? (x4___.async = true):
                                  x4___.w___('async', true);
                                  x6___ = (x5___ = p.i___(dis___),
                                    x5___.feral_v___? x5___.feral:
                                    x5___.v___('feral')), x7___ = method, x8___
                                    = safeUri, x6___.open_m___?
                                    x6___.open(x7___, x8___):
                                  x6___.m___('open', [ x7___, x8___ ]);
                                  break;
                                case 3:
                                  x9___ = p.i___(dis___), x10___ = opt_async,
                                  x9___.async_w___ === x9___? (x9___.async =
                                    x10___): x9___.w___('async', x10___);
                                  x12___ = (x11___ = p.i___(dis___),
                                    x11___.feral_v___? x11___.feral:
                                    x11___.v___('feral')), x14___ = method,
                                  x15___ = safeUri, x13___ =
                                    (IMPORTS___.Boolean_v___?
                                    IMPORTS___.Boolean: ___.ri(IMPORTS___,
                                      'Boolean')).i___(opt_async),
                                  x12___.open_m___? x12___.open(x14___, x15___,
                                    x13___): x12___.m___('open', [ x14___,
                                      x15___, x13___ ]);
                                  break;
                                case 4:
                                  x16___ = p.i___(dis___), x17___ = opt_async,
                                  x16___.async_w___ === x16___? (x16___.async =
                                      x17___): x16___.w___('async', x17___);
                                  x19___ = (x18___ = p.i___(dis___),
                                    x18___.feral_v___? x18___.feral:
                                    x18___.v___('feral')), x22___ = method,
                                  x23___ = safeUri, x20___ =
                                    (IMPORTS___.Boolean_v___?
                                    IMPORTS___.Boolean: ___.ri(IMPORTS___,
                                      'Boolean')).i___(opt_async), x21___ =
                                    (IMPORTS___.String_v___? IMPORTS___.String:
                                    ___.ri(IMPORTS___, 'String'))
                                    .i___(opt_userName), x19___.open_m___?
                                    x19___.open(x22___, x23___, x20___, x21___)
                                    : x19___.m___('open', [ x22___, x23___,
                                      x20___, x21___ ]);
                                  break;
                                case 5:
                                  x24___ = p.i___(dis___), x25___ = opt_async,
                                  x24___.async_w___ === x24___? (x24___.async =
                                    x25___): x24___.w___('async', x25___);
                                  x27___ = (x26___ = p.i___(dis___),
                                    x26___.feral_v___? x26___.feral:
                                    x26___.v___('feral')), x31___ = method,
                                  x32___ = safeUri, x28___ =
                                    (IMPORTS___.Boolean_v___?
                                    IMPORTS___.Boolean: ___.ri(IMPORTS___,
                                      'Boolean')).i___(opt_async), x29___ =
                                    (IMPORTS___.String_v___? IMPORTS___.String:
                                    ___.ri(IMPORTS___, 'String'))
                                    .i___(opt_userName), x30___ =
                                    (IMPORTS___.String_v___? IMPORTS___.String:
                                    ___.ri(IMPORTS___, 'String'))
                                    .i___(opt_password), x27___.open_m___?
                                    x27___.open(x31___, x32___, x28___, x29___,
                                    x30___): x27___.m___('open', [ x31___,
                                      x32___, x28___, x29___, x30___ ]);
                                  break;
                                default:
                                  throw 'XMLHttpRequest cannot accept ' +
                                    a___.length + ' arguments';
                                  break;
                                }
                              })), x8___.open_w___ === x8___? (x8___.open =
                            x9___): x8___.w___('open', x9___);
                          x10___ = TameXMLHttpRequest.prototype_v___?
                            TameXMLHttpRequest.prototype:
                          TameXMLHttpRequest.v___('prototype'), x11___ =
                            method.i___(___.f(function (label, value) {
                                var dis___ = this && this.___? void 0: this;
                                var x0___, x1___, x2___, x3___;
                                x1___ = (x0___ = p.i___(dis___),
                                  x0___.feral_v___? x0___.feral:
                                  x0___.v___('feral')), x2___ =
                                  (IMPORTS___.String_v___? IMPORTS___.String:
                                  ___.ri(IMPORTS___, 'String')).i___(label),
                                x3___ = (IMPORTS___.String_v___?
                                    IMPORTS___.String: ___.ri(IMPORTS___,
                                    'String')).i___(value),
                                x1___.setRequestHeader_m___?
                                  x1___.setRequestHeader(x2___, x3___):
                                x1___.m___('setRequestHeader', [ x2___, x3___ ]
                                );
                              })), x10___.setRequestHeader_w___ === x10___?
                            (x10___.setRequestHeader = x11___):
                          x10___.w___('setRequestHeader', x11___);
                          x12___ = TameXMLHttpRequest.prototype_v___?
                            TameXMLHttpRequest.prototype:
                          TameXMLHttpRequest.v___('prototype'), x13___ =
                            method.i___(___.f(function (opt_data) {
                                {
                                  ___.deodorize(arguments, -6);
                                  var a___ = ___.args(arguments);
                                }
                                var dis___ = this && this.___? void 0: this;
                                var x0___, x1___, x2___, x3___, x4___, x5___,
                                x6___, x7___, x8___, x9___, x10___, evt,
                                x11___, x12___, x13___, x14___;
                                if (a___.length === 0) {
                                  x1___ = (x0___ = p.i___(dis___),
                                    x0___.feral_v___? x0___.feral:
                                    x0___.v___('feral')), x1___.send_m___?
                                    x1___.send(''): x1___.m___('send', [ '' ]);
                                } else if (typeof opt_data === 'string') {
                                  x3___ = (x2___ = p.i___(dis___),
                                    x2___.feral_v___? x2___.feral:
                                    x2___.v___('feral')), x4___ = opt_data,
                                  x3___.send_m___? x3___.send(x4___):
                                  x3___.m___('send', [ x4___ ]);
                                } else {
                                  x6___ = (x5___ = p.i___(dis___),
                                    x5___.feral_v___? x5___.feral:
                                    x5___.v___('feral')), x6___.send_m___?
                                    x6___.send(''): x6___.m___('send', [ '' ]);
                                }
                                if (x8___ = (x7___ = p.i___(dis___),
                                    x7___.feral_v___? x7___.feral:
                                    x7___.v___('feral')),
                                  x8___.overrideMimeType_v___?
                                    x8___.overrideMimeType:
                                  x8___.v___('overrideMimeType')) {
                                  if (! (x9___ = p.i___(dis___),
                                      x9___.async_v___? x9___.async:
                                      x9___.v___('async')) && (x10___ =
                                        p.i___(dis___), x10___.handler_v___?
                                        x10___.handler: x10___.v___('handler'))
                                  ) {
                                    evt = ___.iM([ 'target', dis___ ]);
                                    x12___ = (x11___ = p.i___(dis___),
                                      x11___.handler_v___? x11___.handler:
                                      x11___.v___('handler')), x13___ = void 0,
                                    x14___ = evt, x12___.call_m___?
                                      x12___.call(x13___, x14___):
                                    x12___.m___('call', [ x13___, x14___ ]);
                                  }
                                }
                              })), x12___.send_w___ === x12___? (x12___.send =
                              x13___): x12___.w___('send', x13___);
                          x14___ = TameXMLHttpRequest.prototype_v___?
                            TameXMLHttpRequest.prototype:
                          TameXMLHttpRequest.v___('prototype'), x15___ =
                            method.i___(___.f(function () {
                                var dis___ = this && this.___? void 0: this;
                                var x0___, x1___;
                                x1___ = (x0___ = p.i___(dis___),
                                  x0___.feral_v___? x0___.feral:
                                  x0___.v___('feral')), x1___.abort_m___?
                                  x1___.abort(): x1___.m___('abort', [ ]);
                              })), x14___.abort_w___ === x14___? (x14___.abort
                              = x15___): x14___.w___('abort', x15___);
                          x16___ = TameXMLHttpRequest.prototype_v___?
                            TameXMLHttpRequest.prototype:
                          TameXMLHttpRequest.v___('prototype'), x17___ =
                            method.i___(___.f(function () {
                                var dis___ = this && this.___? void 0: this;
                                var result, x0___, x1___;
                                result = (x1___ = (x0___ = p.i___(dis___),
                                    x0___.feral_v___? x0___.feral:
                                    x0___.v___('feral')),
                                  x1___.getAllResponseHeaders_m___?
                                  x1___.getAllResponseHeaders():
                                  x1___.m___('getAllResponseHeaders', [ ]));
                                return result === (IMPORTS___.undefined_v___?
                                  IMPORTS___.undefined: ___.ri(IMPORTS___,
                                    'undefined')) || result === null ? result:
                                (IMPORTS___.String_v___? IMPORTS___.String:
                                 ___.ri(IMPORTS___, 'String')).i___(result);
                              })), x16___.getAllResponseHeaders_w___ ===
                            x16___? (x16___.getAllResponseHeaders = x17___):
                          x16___.w___('getAllResponseHeaders', x17___);
                          x18___ = TameXMLHttpRequest.prototype_v___?
                            TameXMLHttpRequest.prototype:
                          TameXMLHttpRequest.v___('prototype'), x19___ =
                            method.i___(___.f(function (headerName) {
                                var dis___ = this && this.___? void 0: this;
                                var result, x0___, x1___, x2___;
                                result = (x1___ = (x0___ = p.i___(dis___),
                                    x0___.feral_v___? x0___.feral:
                                    x0___.v___('feral')), x2___ =
                                    (IMPORTS___.String_v___? IMPORTS___.String:
                                    ___.ri(IMPORTS___, 'String'))
                                    .i___(headerName),
                                  x1___.getResponseHeader_m___?
                                    x1___.getResponseHeader(x2___):
                                  x1___.m___('getResponseHeader', [ x2___ ]));
                                return result === (IMPORTS___.undefined_v___?
                                  IMPORTS___.undefined: ___.ri(IMPORTS___,
                                    'undefined')) || result === null ? result:
                                (IMPORTS___.String_v___? IMPORTS___.String:
                                 ___.ri(IMPORTS___, 'String')).i___(result);
                              })), x18___.getResponseHeader_w___ === x18___?
                            (x18___.getResponseHeader = x19___):
                          x18___.w___('getResponseHeader', x19___);
                          setOwn.i___(TameXMLHttpRequest.prototype_v___?
                            TameXMLHttpRequest.prototype:
                            TameXMLHttpRequest.v___('prototype'), 'toString',
                            method.i___(___.f(function () {
                                  return 'Not a real XMLHttpRequest';
                                })));
                          return x20___ = IMPORTS___.cajaVM_v___?
                            IMPORTS___.cajaVM: ___.ri(IMPORTS___, 'cajaVM'),
                          x21___ = TameXMLHttpRequest, x20___.def_m___?
                            x20___.def(x21___): x20___.m___('def', [ x21___ ]);
                        }
                        return ___.f(TameXMLHttpRequest$_meth,
                          'TameXMLHttpRequest$_meth');
                      })(), x35___.TameXMLHttpRequest_w___ === x35___?
                      (x35___.TameXMLHttpRequest = x36___):
                    x35___.w___('TameXMLHttpRequest', x36___);
                    x37___ = IMPORTS___.domitaModules_v___?
                      IMPORTS___.domitaModules: ___.ri(IMPORTS___,
                      'domitaModules'), x38___ = (function () {
                        function CssPropertiesCollection$_meth(aStyleObject) {
                          var canonicalStylePropertyNames, cssPropertyNames,
                          cssPropertyName, x0___, x1___, x2___,
                          baseStylePropertyName, x3___, x4___, x5___,
                          canonStylePropertyName, alts, x6___, i;;
                          canonicalStylePropertyNames = ___.iM([ ]);
                          cssPropertyNames = ___.iM([ ]);
                          {
                            x2___ = Object(IMPORTS___.cssSchema_v___?
                              IMPORTS___.cssSchema: ___.ri(IMPORTS___,
                                'cssSchema')).e___();
                            for (x1___ in x2___) {
                              if (typeof x1___ === 'number' || '' + +x1___ ===
                                x1___) {
                                cssPropertyName = x1___;
                              } else {
                                if (/^NUM___/.test(x1___) && /__$/.test(x1___))
                                  { continue; }
                                x0___ = x1___.match(/([\s\S]*)_e___$/);
                                if (!x0___ || !x2___[ x1___ ]) { continue; }
                                cssPropertyName = x0___[ 1 ];
                              }
                              {
                                baseStylePropertyName = (x5___ =
                                  cssPropertyName, x3___ = new
                                  RegExp.new___('-([a-z])', 'g'), x4___ =
                                  ___.f(function (_, letter) {
                                      return letter.toUpperCase_m___?
                                        letter.toUpperCase():
                                      letter.m___('toUpperCase', [ ]);
                                    }), x5___.replace_m___?
                                  x5___.replace(x3___, x4___):
                                  x5___.m___('replace', [ x3___, x4___ ]));
                                canonStylePropertyName = baseStylePropertyName;
                                cssPropertyNames.w___(baseStylePropertyName,
                                  cssPropertyNames.w___(canonStylePropertyName,
                                    cssPropertyName));
                                alts = (x6___ = (IMPORTS___.cssSchema_v___?
                                    IMPORTS___.cssSchema: ___.ri(IMPORTS___,
                                      'cssSchema')).v___(cssPropertyName),
                                  x6___.cssAlternates_v___?
                                    x6___.cssAlternates:
                                  x6___.v___('cssAlternates'));
                                if (alts) {
                                  for (i = alts.length; --i >= 0;) {
                                    cssPropertyNames.w___(alts[ + +i ],
                                      cssPropertyName);
                                    if (___.i('' + alts[ + +i ], aStyleObject)
                                        && !___.i('' + canonStylePropertyName,
                                        aStyleObject)) {
                                      canonStylePropertyName = alts[ + +i ];
                                    }
                                  }
                                }
                                canonicalStylePropertyNames.w___(cssPropertyName,
                                  canonStylePropertyName);
                              }
                            }
                          }
                          return ___.iM([ 'isCanonicalProp', (function () {
                                  function isCanonicalProp$_lit(p) {
                                    return cssPropertyNames.hasOwnProperty_m___?
                                      cssPropertyNames.hasOwnProperty(p):
                                    cssPropertyNames.m___('hasOwnProperty', [ p
                                      ]);
                                  }
                                  return ___.f(isCanonicalProp$_lit,
                                    'isCanonicalProp$_lit');
                                })(), 'isCssProp', (function () {
                                  function isCssProp$_lit(p) {
                                    return canonicalStylePropertyNames.hasOwnProperty_m___?
                                      canonicalStylePropertyNames.hasOwnProperty(p)
                                      :
                                    canonicalStylePropertyNames.m___('hasOwnProperty',
                                      [ p ]);
                                  }
                                  return ___.f(isCssProp$_lit,
                                    'isCssProp$_lit');
                                })(), 'getCanonicalPropFromCss', (function () {
                                  function getCanonicalPropFromCss$_lit(p) {
                                    return canonicalStylePropertyNames.v___(p);
                                  }
                                  return ___.f(getCanonicalPropFromCss$_lit,
                                    'getCanonicalPropFromCss$_lit');
                                })(), 'getCssPropFromCanonical', (function () {
                                  function getCssPropFromCanonical$_lit(p) {
                                    return cssPropertyNames.v___(p);
                                  }
                                  return ___.f(getCssPropFromCanonical$_lit,
                                    'getCssPropFromCanonical$_lit');
                                })(), 'forEachCanonical', (function () {
                                  function forEachCanonical$_lit(f) {
                                    var p, x0___, x1___, x2___;;
                                    {
                                      x2___ = Object(cssPropertyNames).e___();
                                      for (x1___ in x2___) {
                                        if (typeof x1___ === 'number' || '' +
                                            +x1___ === x1___) { p = x1___; }
                                        else {
                                          if (/^NUM___/.test(x1___) &&
                                              /__$/.test(x1___)) { continue; }
                                          x0___ =
                                            x1___.match(/([\s\S]*)_e___$/);
                                          if (!x0___ || !x2___[ x1___ ]) {
                                            continue; }
                                          p = x0___[ 1 ];
                                        }
                                        {
                                          if
                                            (cssPropertyNames.hasOwnProperty_m___?
                                            cssPropertyNames.hasOwnProperty(p):
                                            cssPropertyNames.m___('hasOwnProperty',
                                              [ p ])) {
                                            f.i___(p);
                                          }
                                        }
                                      }
                                    }
                                  }
                                  return ___.f(forEachCanonical$_lit,
                                    'forEachCanonical$_lit');
                                })() ]);
                        }
                        return ___.f(CssPropertiesCollection$_meth,
                          'CssPropertiesCollection$_meth');
                      })(), x37___.CssPropertiesCollection_w___ === x37___?
                      (x37___.CssPropertiesCollection = x38___):
                    x37___.w___('CssPropertiesCollection', x38___);
                    x39___ = IMPORTS___.cajaVM_v___? IMPORTS___.cajaVM:
                    ___.ri(IMPORTS___, 'cajaVM'), x40___ =
                      IMPORTS___.domitaModules_v___? IMPORTS___.domitaModules:
                    ___.ri(IMPORTS___, 'domitaModules'), x39___.def_m___?
                      x39___.def(x40___): x39___.m___('def', [ x40___ ]);
                    return (function () {
                        function Domado(opt_rulebreaker) {
                          var importsToId, idToImports, nextPluginId,
                          rulebreaker, x0___, x1___, makeDOMAccessible,
                          makeFunctionAccessible, Confidence, x2___,
                          ProxyHandler, x3___, ExpandoProxyHandler, x4___,
                          setOwn, x5___, canHaveEnumerableAccessors, x6___,
                          traceStartup, inherit, definePropertiesAwesomely,
                          forOwnKeys, identity, defaultToEmptyStr, arrayRemove,
                          windowToDomicile, TameEventConf, TameEventT,
                          TameImageDataConf, TameImageDataT, TameGradientConf,
                          TameGradientT, eventMethod, safeHTMLTable, Html,
                          htmlToString, safeHtml, blessHtml, x7___, x8___,
                          XML_SPACE, JS_SPACE, JS_IDENT,
                          SIMPLE_HANDLER_PATTERN, VALID_ID_CHAR, x9___, x10___,
                          x11___, x12___, VALID_ID_PATTERN,
                          VALID_ID_LIST_PATTERN, FORBIDDEN_ID_PATTERN,
                          FORBIDDEN_ID_LIST_PATTERN, isValidId, isValidIdList,
                          trimHTML5Spaces, mimeTypeForAttr, assert,
                          cssSealerUnsealerPair, x13___, tameSetAndClear,
                          makeScrollable, tameScrollTo, tameScrollBy,
                          guessPixelsFromCss, tameResizeTo, tameResizeBy,
                          attachDocument, plugin_dispatchEvent,
                          plugin_dispatchToHandler, x14___, x15___, x16___,
                          x17___;
                          {
                            function traceStartup(var_args) {}
                            ___.f(traceStartup, 'traceStartup');
                          }
                          {
                            function inherit(sub, souper) {
                              var x0___, x1___, x2___, x3___, x4___, x5___,
                              x6___;
                              x3___ = sub, x2___ = (x0___ =
                                  IMPORTS___.Object_v___? IMPORTS___.Object:
                                ___.ri(IMPORTS___, 'Object'), x1___ =
                                  souper.prototype_v___? souper.prototype:
                                souper.v___('prototype'), x0___.create_m___?
                                  x0___.create(x1___): x0___.m___('create', [
                                    x1___ ])), x3___.prototype_w___ === x3___?
                                (x3___.prototype = x2___):
                              x3___.w___('prototype', x2___);
                              x4___ = IMPORTS___.Object_v___?
                                IMPORTS___.Object: ___.ri(IMPORTS___, 'Object')
                                , x5___ = sub.prototype_v___? sub.prototype:
                              sub.v___('prototype'), x6___ = ___.iM([ 'value',
                                  sub, 'writable', true, 'configurable', true ]
                              ), x4___.defineProperty_m___?
                                x4___.defineProperty(x5___, 'constructor',
                                x6___): x4___.m___('defineProperty', [ x5___,
                                  'constructor', x6___ ]);
                            }
                            ___.f(inherit, 'inherit');
                          }
                          {
                            function definePropertiesAwesomely(object,
                              propDescs) {
                              var prop, x0___, x1___, x2___, desc, k, x3___,
                              x4___, x5___, x6___, x7___, x8___, x9___, x10___,
                              x11___, x12___, x13___, x14___, x15___, x16___,
                              x17___, x18___, x19___, x20___, x21___, x22___,
                              x23___, x24___, x25___;;;
                              {
                                x2___ = Object(propDescs).e___();
                                for (x1___ in x2___) {
                                  if (typeof x1___ === 'number' || '' + +x1___
                                    === x1___) { prop = x1___; } else {
                                    if (/^NUM___/.test(x1___) &&
                                      /__$/.test(x1___)) { continue; }
                                    x0___ = x1___.match(/([\s\S]*)_e___$/);
                                    if (!x0___ || !x2___[ x1___ ]) { continue;
                                    }
                                    prop = x0___[ 1 ];
                                  }
                                  {
                                    desc = ___.iM([ ]);
                                    x5___ = Object(propDescs.v___(prop)).e___()
                                      ;
                                    for (x4___ in x5___) {
                                      if (typeof x4___ === 'number' || '' +
                                          +x4___ === x4___) { k = x4___; } else
                                        {
                                        if (/^NUM___/.test(x4___) &&
                                            /__$/.test(x4___)) { continue; }
                                        x3___ = x4___.match(/([\s\S]*)_e___$/);
                                        if (!x3___ || !x5___[ x4___ ]) {
                                          continue; }
                                        k = x3___[ 1 ];
                                      }
                                      {
                                        desc.w___(k, propDescs.v___(prop)
                                            .v___(k));
                                      }
                                    }
                                    if (___.i('' + 'get', desc) || ___.i('' +
                                        'set', desc)) {
                                      x7___ = desc, x6___ =
                                        (desc.enumerable_v___? desc.enumerable:
                                        desc.v___('enumerable')) &&
                                        canHaveEnumerableAccessors,
                                      x7___.enumerable_w___ === x7___?
                                        (x7___.enumerable = x6___):
                                      x7___.w___('enumerable', x6___);
                                    }
                                    if (desc.extendedAccessors_v___?
                                      desc.extendedAccessors:
                                      desc.v___('extendedAccessors')) {
                                      desc.c___('extendedAccessors');
                                      ___.f(function (prop, extGet, extSet) {
                                          var x0___, x1___, x2___, x3___,
                                          x4___, x5___, x6___, x7___;
                                          if (extGet) {
                                            x3___ = desc, x2___ = (x0___ =
                                              IMPORTS___.cajaVM_v___?
                                              IMPORTS___.cajaVM:
                                              ___.ri(IMPORTS___, 'cajaVM'),
                                              x1___ = ___.f(function () {
                                                  var dis___ = this &&
                                                    this.___? void 0: this;
                                                  return extGet.call_m___?
                                                    extGet.call(dis___, prop):
                                                  extGet.m___('call', [ dis___,
                                                      prop ]);
                                                }), x0___.def_m___?
                                              x0___.def(x1___):
                                              x0___.m___('def', [ x1___ ])),
                                            x3___.get_w___ === x3___?
                                              (x3___.get = x2___):
                                            x3___.w___('get', x2___);
                                          }
                                          if (desc.set_v___? desc.set:
                                            desc.v___('set')) {
                                            x7___ = desc, x6___ = (x4___ =
                                              IMPORTS___.cajaVM_v___?
                                              IMPORTS___.cajaVM:
                                              ___.ri(IMPORTS___, 'cajaVM'),
                                              x5___ = ___.f(function (value) {
                                                  var dis___ = this &&
                                                    this.___? void 0: this;
                                                  return extSet.call_m___?
                                                    extSet.call(dis___, value,
                                                    prop): extSet.m___('call',
                                                    [ dis___, value, prop ]);
                                                }), x4___.def_m___?
                                              x4___.def(x5___):
                                              x4___.m___('def', [ x5___ ])),
                                            x7___.set_w___ === x7___?
                                              (x7___.set = x6___):
                                            x7___.w___('set', x6___);
                                          }
                                        }).i___(prop, desc.get_v___? desc.get:
                                        desc.v___('get'), desc.set_v___?
                                          desc.set: desc.v___('set'));
                                    }
                                    if ((desc.get_v___? desc.get:
                                        desc.v___('get')) && ! (x8___ =
                                        IMPORTS___.Object_v___?
                                        IMPORTS___.Object: ___.ri(IMPORTS___,
                                          'Object'), x9___ = desc.get_v___?
                                        desc.get: desc.v___('get'),
                                        x8___.isFrozen_m___?
                                        x8___.isFrozen(x9___):
                                        x8___.m___('isFrozen', [ x9___ ]))) {
                                      if (typeof IMPORTS___.v___('console') !==
                                        'undefined') {
                                        x10___ = IMPORTS___.console_v___?
                                          IMPORTS___.console:
                                        ___.ri(IMPORTS___, 'console'), x11___ =
                                          prop, x12___ = object,
                                        x10___.warn_m___?
                                          x10___.warn('Getter for ', x11___,
                                          ' of ', x12___,
                                          ' is not frozen; fixing.'):
                                        x10___.m___('warn', [ 'Getter for ',
                                            x11___, ' of ', x12___,
                                            ' is not frozen; fixing.' ]);
                                      }
                                      x13___ = IMPORTS___.cajaVM_v___?
                                        IMPORTS___.cajaVM: ___.ri(IMPORTS___,
                                        'cajaVM'), x14___ = desc.get_v___?
                                        desc.get: desc.v___('get'),
                                      x13___.def_m___? x13___.def(x14___):
                                      x13___.m___('def', [ x14___ ]);
                                    }
                                    if ((desc.set_v___? desc.set:
                                        desc.v___('set')) && ! (x15___ =
                                        IMPORTS___.Object_v___?
                                        IMPORTS___.Object: ___.ri(IMPORTS___,
                                          'Object'), x16___ = desc.set_v___?
                                        desc.set: desc.v___('set'),
                                        x15___.isFrozen_m___?
                                        x15___.isFrozen(x16___):
                                        x15___.m___('isFrozen', [ x16___ ]))) {
                                      if (typeof IMPORTS___.v___('console') !==
                                        'undefined') {
                                        x17___ = IMPORTS___.console_v___?
                                          IMPORTS___.console:
                                        ___.ri(IMPORTS___, 'console'), x18___ =
                                          prop, x19___ = object,
                                        x17___.warn_m___?
                                          x17___.warn('Setter for ', x18___,
                                          ' of ', x19___,
                                          ' is not frozen; fixing.'):
                                        x17___.m___('warn', [ 'Setter for ',
                                            x18___, ' of ', x19___,
                                            ' is not frozen; fixing.' ]);
                                      }
                                      x20___ = IMPORTS___.cajaVM_v___?
                                        IMPORTS___.cajaVM: ___.ri(IMPORTS___,
                                        'cajaVM'), x21___ = desc.set_v___?
                                        desc.set: desc.v___('set'),
                                      x20___.def_m___? x20___.def(x21___):
                                      x20___.m___('def', [ x21___ ]);
                                    }
                                    x22___ = IMPORTS___.Object_v___?
                                      IMPORTS___.Object: ___.ri(IMPORTS___,
                                      'Object'), x23___ = object, x24___ =
                                      prop, x25___ = desc,
                                    x22___.defineProperty_m___?
                                      x22___.defineProperty(x23___, x24___,
                                      x25___): x22___.m___('defineProperty', [
                                        x23___, x24___, x25___ ]);
                                  }
                                }
                              }
                            }
                            ___.f(definePropertiesAwesomely,
                              'definePropertiesAwesomely');
                          }
                          {
                            function forOwnKeys(obj, fn) {
                              var i, x0___, x1___, x2___, x3___, x4___, x5___,
                              x6___, x7___;;
                              {
                                x2___ = Object(obj).e___();
                                for (x1___ in x2___) {
                                  if (typeof x1___ === 'number' || '' + +x1___
                                    === x1___) { i = x1___; } else {
                                    if (/^NUM___/.test(x1___) &&
                                        /__$/.test(x1___)) { continue; }
                                    x0___ = x1___.match(/([\s\S]*)_e___$/);
                                    if (!x0___ || !x2___[ x1___ ]) { continue;
                                    }
                                    i = x0___[ 1 ];
                                  }
                                  {
                                    if (! (x5___ = (x4___ = (x3___ =
                                              IMPORTS___.Object_v___?
                                              IMPORTS___.Object:
                                            ___.ri(IMPORTS___, 'Object'),
                                            x3___.prototype_v___?
                                              x3___.prototype:
                                            x3___.v___('prototype')),
                                          x4___.hasOwnProperty_v___?
                                            x4___.hasOwnProperty:
                                          x4___.v___('hasOwnProperty')), x6___
                                          = obj, x7___ = i, x5___.call_m___?
                                          x5___.call(x6___, x7___):
                                        x5___.m___('call', [ x6___, x7___ ])))
                                      { continue; }
                                    fn.i___(i, obj.v___(i));
                                  }
                                }
                              }
                            }
                            ___.f(forOwnKeys, 'forOwnKeys');
                          }
                          {
                            function identity(x) { return x; }
                            ___.f(identity, 'identity');
                          }
                          {
                            function defaultToEmptyStr(x) {
                              return x || '';
                            }
                            ___.f(defaultToEmptyStr, 'defaultToEmptyStr');
                          }
                          {
                            function arrayRemove(array, from, to) {
                              var rest, x0___, x1___, x2___, x3___, x4___,
                              x5___, x6___;
                              rest = (x1___ = array, x0___ = (to || from) + 1
                                  || array.length, x1___.slice_m___?
                                  x1___.slice(x0___): x1___.m___('slice', [
                                    x0___ ]));
                              x3___ = array, x2___ = from < 0? array.length +
                                from: from, x3___.length_w___ === x3___?
                                (x3___.length = x2___): x3___.w___('length',
                                x2___);
                              return x4___ = array.push_v___? array.push:
                              array.v___('push'), x5___ = array, x6___ = rest,
                              x4___.apply_m___? x4___.apply(x5___, x6___):
                              x4___.m___('apply', [ x5___, x6___ ]);
                            }
                            ___.f(arrayRemove, 'arrayRemove');
                          }
                          {
                            function Html(htmlFragment) {
                              var dis___ = this && this.___? void 0: this;
                              var h, x0___, x1___;
                              h = (IMPORTS___.String_v___? IMPORTS___.String:
                                ___.ri(IMPORTS___, 'String')).i___(htmlFragment
                                  == null ? '': htmlFragment);
                              safeHTMLTable.put_m___? safeHTMLTable.put(dis___,
                                htmlFragment): safeHTMLTable.m___('put', [
                                  dis___, htmlFragment ]);
                              return x0___ = IMPORTS___.cajaVM_v___?
                                IMPORTS___.cajaVM: ___.ri(IMPORTS___, 'cajaVM')
                                , x1___ = dis___, x0___.def_m___?
                                x0___.def(x1___): x0___.m___('def', [ x1___ ]);
                            }
                            ___.f(Html, 'Html');
                          }
                          {
                            function htmlToString() {
                              var dis___ = this && this.___? void 0: this;
                              return safeHTMLTable.get_m___?
                                safeHTMLTable.get(dis___):
                              safeHTMLTable.m___('get', [ dis___ ]);
                            }
                            ___.f(htmlToString, 'htmlToString');
                          }
                          {
                            function safeHtml(htmlFragment) {
                              var x0___, x1___;
                              return htmlFragment instanceof Html?
                                safeHTMLTable.get_m___?
                                safeHTMLTable.get(htmlFragment):
                              safeHTMLTable.m___('get', [ htmlFragment ]):
                              (x0___ = IMPORTS___.html_v___? IMPORTS___.html:
                               ___.ri(IMPORTS___, 'html'), x1___ =
                               (IMPORTS___.String_v___? IMPORTS___.String:
                                ___.ri(IMPORTS___, 'String')).i___(htmlFragment
                                   == null ? '': htmlFragment),
                               x0___.escapeAttrib_m___?
                               x0___.escapeAttrib(x1___):
                               x0___.m___('escapeAttrib', [ x1___ ]));
                            }
                            ___.f(safeHtml, 'safeHtml');
                          }
                          {
                            function blessHtml(htmlFragment) {
                              return htmlFragment instanceof Html?
                                htmlFragment: new Html.new___(htmlFragment);
                            }
                            ___.f(blessHtml, 'blessHtml');
                          }
                          {
                            function isValidId(s) {
                              return ! (FORBIDDEN_ID_PATTERN.test_m___?
                                FORBIDDEN_ID_PATTERN.test(s):
                                FORBIDDEN_ID_PATTERN.m___('test', [ s ])) &&
                                (VALID_ID_PATTERN.test_m___?
                                  VALID_ID_PATTERN.test(s):
                                VALID_ID_PATTERN.m___('test', [ s ]));
                            }
                            ___.f(isValidId, 'isValidId');
                          }
                          {
                            function isValidIdList(s) {
                              return ! (FORBIDDEN_ID_LIST_PATTERN.test_m___?
                                  FORBIDDEN_ID_LIST_PATTERN.test(s):
                                FORBIDDEN_ID_LIST_PATTERN.m___('test', [ s ]))
                                && (VALID_ID_LIST_PATTERN.test_m___?
                                  VALID_ID_LIST_PATTERN.test(s):
                                VALID_ID_LIST_PATTERN.m___('test', [ s ]));
                            }
                            ___.f(isValidIdList, 'isValidIdList');
                          }
                          {
                            function trimHTML5Spaces(input) {
                              var x0___, x1___;
                              return x1___ = input, x0___ = new
                                RegExp.new___('^[ \\t\\r\\n\\f]+|[ \\t\\r\\n\\f]+$',
                                'g'), x1___.replace_m___? x1___.replace(x0___,
                                ''): x1___.m___('replace', [ x0___, '' ]);
                            }
                            ___.f(trimHTML5Spaces, 'trimHTML5Spaces');
                          }
                          {
                            function mimeTypeForAttr(tagName, attribName) {
                              if (attribName === 'src') {
                                if (tagName === 'img') { return 'image/*'; }
                                if (tagName === 'script') {
                                  return 'text/javascript';
                                }
                              }
                              return '*/*';
                            }
                            ___.f(mimeTypeForAttr, 'mimeTypeForAttr');
                          }
                          {
                            function assert(cond) {
                              var x0___, x1___;
                              if (!cond) {
                                if (typeof IMPORTS___.v___('console') !==
                                    'undefined') {
                                  x0___ = IMPORTS___.console_v___?
                                    IMPORTS___.console: ___.ri(IMPORTS___,
                                    'console'), x0___.error_m___?
                                    x0___.error('domita assertion failed'):
                                  x0___.m___('error', [
                                      'domita assertion failed' ]);
                                  x1___ = IMPORTS___.console_v___?
                                    IMPORTS___.console: ___.ri(IMPORTS___,
                                    'console'), x1___.trace_m___? x1___.trace()
                                    : x1___.m___('trace', [ ]);
                                }
                                throw new (IMPORTS___.Error_v___?
                                    IMPORTS___.Error: ___.ri(IMPORTS___,
                                    'Error')).new___('Domita assertion failed')
                                  ;
                              }
                            }
                            ___.f(assert, 'assert');
                          }
                          {
                            function tameSetAndClear(target, set, clear,
                              setName, clearName) {
                              var ids, tameSet, tameClear, x0___, x1___, x2___,
                              x3___;
                              {
                                function tameSet(action, delayMillis) {
                                  var id, actionWrapper, tamed;
                                  if (action) {
                                    if (typeof action !== 'function') {
                                      throw new (IMPORTS___.Error_v___?
                                        IMPORTS___.Error: ___.ri(IMPORTS___,
                                          'Error')).new___((setName +
                                            ' called with a ' + typeof action +
                                            '.' +
                                            '  Please pass a function instead of a string of JavaScript')
                                      );
                                    }
                                    actionWrapper = (function () {
                                        function actionWrapper$_var() {
                                          action.i___(); }
                                        return ___.f(actionWrapper$_var,
                                          'actionWrapper$_var');
                                      })();
                                    id = set.i___(actionWrapper, delayMillis |
                                      0);
                                  } else {
                                    id = IMPORTS___.undefined_v___?
                                      IMPORTS___.undefined: ___.ri(IMPORTS___,
                                      'undefined');
                                  }
                                  tamed = ___.iM([ ]);
                                  ids.set_m___? ids.set(tamed, id):
                                  ids.m___('set', [ tamed, id ]);
                                  return tamed;
                                }
                                ___.f(tameSet, 'tameSet');
                              }
                              {
                                function tameClear(id) {
                                  var feral;
                                  if (typeof id !== 'object' || id == null) {
                                    return; }
                                  feral = ids.get_m___? ids.get(id):
                                  ids.m___('get', [ id ]);
                                  if (feral !== (IMPORTS___.undefined_v___?
                                      IMPORTS___.undefined: ___.ri(IMPORTS___,
                                        'undefined'))) clear.i___(feral);
                                }
                                ___.f(tameClear, 'tameClear');
                              }
                              ids = new (IMPORTS___.WeakMap_v___?
                                  IMPORTS___.WeakMap: ___.ri(IMPORTS___,
                                  'WeakMap')).new___;
                              makeFunctionAccessible.i___(set);
                              makeFunctionAccessible.i___(clear);
                              target.w___(setName, (x0___ =
                                  IMPORTS___.cajaVM_v___? IMPORTS___.cajaVM:
                                  ___.ri(IMPORTS___, 'cajaVM'), x1___ =
                                  tameSet, x0___.def_m___? x0___.def(x1___):
                                  x0___.m___('def', [ x1___ ])));
                              target.w___(clearName, (x2___ =
                                  IMPORTS___.cajaVM_v___? IMPORTS___.cajaVM:
                                  ___.ri(IMPORTS___, 'cajaVM'), x3___ =
                                  tameClear, x2___.def_m___? x2___.def(x3___):
                                  x2___.m___('def', [ x3___ ])));
                              return target;
                            }
                            ___.f(tameSetAndClear, 'tameSetAndClear');
                          }
                          {
                            function makeScrollable(bridal, element) {
                              var overflow, x0___, x1___, x2___, x3___, x4___;
                              overflow = (x3___ = (x1___ = bridal, x2___ =
                                  element, x0___ = void 0,
                                  x1___.getComputedStyle_m___?
                                  x1___.getComputedStyle(x2___, x0___):
                                  x1___.m___('getComputedStyle', [ x2___, x0___
                                    ])), x3___.overflow_v___? x3___.overflow:
                                x3___.v___('overflow'));
                              switch (overflow && (overflow.toLowerCase_m___?
                                  overflow.toLowerCase():
                                  overflow.m___('toLowerCase', [ ]))) {
                              case 'visible':
                              case 'hidden':
                                makeDOMAccessible.i___(element.style_v___?
                                    element.style: element.v___('style'));
                                x4___ = element.style_v___? element.style:
                                element.v___('style'), x4___.overflow_w___ ===
                                  x4___? (x4___.overflow = 'auto'):
                                x4___.w___('overflow', 'auto');
                                break;
                              }
                            }
                            ___.f(makeScrollable, 'makeScrollable');
                          }
                          {
                            function tameScrollTo(element, x, y) {
                              if (x !== +x || y !== +y || x < 0 || y < 0) {
                                throw new (IMPORTS___.Error_v___?
                                  IMPORTS___.Error: ___.ri(IMPORTS___, 'Error')
                                ).new___(('Cannot scroll to ' + x + ':' +
                                    typeof x + ',' + y + ' : ' + typeof y));
                              }
                              element.scrollLeft_w___ === element?
                                (element.scrollLeft = x):
                              element.w___('scrollLeft', x);
                              element.scrollTop_w___ === element?
                                (element.scrollTop = y):
                              element.w___('scrollTop', y);
                            }
                            ___.f(tameScrollTo, 'tameScrollTo');
                          }
                          {
                            function tameScrollBy(element, dx, dy) {
                              var x0___, x1___, x2___, x3___;
                              if (dx !== +dx || dy !== +dy) {
                                throw new (IMPORTS___.Error_v___?
                                    IMPORTS___.Error: ___.ri(IMPORTS___,
                                    'Error')).new___(('Cannot scroll by ' + dx
                                    + ':' + typeof dx + ', ' + dy + ':' +
                                    typeof dy));
                              }
                              x1___ = element, x0___ =
                                (element.scrollLeft_v___? element.scrollLeft:
                                element.v___('scrollLeft')) + dx,
                              x1___.scrollLeft_w___ === x1___?
                                (x1___.scrollLeft = x0___):
                              x1___.w___('scrollLeft', x0___);
                              x3___ = element, x2___ = (element.scrollTop_v___?
                                  element.scrollTop: element.v___('scrollTop'))
                                + dy, x3___.scrollTop_w___ === x3___?
                                (x3___.scrollTop = x2___):
                              x3___.w___('scrollTop', x2___);
                            }
                            ___.f(tameScrollBy, 'tameScrollBy');
                          }
                          {
                            function guessPixelsFromCss(cssStr) {
                              var m, x0___, x1___;
                              if (!cssStr) { return 0; }
                              m = (x1___ = cssStr, x0___ = new
                                RegExp.new___('^([0-9]+)'), x1___.match_m___?
                                x1___.match(x0___): x1___.m___('match', [ x0___
                                  ]));
                              return m? +m[ +1 ]: 0;
                            }
                            ___.f(guessPixelsFromCss, 'guessPixelsFromCss');
                          }
                          {
                            function tameResizeTo(element, w, h) {
                              var x0___, x1___, x2___, x3___;
                              if (w !== +w || h !== +h) {
                                throw new (IMPORTS___.Error_v___?
                                    IMPORTS___.Error: ___.ri(IMPORTS___,
                                    'Error')).new___(('Cannot resize to ' + w +
                                    ':' + typeof w + ', ' + h + ':' + typeof h)
                                );
                              }
                              makeDOMAccessible.i___(element.style_v___?
                                  element.style: element.v___('style'));
                              x0___ = element.style_v___? element.style:
                              element.v___('style'), x1___ = w + 'px',
                              x0___.width_w___ === x0___? (x0___.width = x1___)
                                : x0___.w___('width', x1___);
                              x2___ = element.style_v___? element.style:
                              element.v___('style'), x3___ = h + 'px',
                              x2___.height_w___ === x2___? (x2___.height =
                                x3___): x2___.w___('height', x3___);
                            }
                            ___.f(tameResizeTo, 'tameResizeTo');
                          }
                          {
                            function tameResizeBy(element, dw, dh) {
                              var style, x0___, x1___, x2___, x3___, x4___,
                              x5___, extraHeight, extraWidth, goalHeight,
                              goalWidth, h, w, x6___, x7___, x8___, x9___,
                              x10___, x11___, x12___, x13___, hError, x14___,
                              x15___, x16___, x17___, wError, x18___, x19___,
                              x20___, x21___;
                              if (dw !== +dw || dh !== +dh) {
                                throw new (IMPORTS___.Error_v___?
                                  IMPORTS___.Error: ___.ri(IMPORTS___, 'Error')
                                ).new___(('Cannot resize by ' + dw + ':' +
                                    typeof dw + ', ' + dh + ':' + typeof dh));
                              }
                              if (!dw && !dh) { return; }
                              style =
                                makeDOMAccessible.i___(element.currentStyle_v___?
                                element.currentStyle:
                                element.v___('currentStyle'));
                              if (!style) {
                                style = makeDOMAccessible.i___((x3___ = (x0___
                                      = IMPORTS___.bridalMaker_v___?
                                      IMPORTS___.bridalMaker:
                                      ___.ri(IMPORTS___, 'bridalMaker'), x1___
                                      = element, x2___ = makeDOMAccessible,
                                      x0___.getWindow_m___?
                                      x0___.getWindow(x1___, x2___):
                                      x0___.m___('getWindow', [ x1___, x2___ ])
                                    ), x5___ = element, x4___ = void 0,
                                    x3___.getComputedStyle_m___?
                                    x3___.getComputedStyle(x5___, x4___):
                                    x3___.m___('getComputedStyle', [ x5___,
                                        x4___ ])));
                              }
                              makeDOMAccessible.i___(element.style_v___?
                                  element.style: element.v___('style'));
                              extraHeight =
                                guessPixelsFromCss.i___(style.paddingBottom_v___?
                                style.paddingBottom:
                                style.v___('paddingBottom')) +
                                guessPixelsFromCss.i___(style.paddingTop_v___?
                                style.paddingTop: style.v___('paddingTop'));
                              extraWidth =
                                guessPixelsFromCss.i___(style.paddingLeft_v___?
                                  style.paddingLeft: style.v___('paddingLeft'))
                                +
                                guessPixelsFromCss.i___(style.paddingRight_v___?
                                style.paddingRight: style.v___('paddingRight'))
                                ;
                              goalHeight = (element.clientHeight_v___?
                                  element.clientHeight:
                                element.v___('clientHeight')) + dh;
                              goalWidth = (element.clientWidth_v___?
                                  element.clientWidth:
                                element.v___('clientWidth')) + dw;
                              h = goalHeight - extraHeight;
                              w = goalWidth - extraWidth;
                              if (dh) {
                                x6___ = element.style_v___? element.style:
                                element.v___('style'), x9___ = (x7___ =
                                    IMPORTS___.Math_v___? IMPORTS___.Math:
                                  ___.ri(IMPORTS___, 'Math'), x8___ = h,
                                  x7___.max_m___? x7___.max(0, x8___):
                                  x7___.m___('max', [ 0, x8___ ])) + 'px',
                                x6___.height_w___ === x6___? (x6___.height =
                                    x9___): x6___.w___('height', x9___);
                              }
                              if (dw) {
                                x10___ = element.style_v___? element.style:
                                element.v___('style'), x13___ = (x11___ =
                                  IMPORTS___.Math_v___? IMPORTS___.Math:
                                  ___.ri(IMPORTS___, 'Math'), x12___ = w,
                                  x11___.max_m___? x11___.max(0, x12___):
                                  x11___.m___('max', [ 0, x12___ ])) + 'px',
                                x10___.width_w___ === x10___? (x10___.width =
                                    x13___): x10___.w___('width', x13___);
                              }
                              if (dh && (element.clientHeight_v___?
                                  element.clientHeight:
                                  element.v___('clientHeight')) !== goalHeight)
                                {
                                hError = (element.clientHeight_v___?
                                    element.clientHeight:
                                  element.v___('clientHeight')) - goalHeight;
                                x14___ = element.style_v___? element.style:
                                element.v___('style'), x17___ = (x15___ =
                                  IMPORTS___.Math_v___? IMPORTS___.Math:
                                  ___.ri(IMPORTS___, 'Math'), x16___ = h -
                                  hError, x15___.max_m___? x15___.max(0,
                                    x16___): x15___.m___('max', [ 0, x16___ ]))
                                  + 'px', x14___.height_w___ === x14___?
                                  (x14___.height = x17___):
                                x14___.w___('height', x17___);
                              }
                              if (dw && (element.clientWidth_v___?
                                  element.clientWidth:
                                  element.v___('clientWidth')) !== goalWidth) {
                                wError = (element.clientWidth_v___?
                                    element.clientWidth:
                                  element.v___('clientWidth')) - goalWidth;
                                x18___ = element.style_v___? element.style:
                                element.v___('style'), x21___ = (x19___ =
                                    IMPORTS___.Math_v___? IMPORTS___.Math:
                                  ___.ri(IMPORTS___, 'Math'), x20___ = w -
                                    wError, x19___.max_m___? x19___.max(0,
                                    x20___): x19___.m___('max', [ 0, x20___ ]))
                                  + 'px', x18___.width_w___ === x18___?
                                  (x18___.width = x21___): x18___.w___('width',
                                  x21___);
                              }
                            }
                            ___.f(tameResizeBy, 'tameResizeBy');
                          }
                          {
                            function attachDocument(idSuffix, naiveUriPolicy,
                              containerNode, optTargetAttributePresets, taming)
                              {
                              {
                                ___.deodorize(arguments, -6);
                                var a___ = ___.args(arguments);
                              }
                              var domicile, pluginId, document, docEl, bridal,
                              window, x0___, x1___, x2___, elementPolicies,
                              x3___, x4___, x5___, x6___, forceAutocompleteOff,
                              TameNodeConf, TameNodeT, np, x7___, x8___,
                              tamingProxies, finishNode, nodeMethod,
                              sanitizeStyleProperty, sanitizeStyleAttrValue,
                              sanitizeHtml, sanitizeAttrs, htmlSanitizer,
                              x9___, x10___, unsuffix, ID_LIST_PARTS_PATTERN,
                              virtualizeAttributeValue, tameInnerHtml,
                              innerHtmlTamer, x11___, x12___, rewriteAttribute,
                              P_UNIMPLEMENTED, x13___, x14___, P_constant,
                              PropertyTaming, x15___, x16___, NP,
                              NP_tameDescendant, nodeExpandos,
                              getNodeExpandoStorage, makeTameNodeByType,
                              defaultTameNode, tameRelatedNode, x17___, x18___,
                              getNodeListLength, mixinNodeList,
                              rebuildTameListConstructors, makeTameNodeList,
                              TameNodeList, x19___, x20___,
                              makeTameOptionsList, TameOptionsList, x21___,
                              x22___, fakeNodeList, mixinHTMLCollection,
                              tameHTMLCollection, tameGetElementsByTagName,
                              tameGetElementsByClassName,
                              makeEventHandlerWrapper, NOT_EDITABLE,
                              UNSAFE_TAGNAME, UNKNOWN_TAGNAME,
                              INDEX_SIZE_ERROR, tameAddEventListener,
                              tameRemoveEventListener, nodeClasses,
                              tamingClassesByElement, inertCtor, TameNode,
                              x23___, x24___, x25___, x26___,
                              tameNodePublicMembers, x27___, x28___,
                              TameBackedNode, x29___, x30___, x31___, x32___,
                              x33___, x34___, x35___, x36___, x37___, x38___,
                              x39___, x40___, x41___, x42___, x43___, x44___,
                              x45___, x46___, x47___, x48___, x49___, x50___,
                              x51___, x52___, x53___, x54___, x55___, x56___,
                              x57___, x58___, TamePseudoNode, x59___, x60___,
                              x61___, x62___, x63___, x64___, x65___, x66___,
                              x67___, x68___, x69___, x70___,
                              commonElementPropertyHandlers,
                              makeRestrictedNodeType, TameOpaqueNode,
                              TameForeignNode, x71___, x72___, TameTextNode,
                              textAccessor, x73___, x74___, TameCommentNode,
                              x75___, x76___, lookupAttribute,
                              getAttributeType, getLoaderType, getUriEffect,
                              TameBackedAttributeNode, valueAccessor, x77___,
                              x78___, notImplementedNodeMethod, x79___, x80___,
                              x81___, x82___,
                              registerElementScriptAttributeHandlers,
                              TameElement, x83___, x84___, x85___, x86___,
                              x87___, x88___, x89___, x90___, x91___, x92___,
                              x93___, x94___, x95___, x96___, x97___, x98___,
                              x99___, x100___, x101___, x102___, x103___,
                              x104___, x105___, x106___, x107___, x108___,
                              x109___, x110___, x111___, x112___, x113___,
                              x114___, innerTextOf, innerTextProp, tagNameAttr,
                              x115___, x116___, x117___, x118___, x119___,
                              x120___, x121___, x122___, x123___, x124___,
                              x125___, x126___, x127___, defineElement,
                              x128___, x129___, TameBodyElement,
                              FormElementAndExpandoProxyHandler, x130___,
                              x131___, x132___, x133___, x134___,
                              TameFormElement, x135___, x136___, x137___,
                              x138___, x139___, x140___, x141___, x142___,
                              x143___, x144___, x145___, x146___, x147___,
                              x148___, x149___, x150___, P_blacklist,
                              TameIFrameElement, x151___, x152___, x153___,
                              x154___, x155___, x156___, toInt,
                              TameInputElement, x157___, x158___, x159___,
                              x160___, x161___, x162___, x163___, x164___,
                              x165___, x166___, x167___, x168___, x169___,
                              x170___, x171___, x172___, x173___, x174___,
                              x175___, x176___, x177___,
                              dynamicCodeDispatchMaker, TameScriptElement,
                              TameTableCompElement, x178___, x179___, x180___,
                              x181___, requireIntIn, TameTableRowElement,
                              x182___, x183___, x184___, x185___,
                              TameTableElement, x186___, x187___, x188___,
                              x189___, x190___, x191___, x192___, x193___,
                              x194___, x195___, x196___, x197___, x198___,
                              x199___, x200___, x201___, x202___, x203___,
                              x204___, x205___, x206___, x207___, x208___,
                              x209___, x210___, fromInt, tameEvent, ep,
                              x211___, x212___, EP_RELATED, P_e_view,
                              TameEvent, x213___, x214___, x215___, x216___,
                              x217___, x218___, TameCustomHTMLEvent, x219___,
                              x220___, x221___, x222___, viewProperties,
                              TameHTMLDocument, x223___, x224___, x225___,
                              x226___, x227___, x228___, x229___, x230___,
                              x231___, x232___, x233___, x234___, x235___,
                              x236___, x237___, x238___, x239___, x240___,
                              x241___, x242___, x243___, x244___, x245___,
                              x246___, x247___, x248___, x249___, x250___,
                              x251___, x252___, x253___, x254___, x255___,
                              x256___, x257___, x258___, x259___, x260___,
                              x261___, x262___, x263___, x264___, x265___,
                              x266___, x267___, x268___, x269___, x270___,
                              x271___, x272___, x273___, x274___, x275___,
                              x276___, x277___, x278___, x279___, x280___,
                              x281___, x282___, x283___, x284___, x285___,
                              x286___, x287___, x288___, x289___, x290___,
                              x291___, x292___, x293___, x294___, x295___,
                              x296___, x297___, x298___, x299___, x300___,
                              x301___, x302___, x303___, x304___, x305___,
                              x306___, x307___, x308___, x309___, x310___,
                              x311___, x312___, x313___, x314___, x315___,
                              x316___, TameStyle, TameComputedStyle,
                              buildTameStyle, x317___, x318___, x319___,
                              x320___, x321___, x322___, x323___, x324___,
                              x325___, x326___, x327___, x328___, x329___,
                              x330___, x331___, x332___, x333___, x334___,
                              x335___, x336___, x337___, x338___, x339___,
                              x340___, x341___, x342___, x343___, x344___,
                              x345___, x346___, x347___, x348___, x349___,
                              x350___, x351___, x352___, idClass,
                              idClassPattern, x353___, x354___, x355___,
                              x356___, x357___, x358___, x359___, x360___,
                              x361___, x362___, x363___, x364___, x365___,
                              x366___, x367___, x368___, x369___, tameDocument,
                              navigator, tameNavigator, x370___, x371___,
                              PSEUDO_ELEMENT_WHITELIST, TameWindow, x372___,
                              x373___, TameDefaultView, x374___, x375___,
                              x376___, x377___, x378___, x379___, x380___,
                              x381___, x382___, x383___, x384___, x385___,
                              x386___, x387___, x388___, x389___, x390___,
                              x391___, x392___, x393___, x394___, x395___,
                              x396___, x397___, tameWindow, tameDefaultView,
                              x398___, windowProps, wpLen, i, prop, x399___,
                              x400___, x401___, name, x402___, x403___,
                              x404___, ctor, x405___, x406___, x407___,
                              x408___, x409___, x410___, x411___, x412___,
                              allDomNodeClasses, defaultNodeClassCtor, i,
                              className, x413___, x414___, x415___, x416___,
                              x417___, x418___, x419___;
                              {
                                function forceAutocompleteOff(attribs) {
                                  var a, i, n, x0___, x1___, x2___;
                                  a = [ ];
                                  for (i = 0, n = attribs.length; i < n; i = i
                                      + 2) {
                                    if (attribs[ + +i ] !== 'autocomplete') {
                                      x2___ = a, x0___ = attribs[ + +i ], x1___
                                        = attribs[ + (+i + 1) ],
                                      x2___.push_m___? x2___.push(x0___, x1___)
                                        : x2___.m___('push', [ x0___, x1___ ]);
                                    }
                                  }
                                  a.push_m___? a.push('autocomplete', 'off'):
                                  a.m___('push', [ 'autocomplete', 'off' ]);
                                  return a;
                                }
                                ___.f(forceAutocompleteOff,
                                  'forceAutocompleteOff');
                              }
                              {
                                function finishNode(node) {
                                  var feral, x0___, x1___, x2___, x3___,
                                  proxiedNode, x4___, x5___, x6___, x7___,
                                  x8___, x9___, x10___;
                                  feral = (x0___ = np.i___(node),
                                    x0___.feral_v___? x0___.feral:
                                    x0___.v___('feral'));
                                  if (x1___ = IMPORTS___.domitaModules_v___?
                                    IMPORTS___.domitaModules:
                                    ___.ri(IMPORTS___, 'domitaModules'),
                                    x1___.proxiesAvailable_v___?
                                    x1___.proxiesAvailable:
                                    x1___.v___('proxiesAvailable')) {
                                    x2___ = IMPORTS___.Object_v___?
                                      IMPORTS___.Object: ___.ri(IMPORTS___,
                                      'Object'), x3___ = node,
                                    x2___.freeze_m___? x2___.freeze(x3___):
                                    x2___.m___('freeze', [ x3___ ]);
                                    proxiedNode = (x4___ =
                                        IMPORTS___.Proxy_v___?
                                        IMPORTS___.Proxy: ___.ri(IMPORTS___,
                                        'Proxy'), x6___ = (x5___ =
                                        np.i___(node), x5___.proxyHandler_v___?
                                        x5___.proxyHandler:
                                        x5___.v___('proxyHandler')), x7___ =
                                        node, x4___.create_m___?
                                        x4___.create(x6___, x7___):
                                      x4___.m___('create', [ x6___, x7___ ]));
                                    np.i___(node).c___('proxyHandler');
                                    ExpandoProxyHandler.register_m___?
                                      ExpandoProxyHandler.register(proxiedNode,
                                      node):
                                    ExpandoProxyHandler.m___('register', [
                                        proxiedNode, node ]);
                                    TameNodeConf.confide_m___?
                                      TameNodeConf.confide(proxiedNode, node):
                                    TameNodeConf.m___('confide', [ proxiedNode,
                                        node ]);
                                    tamingProxies.set_m___?
                                      tamingProxies.set(node, proxiedNode):
                                    tamingProxies.m___('set', [ node,
                                        proxiedNode ]);
                                    node = proxiedNode;
                                  }
                                  if (feral) {
                                    if ((node.nodeType_v___? node.nodeType:
                                        node.v___('nodeType')) === 1) {
                                      taming.tamesTo_m___?
                                        taming.tamesTo(feral, node):
                                      taming.m___('tamesTo', [ feral, node ]);
                                    } else {
                                      taming.reTamesTo_m___?
                                        taming.reTamesTo(feral, node):
                                      taming.m___('reTamesTo', [ feral, node ])
                                        ;
                                    }
                                  } else {
                                    x9___ = taming, x8___ = ___.iM([ ]), x10___
                                      = node, x9___.tamesTo_m___?
                                      x9___.tamesTo(x8___, x10___):
                                    x9___.m___('tamesTo', [ x8___, x10___ ]);
                                  }
                                  return node;
                                }
                                ___.f(finishNode, 'finishNode');
                              }
                              {
                                function sanitizeStyleProperty(cssPropertyName,
                                  tokens) {
                                  var schema, x0___;
                                  schema = (IMPORTS___.cssSchema_v___?
                                    IMPORTS___.cssSchema: ___.ri(IMPORTS___,
                                      'cssSchema')).v___(cssPropertyName);
                                  if (!schema) {
                                    tokens.length_w___ === tokens?
                                      (tokens.length = 0):
                                    tokens.w___('length', 0);
                                    return false;
                                  }
                                  (IMPORTS___.sanitizeCssProperty_v___?
                                     IMPORTS___.sanitizeCssProperty:
                                   ___.ri(IMPORTS___, 'sanitizeCssProperty'))
                                  .i___(cssPropertyName, schema, tokens,
                                    naiveUriPolicy? ___.f(function (url) {
                                        var x0___, x1___, x2___, x3___;
                                        return uriRewrite.i___(naiveUriPolicy,
                                          url, (x1___ = (x0___ =
                                              IMPORTS___.html4_v___?
                                              IMPORTS___.html4:
                                              ___.ri(IMPORTS___, 'html4'),
                                              x0___.ueffects_v___?
                                              x0___.ueffects:
                                              x0___.v___('ueffects')),
                                            x1___.SAME_DOCUMENT_v___?
                                              x1___.SAME_DOCUMENT:
                                            x1___.v___('SAME_DOCUMENT')),
                                          (x3___ = (x2___ =
                                             IMPORTS___.html4_v___?
                                             IMPORTS___.html4:
                                             ___.ri(IMPORTS___, 'html4'),
                                             x2___.ltypes_v___? x2___.ltypes:
                                             x2___.v___('ltypes')),
                                           x3___.SANDBOXED_v___?
                                           x3___.SANDBOXED:
                                           x3___.v___('SANDBOXED')), ___.iM([
                                              'TYPE', 'CSS', 'CSS_PROP',
                                              cssPropertyName ]));
                                      }): null, (x0___ =
                                        domicile.pseudoLocation_v___?
                                        domicile.pseudoLocation:
                                      domicile.v___('pseudoLocation'),
                                      x0___.href_v___? x0___.href:
                                      x0___.v___('href')));
                                  return tokens.length !== 0;
                                }
                                ___.f(sanitizeStyleProperty,
                                  'sanitizeStyleProperty');
                              }
                              {
                                function sanitizeStyleAttrValue(styleAttrValue)
                                  {
                                  var sanitizedDeclarations;
                                  sanitizedDeclarations = [ ];
                                  (IMPORTS___.parseCssDeclarations_v___?
                                     IMPORTS___.parseCssDeclarations:
                                   ___.ri(IMPORTS___, 'parseCssDeclarations'))
                                  .i___((IMPORTS___.String_v___?
                                        IMPORTS___.String: ___.ri(IMPORTS___,
                                        'String')).i___(styleAttrValue),
                                    ___.iM([ 'declaration', (function () {
                                            function declaration$_lit(property,
                                              value) {
                                              var x0___, x1___;
                                              property =
                                                property.toLowerCase_m___?
                                                property.toLowerCase():
                                              property.m___('toLowerCase', [ ])
                                                ;
                                              sanitizeStyleProperty.i___(property,
                                                value);
                                              x1___ = sanitizedDeclarations,
                                              x0___ = property + ': ' +
                                                (value.join_m___?
                                                  value.join(' '):
                                                value.m___('join', [ ' ' ])),
                                              x1___.push_m___?
                                                x1___.push(x0___):
                                              x1___.m___('push', [ x0___ ]);
                                            }
                                            return ___.f(declaration$_lit,
                                              'declaration$_lit');
                                          })() ]));
                                  return sanitizedDeclarations.join_m___?
                                    sanitizedDeclarations.join(' ; '):
                                  sanitizedDeclarations.m___('join', [ ' ; ' ])
                                    ;
                                }
                                ___.f(sanitizeStyleAttrValue,
                                  'sanitizeStyleAttrValue');
                              }
                              {
                                function sanitizeHtml(htmlText) {
                                  var out;
                                  out = [ ];
                                  htmlSanitizer.i___(htmlText, out);
                                  return out.join_m___? out.join(''):
                                  out.m___('join', [ '' ]);
                                }
                                ___.f(sanitizeHtml, 'sanitizeHtml');
                              }
                              {
                                function sanitizeAttrs(tagName, attribs) {
                                  var n, needsTargetAttrib, x0___, x1___,
                                  x2___, i, attribName, value, atype,
                                  attribKey, x3___, x4___, x5___, x6___, x7___,
                                  x8___, x9___, x10___, x11___, x12___, x13___,
                                  x14___, x15___, x16___, x17___, x18___,
                                  x19___, policy;
                                  n = attribs.length;
                                  needsTargetAttrib = (x1___ = (x0___ =
                                      IMPORTS___.html4_v___? IMPORTS___.html4:
                                      ___.ri(IMPORTS___, 'html4'),
                                      x0___.ATTRIBS_v___? x0___.ATTRIBS:
                                      x0___.v___('ATTRIBS')), x2___ = tagName +
                                    '::target', x1___.hasOwnProperty_m___?
                                    x1___.hasOwnProperty(x2___):
                                    x1___.m___('hasOwnProperty', [ x2___ ]));
                                  for (i = 0; i < n; i = i + 2) {
                                    attribName = attribs[ + +i ];
                                    if ('target' === attribName) {
                                      needsTargetAttrib = false;
                                    }
                                    value = attribs[ + (+i + 1) ];
                                    atype = null;
                                    if ((attribKey = tagName + '::' +
                                        attribName, (x4___ = (x3___ =
                                            IMPORTS___.html4_v___?
                                            IMPORTS___.html4:
                                            ___.ri(IMPORTS___, 'html4'),
                                            x3___.ATTRIBS_v___? x3___.ATTRIBS:
                                            x3___.v___('ATTRIBS')), x5___ =
                                          attribKey, x4___.hasOwnProperty_m___?
                                          x4___.hasOwnProperty(x5___):
                                          x4___.m___('hasOwnProperty', [ x5___
                                            ]))) || (attribKey = '*::' +
                                        attribName, (x7___ = (x6___ =
                                              IMPORTS___.html4_v___?
                                              IMPORTS___.html4:
                                            ___.ri(IMPORTS___, 'html4'),
                                            x6___.ATTRIBS_v___? x6___.ATTRIBS:
                                            x6___.v___('ATTRIBS')), x8___ =
                                            attribKey,
                                          x7___.hasOwnProperty_m___?
                                            x7___.hasOwnProperty(x8___):
                                          x7___.m___('hasOwnProperty', [ x8___
                                            ])))) {
                                      atype = (x9___ = IMPORTS___.html4_v___?
                                        IMPORTS___.html4: ___.ri(IMPORTS___,
                                          'html4'), x9___.ATTRIBS_v___?
                                        x9___.ATTRIBS: x9___.v___('ATTRIBS'))
                                        .v___(attribKey);
                                      value = rewriteAttribute.i___(tagName,
                                        attribName, atype, value);
                                    } else if (! (x10___ = new
                                          RegExp.new___('__$'), x11___ =
                                          attribKey, x10___.test_m___?
                                          x10___.test(x11___):
                                        x10___.m___('test', [ x11___ ]))) {
                                      attribName = (x13___ = attribs, x12___ =
                                        cajaPrefix + attribs[ + +i ],
                                        x13___.NUM____w___ === x13___? (x13___[
                                            + +i ] = x12___): x13___.w___(+ +i,
                                          x12___));
                                    } else { value = null; }
                                    if (value !== null && value !== void 0) {
                                      attribs.NUM____w___ === attribs?
                                        (attribs[ + (+i + 1) ] = value):
                                      attribs.w___(+ (+i + 1), value);
                                    } else {
                                      x15___ = attribs, x14___ = attribs[ + --n
                                      ], x15___.NUM____w___ === x15___?
                                        (x15___[ + (+i + 1) ] = x14___):
                                      x15___.w___(+ (+i + 1), x14___);
                                      x17___ = attribs, x16___ = attribs[ + --n
                                      ], x17___.NUM____w___ === x17___?
                                        (x17___[ + +i ] = x16___):
                                      x17___.w___(+ +i, x16___);
                                      i = i - 2;
                                    }
                                  }
                                  attribs.length_w___ === attribs?
                                    (attribs.length = n):
                                  attribs.w___('length', n);
                                  if (needsTargetAttrib) {
                                    x19___ = attribs, x18___ =
                                      optTargetAttributePresets.v___('default')
                                      , x19___.push_m___? x19___.push('target',
                                      x18___): x19___.m___('push', [ 'target',
                                        x18___ ]);
                                  }
                                  policy = elementPolicies.v___(tagName);
                                  if (policy &&
                                    (elementPolicies.hasOwnProperty_m___?
                                       elementPolicies.hasOwnProperty(tagName):
                                     elementPolicies.m___('hasOwnProperty', [
                                         tagName ]))) {
                                    return policy.i___(attribs);
                                  }
                                  return attribs;
                                }
                                ___.f(sanitizeAttrs, 'sanitizeAttrs');
                              }
                              {
                                function unsuffix(str, suffix, fail) {
                                  var n;
                                  if (typeof str !== 'string') return fail;
                                  n = str.length - suffix.length;
                                  if (0 < n && (str.substring_m___?
                                      str.substring(n): str.m___('substring', [
                                          n ])) === suffix) {
                                    return str.substring_m___? str.substring(0,
                                      n): str.m___('substring', [ 0, n ]);
                                  } else { return fail; }
                                }
                                ___.f(unsuffix, 'unsuffix');
                              }
                              {
                                function virtualizeAttributeValue(attrType,
                                  realValue) {
                                  var x0___, x1___, x2___, x3___, x4___, x5___,
                                  x6___, x7___, x8___, x9___, x10___, x11___,
                                  x12___;
                                  realValue = (IMPORTS___.String_v___?
                                      IMPORTS___.String: ___.ri(IMPORTS___,
                                      'String')).i___(realValue);
                                  switch (attrType) {
                                  case x1___ = (x0___ = IMPORTS___.html4_v___?
                                      IMPORTS___.html4: ___.ri(IMPORTS___,
                                        'html4'), x0___.atype_v___?
                                      x0___.atype: x0___.v___('atype')),
                                    x1___.GLOBAL_NAME_v___? x1___.GLOBAL_NAME:
                                    x1___.v___('GLOBAL_NAME'):
                                  case x3___ = (x2___ = IMPORTS___.html4_v___?
                                        IMPORTS___.html4: ___.ri(IMPORTS___,
                                        'html4'), x2___.atype_v___?
                                        x2___.atype: x2___.v___('atype')),
                                    x3___.ID_v___? x3___.ID: x3___.v___('ID'):
                                  case x5___ = (x4___ = IMPORTS___.html4_v___?
                                      IMPORTS___.html4: ___.ri(IMPORTS___,
                                        'html4'), x4___.atype_v___?
                                      x4___.atype: x4___.v___('atype')),
                                    x5___.IDREF_v___? x5___.IDREF:
                                    x5___.v___('IDREF'):
                                    return unsuffix.i___(realValue, idSuffix,
                                      null);
                                  case x7___ = (x6___ = IMPORTS___.html4_v___?
                                        IMPORTS___.html4: ___.ri(IMPORTS___,
                                        'html4'), x6___.atype_v___?
                                        x6___.atype: x6___.v___('atype')),
                                    x7___.IDREFS_v___? x7___.IDREFS:
                                    x7___.v___('IDREFS'):
                                    return x9___ = realValue, x10___ =
                                      ID_LIST_PARTS_PATTERN, x8___ =
                                      ___.f(function (_, id, spaces) {
                                        return unsuffix.i___(id, idSuffix, '')
                                          + (spaces? ' ': '');
                                      }), x9___.replace_m___?
                                      x9___.replace(x10___, x8___):
                                    x9___.m___('replace', [ x10___, x8___ ]);
                                  case x12___ = (x11___ =
                                        IMPORTS___.html4_v___?
                                        IMPORTS___.html4: ___.ri(IMPORTS___,
                                        'html4'), x11___.atype_v___?
                                        x11___.atype: x11___.v___('atype')),
                                    x12___.URI_FRAGMENT_v___?
                                      x12___.URI_FRAGMENT:
                                    x12___.v___('URI_FRAGMENT'):
                                    if (realValue && '#' ===
                                      (realValue.charAt_m___?
                                       realValue.charAt(0):
                                       realValue.m___('charAt', [ 0 ]))) {
                                      realValue =
                                        unsuffix.i___(realValue.substring_m___?
                                          realValue.substring(1):
                                        realValue.m___('substring', [ 1 ]),
                                        idSuffix, null);
                                      return realValue? '#' + realValue: null;
                                    } else { return null; }
                                    break;
                                  default:
                                    return realValue;
                                  }
                                }
                                ___.f(virtualizeAttributeValue,
                                  'virtualizeAttributeValue');
                              }
                              {
                                function tameInnerHtml(htmlText) {
                                  var out;
                                  out = [ ];
                                  innerHtmlTamer.i___(htmlText, out);
                                  return out.join_m___? out.join(''):
                                  out.m___('join', [ '' ]);
                                }
                                ___.f(tameInnerHtml, 'tameInnerHtml');
                              }
                              {
                                function rewriteAttribute(tagName, attribName,
                                  type, value) {
                                  var x0___, x1___, x2___, x3___, x4___, x5___,
                                  x6___, x7___, x8___, x9___, x10___, x11___,
                                  x12___, x13___, x14___, x15___, x16___,
                                  x17___, x18___, match, doesReturn, fnName,
                                  x19___, x20___, x21___, x22___, x23___,
                                  x24___, x25___, x26___, x27___, x28___,
                                  x29___, cssPropertiesAndValues, css, i,
                                  propName, propValue, semi, x30___, x31___,
                                  x32___, x33___, i;
                                  switch (type) {
                                  case x1___ = (x0___ = IMPORTS___.html4_v___?
                                      IMPORTS___.html4: ___.ri(IMPORTS___,
                                        'html4'), x0___.atype_v___?
                                      x0___.atype: x0___.v___('atype')),
                                    x1___.NONE_v___? x1___.NONE:
                                    x1___.v___('NONE'):
                                    if (attribName === 'autocomplete' &&
                                      (tagName === 'input' || tagName ===
                                       'form')) { return 'off'; }
                                    return (IMPORTS___.String_v___?
                                      IMPORTS___.String: ___.ri(IMPORTS___,
                                        'String')).i___(value);
                                  case x3___ = (x2___ = IMPORTS___.html4_v___?
                                      IMPORTS___.html4: ___.ri(IMPORTS___,
                                        'html4'), x2___.atype_v___?
                                      x2___.atype: x2___.v___('atype')),
                                    x3___.CLASSES_v___? x3___.CLASSES:
                                    x3___.v___('CLASSES'):
                                    value = (IMPORTS___.String_v___?
                                      IMPORTS___.String: ___.ri(IMPORTS___,
                                        'String')).i___(value);
                                    if (! (FORBIDDEN_ID_LIST_PATTERN.test_m___?
                                        FORBIDDEN_ID_LIST_PATTERN.test(value):
                                        FORBIDDEN_ID_LIST_PATTERN.m___('test',
                                          [ value ]))) { return value; }
                                    return null;
                                  case x5___ = (x4___ = IMPORTS___.html4_v___?
                                        IMPORTS___.html4: ___.ri(IMPORTS___,
                                        'html4'), x4___.atype_v___?
                                        x4___.atype: x4___.v___('atype')),
                                    x5___.GLOBAL_NAME_v___? x5___.GLOBAL_NAME:
                                    x5___.v___('GLOBAL_NAME'):
                                  case x7___ = (x6___ = IMPORTS___.html4_v___?
                                      IMPORTS___.html4: ___.ri(IMPORTS___,
                                        'html4'), x6___.atype_v___?
                                      x6___.atype: x6___.v___('atype')),
                                    x7___.ID_v___? x7___.ID: x7___.v___('ID'):
                                  case x9___ = (x8___ = IMPORTS___.html4_v___?
                                      IMPORTS___.html4: ___.ri(IMPORTS___,
                                        'html4'), x8___.atype_v___?
                                      x8___.atype: x8___.v___('atype')),
                                    x9___.IDREF_v___? x9___.IDREF:
                                    x9___.v___('IDREF'):
                                    value = (IMPORTS___.String_v___?
                                        IMPORTS___.String: ___.ri(IMPORTS___,
                                        'String')).i___(value);
                                    if (value && isValidId.i___(value)) {
                                      return value + idSuffix;
                                    }
                                    return null;
                                  case x11___ = (x10___ =
                                      IMPORTS___.html4_v___? IMPORTS___.html4:
                                      ___.ri(IMPORTS___, 'html4'),
                                      x10___.atype_v___? x10___.atype:
                                      x10___.v___('atype')),
                                    x11___.IDREFS_v___? x11___.IDREFS:
                                    x11___.v___('IDREFS'):
                                    value = (IMPORTS___.String_v___?
                                      IMPORTS___.String: ___.ri(IMPORTS___,
                                        'String')).i___(value);
                                    if (value && isValidIdList.i___(value)) {
                                      return x13___ = value, x14___ =
                                        ID_LIST_PARTS_PATTERN, x12___ =
                                        ___.f(function (_, id, spaces) {
                                          return id + idSuffix + (spaces? ' ':
                                            '');
                                        }), x13___.replace_m___?
                                        x13___.replace(x14___, x12___):
                                      x13___.m___('replace', [ x14___, x12___ ]
                                      );
                                    }
                                    return null;
                                  case x16___ = (x15___ =
                                        IMPORTS___.html4_v___?
                                        IMPORTS___.html4: ___.ri(IMPORTS___,
                                        'html4'), x15___.atype_v___?
                                        x15___.atype: x15___.v___('atype')),
                                    x16___.LOCAL_NAME_v___? x16___.LOCAL_NAME:
                                    x16___.v___('LOCAL_NAME'):
                                    value = (IMPORTS___.String_v___?
                                        IMPORTS___.String: ___.ri(IMPORTS___,
                                        'String')).i___(value);
                                    if (value && isValidId.i___(value)) {
                                      return value; }
                                    return null;
                                  case x18___ = (x17___ =
                                        IMPORTS___.html4_v___?
                                        IMPORTS___.html4: ___.ri(IMPORTS___,
                                        'html4'), x17___.atype_v___?
                                        x17___.atype: x17___.v___('atype')),
                                    x18___.SCRIPT_v___? x18___.SCRIPT:
                                    x18___.v___('SCRIPT'):
                                    value = (IMPORTS___.String_v___?
                                        IMPORTS___.String: ___.ri(IMPORTS___,
                                        'String')).i___(value);
                                    match = value.match_m___?
                                      value.match(SIMPLE_HANDLER_PATTERN):
                                    value.m___('match', [
                                        SIMPLE_HANDLER_PATTERN ]);
                                    if (!match) { return null; }
                                    doesReturn = match[ +1 ];
                                    fnName = match[ +2 ];
                                    value = (doesReturn? 'return ': '') +
                                      '___.plugin_dispatchEvent___(' +
                                      'this, event, ' + pluginId + ', \"' +
                                      fnName + '\");';
                                    if (attribName === 'onsubmit') {
                                      value = 'try { ' + value +
                                        ' } finally { return false; }';
                                    }
                                    return value;
                                  case x20___ = (x19___ =
                                        IMPORTS___.html4_v___?
                                        IMPORTS___.html4: ___.ri(IMPORTS___,
                                        'html4'), x19___.atype_v___?
                                        x19___.atype: x19___.v___('atype')),
                                    x20___.URI_v___? x20___.URI:
                                    x20___.v___('URI'):
                                    value = (IMPORTS___.String_v___?
                                        IMPORTS___.String: ___.ri(IMPORTS___,
                                        'String')).i___(value);
                                    value = (x22___ = (x21___ =
                                        IMPORTS___.URI_v___? IMPORTS___.URI:
                                        ___.ri(IMPORTS___, 'URI'),
                                        x21___.utils_v___? x21___.utils:
                                        x21___.v___('utils')), x24___ = (x23___
                                          = domicile.pseudoLocation_v___?
                                          domicile.pseudoLocation:
                                        domicile.v___('pseudoLocation'),
                                        x23___.href_v___? x23___.href:
                                        x23___.v___('href')), x25___ = value,
                                      x22___.resolve_m___?
                                      x22___.resolve(x24___, x25___):
                                      x22___.m___('resolve', [ x24___, x25___ ]
                                      ));
                                    if (!naiveUriPolicy) { return null; }
                                    return uriRewrite.i___(naiveUriPolicy,
                                      value, getUriEffect.i___(tagName,
                                        attribName),
                                      getLoaderType.i___(tagName, attribName),
                                      ___.iM([ 'TYPE', 'MARKUP', 'XML_ATTR',
                                          attribName, 'XML_TAG', tagName ])) ||
                                      null;
                                  case x27___ = (x26___ =
                                      IMPORTS___.html4_v___? IMPORTS___.html4:
                                      ___.ri(IMPORTS___, 'html4'),
                                      x26___.atype_v___? x26___.atype:
                                      x26___.v___('atype')),
                                    x27___.URI_FRAGMENT_v___?
                                      x27___.URI_FRAGMENT:
                                    x27___.v___('URI_FRAGMENT'):
                                    value = (IMPORTS___.String_v___?
                                      IMPORTS___.String: ___.ri(IMPORTS___,
                                        'String')).i___(value);
                                    if ((value.charAt_m___? value.charAt(0):
                                        value.m___('charAt', [ 0 ])) === '#' &&
                                      isValidId.i___(value.substring_m___?
                                        value.substring(1):
                                        value.m___('substring', [ 1 ]))) {
                                      return value + idSuffix;
                                    }
                                    return null;
                                  case x29___ = (x28___ =
                                      IMPORTS___.html4_v___? IMPORTS___.html4:
                                      ___.ri(IMPORTS___, 'html4'),
                                      x28___.atype_v___? x28___.atype:
                                      x28___.v___('atype')), x29___.STYLE_v___?
                                      x29___.STYLE: x29___.v___('STYLE'):
                                    if ('function' !== typeof value) {
                                      return sanitizeStyleAttrValue.i___((IMPORTS___.String_v___?
                                            IMPORTS___.String:
                                          ___.ri(IMPORTS___, 'String'))
                                        .i___(value));
                                    }
                                    cssPropertiesAndValues =
                                      cssSealerUnsealerPair.unseal_m___?
                                      cssSealerUnsealerPair.unseal(value):
                                    cssSealerUnsealerPair.m___('unseal', [
                                        value ]);
                                    if (!cssPropertiesAndValues) { return null;
                                    }
                                    css = [ ];
                                    for (i = 0; i <
                                        cssPropertiesAndValues.length; i = i +
                                        2) {
                                      propName = cssPropertiesAndValues[ + +i ]
                                        ;
                                      propValue = cssPropertiesAndValues.v___(i
                                          + 1);
                                      semi = propName.indexOf_m___?
                                        propName.indexOf(';'):
                                      propName.m___('indexOf', [ ';' ]);
                                      if (semi >= 0) {
                                        propName = propName.substring_m___?
                                          propName.substring(0, semi):
                                        propName.m___('substring', [ 0, semi ])
                                          ;
                                      }
                                      x31___ = css, x30___ = propName + ' : ' +
                                        propValue, x31___.push_m___?
                                        x31___.push(x30___):
                                      x31___.m___('push', [ x30___ ]);
                                    }
                                    return css.join_m___? css.join(' ; '):
                                    css.m___('join', [ ' ; ' ]);
                                  case x33___ = (x32___ =
                                        IMPORTS___.html4_v___?
                                        IMPORTS___.html4: ___.ri(IMPORTS___,
                                        'html4'), x32___.atype_v___?
                                        x32___.atype: x32___.v___('atype')),
                                    x33___.FRAME_TARGET_v___?
                                      x33___.FRAME_TARGET:
                                    x33___.v___('FRAME_TARGET'):
                                    if (value !== null) {
                                      value = (IMPORTS___.String_v___?
                                          IMPORTS___.String: ___.ri(IMPORTS___,
                                          'String')).i___(value);
                                      for (i = 0; i <
                                          (optTargetAttributePresets.whitelist_v___?
                                            optTargetAttributePresets.whitelist:
                                          optTargetAttributePresets.v___('whitelist')
                                        ).length; ++i) {
                                        if
                                          ((optTargetAttributePresets.whitelist_v___?
                                              optTargetAttributePresets.whitelist:
                                            optTargetAttributePresets.v___('whitelist')
                                          ).v___(i) === value) { return value;
                                        }
                                      }
                                    }
                                    return optTargetAttributePresets.v___('default')
                                      ;
                                  default:
                                    return null;
                                  }
                                }
                                ___.f(rewriteAttribute, 'rewriteAttribute');
                              }
                              {
                                function P_constant(value) {
                                  return ___.iM([ 'enumerable', true, 'value',
                                      value ]);
                                }
                                ___.f(P_constant, 'P_constant');
                              }
                              {
                                function PropertyTaming(confidence) {
                                  var p, method, x0___, x1___;
                                  p = confidence.p_v___? confidence.p:
                                  confidence.v___('p');
                                  method = confidence.protectMethod_v___?
                                    confidence.protectMethod:
                                  confidence.v___('protectMethod');
                                  return x0___ = IMPORTS___.cajaVM_v___?
                                    IMPORTS___.cajaVM: ___.ri(IMPORTS___,
                                    'cajaVM'), x1___ = ___.iM([ 'ro', ___.iM([
                                          'enumerable', true,
                                          'extendedAccessors', true, 'get',
                                          method.i___(___.f(function (prop) {
                                                var dis___ = this && this.___?
                                                  void 0: this;
                                                var x0___;
                                                return (x0___ = p.i___(dis___),
                                                  x0___.feral_v___?
                                                  x0___.feral:
                                                  x0___.v___('feral'))
                                                  .v___(prop);
                                              })) ]), 'rw', ___.iM([
                                          'enumerable', true,
                                          'extendedAccessors', true, 'get',
                                          method.i___(___.f(function (prop) {
                                                var dis___ = this && this.___?
                                                  void 0: this;
                                                var x0___;
                                                return (x0___ = p.i___(dis___),
                                                  x0___.feral_v___?
                                                    x0___.feral:
                                                  x0___.v___('feral'))
                                                  .v___(prop);
                                              })), 'set',
                                          method.i___(___.f(function (value,
                                                prop) {
                                                var dis___ = this && this.___?
                                                  void 0: this;
                                                var x0___, x1___;
                                                if (! (x0___ = p.i___(dis___),
                                                    x0___.editable_v___?
                                                      x0___.editable:
                                                    x0___.v___('editable'))) {
                                                  throw new
                                                    (IMPORTS___.Error_v___?
                                                      IMPORTS___.Error:
                                                    ___.ri(IMPORTS___, 'Error')
                                                  ).new___(NOT_EDITABLE);
                                                }
                                                (x1___ = p.i___(dis___),
                                                 x1___.feral_v___? x1___.feral:
                                                 x1___.v___('feral'))
                                                .w___(prop, value);
                                              })) ]), 'RWCond', (function () {
                                          function RWCond$_lit(predicate) {
                                            return ___.iM([ 'enumerable', true,
                                                'extendedAccessors', true,
                                                'get',
                                                method.i___(___.f(function
                                                    (prop) {
                                                      var dis___ = this &&
                                                        this.___? void 0: this;
                                                      var x0___;
                                                      return (x0___ =
                                                          p.i___(dis___),
                                                        x0___.feral_v___?
                                                          x0___.feral:
                                                        x0___.v___('feral'))
                                                        .v___(prop);
                                                    })), 'set',
                                                method.i___(___.f(function
                                                    (value, prop) {
                                                      var dis___ = this &&
                                                        this.___? void 0: this;
                                                      var privates;
                                                      privates = p.i___(dis___)
                                                        ;
                                                      if (!
                                                          (privates.editable_v___?
                                                            privates.editable:
                                                          privates.v___('editable')
                                                        )) {
                                                        throw new
                                                          (IMPORTS___.Error_v___?
                                                            IMPORTS___.Error:
                                                          ___.ri(IMPORTS___,
                                                            'Error'))
                                                          .new___(NOT_EDITABLE)
                                                          ;
                                                      }
                                                      if (predicate.i___(value)
                                                      ) {
                                                        (privates.feral_v___?
                                                         privates.feral:
                                                         privates.v___('feral')
                                                        ).w___(prop, value);
                                                      }
                                                    })) ]);
                                          }
                                          return ___.f(RWCond$_lit,
                                            'RWCond$_lit');
                                        })(), 'Rename', (function () {
                                          function Rename$_lit(mapName, desc) {
                                            return ___.iM([ 'enumerable', true,
                                                'extendedAccessors', true,
                                                'get',
                                                method.i___(___.f(function
                                                      (prop) {
                                                      var dis___ = this &&
                                                        this.___? void 0: this;
                                                      var x0___, x1___, x2___;
                                                      return x0___ =
                                                        desc.get_v___?
                                                        desc.get:
                                                      desc.v___('get'), x1___ =
                                                        dis___, x2___ =
                                                        mapName,
                                                      x0___.call_m___?
                                                        x0___.call(x1___,
                                                        x2___):
                                                      x0___.m___('call', [
                                                          x1___, x2___ ]);
                                                    })), 'set',
                                                method.i___(___.f(function
                                                    (value, prop) {
                                                      var dis___ = this &&
                                                        this.___? void 0: this;
                                                      var x0___, x1___, x2___,
                                                      x3___;
                                                      return x0___ =
                                                        desc.set_v___?
                                                        desc.set:
                                                      desc.v___('set'), x1___ =
                                                        dis___, x2___ = value,
                                                      x3___ = mapName,
                                                      x0___.call_m___?
                                                        x0___.call(x1___,
                                                        x2___, x3___):
                                                      x0___.m___('call', [
                                                          x1___, x2___, x3___ ]
                                                      );
                                                    })) ]);
                                          }
                                          return ___.f(Rename$_lit,
                                            'Rename$_lit');
                                        })(), 'related', ___.iM([ 'enumerable',
                                          true, 'extendedAccessors', true,
                                          'get', method.i___(___.f(function
                                                (prop) {
                                                var dis___ = this && this.___?
                                                  void 0: this;
                                                var x0___, x1___;
                                                if (!___.i('' + 'editable',
                                                    p.i___(dis___))) {
                                                  throw new
                                                    (IMPORTS___.Error_v___?
                                                      IMPORTS___.Error:
                                                    ___.ri(IMPORTS___, 'Error')
                                                  )
                                                    .new___(('Internal error: related property tamer can only'
                                                        +
                                                        ' be applied to objects with an editable flag')
                                                  );
                                                }
                                                return tameRelatedNode.i___((x0___ =
                                                      p.i___(dis___),
                                                    x0___.feral_v___?
                                                      x0___.feral:
                                                    x0___.v___('feral'))
                                                  .v___(prop), (x1___ =
                                                      p.i___(dis___),
                                                    x1___.editable_v___?
                                                      x1___.editable:
                                                    x1___.v___('editable')),
                                                  defaultTameNode);
                                              })) ]), 'filter', (function () {
                                          function filter$_lit(useAttrGetter,
                                            toValue, useAttrSetter, fromValue)
                                            {
                                            var desc, x0___, x1___;
                                            desc = ___.iM([ 'enumerable', true,
                                                'extendedAccessors', true,
                                                'get', useAttrGetter?
                                                method.i___(___.f(function
                                                    (name) {
                                                      var dis___ = this &&
                                                        this.___? void 0: this;
                                                      var x0___, x1___, x2___;
                                                      return x1___ = toValue,
                                                      x2___ = dis___, x0___ =
                                                        dis___.getAttribute_m___?
                                                        dis___.getAttribute(name)
                                                        :
                                                      dis___.m___('getAttribute',
                                                        [ name ]),
                                                      x1___.call_m___?
                                                        x1___.call(x2___,
                                                        x0___):
                                                      x1___.m___('call', [
                                                          x2___, x0___ ]);
                                                    })):
                                                method.i___(___.f(function
                                                    (name) {
                                                      var dis___ = this &&
                                                        this.___? void 0: this;
                                                      var x0___, x1___, x2___,
                                                      x3___;
                                                      return x2___ = toValue,
                                                      x3___ = dis___, x1___ =
                                                        (x0___ = p.i___(dis___)
                                                        , x0___.feral_v___?
                                                        x0___.feral:
                                                        x0___.v___('feral'))
                                                        .v___(name),
                                                      x2___.call_m___?
                                                        x2___.call(x3___,
                                                        x1___):
                                                      x2___.m___('call', [
                                                          x3___, x1___ ]);
                                                    })) ]);
                                            if (fromValue) {
                                              x1___ = desc, x0___ =
                                                useAttrSetter?
                                                method.i___(___.f(function
                                                  (value, name) {
                                                    var dis___ = this &&
                                                      this.___? void 0: this;
                                                    var x0___, x1___, x2___;
                                                    x1___ = dis___, x2___ =
                                                      name, x0___ =
                                                      fromValue.call_m___?
                                                      fromValue.call(dis___,
                                                      value):
                                                    fromValue.m___('call', [
                                                        dis___, value ]),
                                                    x1___.setAttribute_m___?
                                                      x1___.setAttribute(x2___,
                                                      x0___):
                                                    x1___.m___('setAttribute',
                                                      [ x2___, x0___ ]);
                                                  })):
                                              method.i___(___.f(function
                                                    (value, name) {
                                                    var dis___ = this &&
                                                      this.___? void 0: this;
                                                    var x0___, x1___;
                                                    if (! (x0___ =
                                                        p.i___(dis___),
                                                        x0___.editable_v___?
                                                        x0___.editable:
                                                        x0___.v___('editable'))
                                                    ) {
                                                      throw new
                                                        (IMPORTS___.Error_v___?
                                                        IMPORTS___.Error:
                                                        ___.ri(IMPORTS___,
                                                          'Error'))
                                                        .new___(NOT_EDITABLE);
                                                    }
                                                    (x1___ = p.i___(dis___),
                                                     x1___.feral_v___?
                                                       x1___.feral:
                                                     x1___.v___('feral'))
                                                    .w___(name,
                                                      fromValue.call_m___?
                                                        fromValue.call(dis___,
                                                        value):
                                                      fromValue.m___('call', [
                                                          dis___, value ]));
                                                  })), x1___.set_w___ ===
                                                x1___? (x1___.set = x0___):
                                              x1___.w___('set', x0___);
                                            }
                                            return desc;
                                          }
                                          return ___.f(filter$_lit,
                                            'filter$_lit');
                                        })(), 'filterAttr', (function () {
                                          function filterAttr$_lit(toValue,
                                            fromValue) {
                                            return NP.filter_m___?
                                              NP.filter(true, toValue, true,
                                              fromValue): NP.m___('filter', [
                                                true, toValue, true, fromValue
                                              ]);
                                          }
                                          return ___.f(filterAttr$_lit,
                                            'filterAttr$_lit');
                                        })(), 'filterProp', (function () {
                                          function filterProp$_lit(toValue,
                                            fromValue) {
                                            return NP.filter_m___?
                                              NP.filter(false, toValue, false,
                                              fromValue): NP.m___('filter', [
                                                false, toValue, false,
                                                fromValue ]);
                                          }
                                          return ___.f(filterProp$_lit,
                                            'filterProp$_lit');
                                        })() ]), x0___.def_m___?
                                    x0___.def(x1___): x0___.m___('def', [ x1___
                                    ]);
                                }
                                ___.f(PropertyTaming, 'PropertyTaming');
                              }
                              {
                                function getNodeExpandoStorage(node) {
                                  var s, x0___, x1___, x2___;
                                  s = nodeExpandos.get_m___?
                                    nodeExpandos.get(node):
                                  nodeExpandos.m___('get', [ node ]);
                                  if (s === (IMPORTS___.undefined_v___?
                                      IMPORTS___.undefined: ___.ri(IMPORTS___,
                                        'undefined'))) {
                                    x1___ = nodeExpandos, x2___ = node, x0___ =
                                      s = ___.iM([ ]), x1___.set_m___?
                                      x1___.set(x2___, x0___):
                                    x1___.m___('set', [ x2___, x0___ ]);
                                  }
                                  return s;
                                }
                                ___.f(getNodeExpandoStorage,
                                  'getNodeExpandoStorage');
                              }
                              {
                                function makeTameNodeByType(node, editable) {
                                  var tagName, x0___, x1___, x2___, x3___,
                                  x4___, x5___, x6___, x7___, x8___;
                                  switch (node.nodeType_v___? node.nodeType:
                                    node.v___('nodeType')) {
                                  case 1:
                                    tagName = (x0___ = node.tagName_v___?
                                      node.tagName: node.v___('tagName'),
                                      x0___.toLowerCase_m___?
                                      x0___.toLowerCase():
                                      x0___.m___('toLowerCase', [ ]));
                                    if (x2___ = tamingClassesByElement, x1___ =
                                      tagName + '$', x2___.hasOwnProperty_m___?
                                      x2___.hasOwnProperty(x1___):
                                      x2___.m___('hasOwnProperty', [ x1___ ]))
                                      {
                                      return new
                                        (tamingClassesByElement.v___(tagName +
                                            '$').new___)(node, editable);
                                    } else if (! (x4___ = (x3___ =
                                            IMPORTS___.html4_v___?
                                            IMPORTS___.html4:
                                          ___.ri(IMPORTS___, 'html4'),
                                          x3___.ELEMENTS_v___? x3___.ELEMENTS:
                                          x3___.v___('ELEMENTS')), x5___ =
                                          tagName, x4___.hasOwnProperty_m___?
                                          x4___.hasOwnProperty(x5___):
                                        x4___.m___('hasOwnProperty', [ x5___ ])
                                      ) || (x6___ = IMPORTS___.html4_v___?
                                          IMPORTS___.html4: ___.ri(IMPORTS___,
                                          'html4'), x6___.ELEMENTS_v___?
                                          x6___.ELEMENTS:
                                        x6___.v___('ELEMENTS')).v___(tagName) &
                                        (x8___ = (x7___ =
                                          IMPORTS___.html4_v___?
                                          IMPORTS___.html4: ___.ri(IMPORTS___,
                                            'html4'), x7___.eflags_v___?
                                          x7___.eflags: x7___.v___('eflags')),
                                        x8___.UNSAFE_v___? x8___.UNSAFE:
                                        x8___.v___('UNSAFE'))) {
                                      return new TameOpaqueNode.new___(node,
                                        editable);
                                    } else {
                                      return new TameElement.new___(node,
                                        editable, editable);
                                    }
                                  case 2:
                                    throw 'Internal: Attr nodes cannot be generically wrapped';
                                  case 3:
                                  case 4:
                                    return new TameTextNode.new___(node,
                                      editable);
                                  case 8:
                                    return new TameCommentNode.new___(node,
                                      editable);
                                  case 11:
                                    return new TameBackedNode.new___(node,
                                      editable, editable);
                                  default:
                                    return new TameOpaqueNode.new___(node,
                                      editable);
                                  }
                                }
                                ___.f(makeTameNodeByType, 'makeTameNodeByType')
                                  ;
                              }
                              {
                                function defaultTameNode(node, editable,
                                  foreign) {
                                  var tamed;
                                  if (node === null || node === void 0) {
                                    return null; }
                                  node = makeDOMAccessible.i___(node);
                                  if (taming.hasTameTwin_m___?
                                    taming.hasTameTwin(node):
                                    taming.m___('hasTameTwin', [ node ])) {
                                    return taming.tame_m___? taming.tame(node):
                                    taming.m___('tame', [ node ]);
                                  }
                                  tamed = foreign? new
                                    TameForeignNode.new___(node, editable):
                                  makeTameNodeByType.i___(node, editable);
                                  tamed = finishNode.i___(tamed);
                                  return tamed;
                                }
                                ___.f(defaultTameNode, 'defaultTameNode');
                              }
                              {
                                function tameRelatedNode(node, editable,
                                  tameNodeCtor) {
                                  var x0___, x1___, docElem, x2___, ancestor,
                                  x3___, x4___;
                                  if (node === null || node === void 0) {
                                    return null; }
                                  if (node === (x0___ = np.i___(tameDocument),
                                      x0___.feralContainerNode_v___?
                                      x0___.feralContainerNode:
                                      x0___.v___('feralContainerNode'))) {
                                    if ((x1___ = np.i___(tameDocument),
                                        x1___.editable_v___? x1___.editable:
                                        x1___.v___('editable')) && !editable) {
                                      throw new (IMPORTS___.Error_v___?
                                        IMPORTS___.Error: ___.ri(IMPORTS___,
                                          'Error')).new___(NOT_EDITABLE);
                                    }
                                    return tameDocument;
                                  }
                                  node = makeDOMAccessible.i___(node);
                                  try {
                                    docElem = (x2___ = node.ownerDocument_v___?
                                      node.ownerDocument:
                                      node.v___('ownerDocument'),
                                      x2___.documentElement_v___?
                                      x2___.documentElement:
                                      x2___.v___('documentElement'));
                                    for (ancestor = node; ancestor; ancestor =
                                        makeDOMAccessible.i___(ancestor.parentNode_v___?
                                          ancestor.parentNode:
                                        ancestor.v___('parentNode'))) {
                                      if (x4___ = idClassPattern, x3___ =
                                          ancestor.className_v___?
                                          ancestor.className:
                                        ancestor.v___('className'),
                                        x4___.test_m___? x4___.test(x3___):
                                        x4___.m___('test', [ x3___ ])) {
                                        return tameNodeCtor.i___(node,
                                          editable);
                                      } else if (ancestor === docElem) { return null; }
                                    }
                                    return tameNodeCtor.i___(node, editable);
                                  } catch (e) {}
                                  return null;
                                }
                                ___.f(tameRelatedNode, 'tameRelatedNode');
                              }
                              {
                                function getNodeListLength(nodeList) {
                                  var limit;
                                  limit = nodeList.length;
                                  if (limit !== +limit) {
                                    limit = 1 / 0;
                                  }
                                  return limit;
                                }
                                ___.f(getNodeListLength, 'getNodeListLength');
                              }
                              {
                                function mixinNodeList(tamed, nodeList,
                                  editable, opt_tameNodeCtor) {
                                  var limit, i, j, x0___, x1___, x2___, x3___,
                                  x4___, x5___;
                                  limit = getNodeListLength.i___(nodeList);
                                  if (limit > 0 && !opt_tameNodeCtor) {
                                    throw 'Internal: Nonempty mixinNodeList() without a tameNodeCtor';
                                  }
                                  for (i = tamed.length, j = 0; j < limit &&
                                    nodeList[ + +j ]; ++i, ++j) {
                                    x1___ = tamed, x0___ =
                                      opt_tameNodeCtor.i___(nodeList[ + +j ],
                                      editable), x1___.NUM____w___ === x1___?
                                      (x1___[ + +i ] = x0___): x1___.w___(+ +i,
                                      x0___);
                                  }
                                  nodeList = null;
                                  x5___ = tamed, x4___ = (x2___ =
                                      IMPORTS___.cajaVM_v___?
                                      IMPORTS___.cajaVM: ___.ri(IMPORTS___,
                                      'cajaVM'), x3___ = ___.f(function (k) {
                                        k = k & 2147483647;
                                        if (k !== k) {
                                          throw new (IMPORTS___.Error_v___?
                                            IMPORTS___.Error:
                                            ___.ri(IMPORTS___, 'Error'))
                                            .new___;
                                        }
                                        return tamed[ + +k ] || null;
                                      }), x2___.def_m___? x2___.def(x3___):
                                    x2___.m___('def', [ x3___ ])),
                                  x5___.item_w___ === x5___? (x5___.item =
                                      x4___): x5___.w___('item', x4___);
                                  return tamed;
                                }
                                ___.f(mixinNodeList, 'mixinNodeList');
                              }
                              {
                                function rebuildTameListConstructors(ArrayLike)
                                  {
                                  var x0___, x1___, x2___, x3___, x4___, x5___,
                                  x6___, x7___, x8___, x9___, x10___, x11___,
                                  x12___, x13___, x14___, x15___, x16___,
                                  x17___, x18___, x19___, x20___, x21___;
                                  TameNodeList = makeTameNodeList.i___();
                                  x3___ = TameNodeList, x2___ = (x0___ =
                                      IMPORTS___.Object_v___?
                                      IMPORTS___.Object: ___.ri(IMPORTS___,
                                      'Object'), x1___ =
                                      ArrayLike.prototype_v___?
                                      ArrayLike.prototype:
                                    ArrayLike.v___('prototype'),
                                    x0___.create_m___? x0___.create(x1___):
                                    x0___.m___('create', [ x1___ ])),
                                  x3___.prototype_w___ === x3___?
                                    (x3___.prototype = x2___):
                                  x3___.w___('prototype', x2___);
                                  x4___ = IMPORTS___.Object_v___?
                                    IMPORTS___.Object: ___.ri(IMPORTS___,
                                    'Object'), x5___ =
                                    TameNodeList.prototype_v___?
                                    TameNodeList.prototype:
                                  TameNodeList.v___('prototype'), x6___ =
                                    ___.iM([ 'value', TameNodeList ]),
                                  x4___.defineProperty_m___?
                                    x4___.defineProperty(x5___, 'constructor',
                                    x6___): x4___.m___('defineProperty', [
                                      x5___, 'constructor', x6___ ]);
                                  x7___ = IMPORTS___.Object_v___?
                                    IMPORTS___.Object: ___.ri(IMPORTS___,
                                    'Object'), x8___ =
                                    TameNodeList.prototype_v___?
                                    TameNodeList.prototype:
                                  TameNodeList.v___('prototype'),
                                  x7___.freeze_m___? x7___.freeze(x8___):
                                  x7___.m___('freeze', [ x8___ ]);
                                  x9___ = IMPORTS___.Object_v___?
                                    IMPORTS___.Object: ___.ri(IMPORTS___,
                                    'Object'), x10___ = TameNodeList,
                                  x9___.freeze_m___? x9___.freeze(x10___):
                                  x9___.m___('freeze', [ x10___ ]);
                                  TameOptionsList = makeTameOptionsList.i___();
                                  x14___ = TameOptionsList, x13___ = (x11___ =
                                      IMPORTS___.Object_v___?
                                      IMPORTS___.Object: ___.ri(IMPORTS___,
                                      'Object'), x12___ =
                                      ArrayLike.prototype_v___?
                                      ArrayLike.prototype:
                                    ArrayLike.v___('prototype'),
                                    x11___.create_m___? x11___.create(x12___):
                                    x11___.m___('create', [ x12___ ])),
                                  x14___.prototype_w___ === x14___?
                                    (x14___.prototype = x13___):
                                  x14___.w___('prototype', x13___);
                                  x15___ = IMPORTS___.Object_v___?
                                    IMPORTS___.Object: ___.ri(IMPORTS___,
                                    'Object'), x16___ =
                                    TameOptionsList.prototype_v___?
                                    TameOptionsList.prototype:
                                  TameOptionsList.v___('prototype'), x17___ =
                                    ___.iM([ 'value', TameOptionsList ]),
                                  x15___.defineProperty_m___?
                                    x15___.defineProperty(x16___,
                                    'constructor', x17___):
                                  x15___.m___('defineProperty', [ x16___,
                                      'constructor', x17___ ]);
                                  x18___ = IMPORTS___.Object_v___?
                                    IMPORTS___.Object: ___.ri(IMPORTS___,
                                    'Object'), x19___ =
                                    TameOptionsList.prototype_v___?
                                    TameOptionsList.prototype:
                                  TameOptionsList.v___('prototype'),
                                  x18___.freeze_m___? x18___.freeze(x19___):
                                  x18___.m___('freeze', [ x19___ ]);
                                  x20___ = IMPORTS___.Object_v___?
                                    IMPORTS___.Object: ___.ri(IMPORTS___,
                                    'Object'), x21___ = TameOptionsList,
                                  x20___.freeze_m___? x20___.freeze(x21___):
                                  x20___.m___('freeze', [ x21___ ]);
                                }
                                ___.f(rebuildTameListConstructors,
                                  'rebuildTameListConstructors');
                              }
                              {
                                function makeTameNodeList() {
                                  return (function () {
                                      function TNL(nodeList, editable,
                                        tameNodeCtor) {
                                        var getItem, getLength, len, ArrayLike,
                                        x0___, x1___, result, x2___, x3___,
                                        x4___, x5___, x6___;
                                        {
                                          function getItem(i) {
                                            i = +i;
                                            if (i >= nodeList.length) { return void 0; }
                                            return tameNodeCtor.i___(nodeList.v___(i)
                                                , editable);
                                          }
                                          ___.f(getItem, 'getItem');
                                        }
                                        {
                                          function getLength() {
                                            return nodeList.length;
                                          }
                                          ___.f(getLength, 'getLength');
                                        }
                                        nodeList =
                                          makeDOMAccessible.i___(nodeList);
                                        len = +getLength.i___();
                                        ArrayLike = (x0___ =
                                          IMPORTS___.cajaVM_v___?
                                          IMPORTS___.cajaVM: ___.ri(IMPORTS___,
                                            'cajaVM'), x1___ = len,
                                          x0___.makeArrayLike_m___?
                                          x0___.makeArrayLike(x1___):
                                          x0___.m___('makeArrayLike', [ x1___ ]
                                          ));
                                        if (! ((TameNodeList.prototype_v___?
                                                TameNodeList.prototype:
                                              TameNodeList.v___('prototype'))
                                            instanceof ArrayLike)) {
                                          rebuildTameListConstructors.i___(ArrayLike)
                                            ;
                                        }
                                        result =
                                          ArrayLike.i___(TameNodeList.prototype_v___?
                                            TameNodeList.prototype:
                                          TameNodeList.v___('prototype'),
                                          getItem, getLength);
                                        x2___ = IMPORTS___.Object_v___?
                                          IMPORTS___.Object: ___.ri(IMPORTS___,
                                          'Object'), x6___ = result, x5___ =
                                          ___.iM([ 'value', (x3___ =
                                                IMPORTS___.Object_v___?
                                                IMPORTS___.Object:
                                              ___.ri(IMPORTS___, 'Object'),
                                              x4___ = getItem,
                                              x3___.freeze_m___?
                                                x3___.freeze(x4___):
                                              x3___.m___('freeze', [ x4___ ]))
                                          ]), x2___.defineProperty_m___?
                                          x2___.defineProperty(x6___, 'item',
                                          x5___): x2___.m___('defineProperty',
                                          [ x6___, 'item', x5___ ]);
                                        return result;
                                      }
                                      return ___.f(TNL, 'TNL');
                                    })();
                                }
                                ___.f(makeTameNodeList, 'makeTameNodeList');
                              }
                              {
                                function makeTameOptionsList() {
                                  return (function () {
                                      function TOL(nodeList, editable,
                                        opt_tameNodeCtor) {
                                        var getItem, getLength, len, ArrayLike,
                                        x0___, x1___, result, x2___, x3___,
                                        x4___;
                                        {
                                          function getItem(i) {
                                            i = +i;
                                            return opt_tameNodeCtor.i___(nodeList.v___(i)
                                                , editable);
                                          }
                                          ___.f(getItem, 'getItem');
                                        }
                                        {
                                          function getLength() {
                                            return nodeList.length;
                                          }
                                          ___.f(getLength, 'getLength');
                                        }
                                        nodeList =
                                          makeDOMAccessible.i___(nodeList);
                                        len = +getLength.i___();
                                        ArrayLike = (x0___ =
                                          IMPORTS___.cajaVM_v___?
                                          IMPORTS___.cajaVM: ___.ri(IMPORTS___,
                                            'cajaVM'), x1___ = len,
                                          x0___.makeArrayLike_m___?
                                          x0___.makeArrayLike(x1___):
                                          x0___.m___('makeArrayLike', [ x1___ ]
                                          ));
                                        if (! ((TameOptionsList.prototype_v___?
                                                TameOptionsList.prototype:
                                              TameOptionsList.v___('prototype')
                                            ) instanceof ArrayLike)) {
                                          rebuildTameListConstructors.i___(ArrayLike)
                                            ;
                                        }
                                        result =
                                          ArrayLike.i___(TameOptionsList.prototype_v___?
                                          TameOptionsList.prototype:
                                          TameOptionsList.v___('prototype'),
                                          getItem, getLength);
                                        x2___ = IMPORTS___.Object_v___?
                                          IMPORTS___.Object: ___.ri(IMPORTS___,
                                          'Object'), x4___ = result, x3___ =
                                          ___.iM([ 'get', (function () {
                                                function get$_lit() {
                                                  return +
                                                    (nodeList.selectedIndex_v___?
                                                    nodeList.selectedIndex:
                                                    nodeList.v___('selectedIndex')
                                                  );
                                                }
                                                return ___.f(get$_lit,
                                                  'get$_lit');
                                              })() ]),
                                        x2___.defineProperty_m___?
                                          x2___.defineProperty(x4___,
                                          'selectedIndex', x3___):
                                        x2___.m___('defineProperty', [ x4___,
                                            'selectedIndex', x3___ ]);
                                        return result;
                                      }
                                      return ___.f(TOL, 'TOL');
                                    })();
                                }
                                ___.f(makeTameOptionsList,
                                  'makeTameOptionsList');
                              }
                              {
                                function fakeNodeList(array) {
                                  var x0___, x1___, x2___, x3___, x4___, x5___;
                                  x3___ = array, x2___ = (x0___ =
                                      IMPORTS___.cajaVM_v___?
                                      IMPORTS___.cajaVM: ___.ri(IMPORTS___,
                                      'cajaVM'), x1___ = ___.f(function (i) {
                                        return array[ + +i ];
                                      }), x0___.def_m___? x0___.def(x1___):
                                    x0___.m___('def', [ x1___ ])),
                                  x3___.item_w___ === x3___? (x3___.item =
                                    x2___): x3___.w___('item', x2___);
                                  return x4___ = IMPORTS___.Object_v___?
                                    IMPORTS___.Object: ___.ri(IMPORTS___,
                                    'Object'), x5___ = array,
                                  x4___.freeze_m___? x4___.freeze(x5___):
                                  x4___.m___('freeze', [ x5___ ]);
                                }
                                ___.f(fakeNodeList, 'fakeNodeList');
                              }
                              {
                                function mixinHTMLCollection(tamed, nodeList,
                                  editable, opt_tameNodeCtor) {
                                  var tameNodesByName, tameNode, i, name,
                                  x0___, x1___, x2___, x3___, name, x4___,
                                  x5___, x6___, tameNodes, x7___, x8___, x9___,
                                  x10___;;
                                  mixinNodeList.i___(tamed, nodeList, editable,
                                    opt_tameNodeCtor);
                                  tameNodesByName = ___.iM([ ]);
                                  for (i = 0; i < tamed.length && (tameNode =
                                      tamed[ + +i ]); ++i) {
                                    name = tameNode.getAttribute_m___?
                                      tameNode.getAttribute('name'):
                                    tameNode.m___('getAttribute', [ 'name' ]);
                                    if (name && ! ((x1___ = name, x0___ =
                                          name.length - 1, x1___.charAt_m___?
                                          x1___.charAt(x0___):
                                          x1___.m___('charAt', [ x0___ ])) ===
                                        '_' || ___.i('' + name, tamed) || name
                                        === (IMPORTS___.String_v___?
                                            IMPORTS___.String:
                                          ___.ri(IMPORTS___, 'String'))
                                        .i___(name & 2147483647))) {
                                      if (!tameNodesByName.v___(name)) {
                                        tameNodesByName.w___(name, [ ]);
                                      }
                                      x2___ = tameNodesByName.v___(name), x3___
                                        = tameNode, x2___.push_m___?
                                        x2___.push(x3___): x2___.m___('push', [
                                          x3___ ]);
                                    }
                                  }
                                  {
                                    x6___ = Object(tameNodesByName).e___();
                                    for (x5___ in x6___) {
                                      if (typeof x5___ === 'number' || '' +
                                          +x5___ === x5___) { name = x5___; }
                                      else {
                                        if (/^NUM___/.test(x5___) &&
                                            /__$/.test(x5___)) { continue; }
                                        x4___ = x5___.match(/([\s\S]*)_e___$/);
                                        if (!x4___ || !x6___[ x5___ ]) {
                                          continue; }
                                        name = x4___[ 1 ];
                                      }
                                      {
                                        tameNodes = tameNodesByName.v___(name);
                                        if (tameNodes.length > 1) {
                                          tamed.w___(name,
                                            fakeNodeList.i___(tameNodes));
                                        } else {
                                          tamed.w___(name, tameNodes[ +0 ]);
                                        }
                                      }
                                    }
                                  }
                                  x10___ = tamed, x9___ = (x7___ =
                                    IMPORTS___.cajaVM_v___? IMPORTS___.cajaVM:
                                    ___.ri(IMPORTS___, 'cajaVM'), x8___ =
                                    ___.f(function (name) {
                                        var x0___, x1___, x2___, x3___, x4___,
                                        x5___, x6___, x7___, x8___, x9___;
                                        name = (IMPORTS___.String_v___?
                                          IMPORTS___.String: ___.ri(IMPORTS___,
                                            'String')).i___(name);
                                        if ((x1___ = name, x0___ = name.length
                                            - 1, x1___.charAt_m___?
                                            x1___.charAt(x0___):
                                            x1___.m___('charAt', [ x0___ ]))
                                          === '_') { return null; }
                                        if (x4___ = (x3___ = (x2___ =
                                                IMPORTS___.Object_v___?
                                                IMPORTS___.Object:
                                              ___.ri(IMPORTS___, 'Object'),
                                              x2___.prototype_v___?
                                                x2___.prototype:
                                              x2___.v___('prototype')),
                                            x3___.hasOwnProperty_v___?
                                              x3___.hasOwnProperty:
                                            x3___.v___('hasOwnProperty')),
                                          x5___ = tamed, x6___ = name,
                                          x4___.call_m___? x4___.call(x5___,
                                            x6___): x4___.m___('call', [ x5___,
                                              x6___ ])) {
                                          return (x7___ =
                                            IMPORTS___.cajaVM_v___?
                                            IMPORTS___.cajaVM:
                                            ___.ri(IMPORTS___, 'cajaVM'), x9___
                                            = TameNodeT, x8___ =
                                            tamed.v___(name),
                                            x7___.passesGuard_m___?
                                            x7___.passesGuard(x9___, x8___):
                                            x7___.m___('passesGuard', [ x9___,
                                                x8___ ])) ? tamed.v___(name):
                                          (tamed.v___(name))[ +0 ];
                                        }
                                        return null;
                                      }), x7___.def_m___? x7___.def(x8___):
                                    x7___.m___('def', [ x8___ ])),
                                  x10___.namedItem_w___ === x10___?
                                    (x10___.namedItem = x9___):
                                  x10___.w___('namedItem', x9___);
                                  return tamed;
                                }
                                ___.f(mixinHTMLCollection,
                                  'mixinHTMLCollection');
                              }
                              {
                                function tameHTMLCollection(nodeList, editable,
                                  opt_tameNodeCtor) {
                                  var x0___, x1___;
                                  return x0___ = IMPORTS___.Object_v___?
                                    IMPORTS___.Object: ___.ri(IMPORTS___,
                                    'Object'), x1___ =
                                    mixinHTMLCollection.i___([ ], nodeList,
                                    editable, opt_tameNodeCtor),
                                  x0___.freeze_m___? x0___.freeze(x1___):
                                  x0___.m___('freeze', [ x1___ ]);
                                }
                                ___.f(tameHTMLCollection, 'tameHTMLCollection')
                                  ;
                              }
                              {
                                function tameGetElementsByTagName(rootNode,
                                  tagName, editable) {
                                  var eflags, x0___, x1___, x2___, x3___,
                                  x4___, x5___, x6___, x7___, x8___;
                                  tagName = (IMPORTS___.String_v___?
                                    IMPORTS___.String: ___.ri(IMPORTS___,
                                      'String')).i___(tagName);
                                  eflags = 0;
                                  if (tagName !== '*') {
                                    tagName = tagName.toLowerCase_m___?
                                      tagName.toLowerCase():
                                    tagName.m___('toLowerCase', [ ]);
                                    if (! (x2___ = (x1___ = (x0___ =
                                            IMPORTS___.Object_v___?
                                            IMPORTS___.Object:
                                            ___.ri(IMPORTS___, 'Object'),
                                            x0___.prototype_v___?
                                            x0___.prototype:
                                            x0___.v___('prototype')),
                                          x1___.hasOwnProperty_v___?
                                          x1___.hasOwnProperty:
                                          x1___.v___('hasOwnProperty')), x4___
                                        = (x3___ = IMPORTS___.html4_v___?
                                          IMPORTS___.html4: ___.ri(IMPORTS___,
                                            'html4'), x3___.ELEMENTS_v___?
                                          x3___.ELEMENTS:
                                          x3___.v___('ELEMENTS')), x5___ =
                                        tagName, x2___.call_m___?
                                        x2___.call(x4___, x5___):
                                        x2___.m___('call', [ x4___, x5___ ]))
                                      || (eflags = (x6___ =
                                            IMPORTS___.html4_v___?
                                            IMPORTS___.html4:
                                          ___.ri(IMPORTS___, 'html4'),
                                          x6___.ELEMENTS_v___? x6___.ELEMENTS:
                                          x6___.v___('ELEMENTS')).v___(tagName)
                                      ) & (x8___ = (x7___ =
                                          IMPORTS___.html4_v___?
                                          IMPORTS___.html4: ___.ri(IMPORTS___,
                                            'html4'), x7___.ELEMENTS_v___?
                                          x7___.ELEMENTS:
                                          x7___.v___('ELEMENTS')),
                                        x8___.UNSAFE_v___? x8___.UNSAFE:
                                        x8___.v___('UNSAFE'))) {
                                      return new fakeNodeList.new___([ ]);
                                    }
                                    tagName =
                                      virtualToRealElementName.i___(tagName);
                                  }
                                  return new
                                    TameNodeList.new___((rootNode.getElementsByTagName_m___?
                                        rootNode.getElementsByTagName(tagName):
                                      rootNode.m___('getElementsByTagName', [
                                          tagName ])), editable,
                                    defaultTameNode);
                                }
                                ___.f(tameGetElementsByTagName,
                                  'tameGetElementsByTagName');
                              }
                              {
                                function tameGetElementsByClassName(rootNode,
                                  className, editable) {
                                  var classes, x0___, x1___, i, classi, x2___,
                                  x3___, x4___, x5___, x6___, x7___, nClasses,
                                  i, x8___, x9___, candidates, matches, limit,
                                  j, candidate, k, candidateClass, i, x10___,
                                  x11___, tamed;
                                  className = (IMPORTS___.String_v___?
                                      IMPORTS___.String: ___.ri(IMPORTS___,
                                      'String')).i___(className);
                                  classes = (x1___ = className, x0___ = new
                                    RegExp.new___('[^\\t\\n\\f\\r ]+', 'g'),
                                    x1___.match_m___? x1___.match(x0___):
                                    x1___.m___('match', [ x0___ ]));
                                  for (i = classes? classes.length: 0; --i >=
                                    0;) {
                                    classi = classes[ + +i ];
                                    if (FORBIDDEN_ID_PATTERN.test_m___?
                                      FORBIDDEN_ID_PATTERN.test(classi):
                                      FORBIDDEN_ID_PATTERN.m___('test', [
                                          classi ])) {
                                      x3___ = classes, x2___ = classes[ +
                                          (classes.length - 1) ],
                                      x3___.NUM____w___ === x3___? (x3___[ + +i
                                        ] = x2___): x3___.w___(+ +i, x2___);
                                      x5___ = classes, x4___ = classes.length -
                                        1, x5___.length_w___ === x5___?
                                        (x5___.length = x4___):
                                      x5___.w___('length', x4___);
                                    }
                                  }
                                  if (!classes || classes.length === 0) {
                                    return fakeNodeList.i___([ ]);
                                  }
                                  if (typeof
                                      (rootNode.getElementsByClassName_v___?
                                      rootNode.getElementsByClassName:
                                      rootNode.v___('getElementsByClassName'))
                                      === 'function') {
                                    return new TameNodeList.new___((x7___ =
                                        rootNode, x6___ = classes.join_m___?
                                        classes.join(' '): classes.m___('join',
                                          [ ' ' ]),
                                        x7___.getElementsByClassName_m___?
                                        x7___.getElementsByClassName(x6___):
                                        x7___.m___('getElementsByClassName', [
                                            x6___ ])), editable,
                                      defaultTameNode);
                                  } else {
                                    nClasses = classes.length;
                                    for (i = nClasses; --i >= 0;) {
                                      x9___ = classes, x8___ = ' ' + classes[ +
                                        +i ] + ' ', x9___.NUM____w___ ===
                                        x9___? (x9___[ + +i ] = x8___):
                                      x9___.w___(+ +i, x8___);
                                    }
                                    candidates =
                                      rootNode.getElementsByTagName_m___?
                                      rootNode.getElementsByTagName('*'):
                                    rootNode.m___('getElementsByTagName', [ '*'
                                      ]);
                                    matches = [ ];
                                    limit = candidates.length;
                                    if (limit !== +limit) {
                                      limit = 1 / 0;
                                    }
                                    candidate_loop: for (j = 0, k = -1; j <
                                      limit && (candidate = candidates[ + +j ])
                                      ; ++j) {
                                      candidateClass = ' ' +
                                        (candidate.className_v___?
                                        candidate.className:
                                        candidate.v___('className')) + ' ';
                                      for (i = nClasses; --i >= 0;) {
                                        if (-1 === (x11___ = candidateClass,
                                            x10___ = classes[ + +i ],
                                            x11___.indexOf_m___?
                                            x11___.indexOf(x10___):
                                            x11___.m___('indexOf', [ x10___ ]))
                                        ) {
                                          continue candidate_loop;
                                        }
                                      }
                                      tamed = defaultTameNode.i___(candidate,
                                        editable);
                                      if (tamed) {
                                        matches.NUM____w___ === matches?
                                          (matches[ + ++k ] = tamed):
                                        matches.w___(+ ++k, tamed);
                                      }
                                    }
                                    return fakeNodeList.i___(matches);
                                  }
                                }
                                ___.f(tameGetElementsByClassName,
                                  'tameGetElementsByClassName');
                              }
                              {
                                function makeEventHandlerWrapper(thisNode,
                                  listener) {
                                  var x0___, x1___, wrapper;
                                  {
                                    function wrapper(event) {
                                      return plugin_dispatchEvent.i___(thisNode,
                                        event, rulebreaker.getId_m___?
                                        rulebreaker.getId(tameWindow):
                                        rulebreaker.m___('getId', [ tameWindow
                                          ]), listener);
                                    }
                                    ___.f(wrapper, 'wrapper');
                                  }
                                  x0___ = IMPORTS___.domitaModules_v___?
                                    IMPORTS___.domitaModules:
                                  ___.ri(IMPORTS___, 'domitaModules'), x1___ =
                                    listener, x0___.ensureValidCallback_m___?
                                    x0___.ensureValidCallback(x1___):
                                  x0___.m___('ensureValidCallback', [ x1___ ]);
                                  return wrapper;
                                }
                                ___.f(makeEventHandlerWrapper,
                                  'makeEventHandlerWrapper');
                              }
                              {
                                function tameAddEventListener(name, listener,
                                  useCapture) {
                                  var dis___ = this && this.___? void 0: this;
                                  var feral, x0___, x1___, x2___, x3___, x4___,
                                  wrappedListener, x5___, x6___, x7___, x8___,
                                  x9___, x10___, x11___, x12___, x13___,
                                  x14___;
                                  feral = (x0___ = np.i___(dis___),
                                    x0___.feral_v___? x0___.feral:
                                    x0___.v___('feral'));
                                  if (! (x1___ = np.i___(dis___),
                                      x1___.editable_v___? x1___.editable:
                                      x1___.v___('editable'))) {
                                    throw new (IMPORTS___.Error_v___?
                                      IMPORTS___.Error: ___.ri(IMPORTS___,
                                        'Error')).new___(NOT_EDITABLE);
                                  }
                                  if (! (x2___ = np.i___(dis___),
                                      x2___.wrappedListeners_v___?
                                        x2___.wrappedListeners:
                                      x2___.v___('wrappedListeners'))) {
                                    x3___ = np.i___(dis___), x4___ = [ ],
                                    x3___.wrappedListeners_w___ === x3___?
                                      (x3___.wrappedListeners = x4___):
                                    x3___.w___('wrappedListeners', x4___);
                                  }
                                  useCapture = (IMPORTS___.Boolean_v___?
                                    IMPORTS___.Boolean: ___.ri(IMPORTS___,
                                      'Boolean')).i___(useCapture);
                                  wrappedListener =
                                    makeEventHandlerWrapper.i___((x5___ =
                                        np.i___(dis___), x5___.feral_v___?
                                        x5___.feral: x5___.v___('feral')),
                                    listener);
                                  wrappedListener = (x8___ = bridal, x7___ =
                                      (x6___ = np.i___(dis___),
                                      x6___.feral_v___? x6___.feral:
                                      x6___.v___('feral')), x9___ = name,
                                    x10___ = wrappedListener, x11___ =
                                      useCapture, x8___.addEventListener_m___?
                                      x8___.addEventListener(x7___, x9___,
                                      x10___, x11___):
                                    x8___.m___('addEventListener', [ x7___,
                                        x9___, x10___, x11___ ]));
                                  wrappedListener._d_originalListener_w___ ===
                                    wrappedListener?
                                    (wrappedListener._d_originalListener =
                                      listener):
                                  wrappedListener.w___('_d_originalListener',
                                    listener);
                                  x13___ = (x12___ = np.i___(dis___),
                                    x12___.wrappedListeners_v___?
                                    x12___.wrappedListeners:
                                    x12___.v___('wrappedListeners')), x14___ =
                                    wrappedListener, x13___.push_m___?
                                    x13___.push(x14___): x13___.m___('push', [
                                      x14___ ]);
                                }
                                ___.f(tameAddEventListener,
                                  'tameAddEventListener');
                              }
                              {
                                function tameRemoveEventListener(name,
                                  listener, useCapture) {
                                  var dis___ = this && this.___? void 0: this;
                                  var self, feral, x0___, x1___, x2___,
                                  wrappedListener, i, x3___, x4___, x5___,
                                  x6___, x7___, x8___, x9___, x10___, x11___,
                                  x12___, x13___;
                                  self = TameNodeT.coerce_m___?
                                    TameNodeT.coerce(dis___):
                                  TameNodeT.m___('coerce', [ dis___ ]);
                                  feral = (x0___ = np.i___(self),
                                    x0___.feral_v___? x0___.feral:
                                    x0___.v___('feral'));
                                  if (! (x1___ = np.i___(self),
                                      x1___.editable_v___? x1___.editable:
                                      x1___.v___('editable'))) {
                                    throw new (IMPORTS___.Error_v___?
                                      IMPORTS___.Error: ___.ri(IMPORTS___,
                                        'Error')).new___(NOT_EDITABLE);
                                  }
                                  if (! (x2___ = np.i___(dis___),
                                      x2___.wrappedListeners_v___?
                                        x2___.wrappedListeners:
                                      x2___.v___('wrappedListeners'))) {
                                    return; }
                                  wrappedListener = null;
                                  for (i = (x3___ = np.i___(dis___),
                                      x3___.wrappedListeners_v___?
                                        x3___.wrappedListeners:
                                      x3___.v___('wrappedListeners')).length;
                                    --i >= 0;) {
                                    if ((x5___ = (x4___ = np.i___(dis___),
                                          x4___.wrappedListeners_v___?
                                            x4___.wrappedListeners:
                                          x4___.v___('wrappedListeners'))[ + +i
                                        ], x5___._d_originalListener_v___?
                                          x5___._d_originalListener:
                                        x5___.v___('_d_originalListener')) ===
                                      listener) {
                                      wrappedListener = (x6___ =
                                          np.i___(dis___),
                                        x6___.wrappedListeners_v___?
                                          x6___.wrappedListeners:
                                        x6___.v___('wrappedListeners'))[ + +i ]
                                        ;
                                      arrayRemove.i___((x7___ = np.i___(dis___)
                                          , x7___.wrappedListeners_v___?
                                          x7___.wrappedListeners:
                                          x7___.v___('wrappedListeners')), i,
                                        i);
                                      break;
                                    }
                                  }
                                  if (!wrappedListener) { return; }
                                  x10___ = bridal, x9___ = (x8___ =
                                    np.i___(dis___), x8___.feral_v___?
                                    x8___.feral: x8___.v___('feral')), x11___ =
                                    name, x12___ = wrappedListener, x13___ =
                                    useCapture,
                                  x10___.removeEventListener_m___?
                                    x10___.removeEventListener(x9___, x11___,
                                    x12___, x13___):
                                  x10___.m___('removeEventListener', [ x9___,
                                      x11___, x12___, x13___ ]);
                                }
                                ___.f(tameRemoveEventListener,
                                  'tameRemoveEventListener');
                              }
                              {
                                function inertCtor(tamedCtor, someSuper,
                                  opt_name) {
                                  var inert, x0___, x1___, x2___, x3___;
                                  inherit.i___(tamedCtor, someSuper);
                                  inert = (function () {
                                      function inert$_var() {
                                        throw new (IMPORTS___.TypeError_v___?
                                            IMPORTS___.TypeError:
                                          ___.ri(IMPORTS___, 'TypeError'))
                                          .new___('This constructor cannot be called directly.')
                                          ;
                                      }
                                      return ___.f(inert$_var, 'inert$_var');
                                    })();
                                  x1___ = inert, x0___ =
                                    tamedCtor.prototype_v___?
                                    tamedCtor.prototype:
                                  tamedCtor.v___('prototype'),
                                  x1___.prototype_w___ === x1___?
                                    (x1___.prototype = x0___):
                                  x1___.w___('prototype', x0___);
                                  x2___ = IMPORTS___.Object_v___?
                                    IMPORTS___.Object: ___.ri(IMPORTS___,
                                    'Object'), x3___ = inert,
                                  x2___.freeze_m___? x2___.freeze(x3___):
                                  x2___.m___('freeze', [ x3___ ]);
                                  setOwn.i___(tamedCtor.prototype_v___?
                                      tamedCtor.prototype:
                                    tamedCtor.v___('prototype'), 'constructor',
                                    inert);
                                  if (opt_name !== (IMPORTS___.undefined_v___?
                                      IMPORTS___.undefined: ___.ri(IMPORTS___,
                                        'undefined')))
                                    nodeClasses.w___(opt_name, inert);
                                  return inert;
                                }
                                ___.f(inertCtor, 'inertCtor');
                              }
                              {
                                function TameNode(editable) {
                                  var dis___ = this && this.___? void 0: this;
                                  var x0___, x1___;
                                  TameNodeConf.confide_m___?
                                    TameNodeConf.confide(dis___):
                                  TameNodeConf.m___('confide', [ dis___ ]);
                                  x0___ = np.i___(dis___), x1___ = editable,
                                  x0___.editable_w___ === x0___?
                                    (x0___.editable = x1___):
                                  x0___.w___('editable', x1___);
                                  return dis___;
                                }
                                ___.f(TameNode, 'TameNode');
                              }
                              {
                                function TameBackedNode(node, editable,
                                  childrenEditable, opt_proxyType) {
                                  var dis___ = this && this.___? void 0: this;
                                  var x0___, x1___, x2___, x3___, x4___, x5___,
                                  x6___;
                                  node = makeDOMAccessible.i___(node);
                                  if (!node) {
                                    throw new (IMPORTS___.Error_v___?
                                      IMPORTS___.Error: ___.ri(IMPORTS___,
                                        'Error'))
                                      .new___('Creating tame node with undefined native delegate')
                                      ;
                                  }
                                  TameNode.call_m___? TameNode.call(dis___,
                                    editable): TameNode.m___('call', [ dis___,
                                      editable ]);
                                  x0___ = np.i___(dis___), x1___ = node,
                                  x0___.feral_w___ === x0___? (x0___.feral =
                                      x1___): x0___.w___('feral', x1___);
                                  x2___ = np.i___(dis___), x3___ = editable &&
                                    childrenEditable,
                                  x2___.childrenEditable_w___ === x2___?
                                    (x2___.childrenEditable = x3___):
                                  x2___.w___('childrenEditable', x3___);
                                  if (x4___ = IMPORTS___.domitaModules_v___?
                                      IMPORTS___.domitaModules:
                                    ___.ri(IMPORTS___, 'domitaModules'),
                                    x4___.proxiesAvailable_v___?
                                      x4___.proxiesAvailable:
                                    x4___.v___('proxiesAvailable')) {
                                    x5___ = np.i___(dis___), x6___ = new
                                      (opt_proxyType || ExpandoProxyHandler)
                                      .new___(dis___, editable,
                                      (getNodeExpandoStorage.i___(node))),
                                    x5___.proxyHandler_w___ === x5___?
                                      (x5___.proxyHandler = x6___):
                                    x5___.w___('proxyHandler', x6___);
                                  }
                                }
                                ___.f(TameBackedNode, 'TameBackedNode');
                              }
                              {
                                function TamePseudoNode(editable) {
                                  var dis___ = this && this.___? void 0: this;
                                  var x0___, x1___, x2___;
                                  TameNode.call_m___? TameNode.call(dis___,
                                    editable): TameNode.m___('call', [ dis___,
                                      editable ]);
                                  if (x0___ = IMPORTS___.domitaModules_v___?
                                    IMPORTS___.domitaModules:
                                    ___.ri(IMPORTS___, 'domitaModules'),
                                    x0___.proxiesAvailable_v___?
                                    x0___.proxiesAvailable:
                                    x0___.v___('proxiesAvailable')) {
                                    x1___ = np.i___(dis___), x2___ = new
                                      ExpandoProxyHandler.new___(dis___,
                                      editable, (___.iM([ ]))),
                                    x1___.proxyHandler_w___ === x1___?
                                      (x1___.proxyHandler = x2___):
                                    x1___.w___('proxyHandler', x2___);
                                  }
                                }
                                ___.f(TamePseudoNode, 'TamePseudoNode');
                              }
                              {
                                function makeRestrictedNodeType(whitelist) {
                                  var nodeType, safe, x0___, x1___, x2___,
                                  descriptor, x3___, x4___, x5___, x6___,
                                  x7___, x8___, x9___, throwRestricted, x10___,
                                  x11___, i, k, x12___, x13___, x14___, x15___,
                                  x16___, x17___, x18___, x19___;;
                                  {
                                    function throwRestricted() {
                                      throw new (IMPORTS___.Error_v___?
                                        IMPORTS___.Error: ___.ri(IMPORTS___,
                                          'Error'))
                                        .new___('Node is restricted');
                                    }
                                    ___.f(throwRestricted, 'throwRestricted');
                                  }
                                  nodeType = (function () {
                                      function nodeType$_var(node, editable) {
                                        var dis___ = this && this.___? void 0:
                                        this;
                                        TameBackedNode.call_m___?
                                          TameBackedNode.call(dis___, node,
                                          editable, editable):
                                        TameBackedNode.m___('call', [ dis___,
                                            node, editable, editable ]);
                                      }
                                      return ___.f(nodeType$_var,
                                        'nodeType$_var');
                                    })();
                                  inherit.i___(nodeType, TameBackedNode);
                                  {
                                    x2___ = Object(whitelist).e___();
                                    for (x1___ in x2___) {
                                      if (typeof x1___ === 'number' || '' +
                                        +x1___ === x1___) { safe = x1___; }
                                      else {
                                        if (/^NUM___/.test(x1___) &&
                                            /__$/.test(x1___)) { continue; }
                                        x0___ = x1___.match(/([\s\S]*)_e___$/);
                                        if (!x0___ || !x2___[ x1___ ]) {
                                          continue; }
                                        safe = x0___[ 1 ];
                                      }
                                      {
                                        descriptor = whitelist.v___(safe) ===
                                          0? (x3___ =
                                            IMPORTS___.domitaModules_v___?
                                            IMPORTS___.domitaModules:
                                          ___.ri(IMPORTS___, 'domitaModules'),
                                          x4___ =
                                            TameBackedNode.prototype_v___?
                                            TameBackedNode.prototype:
                                          TameBackedNode.v___('prototype'),
                                          x5___ = safe,
                                          x3___.getPropertyDescriptor_m___?
                                            x3___.getPropertyDescriptor(x4___,
                                            x5___):
                                          x3___.m___('getPropertyDescriptor', [
                                              x4___, x5___ ])): ___.iM([
                                            'value', whitelist.v___(safe),
                                            'writable', false, 'configurable',
                                            false, 'enumerable', true ]);
                                        x6___ = IMPORTS___.Object_v___?
                                          IMPORTS___.Object: ___.ri(IMPORTS___,
                                          'Object'), x7___ =
                                          nodeType.prototype_v___?
                                          nodeType.prototype:
                                        nodeType.v___('prototype'), x8___ =
                                          safe, x9___ = descriptor,
                                        x6___.defineProperty_m___?
                                          x6___.defineProperty(x7___, x8___,
                                          x9___): x6___.m___('defineProperty',
                                          [ x7___, x8___, x9___ ]);
                                      }
                                    }
                                  }
                                  definePropertiesAwesomely.i___(nodeType.prototype_v___?
                                    nodeType.prototype:
                                    nodeType.v___('prototype'), ___.iM([
                                        'attributes', ___.iM([ 'enumerable',
                                            canHaveEnumerableAccessors, 'get',
                                            nodeMethod.i___(___.f(function () {
                                                  return new
                                                    TameNodeList.new___([ ],
                                                    false,
                                                    (IMPORTS___.undefined_v___?
                                                       IMPORTS___.undefined:
                                                     ___.ri(IMPORTS___,
                                                       'undefined')));
                                                })) ]) ]));
                                  x10___ = IMPORTS___.cajaVM_v___?
                                    IMPORTS___.cajaVM: ___.ri(IMPORTS___,
                                    'cajaVM'), x11___ = throwRestricted,
                                  x10___.def_m___? x10___.def(x11___):
                                  x10___.m___('def', [ x11___ ]);
                                  for (i = tameNodePublicMembers.length; --i >=
                                      0;) {
                                    k = tameNodePublicMembers[ + +i ];
                                    if (! (x12___ = nodeType.prototype_v___?
                                        nodeType.prototype:
                                        nodeType.v___('prototype'), x13___ = k,
                                        x12___.hasOwnProperty_m___?
                                        x12___.hasOwnProperty(x13___):
                                        x12___.m___('hasOwnProperty', [ x13___
                                          ]))) {
                                      if (typeof
                                        (TameBackedNode.prototype_v___?
                                           TameBackedNode.prototype:
                                         TameBackedNode.v___('prototype'))
                                        .v___(k) === 'Function') {
                                        (nodeType.prototype_v___?
                                           nodeType.prototype:
                                         nodeType.v___('prototype')).w___(k,
                                          throwRestricted);
                                      } else {
                                        x14___ = IMPORTS___.Object_v___?
                                          IMPORTS___.Object: ___.ri(IMPORTS___,
                                          'Object'), x15___ =
                                          nodeType.prototype_v___?
                                          nodeType.prototype:
                                        nodeType.v___('prototype'), x17___ = k,
                                        x16___ = ___.iM([ 'enumerable',
                                            canHaveEnumerableAccessors, 'get',
                                            throwRestricted ]),
                                        x14___.defineProperty_m___?
                                          x14___.defineProperty(x15___, x17___,
                                          x16___):
                                        x14___.m___('defineProperty', [ x15___,
                                            x17___, x16___ ]);
                                      }
                                    }
                                  }
                                  return x18___ = IMPORTS___.cajaVM_v___?
                                    IMPORTS___.cajaVM: ___.ri(IMPORTS___,
                                    'cajaVM'), x19___ = nodeType,
                                  x18___.def_m___? x18___.def(x19___):
                                  x18___.m___('def', [ x19___ ]);
                                }
                                ___.f(makeRestrictedNodeType,
                                  'makeRestrictedNodeType');
                              }
                              {
                                function TameTextNode(node, editable) {
                                  var dis___ = this && this.___? void 0: this;
                                  var pn, x0___, x1___, x2___, x3___;
                                  assert.i___((node.nodeType_v___?
                                      node.nodeType: node.v___('nodeType')) ===
                                    3);
                                  pn = node.parentNode_v___? node.parentNode:
                                  node.v___('parentNode');
                                  if (editable && pn) {
                                    if (1 === (pn.nodeType_v___? pn.nodeType:
                                        pn.v___('nodeType')) && (x0___ =
                                        IMPORTS___.html4_v___?
                                        IMPORTS___.html4: ___.ri(IMPORTS___,
                                          'html4'), x0___.ELEMENTS_v___?
                                        x0___.ELEMENTS: x0___.v___('ELEMENTS'))
                                        .v___((x1___ = pn.tagName_v___?
                                            pn.tagName: pn.v___('tagName'),
                                          x1___.toLowerCase_m___?
                                            x1___.toLowerCase():
                                          x1___.m___('toLowerCase', [ ]))) &
                                        (x3___ = (x2___ =
                                            IMPORTS___.html4_v___?
                                            IMPORTS___.html4:
                                          ___.ri(IMPORTS___, 'html4'),
                                          x2___.eflags_v___? x2___.eflags:
                                          x2___.v___('eflags')),
                                        x3___.UNSAFE_v___? x3___.UNSAFE:
                                        x3___.v___('UNSAFE'))) { editable =
                                        false; }
                                  }
                                  TameBackedNode.call_m___?
                                    TameBackedNode.call(dis___, node, editable,
                                    editable): TameBackedNode.m___('call', [
                                      dis___, node, editable, editable ]);
                                }
                                ___.f(TameTextNode, 'TameTextNode');
                              }
                              {
                                function TameCommentNode(node, editable) {
                                  var dis___ = this && this.___? void 0: this;
                                  assert.i___((node.nodeType_v___?
                                      node.nodeType: node.v___('nodeType')) ===
                                    8);
                                  TameBackedNode.call_m___?
                                    TameBackedNode.call(dis___, node, editable,
                                    editable): TameBackedNode.m___('call', [
                                      dis___, node, editable, editable ]);
                                }
                                ___.f(TameCommentNode, 'TameComm