/**
 * Copyright (C) 2009 eXo Platform SAS.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */
package org.exoplatform.gwtframework.editor.api;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import org.exoplatform.gwtframework.commons.rest.MimeType;
import org.exoplatform.gwtframework.editor.codemirror.CodemirrorProducer;
import org.exoplatform.gwtframework.editor.fckeditor.FCKEditorProducer;

/**
 * Created by The eXo Platform SAS .
 * 
 * @author <a href="mailto:gavrikvetal@gmail.com">Vitaliy Gulyy</a>
 * @version $
 */

public class EditorFactory
{

   private static HashMap<String, List<Editor>> editors = new HashMap<String, List<Editor>>();

   public static List<Editor> getEditors(String mimeType) throws EditorNotFoundException
   {
      List<Editor> list = editors.get(mimeType);
      if (list == null)
      {
         throw new EditorNotFoundException();
      }

      return list;
   }

   public static Editor getDefaultEditor(String mimeType) throws EditorNotFoundException
   {
      List<Editor> list = getEditors(mimeType);
      if (list.size() == 0)
      {
         throw new EditorNotFoundException();
      }

      for (Editor info : list)
      {
         if (info.isDefaultEditor())
         {
            return info;
         }
      }

      return list.get(0);
   }

   private static void addEditor(String mimeType, Editor editorInfo, boolean isDefaultEditor)
   {
      List<Editor> list = editors.get(mimeType);
      if (list == null)
      {
         list = new ArrayList<Editor>();
         editors.put(mimeType, list);
      }

      if (isDefaultEditor)
      {
         for (Editor info : list)
         {
            info.setIsDefaultEditor(false);
         }
         editorInfo.setIsDefaultEditor(true);
      }

      list.add(editorInfo);
   }

   private static void addEditor(String mimeType, Editor editorInfo)
   {
      addEditor(mimeType, editorInfo, false);
   }

   static
   {
      /*
       * TEXT
       */
      addEditor(MimeType.TEXT_PLAIN, new CodemirrorProducer(MimeType.TEXT_PLAIN));

      /*
       * XML
       */
      addEditor(MimeType.TEXT_XML, new CodemirrorProducer(MimeType.TEXT_XML));
      addEditor(MimeType.APPLICATION_XML, new CodemirrorProducer(MimeType.APPLICATION_XML));

      /*
       * HTML
       */
      addEditor(MimeType.TEXT_HTML, new CodemirrorProducer(MimeType.TEXT_HTML), true);
      addEditor(MimeType.TEXT_HTML, new FCKEditorProducer(MimeType.TEXT_HTML));

      /*
       * CSS
       */
      addEditor(MimeType.TEXT_CSS, new CodemirrorProducer(MimeType.TEXT_CSS));

      /*
       * JAVASCRIPT
       */
      addEditor(MimeType.TEXT_JAVASCRIPT, new CodemirrorProducer(MimeType.TEXT_JAVASCRIPT));
      addEditor(MimeType.APPLICATION_JAVASCRIPT, new CodemirrorProducer(MimeType.APPLICATION_JAVASCRIPT));
      addEditor(MimeType.APPLICATION_X_JAVASCRIPT, new CodemirrorProducer(MimeType.APPLICATION_X_JAVASCRIPT));

      /*
       * GROOVY
       */
      addEditor(MimeType.APPLICATION_GROOVY, new CodemirrorProducer(MimeType.APPLICATION_GROOVY));
      addEditor(MimeType.GROOVY_SERVICE, new CodemirrorProducer(MimeType.GROOVY_SERVICE));

      /*
       * GADGET
       */
      addEditor(MimeType.GOOGLE_GADGET, new CodemirrorProducer(MimeType.GOOGLE_GADGET), true);
      addEditor(MimeType.GOOGLE_GADGET, new FCKEditorProducer(MimeType.GOOGLE_GADGET));
      
      /*
       * UWA WIDGET
       */
//      addEditor(MimeType.UWA_WIDGET, new CodemirrorProducer(MimeType.UWA_WIDGET));

      
      /*
       * GROOVY_TEMPLATE
       */
      addEditor(MimeType.GROOVY_TEMPLATE, new CodemirrorProducer(MimeType.GROOVY_TEMPLATE), true);
      addEditor(MimeType.GROOVY_TEMPLATE, new FCKEditorProducer(MimeType.GROOVY_TEMPLATE));      
   }

}
