/*
 * Copyright (C) 2010 eXo Platform SAS.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.exoplatform.gwtframework.editor.codemirror;

import org.exoplatform.gwtframework.commons.rest.MimeType;
import org.exoplatform.gwtframework.editor.api.Token;
import org.exoplatform.gwtframework.editor.api.Token.TokenType;

import com.google.gwt.core.client.JavaScriptObject;

/**
 * @author <a href="mailto:dmitry.ndp@gmail.com">Dmytro Nochevnov</a>
 * @version $Id: $
 *
 */
public class JavaScriptParser extends Parser
{      
   
   private String lastNodeContent;
   
   private String lastNodeType;
   
   private TokenType lastTokenType;
   
   @Override
   Token parseLine(JavaScriptObject node, int lineNumber, Token currentToken, boolean hasParentParser)
   {
      // interrupt at the end of the line or content
      if ((node == null) || getName(node).equals("BR"))
         return currentToken;
      
      String nodeContent = getContent(node).trim(); // returns text without ended space " " in the text
      String nodeType = getType(node);
      
      // to recognize function definition like "a: function(){}" or "var a = function() {}"
      if (isFunctionNode(nodeType, nodeContent))
      {
         // recognize method in the JSON like "a: function() {}"
         if ((lastTokenType == TokenType.PROPERTY) && isColonNode(lastNodeType, lastNodeContent))
         {
            currentToken.updateTypeOfLastSubTokenOfLastToken(TokenType.METHOD);
            lastTokenType = null;
         }
   
         // recognize function definition like "var a = function () {}"
         else if ((lastTokenType == TokenType.VARIABLE) && isEqualNode(lastNodeType, lastNodeContent)) 
         {
            // test if this is the local function therefore last sub token of current token has at least one sub token
            if ((currentToken.getSubTokenList() != null) 
                   && (currentToken.getSubTokenList().size() > 0)
                   && (currentToken.getSubTokenList().get(currentToken.getSubTokenList().size() - 1) != null)
               )   
            {
               currentToken.updateTypeOfLastSubTokenOfLastToken(TokenType.FUNCTION);   // update last subToken of lastToken
            }
            else
            { 
               currentToken.updateTypeOfLastSubToken(TokenType.FUNCTION);   // update last token
            }
         }
      }
      
      // to recognize "var a", "function a"
      else if (nodeType.equals("js-variable"))
      {
         if (isVarNode(lastNodeType, lastNodeContent)) // parse cases like "var a"
         {
            currentToken.addSubToken(new Token(nodeContent, TokenType.VARIABLE, lineNumber, MimeType.APPLICATION_JAVASCRIPT));
            lastTokenType = TokenType.VARIABLE;
         }
         else if (isFunctionNode(lastNodeType, lastNodeContent)) // parse cases like "function a" 
         {
            currentToken.addSubToken(new Token(nodeContent, TokenType.FUNCTION, lineNumber, MimeType.APPLICATION_JAVASCRIPT));
         }
      }
      
      // to recognize local variable defined within the function like "function a() { var a = 1;  }"
      else if (nodeType.equals("js-variabledef"))      
      { 
         // parse local function case like "function a"
         if (isFunctionNode(lastNodeType, lastNodeContent)) 
         {
            currentToken.addSubTokenToTheLastSubToken(new Token(nodeContent, TokenType.FUNCTION, lineNumber, MimeType.APPLICATION_JAVASCRIPT));
            lastTokenType = null;            
         }

         // parse local variable case like "var a"
         else 
         {   
            currentToken.addSubTokenToTheLastSubToken(new Token(nodeContent, TokenType.VARIABLE, lineNumber, MimeType.APPLICATION_JAVASCRIPT));
            lastTokenType = TokenType.VARIABLE;   // to verify code like "var a = function(){}"
         }
      }
            
      // recognize property or method
      else if (isColonNode(nodeType, nodeContent) 
                && (lastNodeType.equals("js-property")     // recognize "a : 1"
                    || lastNodeType.equals("js-string")     // recognize "'a' : 1"
                   )
              )
      {
         currentToken.addSubTokenToTheLastSubToken(new Token(lastNodeContent, TokenType.PROPERTY, lineNumber,
            MimeType.APPLICATION_JAVASCRIPT));
         lastTokenType = TokenType.PROPERTY; // to verify code like "a : function(){}"
      } 
      
      // verify if this is "var a = function(){}" case
      else if ((lastTokenType == TokenType.VARIABLE) && isEqualNode(nodeType, nodeContent))
      {}
      
      else
      {
         lastTokenType = null;
      }
   
      lastNodeContent = nodeContent;
      lastNodeType = nodeType;

      if (hasParentParser) 
      {
         return currentToken; // return current token to parent parser
      } 
      else
      {
         return parseLine(getNext(node), lineNumber, currentToken, false);
      }
   
   }

   private boolean isVarNode(String nodeType, String nodeContent)
   {
      return (nodeType != null) && (nodeContent != null) && nodeType.equals("js-keyword") && nodeContent.equals("var");
   }

   private boolean isFunctionNode(String nodeType, String nodeContent)
   {
      return (nodeType != null) && (nodeContent != null) && nodeType.equals("js-keyword") && nodeContent.equals("function");
   }

   private boolean isEqualNode(String nodeType, String nodeContent)
   {
      return (nodeType != null) && (nodeContent != null) && nodeType.equals("js-operator") && nodeContent.equals("=");
   }

   private boolean isColonNode(String nodeType, String nodeContent)
   {
      return (nodeType != null) && (nodeContent != null) && nodeType.equals("js-punctuation") && nodeContent.equals(":");
   }; 
}