/**
 * Copyright (C) 2010 eXo Platform SAS.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */

package org.exoplatform.gwtframework.editor.api;

import java.util.ArrayList;
import java.util.List;

public class Token
{
   private String name;

   private TokenType type;

   private int lineNumber;

   private List<Token> subTokenList;

   private String shortDescription;

   private String code;

   private String fullDescription;

   private String mimeType;
   
   private Token parentToken;

   public Token(String name, TokenType type)
   {
      this.name = name;
      this.type = type;
   }

   public Token(String name, TokenType type, int lineNumber)
   {
      this(name, type, lineNumber, null, null, null, null, null);
   }

   public Token(String name, TokenType type, int lineNumber, String mimeType)
   {
      this(name, type, lineNumber, mimeType, null, null, null, null);
   }

   public Token(String name, TokenType type, int lineNumber, List<Token> subTokenList)
   {
      this(name, type, lineNumber, null, null, null, null, subTokenList);
   }

   public Token(String name, TokenType type, String shortDescription, String code, String fullDescription)
   {
      this(name, type, 0, null, shortDescription, code, fullDescription, null);
   }

   public Token(String name, TokenType type, int lineNumber, String mimeType, String shortDescription, String code,
      String fullDescription, List<Token> subTokenList)
   {
      this.name = name;
      this.type = type;
      this.lineNumber = lineNumber;
      this.mimeType = mimeType;
      this.shortDescription = shortDescription;
      this.code = code;
      this.fullDescription = fullDescription;
      this.subTokenList = subTokenList;
   }

   /**
    * @return the shortDescription
    */
   public String getShortDescription()
   {
      return shortDescription;
   }

   /**
    * @return the code
    */
   public String getCode()
   {
      return code;
   }

   /**
    * @return the fullDescription
    */
   public String getFullDescription()
   {
      return fullDescription;
   }

   public String getName()
   {
      return name;
   }

   public TokenType getType()
   {
      return type;
   }

   public void setType(TokenType type)
   {
      this.type = type;
   }

   public int getLineNumber()
   {
      return lineNumber;
   }

   public List<Token> getSubTokenList()
   {
      return subTokenList;
   }

   public String getMimeType()
   {
      return mimeType;
   }
   
   /**
    * @param lineNumber the lineNumber to set
    */
   public void setLineNumber(int lineNumber)
   {
      this.lineNumber = lineNumber;
   }

   /**
    * @param shortDescription the shortDescription to set
    */
   public void setShortDescription(String shortDescription)
   {
      this.shortDescription = shortDescription;
   }

   /**
    * @param code the code to set
    */
   public void setCode(String code)
   {
      this.code = code;
   }

   public void setParentToken(Token parentToken)
   {
      this.parentToken = parentToken;
   }
   
   public Token getParentToken()
   {
      return parentToken;
   }
   
   /**
    * @param fullDescription the fullDescription to set
    */
   public void setFullDescription(String fullDescription)
   {
      this.fullDescription = fullDescription;
   }

   /**
    * @param subTokenList the subTokenList to set
    */
   public void setSubTokenList(List<Token> subTokenList)
   {
      this.subTokenList = subTokenList;
   }

   /**
    * @param mimeType the mimeType to set
    */
   public void setMimeType(String mimeType)
   {
      this.mimeType = mimeType;
   }

   /**
    * Adds subToken into the last sub token subTokenList
    * @param token
    * @param subToken
    */
   public void addSubTokenToTheLastSubToken(Token subToken)
   {
      if (subTokenList == null || subTokenList.size() == 0) 
         return;
      
      Token lastToken = subTokenList.get(subTokenList.size() - 1);
      lastToken.addSubToken(subToken);
      
      subToken.parentToken = lastToken;
   }
   
   public void updateTypeOfLastSubTokenOfLastToken(TokenType newType)
   {
      if (subTokenList == null || subTokenList.size() == 0) return;

      Token lastSubToken = subTokenList.get(subTokenList.size() - 1);
      
      lastSubToken.updateTypeOfLastSubToken(newType);
   }    

   public void updateTypeOfLastSubToken(TokenType newType)
   {
      if (subTokenList == null || subTokenList.size() == 0) return;
      
      Token lastSubToken = subTokenList.get(subTokenList.size() - 1);
      
      lastSubToken.setType(newType);
   }
   
   public void addSubToken(Token subToken)
   {
      if (this.subTokenList == null)
      {
         this.subTokenList = new ArrayList<Token>();
      }

      this.subTokenList.add(subToken);
      subToken.setParentToken(this);
   }
   
   public enum TokenType 
   {
      VARIABLE, FUNCTION, PROPERTY, METHOD, KEYWORD, TEMPLATE, TAG, ATTRIBUTE, CDATA, ROOT, TAG_BREAK, CLASS, GROOVY_TAG;
   }

}
