/**
 * Copyright (C) 2009 eXo Platform SAS.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */

 
package org.exoplatform.gwtframework.commons.xml;

import com.google.gwt.xml.client.NamedNodeMap;
import com.google.gwt.xml.client.Node;
import com.google.gwt.xml.client.NodeList;

/**
 * Created by The eXo Platform SAS.
 * 
 * @author <a href="mailto:vitaly.parfonov@gmail.com">Vitaly Parfonov</a>
 * @version $Id: $
 */
public class XMLFinder
{

   /**
    * Method find in document node by <code>parentPath</code> and get value of
    * attribute that set in <code>attribute</code>
    * 
    * @param doc the some XML document
    * @param itemPath the path to node (slash delimited string like
    *          "node/childNode/item")
    * @param attribute the attribute name
    * @return the attribute value or exception if node not found
    * @throws XMLNodeNotFoundException 
    */
   public static String findAttribute(final Node doc, final String itemPath, final String attribute)
      throws XMLNodeNotFoundException
   {
      Node node = findNode(doc, itemPath);
      if (node == null)
      {
         throw new XMLNodeNotFoundException("Node not found " + itemPath);
      }
      NamedNodeMap map = node.getAttributes();
      if (map.getLength() == 0)
      {
         throw new XMLNodeNotFoundException("Node " + itemPath + " don't have attributes");
      }
      Node attr = map.getNamedItem(attribute);
      if (attr == null)
      {
         throw new XMLNodeNotFoundException("Attribute " + attribute + " not found in node " + itemPath);
      }
      return attr.getNodeValue();
   }

   /**
    * Method find in document node by <code>parentPath</code> and get value of
    * node if it's a text value.
    * 
    * @param doc the some XML document
    * @param itemPath the path to the node (slash delimited string like
    *          "node/childNode/item")
    * @return the node value or exception if node not found
    * @throws XMLNodeNotFoundException
    */
   public static String findTextElement(final Node doc, final String itemPath) throws XMLNodeNotFoundException
   {
      Node node = findNode(doc, itemPath);
      if (node == null)
      {
         throw new XMLNodeNotFoundException("Node not found " + itemPath);
      }
      Node firstNode = node.getFirstChild();
      if (firstNode == null)
      {
         throw new XMLNodeNotFoundException("Node " + itemPath + " node is empty");
      }
      if (firstNode.getNodeType() != Node.TEXT_NODE)
      {
         throw new XMLNodeNotFoundException("Node " + itemPath + " not text type");
      }
      return firstNode.getNodeValue();
   }

   /**
    * Find node in parent node by <code>parentPath</code>.
    * 
    * @param parentNode the parent node
    * @param itemPath the path in the parent node (slash delimited string like
    *          "node/childNode/item")
    * @return found node or null if node not found
    */
   private static Node findNode(final Node parentNode, final String itemPath)
   {
      String nodeName;
      String childPath;
      if (itemPath.contains("/"))
      {
         int i = itemPath.indexOf("/", 0);
         nodeName = itemPath.substring(0, i);
         childPath = itemPath.substring(i + 1);
         NodeList nodeList = parentNode.getChildNodes();
         for (int j = 0; j < nodeList.getLength(); j++)
         {
            Node childNode = nodeList.item(j);
            String name = childNode.getNodeName();
            if (name.equals(nodeName))
            {
               return findNode(childNode, childPath);
            }
         }
      }
      else
      {
         nodeName = itemPath;
         NodeList nodeList = parentNode.getChildNodes();
         for (int j = 0; j < nodeList.getLength(); j++)
         {
            Node childNode = nodeList.item(j);
            String name = childNode.getNodeName();
            if (name.equals(nodeName))
            {
               return childNode;
            }
         }
      }
      return null;
   }
}
